/*
 *	EMUL-8: a pdp8e emulator.
 *
 *	Author:
 *		Bill Haygood
 *		41832 Ernest Road
 *		Loon Lake, WA 99148-9607
 *		Internet: billh@comtch.iea.com
 *		Voice/AnsMach/FAX \
 *			or	   509-233-2555
 *		  Cellular/Pager  /
 *
 *	Copyright 1992, 1993, 1994 by the author with all rights reserved.
 *
 *	FPP-8/A module.
 *
 *	If you can answer any of the following questions, please
 *	notify the author and cite the printed reference, if possible.
 *
 *	Questions:
 *	(1) Once the FADDM/FMULM bit gets set, does it stay until the
 *		next FPICL or does it get cleared with each new FPP
 *		instruction execution ?  (Presently implemented as the
 *		latter.)
 *	(2) What do opcodes 5100-5177 do ?  (Presently implemented as
 *		Unknown OpCodes.)
 *
 *	Single Word Direct Addressing Format:
 *	+-----------+---+---+---------------------------+
 *	|  Op Code  | 0 | 1 |          Offset           |
 *	+-----------+---+---+---------------------------+
 *	  0   1   2   3   4   5   6   7   8   9  10  11
 *
 *	Double Word Memory Reference Instruction:
 *	+-----------+---+---+---+-----------+-----------+
 *	|  Op Code  | 1 | 0 | + |     X     |  Address  |
 *	+-----------+---+---+---+-----------+-----------+
 *	  0   1   2   3   4   5   6   7   8   9  10  11
 *	+-----------------------------------------------+
 *	|                    Address                    |
 *	+-----------------------------------------------+
 *	 12  13  14  15  16  17  18  19  20  21  22  23
 *								IMUL,
 *								LEA
 *
 *	Single Word Indirect Addressing Format:
 *	+-----------+---+---+---+-----------+-----------+
 *	|  Op Code  | 1 | 1 | + |     X     |  Offset   |
 *	+-----------+---+---+---+-----------+-----------+
 *	  0   1   2   3   4   5   6   7   8   9  10  11
 *								IMULI,
 *								LEAI
 *
 *	Special Format 1:
 *	+-----------+---+---+---+-----------+-----------+	JNX
 *	|  Op Code  | 0 | 0 | + |     X     |  Address  |
 *	+-----------+---+---+---+-----------+-----------+
 *	  0   1   2   3   4   5   6   7   8   9  10  11
 *	+-----------------------------------------------+
 *	|                    Address                    |
 *	+-----------------------------------------------+
 *	 12  13  14  15  16  17  18  19  20  21  22  23
 *
 *	Special Format 2:					JEQ, JGE,
 *	+-----------+---+---+--------------+------------+	JLE, JA,
 *	|  Op Code  | 0 | 0 |  Extension   |     F      |	JNE, JLT,
 *	+-----------+---+---+--------------+------------+	JGT, JAL,
 *	  0   1   2   3   4   5   6   7   8   9  10  11		JSA, JSR
 *	+-----------------------------------------------+	LDX, ADDX
 *	|                       Y                       |	SETX, SETB
 *	+-----------------------------------------------+	TRAP3,
 *	 12  13  14  15  16  17  18  19  20  21  22  23		TRAP4,
 *
 *	Special Format 3:
 *	+-----------+---+---+--------------+------------+	FEXIT,
 *	|  Op Code  | 0 | 0 |  Extension   |     F      |	FPAUSE,
 *	+-----------+---+---+--------------+------------+	FCLA, FNEG,
 *	  0   1   2   3   4   5   6   7   8   9  10  11		FNORM,
 *								STARTF,
 *								STARTD,
 *								JAC, ALN
 *								ATX, XTA,
 *								FNOP,
 *								STARTE, LTR
 */
#include "pdp8.h"
#include "pdp8regs.h"
#include "pdp8misc.h"

STATIC	UWORD	fppINST, *w ;
STATIC	UBYTE	apt_stored = FALSE ;

/* -------------------------------------------------------------------- */
/*
 *	Output spaces into the trace line.
 */
VOID spaces (INT stop)
{
    INT i = strlen (chrsout) ;
    while (i < stop)
	chrsout [i++] = ' ' ;
}
/* -------------------------------------------------------------------- */
/*
 *	Output the FAC into the trace line.
 */
VOID outfac (INT memdata)
{
/*        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 04200 1031 JA 10000               4056 0001 0002 3110 3755 =  3.141593(FP)
 04201 0000
 04200 1031 JA 10000               4056 0005 0002 3110 3755 2421 0652 3100
 04201 0000                                  =  3.141592653579(EP)
 */
    spaces (40) ;
    sprintf (&chrsout [40], "%04o", fpp_stat) ;
    sprintf (&chrsout [44], " %04o %04o %04o",
				fpp_ac [0], fpp_ac [1], fpp_ac [2]) ;
    if (fpp_data == EP)
    {
	sprintf (&chrsout [59], " %04o %04o %04o\r\n",
				fpp_ac [3], fpp_ac [4], fpp_ac [5]) ;
	if (tracefile)
	    fwrite (chrsout, 1, strlen (chrsout), tracefile) ;
	else
	    fprintf (stderr, "%s", chrsout) ;
	chrsout [0] = '\0' ;
	if (memdata)
	    sprintf (chrsout, "       %04o", *(base + fpp_pc)) ;
	if (--trace <= 0)
	    run = trace = FALSE ;
	spaces (45) ;
	chrsout [45] = '=' ;
	chrsout [46] = ' ' ;
	outfpp (fpp_ac, 47) ;
    }
    else
    {
	spaces (60) ;
	chrsout [60] = '=' ;
	chrsout [61] = ' ' ;
	outfpp (fpp_ac, 62) ;
	if (memdata)
	{
	    if (--trace <= 0)
		run = trace = FALSE ;
	    sprintf (chrsout, "       %04o\r\n", *(base + fpp_pc)) ;
	    if (tracefile)
		fwrite (chrsout, 1, strlen (chrsout), tracefile) ;
	    else
		fprintf (stderr, "%s", chrsout) ;
	}
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Store the APT.
 *
 *	  0   1   2   3   4   5   6   7   8   9  10  11
 *	+-----------------------------------------------+
 *	|   OpAdd   |  BaseReg  |   Index   |    FPC    |
 *	|-----------------------------------------------|
 *	|                      FPC                      |
 *	|-----------------------------------------------|
 *	|                     Index                     |
 *	|-----------------------------------------------|
 *	|                    BaseReg                    |
 *	|-----------------------------------------------|
 *	|                     OpAdd                     |
 *	|-----------------------------------------------|
 *	|                  FAC Exponent                 |
 *	|-----------------------------------------------|
 *	|                   FAC Word1                   |
 *	|-----------------------------------------------|
 *	|                   FAC Word2                   |
 *	|-----------------------------------------------|
 *	|                   FAC Word3                   |
 *	|-----------------------------------------------|
 *	|                   FAC Word4                   |
 *	|-----------------------------------------------|
 *	|                   FAC Word5                   |
 *	+-----------------------------------------------+
 *	  0   1   2   3   4   5   6   7   8   9  10  11
 */
STATIC VOID store_apt (VOID)
{
    w = base + fpp_aptp ;
    *w &= 07770 ;
    *w |= (fpp_pc & 070000) >> 12 ;
    *(w + 1) = fpp_pc & 07777 ;
    if (!(fpp_command & BIT6))	/* Store BaseReg ?	*/
    {
	*w = (*w & 07077) + ((fpp_br & 070000) >> 6) ;
	*(w + 3) = fpp_br & 07777 ;
    }
    if (!(fpp_command & BIT5))	/* Store Index ?	*/
    {
	*w = (*w & 07707) + ((fpp_x0 & 070000) >> 9) ;
	*(w + 2) = fpp_x0 & 07777 ;
    }
    if (!(fpp_command & BIT4))	/* Store Opaddr ?	*/
    {
	*w = (*w & 0777) + ((fpp_opadd & 070000) >> 3) ;
	*(w + 4) = fpp_opadd & 07777 ;
    }
    if (!(fpp_command & BIT7))	/* Store FAC ?		*/
    {
	*(w + 5) = fpp_ac [0] ;
	*(w + 6) = fpp_ac [1] ;
	*(w + 7) = fpp_ac [2] ;
	if (fpp_data == EP)
	{
	    *(w + 8) = fpp_ac [3] ;
	    *(w + 9) = fpp_ac [4] ;
	    *(w + 10) = fpp_ac [5] ;
	}
    }
    apt_stored = TRUE ;
}
/* -------------------------------------------------------------------- */
/*
 *	Perform core divide operations.
 */
STATIC VOID facdiv (UWORD *w)
{
    INT flag ;
    UWORD opr [6] ;
    ULONG hi, lo ;

    opr [0] = *w++ ;
    hi = *w++ << 20 , hi |= *w << 8 ; 
    if (fpp_data == EP)
    {
	hi |= *++w >> 4 ;
	lo = *++w << 28 , lo |= *++w << 16 , lo |= *++w << 4 ;
    }
    else
	lo = 0 ;
    if (flag = hi >> 31)
    {
	hi = ~hi ;
	if (!(lo = -lo))
	    ++hi ;
    }
    while (!(hi & 0x40000000))			/* Normalize divisor	*/
    {
	hi = (hi << 1) | (lo >> 31) ;
	lo <<= 1 ;
	--opr [0] ;
    }
    opr [1] = hi >> 20 ;
    opr [2] = (hi >> 8) & 07777 ;
    if (fpp_data == EP)
    {
	opr [3] = (hi << 4) | (lo >> 28) & 07777 ;
	opr [4] = (lo >> 16) & 07777 ;
	opr [5] = (lo >> 4) & 07777 ;
    }
    else
	opr [3] = opr [4] = opr [5] = 0 ;
    if (!flag)
	storefpp (fpp2d (fpp_ac) / fpp2d (opr), fpp_ac) ;
    else
	storefpp (fpp2d (fpp_ac) / -fpp2d (opr), fpp_ac) ;
}
/* -------------------------------------------------------------------- */
/*
 *	Perform an integer multiply.
 */
STATIC LONG fimul (LONG cand, LONG plier)
{
    DOUBLE d ;
    INT exp ;

    if (cand & 040000000)
	cand |= ~077777777 ;
    if (plier & 040000000)
	plier |= ~077777777 ;
    d = (DOUBLE) cand * (DOUBLE) plier ;
    cand *= plier ;
    frexp (d, &exp) ;
    if (exp > 23)			/* Set overflow & exit	*/
    {
	fpp_stat |= BIT4 ;
	fpp_stat &= ~BIT11 ;		/* Reset RUN in status	*/
	io_flags |= fppFLAG ;		/* Set DONE flag	*/
	fpp_run = fpp_pause = FALSE ;	/* Stop the FPP		*/
    }
    return cand ;
}
/* -------------------------------------------------------------------- */
/*
 *	Unimplemented FPP instruction.
 */
STATIC VOID funim (VOID)
{
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 xxxx Unknown FPP instruction
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "Unknown FPP instruction") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Exit Floating Point - The contents of the FPP registers are
 *	dumped into the active parameter table, the FPP is stopped,
 *	and the FPP flag is set.
 */
STATIC VOID f0000 (VOID)		/* FEXIT   000 000 000 000	*/
{
    store_apt () ;
    fpp_stat &= 07776 ;			/* Reset RUN in status	*/
    io_flags |= fppFLAG ;		/* Set DONE flag	*/
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0000 FEXIT                       0001 0000 0000 0000 0000 0000 0000
                                             =  0.(EP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "FEXIT") ;
	outfac (FALSE) ;
    }
    fpp_run = fpp_pause = FALSE ;
}
/* -------------------------------------------------------------------- */
/*
 *	Suspend FPP operations without updating the APT.  IOT FPST
 *	will cause the FPP to continue.
 */
STATIC VOID f0001 (VOID)		/* FPAUSE  000 000 000 000	*/
{
    fpp_pause = TRUE ;			/* Set pause			*/
    fpp_stat |= BIT10 ;			/* Set PAUSE in status		*/
    fpp_run = FALSE ;			/* Clear FPP run		*/
    fpp_stat &= ~BIT11 ;		/* Clear RUN in status		*/
    io_flags |= fppFLAG ;
    apt_stored = TRUE ;			/* Fake APT storage		*/
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0001 FPAUSE                      0002 0000 0000 0000 =  0.(FP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "FPAUSE") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Clear the FPP Accumulator - Make the FAC fraction zero.  If the
 *	calculating mode is FP or EP, make the FAC exponent zero, also.
 */
STATIC VOID f0002 (VOID)		/* FCLA    000 000 000 010	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] = fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0002 FCLA                        0003  0000 0000 0000 0000 0000 0000
                                             =  0.(EP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "FCLA") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Complement the FPP Accumulator - The FAC fraction is replaced
 *	by its 2's complement.
 */
STATIC VOID f0003 (VOID)		/* FNEG    000 000 000 011	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] ^= 07777 ;
    if (fpp_data == EP)
    {
	fpp_ac [2] ^= 07777 ;
	fpp_ac [3] ^= 07777 ;
	fpp_ac [4] ^= 07777 ;
	if (!(fpp_ac [5] = -fpp_ac [5] & 07777)
			&& !(fpp_ac [4] = ++fpp_ac [4] & 07777)
			&& !(fpp_ac [3] = ++fpp_ac [3] & 07777)
			&& !(fpp_ac [2] = ++fpp_ac [2] & 07777))
	    fpp_ac [1] = ++fpp_ac [1] & 07777 ;
    }
    else
	if (!(fpp_ac [2] = -fpp_ac [2] & 07777))
	    fpp_ac [1] = ++fpp_ac [1] & 07777 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0003 FNEG                        0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "FNEG") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Normalize - If the FAC fraction is non-zero, and if the FPP
 *	mode is FP or EP, the FAC fraction is shifted toward the MSB
 *	until the two MSBs are different from each other or until the
 *	FAC fraction equals 6000 0000.  The FAC exponent is decremented
 *	by one for each position shifted.  If the FAC fraction is 0,
 *	or if the mode is DP, no operation is performed.
 */
STATIC VOID f0004 (VOID)		/* FNORM   000 000 000 100	*/
{
    if (fpp_data > DP)
    {
	INT truth = fpp_ac [1] | fpp_ac [2] ;
	if (fpp_data == EP)
	    truth |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
	if (truth)
	{
	    ULONG high = (fpp_ac [1] << 20) | (fpp_ac [2] << 8) ;
	    ULONG low = 0 ;
	    if (fpp_data == EP)
	    {
		high |= fpp_ac [3] >> 4 ;
		low = (fpp_ac [3] << 28)
				| (fpp_ac [4] << 16) | (fpp_ac [5] << 4) ;
	    }
	    truth = (fpp_ac [1] & 04000) ? TRUE : FALSE ;
	    if (truth)
	    {
		high ^= 0xffffffff ;
		low = -low ;
		if (low == 0)
		    high++ ;
	    }
	    while ((high & 0x40000000) == 0)
	    {
		high = (high << 1) | (low >> 31) ;
		low <<= 1 ;
		fpp_ac [0]-- ;
	    }
	    fpp_ac [0] &= 07777 ;
	    if (truth)
	    {
		high ^= 0xffffffff ;
		low = -low ;
		if (low == 0)
		    high++ ;
	    }
	    fpp_ac [1] = high >> 20 ;
	    fpp_ac [2] = (high >> 8) & 07777 ;
	    if (fpp_data == EP)
	    {
		fpp_ac [3] = ((high << 4) | (low >> 28)) & 07777 ;
		fpp_ac [4] = (low >> 16) & 07777 ;
		fpp_ac [5] = (low >> 4) & 07777 ;
	    }
	}
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0004 FNORM                       0001 0000 0000 0000
                                             = 0.(DP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "FNORM") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Enter 24-bit Floating Point Mode - The FPP enters FP mode.  If
 *	issued in EP mode, the FAC is rounded to 24 bits.
 */
STATIC VOID f0005 (VOID)		/* STARTF  000 000 000 101	*/
{
    apt_stored = FALSE ;
    fpp_stat &= ~(BIT0 | BIT9) ;
    if (fpp_data == EP)
    {
	LONG temp = (fpp_ac [1] << 13) | (fpp_ac [2] << 1)
						| (fpp_ac [3] >> 11) ;
	LONG sign = temp & 0100000000 ;
	if (sign)
	{
	    temp |= ~0177777777 ;
	    temp = -temp ;
	}
	temp++ ;
	if (temp & 0100000000)
	{
	    fpp_ac [0]++ ;
	    temp >>= 1 ;
	}
	if (sign)
	    temp = -temp ;
	fpp_ac [1] = (temp >> 13) & 07777 ;
	fpp_ac [2] = (temp >> 1) & 07777 ;
	fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    }
    fpp_data = FP ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0006 STARTF                      0001
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "STARTF") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Enter Double Precision Mode - The FPP enters DP mode.  If
 *	issued in EP mode, the FAC is chopped to 24 bits.  The FAC
 *	exponent is ignored, but not modified.
 */
STATIC VOID f0006 (VOID)		/* STARTD  000 000 000 110	*/
{
    apt_stored = FALSE ;
    fpp_stat &= ~BIT9 ;
    fpp_stat |= BIT0 ;
    fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    fpp_data = DP ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0006 STARTD                      0001
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "STARTD") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Jump Per FAC - FAC bits 9-23 are loaded into the FPC.
 */
STATIC VOID f0007 (VOID)		/* JAC     000 000 000 111	*/
{
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0007 JAC                         0001 0000 0001 5432
                                             = unnormalized(FP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "JAC") ;
	outfac (FALSE) ;
    }
    apt_stored = FALSE ;
    fpp_pc = ((fpp_ac [1] & 07) << 12) | fpp_ac [2] ;
}
/* -------------------------------------------------------------------- */
/*
 *	In FP and EP modes, the fraction of the FAC is shifted until
 *	the FAC exponent equals the contents of the index register
 *	specified by bits 9-11.  If bits 9-11 of the instruction are
 *	zero, the fraction of the FAC is shifted until the FAC
 *	exponent equals 27 octal (23 decimal).
 *
 *	In DP mode, an arithmetic shift is performed on the FAC.  The
 *	number of shifts is equal to the value of the contents of the
 *	index register specified by bits 9-11.  The sign of the Index
 *	register indicates the direction of shift; a positive sign
 *	causes a shift toward the LSB.  If the shift is toward the
 *	least significant bit, vacated bits are filled with FAC0.  If
 *	the shift is toward the most significant bit, vacated bits are
 *	filled with zeroes.  If bits 9-11 of the instruction are zero,
 *	a 23-bit right shift of the FAC is performed.
 */
STATIC VOID f0010 (VOID)		/* ALN     000 000 001 rrr	*/
{
    LONG high = (fpp_ac [1] << 20) | (fpp_ac [2] << 8) ;
    ULONG low ;
    INT shift = (fppINST & 07) ?
			*(base + (fpp_x0 + (fppINST & 07))) : 027 ;
    if (shift & 04000)
	shift |= ~07777 ;
    switch (fpp_data)
    {
	case EP :
	    high |= (fpp_ac [3] >> 4) ;
	    low = (fpp_ac [3] << 28) | (fpp_ac [4] << 16)
						| (fpp_ac [5] << 4) ;
	    while (fpp_ac [0] != shift)
	    {
		low = (low >> 1) | (high << 31) ;
		high >>= 1 ;
		fpp_ac [0]++ ;
	    }
	    break ;
	case FP :
	    high >>= shift - fpp_ac [0] ;
	    fpp_ac [0] = shift ;
	    break ;
	case DP :
	    if (shift > 0)
		high >>= shift ;
	    else
		high <<= -shift ;
	    break ;
    }
    fpp_ac [1] = (high >> 20) & 07777 ;
    fpp_ac [2] = (high >> 8) & 07777 ;
    if (fpp_data == EP)
    {
	fpp_ac [3] = ((high << 4) | (low >> 28)) & 07777 ;
	fpp_ac [4] = (low >> 16) & 07777 ;
	fpp_ac [5] = (low >> 4) & 07777 ;
    }
    apt_stored = FALSE ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0014 ALN 4                  0006 0001
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "ALN %o",
							fppINST & 07) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
				*(base + fpp_x0 + (fppINST & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	FAC to Index Register - If the mode is FP or EP, the contents
 *	of the FAC are fixed (i.e., shifted until the exponent = 27
 *	octal).  ATX does not test to see if fixing is possible.  If
 *	the mode is DP, the contents of the FAC are already fixed, so
 *	this portion is omitted.  Bits 12-23 of the result are then
 *	loaded into the index register specified by bits 9-11.  The
 *	FAC is not changed by the ATX instruction.
 */
STATIC VOID f0020 (VOID)		/* ATX     000 000 010 rrr	*/
{
    LONG sig = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;

    if (fpp_data > DP)
    {
	if (sig & 040000000)
	{
	    sig |= ~077777777 ;
	    sig = -sig ;
	}
	sig >>= 027 - fpp_ac [0] ;
    }
    *(base + (fpp_x0 + (fppINST & 07))) = sig & 07777 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0024 ATX 4                  4361 0001
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "ATX %o",
							fppINST & 07) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
				*(base + fpp_x0 + (fppINST & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Index Register to FAC - The contents of the Index register
 *	specified by bits 9-11 are loaded into FAC 12-23.  FAC 0-11
 *	is loaded with the contents of FAC12.
 *
 *	FAC 24-59 are clear, 27 octal is then loaded into the FAC
 *	exponent.  If the mode is FP or EP, the FAC is then normalized.
 *	(The normalizing operation is omitted in DP mode.)
 */
STATIC VOID f0030 (VOID)		/* XTA     000 000 011 rrr	*/
{
    fpp_ac [2] = *(base + ((fpp_x0 + (fppINST & 07)) & 077777)) ;
    fpp_ac [1] = (fpp_ac [1] & 04000) ? 07777 : 0 ;
    fpp_ac [0] = 027 ;
    fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data != DP)
    {
	INT sign = fpp_ac [2] & 04000 ;
	fpp_ac [0] = 013 ;
	fpp_ac [1] = fpp_ac [2] ;
	fpp_ac [2] = 0 ;
	if (sign)
	    fpp_ac [1] = - fpp_ac [0] ;
	while ((fpp_ac [1] & 02000) == 0)
	{
	    fpp_ac [1] = fpp_ac [1] << 1 ;
	    fpp_ac [0]-- ;
	}
	if (sign)
	    fpp_ac [0] = - fpp_ac [0] ;
	fpp_ac [0] &= 07777 ;
    }
    apt_stored = FALSE ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0034 XTA 4                  0002 0001 0002 2000 0000
                                             =  2.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "XTA %o",
							fppINST & 07) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
				*(base + fpp_x0 + (fppINST & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	No Operation.
 */
STATIC VOID f0040 (VOID)		/* FNOP	   000 000 100 xxx	*/
{
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0050 FNOP                        0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "FNOP") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Start Extended-Precision Mode - The FPP enters EP mode.  If
 *	the FPP was previously in a mode other than EP, FAC 24-59
 *	are cleared.
 */
STATIC VOID f0050 (VOID)		/* STARTE  000 000 101 xxx	*/
{
    apt_stored = FALSE ;
    if (fpp_data != EP)
	fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    fpp_data = EP ;
    fpp_stat |= BIT9 ;			/* Set EP status bit		*/
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0050 STARTE                      0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	strcpy (&chrsout [strlen (chrsout)], "STARTE") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Load Index Register - Bits 12-23 are loaded into the Index
 *	register specified by bits 9-11.
 */
STATIC VOID f0100 (VOID)		/* LDX	   000 001 000 rrr data	*/
{
    apt_stored = FALSE ;
    *(base + ((fpp_x0 + (fppINST & 07)) & 077777))
					= *(base + fpp_pc) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1101 LDX 4523,1             4523 0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LDX %04o,%o",
					*(base + fpp_pc), fppINST & 07) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
				*(base + fpp_x0 + (fppINST & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Add to Index Register - Bits 12-23 are added to the contents
 *	of the Index register specified by bits 9-11.
 */
STATIC VOID f0110 (VOID)		/* ADDX    000 001 001 rrr data	*/
{
    apt_stored = FALSE ;
    w = (base + ((fpp_x0 + (fppINST & 07)) & 077777)) ;
    *w = (*w + *(base + fpp_pc)) & 07777 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1111 ADDX 4523,1            6254 0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "ADDX %04o,%o",
				*(base + fpp_pc), fppINST & 07) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
				*(base + fpp_x0 + (fppINST & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	The contents of the effective address are loaded into the FAC.
 *	If the mode is DP or FP, bits 24-59 of the FAC fraction are
 *	not used.
 */
STATIC VOID f0200 (VOID)		/* FLDA    000 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data > DP)
	fpp_ac [0] = *w ;
    fpp_ac [1] = *++w , fpp_ac [2] = *++w ;
    if (fpp_data == EP)
	fpp_ac [3] = *++w , fpp_ac [4] = *++w , fpp_ac [5] = *++w ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0201 FLDA BR+1 (25403)           0003  0002 6000 0000 0000 0000 0000
                                              = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FLDA BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f0200.
 */
STATIC VOID f0400 (VOID)		/* FLDA    000 10+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data > DP)
	fpp_ac [0] = *w++ ;
    fpp_ac [1] = *w++ , fpp_ac [2] = *w ;
    if (fpp_data == EP)
	fpp_ac [3] = *++w , fpp_ac [4] = *++w , fpp_ac [5] = *++w ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0401 FLDA 14523,0+ (12345)  1234 0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FLDA %o%04o,%o",
				fppINST & 07, *(base + fpp_pc),
				(fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f0200.
 */
STATIC VOID f0600 (VOID)		/* FLDA    000 11+ rrr off	*/
{
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data > DP)
	fpp_ac [0] = *w++ ;
    fpp_ac [1] = *w++ , fpp_ac [2] = *w ;
    if (fpp_data == EP)
	fpp_ac [3] = *++w , fpp_ac [4] = *++w , fpp_ac [5] = *++w ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0601 FLDA 1,0+ (25403)      1432 0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FLDA %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC = 0.
 */
STATIC VOID f1000 (VOID)		/* JEQ	   001 000 000 msb addr	*/
{
    INT temp ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1001 JEQ 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JEQ %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    temp = fpp_ac [1] | fpp_ac [2] ;
    if (fpp_data == EP)
	temp |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
    if (temp)
	fpp_pc = ++fpp_pc & 077777 ;
    else
	fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC >= 0.
 */
STATIC VOID f1010 (VOID)		/* JGE     001 000 001 msb addr	*/
{
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1011 JGE 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JGE %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    if (fpp_ac [1] < 04000)
	fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
    else
	fpp_pc = ++fpp_pc & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC <= 0.
 */
STATIC VOID f1020 (VOID)		/* JLE     001 000 010 msb addr	*/
{
    INT temp ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1021 JLE 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JLE %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    temp = fpp_ac [1] | fpp_ac [2] ;
    if (fpp_data == EP)
	temp |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
    if ((fpp_ac [1] & 04000) || !temp)
	fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
    else
	fpp_pc = ++fpp_pc & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23.
 */
STATIC VOID f1030 (VOID)		/* JA      001 000 011 msb addr	*/
{
    apt_stored = FALSE ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1031 JA 14523                    0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JA %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC is not zero.
 */
STATIC VOID f1040 (VOID)		/* JNE     001 000 100 msb addr	*/
{
    INT temp ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1041 JNE 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JNE %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    temp = fpp_ac [1] | fpp_ac [2] ;
    if (fpp_data == EP)
	temp |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
    if (temp)
	fpp_pc = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    else
	fpp_pc = ++fpp_pc & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC < 0.
 */
STATIC VOID f1050 (VOID)		/* JLT     001 000 101 msb addr	*/
{
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1051 JLT 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JLT %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    if (fpp_ac [1] & 04000)
	fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
    else
	fpp_pc = ++fpp_pc & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC > 0.
 */
STATIC VOID f1060 (VOID)		/* JGT     001 000 110 msb addr	*/
{
    INT temp ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1061 JGT 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JGT %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    temp = fpp_ac [1] | fpp_ac [2] ;
    if (fpp_data == EP)
	temp |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
    if (temp && fpp_ac [1] < 04000)
	fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
    else
	fpp_pc = ++fpp_pc & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Branch to address specified in bits 9-23 if FAC exponent is
 *	greater than 27 (octal).  This condition signifies that the
 *	FAC contains a number too large to be fixed in 24 bits.
 */
STATIC VOID f1070 (VOID)		/* JAL     001 000 111 msb addr	*/
{
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1071 JEQ 14523                   0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JAL %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    apt_stored = FALSE ;
    if (fpp_ac [0] > 027)
	fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
    else
	fpp_pc = ++fpp_pc & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Set Index Register Pointer - Bits 9-23 are loaded into X0.
 */
STATIC VOID f1100 (VOID)		/* SETX    001 001 000 msb addr	*/
{
    apt_stored = FALSE ;
    fpp_x0 = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1101 SETX 14523                  0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "SETX %05o", fpp_x0) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Set Base Register Pointer - Bits 9-23 are loaded into the BR.
 */
STATIC VOID f1110 (VOID)		/* SETB    001 001 001 msb addr	*/
{
    apt_stored = FALSE ;
    fpp_br = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1111 SETB 14523                  0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "SETB %05o", fpp_br) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Jump and Save at Address - A 'JA' to the current value of the
 *	FPC is constructed and stored in core memory at the address
 *	specified by bits 9-23 of the instruction.  The FPC is then
 *	changed to equal bits 9-23 + 2 of the instruction.
 */
STATIC VOID f1120 (VOID)		/* JSA     001 001 010 msb addr	*/
{
    UWORD next = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    apt_stored = FALSE ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1130 JSA 04523                   0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JSA %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
    w = base + next++ ;
    *w++ = 01030 + ((fpp_pc & 070000) >> 12) ;
    *w = fpp_pc & 07777 ;
    fpp_pc = ++next & 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	Jump and Save Return - A 'JA' to the current value of the FPC
 *	is constructed and stored in core memory locations BR+1 and
 *	BR+2.  Instruction bits 9-23 are then loaded into the FPC.
 */
STATIC VOID f1130 (VOID)		/* JSR     001 001 011 msb addr	*/
{
    ULONG retn = (fpp_pc + 1) & 077777 ;
    apt_stored = FALSE ;
    w = base + (fpp_br + 1) ;
    *w++ = 01030 + (retn >> 12) ;
    *w = retn & 07777 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1130 JSR 04523                   0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JSR %o%04o",
				fppINST & 07, *(base + fpp_pc)) ;
	outfac (TRUE) ;
    }
    fpp_pc = ((fppINST & 07) << 12) + *(base + fpp_pc) ;
}
/* -------------------------------------------------------------------- */
/*
 *	The contents of the effective address are added to the FAC.
 *	In DP mode, no alignment or normalization occurs.  The 24
 *	bits from memory are combined with bits 0-23 of the FAC.  In
 *	FP or EP mode, the two words are aligned by right-shifting
 *	the fraction with the lesser exponent until the two exponents
 *	are the same.  In FP mode, bits shifted out of bit 23 are
 *	shifted into bits 24-35.  Bits shifted out of bit 35 (FP) or
 *	bit 59 (EP) are lost.  The two fractions are then added using
 *	either the 24 MSB (FP) or all 60 bits (EP).  The result is
 *	normalized.  In FP mode, the result is then rounded.  If
 *	either argument is zero, or if its exponent is of such a value
 *	that alignment will shift the fraction completely out of its
 *	register, no shifting occurs.  Under these circumstances, the
 *	FAC is either cleared or loaded with the contents of the
 *	effective address.
 */
STATIC VOID f1200 (VOID)		/* FADD    001 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				+ (((ULONG) *(w + 1) << 12) | *(w + 2)) ;
	fpp_ac [1] = (addend >> 12) & 07777 ;
	fpp_ac [2] = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) + fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1201 FADD BR+1 (25403)           0003 0002 6000 0000
                                             = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FADD BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f1200.
 */
STATIC VOID f1400 (VOID)		/* FADD    001 10+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				+ (((ULONG) *w << 12) | *(w + 1)) ;
	fpp_ac [1] = (addend >> 12) & 07777 ;
	fpp_ac [2] = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) + fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1401 FADD 14523,0+          6637 0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FADD %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >>3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f1200.
 */
STATIC VOID f1600 (VOID)		/* FADD    001 110 rrr off	*/
{					/* FADD    001 111 rrr off	*/
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				+ (((ULONG) *w << 12) | *(w + 1)) ;
	fpp_ac [1] = (addend >> 12) & 07777 ;
	fpp_ac [2] = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) + fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1601 FADD 1,0+ (25403)      2455 0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FADD %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Jump if Non-zero indeX register- The specified index register
 *	is incremented if bit 5 = 1.  If the (incremented) index
 *	register is not zero, bits 9-23 are loaded into the FPC,
 *	causing a jump.
 */
STATIC VOID f2000 (VOID)		/* JNX     010 00+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    w = base + (fpp_x0 + ((fppINST & 070) >> 3)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
/*        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 2111 JNX  14523,1+          4322 0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "JNX %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (*w)
	fpp_pc = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    else
	if (++fpp_pc & 0100000)
	    fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	The contents of the effective address are subtracted from the
 *	FAC.  In DP mode, no alignment or normalization occurs.  The 24
 *	bits from memory are combined with bits 0-23 of the FAC.  In
 *	FP or EP mode, the two words are aligned by right-shifting
 *	the fraction with the lesser exponent until the two exponents
 *	are the same.  In FP mode, bits shifted out of bit 23 are
 *	shifted into bits 24-35.  Bits shifted out of bit 35 (FP) or
 *	bit 59 (EP) are lost.  The two fractions are then subtrtacted
 *	using either the 24 MSB (FP) or all 60 bits (EP).  The result
 *	is normalized.  In FP mode, the result is then rounded.  If
 *	either argument is zero, or if its exponent is of such a value
 *	that alignment will shift the fraction completely out of its
 *	register, no shifting occurs.  Under these circumstances, the
 *	FAC is either cleared or loaded with the contents of the
 *	effective address.
 */
STATIC VOID f2200 (VOID)		/* FSUB    010 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				- (((ULONG) *(w + 1) << 12) | *(w + 2)) ;
	fpp_ac [1] = (addend >> 12) & 07777 ;
	fpp_ac [2] = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) - fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 2201 FSUB BR+1 (25403)           0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FSUB BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f2200.
 */
STATIC VOID f2400 (VOID)		/* FSUB    010 100 rrr msb addr	*/
{					/* FSUB    010 101 rrr msb addr	*/
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				- (((ULONG) *w << 12) | *(w + 1)) ;
	fpp_ac [1] = (addend >> 12) & 07777 ;
	fpp_ac [2] = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) - fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1401 FSUB 14523,0+ (51244)  6724 0003 0002 6000 0000
 12346 4523                                  = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FSUB %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f2200.
 */
STATIC VOID f2600 (VOID)		/* FSUB    010 110 rrr off	*/
{					/* FSUB    010 111 rrr off	*/
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				- (((ULONG) *w << 12) | *(w + 1)) ;
	fpp_ac [1] = (addend >> 12) & 07777 ;
	fpp_ac [2] = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) - fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 2601 FSUB 1,0+ (25403)      2422 0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FSUB %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Trapped Instructions - Set the instruction trap status bit
 *	and exit.  The 15-bit address is placed in the APT OPADD.
 */
STATIC VOID f3000 (VOID)		/* TRAP3   011 00x xxx msb addr	*/
{					/* TRAP4   100 00x xxx msb addr	*/
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    fpp_stat |= BIT1 ;			/* Set TRAP status bit	*/
    fpp_stat &= 07776 ;			/* Reset RUN in status	*/
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 3001 TRAP3 14523                 0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "TRAP%o %05o",
					fppINST >> 9, fpp_opadd) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
    store_apt () ;			/* Save APT and exit	*/
    io_flags |= fppFLAG ;		/* Set DONE flag	*/
    fpp_run = fpp_pause = FALSE ;
}
/* -------------------------------------------------------------------- */
/*
 *	The old FAC is the dividend, the contents of the effective
 *	address are the divisor.  The division is carried 26 or 61
 *	bits.  Lesser bits of the remainder are lost.  In DP mode,
 *	the result is rounded to 24 bits.  In FP mode, the result
 *	is normalized and then rounded to 24 bits.  In EP mode, the
 *	result is normalized and truncated to 60 bits.  In FP and EP
 *	modes, a preliminary test is made to ensure that the divisor
 *	is a normalized number.  If the divisor is not normalized,
 *	it is first normalized before proceeding with the divide.
 *	This operation eliminates the possibility of divide overflow.
 */
STATIC VOID f3200 (VOID)		/* FDIV    011 01x xxx xxx	*/
{
    INT flag = FALSE ;

    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    if ((fpp_data < EP) && *(w + 1) | *(w + 2))
	flag = TRUE ;
    if (fpp_data == EP &&
		(*(w + 1) | *(w + 2) | *(w + 3) | *(w + 4) | *(w + 5)))
	flag = TRUE ;
    if (flag)
    {
	apt_stored = FALSE ;
	if (fpp_data == DP)
	{
	    LONG dividend = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	    LONG divisor = ((ULONG) *(w + 1) << 12) | *(w + 2) ;
	    if (dividend & 040000000)
		dividend |= ~077777777 ;
	    if (divisor & 040000000)
		divisor |= ~077777777 ;
	    dividend /= divisor ;
	    fpp_ac [1] = (dividend >> 12) & 07777 ;
	    fpp_ac [2] = dividend & 07777 ;
	}
	else
	    facdiv (w) ;
	flag = FALSE ;
    }
    else
    {
	fpp_stat |= BIT3 ;		/* Set DIVZ status bit	*/
	store_apt () ;			/* Save APT and exit	*/
	fpp_stat &= 07776 ;		/* Reset RUN in status	*/
	io_flags |= fppFLAG ;		/* Set DONE flag	*/
	flag = TRUE ;
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0201 FDIV BR+1 (25403)           0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FDIV BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
    if (flag)
	fpp_run = fpp_pause = FALSE ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f3200.
 */
STATIC VOID f3400 (VOID)		/* FDIV    011 10+ rrr msb addr	*/
{
    INT flag = FALSE ;

    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data < EP && *w | *(w + 1))
	flag = TRUE ;
    if (fpp_data == EP &&
		(*(w + 1) | *(w + 2) | *(w + 3) | *(w + 4) | *(w + 5)))
	flag = TRUE ;
    if (flag)
    {
	if (fpp_data == DP)
	{
	    LONG dividend = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	    LONG divisor = ((ULONG) *w << 12) | *(w + 1) ;
	    if (dividend & 040000000)
		dividend |= ~077777777 ;
	    if (divisor & 040000000)
		divisor |= ~077777777 ;
	    dividend /= divisor ;
	    fpp_ac [1] = (dividend >> 12) & 07777 ;
	    fpp_ac [2] = dividend & 07777 ;
	}
	else
	    facdiv (w) ;
	flag = FALSE ;
    }
    else
    {
	fpp_stat |= BIT3 ;		/* Set DIVZ status bit	*/
	store_apt () ;			/* Save APT and exit	*/
	fpp_stat &= 07776 ;		/* Reset RUN in status	*/
	io_flags |= fppFLAG ;		/* Set DONE flag	*/
	flag = TRUE ;
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 3502 FDIV 25403,0+ (25403)  5600 0003 0002 6000 0000 0000 0000 0000
 12346 5403                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FDIV %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
    if (flag)
	fpp_run = fpp_pause = FALSE ;
}
/* -------------------------------------------------------------------- */
STATIC VOID f3600 (VOID)		/* FDIV    011 11+ rrr off	*/
{
    INT flag = FALSE ;

    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if ((fpp_data < EP) && *w | *(w + 2))
	flag = TRUE ;
    if (fpp_data == EP &&
		(*(w + 1) | *(w + 2) | *(w + 3) | *(w + 4) | *(w + 5)))
	flag = TRUE ;
    if (flag)
    {
	if (fpp_data == DP)
	{
	    LONG dividend = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	    LONG divisor = ((ULONG) *w << 12) | *(w + 1) ;
	    if (dividend & 040000000)
		dividend |= ~077777777 ;
	    if (divisor & 040000000)
		divisor |= ~077777777 ;
	    dividend /= divisor ;
	    fpp_ac [1] = (dividend >> 12) & 07777 ;
	    fpp_ac [2] = dividend & 07777 ;
	}
	else
	    facdiv (w) ;
	flag = FALSE ;
    }
    else
    {
	fpp_stat |= BIT3 ;		/* Set DIVZ status bit	*/
	store_apt () ;			/* Save APT and exit	*/
	fpp_stat &= 07776 ;		/* Reset RUN in status	*/
	io_flags |= fppFLAG ;		/* Set DONE flag	*/
	flag = TRUE ;
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 3601 FDIV 1,0+ (25403)      4550 0003 0002 6000 0000
                                             = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FDIV %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
    if (flag)
	fpp_run = fpp_pause = FALSE ;
}
/* -------------------------------------------------------------------- */
/*
 *	The old FAC is the multiplicand, the contents of the effective
 *	address are the multiplier.  The 36 (FP) or 72 (EP) MSB of the
 *	product are computed.  In DP mode, the result is rounded to 24
 *	bits.  In FP mode, the result is normalized and then rounded to
 *	24 bits.  In EP mode, the result is normalized and truncated to
 *	60 bits.
 */
STATIC VOID f4200 (VOID)		/* FMUL    100 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG cand = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *(w + 1) << 12) | *(w + 2) ;
	cand = fimul (cand, plier) ;
	fpp_ac [1] = (cand >> 12) & 07777 ;
	fpp_ac [2] = cand & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) * fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 3201 FMUL BR+1 (25403)           0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FMUL BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f4200.
 */
STATIC VOID f4400 (VOID)		/* FMUL    100 10+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG cand = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *w << 12) | *(w + 1) ;
	cand = fimul (cand, plier) ;
	fpp_ac [1] = (cand >> 12) & 07777 ;
	fpp_ac [2] = cand & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) * fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 4502 FMUL 25403,0+ (25403)  4223 0003 0002 6000 0000 0000 0000 0000
 12346 5403                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FMUL %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f4200.
 */
STATIC VOID f4600 (VOID)		/* FMUL    100 11+ rrr off	*/
{
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data == DP)
    {
	LONG cand = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *w << 12) | *(w + 1) ;
	cand = fimul (cand, plier) ;
	fpp_ac [1] = (cand >> 12) & 07777 ;
	fpp_ac [2] = cand & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) * fpp2d (w), fpp_ac) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 3601 FMUL 1,0+ (25403)      0454 0003 0002 6000 0000
                                             = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FMUL %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Load Truth - If the condition is met, +1 (2000 0000 in DP
 *	mode) is loaded into the FAC.  If the condition is not met,
 *	the FAC is cleared.
 */
STATIC VOID f5000 (VOID)		/* LTR EQ  101 000 000 xxx	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] |= fpp_ac [2] ;
    if (fpp_data == EP)
	fpp_ac [1] |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
    fpp_ac [1] = (!fpp_ac [1]) ? 02000 : 0 ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
    {
	fpp_ac [0] = 0 ;
	if (fpp_ac [1])
	    ++fpp_ac [0] ;
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 500x LTR EQ                      0001 0001 2000 0000
                                             =  1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR EQ") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5000.
 */
STATIC VOID f5010 (VOID)		/* LTR GE  101 000 001 xxx	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] = ((fpp_ac [1] & 04000) ^ 04000) >> 1 ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = (fpp_ac [1]) ? 1 : 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 501x LTR GE                      0001 0001 2000 000
                                             =  1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR GE") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5000.
 */
STATIC VOID f5020 (VOID)		/* LTR LE  101 000 010 xxx	*/
{
    apt_stored = FALSE ;
    if (fpp_ac [1] & 04000)
	fpp_ac [1] = 02000 ;
    else
    {
	fpp_ac [1] |= fpp_ac [2] ;
	if (fpp_data == EP)
	    fpp_ac [1] |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
	fpp_ac [1] = (!fpp_ac [1]) ? 02000 : 0 ;
    }
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = (fpp_ac [1]) ? 1 : 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 502x LTR LE                      0001 0001 2000 000
                                             =  1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR LE") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	+1 (2000 0000 in DP mode) is loaded into the FAC.
 */
STATIC VOID f5030 (VOID)		/* LTR SI  101 000 011 xxx	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] = 02000 ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = 1 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 501x LTR SI                      0001 0001 2000 0000 0000 0000 0000
                                             =  1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR SI") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5000.
 */
STATIC VOID f5040 (VOID)		/* LTR NE  101 000 100 xxx	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] |= fpp_ac [2] ;
    if (!fpp_ac [1] && fpp_data == EP)
	fpp_ac [1] |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    fpp_ac [1] = (fpp_ac [1]) ? 02000 : 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = (fpp_ac [1]) ? 1 : 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 504x LTR NE                      0001 0000 0000 0000
                                             =  0.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR NE") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5000.
 */
STATIC VOID f5050 (VOID)		/* LTR LT  101 000 101 xxx	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] = (fpp_ac [1] & 04000) >> 1 ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = (fpp_ac [1]) ? 1 : 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 505x LTR LT                      0001 0000 0000 0000 0000 0000 0000
                                             =  0.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR LT") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5000.
 */
STATIC VOID f5060 (VOID)		/* LTR GT  101 000 110 xxx	*/
{
    apt_stored = FALSE ;
    if (!(fpp_ac [1] & 04000))
    {
	fpp_ac [1] |= fpp_ac [2] ;
	if (fpp_ac [1] == 0 && fpp_data == EP)
	    fpp_ac [1] |= fpp_ac [3] | fpp_ac [4] | fpp_ac [5] ;
	if (fpp_ac [1])
	    fpp_ac [1] = 02000 ;
    }
    else
	fpp_ac [1] = 0 ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = (fpp_ac [1]) ? 1 : 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 506x LTR GT                      0001 0000 0000 0000
                                             =  0.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR GT") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5000.
 */
STATIC VOID f5070 (VOID)		/* LTR AL  101 000 111 xxx	*/
{
    apt_stored = FALSE ;
    fpp_ac [1] = (fpp_ac [0] > 027) ? 02000 : 0 ;
    fpp_ac [2] = fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
    if (fpp_data > DP)
	fpp_ac [0] = (fpp_ac [1]) ? 1 : 0 ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 507x LTR AL                      0001 0000 0000 0000
                                        =  0.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "LTR AL") ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	ADD to Memory - The calculation described above for FADD
 *	occurs, except that the FAC is not changed.  The result of
 *	the computation is stored at the effective address.
 */
STATIC VOID f5200 (VOID)		/* FADDM   101 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    fpp_stat |= BIT7 ;			/* Set FADDM in status		*/
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				+ (((ULONG) *(w + 1) << 12) | *(w + 2)) ;
	*++w = (addend >> 12) & 07777 ;
	*++w = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) + fpp2d (w), w) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1201 FADDM BR+1 (25403)          0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FADDM BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5200.
 */
STATIC VOID f5400 (VOID)		/* FADDM   101 10+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    fpp_stat |= BIT7 ;			/* Set FADDM in status		*/
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				+ (((ULONG) *w << 12) | *(w + 1)) ;
	*w++ = (addend >> 12) & 07777 ;
	*w = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) + fpp2d (w), w) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 5401 FADDM 14523,0+         5612 0003 0002 6000 0000 0000 0000 0000
 12346 4523                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FADDM %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f5200.
 */
STATIC VOID f5600 (VOID)		/* FADDM   101 11+ rrr off	*/
{
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    fpp_stat |= BIT7 ;			/* Set FADDM in status		*/
    if (fpp_data == DP)
    {
	LONG addend = (((ULONG) fpp_ac [1] << 12) | fpp_ac [2])
				+ (((ULONG) *w << 12) | *(w + 1)) ;
	*w++ = (addend >> 12) & 07777 ;
	*w = addend & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) + fpp2d (w), w) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 1601 FADDM 1,0+ (25403)     7462 0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FADDM %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	The author could not make much sense of the documentation in the
 *	"FPP-8/A User's Manual" page 2-8 on the use of these instructions:
 *
 *	IMUL	6	C(FAC*C(Y)->FAC
 *	IMULI	7
 *	Available in DP mode only.  The contents of the effective address
 *	are multiplied by the contents of the FAC, using the rules of
 *	integer arithmetic.  (The binary point is to the right of bit
 *	23.)  The result is loaded into the FAC.  A continuous test of
 *	overflow is maintained.  If overflow occurs, the 24 bits of the
 *	FAC are the 24 LSB of the answer, but an unknown number of MSB
 *	has been lost.
 *
 *	LEA	6	Y->FAC
 *	LEAI	7
 *	Available in FP and EP modes only.  The effective address (not
 *	its contents) is loaded into bits 9-23 of the FAC.  The mode is
 *	then changed to DP.
 *
 *	John Cowan supplied the following documentation from the 1976-77
 *	edition of the "PDP/8A Minicomputer Handbook":
 *
 *	Mnem.	Op Code	Operation	Description
 *
 *	LEA	6	Y -> FAC	The effective address (not its
 *	LEAI	7			contents) is loaded into the FAC
 *					(available only in FP and EP
 *					modes).
 *
 *	LEA and LEAI have only one addressing mode available to each --
 *	the double-word direct reference for the LEA instruction, and the
 *	single-word indirect reference for the LEAI instruction.  Each
 *	operation is begun similarly; i.e., the address calculation is
 *	performed in whichever mode has been programmed; however, when
 *	the address calculation for LEA and LEAI is complete, the FPP
 *	loads the operand address into the FAC and then switches to DP
 *	mode.  In DP mode, the LEA and LEAI instructions are replaced by
 *	IMUL and IMULI respectively.
 *
 *	IMUL	6	C(FAC)*C(Y) -> FAC
 *	IMULI	7
 *	Integer multiply.  An address calculation is accomplished, and
 *	then an integer multiply is performed between the contents of the
 *	effective address and the FAC.  The 24-bit result is loaded into
 *	the FAC.  The old FAC is the multiplier; the contents of the
 *	effective address is the multiplicand.  The operation is similar
 *	to FMUL, except that the binary point is assumed to be the right
 *	of bit 23.  Any propagation of bits out of bit 0 are lost.  A
 *	continuous check of overflow is maintained.  Available only in DP
 *	mode.
 *
 *	Additional note from John Cowan:
 *	You are correct in believing that LEA occupies opcodes 6000-6177
 *	and LEAI occupies 7000-7177.  The trick is to realize that even
 *	though bits 3 and 4 are 00 (which normally indicates a special-
 *	format instruction), LEA behaves as though those bits were 10,
 *	and LEAI behaves as if bits 3 and 4 were 11.  All further bits
 *	are interpreted appropriately.
 *
 *	Therefore, the format of LEA is:
 *
 *	bits:	0    1    2    3    4    5    6    7    8    9    10   11
 *	value:  1    1    0    0    0    +    XR   XR   XR   FLD  FLD  FLD
 *
 *	where + is the autoincrement bit, XR is the index register,
 *	FLD is the field, and bits 12-23 are the absolute address.
 *
 *	Similarly, the format of LEAI is:
 *
 *	bits:	0    1    2    3    4    5    6    7    8    9    10   11
 *	value:	1    1    1    0    0    +    XR   XR   XR   OFF  OFF  OFF
 *
 *	where + and XR is the same, and OFF is the offset into the FPP
 *	base page.
 */
STATIC VOID f6000 (VOID)	/* IMUL  (DP)      110 00+ rrr msb addr	*/
{				/* LEA  (FP or EP) 110 00+ rrr msb addr	*/
    INT data_mode = fpp_data ;

    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    if (fpp_data == DP)
    {
	LONG cand = (fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *w << 12) | *(w + 1) ;
	cand = fimul (cand, plier) ;
	fpp_ac [1] = (cand >> 12) & 07777 ;
	fpp_ac [2] = cand & 07777 ;
    }
    else
    {
	fpp_ac [1] = fpp_opadd >> 12 ;
	fpp_ac [2] = fpp_opadd & 07777 ;
	fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
	fpp_stat &= ~BIT9 ;		/* Status: Clear EP mode	*/
	fpp_stat |= BIT0 ;		/* Status: Set DP mode		*/
	fpp_data = DP ;			/* Set DP mode			*/
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 6002 IMUL 25403                  4003 0002 0007 4426
       5403                                  = unnormalized
 12345 7002 IMULI 25403                 4003 0002 0007 4426
       5403                                  = unnormalized
 12345 6002 LEA 25403                   4003 0002 0002 5403
       4523                                  = unnormalized
 12345 7002 LEAI 25403                  4003 0002 0002 5403
       4523                                  = unnormalized
 */
    if (fpp_trace)
    {
	if (data_mode == DP)
	{
	    strcpy (&chrsout [strlen (chrsout)], "IMUL") ;
	    if (fppINST >> 9 == 7)
		strcpy (&chrsout [strlen (chrsout)], "I") ;
	}
	else
	{
	    strcpy (&chrsout [strlen (chrsout)], "LEA") ;
	    if (fppINST >> 9 == 7)
		strcpy (&chrsout [strlen (chrsout)], "I") ;
	}
	sprintf (&chrsout [strlen (chrsout)], " %05o", fpp_opadd) ;
	outfac (FALSE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	The contents of the FAC are stored at the effective address.
 *	The FAC is not changed.
 */
STATIC VOID f6200 (VOID)		/* FSTA    110 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data > DP)
	*w = fpp_ac [0] ;
    *++w = fpp_ac [1] ;
    *++w = fpp_ac [2] ;
    if (fpp_data == EP)
	*++w = fpp_ac [3] , *++w = fpp_ac [4] , *++w = fpp_ac [5] ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0201 FSTA BR+1 (25403)           0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FSTA BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f6200.
 */
STATIC VOID f6400 (VOID)		/* FSTA    110 10+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data > DP)
	*w = fpp_ac [0] ;
    *++w = fpp_ac [1] ;
    *++w = fpp_ac [2] ;
    if (fpp_data == EP)
	*++w = fpp_ac [3] , *++w = fpp_ac [4] , *++w = fpp_ac [5] ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 6502 FSTA 25403,0+ (25403)  4232 0003  0002 6000 0000 0000 0000 0000
 12346 5403                                   = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FSTA %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f6200.
 */
STATIC VOID f6600 (VOID)		/* FSTA    110 11+ rrr off	*/
{
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    if (fpp_data > DP)
	*w = fpp_ac [0] ;
    *++w = fpp_ac [1] ;
    *++w = fpp_ac [2] ;
    if (fpp_data == EP)
	*++w = fpp_ac [3] , *++w = fpp_ac [4] , *++w = fpp_ac [5] ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 0601 FLDA 1,0+ (25403)      1165 0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FLDA %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	The author could not make much sense of the documentation in the
 *	"FPP-8/A User's Manual" page 2-8 on the use of these instructions:
 *
 *	IMUL	6	C(FAC*C(Y)->FAC
 *	IMULI	7
 *	Available in DP mode only.  The contents of the effective address
 *	are multiplied by the contents of the FAC, using the rules of
 *	integer arithmetic.  (The binary point is to the right of bit
 *	23.)  The result is loaded into the FAC.  A continuous test of
 *	overflow is maintained.  If overflow occurs, the 24 bits of the
 *	FAC are the 24 LSB of the answer, but an unknown number of MSB
 *	has been lost.
 *
 *	LEA	6	Y->FAC
 *	LEAI	7
 *	Available in FP and EP modes only.  The effective address (not
 *	its contents) is loaded into bits 9-23 of the FAC.  The mode is
 *	then changed to DP.
 *
 *	John Cowan supplied the following documentation from the 1976-77
 *	edition of the "PDP/8A Minicomputer Handbook":
 *
 *	Mnem.	Op Code	Operation	Description
 *
 *	LEA	6	Y -> FAC	The effective address (not its
 *	LEAI	7			contents) is loaded into the FAC
 *					(available only in FP and EP
 *					modes).
 *
 *	LEA and LEAI have only one addressing mode available to each --
 *	the double-word direct reference for the LEA instruction, and the
 *	single-word indirect reference for the LEAI instruction.  Each
 *	operation is begun similarly; i.e., the address calculation is
 *	performed in whichever mode has been programmed; however, when
 *	the address calculation for LEA and LEAI is complete, the FPP
 *	loads the operand address into the FAC and then switches to DP
 *	mode.  In DP mode, the LEA and LEAI instructions are replaced by
 *	IMUL and IMULI respectively.
 *
 *	IMUL	6	C(FAC)*C(Y) -> FAC
 *	IMULI	7
 *	Integer multiply.  An address calculation is accomplished, and
 *	then an integer multiply is performed between the contents of the
 *	effective address and the FAC.  The 24-bit result is loaded into
 *	the FAC.  The old FAC is the multiplier; the contents of the
 *	effective address is the multiplicand.  The operation is similar
 *	to FMUL, except that the binary point is assumed to be the right
 *	of bit 23.  Any propagation of bits out of bit 0 are lost.  A
 *	continuous check of overflow is maintained.  Available only in DP
 *	mode.
 *
 *	Additional note from John Cowan:
 *	You are correct in believing that LEA occupies opcodes 6000-6177
 *	and LEAI occupies 7000-7177.  The trick is to realize that even
 *	though bits 3 and 4 are 00 (which normally indicates a special-
 *	format instruction), LEA behaves as though those bits were 10,
 *	and LEAI behaves as if bits 3 and 4 were 11.  All further bits
 *	are interpreted appropriately.
 *
 *	Therefore, the format of LEA is:
 *
 *	bits:	0    1    2    3    4    5    6    7    8    9    10   11
 *	value:  1    1    0    0    0    +    XR   XR   XR   FLD  FLD  FLD
 *
 *	where + is the autoincrement bit, XR is the index register,
 *	FLD is the field, and bits 12-23 are the absolute address.
 *
 *	Similarly, the format of LEAI is:
 *
 *	bits:	0    1    2    3    4    5    6    7    8    9    10   11
 *	value:	1    1    1    0    0    +    XR   XR   XR   OFF  OFF  OFF
 *
 *	where + and XR is the same, and OFF is the offset into the FPP
 *	base page.
 */
STATIC VOID f7000 (VOID)	/* IMULI (DP)      111 00+ rrr off	*/
{				/* LEAI (FP or EP) 111 00+ rrr off	*/
    INT data_mode = fpp_data ;

    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    if (fpp_data == DP)
    {
	LONG cand = (fpp_ac [1] << 12) | fpp_ac [2], plier ;
	w = base + fpp_opadd ;
	plier = ((ULONG) *w << 12) | *(w + 1) ;
	cand = fimul (cand, plier) ;
	fpp_ac [1] = (cand >> 12) & 07777 ;
	fpp_ac [2] = cand & 07777 ;
    }
    else
    {
	fpp_ac [1] = fpp_opadd >> 12 ;
	fpp_ac [2] = fpp_opadd & 07777 ;
	fpp_ac [3] = fpp_ac [4] = fpp_ac [5] = 0 ;
	fpp_stat &= ~BIT9 ;		/* Status: Clear EP mode	*/
	fpp_stat |= BIT0 ;		/* Status: Set DP mode		*/
	fpp_data = DP ;			/* Set DP mode			*/
    }
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 6002 IMUL 25403                  4003 0002 0007 4426
       5403                                  = unnormalized
 12345 7002 IMULI 25403                 4003 0002 0007 4426
       5403                                  = unnormalized
 12345 6002 LEA 25403                   4003 0002 0002 5403
       4523                                  = unnormalized
 12345 7002 LEAI 25403                  4003 0002 0002 5403
       4523                                  = unnormalized
 */
    if (fpp_trace)
    {
	if (data_mode == DP)
	{
	    strcpy (&chrsout [strlen (chrsout)], "IMUL") ;
	    if (fppINST >> 9 == 7)
		strcpy (&chrsout [strlen (chrsout)], "I") ;
	}
	else
	{
	    strcpy (&chrsout [strlen (chrsout)], "LEA") ;
	    if (fppINST >> 9 == 7)
		strcpy (&chrsout [strlen (chrsout)], "I") ;
	}
	sprintf (&chrsout [strlen (chrsout)], " %05o", fpp_opadd) ;
	outfac (FALSE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	MULtiply to Memory - The calculation described above for
 *	FMUL occurs, except that the FAC is not changed.  The result
 *	of the computation is stored at the effective address.
 */
STATIC VOID f7200 (VOID)		/* FMULM   111 01x xxx xxx	*/
{
    apt_stored = FALSE ;
    fpp_opadd = (fpp_br + ((fppINST & 0177) * 3)) & 077777 ;
    w = base + fpp_opadd ;
    fpp_stat |= BIT7 ;			/* Set FMULM in status		*/
    if (fpp_data == DP)
    {
	LONG cand = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *(w + 1) << 12) | *(w + 2) ;
	cand = fimul (cand, plier) ;
	*++w = (cand >> 12) & 07777 , *++w = cand & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) * fpp2d (w), w) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 7201 FMUL BR+1 (25403)           0003 0002 6000 0000 0000 0000 0000
                                             = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FMULM BR+%o (%05o)",
					fppINST & 0177, fpp_opadd) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f7200.
 */
STATIC VOID f7400 (VOID)		/* FMULM   111 10+ rrr msb addr	*/
{
    apt_stored = FALSE ;
    fpp_opadd = ((fppINST & 07) << 12) | *(base + fpp_pc) ;
    w = base + ((fpp_x0 + ((fppINST >> 3) & 07)) & 077777) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    fpp_stat |= BIT7 ;			/* Set FMULM in status		*/
    if (fpp_data == DP)
    {
	LONG cand = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *w << 12) | *(w + 1) ;
	cand = fimul (cand, plier) ;
	*w++ = (cand >> 12) & 07777 ;
	*w = cand & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) * fpp2d (w), w) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 7402 FMULM 25403,0+ (25403) 5631 0003 0002 6000 0000 0000 0000 0000
 12346 5403                                  = -1.(EP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FMULM %o%04o,%o",
		fppINST & 07, *(base + fpp_pc), (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (TRUE) ;
    }
    if (++fpp_pc & 0100000)
	fpp_pc &= 077777 ;
}
/* -------------------------------------------------------------------- */
/*
 *	See comment above for f7200.
 */
STATIC VOID f7600 (VOID)		/* FMULM   111 11+ rrr off	*/
{
    apt_stored = FALSE ;
    w = base + (fpp_br + (3 * (fppINST & 07)) + 1) ;
    fpp_opadd = ((*w & 07) << 12) | *(w + 1) ;
    w = base + (fpp_x0 + ((fppINST >> 3) & 07)) ;
    if (fppINST & BIT5)
	*w = ++*w & 07777 ;
    if (fppINST & 070)
	fpp_opadd = (fpp_opadd + (*w * fpp_data)) & 077777 ;
    w = base + fpp_opadd ;
    fpp_stat |= BIT7 ;			/* Set FMULM in status		*/
    if (fpp_data == DP)
    {
	LONG cand = ((ULONG) fpp_ac [1] << 12) | fpp_ac [2] ;
	LONG plier = ((ULONG) *w << 12) | *(w + 1) ;
	cand = fimul (cand, plier) ;
	*w++ = (cand >> 12) & 07777 ;
	*w = cand & 07777 ;
    }
    else
	storefpp (fpp2d (fpp_ac) * fpp2d (w), w) ;
/*
 *	Output trace information.
 *        1         2         3         4         5         6         7
012345678901234567890123456789012345678901234567890123456789012345678901234
  FPC  CODE FPInstruction          INDX STAT FAC
 12345 7601 FMULM 1,0+ (25403)     4566 0003 0002 6000 0000
                                             = -1.(FP)
 */
    if (fpp_trace)
    {
	sprintf (&chrsout [strlen (chrsout)], "FMULM %o,%o",
				fppINST & 07, (fppINST >> 3) & 07) ;
	if (fppINST & BIT5)
	    strcat (chrsout, "+") ;
	sprintf (&chrsout [strlen (chrsout)], " (%05o)", fpp_opadd) ;
	spaces (35) ;
	sprintf (&chrsout [35], "%04o",
			*(base + fpp_x0 + ((fppINST >> 3) & 07))) ;
	outfac (FALSE) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	FPP-8/A Instruction execution vector table.
 */
STATIC VOID (*ftab [4096]) (VOID) =
{
    f0000, f0001, f0002, f0003, f0004, f0005, f0006, f0007,	/* 0000	*/
    f0010, f0010, f0010, f0010, f0010, f0010, f0010, f0010,
    f0020, f0020, f0020, f0020, f0020, f0020, f0020, f0020,
    f0030, f0030, f0030, f0030, f0030, f0030, f0030, f0030,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0050, f0050, f0050, f0050, f0050, f0050, f0050, f0050,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0100, f0100, f0100, f0100, f0100, f0100, f0100, f0100,	/* 0100 */
    f0110, f0110, f0110, f0110, f0110, f0110, f0110, f0110,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,	/* 0200	*/
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,	/* 0300 */
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0200, f0200, f0200, f0200, f0200, f0200, f0200, f0200,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,	/* 0400	*/
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,	/* 0500	*/
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0400, f0400, f0400, f0400, f0400, f0400, f0400, f0400,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,	/* 0600 */
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,	/* 0700 */
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,
    f0600, f0600, f0600, f0600, f0600, f0600, f0600, f0600,

    f1000, f1000, f1000, f1000, f1000, f1000, f1000, f1000,	/* 1000 */
    f1010, f1010, f1010, f1010, f1010, f1010, f1010, f1010,
    f1020, f1020, f1020, f1020, f1020, f1020, f1020, f1020,
    f1030, f1030, f1030, f1030, f1030, f1030, f1030, f1030,
    f1040, f1040, f1040, f1040, f1040, f1040, f1040, f1040,
    f1050, f1050, f1050, f1050, f1050, f1050, f1050, f1050,
    f1060, f1060, f1060, f1060, f1060, f1060, f1060, f1060,
    f1070, f1070, f1070, f1070, f1070, f1070, f1070, f1070,
    f1100, f1100, f1100, f1100, f1100, f1100, f1100, f1100,	/* 1100 */
    f1110, f1110, f1110, f1110, f1110, f1110, f1110, f1110,
    f1120, f1120, f1120, f1120, f1120, f1120, f1120, f1120,
    f1130, f1130, f1130, f1130, f1130, f1130, f1130, f1130,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f0040, f0040, f0040, f0040, f0040, f0040, f0040, f0040,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,	/* 1200 */
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,	/* 1300 */
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1200, f1200, f1200, f1200, f1200, f1200, f1200, f1200,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,	/* 1400 */
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,	/* 1500 */
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1400, f1400, f1400, f1400, f1400, f1400, f1400, f1400,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,	/* 1600 */
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,	/* 1700 */
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,
    f1600, f1600, f1600, f1600, f1600, f1600, f1600, f1600,

    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,	/* 2000 */
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,	/* 2100 */
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2000, f2000, f2000, f2000, f2000, f2000, f2000, f2000,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,	/* 2200 */
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,	/* 2300 */
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2200, f2200, f2200, f2200, f2200, f2200, f2200, f2200,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,	/* 2400 */
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,	/* 2500 */
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2400, f2400, f2400, f2400, f2400, f2400, f2400, f2400,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,	/* 2600 */
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,	/* 2700 */
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,
    f2600, f2600, f2600, f2600, f2600, f2600, f2600, f2600,

    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,	/* 3000 */
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,	/* 3100 */
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,	/* 3200 */
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,	/* 3300 */
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3200, f3200, f3200, f3200, f3200, f3200, f3200, f3200,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,	/* 3400 */
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,	/* 3500 */
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3400, f3400, f3400, f3400, f3400, f3400, f3400, f3400,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,	/* 3600 */
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,	/* 3700 */
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,
    f3600, f3600, f3600, f3600, f3600, f3600, f3600, f3600,

    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,	/* 4000 */
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,	/* 4100 */
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f3000, f3000, f3000, f3000, f3000, f3000, f3000, f3000,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,	/* 4200 */
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,	/* 4300 */
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4200, f4200, f4200, f4200, f4200, f4200, f4200, f4200,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,	/* 4400 */
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,	/* 4500 */
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4400, f4400, f4400, f4400, f4400, f4400, f4400, f4400,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,	/* 4600 */
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,	/* 4700 */
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,
    f4600, f4600, f4600, f4600, f4600, f4600, f4600, f4600,

    f5000, f5000, f5000, f5000, f5000, f5000, f5000, f5000,	/* 5000 */
    f5010, f5010, f5010, f5010, f5010, f5010, f5010, f5010,
    f5020, f5020, f5020, f5020, f5020, f5020, f5020, f5020,
    f5030, f5030, f5030, f5030, f5030, f5030, f5030, f5030,
    f5040, f5040, f5040, f5040, f5040, f5040, f5040, f5040,
    f5050, f5050, f5050, f5050, f5050, f5050, f5050, f5050,
    f5060, f5060, f5060, f5060, f5060, f5060, f5060, f5060,
    f5070, f5070, f5070, f5070, f5070, f5070, f5070, f5070,
    funim, funim, funim, funim, funim, funim, funim, funim,	/* 5100 */
    funim, funim, funim, funim, funim, funim, funim, funim,
    funim, funim, funim, funim, funim, funim, funim, funim,
    funim, funim, funim, funim, funim, funim, funim, funim,
    funim, funim, funim, funim, funim, funim, funim, funim,
    funim, funim, funim, funim, funim, funim, funim, funim,
    funim, funim, funim, funim, funim, funim, funim, funim,
    funim, funim, funim, funim, funim, funim, funim, funim,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,	/* 5200 */
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,	/* 5300 */
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5200, f5200, f5200, f5200, f5200, f5200, f5200, f5200,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,	/* 5400 */
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,	/* 5500 */
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5400, f5400, f5400, f5400, f5400, f5400, f5400, f5400,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,	/* 5600 */
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,	/* 5700 */
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,
    f5600, f5600, f5600, f5600, f5600, f5600, f5600, f5600,

    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,	/* 6000 */
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,	/* 6100 */
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6000, f6000, f6000, f6000, f6000, f6000, f6000, f6000,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,	/* 6200 */
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,	/* 6300 */
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6200, f6200, f6200, f6200, f6200, f6200, f6200, f6200,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,	/* 6400 */
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,	/* 6500 */
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6400, f6400, f6400, f6400, f6400, f6400, f6400, f6400,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,	/* 6600 */
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,	/* 6700 */
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,
    f6600, f6600, f6600, f6600, f6600, f6600, f6600, f6600,

    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,	/* 7000 */
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,	/* 7100 */
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7000, f7000, f7000, f7000, f7000, f7000, f7000, f7000,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,	/* 7200 */
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,	/* 7300 */
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7200, f7200, f7200, f7200, f7200, f7200, f7200, f7200,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,	/* 7400 */
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,	/* 7500 */
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7400, f7400, f7400, f7400, f7400, f7400, f7400, f7400,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,	/* 7600 */
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,	/* 7700 */
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600,
    f7600, f7600, f7600, f7600, f7600, f7600, f7600, f7600
} ;
/* -------------------------------------------------------------------- */
/*
 *	Execute one or more FPP8-A instructions with impediments.
 */
VOID fppexec1 (VOID)
{
    do
    {
	fpp_stat &= ~BIT7 ;		/* Clear FADDM/FMULM status bit	*/
/*
 *	Fetch the next FPP instruction.
 */
	fppINST = *(base + fpp_pc) ;
/*
 *	Output instruction trace information, if requested.
 */
	if (fpp_trace)
	    sprintf (chrsout, " %05o %04o ", fpp_pc, fppINST) ;
/*
 *	Update the FPP program counter.
 */
	if (++fpp_pc & 0100000)
	    fpp_pc &= 077777 ;
/*
 *	Execute the FPP instruction.
 */
	(*ftab [fppINST]) () ;
/*
 *	Count FPP-8/A instructions executed, if requested.
 */
	if (fpp_count)
	    fpp_count++ ;
/*
 *	If tracing, count the traced instructions.
 */
	if (fpp_trace && --trace <= 0)
	    run = trace = FALSE ;
/*
 *	Continue until stopped by an FPP halt, FPP interleaved mode,
 *	CPU interrupt, FPP stop, or CPU halt.
 */
    } while (fpp_mode == LOCK
		&& !(int_mask & io_flags) && fpp_run && run) ;
    if (!apt_stored)
	store_apt () ;			/* Save APT info		*/
}
/* -------------------------------------------------------------------- */
/*
 *	Execute one or more FPP8-A instructions with
 *	minimum impediments.
 */
VOID fppexec2 (VOID)
{
    do
    {
	fpp_stat &= ~BIT7 ;		/* Clear FADDM/FMULM status bit	*/
/*
 *	Fetch the next FPP instruction.
 */
	fppINST = *(base + fpp_pc) ;
/*
 *	Update the FPP program counter.
 */
	if (++fpp_pc & 0100000)
	    fpp_pc &= 077777 ;
/*
 *	Execute the FPP instruction.
 */
	(*ftab [fppINST]) () ;
/*
 *	Continue until stopped by an FPP halt, FPP interleaved mode,
 *	CPU interrupt, FPP stop, or CPU halt.
 */
    } while (fpp_mode == LOCK
		&& !(int_mask & io_flags) && fpp_run && run) ;
    if (!apt_stored)
	store_apt () ;			/* Save APT info		*/
}
/* -------------------------------------------------------------------- */
