/*	dr11.c		M02	Jun 1 80 */
/*
 * DR11B development driver
 *
 *	This driver is intended for use during development of an interface
 *	based on a dr11b
 *
 *	It permits unusual latitude in allowing processes access to the
 *	device registers (etc) - to permit user level routines to test
 *	& debug the external hardware interfaced via the dr11b, and to
 *	permit the eventual device driver to be developed initially as
 *	a user process.
 *
 *	There are two minor devices (zero & one - or actually even & odd)
 *	Minor dev 0 is the usual interface to the driver, & permits only
 *	single access (only open once)
 *	Minor dev 1 is for special purposes (like monitoring current status,
 *	or examining the hardware when the test prog clags up)
 *	(There are NO access restrictions on minor 1)
 *
 *	It is recommended that the special file(s) giving access to this
 *	driver be highly protected
 *
 *			Robert Elz
 *			Computer Science
 *			Melbourne University
 *
 */

struct dr11regs {
	short	drwc;		/* word count, r/w, no byte ops */
	short	drba;		/* bus address, r/w, no byte ops */
	short	drst;		/* cmd / status, r/w, byte ops OK */
	short	drdb;		/* data buffer, two regs, r & w, no byte ops */
};

/*
 *	Interrupts:
 *
 * If IE is set, then interrupt when READY sets
 *
 * READY sets when ERROR sets, or by drwc overflowing from -1 to 0
 *		(or by INIT which clears IE - no interrupt)
 *	- cleared by GO
 *
 * ERROR sets when user device signals ATTN, NEX sets as result of bus timeout
 *		or when drba increments from 0xfffe to 0 (nb: the XBA bits are
 *		NOT incremented) - also sets if test board not in suitable slot
 *	- cleared by clearing NEX (write a 0 at it) loading drba, or removing
 *	  the error condition (user dev negates ATTN, or boards put back)
 *
 * maintanence:
 *	by plugging the maintanence board into the user slots the following
 *	connections are made (with the noted effects)
 *
 * data out to data in (obvious)
 * fnct3 to dstatA (useless) & to single cycle (if fnct3 is set, dr11b operates
 *	the bus in burst mode, transferring data continuously w/o releasing
 *	the bus till either fnct3 is reset or drwc overflows, or error)
 * fnct2 to dstatB (useless)
 * fnct1 to dstatC (useless) & to C1 control (when fnct1 is set, dr11 transmits
 *	data to slave, when reset, it accepts data)
 * ground to C0 control (no byte operations)
 * ground to ATTN (no device errors)
 * high to drba increment enable (address always increments)
 * high to drwc increment enable (word count always increments)
 * ground to A00 (no odd (byte) addresses)
 * end cycle to cycle request A (the end of one transfer auto starts the next)
 * GO to cycle request B (setting GO starts DMA immediately)
 *
 *	the MAINT bit causes the following when set
 *
 * the fnct[1-3] bits become a cyclic (0 - 7) counter
 *	(if test board is in user slots, then the alternation of fnct1
 *	 caused in turn causes successive cycles to alternate between
 *	 read & write - if fnct1 == 0 initially, a h/w is read then
 *	 written back in next locn, then the next h/w is read ...
 *	 till drwc overflows.
 *	 Also: fnct3 switches state every 4 cycles, causing 4 transfers
 *	 in non-burst mode, then 4 in burst mode ... (or vice versa)  )
 */

#include "../h/param.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/pte.h"
#include "../h/uba.h"
#include "../h/buf.h"
#include "../h/mtpr.h"
#include "../h/dr11.h"

#define	DRSPL	spl5
#define	DRPRI	19
#define	DRPPRI	(PZERO+3)
#define	DRADDR	((struct dr11regs *)(UBA0_DEV + 0172410))


struct dr11 {
	short	dr_state;
	short	dr_drst;
	long	dr_drba;
	short	dr_drwc;
	short	dr_flags;
	long	dr_time;
} dr11;

dr11open(dev, flag)
{
	if (minor(dev)&01)
		return;
	if (dr11.dr_state & DROPEN) {
		u.u_error = EBUSY;
		return;
	}
	dr11.dr_state |= DROPEN;
}

dr11close(dev, flag)
{
	if (minor(dev)&01)
		return;
	dr11.dr_state &= ~DROPEN;
}

dr11strat(bp)
register struct buf *bp;
{
	register i, j;

	i = ubasetup(bp, dr11.dr_flags&DRBDP);
	j = i & 0x3ffff;
		/* this is horrible, but will do if the dr11 is the only
		   dma device on the unibus (if not progs should sleep &
		   retry after an ENOENT error to an i/o sys call)
		 */
	if (j + bp->b_bcount >= 0x10000) {
		ubafree(i);
		bp->b_error = ENOENT;
		bp->b_flags |= B_DONE|B_ERROR;
		return;
	}
	VOID DRSPL();
	dr11.dr_state |= DRIO;
	DRADDR->drwc = -(bp->b_bcount >> 1);
	DRADDR->drba = j;
	dr11.dr_time = mfpr(ICR);
	DRADDR->drst = (dr11.dr_drst&~(XBA16|XBA17)) | GO|IE;
	sleep((caddr_t)&dr11, DRPRI);
	VOID spl0();
	dr11.dr_drba += (int)u.u_base - j;
	ubafree(i);
	dr11.dr_state &= ~DRIO;
	bp->b_resid = -dr11.dr_drwc << 1;
	bp->b_flags |= B_DONE;
}

struct buf dr11_b;

/*
 * this fn should be entered in cdevsw as both the read & write interface
 */
dr11xfer(dev)		/* read or write, we don't care */
{
	if (dr11.dr_state & DRIO) {
		u.u_error = EBUSY;
		return;
	}
		/* the following test is not essential & could be deleted */
	if ( (u.u_count >> 1) != (- dr11.dr_drwc)
	   || ((int)u.u_base != dr11.dr_drba)	 ) {
		u.u_error = EINVAL;
		return;
	}
	physio(dr11strat, &dr11_b, dev, B_WRITE, minphys);
}

dr11intr(d)
{
	if ((dr11.dr_state & DRIO) == 0) {
		if (dr11.dr_state & DRWINTR)
			wakeup((caddr_t) &dr11);
		else
			dr11.dr_state |= DRINTR;
		return;
	}
	dr11.dr_time = mfpr(ICR)-dr11.dr_time;
	dr11.dr_drst = DRADDR->drst;
	dr11.dr_drwc = DRADDR->drwc;
	dr11.dr_drba = DRADDR->drba;
	wakeup((caddr_t) &dr11);
}

dr11ioctl(dev, cmd, addr, flag)
caddr_t addr;
{
	long	data;
	struct dr_iocb iocb;

	switch (cmd) {

	case SETDRWC:
	case SETDRBA:
	case SETDRDB:
	case SETDRST:
		if (copyin(addr, &data, sizeof(data))) {
			u.u_error = EFAULT;
			break;
		}
		switch (cmd) {
		case SETDRWC:
			DRADDR->drwc = data;
			break;
		case SETDRBA:
			DRADDR->drba = data;
			break;
		case SETDRDB:
			DRADDR->drdb = data;
			break;
		case SETDRST:
			DRADDR->drst = data&~GO;
			break;
		}
		break;

	case GETDRWC:
		data = DRADDR->drwc;
		goto xmit;
	case GETDRBA:
		data = DRADDR->drba;
		goto xmit;
	case GETDRDB:
		data = DRADDR->drdb;
		goto xmit;
	case GETDRST:
		data = DRADDR->drst;
	  xmit:
		if (copyout(&data, addr, sizeof(data)))
			u.u_error = EFAULT;
		break;

	case GETTIME:
		data = dr11.dr_time;
		goto xmit;

	case DR11SET:
		if (copyin(addr, &iocb, sizeof(iocb))) {
			u.u_error = EFAULT;
			break;
		}
		dr11.dr_flags = iocb.sg_flags;
		dr11.dr_drst = iocb.sg_drst;
		dr11.dr_drba = iocb.sg_drba;
		dr11.dr_drwc = iocb.sg_drwc;
		break;

	case DR11GET:
		iocb.sg_flags = dr11.dr_flags;
		iocb.sg_state = dr11.dr_state;
		iocb.sg_drst = dr11.dr_drst;
		iocb.sg_drba = dr11.dr_drba;
		iocb.sg_drwc = dr11.dr_drwc;
		iocb.sg_tbase = mfpr(NICR);
		if (copyout(&iocb, addr, sizeof(iocb)))
			u.u_error = EFAULT;
		break;

	case WAKEDR11:
		wakeup((caddr_t) &dr11);
		break;

	case WAITINTR:
		VOID DRSPL();
		if ((dr11.dr_state & DRINTR) == 0) {
			dr11.dr_state |= DRWINTR;
			sleep((caddr_t)&dr11, DRPPRI);
		}
		dr11.dr_state &= ~(DRINTR|DRWINTR);
		spl0();
		return;

	default:
		u.u_error = ENOTTY;
		break;
	}
}
