/*	tty.c	2.1	May 25 80	M05	*/

#
/*
 * general TTY subroutines
 */
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/tty.h"
#include "../h/proc.h"
#include "../h/mx.h"
#include "../h/inode.h"
#include "../h/file.h"
#include "../h/reg.h"
#include "../h/conf.h"
#include "../h/buf.h"

char	partab[];


#if	O_LCASE
/*
 * Input mapping table-- if an entry is non-zero, when the
 * corresponding character is typed preceded by "\" the escape
 * sequence is replaced by the table value.  Mostly used for
 * upper-case only terminals.
 */

char	maptab[] ={
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,'|',000,000,000,000,000,'`',
	'{','}',000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,'~',000,
	000,'A','B','C','D','E','F','G',
	'H','I','J','K','L','M','N','O',
	'P','Q','R','S','T','U','V','W',
	'X','Y','Z',000,000,000,000,000,
};
#endif


/*
 * shorthand
 */
#define	q1	tp->t_rawq
#define	q2	tp->t_canq
#define	q3	tp->t_outq
#define	q4	tp->t_un.t_ctlq
#define	MIN(a,b)	((a<b)?a:b)

#define	OBUFSIZ	100
#define	MAXDELAY	(5 * HZ)	/* max delay thru TIOCDELAY */

/*
 * routine called on first teletype open.
 * establishes a process group for distribution
 * of quits and interrupts from the tty.
 */
ttyopen(dev, tp)
dev_t dev;
register struct tty *tp;
{
	register struct proc *pp;

	pp = u.u_procp;
	tp->t_dev = dev;
	if(pp->p_pgrp == 0) {
		u.u_ttyp = tp;
		u.u_ttyd = dev;
		if (tp->t_pgrp==0)
			tp->t_pgrp = pp->p_pid;
		pp->p_pgrp = tp->t_pgrp;
	}
	tp->t_state &= ~WOPEN;
	tp->t_state |= ISOPEN;
}


/*
 * set default control characters.
 */
ttychars(tp)
register struct tty *tp;
{
	tun.t_intrc = CINTR;
	tun.t_quitc = CQUIT;
	tun.t_startc = CSTART;
	tun.t_stopc = CSTOP;
	tun.t_eofc = CEOT;
	tun.t_brkc = CBRK;
	tp->t_erase = CERASE;
	tp->t_kill = CKILL;
#if	MELB_TTY
	tp->t_escp = CESCAPE;
	tun.t_startc = CSTART;
	tun.t_stopc = CSTOP;
	tp->t_redisp = CREDISP;
	tp->t_delq = CDELQ;
	tp->t_disc = CDISC;
	tp->t_dqueue = CDSPQ;
#if	O_PAGE
	tp->t_pgon = CPAGON;
	tp->t_pgoff = CPAGOFF;
#endif
	tp->t_desc = '\\';
	tp->t_delet = '#';
	tp->t_brkin = CINTR;
	tp->t_dctl = 0xf7ffc97f;
	if (tp->t_state & HUPCLS) {
		tp->t_type = UNKNOWN;
		tp->t_width = 0;
		tp->t_depth = 0;
		tp->t_nldel = 0;
		tp->t_crdel = 0;
		tp->t_tabdel = 0;
		tp->t_ffdel = 0;
		tp->t_dchar = 0;
		tp->t_ddel = 0;
#if	O_PAGE
		tp->t_pgflg = 0;
		tp->t_pgdel = 0;
		tp->t_pglen = 0;
#endif
	}
#endif
}

/*
 * clean tp on last close
 */
ttyclose(tp)
register struct tty *tp;
{

	tp->t_pgrp = 0;
	wflushtty(tp);
#if	! MELB_TTY
	tp->t_state = 0;
#else
	tp->t_state &= HUPCLS;
/**D**
	tp->t_xtra = 0;
 **D**/	tp->t_xtra &= 04000;		/** leave the DTR image bit alone **/
#endif
}

/*
 * stty/gtty writearound
 */
stty()
{
	u.u_arg[2] = u.u_arg[1];
	u.u_arg[1] = TIOCSETP;
	ioctl();
}

gtty()
{
	u.u_arg[2] = u.u_arg[1];
	u.u_arg[1] = TIOCGETP;
	ioctl();
}

/*
 * ioctl system call
 * Check legality, execute common code, and switch out to individual
 * device routine.
 */
ioctl()
{
	register struct file *fp;
	register struct inode *ip;
	register struct a {
		int	fdes;
		int	cmd;
		caddr_t	cmarg;
	} *uap;
	register dev_t dev;
	register fmt;

	uap = (struct a *)u.u_ap;
	if ((fp = getf(uap->fdes)) == NULL)
		return;
	if (uap->cmd==FIOCLEX) {
		u.u_pofile[uap->fdes] |= EXCLOSE;
		return;
	}
	if (uap->cmd==FIONCLEX) {
		u.u_pofile[uap->fdes] &= ~EXCLOSE;
		return;
	}
	ip = fp->f_inode;
	fmt = ip->i_mode & IFMT;
	if (fmt != IFCHR && fmt != IFMPC) {
		u.u_error = ENOTTY;
		return;
	}
	dev = ip->i_un.i_rdev;
	u.u_r.r_val1 = 0;
	(*cdevsw[major(dev)].d_ioctl)(dev, uap->cmd, uap->cmarg, fp->f_flag);
}

/*
 * Common code for several tty ioctl commands
 */
ttioccomm(com, tp, addr, dev)
register struct tty *tp;
caddr_t addr;
{
	unsigned t;
#if	! MELB_TTY
	struct ttiocb iocb;
#else
	union {
		struct ttiocb1 iocb1;
		struct ttiocb5 iocb5;
	} iocb;
#endif
	extern int nldisp;

	switch(com) {

	/*
	 * get discipline number
	 */
	case TIOCGETD:
		t = tp->t_line;
		if (copyout((caddr_t)&t, addr, sizeof(t)))
			u.u_error = EFAULT;
		break;

	/*
	 * set line discipline
	 */
	case TIOCSETD:
		if (copyin(addr, (caddr_t)&t, sizeof(t))) {
			u.u_error = EFAULT;
			break;
		}
		if (t >= nldisp) {
			u.u_error = ENXIO;
			break;
		}
		if (tp->t_line)
			(*linesw[tp->t_line].l_close)(tp);
		if (t)
			(*linesw[t].l_open)(dev, tp, addr);
		if (u.u_error==0)
			tp->t_line = t;
		break;

	/*
	 * prevent more opens on channel
	 */
	case TIOCEXCL:
		tp->t_state |= XCLUDE;
		break;

	case TIOCNXCL:
		tp->t_state &= ~XCLUDE;
		break;

	/*
	 * Set new parameters
	 */
	case TIOCSETP:
		wflushtty(tp);
	case TIOCSETN:
		if (copyin(addr, (caddr_t)&iocb, sizeof(iocb))) {
			u.u_error = EFAULT;
			return(1);
		}
		VOID spl5();
#if	! MELB_TTY
		while (canon(tp)>=0) 
			;
#endif
		if ((tp->t_state&SPEEDS)==0) {
#if	! MELB_TTY
			tp->t_ispeed = iocb.ioc_ispeed;
			tp->t_ospeed = iocb.ioc_ospeed;
#else
			tp->t_ispeed = iocb.iocb1.ioc_ispeed;
			tp->t_ospeed = iocb.iocb1.ioc_ospeed;
#endif
		}
#if	! MELB_TTY
		tp->t_erase = iocb.ioc_erase;
		tp->t_kill = iocb.ioc_kill;
		tp->t_flags = iocb.ioc_flags;
#else
		tp->t_erase = iocb.iocb1.ioc_erase;
		tp->t_kill = iocb.iocb1.ioc_kill;
		tp->t_flags = iocb.iocb1.ioc_flags;
#endif
		VOID spl0();
		break;

	/*
	 * send current parameters to user
	 */
	case TIOCGETP:
#if	! MELB_TTY
		iocb.ioc_ispeed = tp->t_ispeed;
		iocb.ioc_ospeed = tp->t_ospeed;
		iocb.ioc_erase = tp->t_erase;
		iocb.ioc_kill = tp->t_kill;
		iocb.ioc_flags = tp->t_flags;
#else
		iocb.iocb1.ioc_ispeed = tp->t_ispeed;
		iocb.iocb1.ioc_ospeed = tp->t_ospeed;
		iocb.iocb1.ioc_erase = tp->t_erase;
		iocb.iocb1.ioc_kill = tp->t_kill;
		iocb.iocb1.ioc_flags = tp->t_flags;
#endif
		if (copyout((caddr_t)&iocb, addr, sizeof(iocb)))
			u.u_error = EFAULT;
		break;

	/*
	 * Hang up line on last close
	 */

	case TIOCHPCL:
		tp->t_state |= HUPCLS;
		break;

	case TIOCFLUSH:
		flushtty(tp);
		break;

	/*
	 * ioctl entries to line discipline
	 */
	case DIOCSETP:
	case DIOCGETP:
		(*linesw[tp->t_line].l_ioctl)(com, tp, addr);
		break;

	/*
	 * set and fetch special characters
	 */
	case TIOCSETC:
#if	! MELB_TTY
		if (copyin(addr, (caddr_t)&tun, sizeof(struct tc)))
			u.u_error = EFAULT;
#else
		if (copyin(addr, (caddr_t)&tun, sizeof(struct ttiocb2)))
			u.u_error = EFAULT;
#endif
		break;

	case TIOCGETC:
#if	! MELB_TTY
		if (copyout((caddr_t)&tun, addr, sizeof(struct tc)))
			u.u_error = EFAULT;
#else
		if (copyout((caddr_t)&tun, addr, sizeof(struct ttiocb2)))
			u.u_error = EFAULT;
#endif
		break;

#if	MELB_TTY
	/*
	 * set / fetch some more magic characters
	 */
	case TIOCSCHR:
		if (copyin(addr, (caddr_t)&tp->t_ch, sizeof(struct ttiocb3)))
			u.u_error = EFAULT;
		break;

	case TIOCGCHR:
		if (copyout((caddr_t)&tp->t_ch, addr, sizeof(struct ttiocb3)))
			u.u_error = EFAULT;
		break;

	/*
	 * set / get delays
	 */
	case TIOCSDEL:
		if (copyin(addr, (caddr_t)&tp->t_del, sizeof(struct ttiocb4)))
			u.u_error = EFAULT;
		break;

	case TIOCGDEL:
		if (copyout((caddr_t)&tp->t_del, addr, sizeof(struct ttiocb4)))
			u.u_error = EFAULT;
		break;

	/*
	 * set / get control character echo info
	 */
	case TIOCSCTL:
		if (copyin(addr, (caddr_t)&tp->t_ctl, sizeof(struct ttiocb6)))
			u.u_error = EFAULT;
		break;

	case TIOCGCTL:
		if (copyout((caddr_t)&tp->t_ctl, addr, sizeof(struct ttiocb6)))
			u.u_error = EFAULT;
		break;

	/*
	 * set / get state info
	 */
	case TIOCSETS:
		if (copyin(addr, (caddr_t)&iocb, sizeof(struct ttiocb5))) {
			u.u_error = EFAULT;
			break;
		}
		tp->t_type = iocb.iocb5.ioc_ttype;
		tp->t_width = iocb.iocb5.ioc_width;
		tp->t_depth = iocb.iocb5.ioc_depth;
		break;

	case TIOCGETS:
		iocb.iocb5.ioc_state = tp->t_state;
		iocb.iocb5.ioc_ttype = tp->t_type;
		iocb.iocb5.ioc_width = tp->t_width;
		iocb.iocb5.ioc_depth = tp->t_depth;
		if (copyout((caddr_t)&iocb, addr, sizeof(struct ttiocb5)))
			u.u_error = EFAULT;
		break;

	/*
	 * reset DISCARD flag
	 */
	case TIOCRSTO:
		tp->t_xtra &= ~DISCARDING;
		break;

	/*
	 * test to see if input queue(s) is(are) empty
	 */
	case TIOCMPTY:
		t = tp->t_flags&(RAW|CBREAK)
			? tp->t_canq.c_cc == 0
			: tp->t_delct == 0	;
		if (copyout((caddr_t)&t, addr, sizeof(t)))
			u.u_error = EFAULT;
		break;

	/*
	 * place a delay in output stream (arg is delay in ms)
	 */
	case TIOCDELAY:
		if (copyin(addr, (caddr_t)&t, sizeof(t))) {
			u.u_error = EFAULT;
			break;
		}
		if (tp->t_flags & RAW) {
			u.u_error = EINVAL;
			break;
		}
		t *= HZ;
		t /= 1000;
		if (t > MAXDELAY)
			t = MAXDELAY;
		VOID spl5();
		while (tp->t_outq.c_cc >= TTHIWAT || tp->t_xtra&(PG_PEND|PG_WAIT|TTY_BRK)) {
			tp->t_state |= ASLEEP;
			sleep((caddr_t)&tp->t_outq, TTOPRI-1);
		}
		VOID spl0();
		while (t >= 0200) {
			putc(0377, &tp->t_outq);
			t -= 0177;
		}
		if ( t )
			putc(t | 0200, &tp->t_outq);
		break;

#if	O_PAGE
	/*
	 * set / get page mode control data
	 */
	case TIOCSPAG:
		if (copyin(addr, (caddr_t)&tp->t_pag, sizeof(struct ttiocb7)))
			u.u_error = EFAULT;
		else
			tp->t_pgcnt = tp->t_pglen;
		break;

	case TIOCGPAG:
		if (copyout((caddr_t)&tp->t_pag, addr, sizeof(struct ttiocb7)))
			u.u_error = EFAULT;
		break;
#endif
#endif

	default:
		return(0);
	}
	return(1);
}

/*
 * Wait for output to drain, then flush input waiting.
 */
wflushtty(tp)
register struct tty *tp;
{

	VOID spl5();
	while (tp->t_outq.c_cc && tp->t_state&CARR_ON) {
		(*tp->t_oproc)(tp);
		tp->t_state |= ASLEEP;
		sleep((caddr_t)&tp->t_outq, TTOPRI);
	}
	flushtty(tp);
	VOID spl0();
}

/*
 * flush all TTY queues
 */
flushtty(tp)
register struct tty *tp;
{
	register s;

#if	! MELB_TTY
	while (getc(&tp->t_canq) >= 0)
		;
	wakeup((caddr_t)&tp->t_rawq);
#else
	wakeup((caddr_t)&tp->t_canq);
#endif
	wakeup((caddr_t)&tp->t_outq);
	s = spl6();
	tp->t_state &= ~TTSTOP;
#if	MELB_TTY
	tp->t_xtra &= ~(KILL_PEND|ESC_PEND|RO_MODE|DISCARDING|PG_PEND|PG_WAIT|PG_CONT);
#endif
	(*cdevsw[major(tp->t_dev)].d_stop)(tp);
	while (getc(&tp->t_outq) >= 0)
		;
	while (getc(&tp->t_rawq) >= 0)
		;
#if	MELB_TTY
	while (getc(&tp->t_canq) >= 0)
		;
	tp->t_nlpos = NULL;
#if	O_PAGE
	tp->t_pgcnt = tp->t_pglen;
#endif
#endif
	tp->t_delct = 0;
	splx(s);
}



#if	! MELB_TTY
/*
 * transfer raw input list to canonical list,
 * doing erase-kill processing and handling escapes.
 * It waits until a full line has been typed in cooked mode,
 * or until any character has been typed in raw mode.
 */
canon(tp)
register struct tty *tp;
{
	register char *bp;
	char *bp1;
	register int c;
	int mc;
	int s;

	if ((tp->t_flags&(RAW|CBREAK))==0 && tp->t_delct==0
	    || (tp->t_flags&(RAW|CBREAK))!=0 && tp->t_rawq.c_cc==0) {
		return(-1);
	}
	s = spl0();
loop:
	bp = &canonb[2];
	while ((c=getc(&tp->t_rawq)) >= 0) {
		if ((tp->t_flags&(RAW|CBREAK))==0) {
			if (c==0377) {
				tp->t_delct--;
				break;
			}
			if (bp[-1]!='\\') {
				if (c==tp->t_erase) {
					if (bp > &canonb[2])
						bp--;
					continue;
				}
				if (c==tp->t_kill)
					goto loop;
				if (c==tun.t_eofc)
					continue;
			} else {
				mc = maptab[c];
				if (c==tp->t_erase || c==tp->t_kill)
					mc = c;
				if (mc && (mc==c || (tp->t_flags&LCASE))) {
					if (bp[-2] != '\\')
						c = mc;
					bp--;
				}
			}
		}
		*bp++ = c;
		if (bp>=canonb+CANBSIZ)
			break;
	}
	bp1 = &canonb[2];
	VOID b_to_q(bp1, bp-bp1, &tp->t_canq);

	if (tp->t_state&TBLOCK && tp->t_rawq.c_cc < TTYHOG/5) {
		if (putc(tun.t_startc, &tp->t_outq)==0) {
			tp->t_state &= ~TBLOCK;
			ttstart(tp);
		}
		tp->t_char = 0;
	}

	splx(s);
	return(0);
}
#endif


/*
 * block transfer input handler.
 */
ttyrend(tp, pb, pe)
register struct tty *tp;
register char *pb, *pe;
{
	int	tandem;

	tandem = tp->t_flags&TANDEM;
	if (tp->t_flags&RAW) {
		VOID b_to_q(pb, pe-pb, &tp->t_rawq);
		if (tp->t_chan)
			VOID sdata(tp->t_chan); else
			wakeup((caddr_t)&tp->t_rawq);
	} else {
		tp->t_flags &= ~TANDEM;
		while (pb < pe)
			ttyinput(*pb++, tp);
		tp->t_flags |= tandem;
	}
	if (tandem)
		ttyblock(tp);
}

/*
 * Place a character on raw TTY input queue, putting in delimiters
 * and waking up top half as needed.
 * Also echo if required.
 * The arguments are the character and the appropriate
 * tty structure.
 */
#if	! MELB_TTY
ttyinput(c, tp)
register c;
register struct tty *tp;
{
	register int t_flags;
	register struct chan *cp;

	tk_nin += 1;
	c &= 0377;
	t_flags = tp->t_flags;
	if (t_flags&TANDEM)
		ttyblock(tp);
	if ((t_flags&RAW)==0) {
		c &= 0177;
		if (tp->t_state&TTSTOP) {
			if (c==tun.t_startc) {
				tp->t_state &= ~TTSTOP;
				ttstart(tp);
				return;
			}
			if (c==tun.t_stopc)
				return;
			tp->t_state &= ~TTSTOP;
			ttstart(tp);
		} else {
			if (c==tun.t_stopc) {
				tp->t_state |= TTSTOP;
				(*cdevsw[major(tp->t_dev)].d_stop)(tp);
				return;
			}
			if (c==tun.t_startc)
				return;
		}
		if (c==tun.t_quitc || c==tun.t_intrc) {
			flushtty(tp);
			c = (c==tun.t_intrc) ? SIGINT:SIGQUIT;
			if (tp->t_chan)
				scontrol(tp->t_chan, M_SIG, c);
			else
				signal(tp->t_pgrp, c);
			return;
		}
		if (c=='\r' && t_flags&CRMOD)
			c = '\n';
	}
	if (tp->t_rawq.c_cc>TTYHOG) {
		flushtty(tp);
		return;
	}
	if (t_flags&LCASE && c>='A' && c<='Z')
		c += 'a'-'A';
	VOID putc(c, &tp->t_rawq);
	if (t_flags&(RAW|CBREAK)||(c=='\n'||c==tun.t_eofc||c==tun.t_brkc)) {
		if ((t_flags&(RAW|CBREAK))==0 && putc(0377, &tp->t_rawq)==0)
			tp->t_delct++;
		if ((cp=tp->t_chan)!=NULL)
			VOID sdata(cp); else
			wakeup((caddr_t)&tp->t_rawq);
	}
	if (t_flags&ECHO) {
		ttyoutput(c, tp);
		if (c==tp->t_kill && (t_flags&(RAW|CBREAK))==0)
			ttyoutput('\n', tp);
		ttstart(tp);
	}
}
#else

ttyinput(c, tp)
register c;
register struct tty *tp;
{
	register flags = tp->t_flags;
	register struct chan *cp;

	tk_nin++;
	/*
	 * in non-raw mode (also non-fast, or we wouldn't be here)
	 * strip character to 7 bits and check for interrupt/quit.
	 * note: if interrupt & quit are both the same char, quit
	 * will be signalled. (this can be used to disable the interrupt
	 * character if 'break' is to be used instead)
	 */
	if(!(flags & RAW)) {
		c &= 0177;
		if (c == tun.t_quitc || c == tun.t_intrc) {
			flushtty(tp);
			c = (c!=tun.t_quitc) ? SIGINT:SIGQUIT;
			if (tp->t_chan)
				scontrol(tp->t_chan, M_SIG, c);
			else
				signal(tp->t_pgrp, c);
			return;
		}
		if (c == '\r' && flags&CRMOD)
			c = '\n';
	} else
		c &= 0377;

#if	O_PAGE
	/*
	 * If output is stopped at a page boundary, we might start it again now
	 */
	if (tp->t_xtra&PG_WAIT && ( !(tp->t_pgflg&PG_NLGO) || c == tun.t_startc)) {
		tp->t_xtra &= ~(PG_WAIT|PG_PEND);
		if (tp->t_pgflg & PG_CLEAR) {
			tp->t_xtra |= PG_IN;
			ttyoutput('\f', tp);
		}
		ttstart(tp);
		/*
		 * Should we return here or not, or optionally, or depending
		 * on the state of typeahead ???
		 */
		return;
	}
	/*
	 * If the user types something when output is being printed,
	 * what effect should that have on a pending page halt ?
	 * For now - nothing, if the following lines are included,
	 * any typeahead while printing will cause the page wait to
	 * be deleted (but note: typeahead before the page wait is
	 * encountered won't have that effect)
	 *

	if (tp->t_xtra&PG_PEND && !(tp->t_pgflg&PG_NLGO))
		tp->t_xtra |= PG_CONT;

	 */
#endif

	/*
	 * if we are talking to some intelligent being, and we are
	 * approaching the limit of our endurance, let it know, so
	 * it can be more patient with us
	 */
	if (flags&TANDEM)
		ttyblock(tp);

	/*
	 * if the last char was not the escape char, then all kinds
	 * of fancy transformations are possible. But they
	 * don't work in raw mode, and most don't in cbreak mode either
	 */
	if(!(tp->t_xtra&ESC_PEND) && !(flags&RAW)) {
		/*
		 * check for START & STOP commands
		 * - halt or restart output as appropriate
		 */
		if (tp->t_state&TTSTOP) {
			if (c==tun.t_startc) {
				tp->t_state &= ~TTSTOP;
				ttstart(tp);
				return;
			}
			if (c==tun.t_stopc)
				return;
			if (!(flags&HOLDSTOP)) {
				tp->t_state &= ~TTSTOP;
				ttstart(tp);
			}
		} else {
			if (c==tun.t_stopc) {
				tp->t_state |= TTSTOP;
				(*cdevsw[major(tp->t_dev)].d_stop)(tp);
				return;
			}
			if (c==tun.t_startc)
				return;
		}
#if	O_PAGE
		/*
		 * If we have a page mode on/off command - do it
		 */
		if (c == tp->t_pgon || c == tp->t_pgoff) {
			if (c == tp->t_pgon)
				if (c == tp->t_pgoff && tp->t_pglen)
					tp->t_pglen = 0;
				else
					tp->t_pglen = tp->t_depth - 1;
			else
				tp->t_pglen = 0;
			tp->t_pgcnt = tp->t_pglen;
			if (flags & ECHO) {
				echo(c, tp);
				if (tp->t_col)
					ttyoutput('\n', tp);
			}
			if (tp->t_xtra & PG_WAIT) {
				tp->t_xtra &= ~(PG_WAIT|PG_PEND);
				if (tp->t_pgflg & PG_CLEAR) {
					tp->t_xtra |= PG_IN;
					ttyoutput('\f', tp);
				}
			}
			ttstart(tp);
			return;
		}
#endif
		if(!(flags&CBREAK)) {

			/*
			 * if the char is erase, delete the previous
			 * char typed. If that was a kill, it is easy,
			 * just forget we saw it. If it wasn't then extract
			 * it from the buffer (allowing for underflow)
			 * but do nothing at all if there is nothing in the line
			 */
			if(c == tp->t_erase) {
				if(tp->t_xtra&KILL_PEND) {
					tp->t_xtra &= ~KILL_PEND;
					if(flags&KNL) {
						if (flags & ECHO) {
							redispl(tp, tp->t_nlpos);
							ttstart(tp);
						}
						return;
					} else
						c = tp->t_kill;
				} else {
					if (tp->t_canq.c_cf == NULL
						|| tp->t_canq.c_cl == tp->t_nlpos
						|| (c = delc(&tp->t_canq)) < 0)
							return;
				}

				/*
				 * now we let the user know we have obeyed.
				 * This is done in one of three ways.
				 * on a VDU, the char typed (hopefully) is
				 * made to vanish!!
				 * on other terminals either the erase char
				 * is just echoed back, or the char deleted
				 * is echoed. In the latter case, a sequence of
				 * rubbed out characters will be bounded by
				 * a pair of rub-out characters
				 */
				if(flags&ECHO) {
					switch (flags & RO_TYPE) {
					case RO_VDU:
					       del2:
						ttyoutput('\b', tp);
						ttyoutput(' ', tp);
						ttyoutput('\b', tp);
						if (c < ' ' && (tp->t_dctl&(1<<c))) {
							c = ' ';
							goto del2;
						}
						break;
					case RO_HARD:
						if(!(tp->t_xtra&RO_MODE)) {
							tp->t_xtra |= RO_MODE;
							echo(0177, tp);
						}
						echo(c, tp);
						break;
					default:
						c = tp->t_erase;
						echo(c, tp);
						break;
					}
					ttstart(tp);
				}
				return;
			}

			/*
			 * if the last char was an erase, and we are echoing
			 * the chars deleted, it is now time to add the closing
			 * delimiter (the seq of erases has finished.)
			 */
			if(tp->t_xtra&RO_MODE) {
				tp->t_xtra &= ~RO_MODE;
				if(flags&ECHO) {
					echo(0177, tp);
				}
			}
			/*
			 * if we have a redisplay character, re-echo the
			 * whole of the current input line back at the user
			 * take a new line first if not at the left margin
			 */
			if (c == tp->t_redisp || c == tp->t_dqueue) {
				if(flags&ECHO) {
#if	O_PAGE
					tp->t_xtra |= PG_IN;
#endif
					if(tp->t_col)
						ttyoutput('\n', tp);
					redispl(tp, c == tp->t_redispl ?
						tp->t_nlpos : tp->t_canq.c_cf);
					if(tp->t_xtra&KILL_PEND) {
						echo(tp->t_kill, tp);
						if(flags&KNL)
							ttyoutput('\n', tp);
					}
					ttstart(tp);
				}
				return;
			}

			/*
			 * if the char requests us to delete the input
			 * queue, it is done next, NB: this is done at
			 * once, cannot be undone, we do it before handling
			 * a pending kill as otherwise we would do more
			 * or less the same thing twice
			 */
			if (c == tp->t_delq) {
				while (getc(&tp->t_canq) >= 0)
					;
				tp->t_xtra &= ~KILL_PEND;
				tp->t_delct = 0;
				tp->t_nlpos = NULL;
				if (flags & ECHO) {
					echo(c, tp);
					if (tp->t_col)
						ttyoutput('\n', tp);
					ttstart(tp);
				}
				return;
			}

			/*
			 * if the last char was a kill, we will do it now
			 * (we have the following char & it is not erase)
			 * reset all the buffer pointers etc, and discard
			 * any buffers that are no longer needed
			 */
			if(tp->t_xtra&KILL_PEND) {
				while (tp->t_canq.c_cl != NULL
					&& tp->t_canq.c_cl != tp->t_nlpos
					&& delc(&tp->t_canq) >= 0 )
						;

				tp->t_xtra &= ~KILL_PEND;
			}

			/*
			 * if we have a DISCARD char (^O), throw away
			 * any output waiting, & toggle the DISCARDING flag
			 */
			if (c == tp->t_disc) {
				if ((tp->t_xtra ^= DISCARDING) & DISCARDING) {
					(*cdevsw[major(tp->t_dev)].d_stop)(tp);
					while (getc(&tp->t_outq) >= 0)
						;
				}
				if (flags & ECHO) {
					echo(c, tp);
					ttyoutput('\n', tp);
					ttstart(tp);
				} else if (tp->t_state & ASLEEP) {
					tp->t_state &= ~ASLEEP;
					if (tp->t_chan)
						mcstart(tp->t_chan, (caddr_t)&tp->t_outq);
					else
						wakeup((caddr_t)&tp->t_outq);
				}
				return;
			}

			/*
			 * if this char is a kill char, echo it, (and
			 * take a new line if requested)
			 * if there is anything in the current line,
			 * remember we have seen the kill (process it
			 * when we get the next char)
			 */
			if(c == tp->t_kill) {
				if (tp->t_canq.c_cf && tp->t_canq.c_cl != tp->t_nlpos)
					tp->t_xtra |= KILL_PEND;
				if(flags&ECHO) {
					echo(c, tp);
					if(flags&KNL) switch (tp->t_type) {
						case VC404:
							ttyoutput('\r', tp);
							ttyoutput(ctl('v'), tp);
							break;
						case VIS200:
						case ADS520:
						case ADM3A:
						case HZL1500:
						case TRY1061:
							ttyoutput('\r', tp);
							ttyoutput(tp->t_type == HZL1500 ?
								'~' : 033, tp);
							ttyoutput('K', tp);
							tp->t_col = 0;
							break;
#if	O_INT
						case INT8001:
							if (putc(ctl('k'), tp) == 0)
								tp->t_col = 0;
							break;
#endif
						default:
							ttyoutput('\n', tp);
					}
					ttstart(tp);
				}
				return;
			}

			/*
			 * finally, if this is the escape char, just
			 * remember it, so we won't do all this next time
			 */
			if(c==tp->t_escp) {
				tp->t_xtra |= ESC_PEND;
				if(flags&ECHO) {
					echo(c, tp);
					ttstart(tp);
				}
				return;
			}
		}
	}

	/*
	 * discard input from the terminal if nobody seems to want to read it,
	 * notify the user by ringing the bell istead of echoing the char
	 * (but only if there is not much other traffic)
	 */
	if (tp->t_canq.c_cc >= TTYHOG) {
		if (flags&ECHO && tp->t_outq.c_cc < TTLOWAT) {
			ttyoutput('\007', tp);
			ttstart(tp);
		}
		tp->t_xtra &= ~ ESC_PEND;
		return;
	}

#if	O_LCASE || O_UCASE
	if (!(tp->t_xtra & ESC_PEND)) {
		/*
		 * if this is a letter & we are altering cases
		 * (one way or both) do it here
		 */
#if	O_LCASE
		if(c>='A' && c<='Z' && flags&LCASE)
			c += 'a' - 'A';
#if	O_UCASE
		else
#endif
#endif
#if	O_UCASE
		if(c>='a' && c<='z' && flags&UCASE)
			c += 'A' - 'a';
#endif
		if (cc != c && !(flags&DCASE))
	}
#endif

#if	O_PAGE
	/*
	 * If we have an empty line, and we are holding output at
	 * a page boundary, then restart it now
	 */
	if (tp->t_xtra&(PG_WAIT|PG_PEND)
		 && (c == '\n' || !(tp->t_xtra&ESC_PEND) && c == tun.t_eofc)
		 && (tp->t_canq.c_cc == 0 || tp->t_nlpos == tp->t_canq.c_cl)) {
		if (tp->t_pgflg & PG_CLEAR) {
			tp->t_xtra |= PG_IN;
			ttyoutput('\f', tp);
		}
		if (c == '\n')
			tp->t_pgcnt = tp->t_pglen;
		else
			tp->t_pgcnt = (tp->t_pglen >> 1) & 0177;
		if (tp->t_xtra & PG_WAIT)
			tp->t_xtra &= ~(PG_WAIT|PG_PEND);
		else
			tp->t_xtra |= PG_CONT;
		ttstart(tp);
		return;
	}
#endif

	/*
	 * when we finally get here it is time to echo the character
	 */
	if(flags&ECHO) {
		if ((flags&RO_TYPE) == RO_VDU && (tp->t_xtra&ESC_PEND)
				&& (tp->t_desc >= ' ' || tp->t_dctl&(1<<tp->t_desc))) {
			ttyoutput('\b', tp);
			if (tp->t_desc < ' ') {
				ttyoutput(' ', tp);
				ttyoutput('\b', tp);
				ttyoutput('\b', tp);
			}
		}
		echo(c, tp);
		if (tp->t_col && c == tun.t_eofc && !(flags&(RAW|CBREAK))
			&& !(tp->t_xtra&ESC_PEND))
				ttyoutput('\n', tp);
		ttstart(tp);
	}

#if	O_LCASE
	/*
	 * if the char before this was an escape, then see if this is
	 * one of the special chars that represents one of the untypable
	 * special chars ( ~ | { } etc) or an upper case letter.
	 * if it is, switch it for its equivalent
	 */
	if(tp->t_xtra&ESC_PEND && flags&LCASE)
		if(maptab[c])
			c = maptab[c];
#endif


	/*
	 * all chars except eof are added to the input queue here,
	 * eof is added too in raw mode, of if preceded by an escape
	 *
	 * if there is no queue space to put the character on, try warning
	 * the user by ringing the terminal bell. This is unlikely to
	 * be of any use, since if there is no queue space we can't write
	 * either (unless the user has a partly filled output buffer)
	 * --- we can only try
	 */
	if( c!=tun.t_eofc || flags&(RAW|CBREAK) || tp->t_xtra&ESC_PEND ) {
		if (putc(c, &tp->t_canq))
			VOID putc('\007', &tp->t_outq);
		if (tp->t_canq.c_cc == 1)
			tp->t_nlpos = tp->t_canq.c_cf;
	}


	/*
	 * if this char is a line terminator, add the marker char (in non-raw
	 * mode only) and increase the count of completed lines.
	 * Note: eof is a line terminator except if preceded by escape,
	 *       every character is a line terminator in raw mode.
	 * we also remember the start of the next line for use by
	 * kill, erase, and redisplay processing, and wakeup the reader
	 * if it is asleep.
	 * If input lines are considered to start new pages, do it now
	 */
	if(c=='\n' || (c==tun.t_brkc || c==tun.t_eofc) && !(tp->t_xtra&ESC_PEND) || flags&(RAW|CBREAK)) {
		if (!(flags&(RAW|CBREAK)) && putc(0377, &tp->t_canq) == 0)
			tp->t_delct++;
		if (tp->t_chan != NULL)
			VOID sdata(tp->t_chan);
		else
			wakeup((caddr_t)&tp->t_canq);
		tp->t_nlpos = tp->t_canq.c_cl;
#if	O_PAGE
		if ((tp->t_pgflg & PG_FIXD) == 0)
			tp->t_pgcnt = tp->t_pglen;
#endif
	}

	/*
	 * last, and probably least too, clear stray flags, and get out
	 */
	tp->t_xtra &= ~(ESC_PEND|KILL_PEND);
}
#endif


/*
 * Send stop character on input overflow.
 */
ttyblock(tp)
register struct tty *tp;
{
	register x;
	x = q1.c_cc + q2.c_cc;
	if (q1.c_cc > TTYHOG) {
		flushtty(tp);
		tp->t_state &= ~TBLOCK;
	}
	if (x >= TTYHOG/2) {
		if (putc(tun.t_stopc, &tp->t_outq)==0) {
			tp->t_state |= TBLOCK;
			tp->t_char++;
			ttstart(tp);
		}
	}
}

/*
 * put character on TTY output queue, adding delays,
 * expanding tabs, and handling the CR/NL bit.
 * It is called both from the top half for output, and from
 * interrupt level for echoing.
 * The arguments are the character and the tty structure.
 */
#if	! MELB_TTY
ttyoutput(c, tp)
register c;
register struct tty *tp;
{
	register char *colp;
	register ctype;

	tk_nout++;
	/*
	 * Ignore EOT in normal mode to avoid hanging up
	 * certain terminals.
	 * In raw mode dump the char unchanged.
	 */

	if ((tp->t_flags&RAW)==0) {
		c &= 0177;
		if ((tp->t_flags&CBREAK)==0 && c==CEOT)
			return;
	} else {
		VOID putc(c, &tp->t_outq);
		return;
	}

	/*
	 * Turn tabs to spaces as required
	 */
	if (c=='\t' && (tp->t_flags&TBDELAY)==XTABS) {
		c = 8;
		do
			ttyoutput(' ', tp);
		while (--c >= 0 && tp->t_col&07);
		return;
	}
	/*
	 * for upper-case-only terminals,
	 * generate escapes.
	 */
	if (tp->t_flags&LCASE) {
		colp = "({)}!|^~'`";
		while(*colp++)
			if(c == *colp++) {
				ttyoutput('\\', tp);
				c = colp[-2];
				break;
			}
		if ('a'<=c && c<='z')
			c += 'A' - 'a';
	}
	/*
	 * turn <nl> to <cr><lf> if desired.
	 */
	if (c=='\n' && tp->t_flags&CRMOD)
		ttyoutput('\r', tp);
	VOID putc(c, &tp->t_outq);
	/*
	 * Calculate delays.
	 * The numbers here represent clock ticks
	 * and are not necessarily optimal for all terminals.
	 * The delays are indicated by characters above 0200.
	 * In raw mode there are no delays and the
	 * transmission path is 8 bits wide.
	 */
	colp = &tp->t_col;
	ctype = partab[c];
	c = 0;
	switch (ctype&077) {

	/* ordinary */
	case 0:
		(*colp)++;

	/* non-printing */
	case 1:
		break;

	/* backspace */
	case 2:
		if (*colp)
			(*colp)--;
		break;

	/* newline */
	case 3:
		ctype = (tp->t_flags >> 8) & 03;
		if(ctype == 1) { /* tty 37 */
			if (*colp)
				c = max(((unsigned)*colp>>4) + 3, (unsigned)6);
		} else
		if(ctype == 2) { /* vt05 */
			c = 6;
		}
		*colp = 0;
		break;

	/* tab */
	case 4:
		ctype = (tp->t_flags >> 10) & 03;
		if(ctype == 1) { /* tty 37 */
			c = 1 - (*colp | ~07);
			if(c < 5)
				c = 0;
		}
		*colp |= 07;
		(*colp)++;
		break;

	/* vertical motion */
	case 5:
		if(tp->t_flags & VTDELAY) /* tty 37 */
			c = 0177;
		break;

	/* carriage return */
	case 6:
		ctype = (tp->t_flags >> 12) & 03;
		if(ctype == 1) { /* tn 300 */
			c = 5;
		} else if(ctype == 2) { /* ti 700 */
			c = 10;
		} else if(ctype == 3) { /* concept 100 */
			int i;
			for (i= *colp; i<9; i++)
				VOID putc(0177, &tp->t_outq);
		}
		*colp = 0;
	}
	if(c)
		VOID putc(c|0200, &tp->t_outq);
}
#else
ttyoutput(c, tp)
register c;
register struct tty *tp;
{
	register char *colp;
	register int ctype;
	register flags = tp->t_flags;

	tk_nout++;
	/*
	 * Ignore EOT in normal mode to avoid hanging up
	 * certain terminals.
	 */
	if (!(flags & RAW)) {
		if((c &= 0177) == '\04')
			return;
#if	O_INT
		if (c == ctl('x') && tp->t_type == INT8001)
			return;
#endif
	} else {
		VOID putc(c, &tp->t_outq);
		return;
	}

	/*
	 * ignore this char if so instructed
	 */
	if (c == tp->t_dchar && (tp->t_ddel&0377) >= 0200)
		goto delay;

	/*
	 * Turn tabs to spaces as required
	 */
	if (c=='\t' && (tp->t_tabdel&0377)==XTABS) {
		c = 8 - (tp->t_col&07);
		do
			ttyoutput(' ', tp);
		while (--c);
		return;
	}
#if	O_LCASE
	/*
	 * for upper-case-only terminals,
	 * generate escapes.
	 */
	if ((flags & (LCASE|DCASE)) == LCASE) {
		colp = "({)}!|^~'`";
		while(*colp++)
			if(c == *colp++) {
				ctype = tp->t_xtra & PG_IN;
				echo(tp->t_escp, tp);
				tp->t_xtra &= ~PG_IN;
				tp->t_xtra |= ctype;
				c = colp[-2];
				break;
			}
		if ('a'<=c && c<='z')
			c += 'A' - 'a';
#if	O_UCASE
		else if ('A' <= c && c <= 'Z' && flags&UCASE)
			c += 'a' - 'A';
#endif
	}
#endif
	/*
	 * turn <nl> to <cr><lf> if desired.
	 */
	if (c=='\n' && flags&CRMOD)
		ttyoutput('\r', tp);

#if	O_INT
	if (c=='\b' && tp->t_type == INT8001)
		c = ctl('z');		/* believe this: ^Z for bsp !! */
#endif

	/*
	 * if this is a printing char, & we are already at the margin
	 * (and we want software wrapping) - send a newline now
	 */
	if ((flags&WRAP) && tp->t_width && (partab[c]&077)==0 &&
			(tp->t_col&0377) >= (tp->t_width&0377))
		ttyoutput('\n', tp);

	/*
	 * for form feeds
	 *	1) in page mode, mark the end of a page
	 *	2) simulate with newlines if necessary
	 *	3) attempt to clear the screen of a VDU
	 */
	if (c == '\f') {
#if	O_PAGE
		if (tp->t_pglen && ttypage(tp))
			return;
#endif
		if ((tp->t_ffdel&0377) == XFORM) {
			if ((c = tp->t_depth) == 0)
				c = 6;
			else
				c -= tp->t_lcnt;
			while (--c >= 0 && !(tp->t_xtra&PG_PEND))
				ttyoutput('\n', tp);
			return;
		}
		tp->t_lcnt = 0;
		switch (tp->t_type) {
		case VC404:
			if (putc(ctl('x'), &tp->t_outq))
				return;
			goto delay;
		case TEK4012:
			if (putc(033, &tp->t_outq))
				return;
			break;
		case VIS200:
		case ADS520:
		case ADM3A:
			if (putc(033, &tp->t_outq) || putc('w', &tp->t_outq))
				return;
			goto delay;
		case HZL1500:
			if (putc('~', &tp->t_outq) || putc(ctl(']'), &tp->t_outq))
				return;
			goto delay;
		}
	}

	if (c != tp->t_dchar || (tp->t_ddel&0377)<0200)
		if (putc(c, &tp->t_outq))
			return;
   delay:
	/*
	 * Calculate delays.
	 * The numbers here represent clock ticks
	 * and are not necessarily optimal for all terminals.
	 */
	colp = &tp->t_col;
	ctype = partab[c];
	if (c == tp->t_dchar)
		c = tp->t_ddel & 0177;
	else
		c = 0;
	switch (ctype&077) {

	/* ordinary */
	case 0:
		(*colp)++;

	/* non-printing */
	case 1:
		break;

	/* backspace */
	case 2:
#if	O_INT
		if (tp->t_type == INT8001)
			*colp = 0;	/* 'home' returns to left margin */
		else
#endif
		if (*colp)
			(*colp)--;
		break;

	/* newline */
	case 3:
		if (++tp->t_lcnt == tp->t_depth)
			tp->t_lcnt = 0;

		if (!(flags & CRMOD))
			*colp = 0;

#if	O_PAGE
		if (tp->t_pglen && --tp->t_pgcnt == 0)
			if (ttypage(tp))
				return;
#endif

		if ((c = tp->t_nldel&0377) >= 0200) switch (c) {
			case NL_37:
				if (*colp)
					c = max(((unsigned)*colp>>4)+3, (unsigned)6);
				else
					c = 0;
				break;
			default:
				c = 0;
				break;
		}
		break;

	/* tab */
	case 4:
		if ((c = tp->t_tabdel&0377)>=0200) switch (c) {
			case TAB_37:
				if ((c = 1 - (*colp|~07)) < 5)
					c = 0;
				break;
			default:
				c = 0;
		}
		*colp |= 07;
		(*colp)++;
		break;

	/* vertical motion */
	case 5:
		if ((c = tp->t_ffdel&0377) >= 0200) switch (c) {
			default:
				c = 0;
				break;
		}
		*colp = 0;	/* \f usually goes to left margin */
		break;

	/* carriage return */
	case 6:
		if ((c = tp->t_crdel&0377) >= 0200) switch (c) {
			default:
				c = 0;
				break;
		}
		*colp = 0;
		break;

	/* control z (cursor left on Intecolor) */
	case 7:
#if	O_INT
		if (tp->t_type == INT8001 && *colp)
			(*colp)--;
#endif
		break;
	}
	if(c)
		VOID putc(c|0200, &tp->t_outq);
}

echo(ch, tp)
register char ch;
register struct tty *tp;
{
	register char c;

#if	O_PAGE
	tp->t_xtra |= PG_IN;
#endif
	if (!(tp->t_flags&RAW)) {
		c = ch & 0177;
		if (c == 0177)
			ch = c = tp->t_delet;
		else if (c == 033)
			ch = c = tp->t_desc;
		if (c < ' ' && tp->t_dctl&(1<<c)) {
			ttyoutput('^', tp);
			ch = c | '@';
		}
	}
	ttyoutput(ch, tp);
}

#if	O_PAGE
ttypage(tp)
register struct tty *tp;
{
	if (tp->t_pglen == tp->t_pgcnt)		/* ff at top of page */
		return(0);			/* don't stop here */
	tp->t_pgcnt = tp->t_pglen;
	if (tp->t_xtra & PG_IN)
		return(0);
	if (tp->t_pgflg & PG_STOP && putc(0200, &tp->t_outq) == 0)
			tp->t_xtra |= PG_PEND;
	if (tp->t_pgflg & PG_PAUSE && tp->t_pgdel&0177)
		VOID putc(tp->t_pgdel|0200, &tp->t_outq);
	if ((tp->t_pgflg & (PG_STOP|PG_CLEAR)) == PG_CLEAR)
		ttyoutput('\f', tp);
	return(1);
}
#endif
#endif

/*
 * Restart typewriter output following a delay
 * timeout.
 * The name of the routine is passed to the timeout
 * subroutine and it is called during a clock interrupt.
 */
ttrstrt(tp)
register struct tty *tp;
{

	tp->t_state &= ~TIMEOUT;
	ttstart(tp);
}

/*
 * Start output on the typewriter. It is used from the top half
 * after some characters have been put on the output queue,
 * from the interrupt routine to transmit the next
 * character, and after a timeout has finished.
 */
ttstart(tp)
register struct tty *tp;
{
	register s;

	s = spl5();
	if((tp->t_state&(TIMEOUT|TTSTOP|BUSY)) == 0)
		(*tp->t_oproc)(tp);
#if	O_PAGE
	tp->t_xtra &= ~PG_IN;
#endif
	splx(s);
}

/*
 * Called from device's read routine after it has
 * calculated the tty-structure given as argument.
 */
ttread(tp)
register struct tty *tp;
{
	register s;
#if	MELB_TTY
	register raw = tp->t_flags&(RAW|CBREAK);
	register c;
#endif

	if ((tp->t_state&CARR_ON)==0)
		return(-1);
#if	! MELB_TTY
	s = spl5();
	if (tp->t_canq.c_cc==0)
		while (canon(tp)<0)
			if (tp->t_chan==NULL) {
				sleep((caddr_t)&tp->t_rawq, TTIPRI); 
			} else {
				splx(s);
				return(0);
			}
	splx(s);
	while (tp->t_canq.c_cc && passc(getc(&tp->t_canq))>=0)
			;
#else
	s = spl5();
	while (!raw && tp->t_delct==0 || raw && tp->t_canq.c_cc==0) {
		if ((tp->t_state&CARR_ON) == 0 || tp->t_chan != NULL)
			return(0);
		sleep((caddr_t)&tp->t_canq, TTIPRI);
	}
	splx(s);
	while ((c = getc(&tp->t_canq)) >= 0) {
		if (!raw && c == 0377) {
			tp->t_delct--;
			break;
		}
		if (passc(c) < 0)
			break;
	}
	if (!raw && c != 0377 && tp->t_canq.c_cf != NULL && (*tp->t_canq.c_cf&0377) == 0377)
		if (getc(&tp->t_canq) == 0377)
			tp->t_delct--;
	if (tp->t_state&TBLOCK && tp->t_canq.c_cc < TTYHOG/5) {
		if (putc(tun.t_startc, &tp->t_outq) == 0) {
			tp->t_state &= ~TBLOCK;
			ttstart(tp);
		}
		tp->t_char = 0;
	}

#endif
	return(tp->t_rawq.c_cc+tp->t_canq.c_cc);
}

/*
 * Called from the device's write routine after it has
 * calculated the tty-structure given as argument.
 */
caddr_t
ttwrite(tp)
register struct tty *tp;
{
	register char *cp, *ce;
	register int cc;
	register i;
	char obuf[OBUFSIZ];

	if ((tp->t_state&CARR_ON)==0)
		return(NULL);
	while (u.u_count) {
		cc = MIN(u.u_count, OBUFSIZ);
		cp = obuf;
		iomove(cp, (unsigned)cc, B_WRITE);
		if (u.u_error)
			break;
		VOID spl5();
#if	! MELB_TTY
		while (tp->t_outq.c_cc > TTHIWAT) {
#else
		while (tp->t_outq.c_cc > TTHIWAT ||
			(tp->t_xtra & (PG_PEND|PG_WAIT|TTY_BRK)) ) {
#endif
			ttstart(tp);
			tp->t_state |= ASLEEP;
			if (tp->t_chan) {
				u.u_base -= cc;
				u.u_offset -= cc;
				u.u_count += cc;
				VOID spl0();
				return((caddr_t)&tp->t_outq);
			}
			sleep((caddr_t)&tp->t_outq, TTOPRI);
		}
		VOID spl0();
#if	MELB_TTY
		if (tp->t_xtra & DISCARDING) {
			u.u_count = 0;		/* fake a successful write */
			return(NULL);
		}
#endif
		ce = cp + cc;
		if (tp->t_flags&RAW) {
			while (cc) {
				i=b_to_q(cp,cc,&tp->t_outq);
				tk_nout+=cc-i;
				cp+=cc-i; cc=i;
				if (cc) {
					VOID spl5();
#if	! MELB_TTY
					while (tp->t_outq.c_cc > TTHIWAT) {
#else
					while (tp->t_outq.c_cc > TTHIWAT ||
						(tp->t_xtra&(PG_PEND|PG_WAIT|TTY_BRK)) ) {
#endif
						ttstart(tp);
						tp->t_state |= ASLEEP;
						sleep((caddr_t)&tp->t_outq, TTOPRI);
					}
					VOID spl0();
#if	MELB_TTY
					if (tp->t_xtra & DISCARDING) {
						u.u_count = 0;		/* fake a successful write */
						return(NULL);
					}
#endif
				}
			}
		} else
			while (cp < ce) {
#if	MELB_TTY
				if (tp->t_xtra&DISCARDING) {
					u.u_count = 0;
					return;
				}
				while (tp->t_xtra&(PG_PEND|PG_WAIT|TTY_BRK)) {
					ttstart(tp);
					tp->t_state |= ASLEEP;
					sleep((caddr_t)&tp->t_outq, TTOPRI);
				}
#endif
				ttyoutput(*cp++, tp);
			}
	}
	ttstart(tp);
	return(NULL);
}
