#
int dv_debug = 0;	/*%%%*/

/*
 * DV disk driver
 */

#include "../h/param.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/uba.h"

#define DVMAPF	1		/* define to get sector mapping */
#define	NDV	4		/* number of drives on controller */

struct dvregs {
	short	dvcsr;		/* control & status (R/W) */
	short	dvdbr;		/* data buffer register (maint only) (RO) */
	short	dvmar;		/* memory address register (R/W) */
	short	dvwcr;		/* word count register (R/W) */
	short	dvcbr;		/* command buffer register (R/W) */
	short	dvssr;		/* selected status register (RO) */
	short	dvair;		/* attention interrupt register (R/W) */
	short	dvusr;		/* unit status register (RO) */
};

#define	DVADDR	((struct dvregs *)(UBA0_DEV + 0164000))

struct dvsizes {
	unsigned int	nblocks;
	int		cyloff;
} dv_sizes[] = {
	31920,	0,		/* cyl 0 thru 132	*/
	65520,	133,		/* cyl 133 thru 405	*/
	21600,	0,		/* cyl 0 thru 89	*/
	21600,	90,		/* cyl 90 thru 179	*/
	54000,	180,		/* cyl 180 thru 405	*/
	65520,	0,		/* cyl 0 thru 272	*/
	31680,	273,		/* cyl 273 thru 405	*/
	2048,	123,		/* cyl 123 thru 132	*/
};

struct {
	short	hd1;
	short	hd2;
	short	cksum;
	short	chadr;		/* actually char * */
} dvhdr;

int	dv_unit = -1;
int	dv_cyl;
int	dv_head;
int	dv_sctr;
int	dv_count;
int	dv_addr;
int	dv_wcheck;
int	dv_sw = 0;
int	wcwcnt[8];		/* count of bad write checks per drive */
int	dv_ubhinfo;		/* unibus map info for header */
int	dv_ubdinfo;		/* unibus map info for data */

struct	buf	dvtab;
struct	buf	rdvbuf;

# ifdef DVMAPF
char	dvsecmap[] = {
	0, 4, 8,
	1, 5, 9,
	2, 6, 10,
	3, 7, 11,
};
# endif


#define	HTBCOM	000000
#define	CTBCOM	020000
#define	CNBCOM	030000
#define		INCHD	01
#define		RECAL	02
#define		RESET	020
#define		SEEK	040
#define		CLRDA	0100
#define	CHRCOM	070000
#define	CHWCOM	0130000
#define	CHCD	071000
#define	LDSCOM	0140000
#define	CTLRST	0170000
#define CHRDATA	060000
#define CHWHEAD	0110000
#define WENABL	0100000

#define	DRVRDY	04000
#define	ATTN	0400
#define	DONE	0200
#define	IENABLE	0100
#define	GO	01

/*
 * Use b_error to save sector,
 * b_resid for cylinder+track.
 */

#define	dvsec	b_error
#define	cylhd	b_resid

dvstrategy(abp)
struct buf *abp;
{
	register struct buf *bp;
	register struct buf *p1, *p2;
	register struct dvsizes *dvsp;
	register int i;

	if (dv_debug) printf("dvstrategy\n");	/*%%%*/
	bp = abp;

	if (dv_debug && (bp->b_flags & B_PHYS))	/*%%%*/
	{	/*%%%*/
		printf("raw diva %x\n", bp->b_un.b_addr);	/*%%%*/
	}	/*%%%*/

	/*
	** Check to see that device number & addr is in range.
	*/

	dvsp = &dv_sizes[minor(bp->b_dev)&07];
	if ((minor(bp->b_dev)&077) >= (NDV<<3) ||
	    bp->b_blkno >= dvsp->nblocks) {
		bp->b_flags |= B_ERROR;
		iodone(bp);	/* is this needed? */
		return;
	}

	/*
	** Compute physical disk addresses (cylinder, head, etc.)
	*/

	bp->av_forw = 0;
	i = bp->b_blkno / 12;
	bp->cylhd = ((dvsp->cyloff+i/20)<<5)|(i%20);
	bp->dvsec = bp->b_blkno % 12;

	/*
	** Insert this interaction in queue using a shortest
	** seek time first algorithm.
	*/

	spl5();
	if ((p1 = dvtab.b_actf)==0)
		dvtab.b_actf = bp;
	else {
		for (; p2 = p1->av_forw; p1 = p2) {
			if (p1->cylhd <= bp->cylhd
			 && bp->cylhd <  p2->cylhd
			 || p1->cylhd >= bp->cylhd
			 && bp->cylhd >  p2->cylhd) 
				break;
		}
		bp->av_forw = p2;
		p1->av_forw = bp;
	}
	if (dvtab.b_active==0)
		dvstart();
	spl0();
}

/*
** Start disk operation.
**	Fetch first operation on queue.
**	Extract location information.
**	Initialize Unibus mapping registers.
*/

dvstart()
{
	register struct buf *bp;
	int i;	/*%%%*/

	if (dv_debug) printf("dvstart\n");	/*%%%*/

	if ((bp = dvtab.b_actf) == 0) {
		dvtab.b_active = 0;
		return;
	}
	dv_cyl = bp->cylhd>>5;
	dv_head = bp->cylhd&037;
	dv_sctr = bp->dvsec;
	dv_count = (bp->b_bcount + 1) & ~01;
	if (dv_debug) printf("cy%d hd%d se%d cnt%d ", dv_cyl, dv_head, dv_sctr, dv_count);	/*%%%*/
	dv_ubhinfo = uballoc(&dvhdr, sizeof dvhdr, 0);
	dv_ubdinfo = ubasetup(bp, 1);
	dv_addr = dv_ubdinfo & 0x3ffff;
	if (dv_debug)	/*%%%*/
	{	/*%%%*/
		printf("ubh 0x%x ubd 0x%x\n", dv_ubhinfo, dv_ubdinfo);	/*%%%*/
		printf("H UBA:");	/*%%%*/
		for (i = 0; i < 3; i++)	/*%%%*/
			printf(" %x", ((struct uba_regs *) UBA0)->uba_map[((dv_ubhinfo >> 9) & 0777) + i]);	/*%%%*/
		printf("\nD UBA:");	/*%%%*/
		for (i = 0; i < dv_count / 512 + 2; i++)	/*%%%*/
			printf(" %x", ((struct uba_regs *) UBA0)->uba_map[((dv_ubdinfo >> 9) & 0777) + i]);	/*%%%*/
		printf("\n");	/*%%%*/
	}	/*%%%*/
	dvexec();
}

/*
**  Finish diva operation.
**	Free unibus maps.
**	iodone()
*/

dvdone(bp)
register struct buf *bp;
{
	if (dv_debug) printf("dvdone\n");	/*%%%*/
	ubafree(dv_ubhinfo);
	ubafree(dv_ubdinfo);
	iodone(bp);
}

/*
**  DVEXEC -- execute diva command.
**	Select correct unit if necessary.
**	Seek if necessary.
**	Set up "confirm I/O" type header.
**	Select operation.
**	"GO".
*/

dvexec()
{
	register struct buf *bp;
	register sctr, minord;
	register int i;

	if (dv_debug) printf("dvexec\n");	/*%%%*/
	bp = dvtab.b_actf;
	sctr = dv_sctr;
	minord = minor(bp->b_dev);
# ifdef DVMAPF
	if (minord&64)
		sctr = dvsecmap[sctr];
# endif
	dvtab.b_active++;

	/* If we aren't at the correct unit, select it */
	if (dv_unit!=((minord&077)>>3)) {
		dv_unit = (minord&077)>>3;
		dvwait();
		DVADDR->dvcbr = LDSCOM | dv_unit;
	}
	if(dvrdy()) return;

	/* reset and clear unit */
	DVADDR->dvcbr = CNBCOM | RESET | CLRDA;

	/* if we aren't at correct cylinder, seek */
	if (dv_cyl != ((~DVADDR->dvssr) & 0777)) {
		if(dvrdy()) return;
		DVADDR->dvcbr = CTBCOM | dv_cyl;
		if(dvrdy()) return;
		DVADDR->dvcbr = CNBCOM | SEEK | RESET;
		DVADDR->dvair = 1<<dv_unit;
		DVADDR->dvcsr = DONE | IENABLE;
		return;		/* and now for a word from our sponser */
	}
	if(dvrdy()) return;

	/* select correct head */
	DVADDR->dvcbr = HTBCOM | dv_head;

	/* compute transfer byte count */
	DVADDR->dvwcr = -min(dv_count, 512);

	/* set up confirm buffer */
	dvhdr.hd1 = (dv_head<<8)+dv_cyl;	/* set up header */
	dvhdr.hd2 = 0170000|sctr;
	dvhdr.cksum = -dvhdr.hd1-dvhdr.hd2;
	dvhdr.chadr = dv_addr & 0xffff;
	if (dv_debug) printf("dvhdr: %x %x %x %x\n", dvhdr.hd1, dvhdr.hd2, dvhdr.cksum, dvhdr.chadr);
	if(dvrdy()) return;

	/* set up MAR and operation */
	DVADDR->dvmar = dv_ubhinfo & 0xffff;
	if (minord & 128) {
		if (bp->b_flags & B_READ) {
			/* raw (unconfirmed) read */
			DVADDR->dvmar = dv_addr & 0xffff;
			DVADDR->dvcbr = CHRDATA | (sctr<<1);
		} else {
			/* write headers */
			DVADDR->dvcbr = CHWHEAD | (sctr<<1);
			DVADDR->dvair |= WENABL;
		}
	} else {
		if (bp->b_flags & B_READ)
			/* read confirm */
			DVADDR->dvcbr = CHRCOM | (sctr<<1);
		else if (dv_wcheck)
			/* write check */
			DVADDR->dvcbr = CHCD | (sctr<<1);
		else
			/* write confirm */
			DVADDR->dvcbr = CHWCOM | (sctr<<1);
	}

	/* "GO".... (with high order of both addresses) */
	DVADDR->dvcsr = IENABLE | GO | ((dv_addr >> (16-4)) & 060) | ((dv_ubhinfo >> (16-2)) & 014);
}

int	dv_tmp;
int	dv_errct;

/*
** Field interrupt.
**	If seek:
**		On error, print a message.
**		Restart (to do I/O this time).
**	Dispose of errors.
**	If writing, perform a write check (we'll come around again).
**	If transfer is not done, arrange for another.
**	Do some error processing for header write/unconfirmed read (???).
**	Signal I/O complete.
*/

dvintr()
{
	register struct buf *bp;
	register int csr,i;

	if (dv_debug) printf("dvintr %o\n", DVADDR->dvcsr);	/*%%%*/
	if (dvtab.b_active == 0)
		return;

	bp = dvtab.b_actf;
	dvtab.b_active = 0;
	csr = DVADDR->dvcsr;
	DVADDR->dvcsr = DONE;
	if (csr&ATTN) { /* seek complete */
		dv_wcheck = 0;
		DVADDR->dvair &= ~0377;
		if(DVADDR->dvssr>0) { /* error */
			printf("Seek error\n");
			deverror(bp, DVADDR->dvssr, csr);
			DVADDR->dvcbr = CNBCOM | RECAL | RESET;
			dv_unit = -1;
			if(dvrdy()) return;
			DVADDR->dvcbr = CTLRST;
			if (++dvtab.b_errcnt > 10) {
				dvherr(0);
				return;
			}
		}
		dvexec();
		return;
	}
	/* r/w complete */
	i = min(dv_count, 512);
	bp->b_resid = (dv_addr + i - DVADDR->dvmar) & 0xffff;
	if (csr < 0 || (bp->b_resid != 0 && (minor(bp->b_dev)&128)==0))
	{
		if ((dv_addr & 0xffff) - i != DVADDR->dvmar)
		{
			printf("hdr/xfer err ");
			printf("%o %o %o\n", dv_addr, DVADDR->dvmar, csr);
			deverror(bp, DVADDR->dvssr, csr);
			dv_wcheck = 0;
		}
		dv_tmp = csr;
		dv_errct++;
		dv_unit = -1;
		if (dv_wcheck)
		{
			printf("diva bad write\n");
			wcwcnt[(minor(bp->b_dev)&070) >> 3]++;
			deverror(bp, DVADDR->dvssr, csr);
		}
		dv_wcheck = 0;
		if((dvtab.b_errcnt&03)==03) {
			deverror(bp, DVADDR->dvssr, csr);
			DVADDR->dvcbr = CNBCOM | RECAL | RESET;
			if(dvrdy()) return;
		}
		DVADDR->dvcbr = CTLRST;
		if(++dvtab.b_errcnt<=12)
			dvexec();
		else
			dvherr(0);
		return;
	}
	if (dv_sw && (bp->b_flags&B_READ)==B_WRITE &&
	    dv_wcheck==0 && dv_count==-512) {
		dv_wcheck = 1;
		dvexec();
		return;
	}
	dv_wcheck = 0;
	if ((dv_count -= 512) > 0) {
		/* more to do */
		dv_addr += 512;
		if (++dv_sctr>=12) {
			dv_sctr = 0;
			if (++dv_head>=20) {
				dv_head = 0;
				dv_cyl++;
			}
		}
		dvexec();
		return;
	}
	if (csr < 0 && (minor(bp->b_dev)&128)) {
		DVADDR->dvcbr = CTLRST;
		dv_unit = -1;
		dvrdy();
		if (DVADDR->dvcsr < 0) {
			DVADDR->dvcbr = CNBCOM | RECAL | RESET;
			dvrdy();
		}
	}
	dv_wcheck = 0;
	dvtab.b_errcnt = 0;
	dvtab.b_actf = bp->av_forw;
	/* bp->b_resid = DVADDR->dvwcr+0160000; */
	dvdone(bp);
	dvstart();
}


dvrdy()
{
	if ((DVADDR->dvssr & DRVRDY) == 0)
		return(0);
	dvwait();
	if (DVADDR->dvssr & DRVRDY) {
		printf("diva not ready\n");
		dvherr(1);
		return(1);
	}
	return(0);
}

/*
 *	waste a bit of time, max .7(approx) sec on a 70
 */

dvwait()
{
	register int i;

	i = 1 << 17;
	while(--i > 0 && (DVADDR->dvssr&DRVRDY))
		continue;
}


dvherr(n)
{
	register struct buf *bp;

	bp = dvtab.b_actf;
	printf("Hard error on diva\n");
	deverror(bp, DVADDR->dvssr, DVADDR->dvcsr);
	bp->b_flags |= B_ERROR;
	dvtab.b_errcnt = 0;
	dvtab.b_active = 0;
	dvtab.b_actf = bp->av_forw;
	dvdone(bp);
	if(n==0)
		dvstart();
}
dvread(dev)
{
	if(dvphys(dev))
		physio(dvstrategy, &rdvbuf, dev, B_READ);
}

dvwrite(dev)
{
	if(dvphys(dev))
		physio(dvstrategy, &rdvbuf, dev, B_WRITE);
}

dvphys(dev)
{
	register c;

	if (dv_debug) printf("dvphys: %x\n", u.u_base);	/*%%%*/
	c = u.u_offset >> 9;
	c += (u.u_count+511) / 512;
	if(c > dv_sizes[minor(dev) & 07].nblocks) {
		u.u_error = ENXIO;
		return(0);
	}
	return(1);
}
