#
/*
 *	DZ-11A driver (for 8 lines max)
 *	Adapted from Bell Lab's standard
 *	DH and DHDM driver, with inspiration
 *	from the DC11 driver.
 *	A fair amount of code is duplicated.
 *	If you have both this driver and a DH or DC
 *	one resident, it is probably wise to try to
 *	share code by modifying this one.
 *
 *	To adapt this driver to a 16-line DZ-11, all references
 *	to 'dzaddr' must be preceded by a decision (made somehow)
 *	which one of the two Unibus-addresses to pick.
 *	Furthermore, 'dzstatus' should be used to watch 16 carrier bits
 *	now and routine 'dztimer' should be rewritten to make
 *	it word-oriented instead of byte-oriented.
 * Watch out for the implied restriction that minor device number
 * equals the hardware line number currently.
 *
 *	H.J. Thomassen, Computer Graphics Group,
 *	Faculteit W & N, Toernooiveld
 *	Nijmegen University, the Netherlands.
 *	November 1976.
 *
 *	This driver was released to the software
 *	exchange since there seemed to be a need
 *	for DZ-11 support. It has been running
 *	at our installation for several days
 *	without problems. However, this is not
 *	a piece of software with a field proven
 *	record. Please contact the author about
 *	any flaws you may find. A better
 *	version will be released if necessary as
 *	soon as possible.
 */

#include "../conf.h"
#include "../param.h"
#include "../proc.h"
#include "../tty.h"
#include "../user.h"

#define dzaddr	0167740
#define ndz11   8	/* number of lines supported <= 8 */
			/* When changing configuration, be sure to
			   update dzspeedtab too */



struct	tty	dz11[ndz11];/* allocate one tty.h structure for each line */


/*
 *	hardware control bits
 */

#define bits6	010	/* char. length in bits */
#define bits7 	020
#define bits8	030
#define twosb	040	/* want 1.5 or 2 stop bits */
#define penable	0100	/* enable parity */
#define opar	0200	/* odd parity if enabled */

#define	ienable	040100	/* enable all intrpts except silo */
#define scenabl	040	/* master scan enable */
#define done	0200	/* receiver done */
#define xint	0100000	/* xmitter done */

#define dvalid	0100000	/* input data are valid */
#define ferror	020000	/* framing error on input char */
#define perror	010000	/* parity error on input char */



/****************************************************************
/*
/*   Attention: hardware requires most of the registers
/*   to be accessed by certain instructions in assembler only.
/*   Take care that the C-compiler generates proper code !!!!!
/*
/***************************************************************/
struct	{	/* read and write register lay-out */
	int	dzcsr;
	int	x1;
	int	dzstatus;	};
struct	{	/* read only register lay-out */
	int	x2;
	int	dzrbuf;
	char	x3;
	char	x4;
	char	dzring;
	char	dzcarr;		};
struct	{	/* write only register lay-out */
	int	x5;
	int	dzlpr;
	char	x6;
	char	x7;
	char	dztbuf;
	char	dzbrk;		};

/*
	Note: it is wise to connect faster terminals to higher line numbers.
 */
char	dzspeedtab[] { /* speed control bits for 'lpr' register (high byte) */

	/*
	   Index into the table is the same as the speed selector numbers
	   for the DH11 (as is maintained in tp->t_speeds).
	   The DZ-11 does not support 200 bps, ext A and ext B.
	   It does support 2000, 3600 and 7200 bps, which will be accepted
	   by stty as indices 16, 17 and 18.
	*/
	     0>>8,	/* hang up	*/
	010000>>8,	/* 50 baud	*/
	010400>>8, 	/* 75 baud	*/
	011000>>8,	/* 110 baud	*/
	011400>>8,	/* 134,5 baud	*/
	012000>>8,	/* 150 baud	*/
	     0>>8,	/* no 200 baud	*/
	012400>>8,	/* 300 baud	*/
	013000>>8,	/* 600 baud	*/
	013400>>8,	/* 1200 baud	*/
	014000>>8,	/* 1800 baud	*/
	015000>>8,	/* 2400 baud	*/
	016000>>8,	/* 4800 baud	*/
	017000>>8,	/* 9600 baud	*/
	     0>>8,	/* no ext. A	*/
	     0>>8,	/* no ext. B	*/
	014400>>8,	/* 2000 baud	*/
	015400>>8,	/* 3600 baud	*/
	016400>>8,	/* 7200 baud	*/
};
char	dztflag;	/* remember whether timer got activated */

/*
 *	Open a DZ-11 line
 */
dzopen(dev, flag)
{
	register struct tty	*tp;
	extern   dzstart();

	if (!dztflag) {dztimer(); dztflag++;} /* start watching carrier */
	if ( dev.d_minor >= ndz11 ) {/* invalid minor dev. no */
		u.u_error = ENXIO;
		return;
	}
	tp = &dz11[dev.d_minor];
	tp->t_addr = dzstart; /* address of startup function */
	tp->t_dev = dev;
	dzaddr->dzcsr =| ienable|scenabl; /* turn on the thing */
	tp->t_state =| WOPEN|SSTART;	  /* internal state: wait for open */
	if ((tp->t_state & ISOPEN) == 0) {/* fill tty.h with basic info    */
		tp->t_erase = CERASE;
		tp->t_kill  = CKILL;
		tp->t_speeds= 7+(7<<8);	       /* default speed 300 baud */
		tp->t_flags = ODDP|EVENP|ECHO|XTABS; /* defaults for STTY */
		dzparam(tp);		       /* set stuff to DZ11 hardware */
	}
	dzlopen(dev);			       /* open the line, and return
						  when connection is there */
	tp->t_state =& ~WOPEN;		       /* line is open now */
	tp->t_state =| ISOPEN;
	if (u.u_procp->p_ttyp == 0)
		u.u_procp->p_ttyp = tp; /* hook this line to this process */
}

/*
 *	Close a DZ-11 line
 */
dzclose(dev)
{
	register struct tty	*tp;

	tp = &dz11[dev.d_minor];
	dzlclose(dev);
	tp->t_state =& CARR_ON|SSTART;
	wflushtty(tp);
}

/*
 *	Read from a DZ-11 line
 */
dzread(dev)
{
	ttread( &dz11[dev.d_minor]);
}

/*
 *	Write on a DZ-11 line
 */
dzwrite(dev)
{
	ttwrite( &dz11[dev.d_minor]);
}

/*
 *	DZ-11 receiver interrupt
 */
dzrint()
{
	register struct tty *tp;
	register int    c, line;

	while ((c=dzaddr->dzrbuf) < 0) { /* as long as valid char's come in */
		line = ( (c>>8)&07 );
		tp = &dz11[line];	 /* get good tty struct via line no */
		if ( tp >= &dz11[ndz11] )
			continue;	 /* ignore invalid line numbers */
		if ( tp->t_state & ISOPEN == 0) { /* was open done ok?  */
			if ( tp->t_state&WOPEN && dzaddr->dzcarr&(1<<line) )
				tp->t_state =| CARR_ON;
			wakeup(tp);
			return; /* look out: tty not hooked to process yet */
		}
		if ( c&perror ) continue;	/* ignore char. w par. err. */
		if ( c&ferror )			/* break key or bad speed   */
			if ( tp->t_flags & RAW )
			   c = 0;		/* gtty must change speed   */
			else
			   c = 0177;		/* convert to 'del' (intr)  */
		ttyinput(c, tp);
	}
}

/*
 *	DZ-11 transmitter interrupt
 */
dzxint()
{
	extern ttrstrt();
	register struct tty *tp;
	register int    c, line;

	while ( (line=dzaddr->dzcsr) < 0) { /* as long as lines to serve */
		line = (line>>8)&07;
		tp = &dz11[line];
		if (( c= getc (&tp->t_outq) ) >= 0) {
			if ( c<0200 ) {	    /* if no delay wanted */
			   dzaddr->dztbuf = c;/* just send it     */
			   goto out;
			}
		}
		/* if no more characters to xmit or a delay has to
		   be generated, shut off that current DZ-11 line */
		dzaddr->dzstatus =& ~(1<<line);
		if (c >= 0200) {
			timeout(ttrstrt, tp, (c&0177)+6 );
			tp->t_state =| TIMEOUT;
		}

/* if this writer was sleeping on output overflow (ttwrite) or was
   waiting to drain (wflushtty), wake him up when low tide got
   reached or we drained all the way. (asleep-bit in t_state not used)
*/
out:	if ((tp->t_outq.c_cc == 0) || (tp->t_outq.c_cc == TTLOWAT))
		wakeup (&tp->t_outq);
	}
}

/*
 *	Restart transmission on a DZ-11 line
 */
dzstart(atp)
	struct tty *atp;
{
	register struct tty *tp;
	
	tp=atp;
	if (tp->t_state & TIMEOUT) return;	/* we're still waiting t.o */
	dzaddr->dzstatus =| (1<<tp->t_dev.d_minor); /* turn on xmit-enable bit
						       scanner does the rest */
}

/*
 *	Set parameters from open or stty call to DZ-11 hardware registers
 */
dzparam(atp)
	struct tty *atp;
{
	register struct tty *tp;
	register int    lpr;

	tp=atp;
	spl5();
	if (tp->t_speeds.lobyte == 0) { /* hang up the line? */
		tp->t_flags =| HUPCL;
		dzlclose(tp->t_dev);
		return;
	}
	dzaddr->dzcsr.lobyte =| ienable; /* enable interrupts */

	/* build bit pattern to send to line param register */
	lpr = (dzspeedtab[tp->t_speeds.lobyte]<<8) + tp->t_dev.d_minor;
	/* set up parity bits: if stty will accept both odd and even
	   tell line to accept all 8 bits and forget about checking.
	   otherwise, enable parity checking and set whatever desired.
	*/
	if (tp->t_flags & EVENP)
		if (tp->t_flags & ODDP) lpr =| bits8;
		   else                 lpr =| bits7|penable;
	else			        lpr =| bits7|penable|opar;
	/* if appropriate, insert here setting of twosb-bit etc.
	*/
	dzaddr->dzlpr = lpr;
	spl0();
}

/*
 *	STTY and GTTY for DZ-11
 */
dzsgtty(dev, av)
	int *av;
{
	register struct tty *tp;
	register int    r;

	tp = &dz11[dev.d_minor];
	if ( ttystty(tp, av) )
		return;
	if (dzspeedtab[tp->t_speeds.lobyte] == 0)	/* catch bad speeds */
		tp->t_speeds= 7+(7<<8);	       		/* default speed 300 baud */
	dzparam(tp);
}

/*
 *	Open one of the lines, and return when connection is established.
 */
dzlopen(dev)
{
	register struct tty *tp;
	register thisline;

	tp = &dz11[dev.d_minor];
	thisline = (000401<<(dev.d_minor));
	dzaddr->dzstatus =| thisline;	/* set data term. ready and xmit enb */
	if (dzaddr->dzcarr & thisline)  /* make sure this expands as BITB    */
		tp->t_state =| CARR_ON;
	while ((tp->t_state & CARR_ON) == 0)
		sleep (&tp->t_rawq, TTIPRI);
}

/*
 *	Close one of the DZ-11 lines
 */
dzlclose(dev)
{
	register struct tty *tp;
	register line;

	tp = &dz11[dev.d_minor];
	if (tp->t_flags & HUPCL) {	/* if stty has hupcl mode, */
		line = (1<<dev.d_minor);/* remove data term. ready */
	dzaddr->dzstatus.hibyte =& ~line; /* must generate BICB    */
	}
}
/*
 * 	Watch the carrier every second (time-out loop).
 *	Hang up people loosing their carrier, wake up people
 *	sleeping on a carrier to come in.
 */
int	dzstatus;
dztimer()

{
	register struct tty *tp;
	register char *c;
	register int diff;
	int      mask, line;

	if ( (c=dzaddr->dzcarr) != dzstatus ) {
		mask = 0200; line = 7; diff = ((c^dzstatus)<<8);
		do {
		   if (diff < 0) {
				tp = &dz11[line];
				wakeup(tp);
			        if (!(c&mask)) {
				   if ((tp->t_state & WOPEN) == 0) {
					signal (tp, SIGHUP);
					flushtty(tp);
				   }
				   tp->t_state =& ~CARR_ON;
				} else {
				   tp->t_state =| CARR_ON;
				}
		   }
		   line--; mask =>> 1;
		} while ( diff =<< 1);
		dzstatus = c;
	}
	timeout (dztimer, 0, HZ);
}
