#
/*
 *  tape -- manipulate mag-tapes
 *  
 *  usage:
 * 	% tape   [<unit>]  [<command> [<parameters>]] 
 *  
 *  
 *  	<command>	<parameters>
 *  
 *  	skip		[<count>] [files|records] [forward|backward]
 *  	bksp		[<count>] [files|records]
 *	endfile		[<count>]
 *  	rewind
 *  	unload
 *  	<null>
 *  
 *  also recognized anywhere in a command:
 *  
 *  	status		print status after command
 *  	-556		set density to 556 bpi
 *  	-800		set density to 800 bpi
 *  	-800D		set density to 800 dump mode
 *  	-dump		set density to 800 dump mode
 *  	-1600		set density to 1600 bpi
 *  
 *  the following notes also apply:
 *  
 *  	--a missing <count> defaults to 1
 *
 *  	--if direction is not specified, it is assumed to be "forward"
 *
 *  	--"files" is assumed unless "records" is explicitly specified
 *
 *  	--default parity, density  = odd, 800dump and rewind on close.
 *	  (these defaults are determined by the system, not this program.)
 *
 *  	--most commands are uniquely recognizable by the first character
 *  	  or so, and may be abbreviated
 *
 *	--The command's syntax is really more flexible than this documentation
 *  	  implies. The following examples will work (although it may be
 *	  confusing to remember them):
 *  
 *  		%tape 2 rewind
 *  		%tape rewind 2
 *  		%tape 2 skip 3 files forward	(skip forward 3 files on unit 2)
 *  		%tape skip 3 files 2 		(same as above)
 *  		%tape skip files 3 2		(same as above)
 *  		%tape skip 3 2 			(same as above)
 *		%tape forw fil 2 skip 3		(same as above)
 *  
 *page
/*
 *	program definitions
 */

#define FORW	1
#define REV	2
#define FILE	1
#define REC	2
#define READ	0
#define WRITE	1
#define EVEN	04000
#define ODD	0

#define NOP	0
#define SKIP	1
#define REW	2
#define UNL	3
#define WEOF	4

#define CLREW	0100000	/* no rewind on close */
#define DENMASK	 060000 /* bits for density */
#define NINE	 010000	/* unit is 9-track */
#define PEVEN	  04000 /* even parity selected */
#define UMASK	0174000 /* bits available to user */
#define DUMP	 050000 /* dump mode density */

char statf	0;	/* status requested */
char countf	0;	/* count expected (overides unitf) */
char unitf	1;	/* unit number expected  */
char parf	0;	/* parity changed */
char denf	0;	/* density changed */
char rewf	0;	/* rewind flag changed */
char unitsp	0;	/* unit number specified */
char xflg	0;	/* eXtra info flag */
char errorf	0;	/* set to indicate syntax error */

int count	1;	/* skip count */
int direction	FORW;	/* direction */
int fil_or_rec	FILE;	/* files or records */
int rw		READ;	/* read or write? */
int density	DUMP;	/* density */
int parity	ODD;	/* parity */
int no_rewind	0;	/* 1  => no rewind on close */
int mode[3];		/* stty vector */

char tape[] "/dev/rmt1";  /* tape drive to use */
int fd;			/* file descriptor for tape */

char *command[] {
	"status","skip","even","odd","-556","-800","-dump","-1600",
	"rewind","unload","forwards","bksp","backwards","files","records",
	"weof","endfile","eof","-rewind","+rewind","rewind","on","unit","reverse",0};
int pden[] { 0, 020000, 040000, 060000};
char *xden[] {"556","800","dump","1600"};
/*page*/
main(argc,argp)
char **argp;
{
	int i,j,com,num;
	char *a,**b;

	if(argc < 2) help();
	argp++;
	com = NOP;
	while(--argc > 0){
		a = *argp++;
#define elf else if

		if(match(a,"status")){
			statf++;
		} elf(match(a,"skip")) {
			com = SKIP;
			countf++;
			rewf++;		/* disable rewinds */
			no_rewind = 1;
		} elf(match(a,"-x")) {
			xflg++;
		} elf(match(a,"even")) {
			parity = EVEN;
			parf++;
		} elf(match(a,"odd")) {
			parity = ODD;
			parf++;
		} elf(match(a,"-556")) {
			denf++;
			density = pden[0];
		} elf(match(a,"-800")) {
			denf++;
			density = pden[1];
		} elf(match(a,"-dump")) {
			denf++;
			density = pden[2];
		} elf(match(a,"-1600")) {
			denf++;
			density = pden[3];
		} elf(match(a,"rewind")) {
			com = REW;
		} elf(match(a,"unload")) {
			com = UNL;
		} elf(match(a,"forward")) {
			direction = FORW;
		} elf(match(a,"bksp")) {
			com = SKIP;
			direction = REV;
			countf++;
		} elf(match(a,"backwards") || match(a,"reverse")) {
			direction = REV;
		} elf(match(a,"files")) {
			fil_or_rec = FILE;
		} elf(match(a,"records")) {
			fil_or_rec = REC;
		} elf(match(a,"endfile")
		   || match(a,"eof")
		   || match(a,"weof")) {
			com = WEOF;
			countf++;
			rw = WRITE;
		} elf(match(a,"-rewind")) {
			rewf++;
			no_rewind = 1;
		} elf(match(a,"+rewind")) {
			rewf++;
			no_rewind = 0;
		} elf(match(a,"unit") || match(a,"on")) {
			unitf++;
		} elf( (num = number(a)) >= 0){
			if(countf){
				count = num;
				countf = 0;
			} else if(unitf && num < 8){
				tape[8] = *a;
				unitf = 0;
				unitsp++;
			} else {
					printf("illegal number: %s\n",a);
					errorf++;
				}
		} else {
			printf("bad keyword: %s\n", a);
			errorf++;
		}
	}
	if(errorf)
		exit(3);
/*
 *	special case for "status all":
 */
	if(!unitsp && com == NOP && statf){
		stat_all();
		exit(0);
	}
	topen(0);
	if(parf || denf || rewf)
		newmode();
	if(com != NOP)
		ex_command(com);
	if(statf)
		status();
	exit(0);
}
/*page
/* device setup routine */

newmode()
{
	register int mask,val;

	if(getm(fd,mode) < 0){
		printf("getty # 1 failed\n");
		exit(10);
	}
	mask = UMASK;
	*mode =& UMASK;
	val = 0;
	if(parf){
		mask =~PEVEN;
		val =| parity;
	}
	if(denf){
		mask =& ~DENMASK;
		val =| (density&DENMASK);
	}
	if(rewf){
		mask =& ~CLREW;
		val =| (no_rewind? 0 : CLREW);
	}
/* printf("mode, mask, val:  %o, %o, %o\n", *mode, mask, val); */
	if((*mode&~mask) == val){
		statf = 0;
		return;
	}
	*mode =& mask;
	*mode =| val;
	mode[1] = 0;
	mode[2] = 0;
	if(setm(fd,mode) < 0){
		printf("newmode setty failed!\n");
		exit(12);
	}
	status();
}

topen(cont)
{
	if((fd = open(tape,rw)) < 0){
		printf("%s: can't open\n",tape);
		if(!cont)
			exit(1);
	}
	return(fd < 0);
}

tclose()
{
	close(fd);
}
/*page
/* execute commands */

ex_command(com)
{
	switch(com){
	default:
		exit(20);
	case NOP:
		return;
	case UNL:
		com = 7;
		break;
	case REW:
		com = 6;
		break;
	case WEOF:
		com = 1;
		while (count-- > 0)
			tm_com(com,0);
		return;
	case SKIP:
		com = 2;
		if(direction == REV)
			com =+ 1;
		if(fil_or_rec == REC)
			com =+ 2;
		break;
	}
	tm_com(com,count);
}

tm_com(com,cnt)
{
	mode[0] = com;
	mode[1] = cnt;
	mode[2] = 0;

	if(setm(fd,mode) < 0){
		printf("operation could not be performed\n");
		exit(24);
	}
}
/*page*/
stat_all()
{
	register char i;

	for(i='0'; i<'3'; i++){
		tape[8] = i;
		if(!topen(1)){
			status();
			tclose();
		}
	}
}
status()
{
	int n;

	if(getm(fd,mode) < 0){
		printf("can't get status from %s\n",tape);
		exit(30);
	}
	printf("%s:\t",tape);
	printf("%s parity, ",( *mode&PEVEN ? "even" : "odd"));
	n = (*mode>>13) & 03;
	printf("%s %s",xden[n],(n==2? "mode" : "bpi"));
	if(*mode&NINE)
		printf(",  9-trk");
	if((*mode&CLREW) == 0)
		printf(", -rew");
	if(xflg)
		printf(",  (lastc = %d/%d)\n",mode[1],mode[2]);
	putchar('\n');
}

help()
{
	int i;
	char *c, **p;

	printf("keywords are:\n");
	p = command;
	while(c = *p++)
		printf("%s\n",c);
	exit(0);
}
/*page*/
/* lower level stuff */

match(s1,s2)
char *s1,*s2;
{
	while(*s1 != 0)
		if(*s1++ != *s2++)
			return(0);
	return(1);
}

number(s)
char *s;
{
	register int n;
	register char c,*p;

	p = s;
	n = 0;
	while((c = *p++) >= '0' && c <= '9')
		n = n*10 + (c - '0');
	if(c != 0)
		return(-1);
	else
		return(n);
}

setm(f,v)
int *v;
{
	int k;

	return(stty(f,v));
	printf("setm: %2d/ %6o, %6o, %6o;  ",f,v[0],v[1],v[2]);
	k = stty(f,v);
	printf("%d\n",k);
	return(k);
}

getm(f,v)
int *v;
{
	int k;

	return(gtty(f,v));
	printf("getm: %2d/ ",f);
	k = gtty(f,v);
	printf("%6o, %6o, %6o;  %d\n",v[0],v[1],v[2],k);
	return(k);
}
