/*
 * sound.c
 * Original SB16 implementation done by Alex Nash.
 *
 * Sound Blaster 16 Driver Entry Points
 *
 */

#include	<sys/coherent.h>

#include 	<sys/cred.h>
#include 	<sys/devices.h>
#include 	<sys/errno.h>
#include 	<sys/file.h>
#include 	<sys/sound.h>
#include 	<sys/stat.h>

/*--------------------------------------------------------------------------*/
/*	static variables														*/
/*--------------------------------------------------------------------------*/
static int	snd_Installed;

/*----------------------------------------------------------------------------
|	void
|	snd_open (dev_t dev, int mode)
|
|	dev			Device number
|	mode		Open mode
|
|	Opens the selected device if available.  The user's errno is set to
|	ENXIO if the driver was unable to initialize properly during load time.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_open(o_dev_t dev, int mode,
  int __NOTUSED(flags), __cred_t * __NOTUSED(credp))
#else
static void
snd_open (dev, mode, flags, credp)
o_dev_t		dev;
int			mode;
int			flags;
__cred_t	* credp;
#endif
{
	int	retCode;

	if (snd_Installed == 0)
	{
		set_user_error(ENXIO);
		return;
	}
	else
	{
		switch (minor(dev))
		{
			case SOUND_DSP_MINOR:
				retCode	= (*snd_dsp_ftp->open)();
				break;

			case SOUND_MIXER_MINOR:
				retCode	= 0;
				break;

			default:
				retCode	= ENODEV;
		}
	}

	if (retCode != 0)
		set_user_error(retCode);
}

/*----------------------------------------------------------------------------
|	void
|	snd_close (dev_t dev, int mode)
|
|	dev			Device number
|	mode		Mode the device was opened in
|
|	Closes the device.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_close(o_dev_t dev, int mode,
  int __NOTUSED(flags), __cred_t * __NOTUSED(credp))
#else
static void
snd_close(dev, mode, flags, credp)
o_dev_t		dev;
int			mode;
int			flags;
__cred_t	* credp;
#endif
{
	int	retCode;

	switch (minor(dev))
	{
		case SOUND_DSP_MINOR:
			retCode	= (*snd_dsp_ftp->close)();
			break;

		default:
			retCode	= 0;
			break;
	}

	if (retCode != 0)
		set_user_error(retCode);
}

/*----------------------------------------------------------------------------
|	void
|	snd_read (dev_t dev, IO *iop)
|
|	dev			Device number
|	iop			I/O request structure
|
|	Reads data from the device, or sets the user's errno to ENXIO if they
|	are attempting to read from the mixer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_read(o_dev_t dev, IO * iop, __cred_t * __NOTUSED(credp))
#else
static void
snd_read(dev, iop, credp)
o_dev_t		dev;
IO			* iop;
__cred_t	* credp;
#endif
{
	int	retCode	= ENXIO;

	switch (minor(dev))
	{
		case SOUND_DSP_MINOR:
			retCode	= (*snd_dsp_ftp->read)(iop);
			break;

		default:
			break;
	}

	if (retCode != 0)
		set_user_error(retCode);
}

/*----------------------------------------------------------------------------
|	void
|	snd_write (dev_t dev, IO *iop)
|
|	dev			Device number
|	iop			I/O request structure
|
|	Writes data to the device, or sets the user's errno to ENXIO if they
|	are attempting to write to the mixer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_write(o_dev_t dev, IO * iop, __cred_t * __NOTUSED(credp))
#else
static void
snd_write(dev, iop, credp)
o_dev_t		dev;
IO			* iop;
__cred_t	* credp;
#endif
{
	int	retCode	= ENXIO;

	switch (minor(dev))
	{
		case SOUND_DSP_MINOR:
			retCode	= (*snd_dsp_ftp->write)(iop);
			break;

		default:
			break;
	}

	if (retCode != 0)
		set_user_error(retCode);
}

/*----------------------------------------------------------------------------
|	void
|	snd_ioctl (dev_t dev, int cmd, char *data, int mode,
|			   cred_t credptr, int *retptr)
|
|	dev			Device number
|	cmd			IOCTL command
|	data		Pointer to user data
|	mode		Mode the device was opened with
|	credptr		Credentials pointer
|	retptr		Return value pointer
|
|	Dispatches ioctl commands to the DSP or mixer devices.
|
|	Returns 0 to the caller (through retptr) on success, or -1 on failure.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_ioctl(o_dev_t dev, int cmd, __VOID__ * par, int mode,
  __cred_t * __NOTUSED(credp), int * rvalp)
#else
static void
snd_ioctl (dev, cmd, par, mode, credp, rvalp)
o_dev_t		dev;
int			cmd;
char		* par;
int			mode;
__cred_t	* credp;
int			* rvalp;
#endif
{
	int	retCode;

	switch (minor(dev))
	{
		case SOUND_DSP_MINOR:
			retCode	= (*snd_dsp_ftp->ioctl)(cmd, par);
			break;

		case SOUND_MIXER_MINOR:
			retCode	= (*snd_mixer_ftp->ioctl)(cmd, par, mode);
			break;

		default:
			retCode	= ENODEV;
			break;
	}

	if (retCode != 0)
	{
		set_user_error(retCode);
		*rvalp	= -1;
	}
	else
		*rvalp	= 0;
}

/*----------------------------------------------------------------------------
|	void
|	snd_load ()
|
|	Prints an informational message describing the base I/O port, 8/16 bit
|	DMA channels, and IRQ.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_load (void)
#else
static void
snd_load ()
#endif
{
	snd_Installed	= 0;

	if ((*snd_dsp_ftp->init)() == 0 && (*snd_mixer_ftp->init)() == 0)
	{
		snd_Installed	= 1;
	}
}

/*----------------------------------------------------------------------------
|	void
|	snd_uload ()
|
|	Unload routine.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void snd_uload (void)
#else
static void
snd_uload ()
#endif
{
}

/*--------------------------------------------------------------------------*/
/*	CON structure															*/
/*--------------------------------------------------------------------------*/
struct con soundcon	=
{
	DFCHR,						/* flags 		*/
	0,							/* device major	*/
	snd_open,					/* open			*/
	snd_close,					/* close		*/
	NULL,						/* block		*/
	snd_read,					/* read			*/
	snd_write,					/* write		*/
	snd_ioctl,					/* ioctl		*/
	NULL,						/* powerfail	*/
	NULL,						/* timeout		*/
	snd_load,					/* load			*/
	snd_uload,					/* unload		*/
	NULL						/* poll			*/
};
