/* pdp18b_sys.c: 18b PDP's simulator interface

   Copyright (c) 1993, 1994, 1995, 1996, Robert M Supnik,
   Digital Equipment Corporation
   All rights reserved
*/

#include "pdp18b_defs.h"
#include <ctype.h>

extern DEVICE cpu_dev;
extern DEVICE ptr_dev, ptp_dev;
extern DEVICE tti_dev, tto_dev;
extern DEVICE clk_dev;
extern DEVICE lpt_dev;
#if defined (DRM)
extern DEVICE drm_dev;
#endif
#if defined (RF)
extern DEVICE rf_dev;
#endif
#if defined (RP)
extern DEVICE rp_dev;
#endif
#if defined (MTA)
extern DEVICE mt_dev;
#endif
extern REG cpu_reg[];
extern unsigned int M[];
extern int memm;
extern char *get_glyph (char *cptr, char *gbuf, char term);
extern int get_unum (char *cptr, int radix, unsigned int max,
	unsigned int *val);

/* SCP data structures and interface routines

   sim_name		simulator name string
   sim_PC		pointer to saved PC register descriptor
   sim_emax		number of words for examine
   sim_devices		array of pointers to simulated devices
   sim_stop_messages	array of pointers to stop messages
   sim_load		binary loader
*/

#if defined (PDP4)
char sim_name[] = "PDP-4";
#elif defined (PDP7)
char sim_name[] = "PDP-7";
#elif defined (PDP9)
char sim_name[] = "PDP-9";
#elif defined (PDP15)
char sim_name[] = "PDP-15";
#endif

REG *sim_PC = &cpu_reg[0];

int sim_emax = 3;

DEVICE *sim_devices[] = { &cpu_dev,
	&ptr_dev, &ptp_dev, &tti_dev, &tto_dev,
	&clk_dev, &lpt_dev,
#if defined (DRM)
	&drm_dev,
#endif
#if defined (RF)
	&rf_dev,
#endif
#if defined (RP)
	&rp_dev,
#endif
#if defined (MTA)
	&mt_dev,
#endif
	NULL };

const char *sim_stop_messages[] = {
	"Unknown error",
	"Undefined instruction",
	"HALT instruction",
	"Breakpoint",
	"Nested XCT's"  };

/* Binary loader

   Not implemented: format not documented
*/

int sim_load (FILE *fileref)
{
return SCPE_FMT;					/* error */
}

/* Symbol tables */

#define I_V_FL		18				/* inst class */
#define	I_M_FL		017				/* class mask */
#define I_V_DC		22				/* default count */
#define I_V_NPN		0				/* no operand */
#define I_V_NPI		1				/* no operand IOT */
#define I_V_IOT		2				/* IOT */
#define I_V_MRF		3				/* memory reference */
#define I_V_OPR		4				/* OPR */
#define I_V_LAW		5				/* LAW */
#define I_V_XR		6				/* index */
#define I_V_XR9		7				/* index literal */
#define I_V_EST		8				/* EAE setup */
#define I_V_ESH		9				/* EAE shift */
#define I_V_EMD		10				/* EAE mul-div */
#define I_NPN		(I_V_NPN << I_V_FL)		/* no operand */
#define I_NPI		(I_V_NPI << I_V_FL)		/* no operand IOT */
#define I_IOT		(I_V_IOT << I_V_FL)		/* IOT */
#define I_MRF		(I_V_MRF << I_V_FL)		/* memory reference */
#define I_OPR		(I_V_OPR << I_V_FL)		/* OPR */
#define I_LAW		(I_V_LAW << I_V_FL)		/* LAW */
#define I_XR		(I_V_XR << I_V_FL)		/* index */
#define I_XR9		(I_V_XR9 << I_V_FL)		/* index literal */
#define I_EST		(I_V_EST << I_V_FL)		/* EAE setup */
#define I_ESH		(I_V_ESH << I_V_FL)		/* EAE shift */
#define I_EMD		(I_V_EMD << I_V_FL)		/* EAE mul-div */
#define MD(x) ((I_EMD) + ((x) << I_V_DC))

static const int masks[] = {
 0777777, 0777767, 0740000, 0740000,
 0763730, 0760000, 0777000, 0777000,
 0740700, 0760700, 0777700 };

static const char *opcode[] = {
 "CAL", "DAC", "JMS", "DZM",				/* mem refs */
 "LAC", "XOR", "ADD", "TAD",
 "XCT", "ISZ", "AND", "SAD",
 "JMP", "LAW",

 "LACQ", "LACS", "ABS", "GSM", "LMQ",			/* EAE */
 "MUL", "MULS", "DIV", "DIVS",
 "IDIV", "IDIVS", "FRDIV", "FRDIVS",
 "NORM", "NORMS",
 "MUY", "LLK MUY", "DVI", "LLK DVI",
 "NMI", "NMIS", "LRS", "LRSS",
 "LLS", "LLSS", "ALS", "ALSS",
 "EAE-setup", "EAE",					/* setup, general */

 "CLSF", "IOF", "ION", "CLOF", "CLON",			/* standard IO devs */
 "RSF", "RRB", "RCF", "RSA", "RSB",
 "PSF", "PCF", "PSA", "PSB", "PLS",
 "KSF", "KRB", "KCF", "IORS", "IOOS",
 "TSF", "TCF", "TPC", "TLS",
#if defined (TYPE62)					/* PDP-4 LPT */
 "LPSF", "LPCF", "LPLD", "LPSE",
 "LSSF", "LSCF", "LSPR",
#elif defined (TYPE647)					/* PDP-7, PDP-9 LPT */
 "LPSF", "LPCB", "LPCD", "LPCD", "LPCD",
 "LPL2", "LPLD", "LPL1",
 "LPEF", "LPCF", "LPCF", "LPCF", "LPCF",
 "LPPB", "LPLS", "LPPS",
#elif defined (LP15)
 "LPSF", "LPPM", "LPP1", "LPDI",
 "LPRS", "LPOS", "LPEI", "LPCD", "LPCF",
#endif
#if defined (DRM)					/* drum */
 "DRLR", "DRLW", "DRSS", "DRCS",
 "DRSF", "DRSN", "DRCF",
 "DRLCRD", "DRLCWR", "DRLBLK", "DRCONT",
 "DRSF", "DRSOK", "DRCF",
#endif
#if defined (RF)					/* RF09 */
 "DSSF", "DSCC", "DSCF",
 "DRBR", "DRAL", "DSFX", "DRAH",
 "DLBR", "DLAL", "DSCN", "DLAH",
 "DLOK",         "DSCD", "DSRS",
 "DGHS", "DGSS",
#endif
#if defined (RP)
 "DPSF", "DPSA", "DPSJ", "DPSE",
 "DPRSA", "DPOSA", "DPRSB", "DPOSB",
 "DPRM", "DPOM",
 "DPLA", "DPCS", "DPCA", "DPWC",
 "DPLM", "DPEM", "DPSN",
 "DPRU", "DPOU", "DPRA", "DPOA",
 "DPRC", "DPOC", "DPRW", "DPOW",
 "DPCF", "DPLZ", "DPCN", "DPLO", "DPLF",
#endif
#if defined (MTA)					/* TC59 */
 "MTTR", "MTCR", "MTSF", "MTRC", "MTAF",
 "MTRS", "MTGO", "MTCM", "MTLC",
#endif
#if defined (PDP7)
 "ITON", "TTS", "SKP7", "CAF",
 "SEM", "EEM", "EMIR", "LEM",
#elif defined (PDP9)
 "PFSF", "TTS", "SKP7", "CAF",
 "DBR", "SEM", "EEM", "LEM",
 "MPSK", "MPSNE", "MPCV", "MPEU",
 "MPLD", "MPCNE",
 "LPDI", "LPEI",
#elif defined (PDP15)
 "PFSF", "TTS", "SPCO", "CAF",
 "DBR", "SKP15", "SBA", "DBA", "EBA",
 "PAX", "PAL", "AAC", "PXA",
 "AXS", "PXL", "PLA", "PLX",
 "CLX", "CLL", "AXR",
#endif
 "IOT",							/* general */

 "NOP", "STL", "RCL", "RCR",
 "CLC", "LAS", "GLK",
 "OPR", "SMA", "SZA", "SZA SMA",
 "SNL", "SNL SMA", "SNL SZA", "SNL SZA SMA",
 "SKP", "SPA", "SNA", "SNA SPA",
 "SZL", "SZL SPA", "SZL SNA", "SZL SZA SPA",
 "RAL", "SMA RAL", "SZA RAL", "SZA SMA RAL",
 "SNL RAL", "SNL SMA RAL", "SNL SZA RAL", "SNL SZA SMA RAL",
 "SKP RAL", "SPA RAL", "SNA RAL", "SNA SPA RAL",
 "SZL RAL", "SZL SPA RAL", "SZL SNA RAL", "SZL SZA SPA RAL",
 "RAR", "SMA RAR", "SZA RAR", "SZA SMA RAR",
 "SNL RAR", "SNL SMA RAR", "SNL SZA RAR", "SNL SZA SMA RAR",
 "SKP RAR", "SPA RAR", "SNA RAR", "SNA SPA RAR",
 "SZL RAR", "SZL SPA RAR", "SZL SNA RAR", "SZL SZA SPA RAR",
#if defined (PDP15)
 "IAC", "SMA IAC", "SZA IAC", "SZA SMA IAC",
 "SNL IAC", "SNL SMA IAC", "SNL SZA IAC", "SNL SZA SMA IAC",
 "SKP IAC", "SPA IAC", "SNA IAC", "SNA SPA IAC",
 "SZL IAC", "SZL SPA IAC", "SZL SNA IAC", "SZL SZA SPA IAC",
#else
 "RAL RAR", "SMA RAL RAR", "SZA RAL RAR", "SZA SMA RAL RAR",
 "SNL RAL RAR", "SNL SMA RAL RAR", "SNL SZA RAL RAR", "SNL SZA SMA RAL RAR",
 "SKP RAL RAR", "SPA RAL RAR", "SNA RAL RAR", "SNA SPA RAL RAR",
 "SZL RAL RAR", "SZL SPA RAL RAR", "SZL SNA RAL RAR", "SZL SZA SPA RAL RAR",
#endif
 "RTWO", "SMA RTWO", "SZA RTWO", "SZA SMA RTWO",
 "SNL RTWO", "SNL SMA RTWO", "SNL SZA RTWO", "SNL SZA SMA RTWO",
 "SKP RTWO", "SPA RTWO", "SNA RTWO", "SNA SPA RTWO",
 "SZL RTWO", "SZL SPA RTWO", "SZL SNA RTWO", "SZL SZA SPA RTWO",
 "RTL", "SMA RTL", "SZA RTL", "SZA SMA RTL",
 "SNL RTL", "SNL SMA RTL", "SNL SZA RTL", "SNL SZA SMA RTL",
 "SKP RTL", "SPA RTL", "SNA RTL", "SNA SPA RTL",
 "SZL RTL", "SZL SPA RTL", "SZL SNA RTL", "SZL SZA SPA RTL",
 "RTR", "SMA RTR", "SZA RTR", "SZA SMA RTR",
 "SNL RTR", "SNL SMA RTR", "SNL SZA RTR", "SNL SZA SMA RTR",
 "SKP RTR", "SPA RTR", "SNA RTR", "SNA SPA RTR",
 "SZL RTR", "SZL SPA RTR", "SZL SNA RTR", "SZL SZA SPA RTR",
#if defined (PDP15)
 "BSW", "SMA BSW", "SZA BSW", "SZA SMA BSW",
 "SNL BSW", "SNL SMA BSW", "SNL SZA BSW", "SNL SZA SMA BSW",
 "SKP BSW", "SPA BSW", "SNA BSW", "SNA SPA BSW",
 "SZL BSW", "SZL SPA BSW", "SZL SNA BSW", "SZL SZA SPA BSW",
#else
 "RTL RTR", "SMA RTL RTR", "SZA RTL RTR", "SZA SMA RTL RTR",
 "SNL RTL RTR", "SNL SMA RTL RTR", "SNL SZA RTL RTR", "SNL SZA SMA RTL RTR",
 "SKP RTL RTR", "SPA RTL RTR", "SNA RTL RTR", "SNA SPA RTL RTR",
 "SZL RTL RTR", "SZL SPA RTL RTR", "SZL SNA RTL RTR", "SZL SZA SPA RTL RTR",
#endif

 "LLK", "CLQ", "LSN", "OACQ", "ECLA",			/* encode only masks */
 "CMQ", "OMQ", "OSC", 
 "CLA", "CLL", "CML", "CMA",
 "OAS", "HLT",
  NULL  };

static const int opc_val[] = {
 0000000+I_MRF, 0040000+I_MRF, 0100000+I_MRF, 0140000+I_MRF,
 0200000+I_MRF, 0240000+I_MRF, 0300000+I_MRF, 0340000+I_MRF,
 0400000+I_MRF, 0440000+I_MRF, 0500000+I_MRF, 0540000+I_MRF,
 0600000+I_MRF, 0760000+I_LAW,

 0641002+I_NPN, 0641001+I_NPN, 0644000+I_NPN, 0664000+I_NPN, 0652000+I_NPN,
 0653100+MD(022), 0657100+MD(022), 0640300+MD(023), 0644300+MD(023),
 0653300+MD(023), 0657300+MD(023), 0650300+MD(023), 0654300+MD(023),
 0640400+MD(044), 0660400+MD(044),
 0640100+I_ESH, 0660100+I_ESH, 0640300+I_ESH, 0660300+I_ESH,
 0640400+I_ESH, 0660400+I_ESH, 0640500+I_ESH, 0660500+I_ESH,
 0640600+I_ESH, 0660600+I_ESH, 0640700+I_ESH, 0660700+I_ESH,
 0640000+I_EST, 0640000+I_IOT, 

 0700001+I_NPI, 0700002+I_NPI, 0700042+I_NPI, 0700004+I_NPI, 0700044+I_NPI,
 0700101+I_NPI, 0700112+I_NPN, 0700102+I_NPI, 0700104+I_NPI, 0700144+I_NPI,
 0700201+I_NPI, 0700202+I_NPI, 0700204+I_NPI, 0700244+I_NPI, 0700206+I_NPI,
 0700301+I_NPI, 0700312+I_NPN, 0700302+I_NPI, 0700314+I_NPN, 0700304+I_NPI,
 0700401+I_NPI, 0700402+I_NPI, 0700404+I_NPI, 0700406+I_NPI,
#if defined (TYPE62)
 0706501+I_NPI, 0706502+I_NPI, 0706542+I_NPI, 0706506+I_NPI,
 0706601+I_NPI, 0706602+I_NPI, 0706606+I_NPI,
#elif defined (TYPE647)
 0706501+I_NPI, 0706502+I_NPI, 0706522+I_NPI, 0706542+I_NPI, 0706562+I_NPI,
 0706526+I_NPI, 0706546+I_NPI, 0706566+I_NPI,
 0706601+I_NPI, 0706602+I_NPI, 0706622+I_NPI, 0706642+I_NPI, 0706662+I_NPI, 
 0706606+I_NPI, 0706626+I_NPI, 0706646+I_NPI,
#elif defined (LP15)
 0706501+I_NPI, 0706521+I_NPI, 0706541+I_NPI, 0706561+I_NPI,
 0706552+I_NPN, 0706542+I_NPI, 0706544+I_NPI, 0706621+I_NPI, 0706641+I_NPI,
#endif
#if defined (DRM)
 0706006+I_NPI, 0706046+I_NPI, 0706106+I_NPI, 0706204+I_NPI,
 0706101+I_NPI, 0706201+I_NPI, 0706102+I_NPI,
 0706006+I_NPI, 0706046+I_NPI, 0706106+I_NPI, 0706204+I_NPI,
 0706101+I_NPI, 0706201+I_NPI, 0706102+I_NPI,
#endif
#if defined (RF)
 0707001+I_NPI, 0707021+I_NPI, 0707041+I_NPI,
 0707002+I_NPI, 0707022+I_NPI, 0707042+I_NPI, 0707062+I_NPI,
 0707004+I_NPI, 0707024+I_NPI, 0707044+I_NPI, 0707064+I_NPI,
 0707202+I_NPI,                0707242+I_NPI, 0707262+I_NPI,
 0707204+I_NPI, 0707224+I_NPI,
#endif
#if defined (RP)
 0706301+I_NPI, 0706321+I_NPI, 0706341+I_NPI, 0706361+I_NPI,
 0706312+I_NPN, 0706302+I_NPI, 0706332+I_NPN, 0706322+I_NPI, 
 0706342+I_NPN, 0706352+I_NPI,
 0706304+I_NPI, 0706324+I_NPI, 0706344+I_NPI, 0706364+I_NPI,
 0706411+I_NPN, 0706401+I_NPI, 0706421+I_NPI,
 0706412+I_NPN, 0706402+I_NPI, 0706432+I_NPN, 0706422+I_NPI, 
 0706452+I_NPN, 0706442+I_NPI, 0706472+I_NPN, 0706462+I_NPI, 
 0706404+I_NPI, 0706424+I_NPI, 0706454+I_NPN, 0706444+I_NPI, 0706464+I_NPI,
#endif 
#if defined (MTA)
 0707301+I_NPI, 0707321+I_NPI, 0707341+I_NPI, 0707312+I_NPN, 0707322+I_NPI,
 0707352+I_NPN, 0707304+I_NPI, 0707324+I_NPI, 0707326+I_NPI, 
#endif
#if defined (PDP7)
 0703201+I_NPI, 0703301+I_NPI, 0703341+I_NPI, 0703302+I_NPI,
 0707701+I_NPI, 0707702+I_NPI, 0707742+I_NPI, 0707704+I_NPI,
#elif defined (PDP9)
 0700062+I_NPI, 0703301+I_NPI, 0703341+I_NPI, 0703302+I_NPI,
 0703344+I_NPI, 0707701+I_NPI, 0707702+I_NPI, 0707704+I_NPI,
 0701701+I_NPI, 0701741+I_NPI, 0701702+I_NPI, 0701742+I_NPI,
 0701704+I_NPI, 0701644+I_NPI,
 0706504+I_NPI, 0706604+I_NPI,
#elif defined (PDP15)
 0700062+I_NPI, 0703301+I_NPI, 0703341+I_NPI, 0703302+I_NPI,
 0703344+I_NPI, 0707741+I_NPI, 0707761+I_NPI, 0707762+I_NPI, 0707764+I_NPI,
 0721000+I_XR, 0722000+I_XR, 0723000+I_XR9, 0724000+I_XR,
 0725000+I_XR9, 0726000+I_XR, 0730000+I_XR, 0731000+I_XR,
 0735000+I_XR, 0736000+I_XR, 0737000+I_XR9,
#endif
 0700000+I_IOT,

 0740000+I_NPN, 0744002+I_NPN, 0744010+I_NPN, 0744020+I_NPN,
 0750001+I_NPN, 0750004+I_NPN, 0750010+I_NPN,
 0740000+I_OPR, 0740100+I_OPR, 0740200+I_OPR, 0740300+I_OPR,
 0740400+I_OPR, 0740500+I_OPR, 0740600+I_OPR, 0740700+I_OPR,
 0741000+I_OPR, 0741100+I_OPR, 0741200+I_OPR, 0741300+I_OPR,
 0741400+I_OPR, 0741500+I_OPR, 0741600+I_OPR, 0741700+I_OPR,
 0740010+I_OPR, 0740110+I_OPR, 0740210+I_OPR, 0740310+I_OPR,
 0740410+I_OPR, 0740510+I_OPR, 0740610+I_OPR, 0740710+I_OPR,
 0741010+I_OPR, 0741110+I_OPR, 0741210+I_OPR, 0741310+I_OPR,
 0741410+I_OPR, 0741510+I_OPR, 0741610+I_OPR, 0741710+I_OPR,
 0740020+I_OPR, 0740120+I_OPR, 0740220+I_OPR, 0740320+I_OPR,
 0740420+I_OPR, 0740520+I_OPR, 0740620+I_OPR, 0740720+I_OPR,
 0741020+I_OPR, 0741120+I_OPR, 0741220+I_OPR, 0741320+I_OPR,
 0741420+I_OPR, 0741520+I_OPR, 0741620+I_OPR, 0741720+I_OPR,
 0740030+I_OPR, 0740130+I_OPR, 0740230+I_OPR, 0740330+I_OPR,
 0740430+I_OPR, 0740530+I_OPR, 0740630+I_OPR, 0740730+I_OPR,
 0741030+I_OPR, 0741130+I_OPR, 0741230+I_OPR, 0741330+I_OPR,
 0741430+I_OPR, 0741530+I_OPR, 0741630+I_OPR, 0741730+I_OPR,
 0742000+I_OPR, 0742100+I_OPR, 0742200+I_OPR, 0742300+I_OPR,
 0742400+I_OPR, 0742500+I_OPR, 0742600+I_OPR, 0742700+I_OPR,
 0743000+I_OPR, 0743100+I_OPR, 0743200+I_OPR, 0743300+I_OPR,
 0743400+I_OPR, 0743500+I_OPR, 0743600+I_OPR, 0743700+I_OPR,
 0742010+I_OPR, 0742110+I_OPR, 0742210+I_OPR, 0742310+I_OPR,
 0742410+I_OPR, 0742510+I_OPR, 0742610+I_OPR, 0742710+I_OPR,
 0743010+I_OPR, 0743110+I_OPR, 0743210+I_OPR, 0743310+I_OPR,
 0743410+I_OPR, 0743510+I_OPR, 0743610+I_OPR, 0743710+I_OPR,
 0742020+I_OPR, 0742120+I_OPR, 0742220+I_OPR, 0742320+I_OPR,
 0742420+I_OPR, 0742520+I_OPR, 0742620+I_OPR, 0742720+I_OPR,
 0743020+I_OPR, 0743120+I_OPR, 0743220+I_OPR, 0743320+I_OPR,
 0743420+I_OPR, 0743520+I_OPR, 0743620+I_OPR, 0743720+I_OPR,
 0742030+I_OPR, 0742130+I_OPR, 0742230+I_OPR, 0742330+I_OPR,
 0742430+I_OPR, 0742530+I_OPR, 0742630+I_OPR, 0742730+I_OPR,
 0743030+I_OPR, 0743130+I_OPR, 0743230+I_OPR, 0743330+I_OPR,
 0743430+I_OPR, 0743530+I_OPR, 0743630+I_OPR, 0743730+I_OPR,

 0660000+I_EST, 0650000+I_EST, 0644000+I_EST, 0642000+I_EST, 0641000+I_EST,
 0640004+I_EST, 0640002+I_EST, 0640001+I_EST,
 0750000+I_OPR, 0744000+I_OPR, 0740002+I_OPR, 0740001+I_OPR,
 0740004+I_OPR, 0740040+I_OPR,
 -1 };

/* Operate or EAE decode

   Inputs:
	inst	=	mask bits
	class	=	instruction class code
	sp	=	space needed?
   Outputs:
	status	=	space needed?
*/

int oprf (int inst, int class, int sp)
{
int i, j;

for (i = 0; opc_val[i] >= 0; i++) {			/* loop thru ops */
	j = (opc_val[i] >> I_V_FL) & I_M_FL;		/* get class */
	if ((j == class) && (opc_val[i] & inst)) {	/* same class? */
		inst = inst & ~opc_val[i];		/* mask bit set? */
		printf ((sp? " %s": "%s"), opcode[i]);
		sp = 1;  }  }
return sp;
}

/* Symbolic decode

   Inputs:
	addr	=	current PC
	*val	=	pointer to values
	cflag	=	true if decoding for CPU
	sw	=	switches
   Outputs:
	return	=	status code
*/

#define FMTASC(x) ((x) < 040)? "<%03o>": "%c", (x)
#define SIXTOASC(x) (((x) >= 040)? (x): (x) + 0100)
int print_sym (unsigned int addr, unsigned int *val, int cflag, int sw)
{
int i, j, k, sp, inst, disp, ma;

if (sw & SWMASK ('A')) {				/* ASCII? */
	if (val[0] > 0377) return SCPE_ARG;
	printf (FMTASC (val[0] & 0177));
	return SCPE_OK;  }
if (sw & SWMASK ('C')) {				/* character? */
	putchar (SIXTOASC ((val[0] >> 12) & 077));
	putchar (SIXTOASC ((val[0] >> 6) & 077));
	putchar (SIXTOASC (val[0] & 077));
	return SCPE_OK;  }
#if defined (PDP15)
if (sw & SWMASK ('P')) {				/* packed ASCII? */
	printf (FMTASC ((val[0] >> 11) & 0177));
	printf (FMTASC ((val[0] >> 4) & 0177));
	printf (FMTASC (((val[0] << 3) | (val[1] >> 15)) & 0177));
	printf (FMTASC ((val[1] >> 8) & 0177));
	printf (FMTASC ((val[1] >> 1) & 0177));
	return -1;  }
#endif
if (!(sw & SWMASK ('M'))) return SCPE_ARG;

/* Instruction decode */

inst = val[0];
for (i = 0; opc_val[i] >= 0; i++) {			/* loop thru ops */
    j = (opc_val[i] >> I_V_FL) & I_M_FL;		/* get class */
    if ((opc_val[i] & 0777777) == (inst & masks[j])) {	/* match? */

	switch (j) {					/* case on class */
	case I_V_NPN:					/* no operands */
	case I_V_XR:					/* index no opers */
		printf ("%s", opcode[i]);		/* opcode */
		break;
	case I_V_NPI:					/* IOT no operand */
		printf ("%s", opcode[i]);		/* opcode */
		if (inst & 010) printf (" +10");
		break;
	case I_V_IOT:					/* IOT or EAE */
		printf ("%s %-o", opcode[i], inst & 037777);
		break;
	case I_V_MRF:					/* mem ref */
#if defined (PDP15)
		if (memm) {
			disp = inst & 017777;  
			ma = (addr & 0760000) | disp;  }
		else {	disp = inst & 007777;
			ma = (addr & 0770000) | disp;  }
#else
		disp = inst & 017777;
		ma = (addr & 0760000) | disp;
#endif
#if defined (PDP9) || (PDP15)
		if ((disp & ~07) == 00010) ma = ma & 00017;
#endif
#if defined (PDP15)
		printf ("%s%s%-o", opcode[i],
			((inst & 0020000)? " @": " "), 
			(cflag? ma & ADDRMASK: disp));
		if (!memm && (inst & 0010000)) printf (",X");
#else
		printf ("%s%s%-o", opcode[i],
			((inst & 0020000)? " I ": " "),
			(cflag? ma & ADDRMASK: disp));
#endif
		break;
	case I_V_OPR:					/* operate */
		if (sp = (inst & 03730)) printf ("%s", opcode[i]);
		oprf (inst & 014047, I_V_OPR, sp);
		break;
	case I_V_LAW:					/* LAW */
		printf ("%s %-o", opcode[i], inst & 017777);
		break;
	case I_V_XR9:					/* index with lit */
		disp = inst & 0777;
		if (disp & 0400) printf ("%s -%-o", opcode[i], 01000 - disp);
		else printf ("%s %-o", opcode[i], disp);
		break;
	case I_V_EST:					/* EAE setup */
		oprf (inst & 037007, I_V_EST, 0);	/* all masks */
		break;
	case I_V_ESH:					/* EAE shift */
		sp = oprf (inst & 017000, I_V_EST, 0);	/* shift masks */
		printf ((sp? " %s %-o": "%s %-o"), opcode[i], inst & 077);
		break;
	case I_V_EMD:					/* EAE mul-div */
		disp = inst & 077;			/* get actual val */
		k = (opc_val[i] >> I_V_DC) & 077;	/* get default val */
		if (disp == k) printf ("%s", opcode[i]);
		else if (disp < k) printf ("%s -%-o", opcode[i], k - disp);
		else printf ("%s +%-o", opcode[i], disp - k);
		break;  }				/* end case */
	return SCPE_OK;  }				/* end if */
	}						/* end for */
return SCPE_ARG;
}

/* Get 18b signed number

   Inputs:
	*cptr	=	pointer to input string
	*sign	=	pointer to sign
	*val	=	pointer to output value
   Outputs:
	status	=	error status
*/

int get_snum (char *cptr, int *sign, unsigned int *val)
{
*sign = 0;
if (*cptr == '+') {
	*sign = 1;
	cptr++;  }
else if (*cptr == '-') {
	*sign = -1;
	cptr++;  }
return get_unum (cptr, 8, 0777777, val);
}

/* Symbolic input

   Inputs:
	*cptr	=	pointer to input string
	addr	=	current PC
	cflag	=	true if parsing for CPU
	*val	=	pointer to output values
   Outputs:
	status	=	error status
*/

int parse_sym (char *cptr, unsigned int addr, int cflag, unsigned int *val)
{
int i, j, k, sign, dmask, epcmask;
unsigned int r;
char gbuf[CBUFSIZE];

while (isspace (*cptr)) cptr++;
for (i = 1; (i < 5) && (cptr[i] != 0); i++)
	if (cptr[i] == 0) for (j = i + 1; j <= 5; j++) cptr[j] = 0;
if (cptr[0] == '\'') {					/* single char? */
	if (cptr[1] == 0) return SCPE_ARG;
	val[0] = (unsigned int) cptr[1] | 0200;
	return SCPE_OK;  }
if (cptr[0] == '"') {					/* string? */
	if (cptr[1] == 0) return SCPE_ARG;
	val[0] = (((unsigned int) cptr[1] & 077) << 12) |
		 (((unsigned int) cptr[2] & 077) << 6) |
		  ((unsigned int) cptr[3] & 077);
	return SCPE_OK;  }
#if defined (PDP15)
if (cptr[0] == '#') {					/* packed? */
	if (cptr[1] == 0) return SCPE_ARG;
	val[0] = (((unsigned int) cptr[1] & 0177) << 11) |
		 (((unsigned int) cptr[2] & 0177) << 4) |
		 (((unsigned int) cptr[3] & 0170) >> 3);
	val[1] = (((unsigned int) cptr[3] & 0007) << 15) |
		 (((unsigned int) cptr[4] & 0177) << 8) |
		 (((unsigned int) cptr[5] & 0177) << 1);
	return -1;  }
#endif

/* Symbolic input, continued */

cptr = get_glyph (cptr, gbuf, 0);			/* get opcode */
for (i = 0; (opcode[i] != NULL) && (strcmp (opcode[i], gbuf) != 0) ; i++) ;
if (opcode[i] == NULL) return SCPE_ARG;
val[0] = opc_val[i] & 0777777;				/* get value */
j = (opc_val[i] >> I_V_FL) & I_M_FL;			/* get class */

switch (j) {						/* case on class */
case I_V_XR:						/* index */
	break;
case I_V_XR9:						/* index literal */
	cptr = get_glyph (cptr, gbuf, 0);		/* get next field */
	if (get_snum (gbuf, &sign, &r) != SCPE_OK) return SCPE_ARG;
	if (((sign >= 0) && (r > 0377)) || ((sign < 0) && (r > 0400)))
		return SCPE_ARG;
	val[0] = val[0] | ((sign >= 0)? r: (01000 - r));
	break;
case I_V_LAW:						/* law */
	cptr = get_glyph (cptr, gbuf, 0);		/* get next field */
	if (get_unum (gbuf, 8, 017777, &r) != SCPE_OK) return SCPE_ARG;
	val[0] = val[0] | r;
	break;
case I_V_MRF:						/* mem ref */
#if !defined (PDP15)
	dmask = 017777;
	cptr = get_glyph (cptr, gbuf, 0);		/* get next field */
	if (strcmp (gbuf, "I") == 0) {			/* indirect? */
		val[0] = val[0] | 020000;
		cptr = get_glyph (cptr, gbuf, 0);  }
#else
	if (memm) dmask = 017777;
	else dmask = 07777;
	if (*cptr == '@') {				/* indirect? */
		val[0] = val[0] | 020000;
		cptr++;  }
	cptr = get_glyph (cptr, gbuf, ',');		/* get glyph */
#endif
	epcmask = ADDRMASK & ~dmask;
	if (get_unum (gbuf, 8, ADDRMASK, &r) != SCPE_OK) return SCPE_ARG;
	if (r <= dmask) val[0] = val[0] | r;
	else if (cflag && (((addr ^ r) & epcmask) == 0))
		val[0] = val[0] | (r & dmask);
	else return SCPE_ARG;
#if defined (PDP15)
	if (!memm) {
		cptr = get_glyph (cptr, gbuf, 0);
		if (gbuf[0] != 0) {
			if (strcmp (gbuf, "X") != 0) return SCPE_ARG;
			val[0] = val[0] + 010000;  }  }
#endif
	break;
case I_V_EMD:						/* or'able */
	val[0] = val[0] | ((opc_val[i] >> I_V_DC) & 077); /* default shift */
case I_V_EST: case I_V_ESH: 
case I_V_NPN: case I_V_NPI: case I_V_IOT: case I_V_OPR:
	for (cptr = get_glyph (cptr, gbuf, 0); gbuf[0] != 0;
		cptr = get_glyph (cptr, gbuf, 0)) {
		for (i = 0; (opcode[i] != NULL) &&
			(strcmp (opcode[i], gbuf) != 0) ; i++) ;
		if (opcode[i] != NULL) {
			k = opc_val[i] & 0777777;
			if (((k ^ val[0]) & 0740000) != 0) return SCPE_ARG;
			val[0] = val[0] | k;  }
		else {	if (get_snum (gbuf, & sign, &r) != SCPE_OK)
				return SCPE_ARG;
			if (sign > 0) val[0] = val[0] + r;  
			else if (sign < 0) val[0] = val[0] - r;
			else val[0] = val[0] | r;  }  }
	break;  }					/* end case */
if (*cptr != 0) return SCPE_ARG;			/* junk at end? */
return SCPE_OK;
}
