/*
 *	ps - process status
 *	examine and print certain things about processes
 *
 *	Flags:
 *		a - print even processes not owned by user
 *		g - print out getty processes as well
 *		l - long listing
 *		x - print even processes not belonging to a terminal
 *		t - print only processes on named terminal
 */

#include <local-system>
#include <stdio.h>
#include <param.h>
#include <proc.h>
#include <text.h>
#include <inode.h>
#include <tty.h>
#include <dir.h>
#include <signal.h>
#include <table.h>
#include <user.h>
#include <stat.h>

struct	table	table[3];
#define	TPROC	0
#define	TSWAP	1
#define	TSWPDV	2

#define	NCHRS	2		/* number of chars to identify terminal */

char	dev[] = "/dev";
char	devmem[] = "mem";
char	tty[] = "tty";		/* terminals are ttyXX */
char	console[] = "console";	/* ... except for console */
char	devswap[] = "/dev/swap";
char	getty[] = "/etc/getty";
struct stat gettystat;

struct	proc mproc;
struct	user u;

int	retcode = 1;
int	aflg, gflg, lflg, xflg, tflg;
int	ndev;
int	mem;
int	swmem;
int	swap;
int	gettydev;
char	*tptr;
long	lseek();
char	*gettty();
char	*getptr();
char	*strncmp();
int	swapdev;
daddr_t	swplo;

struct devl
{
	char	dname[NCHRS];
	dev_t	dev;
} devl[NTTYS];


main(argc, argv)
char **argv;
{
	int i;
	char *ap;
	int uid;
	char outbuf[BUFSIZ];

	setbuf(stdout, outbuf);
	if (argc > 1)
	{
		ap = argv[1];
		if (*ap == '-')
			ap++;
		while (*ap)
		{
			switch (*ap++)
			{
			case 'a':
				aflg++;
				continue;
	
			case 'g':
				gflg++;
				continue;
	
			case 'l':
				lflg++;
				continue;
	
			case 'x':
				xflg++;
				continue;
	
			case 't':
				tflg++;
				if (*ap)
					tptr = ap;
				aflg++;
				if (*tptr == '?')
				{
					tptr = -1;
					xflg++;
				}
				break;		/* skip remaining args */
	
			default:
				fprintf(stderr, "Usage: ps [aglxt??]\n");
				exit(1);
			}
			break;
		}
	}

	if (chdir(dev) < 0)
	{
		perror(dev);
		exit(1);
	}
	if (getaddr(G_PROC, &table[TPROC]) == -1 ||
	    getaddr(G_SWPLO, &table[TSWAP]) == -1 ||
	    getaddr(G_SWPDV, &table[TSWPDV]) == -1)
	{
		perror("ps (getaddr)");
		exit(1);
	}
	if (table[TPROC].tb_num != NPROC ||
	    table[TPROC].tb_size != sizeof(struct proc) ||
	    table[TSWAP].tb_size != sizeof(swplo) ||
	    table[TSWPDV].tb_size != sizeof(swapdev))
	{
		fprintf(stderr, "\007Warning: recompile\n");
		exit(1);
	}
	if ((mem = open(devmem, 0)) < 0)
	{
		perror(devmem);
		exit(1);
	}
	swmem = open(devmem, 0);
	/*
	 *	Find base of swap
	 */
	lseek(mem, (long)table[TSWAP].tb_addr, 0);
	read(mem, (char *)&swplo, sizeof(swplo));
	lseek(mem, (long)table[TSWPDV].tb_addr, 0);
	read(mem, (char *)&swapdev, sizeof(swapdev));
	/*
	 *	Locate proc table
	 */
	lseek(mem, (long)table[TPROC].tb_addr, 0);
	getdev();
	uid = getuid();
	if (stat(getty, &gettystat) == -1)
	{
		printf("Can't find %s\n", getty);
		gettystat.st_ino = 0;	/* impossible value */
	}
	for (i = 0; i < NPROC; i++)
	{
		read(mem, (char *)&mproc, sizeof mproc);
		if (mproc.p_stat == 0)
			continue;
		if (mproc.p_pgrp == 0 && xflg == 0)
			continue;
		if (uid != mproc.p_uid && aflg == 0)
			continue;
		if (prcom())		/* non-zero ==> something printed */
			printf("\n");
	}
	exit(retcode);
}

getdev()
{
	register struct devl *dp;
	register FILE *df;
	struct stat sbuf;
	struct direct dbuf;

	if ((swap = open(devswap, 0)) != -1)
	{
		fstat(swap, &sbuf);
		if (sbuf.st_rdev != swapdev)
		{
			close(swap);
			swap = -1;
		}
	}
	if (swap == -1)
	{
		fprintf(stderr, "Can't find swap dev\n");
		exit(1);
	}
	if ((df = fopen("", "r")) == NULL)
	{
		perror(dev);
		exit(1);
	}
	ndev = 0;
	dp = &devl[0];
	if (stat(console, &sbuf) == 0)
	{
		strncpy(dp->dname, console, NCHRS);
		dp->dev = sbuf.st_rdev;
		dp++;
		ndev++;
	}
	while (fread((char *)&dbuf, sizeof(dbuf), 1, df) == 1)
	{
		if (ndev >= NTTYS)
		{
			fprintf(stderr, "Too many ttys - recompile\n");
			break;
		}
		if (dbuf.d_ino == 0)
			continue;
		if (strncmp(dbuf.d_name, tty, sizeof(tty)-1))
			continue;
		if (stat(dbuf.d_name, &sbuf) < 0)
			continue;
		if ((sbuf.st_mode & S_IFMT) != S_IFCHR)
			continue;
		strncpy(dp->dname, &dbuf.d_name[sizeof(tty)-1], NCHRS);
		dp->dev = sbuf.st_rdev;
		dp++;
		ndev++;
	}
	fclose(df);
	/*
	 * now fix up t-option (if any)
	 */
	if (tflg == 0 || tptr == (char *)-1)
		return;
	if (tptr == 0)
	{
		fstat(2, &gettystat);	/* use fd2 for terminal */
		for (dp = &devl[0]; dp < &devl[ndev]; dp++)
			if (dp->dev == gettystat.st_rdev)
				break;
	}
	else
		for (dp = &devl[0]; dp < &devl[ndev]; dp++)
			if (strncmp(dp->dname, tptr, NCHRS) == 0)
				break;
	if (dp == &devl[ndev])	/* oh dear couldn't find him */
		tptr = -1;
	else
		tptr = dp->dname;
}

prcom()
{
	char abuf[512];
	long addr;
	register int *ip;
	register char *cp, *cp1;
	int c, nbad;
	register char *tp;
	char **ap;
	int file;

	if (mproc.p_flag & SLOAD)
	{
		addr = ctob((long)mproc.p_addr);
		file = swmem;
	}
	else
	{
		addr = (mproc.p_addr + swplo) << 9;
		file = swap;
	}
	lseek(file, addr, 0);
	if (read(file, (char *)&u, sizeof(u)) != sizeof(u))
		return(0);

	if (!gflg && mproc.p_textp && isgetty())
		return(0);
	tp = gettty();
	if (tflg && tp != tptr)
		return(0);
	if (tp == -1)
		tp = "?";
	if (retcode)
	{
		if (lflg)
			printf(" F S UID   PID  PPID CPU PRI NICE  ADDR  SZ  WCHAN TTY TIME CMD\n");
		else
			printf("   PID TTY TIME CMD\n");
		retcode = 0;
	}
	if (lflg)
		printf("%2o %c%4d", mproc.p_flag, "0SWRIZT"[mproc.p_stat], mproc.p_uid);
	printf("%6u", mproc.p_pid);
	if (lflg)
	{
		printf("%6u%4d%4d%5d%6o%4d", mproc.p_ppid, mproc.p_cpu & 0377,
			mproc.p_pri,
			mproc.p_nice,
			mproc.p_addr,
			ctod(mproc.p_size));
		if (mproc.p_wchan)
			printf("%7o", mproc.p_wchan);
		else
			printf("       ");
	}
	printf(" %-2.2s", tp);
	if (mproc.p_stat == SZOMB)
	{
		printf("  <defunct>");
		return(1);
	}
	prtime(u.u_utime + u.u_stime);
	if (mproc.p_pid == 0)
	{
		printf(" UNIX");
		return(1);
	}
	addr += ctob((long)mproc.p_size) - sizeof(abuf);

	lseek(file, addr, 0);
	if (read(file, abuf, sizeof(abuf)) != sizeof(abuf))
		return(1);
	for (ip = (int *)&abuf[sizeof(abuf)] - 2; ip > (int *)abuf; )
	{
		if (*--ip == 0)
		{
			cp = (char *)(ip + 1);
			if (*cp == 0)
				cp++;
			nbad = 0;
			for (cp1 = cp; cp1 < &abuf[sizeof(abuf)]; cp1++)
			{
				c = *cp1 & 0177;
				if (c == 0)
					*cp1 = ' ';
				else if (c < ' ' || c > 0176)
				{
					if (++nbad >= 5)
					{
						*cp1++ = ' ';
						break;
					}
					*cp1 = '?';
				}
				/*
				 * avoid printing out the environment
				 * (unfortunately, this is a bit messy)
				 */
				else if (c == '=' && cp1[-1] >= 'A' && cp1[-1] <= 'Z')
				{
					while (cp1 > cp && *--cp1 != ' ');
					*cp1 = 0;
					break;
				}
			}
			while (*--cp1 == ' ')
				*cp1 = 0;
			printf(lflg ? " %.30s" : " %.60s", cp);
			return(1);
		}
	}
	/*
	 * couldn't find command name - use u_comm
	 */
	printf(" [ %.8s ]", u.u_comm);
	return(1);
}

char *
gettty()
{
	register char *p;
	register struct devl *dp;

	if (u.u_ttyp)
		for (dp = &devl[0]; dp < &devl[ndev]; dp++)
			if (dp->dev == u.u_ttyd)
				return(dp->dname);
	return(-1);
}

/*
 * return non-zero if the current (sharable) process is a getty
 */
isgetty()
{
	static unsigned txt[NTEXT];
	register unsigned *tp;
	static unsigned *tphi = txt;
	static struct text *gettyp;
	unsigned fred;
	struct inode *t_iptr;
	unsigned i_ino, i_dev;

	if (mproc.p_textp == gettyp)
		return(1);
	if (gettyp)
		return(0);
	/*
	 * if we get here, we haven't even found the getty text struct yet
	 */
	for (tp = txt; tp < tphi; tp++)
		if (*tp == (unsigned)mproc.p_textp)
			break;
	if (tp != tphi)
		return(0);	/* already found - not a getty */
	/*
	 * at this stage, we have a textp that
	 * we have not previously tested
	 */
	fred = (unsigned)&mproc.p_textp->x_iptr;
	lseek(swmem, (long)fred, 0);
	if (read(swmem, &t_iptr, sizeof(t_iptr)) != sizeof(t_iptr))
	{
		fprintf(stderr, "read text err\n");
		exit(1);
	}
	fred = (unsigned)(&t_iptr->i_dev);
	lseek(swmem, (long)fred, 0);
	if (read(swmem, &i_dev, sizeof(i_dev)) != sizeof(i_dev))
	{
		fprintf(stderr, "read inode err 0\n");
		exit(1);
	}
	fred = (unsigned)(&t_iptr->i_number);
	lseek(swmem, (long)fred, 0);
	if (read(swmem, &i_ino, sizeof(i_ino)) != sizeof(i_ino))
	{
		fprintf(stderr, "read inode err 1\n");
		exit(1);
	}
	if (i_dev == gettystat.st_dev && i_ino == gettystat.st_ino)
	{
		gettyp = mproc.p_textp;
		return(1);
	}
	*tphi++ = mproc.p_textp;
	return(0);
}

/*
 * prtime - print out argument (in ticks) in suitable form of type " xx:xx"
 * (unfortunately, the least-significant digit has been truncated, not rounded)
 */
prtime(tm)
long tm;
{
	register rmndr;

	printf(" ");
	rmndr = (short)tm % HZ;	/* only used for small tm */
	tm /= HZ;		/* now in seconds */
	if (tm < 60)		/* less than a minute */
		printf("%2d.", (short)tm);
	else
	{
		rmndr = (short)tm % 60;
		tm /= 60;		/* now in minutes */
		if (tm < 60)		/* less than an hour */
			printf("%2dm", (short)tm);
		else
		{
			rmndr = (short)tm % 60;
			tm /= 60;		/* now in hours */
			if (tm < 24)		/* less than a day */
				printf("%2dh", (short)tm);
			else
			{
				rmndr = tm % 24;
				tm /= 24;		/* now in days */
				printf("%2ldd", tm);
			}
		}
	}
	printf(rmndr < 10 ? "0%d" : "%d", rmndr);
}
