#include <param.h>
#include <systm.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <inode.h>
#include <file.h>
#include <fcntl.h>
#include <reg.h>
#include <errno.h>

extern struct user u;

/*
 * Max allowable buffering per pipe.
 * This is also the max size of the
 * file created to implement the pipe.
 * If this size is bigger than 5120,
 * pipes will be implemented with large
 * files, which is probably not good.
 */
#define	PIPSIZ	4096

/*
 * The sys-pipe entry.
 * Allocate an inode on the root device.
 * Allocate 2 file structures.
 * Put it all together with flags.
 */
pipe()
{
	register struct inode *ip;
	register struct file *rf, *wf;
	int r;

	ip = ialloc(pdevmnt ? pipedev : rootdev);
	if (ip == NULL)
		return;
	rf = falloc();
	if (rf == NULL)
	{
		iput(ip);
		return;
	}
	r = u.u_r.r_r0;
	wf = falloc();
	if (wf == NULL)
	{
		rf->f_count = 0;
		u.u_ofile[r] = NULL;
		iput(ip);
		return;
	}
	u.u_r.r_r1 = u.u_r.r_r0;
	u.u_r.r_r0 = r;
	wf->f_flag = FWRITE|O_FREE;
	wf->f_inode = ip;
	rf->f_flag = FREAD|O_FREE;
	rf->f_inode = ip;
	ip->i_count = 2;
	ip->i_mode = IFIFO;
	ip->i_uid = u.u_uid;
	ip->i_gid = u.u_gid;
	ip->i_flag = IACC|IUPD|ICHG;
}

/*
 *	Circular read from a pipe.
 */
readp(fp)
register struct file *fp;
{
	register struct inode *ip;
	register r;
	int ct;

	ip = fp->f_inode;
	do
	{
		if ((ct = u.u_count) == 0)
			return;
		plock(ip);

		/*
		 * If there is nothing there sleep and try again.
		 */

		while ((r = ip->i_size) == 0)	/* fits as PIPSIZ < 2^16 */
		{
			/*
			 * If there is not a read and a write pointer
			 * then return without reading.
			 */

			prele(ip);
			if (ip->i_count < 2)
				return;
			ip->i_mode |= IREAD;
			sleep((caddr_t)ip+2, PPIPE);
			plock(ip);
		}
		ip->i_size = (off_t)min((unsigned)fp->f_offset + r, PIPSIZ);
		u.u_offset = fp->f_offset;
		readi(ip);
		ip->i_size = r - (ct - u.u_count);

		/*
		 * If the read pointer has reached the end of the
		 * pipe buffer reset it to the front.
		 */

		if ((fp->f_offset = u.u_offset) == PIPSIZ)
			fp->f_offset = 0;

		/*
		 * If the write proccess was sleeping wake it up.
		 */

		if (ip->i_mode & IWRITE)
		{
			ip->i_mode &= ~IWRITE;
			wakeup((caddr_t)ip+1);
		}

		prele(ip);
	}
	while (fp->f_offset == 0 && ip->i_size != 0 && u.u_error == 0);
}

/*
 * Circular write on a pipe.
 */
writep(fp)
register struct file *fp;
{
	register struct inode *ip;
	register c;
	int sz;

	ip = fp->f_inode;
	c = u.u_count;
	while (c && u.u_error == 0)
	{
		plock(ip);

		/*
		 * If there is not both a read and a write pointer
		 * set the broken pipe error and return.
		 */

		if (ip->i_count < 2)
		{
			prele(ip);
			u.u_error = EPIPE;
			psignal(u.u_procp, SIGPIPE);
			return;
		}

		/*
		 * If the pipe is already full wait till
		 * some of it is read.
		 */

		if ((sz = ip->i_size) >= PIPSIZ)
		{
			prele(ip);
			ip->i_mode |= IWRITE;
			sleep((caddr_t)ip+1, PPIPE);
			continue;
		}
		u.u_offset = fp->f_offset;
		u.u_count = min(c, PIPSIZ - max((unsigned)fp->f_offset, sz));
		c -= u.u_count;
		sz += u.u_count;
		writei(ip);
		ip->i_size = (off_t)sz;
		if ((fp->f_offset = u.u_offset) == PIPSIZ)
			fp->f_offset = 0;
		prele(ip);

		/*
		 * If the read proccess had been waiting wake it up.
		 */

		if (ip->i_mode & IREAD)
		{
			ip->i_mode &= ~IREAD;
			wakeup((caddr_t)ip+2);
		}
	}
}

/*
 * Lock a pipe.
 * If its already locked,
 * set the WANT bit and sleep.
 */
plock(ip)
register struct inode *ip;
{
	while (ip->i_flag & ILOCK)
	{
		ip->i_flag |= IWANT;
		sleep((caddr_t)ip, PINOD);
	}
	ip->i_flag |= ILOCK;
}

/*
 * Unlock a pipe.
 * If WANT bit is on,
 * wakeup.
 * This routine is also used
 * to unlock inodes in general.
 */
prele(ip)
register struct inode *ip;
{
	ip->i_flag &= ~ILOCK;
	if (ip->i_flag & IWANT)
	{
		ip->i_flag &= ~IWANT;
		wakeup((caddr_t)ip);
	}
}
