/* single.c - Single operand instructions. */


#include "defines.h"


/* adc() - Add Carry Instruction. */

void 
adc()
{
    uint16_t data;

    load_dst(&data);

    if (psw & CC_C) {		/* do if carry is set */
	if (data == MPI)
	    SET_CC_V();
	else
	    CLR_CC_V();
	if (data == NEG_1)
	    SET_CC_C();
	else
	    CLR_CC_C();
	data++;			/* add the carry */
    } else {
	CLR_CC_V();
	CLR_CC_C();
    }

    CHG_CC_N(data);
    CHG_CC_Z(data);

    store_dst_2(data);
}


/* asl() - Arithmetic Shift Left Instruction. */

void 
asl()
{
    uint16_t data;

    load_dst(&data);

    if (data & SIGN)
	SET_CC_C();
    else
	CLR_CC_C();

    data <<= 1;

    CHG_CC_N(data);
    CHG_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    store_dst_2(data);
}


/* asr() - Arithmetic Shift Right Instruction. */

void 
asr()
{
    uint16_t data;

    load_dst(&data);

    if (data & LSBIT)
	SET_CC_C();
    else
	CLR_CC_C();

    data = (data >> 1) + (data & SIGN);	/* shift and replicate */

    CHG_CC_N(data);
    CHG_CC_Z(data);

    CHG_CC_V_XOR_C_N();

    store_dst_2(data);
}


/* clr() - Clear Instruction. */

void 
clr()
{

    CLR_CC_ALL();
    SET_CC_Z();

    store_dst((uint16_t) 0);
}


/* com() - Complement Instrcution. */

void 
com()
{
    uint16_t data;

    load_dst(&data);

    data = ~data;

    CHG_CC_N(data);
    CHG_CC_Z(data);
    CLR_CC_V();
    SET_CC_C();

    store_dst_2(data);
}


/* dec() - Decrement Instrcution. */

void 
dec()
{
    uint16_t data;

    load_dst(&data);

    if (data == MNI)
	SET_CC_V();
    else
	CLR_CC_V();

    --data;

    CHG_CC_N(data);
    CHG_CC_Z(data);

    store_dst_2(data);
}


/* inc() - Increment Instruction. */

void 
inc()
{
    uint16_t data;

    load_dst(&data);

    if (data == MPI)
	SET_CC_V();
    else
	CLR_CC_V();

    ++data;

    CHG_CC_N(data);
    CHG_CC_Z(data);

    store_dst_2(data);
}


/* neg() - Negate Instruction. */

void 
neg()
{
    uint16_t data;

    load_dst(&data);

    data = (NEG_1 - data) + 1;

    CHG_CC_N(data);
    CHG_CC_Z(data);

    if (data == MNI)
	SET_CC_V();
    else
	CLR_CC_V();

    if (data == 0)
	CLR_CC_C();
    else
	SET_CC_C();

    store_dst_2(data);
}


/* rol() - Rotate Left Instruction. */

void 
rol()
{
    uint16_t data;
    uint16_t temp;

    load_dst(&data);

    temp = data & SIGN;		/* get sign bit */
    data <<= 1;			/* shift */

    if (psw & CC_C)		/* roll in carry */
	data += LSBIT;

    if (temp)			/* roll out to carry */
	SET_CC_C();
    else
	CLR_CC_C();

    CHG_CC_N(data);
    CHG_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    store_dst_2(data);
}


/* ror() - Rotate Right Instruction. */

void 
ror()
{
    uint16_t data;
    uint16_t temp;

    load_dst(&data);

    temp = data & LSBIT;	/* get low bit */
    data >>= 1;			/* shift */

    if (psw & CC_C)		/* roll in carry */
	data += SIGN;

    if (temp)			/* roll out to carry */
	SET_CC_C();
    else
	CLR_CC_C();

    CHG_CC_N(data);
    CHG_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    store_dst_2(data);
}


/* sbc() - Subtract Carry Instruction. */

void 
sbc()
{
    uint16_t data;

    load_dst(&data);

    if (data == MNI)
	SET_CC_V();
    else
	CLR_CC_V();

    if (psw & CC_C) {		/* do if carry is set */
	if (data)
	    CLR_CC_C();
	else
	    SET_CC_C();
	--data;			/* subtract carry */
    } else {
	CLR_CC_C();
    }

    CHG_CC_N(data);
    CHG_CC_Z(data);

    store_dst_2(data);
}


/* swabi() - Swap Bytes Instruction. */

void 
swabi()
{
    uint16_t data1;
    uint16_t data2;
    uint16_t data3;

    load_dst(&data1);

    data2 = (data1 << 8) & 0xff00;
    data3 = (data1 >> 8) & 0x00ff;
    data1 = data2 + data3;

    CLR_CC_ALL();
    CHGB_CC_N(data3);		/* cool, negative and zero */
    CHGB_CC_Z(data3);		/* checks done on low byte only */

    store_dst_2(data1);
}


/* sxt() - Sign Extend Instruction. */

void 
sxt()
{
    uint16_t data;

    if (psw & CC_N) {
	data = NEG_1;
	CLR_CC_Z();
    } else {
	data = 0;
	SET_CC_Z();
    }
    CLR_CC_V();

    store_dst(data);
}


/* tst() - Test Instruction. */

void 
tst()
{
    uint16_t data;

    load_dst(&data);

    CLR_CC_ALL();
    CHG_CC_N(data);
    CHG_CC_Z(data);

}


/* tstb() - Test Byte Instruction. */

void 
tstb()
{
    uint8_t data;

    loadb_dst(&data);

    CHGB_CC_N(data);
    CHGB_CC_Z(data);
    CLR_CC_V();
    CLR_CC_C();

}

/* aslb() - Arithmetic Shift Left Byte Instruction. */

void 
aslb()
{
    uint8_t data;

    loadb_dst(&data);

    if (data & SIGN_B)
	SET_CC_C();
    else
	CLR_CC_C();

    data <<= 1;

    CHGB_CC_N(data);
    CHGB_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    storeb_dst_2(data);
}


/* asrb() - Arithmetic Shift Right Byte Instruction. */

void 
asrb()
{
    uint8_t data;

    loadb_dst(&data);

    if (data & LSBIT)
	SET_CC_C();
    else
	CLR_CC_C();

    data = (data >> 1) + (data & SIGN_B);	/* shift and replicate */

    CHGB_CC_N(data);
    CHGB_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    storeb_dst_2(data);
}


/* clrb() - Clear Byte Instruction. */

void 
clrb()
{
    CLR_CC_ALL();
    SET_CC_Z();

    storeb_dst((uint8_t) 0);
}


/* comb() - Complement Byte Instrcution. */

void 
comb()
{
    uint8_t data;

    loadb_dst(&data);

    data = ~data;

    CHGB_CC_N(data);
    CHGB_CC_Z(data);
    CLR_CC_V();
    SET_CC_C();

    storeb_dst_2(data);
}


/* decb() - Decrement Byte Instrcution. */

void 
decb()
{
    uint8_t data;

    loadb_dst(&data);

    if (data == MNI_B)
	SET_CC_V();
    else
	CLR_CC_V();

    --data;

    CHGB_CC_N(data);
    CHGB_CC_Z(data);

    storeb_dst_2(data);
}


/* incb() - Increment Byte Instruction. */

void 
incb()
{
    uint8_t data;

    loadb_dst(&data);

    if (data == MPI_B)
	SET_CC_V();
    else
	CLR_CC_V();

    ++data;

    CHGB_CC_N(data);
    CHGB_CC_Z(data);

    storeb_dst_2(data);
}


/* negb() - Negate Byte Instruction. */

void 
negb()
{
    uint8_t data;

    loadb_dst(&data);

    data = (NEG_1_B - data) + 1;/* hope this is right */

    CHGB_CC_N(data);
    CHGB_CC_Z(data);

    if (data == MNI_B)
	SET_CC_V();
    else
	CLR_CC_V();

    if (data)
	SET_CC_C();
    else
	CLR_CC_C();

    storeb_dst_2(data);
}


/* rolb() - Rotate Left Byte Instruction. */

void 
rolb()
{
    uint8_t data;
    uint8_t temp;

    loadb_dst(&data);

    temp = data & SIGN_B;	/* get top bit */
    data <<= 1;			/* shift */

    if (psw & CC_C)		/* roll in carry */
	data = data + LSBIT;

    if (temp)			/* roll out to carry */
	SET_CC_C();
    else
	CLR_CC_C();

    CHGB_CC_N(data);
    CHGB_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    storeb_dst_2(data);
}


/* rorb() - Rotate Right Byte Instruction. */

void 
rorb()
{
    uint8_t data;
    uint8_t temp;

    loadb_dst(&data);

    temp = data & LSBIT;	/* get low bit */
    data >>= 1;			/* shift */

    if (psw & CC_C)		/* roll in carry */
	data += SIGN_B;

    if (temp)			/* roll out to carry */
	SET_CC_C();
    else
	CLR_CC_C();

    CHGB_CC_N(data);
    CHGB_CC_Z(data);
    CHG_CC_V_XOR_C_N();

    storeb_dst_2(data);
}


/* adcb() - Add Carry Byte Instruction. */

void 
adcb()
{
    uint8_t data;

    loadb_dst(&data);

    if (psw & CC_C) {		/* do if carry is set */
	if (data == MPI_B)
	    SET_CC_V();
	else
	    CLR_CC_V();
	if (data == NEG_1_B)
	    SET_CC_C();
	else
	    CLR_CC_C();
	++data;			/* add the carry */
    } else {
	CLR_CC_V();
	CLR_CC_C();
    }

    CHGB_CC_N(data);
    CHGB_CC_Z(data);

    storeb_dst_2(data);
}


/* sbcb() - Subtract Carry Byte Instruction. */

void 
sbcb()
{
    uint8_t data;

    loadb_dst(&data);

    if (psw & CC_C) {		/* do if carry is set */
	if (data)
	    CLR_CC_C();
	else
	    SET_CC_C();

	--data;			/* subtract carry */
    } else {
	CLR_CC_C();
    }

    if (data == MNI_B)
	SET_CC_V();
    else
	CLR_CC_V();

    CHGB_CC_N(data);
    CHGB_CC_Z(data);

    storeb_dst_2(data);
}
