/* Copyright (c)1994-1999 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

/*
 * MRV12 option
 *
 * this emulates an MRV12 boot prom including maintenance, configuration
 * and display registers.
 *
 * You can only define a controller, no devices.
 *
 * Arguments:
 *	ctrl mr <csr-address> <prom-file>
 */
# include "proc.h"

# define LED
# define D(C)

typedef struct Boot Boot;

struct Boot {
	unsigned	csr_base;
	unsigned	page0;
	unsigned	page1;
	unsigned	maint;
	unsigned	config;
	unsigned	display;
	ushort		prog[8192];
};

void	boot_ctrl_create(IODev *, int, char **);
void	boot_dev_create(IODev *, int, char **);
void	boot_ctrl_complete(IODev *);
void	boot_reset(IODev *);
ushort	boot_fetch(IODev *, unsigned);
void	boot_store(IODev *, unsigned, int, ushort);
void	boot_info(IODev *);

static int	bootc_info(IODev *dev, int argc, char **argv);

IODevCmd boot_cmds[] = {
	{ "?",		"[command ...]",	dev_help },
	{ "help",	"[command ...]",	dev_help },
	{ "info",	"",			bootc_info },
	{ NULL }
};

IOOps mr_ops = {
	boot_ctrl_create,	/* ctrl_create */
	boot_dev_create,	/* dev_create */
	boot_ctrl_complete,	/* ctrl_complete */
	boot_reset,		/* reset */
	boot_fetch,		/* fetch */
	boot_store,		/* store */
	0,			/* vector */
	0,			/* dma */
	0,			/* async */
	boot_info,		/* info */
	0,			/* flush */
	boot_cmds,		/* cmds */
};

enum {
	MR_Page		= 00,
	MR_Maint	= 02,
	MR_Disp		= 04,
	MR_Page0	= 017773000,
	MR_Page1	= 017765000,
	MR_PageSize	= 000001000,
	MR_SizeMask	= 000000776,
};


void
boot_ctrl_create(IODev *dev, int argc, char **argv)
{
	Boot	*d;
	int	i, fd;
	struct stat statb;

	if(argc != 2)
		conf_panic("mr: bad args in configuration");

	d = dev->data = xalloc(sizeof(Boot));
	(void)memset(d, 0, sizeof(Boot));

	d->csr_base = parse_csr(argv[0], "mr");

	for(i = MR_Page0; i < MR_Page0 + MR_PageSize; i += 2)
		proc.iopage[IOP(i)] = dev;
	for(i = MR_Page1; i < MR_Page1 + MR_PageSize; i += 2)
		proc.iopage[IOP(i)] = dev;
	proc.iopage[IOP(d->csr_base + MR_Page )] = dev;
	proc.iopage[IOP(d->csr_base + MR_Maint)] = dev;
	proc.iopage[IOP(d->csr_base + MR_Disp )] = dev;

	if((fd = open(argv[1], O_RDONLY)) < 0)
		conf_panic("mr: can't open file %s: %s", argv[1], strerror(errno));

	if(fstat(fd, &statb))
		conf_panic("mr: can't stat file %s: %s", argv[1], strerror(errno));

	if(statb.st_size > sizeof(d->prog))
		conf_panic("mr: boot program to large");

	if(read(fd, d->prog, statb.st_size) != statb.st_size)
		conf_panic("mr: read error on %s: %s", argv[1], strerror(errno));

	close(fd);
}


void
boot_dev_create(IODev *dev UNUSED, int argc UNUSED, char **argv UNUSED)
{
	panic("mr: can't have any devices");
}


void
boot_ctrl_complete(IODev *dev UNUSED)
{
	/* nix */
}


void
boot_reset(IODev *dev UNUSED)
{
	/* nix */
}


ushort
boot_fetch(IODev *dev, unsigned a)
{
	ushort	v;
	Boot	*b = dev->data;

	if(a >= MR_Page0 && a < MR_Page0 + MR_PageSize) {
		v = b->prog[(b->page0 + (a & MR_SizeMask)) >> 1];
		v = GETW(v);
	} else if(a >= MR_Page1 && a < MR_Page1 + MR_PageSize) {
		v = b->prog[(b->page1 + (a & MR_SizeMask)) >> 1];
		v = GETW(v);
	} else switch(a - b->csr_base) {

	case MR_Page:
		v = (b->page0 >> 9) + (b->page1 >> 1);
		break;

	case MR_Maint:
		v = b->maint;
		break;

	case MR_Disp:
		v = b->config;
		break;

	default:
		warn("boot_fetch(%o)", a);
		Trap4(020);
	}
	return v;
}

void
boot_store(IODev *dev, unsigned a, int mode, ushort v)
{
	Boot	*b = dev->data;

	if((a >= MR_Page0 && a < MR_Page0 + MR_PageSize) || (a >= MR_Page1 && a < MR_Page1 + MR_PageSize))
		return;

	switch(a - b->csr_base) {

	case MR_Page:
		if(!(mode & M_High))
			b->page0 = (v & 037) << 9;
		if(!(mode & M_Low))
			b->page1 = (v & 017400) << 1;
		D(printf("boot map: %2o %2o - %06ho\n", b->page0, b->page1, proc.reg[7]));
		break;

	case MR_Maint:
		if(mode & M_Low)
			b->maint = (b->maint & 0177400) | (v & 0377);
		else if(mode & M_High)
			b->maint = (b->maint & 0377) | (v & 0177400);
		else
			b->maint = v;
		break;

	case MR_Disp:
		v &= 017;
		if(!(mode & M_High) && b->display != v)  {
			b->display = v;
# ifdef LED
			printf("LED %c%c%c%c\n", "O."[(v&010)!=0], "O."[(v&4)!=0],
						 "O."[(v&2)!=0], "O."[(v&1)!=0]);
# endif
		}
		break;

	default:
		warn("boot_fetch(%o, %d)", a, mode);
		Trap4(020);
	}

}

void
boot_info(IODev *dev)
{
	Boot	*b = dev->data;

	printf("BOOT PROM\n");
	printf("Page register        %08o: %02o %02o\n", b->csr_base + MR_Page, 
						b->page0, b->page1);
	printf("Maintenance register %08o: %06o\n", b->csr_base + MR_Maint, 
						b->maint);
	printf("Display register     %08o: %c%c%c%c\n", b->csr_base + MR_Disp,
						"O."[(b->display & 010)!=0],
						"O."[(b->display & 004)!=0],
						"O."[(b->display & 002)!=0],
						"O."[(b->display & 001)!=0]);
	printf("Page0 (fixed): %08o - %08o\n", MR_Page0, MR_Page0 + MR_PageSize - 2);
	printf("Page1 (fixed): %08o - %08o\n", MR_Page1, MR_Page1 + MR_PageSize - 2);
}

static int
bootc_info(IODev *dev, int argc, char **argv UNUSED)
{
	if(argc != 0)
		return 1;
	boot_info(dev);
	return 0;
}
