/* i1401_iq.c: IBM 1407 inquiry terminal

   Copyright (c) 1996 Robert M Supnik, Digital Equipment Corporation
   Commercial use prohibited
*/

#include "i1401_defs.h"
#include <ctype.h>

extern volatile int stop_cpu;
extern unsigned char M[];
extern int iochk, ind[64];
extern char ascii_to_bcd[128], bcd_to_ascii[64];
extern UNIT cpu_unit;
int inq_char = 033;					/* request inq */
int inq_svc (UNIT *uptr);
int inq_reset (DEVICE *dptr);
void puts_tty (char *cptr);
extern int sim_activate (UNIT *uptr, int interval);
extern int sim_poll_kbd (void);
extern int sim_putchar (int out);

/* INQ data structures

   inq_dev	INQ device descriptor
   inq_unit	INQ unit descriptor
   inq_reg	INQ register list
*/

UNIT inq_unit = { UDATA (&inq_svc, 0, 0), KBD_POLL_WAIT };

REG inq_reg[] = {
	{ ORDATA (INQC, inq_char, 7) },
	{ FLDATA (INR, ind[IN_INR], 0) },
	{ FLDATA (INC, ind[IN_INC], 0) },
	{ DRDATA (TIME, inq_unit.wait, 24), REG_NZ + PV_LEFT },
	{ NULL }  };

DEVICE inq_dev = {
	"INQ", &inq_unit, inq_reg, NULL,
	1, 10, 31, 1, 8, 7,
	NULL, NULL, &inq_reset,
	NULL, NULL, NULL };

/* Terminal I/O

   Modifiers have not been checked; legal modifiers are R and W
*/

int inq_io (int flag, int addr, int mod, int *new)
{
int i, t, wm_seen = 0;

ind[IN_INC] = 0;					/* clear inq clear */
switch (mod) {						/* case on mod */
case BCD_R:						/* input */
/*	if (ind[IN_INR] == 0) return SCPE_OK;		/* return if no req */
	ind[IN_INR] = 0;				/* clear req */
	puts_tty ("[Enter]\r\n");				/* prompt */
	for (i = 0; M[addr] != (BCD_GRPMRK + WM); i++) { /* until GM + WM */
		while ((t = sim_poll_kbd ()) == SCPE_OK)
			if (stop_cpu) return SCPE_STOP;	/* interrupt? */
		if (t < SCPE_KFLAG) return t;		/* if not char, err */
		t = t & 0177;
		if ((t == '\r') || (t == '\n')) break;
		if (t == inq_char) {			/* cancel? */
			ind[IN_INC] = 1;		/* set indicator */
			puts_tty ("\r\n[Canceled]\r\n");
			*new = addr;
			return SCPE_OK;  }
		if (i && ((i % INQ_WIDTH) == 0)) puts_tty ("\r\n");
		sim_putchar (t);			/* echo */
		if (flag == MD_WM) {			/* word mark mode? */
			if ((t == '~') && (wm_seen == 0)) wm_seen = WM;
			else {	M[addr] = wm_seen | ascii_to_bcd[t];
				wm_seen = 0; }  }
		else M[addr] = (M[addr] & WM) | ascii_to_bcd[t];
		if (!wm_seen) addr++;
		if (addr >= MEMSIZE) {
			*new = addr % MAXMEMSIZE;
			return STOP_NXM;  }  }
	puts_tty ("\r\n");
	M[addr] = BCD_GRPMRK + WM;
	*new = addr + 1;
	return SCPE_OK;
case BCD_W:						/* output */
	for (i = 0; (t = M[addr++]) != (BCD_GRPMRK + WM); i++) {
		if ((flag == MD_WM) && (t & WM)) {
			if (i && ((i % INQ_WIDTH) == 0)) puts_tty ("\r\n");
			sim_putchar ('~');  }
		if (i && ((i % INQ_WIDTH) == 0)) puts_tty ("\r\n");
		sim_putchar (bcd_to_ascii[t & CHAR]);
		if (addr >= MEMSIZE) {
			*new = addr % MAXMEMSIZE;
			return STOP_NXM;  }  }
	puts_tty ("\r\n");
	*new = addr;
	return SCPE_OK;
default:
	return STOP_INVM;  }				/* invalid mod */
return SCPE_OK;
}

/* Unit service - polls for WRU or inquiry request */

int inq_svc (UNIT *uptr)
{
int temp;

sim_activate (&inq_unit, inq_unit.wait);		/* continue poll */
if ((temp = sim_poll_kbd ()) < SCPE_KFLAG) return temp;	/* no char or error? */
if ((temp & 0177) == inq_char) ind[IN_INR] = 1;		/* set indicator */
return SCPE_OK;
}

/* Output multiple characters */

void puts_tty (char *cptr)
{
if (cptr == NULL) return;
while (*cptr != 0) sim_putchar (*cptr++);
return;
}

/* Reset routine */

int inq_reset (DEVICE *dptr)
{
ind[IN_INR] = ind[IN_INC] = 0;				/* clear indicators */
sim_activate (&inq_unit, inq_unit.wait);		/* activate poll */
return SCPE_OK;
}
