/*	v7fs -- a utility to read from an unmounted version 7	*/
/*		unix filesystem.				*/
/*		Mike Karels, Dept. of Molecular Biology		*/
/*		University of California, Berkeley		*/
/*								*/
/*		Heavily modified by Jonathan Naylor		*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

/*
 *	PDP-11 data types & lengths, you may need to edit these.
 */
typedef	unsigned char	int8;
typedef	unsigned short	int16;
typedef	unsigned int	int32;

/*
 *	V7 File system specific data types.
 */
typedef	int32		v7_daddr_t;
typedef	int16		v7_ino_t;
typedef	int32		v7_time_t;
typedef	int32		v7_off_t;

/*
 *	V7 File system defines.
 */
#define	V7_BLKSIZE	512		/* block size in bytes */
#define	V7_DIRSIZ	14		/* bytes allocated to file names */

#define	V7_NADDR	13

#define	V7_NICFREE	50		/* number of superblock free blocks */

#define	V7_ROOTINO	((v7_ino_t)2)	/* i number of all roots */

#define	V7_IFDIR	0040000		/* directory */
#define	V7_IFREG	0100000		/* regular */

/*
 *	V7 File system data structures.
 */
struct v7_direct {
	v7_ino_t	d_ino;
	int8		d_name[V7_DIRSIZ];
};
#define	V7_DIRECT	16		/* size of struct v7_direct */

struct v7_dinode {
	int16		di_mode;	/* mode and type of file */
	int16		di_nlink;	/* number of links to file */
	int16		di_uid;		/* owner's user id */
	int16		di_gid;		/* owner's group id */
	v7_off_t	di_size;	/* number of bytes in file */
	v7_daddr_t	di_addr[V7_NADDR]; /* disk block addresses */
	v7_time_t	di_atime;	/* time last accessed */
	v7_time_t	di_mtime;	/* time last modified */
	v7_time_t	di_ctime;	/* time created */
};
#define	V7_DINODE	64		/* size of struct v7_dinode */

struct v7_filsys {
	int16		s_isize;	/* size in blocks of i-list */
	v7_daddr_t	s_fsize;	/* size in blocks of entire volume */
	int16		s_nfree;	/* number of addresses in s_free */
	v7_daddr_t	s_free[V7_NICFREE]; /* free block list */
	v7_time_t	s_time;		/* last super block update */
};

/*
 *	Function prototypes.
 */
static int getline(char *);
static int parse(char *);
static void null(int);
static void ls(int , char **);
static int cpdir(int, char **);
static int cd(int, char **);
static int iindex(char *, char);
static struct v7_direct *find(char *);
static void cp(int, char **);
static void rindir(FILE *, v7_daddr_t, int);
static void riindir(FILE *, v7_daddr_t, int);
static void riiindir(FILE *, v7_daddr_t, int);
static int readdir(struct v7_direct *);
static int dirsort(const void *, const void *);
static int readi(v7_ino_t, struct v7_dinode *);
static int readblk(v7_daddr_t, void *);
static void flush(int);
static void msg(void);
static void printi(int, char **);
static void printsb(int, char **);
static void printblk(int, char **);
static void hexdump(unsigned char *, int);
static void dumpblk(int, char **);
static void dumpboot(int, char **);
static void cpblk(int, char **);
static void readroot(v7_ino_t);
static int16 extract16(unsigned char *);
static int32 extract24(unsigned char *);
static int32 extract32(unsigned char *);

#define	MAXLINE 	2000		/* maximum input line length */

static char		linein[MAXLINE];

#define MAXARG		30	/* maximum size of input args */

static int		ac;	/* number of arguments to subroutine */
static char		*arg[MAXARG];	/* pointers to subroutine arguments */

static unsigned char	buf[V7_BLKSIZE];
static int		size;

#define MAXSIZE 	500	/* max number of directory entries	*/

#define V7_DAPB		(V7_BLKSIZE / sizeof(v7_daddr_t))
#define	V7_DIRBLK	(V7_BLKSIZE / V7_DIRECT)

static unsigned char bb[V7_BLKSIZE];	/* boot block */
static struct v7_filsys	sb;		/* superblock */
static struct v7_dinode	dir_ino, fil_ino;	/* current dir, file inodes */
static struct v7_direct	dir[MAXSIZE], cur_dir, root = { V7_ROOTINO, "/" };

static int		dir_sz;
static FILE		*fp, *fpout;
static v7_ino_t		imax;

static char		*cmd[] = { "ls", "cd", "cp", "cat", "?", "", "q",
				"cpdir", "lcd", "printsb", "printi",
				"printblk", "dumpblk", "cpblk",	"rootino",
				"dumpboot" };

#define NCMD		16	/* number of commands in cmd[] */

int main(int argc, char **argv)
{
	int i;

	if (argc < 2) {
		fprintf(stderr,"Usage: v7 file, where file is block special\n");
		return 1;
	}

	if ((fp = fopen(argv[1], "r")) == NULL) {
		fprintf(stderr, "v7fs: cannot open %s\n", argv[1]);
		return 1;
	}

	if (fread(bb, 1, V7_BLKSIZE, fp) < V7_BLKSIZE) {
		fprintf(stderr, "v7fs: cannot read boot block\n");
		return 1;
	}

	if (fread(buf, 1, V7_BLKSIZE, fp) < V7_BLKSIZE) {
		fprintf(stderr,"v7fs: cannot read superblock\n");
		return 1;
	}

	sb.s_isize = extract16(buf + 0);
	sb.s_fsize = extract32(buf + 2);
	sb.s_nfree = extract16(buf + 6);

	for (i = 0; i < V7_NICFREE; i++)
		sb.s_free[i] = extract32(buf + 8 + i * 4);

	sb.s_time  = extract32(buf + 414);

	imax = (V7_BLKSIZE / V7_DINODE) * (sb.s_isize - 1);

	if (readdir(&root) < 0)
		return 1;

	cur_dir.d_ino = root.d_ino;
	strcpy(cur_dir.d_name, root.d_name);

	signal(SIGINT, SIG_IGN);

	while (1) {
		fprintf(stdout, "* ");

		if (getline(linein) == EOF) return 0;

		if (linein[0] == '!') {
			signal(SIGINT, null);
			system(linein + 1);
			signal(SIGINT, SIG_IGN);
			continue;
		}

		if (parse(linein) == 0)
			continue;

		for (i = 0; i < NCMD && strcmp(arg[0], cmd[i]) != 0; i++)
			;

		switch (i) {
			case 0: 
				ls(ac, arg);
				break;
			case 1: 
				cd(ac, arg);
				break;
			case 2: 
				cp(ac, arg);
				break;
			case 3: 
				cp(ac, arg);  /* arg[0]="cat" */
				break;
			case 4: 
				msg();
				break;
			case 5: 
				break;
			case 6: 
				fclose(fp);
				return 0;
			case 7: 
				cpdir(ac, arg);
				break;
			case 8: 
				if (chdir(arg[1]) < 0)
				perror("lcd");
				break;
			case 9: 
				printsb(ac, arg);
				break;
			case 10:
				printi(ac, arg);
				break;
			case 11:
				printblk(ac, arg);
				break;
			case 12:
				dumpblk(ac, arg);
				break;
			case 13:
				cpblk(ac, arg);
				break;
			case 14:
				readroot((ino_t)atoi(arg[1]));
				break;
			case 15:
				dumpboot(ac, arg);
				break;
			case NCMD:
				fprintf(stderr,"v7fs: invalid command\n");
				break;
		}
	}
}

static int getline(char *linein)	/* reads input line into linein */
{
	int i, c;

	for (i = 0; i < MAXLINE - 1 && (c = getchar()) != '\n' && c != EOF; i++)
		linein[i] = c;

	if (c == '\n') {
		linein[i] = '\0';
		return i;
	} else if (c == EOF) {
		return EOF;
	} else {
		fprintf(stderr, "v7fs: input line too long\n");
		fflush(stdin);
		linein[0] = '\0';
		return 0;
	}
}

static int parse(char *line)	/* parse into words, producing ac and *arg */
{
	int inword;
	char *c;
	ac = 0;
	inword = 0;

	for (c = line; *c && ac < MAXARG; c++) {
		if (*c == ' ' || *c == '\t') {
			if (inword)
				inword = 0;
			*c = '\0';
			continue;
		} else if (inword == 0) {
			arg[ac++] = c;
			inword = 1;
		}
	}

	if (ac == MAXARG) {
		fprintf(stderr,"v7fs: too many arguments\n");
		return 0;
	}

	arg[ac] = NULL;

	return ac;
}

static void null(int sig) {}		/* trap here on interrupt during system() */

static void ls(int ac, char **av)	/* list directory 'arg[1]', current dir if no name */
{	/* -i option gives inode numbers */
	int i, j, flag, iflag, col;
	struct v7_direct save_dir;

	if (ac > 1 && strcmp(av[1], "-i") == 0) {
		ac--;
		av++;
		iflag = 1;
		col = 3;
	} else {
		iflag = 0;
		col = 4;
	}

	flag = 0;

	if (ac > 1) {
		flag = 1;
		save_dir.d_ino = cur_dir.d_ino;
		strcpy(save_dir.d_name,cur_dir.d_name);

		if (cd(ac, av) < 0)
			return;
	}

	j = 0;

	for (i = 0; i < dir_sz; i++) {
		if (dir[i].d_ino != 0) {
			if (iflag)
				fprintf(stdout, "%6d %-15.14s%c", dir[i].d_ino, dir[i].d_name,
					++j % col ? ' ' : '\n');
			else
				fprintf(stdout, "%-15.14s%c", dir[i].d_name,
					++j % col ? ' ' : '\n');
		}
	}

	if (j % col)
		fprintf(stdout, "\n");

	if (flag) {
		readdir(&save_dir);
		cur_dir.d_ino = save_dir.d_ino;
		strcpy(cur_dir.d_name,save_dir.d_name);
	}
}

static int cpdir(int ac, char **av)	/* copy contents of current directory */
{
	char *arg[2];
	int i;

	for (i = 0; i < dir_sz; i++) {
		if (dir[i].d_ino != 0 && dir[i].d_name[0] != '.') {
			arg[0] = "cp";
			arg[1] = dir[i].d_name;
			cp(1, arg);
		}
	}

	return 0;
}

static int cd(int ac, char **av)		/* returns 0 if successful, else -1 */
{	/* returns to previous directory if unsuccesssful */
	int i, ino;
	char *c, *name;
	struct v7_direct *pdir;

	name = av[1];

	if ((i = iindex(name, '/')) >= 0) {
		if (i == 0) {
			readdir(&root);
			name++;
		}

		if (name[0] == '\0') {
			cur_dir.d_ino = root.d_ino;
			strcpy(cur_dir.d_name,root.d_name);
			return 0;
		}

		if (*((c = strrchr(name, '/')) + 1) == '\0')
			*c = '\0';
			/* removes trailing '/' if present */

		while ((i = iindex(name, '/')) != -1) {
			name[i] = '\0';

			if ((pdir = find(name)) == NULL) {
				fprintf(stderr, "v7fs: cannot find %s\n", name);
				readdir(&cur_dir);
				return -1;
			}

			if (readdir(pdir) < 0)
				return -1;

			name += i + 1;
		}
	}

	if ((pdir = find(name)) == NULL) {
		fprintf(stderr, "v7fs: cannot find %s\n", name);
		readdir(&cur_dir);
		return -1;
	}

	ino = pdir->d_ino;

	if (readdir(pdir) >= 0) {
		cur_dir.d_ino = ino;
		strcpy(cur_dir.d_name, name);
		return 0;
	} else {
		return -1;
	}
}

static int iindex(char *s, char c)
{
	int i;

	for (i = 0; ; i++) {
		if (s[i] == c)
			return i;

		if (s[i] == '\0')
			return -1;
	}
}

static struct v7_direct *find(char *name)	/* returns pointer to "name" entry */
{						/* in dir[], NULL if not found */
	int i;

	for (i = 0; i < dir_sz; i++)
		if (strcmp(dir[i].d_name, name) == 0 && dir[i].d_ino != 0)
			break;

	if (i == dir_sz)
		return NULL;

	return &(dir[i]);
}

static void cp(int ac, char **arg)	/* copies ifile to ofile if arg[0]=="cp" */
					/* cats ifile if arg[0]=="cat" */
{
	int n, blk, flag, mode;
	v7_ino_t ino;
	char *tname, *ofile;
	struct v7_direct *pdir, save_dir;

	flag = 0;

	if ((ino = (v7_ino_t)atoi(arg[1])) > 0) {
		if (readi(ino, &fil_ino) < 0)
			return;
	} else {
		if (iindex(arg[1], '/') != -1) {
			flag = 1;
			save_dir.d_ino = cur_dir.d_ino;
			strcpy(save_dir.d_name, cur_dir.d_name);
			tname = strchr(arg[1], '/');

			if (tname == arg[1])	/* file in root directory */
				arg[1] = root.d_name;
			*tname = '\0';

			if (cd(ac, arg) < 0)
				return;
			arg[1] = tname + 1;
		}

		if ((pdir = find(arg[1])) == NULL) {
			fprintf(stderr, "v7fs: cannot find %s\n",arg[1]);
			goto quit;
		}

		if (readi(pdir->d_ino, &fil_ino) < 0)
			goto quit;
	}

	if (strcmp(arg[0], "cp") == 0)
		mode = 1;
	else
		mode = 0;

	if (ac < 3 && mode == 1)
		ofile = arg[1];
	else
		ofile = arg[2];

	if (mode == 1 && (fpout = fopen(ofile, "r")) != NULL) {
		fclose(fpout);
		fprintf(stderr, "v7fs: %s already exists\n", ofile);
		goto quit;
	}

	if (mode == 1)  {
		if ((fpout = fopen(ofile, "w")) == NULL) {
			fprintf(stderr, "v7fs: cannot create %s\n", ofile);
			goto quit;
		}
	} else {
		signal(SIGINT,flush);
		fpout = stdout;
	}

	if (!(fil_ino.di_mode & V7_IFREG)) {
		/* is special file or directory */
		fprintf(stderr, "v7fs: %s not a regular file\n", arg[1]);
		goto quit;
	}

	size = fil_ino.di_size;

	for (blk = 0; size > 0L && blk < V7_NADDR - 3; blk++, size -= V7_BLKSIZE) {
		n = readblk(fil_ino.di_addr[blk], buf);

		if (n > 0) {
			if (fwrite(buf, 1, (int)size > n ? n : (int)size, fpout) < 0) {
				if (mode == 1)
					fprintf(stderr, "v7fs: write error\n");
				return;
			}
		}
	}

	if (size > 0L)
		rindir(fpout, fil_ino.di_addr[V7_NADDR - 3], mode);

	if (size > 0L)
		riindir(fpout, fil_ino.di_addr[V7_NADDR - 2], mode);

	if (size > 0L)
		riiindir(fpout, fil_ino.di_addr[V7_NADDR - 1], mode);

quit:
	if (fpout != stdout)
		fclose(fpout);

	signal(SIGINT,SIG_IGN);

	if (flag) {
		readdir(&save_dir);
		cur_dir.d_ino = save_dir.d_ino;
		strcpy(cur_dir.d_name, save_dir.d_name);
	}
}


static void rindir(FILE *fpout, v7_daddr_t blk, int mode)	/* read indirect block and blocks it specifies */
								/* mode as in cp()	*/
{
	int i, count;
	v7_daddr_t iblk[V7_DAPB];

	if (readblk(blk, buf) < 0) {
		size = 0;
		return;
	}

	for (i = 0; i < V7_DAPB; i++)
		iblk[i] = extract32(buf + i * 4);

	for (i = 0; size > 0L && i < V7_DAPB; i++, size -= V7_BLKSIZE) {
		count = readblk(iblk[i], buf);

		if (count > 0) {
			if (fwrite(buf, 1, ((int)size>count) ? count: (int)size, fpout) < 0) {
				if (mode == 1)
					fprintf(stderr, "v7fs: write error\n");
				return;
			}
		}
	}
}

static void riindir(FILE *fpout, v7_daddr_t blk, int mode)	/* read double-indirect blocks */
{
	int i;
	v7_daddr_t iiblk[V7_DAPB];

	if (readblk(blk, buf) < 0) {
		size = 0;
		return;
	}

	for (i = 0; i < V7_DAPB; i++)
		iiblk[i] = extract32(buf + i * 4);

	for (i = 0; size > 0L && i < V7_DAPB; i++, size -= V7_BLKSIZE)
		rindir(fpout, iiblk[i], mode);
}

static void riiindir(FILE *fpout, v7_daddr_t blk, int mode)	/* read triple-indirect blocks */
{
	int i;
	v7_daddr_t iiiblk[V7_DAPB];

	if (readblk(blk, buf) < 0) {
		size = 0;
		return;
	}

	for (i = 0; i < V7_DAPB; i++)
		iiiblk[i] = extract32(buf + i * 4);

	for (i = 0; size > 0L && i < V7_DAPB; i++, size -= V7_BLKSIZE)
		riindir(fpout, iiiblk[i], mode);
}

static int readdir(struct v7_direct *pdir)		/* reads pdir->d_inode into dir_ino, then */
							/* reads corresponding directory into dir */
{							/* reads cur_dir on error, then returns -1 */
	int blk, i;
	v7_daddr_t indir[V7_DAPB];
	v7_off_t sz;

	if (pdir->d_ino == 0)
		return -1;

	if (readi(pdir->d_ino, &dir_ino) < 0)
		return -1;

	if (!(dir_ino.di_mode & V7_IFDIR)) {	/* not a directory */
		fprintf(stderr, "v7fs: %s not a directory\n", pdir->d_name);
		readdir(&cur_dir);
		return -1;
	}

	size = dir_ino.di_size;
	sz   = 0;

	for (blk = 0; size > 0 && blk < 10; blk++) {
		if (readblk(dir_ino.di_addr[blk], buf) < V7_BLKSIZE)
			return -1;

		for (i = 0; i < V7_DIRBLK && size > 0; i++) {
			dir[sz].d_ino = extract16(buf + 0 + i * V7_DIRECT);
			memcpy(dir[sz].d_name, buf + 2 + i * V7_DIRECT, V7_DIRSIZ);
			size -= V7_DIRECT;
			sz++;
		}
	}

	if (size > 0L) {
		if (readblk(dir_ino.di_addr[10], buf) < V7_BLKSIZE)
			return -1;

		for (i = 0; i < V7_DAPB; i++)
			indir[i] = extract32(buf + i * 4);

		for (blk = 0; size > 0 && sz < MAXSIZE && blk < V7_DAPB; blk++) {
			if (readblk(indir[blk], buf) < V7_BLKSIZE)
				return -1;

			for (i = 0; i < V7_DIRBLK && size > 0; i++) {
				dir[sz].d_ino = extract16(buf + 0 + i * V7_DIRECT);
				memcpy(dir[sz].d_name, buf + 2 + i * V7_DIRECT, V7_DIRSIZ);
				size -= V7_DIRECT;
				sz++;
			}
		}
	}

	if (size > 0L) {
		fprintf(stderr, "v7fs: directory has more than %d entries\n", MAXSIZE);
		return -1;
	}

	dir_sz = sz;

	qsort(dir, dir_sz, sizeof(struct v7_direct), dirsort);

	return dir_sz;
}

static int dirsort(const void *a, const void *b)
{
	const struct v7_direct *sa = (struct v7_direct *)a;
	const struct v7_direct *sb = (struct v7_direct *)b;

	return strcmp(sa->d_name, sb->d_name);
}

static int readi(v7_ino_t inum, struct v7_dinode *pinode)	/* reads inode inum into *pinode */
{
	int i;

	if (inum < 1 || inum > imax) {
		fprintf(stderr, "v7fs: bad inode number, %u\n", inum);
		return -1;
	}

	if ((fseek(fp, (long)2 * V7_BLKSIZE + (long)(inum - 1) * V7_DINODE, SEEK_SET) < 0) ||
	    (fread(buf, 1, V7_DINODE, fp) < 0)) {
		fprintf(stderr, "v7fs: can't read inode %u\n", inum);
		return -1;
	}

	pinode->di_mode  = extract16(buf + 0);
	pinode->di_nlink = extract16(buf + 2);
	pinode->di_uid   = extract16(buf + 4);
	pinode->di_gid   = extract16(buf + 6);
	pinode->di_size  = extract32(buf + 8);

	for (i = 0; i < V7_NADDR; i++)
		pinode->di_addr[i] = extract24(buf + 12 + i * 3);

	pinode->di_atime = extract32(buf + 52);
	pinode->di_mtime = extract32(buf + 56);
	pinode->di_ctime = extract32(buf + 60);

	return 0;
}

static int readblk(v7_daddr_t blk, void *buff)	/* reads block blk into buff */
{
	int n;

	if (blk == 0) {
		memset(buff, '\0', V7_BLKSIZE);
		return V7_BLKSIZE;
	}

	if (blk < sb.s_isize - 1 || blk >= sb.s_fsize) {
		fprintf(stderr, "v7fs: bad block number, %d\n", blk);
		return -1;
	}

	if (fseek(fp, (long)(V7_BLKSIZE * blk), SEEK_SET) < 0) {
		fprintf(stderr, "v7fs: seek error, block %d\n", blk);
		return -1;
	}

	if ((n = fread(buff, 1, V7_BLKSIZE, fp)) != V7_BLKSIZE)
		fprintf(stderr, "v7fs: read error, block %d\n", blk);

	return n;
}

static void flush(int sig) {		/* closes fo to terminate a cat() */
	fclose(fpout);
}

static void msg(void) {
	printf("commands:\n");
	printf("\t ls [-i] [dir]: list directory contents, current dir default\n");
	printf("\t cd name: change to directory 'name'\n");
	printf("\t cat name1: print file 'name1' on terminal\n");
	printf("\t cp name1 [name2]: copy internal file 'name1' to external 'name2'\n");
	printf("\t\tname2 defaults to name1.\n");
	printf("\t\t(An i-number can be used instead of name1 for cp or cat.)\n");
	printf("\t cpdir: copy all files in current internal directory\n");
	printf("\t\tto current external directory\n");
	printf("\t lcd name: change to local directory 'name'\n");
	printf("\t printi ino ...: print contents of inode 'ino'\n");
	printf("\t printblk blk ...: print contents of block 'blk'\n");
	printf("\t printsb: print contents of the super block\n");
	printf("\t dumpblk blk ...: hex dump of block 'blk'\n");
	printf("\t dumpboot: hex dump of the boot block\n");
	printf("\t cpblk file blk ...: copy contents of 'blk' to external ");
	printf("file 'file'\n\t\t(append to file if it exists)\n");
	printf("\t rootino ino: read directory with inode 'ino', making it");
	printf("\n\t\tthe root directory\n");
	printf("\t ! : shell escape; the rest of the line is passed to the shell\n");
	printf("\t q or ^d: quit\n");
}

static void printi(int ac, char **av)
{
	int i;
	v7_ino_t inum;
	time_t timec;

	while (--ac) {
		inum = (v7_ino_t)atoi(*++av);

		fprintf(stdout, "\nInode %u\n", inum);

		if (readi(inum, &fil_ino) < 0)
			return;

		fprintf(stdout, "flags: %o\tnlinks: %d\tuid: %d\tgid: %d\n",
			fil_ino.di_mode, fil_ino.di_nlink, fil_ino.di_uid,
			fil_ino.di_gid);
		fprintf(stdout, "size: %d\n", fil_ino.di_size);
		fprintf(stdout, "blocks:");

		for (i = 0; i < V7_NADDR; i++)
			fprintf(stdout, " %d", fil_ino.di_addr[i]);
		fprintf(stdout, "\n");

		timec = fil_ino.di_atime;
		fprintf(stdout, "access time: %s", ctime(&timec));

		timec = fil_ino.di_mtime;
		fprintf(stdout, "last modified: %s", ctime(&timec));

		timec = fil_ino.di_ctime;
		fprintf(stdout, "time created: %s", ctime(&timec));
	}
}

static void printsb(int ac, char **av)
{
	time_t timec;
	int i, j, space;

	fprintf(stdout, "Super Block\n");

	fprintf(stdout, "i-list size: %d blks\n", sb.s_isize);
	fprintf(stdout, "volume size: %d blks\n", sb.s_fsize);

	fprintf(stdout, "free blocks: ");

	space = 0;
	for (i = 0, j = 1; i < sb.s_nfree; i++, j++) {
		if (space)
			fprintf(stdout, "             ");

		fprintf(stdout, "%d", sb.s_free[i]);

		if ((j % 10) == 0) {
			fprintf(stdout, "\n");
			space = 1;
		} else {
			fprintf(stdout, " ");
			space = 0;
		}
	}
	fprintf(stdout, "\n");

	timec = sb.s_time;
	fprintf(stdout, "last update: %s", ctime(&timec));
}

static void printblk(int ac, char **av)
{
	int count;
	v7_daddr_t blk;

	while (--ac) {
		blk = (v7_daddr_t)atoi(*++av);

		if ((count = readblk(blk, buf)) > 0)
			fwrite(buf, 1, count, stdout);
	}
}

static void hexdump(unsigned char *buf, int count)
{
	int offset, i;

	for (offset = 0; offset < count; offset += 16) {
		fprintf(stdout, "%04X   ", offset);

		for (i = 0; i < 16; i++) {
			if ((offset + i) < count)
				fprintf(stdout, "%02X ", buf[offset + i]);
			else
				fprintf(stdout, "   ");
		}

		fprintf(stdout, "   *");

		for (i = 0; i < 16; i++) {
			if ((offset + i) < count)
				if (buf[offset + i] >= 0x20 && buf[offset + i] < 0x7F)
					fprintf(stdout, "%c", buf[offset + i]);
				else
					fprintf(stdout, ".");
		}

		fprintf(stdout, "*\n");
	}
}

static void dumpblk(int ac, char **av)
{
	int count;
	v7_daddr_t blk;

	while (--ac) {
		blk = (v7_daddr_t)atoi(*++av);

		if ((count = readblk(blk, buf)) > 0) {
			fprintf(stdout, "block %d:\n", blk);
			hexdump(buf, count);
		}
	}
}

static void dumpboot(int ac, char **av)
{
	fprintf(stdout, "boot block:\n");
	hexdump(bb, V7_BLKSIZE);
}

static void cpblk(int ac, char **av)
{
	int count;
	v7_daddr_t blk;
	FILE *fpout;

	if (ac < 3) {
		fprintf(stdout, "usage: cpblk file blk ...\n");
		return;
	}

	if ((fpout = fopen(av[1], "w")) < 0) {
		fprintf(stderr,"v7fs: cannot create %s\n", av[1]);
		return;
	}

	fseek(fp, 0L, SEEK_END);

	--ac;
	av++;

	while (--ac) {
		blk = (v7_daddr_t)atoi(*++av);

		if ((count = readblk(blk, buf)) < 0)
			return;

		fwrite(buf, 1, count, fpout);
	}

	fclose(fpout);
}

static void readroot(v7_ino_t ino)	/* make root directory that specified by ino */
{
	root.d_ino = ino;

	readdir(&root);
}

static int16 extract16(unsigned char *s)
{
	return (s[0] << 0) | (s[1] << 8);
}

static int32 extract24(unsigned char *s)
{
	return (s[1] << 0) | (s[2] << 8) | (s[0] << 16);
}

static int32 extract32(unsigned char *s)
{
	return (s[2] << 0) | (s[3] << 8) | (s[0] << 16) | (s[1] << 24);
}
