/******************************************************************
Copyright 1987 by Apollo Computer Inc., Chelmsford, Massachusetts.
Copyright 1989 by Hewlett-Packard Company.

                        All Rights Reserved

Permission to use, duplicate, change, and distribute this software and
its documentation for any purpose and without fee is granted, provided
that the above copyright notice appear in such copy and that this
copyright notice appear in all supporting documentation, and that the
names of Apollo Computer Inc., the Hewlett-Packard Company, or MIT not
be used in advertising or publicity pertaining to distribution of the
software without written prior permission.

HEWLETT-PACKARD MAKES NO WARRANTY OF ANY KIND WITH REGARD
TO THIS SOFTWARE, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE.  Hewlett-Packard shall not be liable for errors
contained herein or direct, indirect, special, incidental or
consequential damages in connection with the furnishing,
performance, or use of this material.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.
******************************************************************/

/*
 * Functions implementing Apollo-keyboard-specific parts of the driver
 * having to do with input handling.
 *
 * This code handles "North American" keyboards, that is keyboard types 2 and
 * 3 (but not 3a, 3b, etc.).  Since IDM simulates (as well as it can) a
 * keyboard number 3 when we have a keyboard 2, we have nearly common code
 * for those two keyboards.
 */

#include "apollo.h"

/* ignore whole rest of file if pre-SR10.2 */
#ifndef NO_PHYS_KEYS

#define XK_MISCELLANY
#define XK_LATIN1
#include "keysymdef.h"
#include "ap_keysym.h"

#include <apollo/kbd.h>

typedef unsigned long idm_$keymap_t[4];

typedef unsigned short aws_$kbd_modifiers_t;
typedef unsigned short aws_$kbd_modifiers_set_t;

#define  aws_$shift     0
#define  aws_$lock      1
#define  aws_$control   2
#define  aws_$mod1      3
#define  aws_$mod2      4
#define  aws_$mod3      5
#define  aws_$mod4      6
#define  aws_$mod5      7
#define  aws_$repeating 8

extern void aws_$set_modifiers (
    aws_$kbd_modifiers_t       &modf,
    idm_$keymap_t               map,
    status_$t                  *status
);

typedef short enum {
        aws_$repeat_on,
        aws_$repeat_off,
        aws_$repeat_default
} aws_$auto_repeat_mode_t;

typedef short enum {
        aws_$delay_fast,
        aws_$delay_normal,
        aws_$delay_default
} aws_$auto_repeat_delay_t;

extern void aws_$set_auto_repeat_mode (
    aws_$auto_repeat_mode_t    &mode,
    status_$t                  *status
);

extern void aws_$get_auto_repeat_mode (
    aws_$auto_repeat_mode_t    *mode,
    status_$t                  *status
);

extern void aws_$set_auto_repeat (
    aws_$auto_repeat_mode_t    &mode,
    idm_$keymap_t               map,
    status_$t                  *status
);

extern void aws_$get_auto_repeat (
    idm_$keymap_t               map,
    status_$t                  *status
);

extern void aws_$set_auto_repeat_delay (
    aws_$auto_repeat_delay_t   &delay,
    idm_$keymap_t               map,
    status_$t                  *status
);

extern void aws_$get_auto_repeat_delay (
    aws_$auto_repeat_delay_t   &delay,
    idm_$keymap_t               map,
    status_$t                  *status
);


/*
 * Save autorepeat settings here to put them back on exit.
 */
static aws_$auto_repeat_mode_t  savedARMode;
static idm_$keymap_t            savedAR;
static idm_$keymap_t            savedARDelay;

/*
 * Keyboard type number.  If initialized, will equal either 2 or 3.
 */
static int kbdType;

#define N_KEY_CODES 128

/* 
 * Entry in the keyboard description table (data for initial keymap).
 */
#define NA_GLYPHS_PER_KEY 2

typedef KeySym apNAKeySymList[NA_GLYPHS_PER_KEY];

typedef struct _keyDesc
{
    int             key_type;
    apNAKeySymList  key_syms;
} keyDescRec;

/* 
 * Values for the key_type field.
 */
#define KEYT_NONEXISTENT   0
#define KEYT_NOT_ON_KBD2   1
#define KEYT_NO_UP_ON_KBD2 2
#define KEYT_UP_ON_BOTH    3

/*
 * The keyboard description table, indexed by hardware keycode (hereafter called "action").
 * This table contains inherent hardware information about key codes on the 2 North American
 * keyboards, and also the info for the initial contents of the keymap.
 *
 * Meaning of the key_type field:
 *      KEYT_NONEXISTENT        Neither North American kbd has this action.
 *      KEYT_NOT_ON_KBD2        This action cannot be generated by kbd 2.
 *      KEYT_NO_UP_ON_KBD2      This key has no hardware up transition on kbd 2 (therefore not
 *                              a candiate for modifiers -- see LegalModifier).
 *      KEYT_UP_ON_BOTH         This key exists and has hardware up transitions on both
 *                              North American kbds.
 *
 * Meaning of the key_syms field:
 *      A list of the keysyms initially bound to this key.
 */
static keyDescRec kbdDescTable[N_KEY_CODES] = {
/* 0x00 = ESC  */   KEYT_NO_UP_ON_KBD2, { XK_Escape         , NoSymbol           },
/* 0x01 = L1   */   KEYT_UP_ON_BOTH   , { XK_Select         , XK_Insert          },
/* 0x02 = L2   */   KEYT_UP_ON_BOTH   , { apXK_LineDel      , NoSymbol           },
/* 0x03 = L3   */   KEYT_UP_ON_BOTH   , { apXK_CharDel      , NoSymbol           },
/* 0x04 = L1A  */   KEYT_UP_ON_BOTH   , { apXK_Copy         , apXK_Cut           },
/* 0x05 = L2A  */   KEYT_UP_ON_BOTH   , { apXK_Paste        , XK_Undo            },
/* 0x06 = L3A  */   KEYT_UP_ON_BOTH   , { apXK_Grow         , apXK_Move          },
/* 0x07 = L4   */   KEYT_UP_ON_BOTH   , { apXK_LeftBar      , NoSymbol           },
/* 0x08 = L5   */   KEYT_UP_ON_BOTH   , { apXK_Cmd          , apXK_Shell         },
/* 0x09 = L6   */   KEYT_UP_ON_BOTH   , { apXK_RightBar     , NoSymbol           },
/* 0x0A = L7   */   KEYT_UP_ON_BOTH   , { apXK_LeftBox      , NoSymbol           },
/* 0x0B = L8   */   KEYT_UP_ON_BOTH   , { XK_Up             , NoSymbol           },
/* 0x0C = L9   */   KEYT_UP_ON_BOTH   , { apXK_RightBox     , NoSymbol           },
/* 0x0D = LA   */   KEYT_UP_ON_BOTH   , { XK_Left           , NoSymbol           },
/* 0x0E = LB   */   KEYT_UP_ON_BOTH   , { XK_Next           , NoSymbol           },
/* 0x0F = LC   */   KEYT_UP_ON_BOTH   , { XK_Right          , NoSymbol           },
/* 0x10 = LD   */   KEYT_UP_ON_BOTH   , { apXK_UpBox        , NoSymbol           },
/* 0x11 = LE   */   KEYT_UP_ON_BOTH   , { XK_Down           , NoSymbol           },
/* 0x12 = LF   */   KEYT_UP_ON_BOTH   , { apXK_DownBox      , NoSymbol           },
/* 0x13 = F0   */   KEYT_UP_ON_BOTH   , { XK_F10            , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x14 = F1   */   KEYT_UP_ON_BOTH   , { XK_F1             , NoSymbol           },
/* 0x15 = F2   */   KEYT_UP_ON_BOTH   , { XK_F2             , NoSymbol           },
/* 0x16 = F3   */   KEYT_UP_ON_BOTH   , { XK_F3             , NoSymbol           },
/* 0x17 = F4   */   KEYT_UP_ON_BOTH   , { XK_F4             , NoSymbol           },
/* 0x18 = F5   */   KEYT_UP_ON_BOTH   , { XK_F5             , NoSymbol           },
/* 0x19 = F6   */   KEYT_UP_ON_BOTH   , { XK_F6             , NoSymbol           },
/* 0x1A = F7   */   KEYT_UP_ON_BOTH   , { XK_F7             , NoSymbol           },
/* 0x1B = F8   */   KEYT_UP_ON_BOTH   , { XK_F8             , NoSymbol           },
/* 0x1C = F9   */   KEYT_UP_ON_BOTH   , { XK_F9             , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x1D = R1   */   KEYT_UP_ON_BOTH   , { apXK_Pop          , NoSymbol           },
/* 0x1E = R2   */   KEYT_UP_ON_BOTH   , { XK_Redo           , NoSymbol           },
/* 0x1F = R3   */   KEYT_UP_ON_BOTH   , { apXK_Read         , NoSymbol           },
/* 0x20 = R4   */   KEYT_UP_ON_BOTH   , { apXK_Edit         , apXK_Save          },
/* 0x21 = R5   */   KEYT_UP_ON_BOTH   , { apXK_Exit         , XK_Cancel          },
/* 0x22 = R6   */   KEYT_UP_ON_BOTH   , { XK_Pause          , XK_Help            },
/* 0x23 = NP0  */   KEYT_NO_UP_ON_KBD2, { XK_KP_0           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x24 = NP1  */   KEYT_NO_UP_ON_KBD2, { XK_KP_1           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x25 = NP2  */   KEYT_NO_UP_ON_KBD2, { XK_KP_2           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x26 = NP3  */   KEYT_NO_UP_ON_KBD2, { XK_KP_3           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x27 = NP4  */   KEYT_NO_UP_ON_KBD2, { XK_KP_4           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x28 = NP5  */   KEYT_NO_UP_ON_KBD2, { XK_KP_5           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x29 = NP6  */   KEYT_NO_UP_ON_KBD2, { XK_KP_6           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x2A = NP7  */   KEYT_NO_UP_ON_KBD2, { XK_KP_7           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x2B = NP8  */   KEYT_NO_UP_ON_KBD2, { XK_KP_8           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x2C = NP9  */   KEYT_NO_UP_ON_KBD2, { XK_KP_9           , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x2D = NPA  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x2E = NPB  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x2F = NPC  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x30 = NPD  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x31 = NPE  */   KEYT_NO_UP_ON_KBD2, { XK_KP_Enter       , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x32 = NPF  */   KEYT_NO_UP_ON_KBD2, { XK_KP_Subtract    , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x33 = NPG  */   KEYT_NO_UP_ON_KBD2, { XK_KP_Add         , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x34 = NPP  */   KEYT_NO_UP_ON_KBD2, { XK_KP_Decimal     , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x35 = ALTL */   KEYT_UP_ON_BOTH   , { XK_Alt_L          , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x36 = ALTR */   KEYT_UP_ON_BOTH   , { XK_Alt_R          , NoSymbol           },	/* NOT_ON_KBD2 */
/* 0x37 = SHFL */   KEYT_UP_ON_BOTH   , { XK_Shift_L        , NoSymbol           },	/* NO_UP_ON_KBD2 */
/* 0x38 = SHFR */   KEYT_UP_ON_BOTH   , { XK_Shift_R        , NoSymbol           },	/* NO_UP_ON_KBD2 */
/* 0x39 = LOCK */   KEYT_UP_ON_BOTH   , { XK_Caps_Lock      , NoSymbol           },	/* NO_UP_ON_KBD2 */
/* 0x3A = CTRL */   KEYT_UP_ON_BOTH   , { XK_Control_L      , NoSymbol           },	/* NO_UP_ON_KBD2 */
/* 0x3B = REPT */   KEYT_NO_UP_ON_KBD2, { apXK_Repeat       , NoSymbol           },
/* 0x3C = TAB  */   KEYT_NO_UP_ON_KBD2, { XK_Tab            , NoSymbol           },
/* 0x3D = RET  */   KEYT_NO_UP_ON_KBD2, { XK_Return         , NoSymbol           },
/* 0x3E = BS   */   KEYT_NO_UP_ON_KBD2, { XK_BackSpace      , NoSymbol           },
/* 0x3F = DEL  */   KEYT_NO_UP_ON_KBD2, { XK_Delete         , NoSymbol           },
/* 0x40 = B00  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x41 = B01  */   KEYT_NO_UP_ON_KBD2, { XK_z              , XK_Z               },
/* 0x42 = B02  */   KEYT_NO_UP_ON_KBD2, { XK_x              , XK_X               },
/* 0x43 = B03  */   KEYT_NO_UP_ON_KBD2, { XK_c              , XK_C               },
/* 0x44 = B04  */   KEYT_NO_UP_ON_KBD2, { XK_v              , XK_V               },
/* 0x45 = B05  */   KEYT_NO_UP_ON_KBD2, { XK_b              , XK_B               },
/* 0x46 = B06  */   KEYT_NO_UP_ON_KBD2, { XK_n              , XK_N               },
/* 0x47 = B07  */   KEYT_NO_UP_ON_KBD2, { XK_m              , XK_M               },
/* 0x48 = B08  */   KEYT_NO_UP_ON_KBD2, { XK_comma          , XK_less            },
/* 0x49 = B09  */   KEYT_NO_UP_ON_KBD2, { XK_period         , XK_greater         },
/* 0x4A = B10  */   KEYT_NO_UP_ON_KBD2, { XK_slash          , XK_question        },
/* 0x4B = B11  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x4C = B12  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x4D = SP   */   KEYT_NO_UP_ON_KBD2, { XK_space          , NoSymbol           },
/* 0x4E = B98  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x4F = B99  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x50 = C00  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x51 = C01  */   KEYT_NO_UP_ON_KBD2, { XK_a              , XK_A               },
/* 0x52 = C02  */   KEYT_NO_UP_ON_KBD2, { XK_s              , XK_S               },
/* 0x53 = C03  */   KEYT_NO_UP_ON_KBD2, { XK_d              , XK_D               },
/* 0x54 = C04  */   KEYT_NO_UP_ON_KBD2, { XK_f              , XK_F               },
/* 0x55 = C05  */   KEYT_NO_UP_ON_KBD2, { XK_g              , XK_G               },
/* 0x56 = C06  */   KEYT_NO_UP_ON_KBD2, { XK_h              , XK_H               },
/* 0x57 = C07  */   KEYT_NO_UP_ON_KBD2, { XK_j              , XK_J               },
/* 0x58 = C08  */   KEYT_NO_UP_ON_KBD2, { XK_k              , XK_K               },
/* 0x59 = C09  */   KEYT_NO_UP_ON_KBD2, { XK_l              , XK_L               },
/* 0x5A = C10  */   KEYT_NO_UP_ON_KBD2, { XK_semicolon      , XK_colon           },
/* 0x5B = C11  */   KEYT_NO_UP_ON_KBD2, { XK_quoteright     , XK_quotedbl        },
/* 0x5C = C12  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x5D = C13  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x5E = C14  */   KEYT_NO_UP_ON_KBD2, { XK_backslash      , XK_bar             },
/* 0x5F = C99  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x60 = D00  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x61 = D01  */   KEYT_NO_UP_ON_KBD2, { XK_q              , XK_Q               },
/* 0x62 = D02  */   KEYT_NO_UP_ON_KBD2, { XK_w              , XK_W               },
/* 0x63 = D03  */   KEYT_NO_UP_ON_KBD2, { XK_e              , XK_E               },
/* 0x64 = D04  */   KEYT_NO_UP_ON_KBD2, { XK_r              , XK_R               },
/* 0x65 = D05  */   KEYT_NO_UP_ON_KBD2, { XK_t              , XK_T               },
/* 0x66 = D06  */   KEYT_NO_UP_ON_KBD2, { XK_y              , XK_Y               },
/* 0x67 = D07  */   KEYT_NO_UP_ON_KBD2, { XK_u              , XK_U               },
/* 0x68 = D08  */   KEYT_NO_UP_ON_KBD2, { XK_i              , XK_I               },
/* 0x69 = D09  */   KEYT_NO_UP_ON_KBD2, { XK_o              , XK_O               },
/* 0x6A = D10  */   KEYT_NO_UP_ON_KBD2, { XK_p              , XK_P               },
/* 0x6B = D11  */   KEYT_NO_UP_ON_KBD2, { XK_bracketleft    , XK_braceleft       },
/* 0x6C = D12  */   KEYT_NO_UP_ON_KBD2, { XK_bracketright   , XK_braceright      },
/* 0x6D = D13  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x6E = D98  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x6F = D99  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x70 = E00  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x71 = E01  */   KEYT_NO_UP_ON_KBD2, { XK_1              , XK_exclam          },
/* 0x72 = E02  */   KEYT_NO_UP_ON_KBD2, { XK_2              , XK_at              },
/* 0x73 = E03  */   KEYT_NO_UP_ON_KBD2, { XK_3              , XK_numbersign      },
/* 0x74 = E04  */   KEYT_NO_UP_ON_KBD2, { XK_4              , XK_dollar          },
/* 0x75 = E05  */   KEYT_NO_UP_ON_KBD2, { XK_5              , XK_percent         },
/* 0x76 = E06  */   KEYT_NO_UP_ON_KBD2, { XK_6              , XK_asciicircum     },
/* 0x77 = E07  */   KEYT_NO_UP_ON_KBD2, { XK_7              , XK_ampersand       },
/* 0x78 = E08  */   KEYT_NO_UP_ON_KBD2, { XK_8              , XK_asterisk        },
/* 0x79 = E09  */   KEYT_NO_UP_ON_KBD2, { XK_9              , XK_parenleft       },
/* 0x7A = E10  */   KEYT_NO_UP_ON_KBD2, { XK_0              , XK_parenright      },
/* 0x7B = E11  */   KEYT_NO_UP_ON_KBD2, { XK_minus          , XK_underscore      },
/* 0x7C = E12  */   KEYT_NO_UP_ON_KBD2, { XK_equal          , XK_plus            },
/* 0x7D = E13  */   KEYT_NO_UP_ON_KBD2, { XK_quoteleft      , XK_asciitilde      },
/* 0x7E = E14  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
/* 0x7F = E99  */   KEYT_NONEXISTENT  , { NoSymbol          , NoSymbol           },
};

/*
 * Table describing what keys exist for the server (may differ between kbd2 and kbd3).
 * Indexed by action, as above.
 */
static Bool             keyExists[N_KEY_CODES];
static idm_$keymap_t    idmKeyset;

/*
 * Macros translating between "actions" (range 0-127) and protocol keysyms
 * (range 128-255).
 */
#define AP_ACTION_TO_KEYCODE(action) ((action)+128)
#define AP_KEYCODE_TO_ACTION(keycode) ((keycode)-128)


/*
 * apNALegalModifier -- Driver internal code
 *      This is the implementation of LegalModifier for North American kbds.
 */
Bool
apNALegalModifier(keycode)
    BYTE keycode;
{
    if (!keyExists[AP_KEYCODE_TO_ACTION(keycode)])
        return FALSE;
    if ((kbdType == 2) &&
        (kbdDescTable[AP_KEYCODE_TO_ACTION(keycode)].key_type == KEYT_NO_UP_ON_KBD2))
        return FALSE;

    return TRUE;
}

/*
 * apNAHWInitKbd -- Driver internal code
 *      Perform all "hardware" operations needed to initialize keyboard.
 *      Input argument is the Apollo keyboard type code (as a character '2' or '3').
 */
void
apNAHWInitKbd (kbd_type)
    char        kbd_type;
{
    int             action, kt;
    KeySym          ks;
    gpr_$keyset_t   gprKeyset;
    idm_$keymap_t   idmKeysetNull;
    status_$t       status;

    kbdType = kbd_type - '0';

    for (action = 0 ; action < N_KEY_CODES ; action++)
    {
        kt = kbdDescTable[action].key_type;
        keyExists[action] = ((kt != KEYT_NONEXISTENT) &&
                             ((kbdType != 2) || (kt != KEYT_NOT_ON_KBD2)));
    }

    /*
     * Workaround for bug in beta versions of SR10.2 (fixed in release version):
     *
     * We must enable at least one other key-type event for the
     * gpr_$physical_keys to work.  Since physical_keys events take precedence
     * over function_keys events, the function_keys event here will never be
     * delivered.
     */
    lib_$init_set (gprKeyset, sizeof(gprKeyset)*8);
    lib_$add_to_set (gprKeyset, sizeof(gprKeyset)*8, (short) KBD3_$ESC);
    gpr_$enable_input (gpr_$function_keys, gprKeyset, &status);

    lib_$init_set (gprKeyset, sizeof(gprKeyset)*8);
    lib_$init_set (idmKeyset, sizeof(idmKeyset)*8);
    for (action = 0 ; action < N_KEY_CODES ; action++)
        if (keyExists[action])
        {
            lib_$add_to_set (gprKeyset, sizeof(gprKeyset)*8, (short) action);
            lib_$add_to_set (gprKeyset, sizeof(gprKeyset)*8, (short) (action + KBD3_$KEY_UP));
            lib_$add_to_set (idmKeyset, sizeof(idmKeyset)*8, (short) action);
        }
    gpr_$enable_input (gpr_$physical_keys, gprKeyset, &status);
    aws_$get_auto_repeat_mode (&savedARMode, &status);
    aws_$get_auto_repeat (savedAR, &status);
    aws_$get_auto_repeat_delay (aws_$delay_fast, savedARDelay, &status);

    aws_$set_auto_repeat_delay (aws_$delay_normal, idmKeyset, &status);

    lib_$init_set (idmKeysetNull, sizeof(idmKeysetNull)*8);
    aws_$set_modifiers (aws_$shift, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$lock, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$control, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$mod1, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$mod2, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$mod3, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$mod4, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$mod5, idmKeysetNull, &status);
    aws_$set_modifiers (aws_$repeating, idmKeysetNull, &status);

#ifdef NO_GPR_QUIT
    smd_$set_quit_char ((char) 0, &status);
#else
    gpr_$set_quit_event (gpr_$no_event, 0, &status);
#endif
}

/*
 * apNAHWCloseKbd -- Driver internal code
 *      Perform all "hardware" operations needed to close down keyboard.
 *      In this case, put back autorepeat as we found it (DM should do this).
 */
void
apNAHWCloseKbd()
{
    status_$t   status;

    aws_$set_auto_repeat_mode (savedARMode, &status);
    aws_$set_auto_repeat (aws_$repeat_off, idmKeyset, &status);
    aws_$set_auto_repeat (aws_$repeat_on, savedAR, &status);
    aws_$set_auto_repeat_delay (aws_$delay_fast, savedARDelay, &status);
}

/*
 * apNAGetMappings -- Driver internal code
 *      This code creates the default keymap for North American keyboards.
 */
Bool
apNAGetMappings(pKeySyms, pModMap)
    KeySymsPtr  pKeySyms;
    CARD8      *pModMap;
{
    int             action, min_action, max_action;
    int             kc, min_kc, max_kc, i;
    apNAKeySymList *map;

    min_action = N_KEY_CODES;
    max_action = 0;
    for (action = 0 ; action < N_KEY_CODES ; action++)
        if (keyExists[action])
        {
            max_action = action;
            if (min_action > action) min_action = action;
        }

    map = (apNAKeySymList *) xalloc (sizeof(apNAKeySymList) * MAP_LENGTH);
    if (!map)
        return FALSE;

    pKeySyms->minKeyCode = min_kc = AP_ACTION_TO_KEYCODE(min_action);
    pKeySyms->maxKeyCode = max_kc = AP_ACTION_TO_KEYCODE(max_action);
    pKeySyms->mapWidth = NA_GLYPHS_PER_KEY;
    pKeySyms->map = (KeySym *) map;

    for (kc = 0; kc < MAP_LENGTH; kc++)
    {
        for (i = 0 ; i < NA_GLYPHS_PER_KEY ; i++)
            map[kc][i] = NoSymbol;  /* make sure it is restored */
    }

    for (action = min_action ; action <= max_action ; action++)
        if (keyExists[action])
        {
            for (i = 0 ; i < NA_GLYPHS_PER_KEY ; i++)
                map[AP_ACTION_TO_KEYCODE(action) - min_kc][i] = kbdDescTable[action].key_syms[i];
        }

    for (kc = 0; kc < MAP_LENGTH; kc++)
    {
        if ((kc < min_kc) || (kc > max_kc))
            pModMap[kc] = NoSymbol;
        else
            switch (map[kc - min_kc][0])
            {
            case XK_Shift_L:
            case XK_Shift_R:
                pModMap[kc] = ShiftMask;
                break;
            case XK_Control_L:
                pModMap[kc] = ControlMask;
                break;
            case XK_Caps_Lock:
                pModMap[kc] = LockMask;
                break;
            case XK_Alt_L:
            case XK_Alt_R:
                pModMap[kc] = Mod1Mask;
                break;
            default:
                pModMap[kc] = NoSymbol;
                break;
            }
    }
    return TRUE;
}

/*
 * apNAHandleKey -- Driver internal code
 *      Given the action, generate the X event.
 */
void
apNAHandleKey (xEp, action)
    xEvent         *xEp;
    unsigned char   action;
{
    if (action >= KBD3_$KEY_UP)
    {
        xEp->u.u.type = KeyRelease;
        xEp->u.u.detail = AP_ACTION_TO_KEYCODE(action - KBD3_$KEY_UP);
    }
    else
    {
        xEp->u.u.type = KeyPress;
        xEp->u.u.detail = AP_ACTION_TO_KEYCODE(action);
    }

    (*apKeyboard->processInputProc) (xEp, apKeyboard, 1);
}

/*
 * apNASetRepeat -- Driver internal code
 *      Set autorepeat functions for the keyboard.
 */
void
apNASetRepeat (ctrl)
    KeybdCtrl  *ctrl;
{
    idm_$keymap_t           repeatKeyset;
    int                     kc, action;
    int                     i, bit;
    aws_$auto_repeat_mode_t mode;
    status_$t               status;

    aws_$set_auto_repeat (aws_$repeat_off, idmKeyset, &status);
    lib_$init_set (repeatKeyset, sizeof(repeatKeyset)*8);
    for (kc = 0; kc < MAP_LENGTH; kc++)
    {
        i = kc >> 3;
        bit = 1 << (kc & 7);
        if ((ctrl->autoRepeats[i]) & bit)
            lib_$add_to_set (repeatKeyset, sizeof(repeatKeyset)*8,
                             (short) AP_KEYCODE_TO_ACTION(kc));
    }
    aws_$set_auto_repeat (aws_$repeat_on, repeatKeyset, &status);
    mode = (ctrl->autoRepeat) ? aws_$repeat_on : aws_$repeat_off;
    aws_$set_auto_repeat_mode (mode, &status);
}

#endif  /* NO_PHYS_KEYS */
