/* 
 * cc autoclock.c -bx autoclock -info 4 -db -runtype any -systype any -pic
 */

#include <apollo/base.h>
#include <apollo/cal.h>
#include <apollo/errlog.h>
#include <apollo/fault.h>
#include <apollo/ios.h>
#include <apollo/pad.h>
#include <apollo/pfm.h>
#include <apollo/pgm.h>
#include <apollo/pad.h>
#include <apollo/time.h>
#include <apollo/vfmt.h>

void
pm_$set_my_name
(
    char *proc_name,
    short &name_len,
    status_$t *status
);

#define string(x) (x),(sizeof(x)-1)

#define version string("Autoclock version 1.1")
#define logfile string("`node_data/system_logs/autoclock")

static ios_$id_t pad;
static boolean pad_valid = false;

void
oops
(
    status_$t   &status,
    char        *message
)
{
    status_$t st2;
    ios_$id_t log;

    if( status.all == fault_$quit || status.all == fault_$stop )
    {
        if( pad_valid )
        {
            pad_$dm_cmd(pad, string("msg ' * Autoclock shutdown * '"), &st2);
            pad_valid = false;
            ios_$close(pad, &st2);
        }

        pgm_$set_severity(pgm_$ok);
        pgm_$exit();
    }

    log = errlog_$open(logfile, 4096, 60, version, status.all, 
        (errlog_$silent|errlog_$no_divert), &st2);

    if( st2.all == status_$ok )
    {
        vfmt_$ws(log, message);
        errlog_$traceback(log, 0, 0);
        errlog_$close(log);
    }

    if( pad_valid )
    {
        pad_$dm_cmd(pad, string("msg ' * Autoclock faulted * '"), &st2);
        pad_valid = false;
        ios_$close(pad, &st2);  
    }

    pgm_$set_severity(pgm_$error);
    pgm_$exit();
}

void main(void)
{
    status_$t s;
    pfm_$cleanup_rec cleanup_record;

    s = pfm_$cleanup(&cleanup_record);

    if( s.all != pfm_$cleanup_set )
    {
        oops(s, "Autoclock caught a fault%.");
    }
    else
    {
        static pad_$window_desc_t wd = { 0, 0, 500, 100 };

        pm_$set_my_name(string("autoclock"), &s);

        pad_$create_window(string("`node_data/usrtmp/dm_msg_pad"),
            pad_$transcript, 1, wd, &pad, &s);
        if( s.all != status_$ok ) oops(s, "Cannot open pad%.");
        pad_valid = true;

        pad_$make_invisible(pad, 1, &s);
        if( s.all != status_$ok ) oops(s, "Cannot make pad invisible%.");

        pad_$set_auto_close(pad, 1, true, &s);
        if( s.all != status_$ok ) oops(s, "Cannot set pad autoclose%.");

        while( 1 )
        {
            char buf[ 32 ];
            short buflen;
            cal_$timedate_rec_t l;
            time_$clock_t w, i;

            cal_$decode_local_time(&l);
            vfmt_$encode("msg ' * %2zwd/%2zwd %2zwd:%2zwd * '%$", buf, sizeof buf, 
                &buflen, &l.day, &l.month, &l.hour, &l.minute);

            pad_$dm_cmd(pad, buf, buflen, &s);
            if( s.all != status_$ok ) oops(s, "Can't write the time%.");

            cal_$encode_time(l, &w);
            cal_$sec_to_clock(60-l.second, &i);
            cal_$add_clock(&w, i);
            cal_$remove_local_offset(&w);
            time_$wait(time_$absolute, w, &s);
            if( s.all != status_$ok ) oops(s, "Can't wait out the minute%.");
        }
    }
}

