program shell_with_log;

{ Author:  Steven Pucci,  Mentor APD
  Date:    May 29, 1985

  This program is functionally equivalent to the /com/sh program, with
  the exception that it directs both error and standard output to a
  file specified with the -l argument.  It parses off the -l logname
  arguments from its own argument list, opens the specified file,
  and invokes /com/sh with the remaining arguments.  It passes the
  stream number for the opened file to both error output and standard
  output for the invoked program.  This program should work even if
  /com/sh is changed substantially, since the real /com/sh is invoked
  to do the real work.  The only change to /com/sh which would
  necessitate recompiling this program would be the addition of an
  argument to /com/sh which starts with -l. }
 
  %include '/sys/ins/base.ins.pas';
  %include '/sys/ins/error.ins.pas';
  %include '/sys/ins/pgm.ins.pas';
  %include '/sys/ins/streams.ins.pas';
  %include '/sys/ins/vfmt.ins.pas';

  const program_version = 1.1;

     {History:      1.1     2/12/86    slp    Allowed sh_log with no -l to invoke sh transparently so the "batch -l /dev/null" command will work}

  type pathtype = record
                    name: name_$pname_t;
                    length: integer
                  end;

  var  status:  status_$t;
       stream_id:  stream_$id_t; {stream id of log file}
       ec:  uid_$t; {reserved -- no documentation}
       stream_vector: array[1..4] of stream_$id_t; {to pass to /com/sh}
       wait_mode: pgm_$mode;      {of invoke of /com/sh}
       logfile_name: pathtype;
       logfile_specified: boolean;
       argument_vector_pointer: pgm_$argv_ptr;
       num_args: integer;
       arg: pathtype;  {holds arguments for comparison}

  procedure output_help;

    begin {output_help}
      vfmt_$write2('sh_log (shell_with_log_file) revision %3.1F, 85/3/xx%.',program_version,0);
      writeln('SH_LOG');
      writeln('  Author:   Steve Pucci at Mentor APD');
      writeln;
      writeln('This program is functionally equivalent to the sh program, with one exception:');
      writeln('there is an additional switch option to create a log file.  All other options');
      writeln('to sh are available; type HELP SH for details on the other options.  Please note');
      writeln('that this program is most useful when run as a background process.');
      writeln;
      writeln('FORMAT:');
      writeln;
      writeln('   sh_log  [options to the sh command]  -l logfilename');
      writeln;
      writeln('OPTIONS:');
      writeln;
      writeln('   -l  logfilename        The name of a file to be created which will contain');
      writeln('                          all standard and error output from the shell program.');
      writeln('                          Note that if the -v switch is also specified, the');
      writeln('                          commands executed by the shell program will also be');
      writeln('                          written to the logfile.');
      writeln;
      writeln('   all options to sh      See HELP SH for details.');
      writeln;
      writeln;
      writeln('EXAMPLES:');
      writeln;
      writeln('   1.  sh_log  pas myfile  -l myfilename');
      writeln;
      writeln('         Executes the shell command "pas myfile," and directs the output to the');
      writeln('         file myfilename.');
      writeln;
      writeln('   2.  sh_log   myscript  -v  -l  myfilename');
      writeln;
      writeln('         Executes the shell script myscript, and directs the output to the file');
      writeln('         myfilename.  Note that since -v is specified, the commands in myscript');
      writeln('         are also written to the file.');
    end; {output_help}

  procedure parse_arguments;

  { The idea is to pass everything through to /com/sh, with the
    exception of the -l logfile argument which we strip off, and
    the -h argument for help.
    The strange loop logic is necessary to retrieve the arguments
    properly.  We need to get num_args arguments, but the argument
    pointer changes as we delete arguments from the vector.  See
    the description of the pgm routines for details. }

    var i: integer; {loop counter}
        arg_no: integer; {argument number}

    begin {parse_arguments}
      logfile_specified := false;
      pgm_$get_args( num_args, argument_vector_pointer );
      i := 0;  {i is number of arguments retrieved so far}
      arg_no := 1; {arg_no is pointer to next argument -- no, I can't do this with one loop variable}
      while (i < num_args-1) do begin  {the 0th arg is the program name}
        arg.length := pgm_$get_arg( arg_no, arg.name, status, name_$pnamlen_max );
        if ( (arg.name[1] = '-') and ((arg.name[2] = 'l') or (arg.name[2] = 'L'))) then begin
          if logfile_specified then begin
            vfmt_$ws2(stream_$errout,'?More than one logfile specified%.',0,0);
            pgm_$exit;
          end; {if more than one logfile}
          pgm_$del_arg(arg_no);  {remove the -l from the arg list}
          logfile_name.length := pgm_$get_arg(arg_no, logfile_name.name, status, name_$pnamlen_max );
          if (status.all <> status_$ok) then begin
            vfmt_$ws2(stream_$errout,'?Trouble getting logfile argument%.',0,0);
            error_$print(status);
          end;
          logfile_specified := true;
          pgm_$del_arg(arg_no);  {remove the logfile name from the arg list}
          i := i + 2;  {we already did two more args}
        end else if ( (arg.name[1] = '-') and ((arg.name[2] = 'h') or (arg.name[2] = 'H'))) then begin
          output_help;
          pgm_$exit;
        end else begin  {if not the -l or the -h argument, increment arg ptr}
          arg_no := arg_no + 1;
          i := i + 1;
        end; {if it was the -l argument}
      end;  {while loop on argument number}
      if not logfile_specified then begin
        vfmt_$ws2(stream_$errout,'?No logfile specified -- use /com/sh; continuing anyway...%.',0,0);
{        pgm_$exit;  We want it to continue anyway so that batch works properly when /dev/null is specified}
      end; {if no logfile specified}
    end; {parse_arguments}

  procedure open_logfile;

    begin {open_logfile}
      stream_$create(logfile_name.name, logfile_name.length, stream_$write, stream_$unregulated, 
                     stream_id, status);
      if status.all <> status_$ok then begin
        vfmt_$ws2(stream_$errout,'?Trouble creating file %A%.',
                  logfile_name.name, logfile_name.length);
        error_$print(status);
        pgm_$exit;
      end;
    end; {open_logfile}

  procedure invoke_shell;

    begin {invoke shell}
      stream_vector[1] := stream_$stdin;  {std input for /com/sh}
      stream_vector[3] := stream_$errin;  {err input for /com/sh}
      if logfile_specified then begin
        stream_vector[2] := stream_id;      {std output for /com/sh}
        stream_vector[4] := stream_id;      {err output for /com/sh}
        num_args := num_args - 2;     {leave off file specification}
      end else begin
        stream_vector[2] := stream_$stdout; {std output for /com/sh}
        stream_vector[4] := stream_$errout; {err output for /com/sh}
      end;
      wait_mode := [pgm_$wait];
      pgm_$invoke('/com/sh', 7, num_args, argument_vector_pointer^, 4, stream_vector, wait_mode, ec, status );
      if status.all <> status_$ok then begin
        vfmt_$ws2(stream_$errout,'?Trouble with invoke of /com/sh%.',0,0);
        error_$print(status);
        pgm_$exit;
      end;
    end; {invoke shell}

  begin {shell_with_log}
    parse_arguments;
    if logfile_specified then
      open_logfile;
    invoke_shell;
  end.  {shell_with_log}
