program parse_batch(input, output);

{ This program parses batch input and outputs a command to be passed
  to the command interpreter.  The idea is to change standard Apollo
  input format into the strange format required for creating remote
  processes.  The real advantage is when names are defaulted.

  The following major change was introduced with version 1.0:  Make
  batch take advantage of the sh_log program, which uses a log file
  to keep track of BOTH error and standard input.

    batch -pn name -n node_id -log logfilename shell_command  shell_parameters
                                        
  is converted to

    crp '/com/sh_log -c @'shell_command  shell_params@' -l logfilename' -on node_id -n name -cps -me

  with appropriate defaults 

  IMPLEMENTATION NOTES:

   This program depends on there being a "sh_log" program available.  Currently this needs to be in
 /com/sh_log.  This can be changed by changing the line in the start_output procedure which refers to
 /com/sh_log to be anywhere the user desires.

  NOTES, PROBLEMS, RESTRICTIONS, ETC:

  See the output_help procedure for these.

  Algorithm summary:
   This is mostly a bookkeeping problem.  The arguments to batch are parsed off and written in the
   other syntax for input to sh_log.  Note that any parameter to batch which is NOT a switch is
   assumed to be the shell command (or script) to execute, along with any of its parameters.
   The command itself (shell_command above) is parsed to obtain the leaf name; this leaf name
   is used to derive the default logfile and process names.

  }

  const max_arguments = 30;  {maximum # of arguments on command line}


        program_version = 1.35;
               {update history:   version 1.1  10/14/85  slp  change sh_log to `node_data/sh_log
                                  version 1.2  10/29/85  slp  for SR9 -- use CRP -ME to transfer SID
                                  version 1.3   2/12/86  slp  change help file to reflect version 1.2 change; also allow /dev/null for logfile
                                  version 1.35  3/21/86  slp  for ADUS -- use /com/sh_log instead of `node_data/com/sh_log}


  %include '/sys/ins/base.ins.pas';
  %include '/sys/ins/name.ins.pas';
  %include '/sys/ins/pgm.ins.pas';
  %include '/sys/ins/pm.ins.pas';
  %include '/sys/ins/streams.ins.pas';
  %include '/sys/ins/vfmt.ins.pas';

  type pathtype = record
                    name: name_$pname_t;
                    length: integer
                  end;

  var process_name,
      logfile_name,
      full_log,
      command,
      full_command,
      username,
      wd,
      node_id:  pathtype;
      name_specified,
      node_specified,
      logfile_specified,
      command_specified,
      silent_specified: boolean;
      misc_status: status_$t;
      i: integer;
      devnull: array[1..*] of char := '/dev/null';   {special name for logfile directed to the null device}
      devnull_size: integer := sizeof(devnull);

  procedure init_globals;

    begin {init_globals}
      node_id.length := 1;
      node_id.name[1] := '/';
      name_specified := false;
      logfile_specified := false;
      command_specified := false;
      node_specified := false;
      silent_specified := false;
    end; {init_globals}

  function exists(var f: pathtype): boolean;
    {This function returns true if the specified file exists}

    var 
       file_attributes:  stream_$ir_rec_t;  
       stream_mask: stream_$inquire_mask_t; {which attributes to get}
       stream_inq_type:  stream_$ir_opt; {how to access file (by name)}
       stream_error_mask: stream_$inquire_mask_t; {status returned}
       i: integer;

    begin {exists}
      stream_mask:= []; {Don't really need any information}
      stream_inq_type := stream_$name_unconditional;
      for i := 1 to f.length do begin
        file_attributes.obj_name[i] := f.name[i];
      end;
      file_attributes.obj_namlen := f.length;
      file_attributes.pre_exist := false;
      stream_$inquire( stream_mask, stream_inq_type, file_attributes, 
                       stream_error_mask, misc_status);
      exists := (misc_status.code <> stream_$name_not_found);
    end;  {exists}

  function is_null(var f: pathtype): boolean;
    {This function returns true if the specified file is /dev/null}

    var i: integer;
        null: boolean;  {local variable for value holder}

    begin {is_null}
      if f.length <> devnull_size then
        is_null := false
      else begin
        null := true;
        i := 1;
        repeat
          if f.name[i] <> devnull[i] then null := false;
          i := i + 1;
        until (not null) or (i > devnull_size);
        is_null := null;
      end;
    end; {is_null}

  procedure start_output;

    begin
      write('crp ''/com/sh_log -v -c @''wd '); {Note:  this can be customized to include a startup script}
      for i := 1 to wd.length do write(wd.name[i]);
      write(';');
    end;

  procedure output_help;

    begin {output_help}
      vfmt_$ws2(stream_$errout,'batch revision %4.2F, 85/5/xx by SLP at Mentor APD%.',program_version,0);
      vfmt_$ws2(stream_$errout,'BATCH  --  submit a command as a background (CPS) process%.',i,i);
      vfmt_$ws2(stream_$errout,'usage:  BATCH  pathname  [-N nodeid] [-PN process_name]%.',i,i);
      vfmt_$ws2(stream_$errout,'                         [-L logfile] [-Silent]%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'FORMAT%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  BATCH  pathname %.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  BATCH creates a background process on the given node and has the%.',i,i);
      vfmt_$ws2(stream_$errout,'  process execute the given command.  The process executes with the same%.',i,i);
      vfmt_$ws2(stream_$errout,'  id as the calling process, in the same working directory as the process.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'ARGUMENTS%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  pathname      The shell command to be executed.  This may be any%.',i,i);
      vfmt_$ws2(stream_$errout,'                command recognized by the command interpreter; thus%.',i,i);
      vfmt_$ws2(stream_$errout,'                it may be a shell script or an object file.%.',i,i);
      vfmt_$ws2(stream_$errout,'                If the command has arguments, enclose the command%.',i,i);
      vfmt_$ws2(stream_$errout,'                AND any arguments in quotes.  For example:%.',i,i);
      vfmt_$ws2(stream_$errout,'                   BATCH  "netstat -a"%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'OPTIONS%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  -N nodeid     Execute the process on the given node.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'                Default if omitted:  Execute on current node%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  -PN name      Use given name as process name.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'                Default if omitted:  Use  BATCH.username.commandname%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  -L logfile    Direct output to given file.  Note that if no output%.',i,i);
      vfmt_$ws2(stream_$errout,'                is desired, logfile may be specified as /dev/null.%.',i,i);              
      vfmt_$ws2(stream_$errout,'                Please note also that with version 1.0 of BATCH, both standard%.',i,i);
      vfmt_$ws2(stream_$errout,'                and error output are written to the same log file, and that%.',i,i);
      vfmt_$ws2(stream_$errout,'                this log file also includes the commands executed.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'                Default if omitted:  Use  commandname.log in current%.',i,i);
      vfmt_$ws2(stream_$errout,'                working directory.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'  -Silent       Don''t tell me about the process to be created%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'PROBLEMS, RESTRICTIONS%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'   1. Error reporting is minimal.  The following errors are not%.',i,i);
      vfmt_$ws2(stream_$errout,'      reported;%.',i,i);
      vfmt_$ws2(stream_$errout,'       a.  pname same as another process%.',i,i);
      vfmt_$ws2(stream_$errout,'       b.  log file same as that of another process%.',i,i);
      vfmt_$ws2(stream_$errout,'       c.  command doesn''t exist%.',i,i);
      vfmt_$ws2(stream_$errout,'       d.  logfile or process name invalid%.',i,i);
      vfmt_$ws2(stream_$errout,'       e.  remote node not set up for remote server processes%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'   2. Ideally we''d like to transfer across csr to background%.',i,i);
      vfmt_$ws2(stream_$errout,'       process.  Currently only wd is transferred across directly.  The%.',i,i);
      vfmt_$ws2(stream_$errout,'       user could customize by adding a startup script.  The aim here%.',i,i);
      vfmt_$ws2(stream_$errout,'       would be:  If we can type a command at the shell level, we can%.',i,i);
      vfmt_$ws2(stream_$errout,'       precede the command by "batch" and achieve the same results.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'EXAMPLES%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'    1.  $ BATCH pas myfile%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'             This creates a process to compile the file myfile.pas%.',i,i);
      vfmt_$ws2(stream_$errout,'             in the current directory.  The process name will be%.',i,i);
      vfmt_$ws2(stream_$errout,'             batch.slp.pas, and the log file will be pas.log%.',i,i);
      vfmt_$ws2(stream_$errout,'             in the current working directory.%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'    2.  $ BATCH my_shell_script -n 2846%.',i,i);
      vfmt_$ws2(stream_$errout,'%.',i,i);
      vfmt_$ws2(stream_$errout,'             This process is created on another node.%.',i,i);
    end; {output_help}

  procedure get_arguments;

    var arg_no: integer;
        arg: pathtype;
        misc_status: status_$t;
        i,j: integer; {loop counters}

    procedure get_switch;

      begin {get_switch}
        for i := 1 to arg.length do   {Convert to lower case}
          if (arg.name[i] >= 'A') and (arg.name[i] <= 'Z')  then
            arg.name[i] := chr (ord (arg.name[i]) + 32);
        if (arg.name[2] = 'h') then begin
          output_help;
          pgm_$exit;
        end else if (arg.name[2] = 'p') then begin
          arg_no := arg_no + 1;
          process_name.length := pgm_$get_arg( arg_no, process_name.name, 
                                             misc_status, name_$pnamlen_max);
          name_specified := true;
        end else if ((arg.name[2] = 'o') or (arg.name[2] = 'n')) then begin
          arg_no := arg_no + 1;
          node_id.length := pgm_$get_arg( arg_no, node_id.name, 
                                             misc_status, name_$pnamlen_max);
          node_specified := true;
        end else if (arg.name[2] = 'l') then begin
          arg_no := arg_no + 1;
          logfile_name.length := pgm_$get_arg( arg_no, logfile_name.name, 
                                             misc_status, name_$pnamlen_max);
          logfile_specified := true;
        end else if (arg.name[2] = 's') then begin
          silent_specified := true;
        end else begin
          vfmt_$ws2(stream_$errout,'?Unrecognized switch %A%.',arg.name, arg.length);
          pgm_$exit;
        end; {case if on switch type}
      end; {get_switch}

    begin {get_arguments}
      for arg_no := 1 to max_arguments do begin
        arg.length := pgm_$get_arg( arg_no, arg.name, misc_status, 
                                    name_$pnamlen_max);
        if (arg.length > 0) then begin
          if (arg.name[1] = '-') then {if it's a switch}
            get_switch
          else begin           {if not a switch, assume must be written to command}
            for i := 1 to arg.length do begin
              write(arg.name[i]);
            end;
            write(' ');
            if not command_specified then begin  {if first non-switch, parse off leaf as command}
              j := 0;
              i := 0;
              while (i<arg.length) do begin
                j := j + 1;
                i := i + 1;
                if arg.name[i] = '/' then
                  j := 0
                else if arg.name[i] = ' ' then begin
                  i := arg.length;
                  j := j - 1;
                end else begin
                  command.name[j] := arg.name[i];
                end;
              end; {for i implemented as while i}
              command.length := j;
              command_specified := true;
            end; {if not command_specified}
          end; {if switch then else}
        end; {if non-null arg}
      end {for arg_no}
    end; {get_arguments}

  procedure get_procinfo;

    {Here we first get the full SID text, then parse off the first part
     for username.  Next we get the working directory}

    var i: integer;

    begin {get_procinfo}
      pm_$get_sid_txt(256,username.name,username.length);
      i := 1;
      while ((i <= username.length) and (username.name[i] <> '.')) do begin
        i := i + 1;
      end;
      username.length := i - 1;
      name_$get_wdir(wd.name,wd.length,misc_status);
      for i := 1 to wd.length do begin
        if (wd.name[i] >= 'A') and (wd.name[i] <= 'Z')  then
            wd.name[i] := chr (ord (wd.name[i]) + 32);
      end;
    end; {get_procinfo}

  procedure do_logfile;

    var i_full: integer;  {location in full_log name}

    begin
      if (not silent_specified) then begin
        vfmt_$ws2(stream_$errout,'Working directory of created process will be %A%.',wd.name,wd.length);
      end;
      if logfile_specified then begin
        if logfile_name.name[1] = '/' then begin
          if ((logfile_name.name[2] <> '/') and node_specified) then begin
            vfmt_$ws2(stream_$errout,'?Warning: logfile specification begins with ''/''%.', i,i);
          end;
          for i_full := 1 to logfile_name.length do begin
            full_log.name[i_full] := logfile_name.name[i_full];
          end;
          full_log.length := logfile_name.length;
        end else begin
          for i_full := 1 to wd.length do begin
            full_log.name[i_full] := wd.name[i_full];
          end;
          full_log.name[wd.length+1] := '/';
          for i_full := wd.length+2 to (wd.length + logfile_name.length+1) do begin
            full_log.name[i_full] := logfile_name.name[i_full-wd.length-1];
          end;
          full_log.length := wd.length + logfile_name.length + 1;
        end;
      end else begin
        for i_full := 1 to wd.length do begin
          full_log.name[i_full] := wd.name[i_full];
        end;
        full_log.name[wd.length+1] := '/';
        for i_full := wd.length+2 to (wd.length + command.length + 1) do begin
          full_log.name[i_full] := command.name[i_full - wd.length-1];
        end;
        i_full := wd.length + command.length + 1;
        full_log.name[i_full+1] := '.';
        full_log.name[i_full+2] := 'l';
        full_log.name[i_full+3] := 'o';
        full_log.name[i_full+4] := 'g';
        full_log.length := wd.length + command.length + 5;
      end; {if specified else endif}
      if (not is_null(full_log)) then begin   {If we haven't specified /dev/null as the filename}
        if (exists(full_log)) then begin
          vfmt_$ws2(stream_$errout,'?logfile %A already exists.%.',full_log.name,full_log.length);
          pgm_$exit;
        end;
        write('@'' -l ');
        for i := 1 to full_log.length do write(full_log.name[i]);
        write('''');
        if (not silent_specified) then begin
          vfmt_$ws2(stream_$errout,'Logfile will be %A%.',full_log.name,full_log.length);
        end;
      end else begin  {else it was /dev/null}
        write('@''''');
        if (not silent_specified) then begin
          vfmt_$ws2(stream_$errout,'Logfile will not be created%.',0,0);
        end;
      end; {if not /dev/null else}
    end;


  procedure do_name;

    var i: integer;

    begin
      write(' -n ');
      if name_specified then begin
        for i := 1 to process_name.length do write(process_name.name[i]);
        if (not silent_specified) then begin
          vfmt_$ws2(stream_$errout,'Process name will be %A%.',process_name.name,process_name.length);
        end;
      end else begin
        write('batch.');
        for i := 1 to username.length do write(username.name[i]);
        write('.');
        for i := 1 to command.length do write(command.name[i]);
        if (not silent_specified) then begin
          vfmt_$ws5(stream_$errout,'Process name will be batch.%A.%A%.',username.name,username.length,command.name,command.length,0);
        end;
      end;
    end;

  procedure do_nodeid;

    var i: integer;

    begin
      write (' -on ');
      for i := 1 to node_id.length do write(node_id.name[i]);
      writeln(' -cps -me');
      if (not silent_specified) then begin
        if (node_specified) then begin
          vfmt_$ws2(stream_$errout,'Process will be created on node %A%.',node_id.name,node_id.length);
        end else begin
          vfmt_$ws2(stream_$errout,'Process will be created on this node.%.',0,0);
        end;
      end;
    end;

  begin {parse_batch main}
    init_globals;
    get_procinfo;
    start_output;
    get_arguments;
    if not command_specified then begin
      vfmt_$ws2(stream_$errout,'?No command specified%.', i,i);
      pgm_$exit;
    end;
    do_logfile;
    do_name;
    do_nodeid;
  end. {parse_batch main}
