/*********************************************************************** 
*                                                                      *
*         University of Utah - Jim Schimpf System Support Code         *
*                  Copyright 1986 University of Utah                   *
*                                                                      *
*                      MAIL CHECK PROGRAM                              *
*                                                                      *
*      Module: ckmail.c                                                *
*                                                                      *
*   Version    Date    Person          Description                     *
*----------------------------------------------------------------------*
*      0.1  20-Apr-86  J.Schimpf       Initial Version                 *
*      0.2  22-Apr-86  J.Schimpf       New Options                     *
*      0.3  30-Jul-86  M.Hucka         Added -f option
*                                                                      *
*      DESCRIPTION:                                                    *
*              This program checks for new mail and if received it     *
*      will inform the user of this change in status                   *
*                                                                      *
*      SYNTAX: ckmail [-r [<seconds>]] [-b [S]] [-S]                   *
*                                                                      *
*              -r [<seconds>]  - Run the program every <seconds>       *
*                                continuously                          *
*                                If <seconds> omitted run every minute *
*              -b [S]          - If present program will create a      *
*                                window with messages on mail status   *
*                                If not present uses STDOUT            *
*                                If S present then pop up windows will *
*                                always appear in the same spot        *
*                                (Upper left corner) of screen         *
*              -S              - If present and b is not selected then *
*                                stdout will not output NO MAIL MSGS   *
*              -f <file>       - Uses <file> as the user's mail file   *
*                                instead of the default mail file,     *
*                                /usr/spool/mail/<usrname>.            *
*                                                                      *
*   Procedures:                                                        *
*                                                                      *
*   Written+   Name                    Description                     *
*----------------------------------------------------------------------*
*      y       get_usr_name()          Get the user's name from sid    *
*      y       check_mail()            Check mailbox status            *
*      y       send_window()           Send message to created window  *
*      y       send_out()              Send message to stdout          *
*                                                                      *
*      + (N - no; Y - yes (y - untested); P - partially written)       *
*                                                                      *
***********************************************************************/

/********* Include Files ********/

#include <stdio.h>
#include <ctype.h>
#include "/sys/ins/base.ins.c"
#include "/sys/ins/pad.ins.c"
#include "/sys/ins/streams.ins.c"
#include <sys/types.h>
#include <sys/stat.h>

/********* Program Constants **/

#define  TRUE  0xff
#define  FALSE 0
#define  OK    0
#define  BAD   -1
#define SCREEN_MAX     800     /* Std apollo screen height */

#define  MAIL_STAT     "~/user_data/mail.stat" /* Prev Mail status */
#define  MAIL_FILE     "/usr/spool/mail/"       /* Default mail spool file
                                                  append usr name  */

#define REPEAT         "-r"
#define REPEAT_U       "-R"
#define REPEAT_TM_DEF  60

#define WINDOW_OUT     "-b"
#define WINDOW_OUT_U   "-B"
#define WINDOW_STATIC   'S'

#define STD_OUT_SILENCE "-S"

#define ALT_MAIL_FILE  "-f"

#define BEEP_TIME_H    0x2             /* ~ 1/4 second */
#define BEEP_TIME_L    0

#define BUF_SZ     256

/***********************************************************************
*
*      MAIN(ARGS)
*
*      INPUT:  Command line arguments
*
*      OUTPUT: Check user mail file
*
***********************************************************************/

main(argc,argv)
int argc;
char *argv[];
{            
       int alt_mail_file = FALSE;      /* Use alternate mail file */
       int repeat = FALSE;             /* Repeat parameters */
       int repeat_time = REPEAT_TM_DEF;
       int window_out  = FALSE;        /* Output to stdout */
       int window_static = FALSE;      /* Hold windows in place */

       int out_once = FALSE;           /* Stdout continuous output */

       char *uname,*get_usr_name();    /* File & user id values */
       FILE *mstat,*mail,*fopen();
       char mail_file[BUF_SZ];

       time_t last_time;               /* Previous mail values */
       int  last_size;

       char in [BUF_SZ];               /* Useful stuff */
       int i;
       int status;

       /*  *** Parse Command line (if none repeat values are set
               OK).  Get repeat and time (if any)
                     Get if we are to use windows
       */

       for( i=1; i<argc; i++ )
       {                               
               /*  Check for non-continuous output */

               if( strcmp(argv[i],STD_OUT_SILENCE) == 0 )
                       out_once = TRUE;

               /*  If window output check for static type pop
                   ups
               */

               if( strcmp(argv[i],WINDOW_OUT) == 0 ||
                   strcmp(argv[i],WINDOW_OUT_U) == 0)
               {
                       window_out = TRUE;

               /*  Check for static windows */

                       if( ((i+1) < argc) && 
                           toupper(argv[i+1][0]) == WINDOW_STATIC )
                               window_static = TRUE;
               }
                                                 
               if( strcmp(argv[1],REPEAT) == 0 ||
                   strcmp(argv[i],REPEAT_U) == 0 )
               {        
                       repeat = TRUE;

               /*  Check for the time (if any) */

                       if( ((i+1) < argc) && isdigit(argv[i+1][0]) )
                               sscanf(argv[i+1],"%d",&repeat_time);
               }

               if (strcmp(argv[i],ALT_MAIL_FILE) == 0)
               {
                       alt_mail_file = TRUE;
                       strcpy(mail_file,argv[++i]);
               } 

       }

       /*  0) Build mail file name */

       if (alt_mail_file == FALSE)
       {
                uname = get_usr_name();
                strcpy(mail_file,MAIL_FILE);
                strcat(mail_file,uname);
       }


       /*  1) Open mail stat file (if possible) and get data
              If it doesn't exist then set our times and sizes
              to 0
       */

       mstat = fopen(MAIL_STAT,"r");
       if( mstat != NULL && fgets(in,BUF_SZ,mstat) != NULL)
       {
               sscanf(in,"%ld %d",&last_time,&last_size);
               fclose( mstat );
       }
       else
       {
               last_time = (time_t) 0;
               last_size = 0;
       }
       
       /*  LOOP here if repeat and check for mail arrival
           Mail arrival is true if:
                       1) mail file size != 0 & has changed
                       2) Updated since last time saved
       */

       do{
               status = check_mail(mail_file,
                        &last_time,&last_size,window_out,
                        window_static,out_once);
               if( status )
               {
                       fprintf(stderr,"CKMAIL:Cannot status %s quitting\n",
                               mail_file);
                       exit(-1);
               }
               if( repeat )
                       sleep( repeat_time );
       } while( repeat );
}

       
/***********************************************************************
*
*      GET_USR_NAME()
*
*      INPUT:  NONE - program environment
*
*      OUTPUT: Pointer to user name as a string
*
***********************************************************************/

#define SID_MARK       '.'
                                        
char *get_usr_name()
{
       short len;
       short max_len = BUF_SZ;
       char  sid[BUF_SZ],*usr_name;
       int i,name_len;

       /*  Get the name from the process space & determine the 
           length of the user's name 
           sid = person.group.project.nodeid
           ***** WEIRDNESS - NOTE POINTERS HERE ?????!!!!! *******
       */

       pm_$get_sid_txt(&max_len,sid,&len);
       for( i=0; i<len; i++ )
       {
               if( sid[i] == SID_MARK )
               {
                       sid[i] = '\0';
                       break;
               }
       }

       /*  Name marked so create an area for it and return to usr */

       name_len = strlen( sid );
       usr_name = (char *) malloc( name_len + 2 );
       if( usr_name == NULL )
       {
               fprintf(stderr,"**ckmail: MALLOC error in get_usr_name\n");
               exit(-1);
       }
       strcpy(usr_name,sid);
       return( usr_name );
}

/***********************************************************************
*
*      CHECK_MAIL(mail,&last_time,&size,window_flg,window_static,out_once)
*
*      INPUT:  mail         - Mail file name
*              last_time    - last time we got mail
*              size         - Size of mail file at time time
*              window_flg   - If TRUE use window output else use
*                             std out
               window_static - If true pop all windows in same lcn
               out_once      - If true don't output NO MAIL msg to 
                               stdout
*
*      OUTPUT: NONE
*              If mail received output message to user & update time,size
*              and mail status file
*
***********************************************************************/

check_mail(mail,last_time,size,window_flg,window_static,out_once)
char mail[];
time_t *last_time;
int *size;
int window_flg;
int window_static;
int out_once;
{                              
       static int first_time = TRUE;
       struct stat file_stat;
       FILE *mstat,*fopen();
       char *ctime();
       char out[BUF_SZ],*timbuf,*my_time();

       /*  0) Get the status of the mail file */

       if( stat(mail,&file_stat) == BAD )
               return( BAD );

       /*  Ok good status so start checking stuff
               a) If file size == 0 we can quit now and return 0 
               b) If time == last time then quit now also
               
       */

       if( file_stat.st_size == (off_t) 0 )
       {       
               if( !window_flg && !out_once ) 
               {
                       timbuf = my_time(last_time);
                       printf("** No mail since %s\n",timbuf);
               }
               return( OK );
       }

       if( file_stat.st_mtime == *last_time )
       {
               if( first_time )
               {
                       sprintf(out,"** Unread Mail in mailbox CHECK it **");
                       if( window_flg )
                               send_window(out,window_static);
                       else
                               send_out( out );
                       first_time = FALSE;
               }
               return( OK );
       }

       /*  3) File has been changed, output this for the status file and
           tell the user
       */

       first_time = FALSE;
       *last_time = file_stat.st_mtime;
       *size      = file_stat.st_size;

       mstat = fopen( MAIL_STAT,"w");
       if( mstat == NULL )
       {
               fprintf(stderr,"CKMAIL:Problem writing %s check ACL's\n",
                       MAIL_STAT);
               fprintf(stderr,"CKMAIL:Continuing -- Mail status not updated\n");
       }
       else
       {
               fprintf(mstat,"%ld %d",*last_time,*size);
               fclose( mstat );
       }

       /*  4) Ok tell the user and return OK status to main */
                                           
       timbuf = my_time(last_time);
       sprintf(out,"**You have new mail %s",timbuf);
       if( window_flg )
               send_window(out,window_static);
       else
               send_out( out );
       return(OK);
}

/***********************************************************************
*
*      SEND_WINDOW(out,window_static)
*
*      INPUT:  OUT     - Message to output to a temp window
*              window_static - If true pop window in same place
*
*      OUTPUT: Put up temporary window on screen with message and ring
*              bell, window large enough to hold message
*
***********************************************************************/

send_window(out,window_static)
char out[];
int window_static;
{
       time_$clock_t tone_time;
       int i,lines;
#define display_unit           (short)1        /* Window stuff */
       stream_$id_t            window_stream;
       pad_$window_desc_t      window;
       status_$t               status;
       short   fnsize = 0;                     /* Font inquire stuff */
       short   fwidth,fheight,fnlen;
       char    fname[BUF_SZ];
       short x,y;                              /* Window psn for cursor psn */
       stream_$sk_t seek_key[3];               /* Stream output stuff */
       static short win_psn_top  = 0;
       static short win_psn_left = 0;

       /********************** CODE *******************/            
       /*  Count the number of lines (min 1) and make the window 
           large enough to hold them 
       */

       for( i=0,lines=1; i<strlen(out); i++ )
               if( out[i] == '\n' )
                       lines++;

       /*  Beep at the user */

       tone_time.high = BEEP_TIME_H;          
       tone_time.low = BEEP_TIME_L;
       tone_$time( &tone_time );

       /*  Create a window for the user to display the stuff
           First create a tiny window and then calculate the real
           size
       */

       window.top    = win_psn_top;    /* Tiny window at top left */
       window.left   = win_psn_left;   /* corner of screen */
       window.width  = 5;
       window.height = 5;
                     
       pad_$create_window( (char *)NULL,       /* No path for pad */
                           (short)0,           /* Name length = 0 */
                           pad_$transcript,    /* Read only type pad */
                           display_unit,       /* put on my screen */
                           window,             /* Window size */
                           window_stream,      /* Stream to window */
                           status);

       /*  Get the font size and use this to calculate the real size */

       pad_$inq_font(window_stream,fwidth,fheight,
                     fname,fnsize,fnlen,status);

       /*  Now calculate the from the fontsize and the message length the
           size of a window to fit on the screen
       */

       window.width  = fwidth * ( strlen(out) + 5 );
       window.height = fheight * (2 + lines);

       pad_$set_full_window(window_stream,(short)1,window,status);

       /*  Put cursor on the middle line for the write & output message */
       
       x = (short)0;
       y = (short)2;           /* Window is 3+ fonts high */

       pad_$move(window_stream,pad_$absolute,x,y,status);
       stream_$put_rec(window_stream,out,(long)strlen(out),seek_key,status);
       
       /* Update the position for the next time 

          If window_static FALSE 
        (so we can see it again) */
       
       if( !window_static )
       {
               win_psn_top += window.height;
               if( win_psn_top > SCREEN_MAX )
                       win_psn_top = 0;
       }

       /*  Do a TLW window command to the DM to get back */

       strcpy(fname,"TLW;TI");
       fnsize = strlen(fname);
       pad_$dm_cmd(window_stream,fname,fnsize,status);

       /*  Close the window so we can re-use it */

       stream_$close(window_stream,status);
}                                                                          

/***********************************************************************
*
*      SEND_OUT(out)
*
*      INPUT:  out     - Message to be sent to user
*
*      OUTPUT: Output with bell to stdout
*
***********************************************************************/

send_out(out)
char out[];
{
       time_$clock_t tone_time;

       /*  Beep at the user */

       tone_time.high = BEEP_TIME_H;          
       tone_time.low = BEEP_TIME_L;
       tone_$time( &tone_time );

       /* Output the message and a newline */

       printf("%s\n",out);
}

/***********************************************************************
*
*      MY_TIME(tim)
*
*      INPUT:  tim             - Pointer to time_t value
*
*      OUTPUT: String with time converted to ASCII with no <CR>
*
***********************************************************************/
                                                               
char *my_time(tim)
time_t *tim;
{
       char *tim_val;
       int psn;

       tim_val = ctime( tim );

       /*  Now remove the \n at the end of the string */

       psn = strlen(tim_val);
       tim_val[psn-1] = '\0';
       return( tim_val );
}
