/*
 * Apollo lights, X version.
 * Click on a light to toggle it.
 * Click on heartbeat light or type 'q' to quit.
 *
 * Jim Rees, University of Michigan, June, 1998
 */

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>

#include "Draw.h"
#include "kread.h"
#include "dnl.h"

#define NCELLS 64
#define GSCL 65536
#define GSCLD (GSCL - 1)
#define HASH 131
#define LY 124
#define LW 24
#define LH 22
#define BLINKINT 105 /* 2^18/5000 ~= 104.8576 */

static char appname[] = "Dnlights";

static
String fallback_resources[] = {
    "zpixmap:					True",
    "ppmdir:					.",
    "*foreground:				black",
    "*background:				white",
    NULL
};

typedef struct _AppResources {
    Boolean	debug;
    Boolean	gamma;
    Boolean	zpixmap;
    String	ppmdir;
    String	net;
    String	disk;
} AppResources;

AppResources app_resources;

static
XtResource resources[] = {
    {"debug", "Debug", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, debug), XtRImmediate, False},
    {"gamma", "Gamma", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, gamma), XtRImmediate, False},
    {"zpixmap", "Zpixmap", XtRBoolean, sizeof(Boolean),
     XtOffset(AppResources *, zpixmap), XtRImmediate, False},
    {"ppmdir", "Ppmdir", XtRString, sizeof(String),
     XtOffset(AppResources *, ppmdir), XtRImmediate, ""},
    {"net", "Net", XtRString, sizeof(String),
     XtOffset(AppResources *, net), XtRImmediate, ""},
    {"disk", "Disk", XtRString, sizeof(String),
     XtOffset(AppResources *, disk), XtRImmediate, ""},
};

static
XrmOptionDescRec options[] = {
    {"-d", 	"debug",		XrmoptionNoArg,		"True"},
    {"-g", 	"gamma",		XrmoptionNoArg,		"True"},
    {"-ppmdir",	"ppmdir",		XrmoptionSepArg,	NULL},
    {"-net",	"net",			XrmoptionSepArg,	NULL},
    {"-disk",	"disk",			XrmoptionSepArg,	NULL},
};

static void Close(), Quit();

static XtActionsRec actions[] = {
    { "quit", Close },
};

int dflag;
XtAppContext app;
Display *dpy;
int screen;
Atom wm_delete_window;
Colormap colormap;
Window win;
GC gc0;
int width, height, dpyform, exposed;
FILE *bkgf;
XImage *bkgimg, *onimg, *offimg, *pwrimg;
u_long tick, ifaddr;

struct {
    int on, pos;
    XImage *img;
} light[5] = {
    {0,  27, NULL},
    {0,  52, NULL},
    {0,  76, NULL},
    {0, 101, NULL},
    {1, 125, NULL},
};

static void excb(), incb();
XtTimerCallbackProc beat();
FILE *openpbm();
XImage *readpbm(), *loadimage();

extern char *malloc();
extern int abort();
extern unsigned char gammatab[];

main(ac, av)
int ac;
char *av[];
{
    Widget top, draw;
    XGCValues gcv;
    Visual *v;
    struct ifst *ifp = NULL;

    top = XtVaAppInitialize(&app, appname, options, XtNumber(options), &ac, av, fallback_resources, NULL);
    XtGetApplicationResources(top, (XtPointer) &app_resources, resources, XtNumber(resources), NULL, 0);
    XtAppAddActions(app, actions, XtNumber(actions));
    XtOverrideTranslations(top, XtParseTranslationTable ("<Message>WM_PROTOCOLS: quit()"));
    dpy = XtDisplay(top);
    screen = DefaultScreen(dpy);

    dflag = app_resources.debug;
    if (dflag) {
	XSynchronize(dpy, True);
	XSetErrorHandler(abort);
    }

    bkgf = openpbm("bkg.ppm", &width, &height);
    if (!bkgf) {
	fprintf(stderr, "can't open ppm files\n");
	Quit();
    }

    dpyform = (app_resources.zpixmap) ? ZPixmap : XYPixmap;

    draw = XtVaCreateManagedWidget("draw", drawWidgetClass, top,
	XtNwidth, width, XtNheight, height, NULL);

    XtAddCallback(draw, XtNexpose, excb, NULL);
    XtAddCallback(draw, XtNcallback, incb, NULL);

    XtRealizeWidget(top);

    win = XtWindow(draw);

    XtVaGetValues(draw, XtNcolormap, &colormap, 0);

    gcv.foreground = BlackPixel(dpy, screen);
    gcv.background = WhitePixel(dpy, screen);
    gc0 = XCreateGC(dpy, win, (GCForeground | GCBackground), &gcv);

    wm_delete_window = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(dpy, win, &wm_delete_window, 1);

#ifndef NONET
    if ((kmem = kopen()) < 0)
	return -1;

    /* Get a list of net interfaces and decide which one to use */
    getiflist(&ifp);
    if (ifp) {
	if (app_resources.net && app_resources.net[0]) {
	    while (ifp->name && strcmp(ifp->name, app_resources.net))
		ifp++;
	} else {
	    while (ifp->name && !strncmp(ifp->name, "lo", 2))
		ifp++;
	}
	if (ifp->name)
	    ifaddr = ifp->addr;
    }
#endif

    XtAppMainLoop(app);
}

static void
incb(w, a, ev)
Widget w;
XtPointer a;
XEvent *ev;
{
    static int x0, y0;
    int i;
    char buf[4];
    XComposeStatus compstatus;

    switch(ev->xany.type) {
    case ButtonPress:
	x0 = ev->xmotion.x;
	y0 = ev->xmotion.y;
	break;
    case ButtonRelease:
	switch(ev->xbutton.button) {
	case Button1:
	    /* figure out which light (if any) was clicked */
	    if (y0 >= LY && y0 < LY + LH) {
		for (i = 0; i < 5; i++)
		    if (x0 >= light[i].pos && x0 < light[i].pos + LW)
			break;
		if (i == 3) {
		    /* click on heartbeat light?  quit. */
		    Quit();
		} else {
		    blink(i, -1);
		    if (i == 4)
			/* power light */
			beat(w, 0);
		}
	    }
	}
	break;
    case KeyPress:
	i = XLookupString((XKeyEvent *) ev, buf, sizeof buf, NULL, &compstatus);
	buf[i] = '\0';
	switch (buf[0]) {
	case 'q':
	    Quit();
	    break;
	}
	break;
    }
}

static void
excb(w, a)
Widget w;
XtPointer a;
{
    int i, lw, lh;

    if (!exposed++) {
	/* First time through; load all images */
	if (bkgf != NULL) {
	    bkgimg = readpbm(bkgf, width, height);
	    fclose(bkgf);
	    bkgf = NULL;
	}
	onimg = loadimage("on.ppm", &lw, &lh);
	offimg = loadimage("off.ppm", &lw, &lh);
	pwrimg = loadimage("pwr.ppm", &lw, &lh);
	XtAppAddTimeOut(app, BLINKINT, beat, w);
    }
    if (!bkgimg)
	return;

    /* Put up the background */
    XPutImage(dpy, win, gc0, bkgimg, 0, 0, 0, 0, width, height);

    /* Now, some lights */
    for (i = 0; i < 5; i++) {
	light[i].img = (i == 4) ? pwrimg : onimg;
	blink(i, light[i].on);
    }
}

/* Turn a light on or off (0 or 1), or toggle (-1). */

blink(i, onoff)
int i, onoff;
{
    light[i].on = (onoff < 0) ? !light[i].on : onoff;
    XPutImage(dpy, win, gc0, light[i].on ? light[i].img : offimg, 0, 0, light[i].pos, LY, LW, LH);
}

XtTimerCallbackProc
beat(Widget w, XtIntervalId id)
{
    int i, ntx, nrx, ndk;
    static int ontx, onrx, ondk;

    /* if power is off, turn off all lights and return */
    if (!light[4].on) {
	tick = 0;
	for (i = 0; i < 5; i++)
	    blink(i, 0);
	return;
    }

#ifndef NONET
    if (ifaddr) {
	/* get packet counts, toggle 'A' and 'B' if changed */
	getifstats(ifaddr, &ntx, &nrx);
	if (ntx != ontx)
	    blink(0, -1);
	if (nrx != onrx)
	    blink(1, -1);
	ontx = ntx;
	onrx = nrx;
    }
#endif

#ifndef NODISK
    ndk = getdkstats();
    if (ndk != ondk)
	blink(2, -1);
    ondk = ndk;
#endif

    /* toggle 'D' every fifth tick */
    if (tick++ % 5 == 0)
	blink(3, -1);

    XtAppAddTimeOut(app, BLINKINT, beat, w);
}

FILE *
openpbm(name, wp, hp)
char *name;
int *wp, *hp;
{
    FILE *fp;
    char buf[200];
    char *dirname;

    if (app_resources.ppmdir && app_resources.ppmdir[0])
	dirname = app_resources.ppmdir;
    else
	dirname = ".";
    sprintf(buf, "%s/%s", dirname, name);
    fp = fopen(buf, "r");
    if (!fp)
	return NULL;
    if (fgets(buf, sizeof buf, fp) == NULL || strncmp(buf, "P6", 2)) {
	fclose(fp);
	return NULL;
    }
    buf[0] = '#';
    while (buf[0] == '#') {
	if (fgets(buf, sizeof buf, fp) == NULL) {
	    fclose(fp);
	    return NULL;
	}
    }

    sscanf(buf, "%d %d", wp, hp);

    return fp;
}

XImage *
readpbm(fp, width, height)
FILE *fp;
int width, height;
{
    XImage *ip;
    char *bp;
    int npix, x, y;
    XColor c;

    fscanf(fp, "%d", &npix);
    getc(fp);

    bp = (char *) malloc(width * height * (DisplayPlanes(dpy, screen) + 7) / 8);
    ip = XCreateImage(dpy, DefaultVisual(dpy, screen),
	DisplayPlanes(dpy, screen), dpyform, 0, bp, width, height, 32, 0);

    for (y = 0; y < height; y++) {
	for (x = 0; x < width; x++) {
	    if (app_resources.gamma) {
		c.red = gammatab[getc(fp) * 255 / npix] * GSCLD / 255;
		c.green = gammatab[getc(fp) * 255 / npix] * GSCLD / 255;
		c.blue = gammatab[getc(fp) * 255 / npix] * GSCLD / 255;
	    } else {
		c.red = getc(fp) * GSCLD / npix;
		c.green = getc(fp) * GSCLD / npix;
		c.blue = getc(fp) * GSCLD / npix;
	    }
	    rgbtopix(&c);
	    XPutPixel(ip, x, y, c.pixel);
	}
    }
    return ip;
}

XImage *
loadimage(fname, wp, hp)
char *fname;
int *wp, *hp;
{
    FILE *f;
    XImage *ip;

    f = openpbm(fname, wp, hp);
    if (!f)
	return NULL;
    ip = readpbm(f, *wp, *hp);
    fclose(f);
    return ip;
}

#define hashfunc(r, g, b) (((r) * 3 + (g) * 5 + (b) * 7) % HASH)

rgbtopix(cp)
XColor *cp;
{
    int h, h0, i;
    static short hashtab[HASH], cmx;
    static XColor *cm;
    XColor ca;

    if (!cm) {
	cm = (XColor *) malloc(NCELLS * sizeof (XColor));
	for (h = 0; h < HASH; h++)
	    hashtab[h] = -1;
	for (i = 0; i < NCELLS; i++)
	    cm[i].pixel = -1;
    }

    h = h0 = hashfunc(cp->red, cp->green, cp->blue);

    while ((i = hashtab[h]) >= 0) {
	if (cm[i].red == cp->red && cm[i].green == cp->green && cm[i].blue == cp->blue)
	    break;
	if (++h >= HASH)
	    h = 0;
	if (h == h0)
	    break;
    }
    if (i >= 0) {
	cp->pixel = cm[i].pixel;
	return;
    }

    /* Didn't find it.  Allocate a (possibly new) cell. */
    ca = *cp;
    if (!XAllocColor(dpy, colormap, &ca)) {
	fprintf(stderr, "too many colors\n");
	Quit();
    }

    i = hashtab[h] = cmx++;
    cp->pixel = ca.pixel;
    cm[i] = *cp;
}

static void
Close(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
    if (w && event->type == ClientMessage && event->xclient.data.l[0] != wm_delete_window) {
	XBell(dpy, 0);
	return;
    }
    Quit();
}

static void
Quit()
{
    XtDestroyApplicationContext(app);
    exit(0);
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
