/*	APOLLO -- Terminal Driver for Apollo Display using Domain	     */
/*		  Graphics Primitives (GPR) or Network Graphics		     */
/*		  Terminal (NGT) Protocol.				     */
/*									     */
/*	Copyright (C) 1987 by Leonard N. Zubkoff, All Rights Reserved	     */
/*									     */
/*	This software is provided free and without any warranty.	     */
/*	Permission to copy for any purpose is hereby granted so		     */
/*	long as this copyright notice remains intact.			     */
/*									     */
/*	Revision:	18-Aug-92 11:58:34				     */


#define hidden	    static
#define visible
#define procedure   void


typedef unsigned long	    word;
typedef unsigned short	    halfword;
typedef unsigned char	    byte;


#include <stdio.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#define NIL 0L
#undef NULL
#include "config.h"
#include "lisp.h"
#include "paths.h"
#include "buffer.h"
#include "termchar.h"
#include "termhooks.h"
#include "termopts.h"
#include "dispextern.h"
#include "ngt.h"


#ifdef APOLLO			/* Apollo Display support using GPR or NGT */


#include <apollo/base.h>
#include <apollo/ec2.h>
#include <apollo/fontn.h>
#include <apollo/gpr.h>
#include <apollo/ios.h>
#include <apollo/io_traits.h>
#include <apollo/kbd.h>
#include <apollo/ms.h>
#include <apollo/pad.h>
#include <apollo/tone.h>
#include <apollo/trait.h>

#define Ctrl(Character)	    ((Character) & 0x1F)
#define Meta(Character)	    ((Character) | 0x80)


#define MouseCursorHeight   16
#define MouseCursorWidth    8
#define MouseCursorXOrigin  0
#define MouseCursorYOrigin  0
#define VisibleListSize	    64


extern char
  *NetworkDisplay;


hidden int
  Keyboard,
  ScreenLines,
  ScreenColumns,
  AcquireCount,
  CursorLine,
  CursorColumn,
  WindowLines,
  LineClear[256];


hidden short
  FontHeight,
  FontWidth,
  FontOffset,
  FontVerticalSpacing,
  FontHorizontalSpacing,
  FontNameLength,
  ScreenVisibleCount,
  FixedMappings[256],
  MouseCursorPattern[MouseCursorHeight] =
    { 0x8000, 0xC000, 0xE000, 0xF000,
	0xF800, 0xFC00, 0xFE00, 0xF000,
	0xD800, 0x9800, 0x0C00, 0x0C00,
	0x0600, 0x0600, 0x0300, 0x0300 };


hidden boolean
  ColorDisplay,
  HighlightMode,
  WantHighlightMode,
  ScreenUnobscured,
  UpdateInProgress,
  RefreshDuringUpdate,
  MouseCursorInWindow,
  ApolloDisplayActivated,
  ApolloUnixProtectionEnabled;

hidden ios_$id_t
  PadStreamID;


hidden pad_$string_t
  FontName;


hidden gpr_$bitmap_desc_t
  ScreenBitmap,
  MouseCursorBitmap,
  MouseCursorTempBitmap;


hidden gpr_$position_t
  MouseCursorPosition;


hidden gpr_$pixel_value_t
  Background,
  Characters;


hidden gpr_$raster_op_t
  ColorCursor1[4] =
    { gpr_$rop_zeros, gpr_$rop_src, gpr_$rop_zeros, gpr_$rop_src },
  ColorCursor2[4] =
    { gpr_$rop_zeros, gpr_$rop_zeros, gpr_$rop_src, gpr_$rop_src };


hidden gpr_$mask_32_t
  NormalPlaneMask,
  FullPlaneMask,
  PlaneMask123;


hidden gpr_$window_t
  ScreenVisibleList[VisibleListSize];


hidden ec2_$ptr_t
  GPREventCount;


hidden short
  MetaKey;


hidden gpr_$keyset_t
  CodedKeysSet,
  FunctionKeysSet,
  PhysicalKeysSet,
  MouseButtonsSet;

hidden procedure ClearRectangle(int Line,
				int Column,
				int LineCount,
				int ColumnCount)
{
  gpr_$window_t Rectangle;
  status_$t Status;
  int i;
  Rectangle.window_base.x_coord = Column * FontWidth;
  Rectangle.window_base.y_coord = Line * FontHeight;
  Rectangle.window_size.x_size = ColumnCount * FontWidth;
  Rectangle.window_size.y_size = LineCount * FontHeight;
  for (i=0; i<ScreenVisibleCount; i++)
    {
      if (!ScreenUnobscured)
	gpr_$set_clip_window(ScreenVisibleList[i],&Status);
      gpr_$rectangle(Rectangle,&Status);
    }
  if (ColumnCount == ScreenColumns)
    for (i=Line; i<Line+LineCount; i++)
      LineClear[i] = true;
}


hidden procedure CopyRectangle(int SourceLine,
			       int SourceColumn,
			       int DestinationLine,
			       int DestinationColumn,
			       int LineCount,
			       int ColumnCount)
{
  gpr_$window_t SourceWindow;
  gpr_$position_t DestinationOrigin;
  status_$t Status;
  int i;
  SourceWindow.window_base.x_coord = SourceColumn * FontWidth;
  SourceWindow.window_base.y_coord = SourceLine * FontHeight;
  SourceWindow.window_size.x_size = ColumnCount * FontWidth;
  SourceWindow.window_size.y_size = LineCount * FontHeight;
  DestinationOrigin.x_coord = DestinationColumn * FontWidth;
  DestinationOrigin.y_coord = DestinationLine * FontHeight;
  if (ScreenUnobscured)
    gpr_$pixel_blt(ScreenBitmap,SourceWindow,DestinationOrigin,&Status);
  if (ColumnCount == ScreenColumns)
    if (SourceLine > DestinationLine)
      for (i=0; i<LineCount; i++)
	LineClear[DestinationLine+i] = LineClear[SourceLine+i];
    else for (i=LineCount-1; i>=0; --i)
      LineClear[DestinationLine+i] = LineClear[SourceLine+i];
}

hidden procedure DisplayCharacterCursor(boolean Activate)
{
  gpr_$window_t SourceWindow;
  status_$t Status;
  int i;
  SourceWindow.window_base.x_coord = CursorColumn * FontWidth-1;
  SourceWindow.window_base.y_coord = CursorLine * FontHeight;
  SourceWindow.window_size.x_size = FontWidth - FontHorizontalSpacing + 2;
  SourceWindow.window_size.y_size = FontHeight - FontVerticalSpacing + 2;
  if (!ColorDisplay)
    {
      gpr_$set_raster_op(0,gpr_$rop_not_dst,&Status);
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$pixel_blt(ScreenBitmap,SourceWindow,
			 SourceWindow.window_base,&Status);
	}
      gpr_$set_raster_op(0,gpr_$rop_src,&Status);
    }
  else if (Activate)
    {
      gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
      gpr_$set_raster_op_mask(PlaneMask123,gpr_$rop_zeros,&Status);
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$additive_blt(ScreenBitmap,SourceWindow,3,
			    SourceWindow.window_base,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
  else
    {
      int Option = (Background & 7) >> 1;
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
	  gpr_$bit_blt(ScreenBitmap,SourceWindow,3,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_zeros,&Status);
	  gpr_$set_raster_op(1,ColorCursor1[Option],&Status);
	  gpr_$set_raster_op(2,ColorCursor2[Option],&Status);
	  gpr_$additive_blt(ScreenBitmap,SourceWindow,0,
			    SourceWindow.window_base,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
}

hidden procedure DisplayMouseCursor(boolean Activate)
{
  static gpr_$window_t SourceWindow =
    { 0, 0, MouseCursorWidth, MouseCursorHeight };
  static gpr_$window_t SourceWindow2 =
    { 0, 0, MouseCursorWidth, MouseCursorHeight };
  status_$t Status;
  int i;
  if (!MouseCursorInWindow) return;
  SourceWindow2.window_base = MouseCursorPosition;
  if (!ColorDisplay)
    {
      gpr_$set_raster_op(0,gpr_$rop_src_xor_dst,&Status);
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$pixel_blt(MouseCursorBitmap,SourceWindow,
			 MouseCursorPosition,&Status);
	}
      gpr_$set_raster_op(0,gpr_$rop_src,&Status);
    }
  else if (Activate)
    {
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$set_bitmap(MouseCursorTempBitmap,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src,&Status);
	  gpr_$bit_blt(ScreenBitmap,SourceWindow2,0,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
	  gpr_$bit_blt(ScreenBitmap,SourceWindow2,3,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_and_dst,&Status);
	  gpr_$bit_blt(MouseCursorBitmap,SourceWindow,0,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_bitmap(ScreenBitmap,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
	  gpr_$set_raster_op_mask(PlaneMask123,
				  gpr_$rop_not_src_and_dst,&Status);
	  gpr_$additive_blt(MouseCursorTempBitmap,SourceWindow,0,
			    MouseCursorPosition,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
  else
    {
      int Option = (Background & 7) >> 1;
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);

	  gpr_$bit_blt(ScreenBitmap,SourceWindow2,3,
		       MouseCursorPosition,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_zeros,&Status);
	  gpr_$set_raster_op(1,ColorCursor1[Option],&Status);
	  gpr_$set_raster_op(2,ColorCursor2[Option],&Status);
	  gpr_$additive_blt(ScreenBitmap,SourceWindow2,0,
			    MouseCursorPosition,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
}


hidden procedure SetInsertDeleteAllowed(boolean ScreenUnobscured)
{
  line_ins_del_ok = ScreenUnobscured;
  char_ins_del_ok = ScreenUnobscured;
}


hidden procedure AcquireDisplay(boolean RemoveCharacterCursor)
{
  status_$t Status;
  if (AcquireCount++ > 0) return;
  ScreenUnobscured = gpr_$acquire_display(&Status);
  if (!ScreenUnobscured)
    gpr_$inq_vis_list(VisibleListSize,&ScreenVisibleCount,
		      ScreenVisibleList,&Status);
  else ScreenVisibleCount = 1;
  if (RemoveCharacterCursor) DisplayCharacterCursor(false);
  DisplayMouseCursor(false);
  SetInsertDeleteAllowed(ScreenUnobscured);
}


hidden procedure ReleaseDisplay(boolean RestoreCharacterCursor)
{
  static gpr_$window_t FullClipWindow =
    { 0, 0, gpr_$max_x_size, gpr_$max_y_size };
  status_$t Status;
  if (--AcquireCount > 0) return;
  DisplayMouseCursor(true);
  if (RestoreCharacterCursor) DisplayCharacterCursor(true);
  if (!ScreenUnobscured) gpr_$set_clip_window(FullClipWindow,&Status);
  gpr_$release_display(&Status);
}

visible procedure LockDisplayDataStructures()
{
  if (!ApolloDisplayActivated) return;
  UpdateInProgress = true;
  RefreshDuringUpdate = false;
}


visible procedure UnlockDisplayDataStructures()
{
  if (!ApolloDisplayActivated) return;
  UpdateInProgress = false;
  if (!RefreshDuringUpdate) return;
  screen_garbaged = 1;
  if (ScreenLines != screen_height || ScreenColumns != screen_width)
    change_screen_size(ScreenLines,ScreenColumns,0);
  else redisplay_preserve_echo_area();
}


hidden procedure RefreshProcedure(boolean &Unobscured,
				  boolean &PositionChanged)
{
  pad_$window_desc_t CurrentWindow;
  gpr_$position_t WindowOrigin;
  status_$t Status;
  short WindowCount, AcquireCount;
  if (PositionChanged)
    {
      gpr_$force_release(&AcquireCount,&Status);
      pad_$inq_windows(ios_$stdout,&CurrentWindow,1,&WindowCount,&Status);
      while (--AcquireCount >= 0)
	Unobscured = gpr_$acquire_display(&Status);
      ScreenLines = (CurrentWindow.height-2)/FontHeight;
      ScreenColumns = (CurrentWindow.width-2)/FontWidth;
      WindowOrigin.x_coord = (CurrentWindow.width-ScreenColumns*FontWidth)/2;
      WindowOrigin.y_coord = (CurrentWindow.height-ScreenLines*FontHeight)/2;
      gpr_$set_coordinate_origin(WindowOrigin,&Status);
    }
  RefreshDuringUpdate = UpdateInProgress;
  if (UpdateInProgress) return;
  ScreenUnobscured = Unobscured;
  if (!ScreenUnobscured)
    gpr_$inq_vis_list(VisibleListSize,&ScreenVisibleCount,
		      ScreenVisibleList,&Status);
  else ScreenVisibleCount = 1;
  screen_garbaged = 1;
  if (ScreenLines != screen_height || ScreenColumns != screen_width)
    change_screen_size(ScreenLines,ScreenColumns,0);
  else redisplay_preserve_echo_area();
}

hidden InitializeTerminal()
{
  AcquireDisplay(true);
  CursorLine = 0;
  CursorColumn = 0;
  WindowLines = ScreenLines;
  HighlightMode = true;
  WantHighlightMode = false;
  UpdateInProgress = false;
  BackgroundHighlight();
  ReleaseDisplay(true);
}


hidden ResetTerminal()
{
  status_$t Status;
  pad_$cooked(PadStreamID,&Status);
}


hidden RingBell()
{
  static time_$clock_t BellTime = { 0, 32768 };
  tone_$time(BellTime);
}


hidden DefineWindow(int NewWindowLines)
{
  if (NewWindowLines > 0 && NewWindowLines < ScreenLines)
    WindowLines = NewWindowLines;
  else WindowLines = ScreenLines;
}


hidden WriteCursorPosition(int Line,
			   int Column)
{
  gpr_$coordinate_t CursorX, CursorY;
  status_$t Status;
  CursorLine = Line;
  CursorColumn = Column;
  CursorX = CursorColumn * FontWidth;
  CursorY = FontOffset + CursorLine * FontHeight;
  gpr_$move(CursorX,CursorY,&Status);
}

hidden WriteCharacters(char *CharacterText,
		       int CharacterCount)
{
  gpr_$coordinate_t CursorX, CursorY;
  status_$t Status;
  int i;
  AcquireDisplay(true);
  CheckHighlightMode();
  if (!LineClear[CursorLine])
    ClearRectangle(CursorLine,CursorColumn,1,CharacterCount);
  for (i=0; i<ScreenVisibleCount; i++)
    {
      if (!ScreenUnobscured)
	{
	  gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  CursorX = CursorColumn * FontWidth;
	  CursorY = FontOffset + CursorLine * FontHeight;
	  gpr_$move(CursorX,CursorY,&Status);
	}
      gpr_$text(CharacterText,(short)CharacterCount,&Status);
    }
  CursorColumn += CharacterCount;
  LineClear[CursorLine] = false;
  ReleaseDisplay(true);
}


hidden InsertCharacters(char *CharacterText,
			int CharacterCount)
{
  AcquireDisplay(true);
  CheckHighlightMode();
  CopyRectangle(CursorLine,CursorColumn,
		CursorLine,CursorColumn+CharacterCount,
		1,ScreenColumns-(CursorColumn+CharacterCount));
  if (CharacterText == NIL)
    ClearRectangle(CursorLine,CursorColumn,1,CharacterCount);
  else WriteCharacters(CharacterText,CharacterCount);
  ReleaseDisplay(true);
}


hidden DeleteCharacters(int DeleteCount)
{
  AcquireDisplay(true);
  CheckHighlightMode();
  CopyRectangle(CursorLine,CursorColumn+DeleteCount,CursorLine,CursorColumn,
		1,ScreenColumns-(CursorColumn+DeleteCount));
  ClearRectangle(CursorLine,ScreenColumns-DeleteCount,1,DeleteCount);
  ReleaseDisplay(true);
}

hidden InsertOrDeleteLines(int Line,
			   int InsertOrDeleteCount)
{
  AcquireDisplay(true);
  BackgroundHighlight();
  if (InsertOrDeleteCount < 0)
    {
      CopyRectangle(Line-InsertOrDeleteCount,0,Line,0,
		    WindowLines-(Line-InsertOrDeleteCount),ScreenColumns);
      ClearRectangle(WindowLines+InsertOrDeleteCount,0,
		     -InsertOrDeleteCount,ScreenColumns);
    }
  else
    {
      CopyRectangle(Line,0,Line+InsertOrDeleteCount,0,
		    WindowLines-(Line+InsertOrDeleteCount),ScreenColumns);
      ClearRectangle(Line,0,InsertOrDeleteCount,ScreenColumns);
    }
  ReleaseDisplay(true);
}


hidden ClearScreen()
{
  status_$t Status;
  int i;
  AcquireDisplay(true);
  BackgroundHighlight();
  if (ColorDisplay) gpr_$set_plane_mask_32(FullPlaneMask,&Status);
  for (i=0; i<ScreenVisibleCount; i++)
    {
      if (!ScreenUnobscured)
	gpr_$set_clip_window(ScreenVisibleList[i],&Status);
      gpr_$clear(Background,&Status);
    }
  if (ColorDisplay) gpr_$set_plane_mask_32(NormalPlaneMask,&Status);
  WriteCursorPosition(0,0);
  ReleaseDisplay(true);
}


hidden ClearToEndOfLine(int FirstUnusedColumn)
{
  AcquireDisplay(true);
  CheckHighlightMode();
  ClearRectangle(CursorLine,CursorColumn,1,FirstUnusedColumn-CursorColumn);
  ReleaseDisplay(true);
}

hidden ClearToEndOfWindow()
{
  AcquireDisplay(true);
  BackgroundHighlight();
  if (CursorColumn > 0)
    {
      ClearToEndOfLine(ScreenColumns);
      if (CursorLine < WindowLines-1)
	ClearRectangle(CursorLine+1,0,WindowLines-CursorLine-1,ScreenColumns);
    }
  else ClearRectangle(CursorLine,0,WindowLines-CursorLine,ScreenColumns);
  ReleaseDisplay(true);
}


hidden BackgroundHighlight()
{
  status_$t Status;
  if (!HighlightMode) return;
  HighlightMode = false;
  gpr_$set_text_background_value(Background,&Status);
  gpr_$set_text_value(Characters,&Status);
  gpr_$set_fill_value(Background,&Status);
}


hidden CheckHighlightMode()
{
  status_$t Status;
  if (HighlightMode == WantHighlightMode) return;
  HighlightMode = WantHighlightMode;
  gpr_$set_text_background_value((HighlightMode ? Characters : Background),
				 &Status);
  gpr_$set_text_value((HighlightMode ? Background : Characters),&Status);
  gpr_$set_fill_value((HighlightMode ? Characters : Background),&Status);
}

hidden ReassertLineHighlight(int OldHighlightMode,
			     int Line)
{
  WantHighlightMode = OldHighlightMode;
}


hidden ChangeLineHighlight(int NewHighlightMode,
			   int Line,
			   int FirstUnusedColumn)
{
  AcquireDisplay(true);
  WantHighlightMode = NewHighlightMode;
  WriteCursorPosition(Line,0);
  ClearToEndOfLine(ScreenColumns);
  ReleaseDisplay(true);
}


hidden BeginScreenUpdate()
{
  AcquireDisplay(true);
}


hidden EndScreenUpdate()
{
  BackgroundHighlight();
  WantHighlightMode = false;
  ReleaseDisplay(true);
}


hidden procedure SetApolloMetaKey(int MetaKeyCode)
{
  status_$t Status;
  MetaKey = MetaKeyCode;
  lib_$init_set(PhysicalKeysSet,256);
  if (MetaKey == KBD3_$AL || MetaKey == KBD3_$AR)
    {
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AL);
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AR);
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AL+KBD3_$KEY_UP);
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AR+KBD3_$KEY_UP);
    }
  else
    {
      lib_$add_to_set(PhysicalKeysSet,256,MetaKey);
      lib_$add_to_set(PhysicalKeysSet,256,MetaKey+KBD3_$KEY_UP);
    }
  gpr_$enable_input(gpr_$physical_keys,PhysicalKeysSet,&Status);
}

hidden long gpr_io_$get(void *&Handle,
			ios_$put_get_opts_t &PutGetOptions,
			void *Buffer,
			long &BufferLength,
			status_$t *Status)
{
  static char InputBuffer[10];
  static long InputBufferLength;
  static boolean MetaDepressed = false;
  static boolean HaveInputBuffer = false;
  boolean NoWait = ((PutGetOptions & ios_$cond_opt) != 0);
  boolean Preview = ((PutGetOptions & ios_$preview_opt ) != 0);
  gpr_$event_t EventType;
  unsigned char EventData[1];
  gpr_$position_t EventPosition;
  short AcquireCount;
  int Length;
  if (HaveInputBuffer)
    {
      Status->all = status_$ok;
      Length = BufferLength;
      if (Length > InputBufferLength)
	Length = InputBufferLength;
      bcopy(InputBuffer,Buffer,Length);
      if (!Preview && Length < InputBufferLength)
	{
	  InputBufferLength -= Length;
	  bcopy(InputBuffer,&InputBuffer[Length],InputBufferLength);
	}
      else HaveInputBuffer = Preview;
      return Length;
    }
  InputBufferLength = 0;
  while (InputBufferLength == 0)
    {
      if (NoWait)
	{
	  gpr_$cond_event_wait(&EventType,&EventData[0],&EventPosition,Status);
	  if (EventType == gpr_$no_event)
	    Status->all = ios_$get_conditional_failed;
	}
      else gpr_$event_wait(&EventType,&EventData[0],&EventPosition,Status);
      if (Status->all != status_$ok) return 0;

      if (EventType == gpr_$coded_keys)
	{
	  int Keystroke = EventData[0];
	  if (MetaDepressed)
	    InputBuffer[0] = Meta(Keystroke);
	  else InputBuffer[0] = Keystroke;
	  InputBufferLength = 1;
	}
      else if (EventType == gpr_$function_keys)
	{
	  int Keystroke = EventData[0];
	  if (FixedMappings[Keystroke] > 0)
	    {
	      InputBuffer[0] =
		(MetaDepressed ? Meta(FixedMappings[Keystroke])
			       : FixedMappings[Keystroke]);
	      InputBufferLength = 1;
	    }
	  else
	    {
	      InputBuffer[0] = Ctrl('^');
	      InputBuffer[1] = (MetaDepressed ? 0x4C : 0x48)
			       | (Keystroke >> 6);
	      InputBuffer[2] = 0x40 | (Keystroke & 0x3F);
	      InputBufferLength = 3;
	    }
	}
      else if (EventType == gpr_$physical_keys)
	MetaDepressed = (EventData[0] < KBD3_$KEY_UP);
      else if (EventType == gpr_$locator_update)
	{
	  AcquireDisplay(false);
	  MouseCursorPosition = EventPosition;
	  ReleaseDisplay(ColorDisplay);
	}
      else if (EventType == gpr_$buttons)
	{
	  AcquireDisplay(false);
	  MouseCursorPosition = EventPosition;
	  ReleaseDisplay(ColorDisplay);
	  InputBuffer[0] = Ctrl('^');
	  InputBuffer[1] = EventData[0]+(MetaDepressed ? 16 : 0);
	  InputBuffer[2] = 8+(MouseCursorPosition.x_coord
			      +MouseCursorXOrigin)/FontWidth;
	  InputBuffer[3] = 8+(MouseCursorPosition.y_coord
			      +MouseCursorYOrigin)/FontHeight;
	  InputBufferLength = 4;
	}

      else if (EventType == gpr_$entered_window)
	{
	  AcquireDisplay(false);
	  MouseCursorInWindow = true;
	  MouseCursorPosition = EventPosition;
	  ReleaseDisplay(ColorDisplay);
	}
      else if (EventType == gpr_$left_window)
	{
	  AcquireDisplay(false);
	  MouseCursorInWindow = false;
	  ReleaseDisplay(ColorDisplay);
	}
    }
  HaveInputBuffer = Preview;
  bcopy(InputBuffer,Buffer,InputBufferLength);
  if (InputBufferLength > 0)
    {
      gpr_$force_release(&AcquireCount,Status);
      pad_$pop_push_window(ios_$stdout,1,true,Status);
      while (--AcquireCount >= 0)
	ScreenUnobscured = gpr_$acquire_display(Status);
    }
  return InputBufferLength;
}


hidden ios_$conn_flag_set gpr_io_$inq_conn_flags(void *&Handle,
						 status_$t *Status)
{
  Status->all = status_$ok;
  return (ios_$cf_tty_mask | ios_$cf_vt_mask);
}


hidden procedure gpr_io_$get_ec(void *&Handle,
				ios_$ec_key_t &ECkey,
				ec2_$ptr_t *EventCountP,
				status_$t *Status)
{
  *EventCountP = GPREventCount;
  Status->all = status_$ok;
}


hidden boolean gpr_io_$close(void *&Handle,
			     status_$t *Status)
{
  Status->all = status_$ok;
  return false;
}

hidden procedure gpr_io_$export(void *&Handle,
				void *ManagerData,
				short &BufferSize,
				short *ManagerDataLength,
				status_$t *Status)
{
  *ManagerDataLength = 0;
  Status->all = status_$ok;
}


hidden procedure gpr_io_$import(void *ManagerData,
				void **HandleP,
				status_$t *Status)
{
  *HandleP = NIL;
  Status->all = status_$ok;
}


hidden io_$epv
  gpr_io_$epv =
    { gpr_io_$export,
      gpr_io_$import,
      NIL,
      NIL,
      gpr_io_$close,
      gpr_io_$get_ec,
      NIL,
      NIL,
      NIL,
      gpr_io_$inq_conn_flags,
      NIL,
      gpr_io_$get,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL,
      NIL };

hidden procedure InitializeGraphics()
{
  static uid_$t gpr_io_$uid = { 0x2F3E1E7A, 0x10003166 };
  io_$epv_ptr gpr_io_$epv_ptr = &gpr_io_$epv;
  pad_$string_t KeyboardType;
  fontn_$table_tver1 *FontTable;
  pad_$window_desc_t CurrentWindow;
  pad_$position_t IconPosition;
  gpr_$offset_t ScreenBitmapSize, MouseCursorBitmapSize;
  gpr_$rgb_plane_t HiPlane;
  gpr_$position_t WindowOrigin;
  gpr_$attribute_desc_t CursorAttributeBlock;
  gpr_$keyset_t KeySet;
  ios_$id_t StreamID;
  status_$t Status;
  unsigned long LengthMapped;
  short KeyboardLength, WindowCount, FontID, Key;
  short *MouseCursorBitmapPointer, CursorLineWidth, i;
  boolean IconMoved;
  AcquireCount = 0;
  MouseCursorInWindow = true;
  pad_$inq_kbd(ios_$stdin,sizeof(KeyboardType),KeyboardType,
	       &KeyboardLength,&Status);
  Keyboard = KeyboardType[0]-'0';
  if (Status.all == pad_$stream_not_pad ||
      KeyboardLength == 0 || Keyboard == 1)
    {
      fprintf(stderr,"GPR Emacs cannot be run on this Terminal\n");
      exit(1);
    }
  while (true)
    {
      pad_$icon_wait(ios_$stdout,1,&IconMoved,&IconPosition,&Status);
      if (!IconMoved) break;
    }
  pad_$inq_font(ios_$stdout,&FontWidth,&FontHeight,
		FontName,sizeof(FontName),&FontNameLength,&Status);
  if (Status.all != status_$ok)
    {
      fprintf(stderr,"Pad_$Inq_Font failed, using f7x13.b\n");
      strcpy(FontName,"/sys/dm/fonts/f7x13.b");
      FontNameLength = strlen(FontName);
    }
  pad_$raw(ios_$stdin,&Status);
  PadStreamID = dup(ios_$stdin);
  fcntl(PadStreamID,F_SETFD,1);
  FontTable =
    (fontn_$table_tver1 *) ms_$mapl(FontName,FontNameLength,
				    0,sizeof(fontn_$table_tver1),
				    ms_$nr_xor_1w,ms_$r,
				    false,&LengthMapped,&Status);
  FontHeight = FontTable->max_height + FontTable->v_spacing;
  FontWidth = FontTable->space_size + FontTable->inter_space_dflt;
  FontOffset = FontTable->baseline_offset + FontTable->v_spacing/2-1
	       - (FontTable->font_height+1-FontTable->max_height)/2;
  FontVerticalSpacing = FontTable->v_spacing;
  FontHorizontalSpacing = FontTable->inter_space_dflt;
  ms_$unmap(FontTable,LengthMapped,&Status);

  pad_$set_scale(ios_$stdout,1,1,&Status);
  pad_$inq_windows(ios_$stdout,&CurrentWindow,1,&WindowCount,&Status);
  ScreenLines = (CurrentWindow.height-2)/FontHeight;
  ScreenColumns = (CurrentWindow.width-2)/FontWidth;
  ScreenBitmapSize.x_size = gpr_$max_x_size;
  ScreenBitmapSize.y_size = gpr_$max_y_size;
  gpr_$init(gpr_$direct,ios_$stdout,ScreenBitmapSize,
	    gpr_$highest_plane,&ScreenBitmap,&Status);
  gpr_$inq_bitmap_dimensions(ScreenBitmap,&ScreenBitmapSize,&HiPlane,&Status);
  ColorDisplay = (HiPlane > 0);
  Background = (ColorDisplay ? gpr_$inq_background(&Status) : 0);
  Characters = (ColorDisplay ? 0 : 1);
  if (ColorDisplay)
    {
      NormalPlaneMask = (1<<4)-1;
      FullPlaneMask = (1<<(HiPlane+1))-1;
      PlaneMask123 = (1<<1)|(1<<2)|(1<<3);
      gpr_$set_plane_mask_32(NormalPlaneMask,&Status);
    }
  WindowOrigin.x_coord = (CurrentWindow.width-ScreenColumns*FontWidth)/2;
  WindowOrigin.y_coord = (CurrentWindow.height-ScreenLines*FontHeight)/2;
  gpr_$set_coordinate_origin(WindowOrigin,&Status);
  gpr_$set_obscured_opt(gpr_$input_ok_if_obs,&Status);
  gpr_$set_refresh_entry(RefreshProcedure,(gpr_$rhdm_pr_t)NIL,&Status);
  gpr_$load_font_file(FontName,FontNameLength,&FontID,&Status);
  gpr_$set_text_font(FontID,&Status);
  MouseCursorBitmapSize.x_size = MouseCursorWidth;
  MouseCursorBitmapSize.y_size = MouseCursorHeight;
  gpr_$allocate_attribute_block(&CursorAttributeBlock,&Status);
  gpr_$allocate_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
		       &MouseCursorBitmap,&Status);
  gpr_$acquire_display(&Status);
  gpr_$allocate_hdm_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
			   &MouseCursorTempBitmap,&Status);
  if (Status.all != status_$ok)
    gpr_$allocate_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
			 &MouseCursorTempBitmap,&Status);
  gpr_$release_display(&Status);
  gpr_$inq_bitmap_pointer(MouseCursorBitmap,(char **)&MouseCursorBitmapPointer,
			  &CursorLineWidth,&Status);
  for (i=0; i<MouseCursorHeight; i++)
    {
      *MouseCursorBitmapPointer = MouseCursorPattern[i];
      MouseCursorBitmapPointer += CursorLineWidth;
    }

  bzero(FixedMappings,sizeof(FixedMappings));
  FixedMappings[KBD3_$ESC] = Ctrl('[');
  FixedMappings[KBD3_$TAB] = Ctrl('I');
  FixedMappings[KBD3_$RET] = Ctrl('M');
  FixedMappings[KBD3_$BS] = 0x7F;
  FixedMappings[KBD3_$DEL] = 0x7F;
  for (i=0; i<=9; i++)
    FixedMappings[KBD3_$NP0+i] = '0'+i;
  FixedMappings[KBD3_$NPE] = Ctrl('M');
  FixedMappings[KBD3_$NPF] = '-';
  FixedMappings[KBD3_$NPG] = '+';
  FixedMappings[KBD3_$NPP] = '.';
  lib_$init_set(CodedKeysSet,256);
  lib_$init_set(FunctionKeysSet,256);
  lib_$init_set(PhysicalKeysSet,256);
  lib_$init_set(MouseButtonsSet,256);
  for (Key=0; Key<0x80; Key++)
    lib_$add_to_set(CodedKeysSet,256,Key);
  for (Key=0; Key<256; Key++)
    if (FixedMappings[Key] > 0) lib_$add_to_set(FunctionKeysSet,256,Key);
  SetApolloMetaKey(KBD3_$L9);
  gpr_$enable_input(gpr_$coded_keys,CodedKeysSet,&Status);
  gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
  gpr_$enable_input(gpr_$locator_update,MouseButtonsSet,&Status);
  gpr_$enable_input(gpr_$entered_window,MouseButtonsSet,&Status);
  gpr_$enable_input(gpr_$left_window,MouseButtonsSet,&Status);
  gpr_$get_ec(gpr_$input_ec,&GPREventCount,&Status);
  gpr_$set_quit_event(gpr_$coded_keys,Ctrl('G'),&Status);
  trait_$mgr_dcl(gpr_io_$uid,&io_$trait,trait_$member_kind_local,
		 gpr_io_$epv_ptr,&Status);
  StreamID =
    ios_$connect("",0,gpr_io_$uid,(void *)NIL,gpr_io_$epv_ptr,&Status);
  dup2(StreamID,0);
  close(StreamID);
}

hidden byte
  *InputPointer,
  *OutputPointer,
  InputBuffer[1024],
  OutputBuffer[16384];


#define OutputByte(Character)	*OutputPointer++ = (Character)


#include "toktbl.h"
#include "txtcmp.h"


hidden FlushOutputBuffer()
{
  status_$t Status;
  long BytesRemaining = OutputPointer-OutputBuffer, ByteCount;
  OutputPointer = OutputBuffer;
  while (BytesRemaining > 0)
    {
      ByteCount = ios_$putp(ios_$stdout,ios_$partial_ok_opt,
			    (char *)OutputPointer,BytesRemaining,&Status);
      if (Status.all != status_$ok) pfm_$error_trap(Status);
      BytesRemaining -= ByteCount;
      OutputPointer += ByteCount;
    }
  OutputPointer = OutputBuffer;
}

hidden N_InitializeTerminal()
{
  Z_InitializeTerminal();
  CursorLine = 0;
  CursorColumn = 0;
  WindowLines = ScreenLines;
  HighlightMode = false;
  WantHighlightMode = false;
  InitTextCompressor();
}


hidden N_ResetTerminal()
{
  Z_ResetTerminal();
  FlushOutputBuffer();
}


hidden N_RingBell()
{
  Z_RingBell();
}


hidden N_DefineWindow(int NewWindowLines)
{
  if (NewWindowLines == WindowLines) return;
  if (NewWindowLines < 0 || NewWindowLines > ScreenLines)
    NewWindowLines = ScreenLines;
  Z_DefineWindow(NewWindowLines);
  WindowLines = NewWindowLines;
}


hidden N_WriteCursorPosition(int Line,
			     int Column)
{
  if (Line == CursorLine && Column == CursorColumn) return;
  if (Line == 0 && Column == 0) Z_HomeCursor()
  else if (Line == CursorLine+1)
    if (Column == 0) Z_CursorDownAndReturn()
    else Z_CursorDownAndWriteColumn(Column)
  else Z_WriteCursorPosition(Line,Column);
  CursorLine = Line;
  CursorColumn = Column;
}


hidden N_WriteCharacters(char *CharacterText,
			 int CharacterCount)
{
  N_CheckHighlightMode();
  OutputPointer += CompressText(CharacterText,CharacterCount,OutputPointer);
  CursorColumn += CharacterCount;
}

hidden N_InsertCharacters(char *CharacterText,
			  int CharacterCount)
{
  if (CharacterText == NIL)
    {
      Z_InsertSpaces(CharacterCount);
      CursorColumn += CharacterCount;
    }
  else
    {
      Z_OpenSpaces(CharacterCount);
      N_WriteCharacters(CharacterText,CharacterCount);
    }
}


hidden N_DeleteCharacters(int DeleteCount)
{
  Z_DeleteCharacters(DeleteCount);
}


hidden N_InsertOrDeleteLines(int Line,
			     int InsertOrDeleteCount)
{
  N_BackgroundHighlight();
  if (InsertOrDeleteCount < 0)
    Z_DeleteLines(Line,-InsertOrDeleteCount)
  else Z_InsertLines(Line,InsertOrDeleteCount);
}


hidden N_ClearScreen()
{
  N_BackgroundHighlight();
  Z_ClearScreen();
  CursorLine = 0;
  CursorColumn = 0;
}


hidden N_ClearToEndOfLine(int FirstUnusedColumn)
{
  N_CheckHighlightMode();
  Z_ClearToEndOfLine();
}


hidden N_ClearToEndOfWindow()
{
  N_BackgroundHighlight();
  Z_ClearToEndOfWindow();
}

hidden N_BackgroundHighlight()
{
  if (!HighlightMode) return;
  Z_HighlightModeOff();
  HighlightMode = false;
}


hidden N_CheckHighlightMode()
{
  if (HighlightMode == WantHighlightMode) return;
  if (WantHighlightMode)
    Z_HighlightModeOn()
  else Z_HighlightModeOff();
  HighlightMode = WantHighlightMode;
}


hidden N_ReassertLineHighlight(int OldHighlightMode,
			       int Line)
{
  WantHighlightMode = OldHighlightMode;
}


hidden N_ChangeLineHighlight(int NewHighlightMode,
			     int Line,
			     int FirstUnusedColumn)
{
  WantHighlightMode = NewHighlightMode;
  N_WriteCursorPosition(Line,0);
  N_ClearToEndOfLine(ScreenColumns);
}


hidden N_BeginScreenUpdate()
{
}


hidden N_EndScreenUpdate()
{
  N_BackgroundHighlight();
  WantHighlightMode = false;
  FlushOutputBuffer();
}


hidden procedure FatalError(char *Message,
			    char *Argument)
{
  extern int errno;
  extern char *sys_errlist[];
  if (Argument == NIL)
    fprintf(stderr,Message,sys_errlist[errno]);
  else fprintf(stderr,Message,Argument);
  exit(1);
}

hidden int N_ReadSocket(int Descriptor,
			byte *Buffer,
			int ByteCount)
{
  extern int input_pending;
  byte *SourcePointer, *TargetPointer, *BufferLimit;
  int BytesRead, Count;
  boolean RefreshPending = false;
  TargetPointer = Buffer;
  if ((BytesRead = InputPointer-InputBuffer) > 0)
    {
      SourcePointer = InputBuffer;
      while (SourcePointer < InputPointer)
	*TargetPointer++ = *SourcePointer++;
      InputPointer = InputBuffer;
    }
  if (ByteCount > 1)
    BytesRead += read(Descriptor,&Buffer[BytesRead],ByteCount);
  if (BytesRead <= 0) return BytesRead;
  SourcePointer = Buffer;
  BufferLimit = &Buffer[BytesRead];
  while (SourcePointer < BufferLimit)
    switch (*SourcePointer++)
      {
	case N_Keystroke:
	  *TargetPointer++ = *SourcePointer++;
	  break;
	case N_FunctionKey:
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  break;
	case N_Buttons:
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  break;
	case N_Refresh:
	  ScreenLines = *SourcePointer++;
	  ScreenColumns = *SourcePointer++;
	  SetInsertDeleteAllowed(*SourcePointer++);
	  RefreshPending = true;
	  break;
	case N_SetObscured:
	  SetInsertDeleteAllowed(*SourcePointer++);
	  break;
      }
  if (!RefreshPending) return TargetPointer-Buffer;
  input_pending = TargetPointer > Buffer;
  screen_garbaged = 1;
  if (ScreenLines != screen_height || ScreenColumns != screen_width)
    change_screen_size(ScreenLines,ScreenColumns,0);
  else redisplay_preserve_echo_area();
  return TargetPointer-Buffer;
}

hidden procedure InitializeNetwork()
{
  char HostName[256], *HostAndPort, *Pointer;
  struct hostent *HostEntry;
  struct sockaddr_in InternetSocketAddress;
  struct sockaddr *SocketAddress;
  int Stream, NetworkStream, SocketAddressLength, One = 1;
  word HostAddress;
  short Port = 6900;
  boolean DetachProcess = true;
  HostAndPort = NetworkDisplay;
  if (*HostAndPort == '=')
    {
      HostAndPort++;
      DetachProcess = false;
    }
  Pointer = HostName;
  while (*HostAndPort != ':' && *HostAndPort != '#' && *HostAndPort != '\0')
    *Pointer++ = *HostAndPort++;
  *Pointer = '\0';
  if (*HostAndPort == ':')
    Port += atoi(++HostAndPort);
  else if (*HostAndPort == '#')
    Port = atoi(++HostAndPort);
  if ((HostAddress = inet_addr(HostName)) == -1)
    {
      HostEntry = gethostbyname(HostName);
      if (HostEntry == NIL || HostEntry->h_addrtype != AF_INET)
	FatalError("Unknown Host '%s'\n",HostName);
      bcopy(HostEntry->h_addr,&HostAddress,sizeof(HostAddress));
    }
  InternetSocketAddress.sin_family = AF_INET;
  InternetSocketAddress.sin_addr.s_addr = INADDR_ANY;
  InternetSocketAddress.sin_port = htons(Port);
  SocketAddress = (struct sockaddr *) &InternetSocketAddress;
  SocketAddressLength = sizeof(InternetSocketAddress);
  Stream = socket(SocketAddress->sa_family,SOCK_STREAM,0);
  if (Stream < 0)
    FatalError("Unable to create socket - %s\n",NIL);
  setsockopt(Stream,SOL_SOCKET,SO_REUSEADDR,(char *)&One,sizeof(int));
  if (bind(Stream,SocketAddress,SocketAddressLength) == -1)
    FatalError("Unable to bind socket - %s\n",NIL);
  if (listen(Stream,1) == -1)
    FatalError("Unable to listen on socket - %s\n",NIL);
  if (DetachProcess && fork() > 0) exit(0);
  setsid();
  NetworkStream = accept(Stream,SocketAddress,&SocketAddressLength);
  if (NetworkStream < 0)
    FatalError("Unable to accept on socket - %s\n",NIL);
  if (InternetSocketAddress.sin_addr.s_addr != HostAddress)
    {
      HostEntry =
	gethostbyaddr((char *)&InternetSocketAddress.sin_addr.s_addr,
		      sizeof(InternetSocketAddress.sin_addr.s_addr),AF_INET);

      if (HostEntry == NIL)
	FatalError("Client from wrong Host\n","");
      strcpy(HostName,HostEntry->h_name);
      HostEntry = gethostbyaddr((char *)&HostAddress,
				sizeof(HostAddress),AF_INET);
      if (HostEntry == NIL || strcmp(HostName,HostEntry->h_name) != 0)
	FatalError("Client from wrong Host\n","");
    }
  setsockopt(NetworkStream,IPPROTO_TCP,TCP_NODELAY,(char *)&One,sizeof(int));
  close(Stream);
  dup2(NetworkStream,0);
  dup2(NetworkStream,1);
  if (DetachProcess) dup2(NetworkStream,2);
  InputPointer = InputBuffer;
  OutputPointer = OutputBuffer;
}


hidden procedure GetDisplayConfiguration()
{
  status_$t Status;
  byte DisplayConfiguration[3];
  int BytesRead;
  BytesRead = ios_$get(ios_$stdin,ios_$no_rec_bndry_opt,
		       (char *)DisplayConfiguration,
		       sizeof(DisplayConfiguration),&Status);
  if (BytesRead != sizeof(DisplayConfiguration))
    FatalError("Unable to get Display Configuration\n","");
  Keyboard = DisplayConfiguration[0];
  ScreenLines = DisplayConfiguration[1];
  ScreenColumns = DisplayConfiguration[2];
}


visible procedure InitializeApolloDisplay()
{
  extern Lisp_Object Vwindow_system;
  ApolloDisplayActivated = true;
  Vwindow_system = intern("apollo");
  meta_key = 1;
  line_ins_del_ok = 1;
  char_ins_del_ok = 1;
  scroll_region_ok = 1;
  fast_clear_end_of_line = 1;
  memory_below_screen = 0;
  must_write_spaces = 0;
  dont_calculate_costs = 1;
  baud_rate = 9600;
  min_padding_speed = 100000;

  if (!NetworkDisplay)
    {
      InitializeGraphics();
      screen_height = ScreenLines;
      screen_width = ScreenColumns;
      set_terminal_modes_hook = InitializeTerminal;
      reset_terminal_modes_hook = ResetTerminal;
      ring_bell_hook = RingBell;
      set_terminal_window_hook = DefineWindow;
      move_cursor_hook = WriteCursorPosition;
      raw_move_cursor_hook = WriteCursorPosition;
      output_chars_hook = WriteCharacters;
      insert_chars_hook = InsertCharacters;
      delete_chars_hook = DeleteCharacters;
      ins_del_lines_hook = InsertOrDeleteLines;
      clear_screen_hook = ClearScreen;
      clear_end_of_line_hook = ClearToEndOfLine;
      clear_to_end_hook = ClearToEndOfWindow;
      reassert_line_highlight_hook = ReassertLineHighlight;
      change_line_highlight_hook = ChangeLineHighlight;
      update_begin_hook = BeginScreenUpdate;
      update_end_hook = EndScreenUpdate;
    }
  else
    {
      InitializeNetwork();
      GetDisplayConfiguration();
      screen_height = ScreenLines;
      screen_width = ScreenColumns;
      set_terminal_modes_hook = N_InitializeTerminal;
      reset_terminal_modes_hook = N_ResetTerminal;
      ring_bell_hook = N_RingBell;
      set_terminal_window_hook = N_DefineWindow;
      move_cursor_hook = N_WriteCursorPosition;
      raw_move_cursor_hook = N_WriteCursorPosition;
      output_chars_hook = N_WriteCharacters;
      insert_chars_hook = N_InsertCharacters;
      delete_chars_hook = N_DeleteCharacters;
      ins_del_lines_hook = N_InsertOrDeleteLines;
      clear_screen_hook = N_ClearScreen;
      clear_end_of_line_hook = N_ClearToEndOfLine;
      clear_to_end_hook = N_ClearToEndOfWindow;
      reassert_line_highlight_hook = N_ReassertLineHighlight;
      change_line_highlight_hook = N_ChangeLineHighlight;
      update_begin_hook = N_BeginScreenUpdate;
      update_end_hook = N_EndScreenUpdate;
      read_socket_hook = N_ReadSocket;
      fix_screen_hook = FlushOutputBuffer;
    }
}

visible boolean apollo_display_available_p()
{
  status_$t Status;
  if (NetworkDisplay) return true;
  pad_$isa_dm_pad(ios_$stdout,&Status);
  return (Status.all == status_$ok);
}


visible boolean apollo_display_suspend_emacs()
{
  pad_$position_t IconPosition;
  status_$t Status;
  char IconFont[256];
  boolean IconMoved;
  if (!ApolloDisplayActivated) return false;
  if (NetworkDisplay)
    {
      Z_IconizeTerminal();
      return true;
    }
  strcpy(IconFont,PATH_EXEC);
  strcat(IconFont,"/apollo.icons");
  pad_$set_icon_font(ios_$stdout,1,IconFont,strlen(IconFont),&Status);
  if (Status.all == status_$ok)
    pad_$make_icon(ios_$stdout,1,(char)'J',&Status);
  else pad_$make_icon(ios_$stdout,1,(char)'E',&Status);
  while (true)
    {
      pad_$icon_wait(ios_$stdout,1,&IconMoved,&IconPosition,&Status);
      if (!IconMoved) break;
    }
  AcquireDisplay(false);
  DisplayMouseCursor(true);
  ReleaseDisplay(true);
  return true;
}

visible boolean apollo_unix_protection_enabled()
{
  return ApolloUnixProtectionEnabled;
}


visible boolean apollo_file_default_type_p(char *file_name)
{
  extern uid_$t unstruct_$uid, uasc_$uid;
  ios_$id_t StreamID;
  uid_$t FileTypeUID;
  status_$t Status;
  StreamID = ios_$open(file_name,strlen(file_name),
		       ios_$inquire_only_opt,&Status);
  if (Status.all != status_$ok) return false;
  ios_$inq_type_uid(StreamID,&FileTypeUID,&Status);
  ios_$close(StreamID,&Status);
  return (FileTypeUID == unstruct_$uid || FileTypeUID == uasc_$uid
	  ? true : false);
}


visible int apollo_create_related_file(char *file_name,
				       int mode,
				       int related_stream_id)
{
  ios_$id_t StreamID;
  uid_$t FileTypeUID;
  status_$t Status;
  boolean NewlyCreatedFile = access(file_name,F_OK);
  if (related_stream_id >= 0)
    {
      ios_$inq_type_uid((ios_$id_t)related_stream_id,&FileTypeUID,&Status);
      if (Status.all != status_$ok) return -1;
    }
  else FileTypeUID = uid_$nil;
  ios_$create(file_name,strlen(file_name),FileTypeUID,ios_$truncate_mode,
	      ios_$write_opt+ios_$unregulated_opt,&StreamID,&Status);
  if (Status.all == ios_$cant_change_type)
    ios_$create(file_name,strlen(file_name),FileTypeUID,ios_$recreate_mode,
		ios_$write_opt+ios_$unregulated_opt,&StreamID,&Status);
  if (Status.all != status_$ok) return -1;
  if (ApolloUnixProtectionEnabled && NewlyCreatedFile)
    {
      int oumask = umask(022);
      umask(oumask);
      fchmod(StreamID,(mode & ~oumask));
    }
  return StreamID;
}

DEFUN("apollo-keyboard-type",Fapollo_keyboard_type,
      Sapollo_keyboard_type,0,0,0,
    "Return the Apollo Keyboard Type (2 or 3).")
   ()
{
  return make_number(Keyboard);
}


DEFUN("enable-apollo-function-key",Fenable_apollo_function_key,
      Senable_apollo_function_key,1,1,
    "nFunction Key: ",
    "Enable the Apollo Function Key Transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
   (Lisp_Object KeyCode)
{
  status_$t Status;
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  if (NetworkDisplay)
    {
      Z_EnableApolloFunctionKey(Key);
      return Qt;
    }
  FixedMappings[KeyCode] = -abs(FixedMappings[KeyCode]);
  if (FixedMappings[KeyCode] < 0) return Qt;
  lib_$add_to_set(FunctionKeysSet,256,Key);
  gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
  return (Status.all == status_$ok ? Qt : Qnil);
}


DEFUN("disable-apollo-function-key",Fdisable_apollo_function_key,
      Sdisable_apollo_function_key,1,1,
    "nFunction Key: ",
    "Disable the Apollo Function Key Transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
   (Lisp_Object KeyCode)
{
  status_$t Status;
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  if (NetworkDisplay)
    {
      Z_DisableApolloFunctionKey(Key);
      return Qt;
    }
  FixedMappings[KeyCode] = abs(FixedMappings[KeyCode]);
  if (FixedMappings[KeyCode] > 0) return Qt;
  lib_$clr_from_set(FunctionKeysSet,256,Key);
  gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
  return (Status.all == status_$ok ? Qt : Qnil);
}

DEFUN("enable-apollo-mouse-button",Fenable_apollo_mouse_button,
      Senable_apollo_mouse_button,1,1,
    "nMouse Button: ",
    "Enable an Apollo Mouse Button transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
   (Lisp_Object KeyCode)
{
  status_$t Status;
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  if (NetworkDisplay)
    {
      Z_EnableApolloMouseButton(Key);
      return Qt;
    }
  lib_$add_to_set(MouseButtonsSet,256,Key);
  gpr_$enable_input(gpr_$buttons,MouseButtonsSet,&Status);
  return (Status.all == status_$ok ? Qt : Qnil);
}


DEFUN("disable-apollo-mouse-button",Fdisable_apollo_mouse_button,
      Sdisable_apollo_mouse_button,1,1,
    "nMouse Button: ",
    "Disable an Apollo Mouse Button transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
   (Lisp_Object KeyCode)
{
  status_$t Status;
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  if (NetworkDisplay)
    {
      Z_DisableApolloMouseButton(Key);
      return Qt;
    }
  lib_$clr_from_set(MouseButtonsSet,256,Key);
  gpr_$enable_input(gpr_$buttons,MouseButtonsSet,&Status);
  return (Status.all == status_$ok ? Qt : Qnil);
}

DEFUN("set-apollo-meta-key",Fset_apollo_meta_key,
      Sset_apollo_meta_key,1,1,
    "nMeta Key Code: ",
    "Select the Meta Key to use by specifying the key code for thewn\n\
key transition.")
   (Lisp_Object MetaKeyCode)
{
  if (XINT(MetaKeyCode) >= 0x40) return Qnil;
  if (NetworkDisplay)
    {
      Z_SetApolloMetaKey(XINT(MetaKeyCode));
      return Qt;
    }
  SetApolloMetaKey(XINT(MetaKeyCode));
  return Qt;
}


DEFUN("select-apollo-protection-style",Fselect_apollo_protection_style,
      Sselect_apollo_protection_style,1,1,
    "SProtection-Style: ",
    "Enable Domain ACL protection if PROTECTION-STYLE is non-NIL or\n\
UNIX Mode protection if PROTECTION-STYLE is NIL.")
   (Lisp_Object protection_style)
{
  if (protection_style == Qnil)
    ApolloUnixProtectionEnabled = true;
  else ApolloUnixProtectionEnabled = false;
  return Qt;
}

DEFUN("write-region-to-default-apollo-paste-buffer",
      Fwrite_region_to_default_apollo_paste_buffer,
      Swrite_region_to_default_apollo_paste_buffer,2,2,"r",
    "Write the region to the default Apollo Paste Buffer.")
   (Lisp_Object Region_Start, Lisp_Object Region_End)
{
  int RegionStart, RegionEnd;
  validate_region(&Region_Start,&Region_End);
  RegionStart = XINT(Region_Start);
  RegionEnd = XINT(Region_End);
  if (NetworkDisplay)
    {
      Z_WritePasteBuffer(RegionEnd-RegionStart);
      while (RegionStart < RegionEnd)
	{
	  if (OutputPointer-OutputBuffer == sizeof(OutputBuffer))
	    FlushOutputBuffer();
	  OutputByte(FETCH_CHAR(RegionStart));
	  RegionStart++;
	}
      FlushOutputBuffer();
    }
  else
    {
      int count = specpdl_ptr-specpdl;
      int FileDescriptor =
	open("`node_data/paste_buffers/default.txt",
	     O_RDWR|O_CREAT|O_TRUNC,0777);
      int RegionStart1, RegionEnd1;
      extern close_file_unwind();
      if (FileDescriptor < 0)
	report_file_error("opening default paste buffer",Qnil);
      record_unwind_protect(close_file_unwind,make_number(FileDescriptor));
      if (RegionStart > GPT)
	RegionStart1 = RegionStart;
      else RegionStart1 = GPT;
      if (RegionEnd < GPT)
	RegionEnd1 = RegionEnd;
      else RegionEnd1 = GPT;
      if (RegionStart != RegionEnd && RegionStart < GPT)
	if (e_write(FileDescriptor,&FETCH_CHAR(RegionStart),
		    RegionEnd1-RegionStart) < 0)
	  report_file_error("I/O error writing paste buffer",Qnil);
      if (RegionStart != RegionEnd && GPT < RegionEnd)
	if (e_write(FileDescriptor,&FETCH_CHAR(RegionStart1),
		    RegionEnd-RegionStart1) < 0)
	  report_file_error("I/O error writing paste buffer",Qnil);
      if (close(FileDescriptor) < 0)
	report_file_error("closing default paste buffer",Qnil);
      specpdl_ptr = specpdl+count;
    }
  return Qt;
}

DEFUN("insert-contents-of-default-apollo-paste-buffer",
      Finsert_contents_of_default_apollo_paste_buffer,
      Sinsert_contents_of_default_apollo_paste_buffer,0,0,0,
    "Insert the contents of the default Apollo Paste Buffer.")
   ()
{
  int count = specpdl_ptr-specpdl;
  int FileDescriptor = 0, FileLength = -1, Inserted = 0, BytesRead;
  if (!NULL(current_buffer->read_only))
    Fbarf_if_buffer_read_only();
  if (NetworkDisplay)
    {
      Z_ReadPasteBuffer();
      FlushOutputBuffer();
      while (FileLength < 0)
	{
	  InputPointer += read(0,InputPointer,1);
	  switch (InputPointer[-1])
	    {
	      case N_Keystroke:
	      case N_SetObscured:
		InputPointer += read(0,InputPointer,1);
		break;
	      case N_FunctionKey:
	      case N_Refresh:
		InputPointer += read(0,InputPointer,3);
		break;
	      case N_Buttons:
		InputPointer += read(0,InputPointer,4);
		break;
	      case N_PasteBuffer:
		InputPointer--;
		read(0,InputPointer,3);
		FileLength = (InputPointer[0]<<16)
			     + (InputPointer[1]<<8) + InputPointer[2];
		break;
	    }
	}
    }
  else
    {
      struct stat FileStatBuffer;
      extern close_file_unwind();
      FileDescriptor = open("`node_data/paste_buffers/default.txt",O_RDONLY);
      if (FileDescriptor < 0)
	report_file_error("opening default paste buffer",Qnil);
      record_unwind_protect(close_file_unwind,make_number(FileDescriptor));
      if (fstat(FileDescriptor,&FileStatBuffer) < 0)
	report_file_error("fstat default paste buffer",Qnil);
      FileLength = FileStatBuffer.st_size;
    }

  if (FileLength <= 0) return make_number(0);
  prepare_to_modify_buffer();
  move_gap(point);
  if (FileLength > GAP_SIZE) make_gap(FileLength-GAP_SIZE);
  while (Inserted < FileLength)
    {
      if (NetworkDisplay)
	{
	  byte ByteCount;
	  read(0,&ByteCount,1);
	  if (ByteCount == 0) break;
	  BytesRead = 0;
	  while (BytesRead < ByteCount)
	    BytesRead +=
	      read(0,&FETCH_CHAR(point+Inserted-1)+1,ByteCount-BytesRead);
	}
      else BytesRead = read(FileDescriptor,&FETCH_CHAR(point+Inserted-1)+1,
			    FileLength-Inserted);
      if (BytesRead <= 0) break;
      GPT += BytesRead;
      GAP_SIZE -= BytesRead;
      ZV += BytesRead;
      Z += BytesRead;
      Inserted += BytesRead;
    }
  if (Inserted > 0) MODIFF++;
  record_insert(point,Inserted);
  if (FileDescriptor > 0)
    {
      if (close(FileDescriptor) < 0)
	report_file_error("closing default paste buffer",Qnil);
      specpdl_ptr = specpdl+count;
    }
  return make_number(Inserted);
}

DEFUN("execute-dm-command",Fexecute_dm_command,
      Sexecute_dm_command,1,1,
    "sExecute DM Command: ",
    "Give the string to the Apollo Display Manager for execution.\n\
Use with caution.")
   (Lisp_Object DMString)
{
  if (!NetworkDisplay)
    {
      char Buffer[20];
      status_$t Status;
      pad_$dm_cmd(ios_$stdout,(char *)(XSTRING(DMString)->data),
		  strlen(XSTRING(DMString)->data),&Status);
      /*
	Status code is too big for a Lisp number, return hex string.
      */
      sprintf(Buffer,"%X",Status);
      return build_string(Buffer);
    }
  else
    {
      char *Pointer = (char *)(XSTRING(DMString)->data);
      int Length = strlen(XSTRING(DMString)->data);
      Z_ExecuteDMCommand(Length);
      while (--Length >= 0) OutputByte(*Pointer++);
    }
}


visible procedure syms_of_apollo()
{
  ApolloDisplayActivated = false;
  ApolloUnixProtectionEnabled = true;
  defsubr(&Sapollo_keyboard_type);
  defsubr(&Senable_apollo_function_key);
  defsubr(&Sdisable_apollo_function_key);
  defsubr(&Senable_apollo_mouse_button);
  defsubr(&Sdisable_apollo_mouse_button);
  defsubr(&Sset_apollo_meta_key);
  defsubr(&Sselect_apollo_protection_style);
  defsubr(&Swrite_region_to_default_apollo_paste_buffer);
  defsubr(&Sinsert_contents_of_default_apollo_paste_buffer);
  defsubr(&Sexecute_dm_command);
}

#else				/* Apollo Display support using NGT */


typedef enum {false, true}  boolean;


extern char
  *NetworkDisplay;


hidden int
  Keyboard,
  ScreenLines,
  ScreenColumns,
  CursorLine,
  CursorColumn,
  WindowLines;


hidden boolean
  HighlightMode,
  WantHighlightMode,
  ApolloDisplayActivated;


hidden byte
  *InputPointer,
  *OutputPointer,
  InputBuffer[1024],
  OutputBuffer[16384];


#define OutputByte(Character)	*OutputPointer++ = (Character)


#include "toktbl.h"
#include "txtcmp.h"


hidden FlushOutputBuffer()
{
  write(1,OutputBuffer,OutputPointer-OutputBuffer);
  OutputPointer = OutputBuffer;
}


hidden procedure SetInsertDeleteAllowed(ScreenUnobscured)
     boolean ScreenUnobscured;
{
  line_ins_del_ok = ScreenUnobscured;
  char_ins_del_ok = ScreenUnobscured;
}

hidden N_InitializeTerminal()
{
  Z_InitializeTerminal();
  CursorLine = 0;
  CursorColumn = 0;
  WindowLines = ScreenLines;
  HighlightMode = false;
  WantHighlightMode = false;
  InitTextCompressor();
}


hidden N_ResetTerminal()
{
  Z_ResetTerminal();
  FlushOutputBuffer();
}


hidden N_RingBell()
{
  Z_RingBell();
}


hidden N_DefineWindow(NewWindowLines)
     int NewWindowLines;
{
  if (NewWindowLines == WindowLines) return;
  if (NewWindowLines < 0 || NewWindowLines > ScreenLines)
    NewWindowLines = ScreenLines;
  Z_DefineWindow(NewWindowLines);
  WindowLines = NewWindowLines;
}


hidden N_WriteCursorPosition(Line,Column)
     int Line, Column;
{
  if (Line == CursorLine && Column == CursorColumn) return;
  if (Line == 0 && Column == 0) Z_HomeCursor()
  else if (Line == CursorLine+1)
    if (Column == 0) Z_CursorDownAndReturn()
    else Z_CursorDownAndWriteColumn(Column)
  else Z_WriteCursorPosition(Line,Column);
  CursorLine = Line;
  CursorColumn = Column;
}


hidden N_WriteCharacters(CharacterText,CharacterCount)
     char *CharacterText;
     int CharacterCount;
{
  N_CheckHighlightMode();
  OutputPointer += CompressText(CharacterText,CharacterCount,OutputPointer);
  CursorColumn += CharacterCount;
}

hidden N_InsertCharacters(CharacterText,CharacterCount)
     char *CharacterText;
     int CharacterCount;
{
  if (CharacterText == NIL)
    {
      Z_InsertSpaces(CharacterCount);
      CursorColumn += CharacterCount;
    }
  else
    {
      Z_OpenSpaces(CharacterCount);
      N_WriteCharacters(CharacterText,CharacterCount);
    }
}


hidden N_DeleteCharacters(DeleteCount)
     int DeleteCount;
{
  Z_DeleteCharacters(DeleteCount);
}


hidden N_InsertOrDeleteLines(Line,InsertOrDeleteCount)
     int Line, InsertOrDeleteCount;
{
  N_BackgroundHighlight();
  if (InsertOrDeleteCount < 0)
    Z_DeleteLines(Line,-InsertOrDeleteCount)
  else Z_InsertLines(Line,InsertOrDeleteCount);
}


hidden N_ClearScreen()
{
  N_BackgroundHighlight();
  Z_ClearScreen();
  CursorLine = 0;
  CursorColumn = 0;
}


hidden N_ClearToEndOfLine(FirstUnusedColumn)
     int FirstUnusedColumn;
{
  N_CheckHighlightMode();
  Z_ClearToEndOfLine();
}


hidden N_ClearToEndOfWindow()
{
  N_BackgroundHighlight();
  Z_ClearToEndOfWindow();
}

hidden N_BackgroundHighlight()
{
  if (!HighlightMode) return;
  Z_HighlightModeOff();
  HighlightMode = false;
}


hidden N_CheckHighlightMode()
{
  if (HighlightMode == WantHighlightMode) return;
  if (WantHighlightMode)
    Z_HighlightModeOn()
  else Z_HighlightModeOff();
  HighlightMode = WantHighlightMode;
}


hidden N_ReassertLineHighlight(OldHighlightMode,Line)
     int OldHighlightMode, Line;
{
  WantHighlightMode = OldHighlightMode;
}


hidden N_ChangeLineHighlight(NewHighlightMode,Line,FirstUnusedColumn)
     int NewHighlightMode, Line, FirstUnusedColumn;
{
  WantHighlightMode = NewHighlightMode;
  N_WriteCursorPosition(Line,0);
  N_ClearToEndOfLine(ScreenColumns);
}


hidden N_BeginScreenUpdate()
{
}


hidden N_EndScreenUpdate()
{
  N_BackgroundHighlight();
  WantHighlightMode = false;
  FlushOutputBuffer();
}


hidden procedure FatalError(Message,Argument)
     char *Message, *Argument;
{
  extern int errno;
  extern char *sys_errlist[];
  if (Argument == NIL)
    fprintf(stderr,Message,sys_errlist[errno]);
  else fprintf(stderr,Message,Argument);
  exit(1);
}

hidden int N_ReadSocket(Descriptor,Buffer,ByteCount)
     int Descriptor;
     byte *Buffer;
     int ByteCount;
{
  extern int input_pending;
  byte *SourcePointer, *TargetPointer, *BufferLimit;
  int BytesRead, Count;
  boolean RefreshPending = false;
  TargetPointer = Buffer;
  if ((BytesRead = InputPointer-InputBuffer) > 0)
    {
      SourcePointer = InputBuffer;
      while (SourcePointer < InputPointer)
	*TargetPointer++ = *SourcePointer++;
      InputPointer = InputBuffer;
    }
  if (ByteCount > 1)
    BytesRead += read(Descriptor,&Buffer[BytesRead],ByteCount);
  if (BytesRead <= 0) return BytesRead;
  SourcePointer = Buffer;
  BufferLimit = &Buffer[BytesRead];
  while (SourcePointer < BufferLimit)
    switch (*SourcePointer++)
      {
	case N_Keystroke:
	  *TargetPointer++ = *SourcePointer++;
	  break;
	case N_FunctionKey:
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  break;
	case N_Buttons:
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  *TargetPointer++ = *SourcePointer++;
	  break;
	case N_Refresh:
	  ScreenLines = *SourcePointer++;
	  ScreenColumns = *SourcePointer++;
	  SetInsertDeleteAllowed(*SourcePointer++);
	  RefreshPending = true;
	  break;
	case N_SetObscured:
	  SetInsertDeleteAllowed(*SourcePointer++);
	  break;
      }
  if (!RefreshPending) return TargetPointer-Buffer;
  input_pending = TargetPointer > Buffer;
  screen_garbaged = 1;
  if (ScreenLines != screen_height || ScreenColumns != screen_width)
    change_screen_size(ScreenLines,ScreenColumns,0);
  else redisplay_preserve_echo_area();
  return TargetPointer-Buffer;
}

hidden procedure InitializeNetwork()
{
  char HostName[256], *HostAndPort, *Pointer;
  struct hostent *HostEntry;
  struct sockaddr_in InternetSocketAddress;
  struct sockaddr *SocketAddress;
  int Stream, NetworkStream, SocketAddressLength;
  int One = 1, FileDescriptor;
  word HostAddress;
  short Port = 6900;
  boolean DetachProcess = true;
  HostAndPort = NetworkDisplay;
  if (*HostAndPort == '=')
    {
      HostAndPort++;
      DetachProcess = false;
    }
  Pointer = HostName;
  while (*HostAndPort != ':' && *HostAndPort != '#' && *HostAndPort != '\0')
    *Pointer++ = *HostAndPort++;
  *Pointer = '\0';
  if (*HostAndPort == ':')
    Port += atoi(++HostAndPort);
  else if (*HostAndPort == '#')
    Port = atoi(++HostAndPort);
  if ((HostAddress = inet_addr(HostName)) == -1)
    {
      HostEntry = gethostbyname(HostName);
      if (HostEntry == NIL || HostEntry->h_addrtype != AF_INET)
	FatalError("Unknown Host '%s'\n",HostName);
      bcopy(HostEntry->h_addr,&HostAddress,sizeof(HostAddress));
    }
  InternetSocketAddress.sin_family = AF_INET;
  InternetSocketAddress.sin_addr.s_addr = INADDR_ANY;
  InternetSocketAddress.sin_port = htons(Port);
  SocketAddress = (struct sockaddr *) &InternetSocketAddress;
  SocketAddressLength = sizeof(InternetSocketAddress);
  Stream = socket(SocketAddress->sa_family,SOCK_STREAM,0);
  if (Stream < 0)
    FatalError("Unable to create socket - %s\n",NIL);
  setsockopt(Stream,SOL_SOCKET,SO_REUSEADDR,(char *)&One,sizeof(int));
  if (bind(Stream,SocketAddress,SocketAddressLength) == -1)
    FatalError("Unable to bind socket - %s\n",NIL);
  if (listen(Stream,1) == -1)
    FatalError("Unable to listen on socket - %s\n",NIL);

  if (DetachProcess && fork() > 0) exit(0);
#ifdef HAVE_SETSID
  setsid();
#else
#ifdef TIOCNOTTY
  FileDescriptor = open("/dev/tty",O_RDWR);
  ioctl(FileDescriptor,TIOCNOTTY,0);
  close(FileDescriptor);
#endif
#endif
  NetworkStream = accept(Stream,SocketAddress,&SocketAddressLength);
  if (NetworkStream < 0)
    FatalError("Unable to accept on socket - %s\n",NIL);
  if (InternetSocketAddress.sin_addr.s_addr != HostAddress)
    {
      HostEntry =
	gethostbyaddr((char *)&InternetSocketAddress.sin_addr.s_addr,
		      sizeof(InternetSocketAddress.sin_addr.s_addr),AF_INET);
      if (HostEntry == NIL)
	FatalError("Client from wrong Host\n","");
      strcpy(HostName,HostEntry->h_name);
      HostEntry = gethostbyaddr((char *)&HostAddress,
				sizeof(HostAddress),AF_INET);
      if (HostEntry == NIL || strcmp(HostName,HostEntry->h_name) != 0)
	FatalError("Client from wrong Host\n","");
    }
#ifdef TCP_NODELAY
  setsockopt(NetworkStream,IPPROTO_TCP,TCP_NODELAY,(char *)&One,sizeof(int));
#endif
  close(Stream);
  dup2(NetworkStream,0);
  dup2(NetworkStream,1);
  if (DetachProcess) dup2(NetworkStream,2);
  InputPointer = InputBuffer;
  OutputPointer = OutputBuffer;
}


hidden procedure GetDisplayConfiguration()
{
  byte DisplayConfiguration[3];
  int BytesRead;
  BytesRead = read(0,DisplayConfiguration,sizeof(DisplayConfiguration));
  if (BytesRead != sizeof(DisplayConfiguration))
    FatalError("Unable to get Display Configuration\n","");
  Keyboard = DisplayConfiguration[0];
  ScreenLines = DisplayConfiguration[1];
  ScreenColumns = DisplayConfiguration[2];
}

visible procedure InitializeApolloDisplay()
{
  extern Lisp_Object Vwindow_system;
  ApolloDisplayActivated = true;
  Vwindow_system = intern("apollo");
  InitializeNetwork();
  GetDisplayConfiguration();
  screen_height = ScreenLines;
  screen_width = ScreenColumns;
  meta_key = 1;
  line_ins_del_ok = 1;
  char_ins_del_ok = 1;
  scroll_region_ok = 1;
  fast_clear_end_of_line = 1;
  memory_below_screen = 0;
  must_write_spaces = 0;
  dont_calculate_costs = 1;
  baud_rate = 9600;
  min_padding_speed = 100000;
  set_terminal_modes_hook = N_InitializeTerminal;
  reset_terminal_modes_hook = N_ResetTerminal;
  ring_bell_hook = N_RingBell;
  set_terminal_window_hook = N_DefineWindow;
  move_cursor_hook = N_WriteCursorPosition;
  raw_move_cursor_hook = N_WriteCursorPosition;
  output_chars_hook = N_WriteCharacters;
  insert_chars_hook = N_InsertCharacters;
  delete_chars_hook = N_DeleteCharacters;
  ins_del_lines_hook = N_InsertOrDeleteLines;
  clear_screen_hook = N_ClearScreen;
  clear_end_of_line_hook = N_ClearToEndOfLine;
  clear_to_end_hook = N_ClearToEndOfWindow;
  reassert_line_highlight_hook = N_ReassertLineHighlight;
  change_line_highlight_hook = N_ChangeLineHighlight;
  update_begin_hook = N_BeginScreenUpdate;
  update_end_hook = N_EndScreenUpdate;
  read_socket_hook = N_ReadSocket;
  fix_screen_hook = FlushOutputBuffer;
}

visible boolean apollo_display_available_p()
{
  return (NetworkDisplay != NIL);
}


visible boolean apollo_display_suspend_emacs()
{
  if (!ApolloDisplayActivated) return false;
  Z_IconizeTerminal();
  return true;
}


DEFUN("apollo-keyboard-type",Fapollo_keyboard_type,
      Sapollo_keyboard_type,0,0,0,
    "Return the Apollo Keyboard Type (2 or 3).")
   ()
{
  return make_number(Keyboard);
}


DEFUN("enable-apollo-function-key",Fenable_apollo_function_key,
      Senable_apollo_function_key,1,1,
    "nFunction Key: ",
    "Enable the Apollo Function Key Transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
   (KeyCode)
     Lisp_Object KeyCode;
{
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  Z_EnableApolloFunctionKey(Key);
  return Qt;
}

DEFUN("disable-apollo-function-key",Fdisable_apollo_function_key,
      Sdisable_apollo_function_key,1,1,
    "nFunction Key: ",
    "Disable the Apollo Function Key Transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
  (KeyCode)
     Lisp_Object KeyCode;
{
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  Z_DisableApolloFunctionKey(Key);
  return Qt;
}


DEFUN("enable-apollo-mouse-button",Fenable_apollo_mouse_button,
      Senable_apollo_mouse_button,1,1,
    "nMouse Button: ",
    "Enable an Apollo Mouse Button transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
   (KeyCode)
     Lisp_Object KeyCode;
{
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  Z_EnableApolloMouseButton(Key);
  return Qt;
}


DEFUN("disable-apollo-mouse-button",Fdisable_apollo_mouse_button,
      Sdisable_apollo_mouse_button,1,1,
    "nMouse Button: ",
    "Disable an Apollo Mouse Button transition by specifying the transition\n\
key code.  See '/usr/include/apollo/kbd.h' for definitions of the key codes.")
  (KeyCode)
     Lisp_Object KeyCode;
{
  short Key = XINT(KeyCode);
  if (Key < 0 || Key > 0xFF) return Qnil;
  Z_DisableApolloMouseButton(Key);
  return Qt;
}

DEFUN("set-apollo-meta-key",Fset_apollo_meta_key,
      Sset_apollo_meta_key,1,1,
    "nMeta Key Code: ",
    "Select the Meta Key to use by specifying the key code for thewn\n\
key transition.")
   (MetaKeyCode)
     Lisp_Object MetaKeyCode;
{
  int MetaKey = XINT(MetaKeyCode);
  if (XINT(MetaKeyCode) >= 0x40) return Qnil;
  Z_SetApolloMetaKey(MetaKey);
  return Qt;
}


DEFUN("write-region-to-default-apollo-paste-buffer",
      Fwrite_region_to_default_apollo_paste_buffer,
      Swrite_region_to_default_apollo_paste_buffer,2,2,"r",
    "Write the region to the default Apollo Paste Buffer.")
   (Region_Start, Region_End)
     Lisp_Object Region_Start, Region_End;
{
  int RegionStart, RegionEnd;
  validate_region(&Region_Start,&Region_End);
  RegionStart = XINT(Region_Start);
  RegionEnd = XINT(Region_End);
  Z_WritePasteBuffer(RegionEnd-RegionStart);
  while (RegionStart < RegionEnd)
    {
      if (OutputPointer-OutputBuffer == sizeof(OutputBuffer))
	FlushOutputBuffer();
      OutputByte(FETCH_CHAR(RegionStart));
      RegionStart++;
    }
  FlushOutputBuffer();
  return Qt;
}

DEFUN("insert-contents-of-default-apollo-paste-buffer",
      Finsert_contents_of_default_apollo_paste_buffer,
      Sinsert_contents_of_default_apollo_paste_buffer,0,0,0,
    "Insert the contents of the default Apollo Paste Buffer.")
   ()
{
  int FileLength = -1, Inserted = 0, BytesRead;
  if (!NULL(current_buffer->read_only))
    Fbarf_if_buffer_read_only();
  Z_ReadPasteBuffer();
  FlushOutputBuffer();
  while (FileLength < 0)
    {
      InputPointer += read(0,InputPointer,1);
      switch (InputPointer[-1])
	{
	  case N_Keystroke:
	  case N_SetObscured:
	    InputPointer += read(0,InputPointer,1);
	    break;
	  case N_FunctionKey:
	  case N_Refresh:
	    InputPointer += read(0,InputPointer,3);
	    break;
	  case N_Buttons:
	    InputPointer += read(0,InputPointer,4);
	    break;
	  case N_PasteBuffer:
	    InputPointer--;
	    read(0,InputPointer,3);
	    FileLength = (InputPointer[0]<<16)
			 +(InputPointer[1]<<8)+InputPointer[2];
	    break;
	}
    }
  if (FileLength <= 0) return make_number(0);
  prepare_to_modify_buffer();
  move_gap(point);
  if (FileLength > GAP_SIZE) make_gap(FileLength-GAP_SIZE);
  while (Inserted < FileLength)
    {
      byte ByteCount;
      read(0,&ByteCount,1);
      if (ByteCount == 0) break;
      BytesRead = 0;
      while (BytesRead < ByteCount)
	BytesRead += read(0,&FETCH_CHAR(point+Inserted-1)+1,
			  ByteCount-BytesRead);
      GPT += BytesRead;
      GAP_SIZE -= BytesRead;
      ZV += BytesRead;
      Z += BytesRead;
      Inserted += BytesRead;
    }
  if (Inserted > 0) MODIFF++;
  record_insert(point,Inserted);
  return make_number(Inserted);
}

DEFUN("execute-dm-command",Fexecute_dm_command,
      Sexecute_dm_command,1,1,
    "sExecute DM Command: ",
    "Give the string to the Apollo Display Manager for execution.\n\
Use with caution.")
   (DMString)
     Lisp_Object DMString;
{
  char *Pointer = (char *)(XSTRING(DMString)->data);
  int Length = strlen(XSTRING(DMString)->data);
  Z_ExecuteDMCommand(Length);
  while (--Length >= 0) OutputByte(*Pointer++);
}


visible procedure syms_of_apollo()
{
  ApolloDisplayActivated = false;
  defsubr(&Sapollo_keyboard_type);
  defsubr(&Senable_apollo_function_key);
  defsubr(&Sdisable_apollo_function_key);
  defsubr(&Senable_apollo_mouse_button);
  defsubr(&Sdisable_apollo_mouse_button);
  defsubr(&Sset_apollo_meta_key);
  defsubr(&Swrite_region_to_default_apollo_paste_buffer);
  defsubr(&Sinsert_contents_of_default_apollo_paste_buffer);
  defsubr(&Sexecute_dm_command);
}


#endif
