/*	NGT -- Network Graphics Terminal for Apollo Emacs using GPR	     */
/*									     */
/*	Copyright (C) 1987 by Leonard N. Zubkoff, All Rights Reserved	     */
/*									     */
/*	This software is provided free and without any warranty.	     */
/*	Permission to copy for any purpose is hereby granted so		     */
/*	long as this copyright notice remains intact.			     */
/*									     */
/*	Revision:	18-Aug-92 11:59:15				     */


#define hidden	    static
#define visible
#define procedure   void


typedef unsigned long	    word;
typedef unsigned short	    halfword;
typedef unsigned char	    byte;


#include <fcntl.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/tcp.h>


#include <apollo/base.h>
#include <apollo/ec2.h>
#include <apollo/fontn.h>
#include <apollo/gpr.h>
#include <apollo/ios.h>
#include <apollo/kbd.h>
#include <apollo/ms.h>
#include <apollo/pad.h>
#include <apollo/pfm.h>
#include <apollo/tone.h>
#include "ngt.h"
#include "tokstr.h"


#define NIL	    0L


#define Ctrl(Character)	    ((Character) & 0x1F)
#define Meta(Character)	    ((Character) | 0x80)

#define MouseCursorHeight   16
#define MouseCursorWidth    8
#define MouseCursorXOrigin  0
#define MouseCursorYOrigin  0
#define VisibleListSize	    64


hidden int
  Keyboard,
  AcquireCount,
  CursorLine,
  CursorColumn,
  ScreenLines,
  ScreenColumns,
  WindowLines;


hidden short
  FontHeight,
  FontWidth,
  FontOffset,
  FontVerticalSpacing,
  FontHorizontalSpacing,
  FontNameLength,
  ScreenVisibleCount,
  FixedMappings[256],
  MouseCursorPattern[MouseCursorHeight] =
    { 0x8000, 0xC000, 0xE000, 0xF000,
      0xF800, 0xFC00, 0xFE00, 0xF000,
      0xD800, 0x9800, 0x0C00, 0x0C00,
      0x0600, 0x0600, 0x0300, 0x0300 };


hidden boolean
  ColorDisplay,
  LineClear[256],
  ScreenUnobscured,
  LastScreenUnobscured = true,
  MouseCursorInWindow = true;


hidden pad_$string_t
  FontName;


hidden gpr_$bitmap_desc_t
  ScreenBitmap,
  MouseCursorBitmap,
  MouseCursorTempBitmap;

hidden gpr_$position_t
  MouseCursorPosition;


hidden gpr_$pixel_value_t
  Background,
  Characters;


hidden gpr_$raster_op_t
  ColorCursor1[4] =
    { gpr_$rop_zeros, gpr_$rop_src, gpr_$rop_zeros, gpr_$rop_src },
  ColorCursor2[4] =
    { gpr_$rop_zeros, gpr_$rop_zeros, gpr_$rop_src, gpr_$rop_src };


hidden gpr_$mask_32_t
  NormalPlaneMask,
  FullPlaneMask,
  PlaneMask123;


hidden gpr_$window_t
  ScreenVisibleList[VisibleListSize];


hidden ios_$id_t
  NetworkStream;


#define GraphicsEventCount  0
#define NetworkEventCount   1


hidden ec2_$ptr_t
  EventCountPointers[2];


hidden long
  EventCountValues[2];


hidden short
  MetaKey;


hidden gpr_$keyset_t
  CodedKeysSet,
  FunctionKeysSet,
  PhysicalKeysSet,
  MouseButtonsSet;

hidden byte
  InputBuffer[16384+1],
  OutputBuffer[4096],
  DisplayBuffer[256],
  *InputLimit,
  *InputPointer = InputBuffer,
  *OutputPointer = OutputBuffer,
  *DisplayPointer = DisplayBuffer;


#define InputByte()			(*InputPointer++)
#define OutputByte(Character)		*OutputPointer++ = (Character)
#define DisplayCharacter(Character)	*DisplayPointer++ = (Character)


hidden procedure RemoveAcceptedInput()
{
  if (InputPointer < InputLimit)
    {
      byte *Pointer = InputPointer;
      InputPointer = InputBuffer;
      while (Pointer < InputLimit)
	*InputPointer++ = *Pointer++;
    }
  else InputPointer = InputBuffer;
}


hidden procedure FatalError(char *Message,
			    char *Argument)
{
  extern int errno;
  extern char *sys_errlist[];
  if (Argument == NIL)
    printf(Message,sys_errlist[errno]);
  else printf(Message,Argument);
  exit(1);
}


hidden boolean FillInputBuffer()
{
  status_$t Status;
  long ByteCount, BytesRead;
  boolean AnyInput = false;
  while (true)
    {
      ByteCount = sizeof(InputBuffer)-1 - (InputPointer-InputBuffer);
      BytesRead = ios_$get(NetworkStream,ios_$cond_opt+ios_$no_rec_bndry_opt,
			   (char *)InputPointer,ByteCount,&Status);
      if (Status.all == ios_$end_of_file) exit(0);
      if (Status.all == ios_$get_conditional_failed) break;
      if (Status.all != status_$ok) pfm_$error_trap(Status);
      InputPointer += BytesRead;
      if (BytesRead > 0) AnyInput = true;
    }
  InputLimit = InputPointer;
  return AnyInput;
}

hidden procedure FlushOutputBuffer()
{
  status_$t Status;
  long BytesRemaining = OutputPointer-OutputBuffer, ByteCount;
  OutputPointer = OutputBuffer;
  while (BytesRemaining > 0)
    {
      ByteCount = ios_$putp(NetworkStream,ios_$partial_ok_opt,
			    (char *)OutputPointer,BytesRemaining,&Status);
      if (Status.all != status_$ok) pfm_$error_trap(Status);
      BytesRemaining -= ByteCount;
      OutputPointer += ByteCount;
    }
  OutputPointer = OutputBuffer;
}


hidden procedure FlushDisplayBuffer()
{
  extern procedure ClearRectangle();
  gpr_$coordinate_t CursorX, CursorY;
  status_$t Status;
  int DisplayCount = DisplayPointer-DisplayBuffer, i;
  if (DisplayCount == 0) return;
  if (!LineClear[CursorLine])
    ClearRectangle(CursorLine,CursorColumn,1,DisplayCount);
  for (i=0; i<ScreenVisibleCount; i++)
    {
      if (!ScreenUnobscured)
	{
	  gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  CursorX = CursorColumn * FontWidth;
	  CursorY = FontOffset + CursorLine * FontHeight;
	  gpr_$move(CursorX,CursorY,&Status);
	}
      gpr_$text((char *)DisplayBuffer,(short)DisplayCount,&Status);
    }
  CursorColumn += DisplayCount;
  LineClear[CursorLine] = false;
  DisplayPointer = DisplayBuffer;
}

hidden procedure ClearRectangle(int Line,
				int Column,
				int LineCount,
				int ColumnCount)
{
  gpr_$window_t Rectangle;
  status_$t Status;
  int i;
  Rectangle.window_base.x_coord = Column * FontWidth;
  Rectangle.window_base.y_coord = Line * FontHeight;
  Rectangle.window_size.x_size = ColumnCount * FontWidth;
  Rectangle.window_size.y_size = LineCount * FontHeight;
  for (i=0; i<ScreenVisibleCount; i++)
    {
      if (!ScreenUnobscured)
	gpr_$set_clip_window(ScreenVisibleList[i],&Status);
      gpr_$rectangle(Rectangle,&Status);
    }
  if (ColumnCount == ScreenColumns)
    for (i=Line; i<Line+LineCount; i++)
      LineClear[i] = true;
}


hidden procedure CopyRectangle(int SourceLine,
			       int SourceColumn,
			       int DestinationLine,
			       int DestinationColumn,
			       int LineCount,
			       int ColumnCount)
{
  gpr_$window_t SourceWindow;
  gpr_$position_t DestinationOrigin;
  status_$t Status;
  int i;
  SourceWindow.window_base.x_coord = SourceColumn * FontWidth;
  SourceWindow.window_base.y_coord = SourceLine * FontHeight;
  SourceWindow.window_size.x_size = ColumnCount * FontWidth;
  SourceWindow.window_size.y_size = LineCount * FontHeight;
  DestinationOrigin.x_coord = DestinationColumn * FontWidth;
  DestinationOrigin.y_coord = DestinationLine * FontHeight;
  if (ScreenUnobscured)
    gpr_$pixel_blt(ScreenBitmap,SourceWindow,
		   DestinationOrigin,&Status);
  if (ColumnCount == ScreenColumns)
    if (SourceLine > DestinationLine)
      for (i=0; i<LineCount; i++)
	LineClear[DestinationLine+i] = LineClear[SourceLine+i];
    else for (i=LineCount-1; i>=0; --i)
      LineClear[DestinationLine+i] = LineClear[SourceLine+i];
}

hidden procedure DisplayCharacterCursor(boolean Activate)
{
  gpr_$window_t SourceWindow;
  status_$t Status;
  int i;
  SourceWindow.window_base.x_coord = CursorColumn * FontWidth - 1;
  SourceWindow.window_base.y_coord = CursorLine * FontHeight;
  SourceWindow.window_size.x_size = FontWidth - FontHorizontalSpacing + 2;
  SourceWindow.window_size.y_size = FontHeight - FontVerticalSpacing + 2;
  if (!ColorDisplay)
    {
      gpr_$set_raster_op(0,gpr_$rop_not_dst,&Status);
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$pixel_blt(ScreenBitmap,SourceWindow,
			 SourceWindow.window_base,&Status);
	}
      gpr_$set_raster_op(0,gpr_$rop_src,&Status);
    }
  else if (Activate)
    {
      gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
      gpr_$set_raster_op_mask(PlaneMask123,gpr_$rop_zeros,&Status);
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$additive_blt(ScreenBitmap,SourceWindow,3,
			    SourceWindow.window_base,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
  else
    {
      int Option = (Background & 7) >> 1;
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
	  gpr_$bit_blt(ScreenBitmap,SourceWindow,3,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_zeros,&Status);
	  gpr_$set_raster_op(1,ColorCursor1[Option],&Status);
	  gpr_$set_raster_op(2,ColorCursor2[Option],&Status);
	  gpr_$additive_blt(ScreenBitmap,SourceWindow,0,
			    SourceWindow.window_base,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
}

hidden procedure DisplayMouseCursor(boolean Activate)
{
  static gpr_$window_t SourceWindow =
    { 0, 0, MouseCursorWidth, MouseCursorHeight };
  static gpr_$window_t SourceWindow2 =
    { 0, 0, MouseCursorWidth, MouseCursorHeight };
  status_$t Status;
  int i;
  if (!MouseCursorInWindow) return;
  SourceWindow2.window_base = MouseCursorPosition;
  if (!ColorDisplay)
    {
      gpr_$set_raster_op(0,gpr_$rop_src_xor_dst,&Status);
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$pixel_blt(MouseCursorBitmap,SourceWindow,
			 MouseCursorPosition,&Status);
	}
      gpr_$set_raster_op(0,gpr_$rop_src,&Status);
    }
  else if (Activate)
    {
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$set_bitmap(MouseCursorTempBitmap,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src,&Status);
	  gpr_$bit_blt(ScreenBitmap,SourceWindow2,0,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
	  gpr_$bit_blt(ScreenBitmap,SourceWindow2,3,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_and_dst,&Status);
	  gpr_$bit_blt(MouseCursorBitmap,SourceWindow,0,
		       SourceWindow.window_base,0,&Status);
	  gpr_$set_bitmap(ScreenBitmap,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
	  gpr_$set_raster_op_mask(PlaneMask123,
				  gpr_$rop_not_src_and_dst,&Status);
	  gpr_$additive_blt(MouseCursorTempBitmap,SourceWindow,0,
			    MouseCursorPosition,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
  else
    {
      int Option = (Background & 7) >> 1;
      for (i=0; i<ScreenVisibleCount; i++)
	{
	  if (!ScreenUnobscured)
	    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
	  gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);

	  gpr_$bit_blt(ScreenBitmap,SourceWindow2,3,
		       MouseCursorPosition,0,&Status);
	  gpr_$set_raster_op(0,gpr_$rop_zeros,&Status);
	  gpr_$set_raster_op(1,ColorCursor1[Option],&Status);
	  gpr_$set_raster_op(2,ColorCursor2[Option],&Status);
	  gpr_$additive_blt(ScreenBitmap,SourceWindow2,0,
			    MouseCursorPosition,&Status);
	}
      gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
    }
}


hidden procedure AcquireDisplay(boolean RemoveCharacterCursor)
{
  status_$t Status;
  if (AcquireCount++ > 0) return;
  ScreenUnobscured = gpr_$acquire_display(&Status);
  if (!ScreenUnobscured)
    gpr_$inq_vis_list(VisibleListSize,&ScreenVisibleCount,
		      ScreenVisibleList,&Status);
  else ScreenVisibleCount = 1;
  if (RemoveCharacterCursor) DisplayCharacterCursor(false);
  DisplayMouseCursor(false);
  if (ScreenVisibleCount == 0) return;
  if (ScreenUnobscured == LastScreenUnobscured) return;
  OutputByte(N_SetObscured);
  OutputByte(ScreenUnobscured);
  FlushOutputBuffer();
  LastScreenUnobscured = ScreenUnobscured;
}


hidden procedure ReleaseDisplay(boolean RestoreCharacterCursor)
{
  static gpr_$window_t FullClipWindow =
    { 0, 0, gpr_$max_x_size, gpr_$max_y_size };
  status_$t Status;
  if (--AcquireCount > 0) return;
  DisplayMouseCursor(true);
  if (RestoreCharacterCursor) DisplayCharacterCursor(true);
  if (!ScreenUnobscured) gpr_$set_clip_window(FullClipWindow,&Status);
  gpr_$release_display(&Status);
}

hidden procedure RefreshProcedure(boolean &Unobscured,
				  boolean &PositionChanged)
{
  pad_$window_desc_t CurrentWindow;
  gpr_$position_t WindowOrigin;
  status_$t Status;
  short WindowCount, AcquireCount;
  if (PositionChanged)
    {
      gpr_$force_release(&AcquireCount,&Status);
      pad_$inq_windows(ios_$stdout,&CurrentWindow,1,&WindowCount,&Status);
      while (--AcquireCount >= 0)
	Unobscured = gpr_$acquire_display(&Status);
      ScreenLines = (CurrentWindow.height-2)/FontHeight;
      ScreenColumns = (CurrentWindow.width-2)/FontWidth;
      WindowOrigin.x_coord = (CurrentWindow.width-ScreenColumns*FontWidth)/2;
      WindowOrigin.y_coord = (CurrentWindow.height-ScreenLines*FontHeight)/2;
      gpr_$set_coordinate_origin(WindowOrigin,&Status);
    }
  ScreenUnobscured = Unobscured;
  if (!ScreenUnobscured)
    gpr_$inq_vis_list(VisibleListSize,&ScreenVisibleCount,
		      ScreenVisibleList,&Status);
  else ScreenVisibleCount = 1;
  OutputByte(N_Refresh);
  OutputByte(ScreenLines);
  OutputByte(ScreenColumns);
  OutputByte(ScreenUnobscured);
  FlushOutputBuffer();
  LastScreenUnobscured = ScreenUnobscured;
}

hidden procedure InitializeTerminal()
{
  extern procedure InitDecompressor();
  extern procedure HighlightModeOff();
  extern procedure ClearScreen();
  CursorLine = 0;
  CursorColumn = 0;
  WindowLines = ScreenLines;
  InitDecompressor();
  HighlightModeOff();
  ClearScreen();
}


hidden procedure ResetTerminal()
{
}


hidden procedure IconizeTerminal()
{
  pad_$position_t IconPosition;
  status_$t Status;
  char IconFont[256];
  boolean IconMoved;
  strcpy(IconFont,"/gnuemacs/etc/apollo.icons");
  pad_$set_icon_font(ios_$stdout,1,IconFont,strlen(IconFont),&Status);
  if (Status.all == status_$ok)
    pad_$make_icon(ios_$stdout,1,(char)'J',&Status);
  else pad_$make_icon(ios_$stdout,1,(char)'E',&Status);
  while (true)
    {
      pad_$icon_wait(ios_$stdout,1,&IconMoved,&IconPosition,&Status);
      if (!IconMoved) break;
    }
}


hidden procedure RingBell()
{
  static time_$clock_t BellTime = { 0, 32768 };
  tone_$time(BellTime);
}

hidden procedure DefineWindow(int NewWindowLines)
{
  WindowLines = NewWindowLines;
}


hidden procedure ClearScreen()
{
  extern procedure WriteCursorPosition();
  status_$t Status;
  int i;
  if (ColorDisplay) gpr_$set_plane_mask_32(FullPlaneMask,&Status);
  for (i=0; i<ScreenVisibleCount; i++)
    {
      if (!ScreenUnobscured)
	gpr_$set_clip_window(ScreenVisibleList[i],&Status);
      gpr_$clear(Background,&Status);
    }
  if (ColorDisplay) gpr_$set_plane_mask_32(NormalPlaneMask,&Status);
  WriteCursorPosition(0,0);
}


hidden procedure ClearToEndOfLine()
{
  ClearRectangle(CursorLine,CursorColumn,1,ScreenColumns-CursorColumn);
}


hidden procedure ClearToEndOfWindow()
{
  if (CursorColumn > 0)
    {
      ClearToEndOfLine(ScreenColumns);
      if (CursorLine < WindowLines-1)
	ClearRectangle(CursorLine+1,0,WindowLines-CursorLine-1,ScreenColumns);
    }
  else ClearRectangle(CursorLine,0,WindowLines-CursorLine,ScreenColumns);
}


hidden procedure WriteCursorPosition(int Line,
				     int Column)
{
  gpr_$coordinate_t CursorX, CursorY;
  status_$t Status;
  CursorLine = Line;
  CursorColumn = Column;
  CursorX = CursorColumn * FontWidth;
  CursorY = FontOffset + CursorLine * FontHeight;
  gpr_$move(CursorX,CursorY,&Status);
}

hidden procedure DeleteCharacters(int DeleteCount)
{
  CopyRectangle(CursorLine,CursorColumn+DeleteCount,
		CursorLine,CursorColumn,
		1,ScreenColumns-(CursorColumn+DeleteCount));
  ClearRectangle(CursorLine,ScreenColumns-DeleteCount,1,DeleteCount);
}


hidden procedure OpenSpaces(int OpenCount)
{
  CopyRectangle(CursorLine,CursorColumn,
		CursorLine,CursorColumn+OpenCount,
		1,ScreenColumns-(CursorColumn+OpenCount));
  ClearRectangle(CursorLine,CursorColumn,1,OpenCount);
}


hidden procedure InsertSpaces(int InsertCount)
{
  CopyRectangle(CursorLine,CursorColumn,
		CursorLine,CursorColumn+InsertCount,
		1,ScreenColumns-(CursorColumn+InsertCount));
  ClearRectangle(CursorLine,CursorColumn,1,InsertCount);
  WriteCursorPosition(CursorLine,CursorColumn+InsertCount);
}


hidden procedure OverwriteSpaces(int OverwriteCount)
{
  ClearRectangle(CursorLine,CursorColumn,1,OverwriteCount);
  WriteCursorPosition(CursorLine,CursorColumn+OverwriteCount);
}


hidden procedure OverwriteCharacters(int OverwriteCount,
				     int OverwriteCharacter)
{
  while (--OverwriteCount >= 0)
    DisplayCharacter(OverwriteCharacter);
}


hidden procedure DeleteLines(int Line,
			     int DeleteCount)
{
  CopyRectangle(Line+DeleteCount,0,Line,0,
		WindowLines-(Line+DeleteCount),ScreenColumns);
  ClearRectangle(WindowLines-DeleteCount,0,DeleteCount,ScreenColumns);
}


hidden procedure InsertLines(int Line,
			     int InsertCount)
{
  CopyRectangle(Line,0,Line+InsertCount,0,
		WindowLines-(Line+InsertCount),ScreenColumns);
  ClearRectangle(Line,0,InsertCount,ScreenColumns);
}

hidden procedure HighlightModeOff()
{
  status_$t Status;
  gpr_$set_text_background_value(Background,&Status);
  gpr_$set_text_value(Characters,&Status);
  gpr_$set_fill_value(Background,&Status);
}


hidden procedure HighlightModeOn()
{
  status_$t Status;
  gpr_$set_text_background_value(Characters,&Status);
  gpr_$set_text_value(Background,&Status);
  gpr_$set_fill_value(Characters,&Status);
}


hidden procedure EnableApolloFunctionKey(int KeyCode)
{
  status_$t Status;
  FixedMappings[KeyCode] = -abs(FixedMappings[KeyCode]);
  if (FixedMappings[KeyCode] < 0) return;
  lib_$add_to_set(FunctionKeysSet,256,(short)KeyCode);
  gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
}


hidden procedure DisableApolloFunctionKey(int KeyCode)
{
  status_$t Status;
  FixedMappings[KeyCode] = abs(FixedMappings[KeyCode]);
  if (FixedMappings[KeyCode] > 0) return;
  lib_$clr_from_set(FunctionKeysSet,256,(short)KeyCode);
  gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
}


hidden procedure EnableApolloMouseButton(int KeyCode)
{
  status_$t Status;
  lib_$add_to_set(MouseButtonsSet,256,(short)KeyCode);
  gpr_$enable_input(gpr_$buttons,MouseButtonsSet,&Status);
}


hidden procedure DisableApolloMouseButton(int KeyCode)
{
  status_$t Status;
  lib_$clr_from_set(MouseButtonsSet,256,(short)KeyCode);
  gpr_$enable_input(gpr_$buttons,MouseButtonsSet,&Status);
}

hidden procedure SetApolloMetaKey(int MetaKeyCode)
{
  status_$t Status;
  MetaKey = MetaKeyCode;
  lib_$init_set(PhysicalKeysSet,256);
  if (MetaKey == KBD3_$AL || MetaKey == KBD3_$AR)
    {
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AL);
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AR);
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AL+KBD3_$KEY_UP);
      lib_$add_to_set(PhysicalKeysSet,256,KBD3_$AR+KBD3_$KEY_UP);
    }
  else
    {
      lib_$add_to_set(PhysicalKeysSet,256,MetaKey);
      lib_$add_to_set(PhysicalKeysSet,256,MetaKey+KBD3_$KEY_UP);
    }
  gpr_$enable_input(gpr_$physical_keys,PhysicalKeysSet,&Status);
}


hidden procedure WritePasteBuffer(int PasteBufferLength)
{
  int FileDescriptor = open("`node_data/paste_buffers/default.txt",
			    O_RDWR|O_CREAT|O_TRUNC,0777);
  int ByteCount;
  if (FileDescriptor < 0) return;
  while (PasteBufferLength > 0)
    {
      ByteCount = InputLimit-InputPointer;
      if (ByteCount == 0)
	{
	  RemoveAcceptedInput();
	  FillInputBuffer();
	  InputPointer = InputBuffer;
	  ByteCount = InputLimit - InputPointer;
	  if (ByteCount == 0) sleep(1);
	}
      if (ByteCount > PasteBufferLength)
	ByteCount = PasteBufferLength;
      write(FileDescriptor,InputPointer,ByteCount);
      InputPointer += ByteCount;
      PasteBufferLength -= ByteCount;
    }
  close(FileDescriptor);
}

hidden procedure ReadPasteBuffer()
{
  int FileDescriptor = open("`node_data/paste_buffers/default.txt",O_RDONLY);
  struct stat FileStatBuffer;
  if (FileDescriptor < 0) return;
  if (fstat(FileDescriptor,&FileStatBuffer) == 0)
    {
      int FileLength = FileStatBuffer.st_size;
      OutputByte(N_PasteBuffer);
      OutputByte((FileLength>>16));
      OutputByte((FileLength>>8));
      OutputByte((FileLength));
      FlushOutputBuffer();
      while (FileLength > 0)
	{
	  int ByteCount = FileLength;
	  if (ByteCount > 255) ByteCount = 255;
	  OutputByte(0);
	  ByteCount = read(FileDescriptor,OutputPointer,ByteCount);
	  if (ByteCount > 0)
	    {
	      OutputBuffer[0] = ByteCount;
	      OutputPointer += ByteCount;
	    }
	  FlushOutputBuffer();
	  FileLength -= ByteCount;
	  if (ByteCount == 0) break;
	}
    }
  close(FileDescriptor);
}


hidden procedure ExecuteDMCommand(byte *DMString,
				  int DMStringLength)
{
  status_$t Status;
  pad_$dm_cmd(ios_$stdout,(char *)DMString,
	      (short)DMStringLength,&Status);
}

hidden int
  NextDynamicTokenNumber;


hidden boolean
  TrailingSpace[8] =
    { false, false, false, true, false, true, false, true };


hidden char
  CaseConvert1[8] =
    { 0, 0, ' ', ' ', 0, 0, ' ', ' ' },
  CaseConvert2[8] =
    { 0, 0, ' ', ' ', 0, 0, 0, 0 },
  Letters1[16] =
    { 0, 'e', 't', 'i', 'o', 'n', 'a', 's',
      'r', 'l', 'h', 'c', 'd', 'm', 'u', 'p' },
  Letters2[12] =
    { 0, 'f', 'g', 'b', 'y', 'w', 'v', 'k', 'x', 'j', 'q', 'z' },
  Shifts[16] =
    { 24, 28, 16, 20, 8, 12, 0, 4, 24, 28, 16, 20, 8, 12, 0, 4 };


hidden struct
  {
    word Word0, Word1;
  }
TokenWords[256];


hidden procedure InitDecompressor()
{
  NextDynamicTokenNumber = 0;
}

hidden procedure DisplayDynamicToken(int TokenNumber,
				     int TokenType)
{
  word TokenWord = TokenWords[TokenNumber].Word0;
  int Letter, Nibble = 0;
  Letter = Letters1[(TokenWord>>Shifts[Nibble++])&15];
  if (Letter == 0) Letter = Letters2[(TokenWord>>Shifts[Nibble++])&15];
  DisplayCharacter(Letter-CaseConvert1[TokenType]);
  while (true)
    {
      Letter = Letters1[(TokenWord>>Shifts[Nibble++])&15];
      if (Letter == 0)
	{
	  if (Nibble == 8)
	    TokenWord = TokenWords[TokenNumber].Word1;
	  else if (Nibble == 16) break;
	  Letter = Letters2[(TokenWord>>Shifts[Nibble++])&15];
	}
      if (Letter == 0) break;
      DisplayCharacter(Letter-CaseConvert2[TokenType]);
      if (Nibble == 8)
	TokenWord = TokenWords[TokenNumber].Word1;
      else if (Nibble == 16) break;
    }
  if (TrailingSpace[TokenType]) DisplayCharacter(' ');
}

hidden procedure ProcessInputBuffer()
{
  DefineCommandArgumentCounts;
  InputPointer = InputBuffer;
  *InputLimit = 0xFF;
  while (InputPointer < InputLimit)
    {
      int Character = InputByte();
      int ArgumentCount = CommandArgumentCounts[Character];
      int Argument;
      if (ArgumentCount == 8)
	ArgumentCount = 1 + (Character & 7);
      else if (ArgumentCount == 9)
	ArgumentCount = InputPointer[0] + 1;
      if (InputLimit-InputPointer < ArgumentCount)
	{
	  --InputPointer;
	  break;
	}
      if (Character >= L_DefineAndDisplayDynamicToken &&
	  Character <= U_DefineAndDisplayDynamicToken)
	{
	  int TokenType = (Character>>3) & 7;
	  int TokenByteCount = 1 + (Character & 7);
	  byte *Pointer = (byte *) &TokenWords[NextDynamicTokenNumber];
	  TokenWords[NextDynamicTokenNumber].Word0 = 0;
	  TokenWords[NextDynamicTokenNumber].Word1 = 0;
	  while (--TokenByteCount >= 0)
	    *Pointer++ = InputByte();
	  DisplayDynamicToken(NextDynamicTokenNumber,TokenType);
	  NextDynamicTokenNumber++;
	  NextDynamicTokenNumber &= 255;
	}
      else if (Character >= L_DisplayDynamicToken &&
	       Character <= U_DisplayDynamicToken)
	DisplayDynamicToken(InputByte(),Character & 7);
      else if (Character >= L_DisplayStaticToken &&
	       Character <= U_DisplayStaticToken)
	{
	  char *Pointer = StaticTokens[((Character & 3)<<8)+InputByte()];
	  int TokenType = (Character>>2) & 7;
	  Character = *Pointer++;
	  DisplayCharacter(Character-CaseConvert1[TokenType]);
	  while ((Character = *Pointer++) != '\0')
	    DisplayCharacter(Character-CaseConvert2[TokenType]);
	  if (TrailingSpace[TokenType]) DisplayCharacter(' ');
	}

      else if (Character >= L_DisplayQuickToken &&
	       Character <= U_DisplayQuickToken)
	{
	  char *Pointer = QuickTokens[Character & 0x1F];
	  while ((Character = *Pointer++) != '\0')
	    DisplayCharacter(Character);
	}
      else
	{
	  if (Character < ' ' || Character > '~') FlushDisplayBuffer();
	  switch (Character)
	    {
	      case C_InitializeTerminal:
	        InitializeTerminal();
		break;
	      case C_ResetTerminal:
		ResetTerminal();
		break;
	      case C_IconizeTerminal:
		ReleaseDisplay(false);
		IconizeTerminal();
		AcquireDisplay(false);
		break;
	      case C_RingBell:
		RingBell();
		break;
	      case C_DefineWindow:
		DefineWindow(InputByte());
		break;
	      case C_ClearScreen:
		ClearScreen();
		break;
	      case C_ClearToEndOfLine:
		ClearToEndOfLine();
		break;
	      case C_ClearToEndOfWindow:
		ClearToEndOfWindow();
		break;
	      case C_HomeCursor:
		WriteCursorPosition(0,0);
		break;
	      case C_CursorDownAndReturn:
		WriteCursorPosition(CursorLine+1,0);
		break;

	      case C_CursorDownAndWriteColumn:
		WriteCursorPosition(CursorLine+1,InputByte());
		break;
	      case C_WriteCursorPosition:
		Argument = InputByte();
		WriteCursorPosition(Argument,InputByte());
		break;
	      case C_DeleteCharacters:
		DeleteCharacters(InputByte());
		break;
	      case C_OpenSpaces:
		OpenSpaces(InputByte());
		break;
	      case C_InsertSpaces:
		InsertSpaces(InputByte());
		break;
	      case C_OverwriteSpaces:
		OverwriteSpaces(InputByte());
		break;
	      case C_OverwriteCharacters:
		Argument = InputByte();
		OverwriteCharacters(Argument,InputByte());
		break;
	      case C_DeleteLines:
		Argument = InputByte();
		DeleteLines(Argument,InputByte());
		break;
	      case C_InsertLines:
		Argument = InputByte();
		InsertLines(Argument,InputByte());
		break;
	      case C_HighlightModeOff:
		HighlightModeOff();
		break;
	      case C_HighlightModeOn:
		HighlightModeOn();
		break;
	      case C_EnableApolloFunctionKey:
		EnableApolloFunctionKey(InputByte());
		break;
	      case C_DisableApolloFunctionKey:
		DisableApolloFunctionKey(InputByte());
		break;

	      case C_EnableApolloMouseButton:
		EnableApolloMouseButton(InputByte());
		break;
	      case C_DisableApolloMouseButton:
		DisableApolloMouseButton(InputByte());
		break;
	      case C_SetApolloMetaKey:
		SetApolloMetaKey(InputByte());
		break;
	      case C_WritePasteBuffer:
		ReleaseDisplay(false);
		Argument = InputByte();
		Argument = (Argument<<8)+InputByte();
		Argument = (Argument<<8)+InputByte();
		WritePasteBuffer(Argument);
		AcquireDisplay(false);
		break;
	      case C_ReadPasteBuffer:
		ReleaseDisplay(false);
		ReadPasteBuffer();
		AcquireDisplay(false);
		break;
	      case C_ExecuteDMCommand:
		ReleaseDisplay(false);
		Argument = InputByte();
		ExecuteDMCommand(InputPointer,Argument);
		InputPointer += Argument;
		AcquireDisplay(false);
		break;
	      default:
		DisplayCharacter(Character);
		break;
	    }
	}
    }
  RemoveAcceptedInput();
  FlushDisplayBuffer();
}

hidden procedure ProcessNetworkInput()
{
  while (FillInputBuffer())
    {
      AcquireDisplay(true);
      ProcessInputBuffer();
      ReleaseDisplay(true);
    }
}


hidden procedure InitializeNetwork(char *NetworkDisplay)
{
  status_$t Status;
  char HostName[256], *HostAndPort = NetworkDisplay, *Pointer = HostName;
  struct hostent *HostEntry;
  struct sockaddr_in InternetSocketAddress;
  struct sockaddr *SocketAddress;
  int SocketAddressLength, One = 1;
  word HostAddress;
  short Port = 6900;
  while (*HostAndPort != ':' && *HostAndPort != '#' && *HostAndPort != '\0')
    *Pointer++ = *HostAndPort++;
  *Pointer = '\0';
  if (*HostAndPort == ':')
    Port += atoi(++HostAndPort);
  else if (*HostAndPort == '#')
    Port = atoi(++HostAndPort);
  if ((HostAddress = inet_addr(HostName)) == -1)
    {
      HostEntry = gethostbyname(HostName);
      if (HostEntry == NIL || HostEntry->h_addrtype != AF_INET)
	FatalError("Unknown Host '%s'\n",HostName);
      bcopy(HostEntry->h_addr,&InternetSocketAddress.sin_addr,
	    sizeof(InternetSocketAddress.sin_addr));
    }
  else InternetSocketAddress.sin_addr.s_addr = HostAddress;
  InternetSocketAddress.sin_family = AF_INET;
  InternetSocketAddress.sin_port = htons(Port);
  SocketAddress = (struct sockaddr *) &InternetSocketAddress;
  SocketAddressLength = sizeof(InternetSocketAddress);
  while (true)
    {
      NetworkStream = socket(SocketAddress->sa_family,SOCK_STREAM,0);
      if (NetworkStream < 0)
	FatalError("Unable to create socket - %s\n",NIL);
      setsockopt(NetworkStream,IPPROTO_TCP,TCP_NODELAY,
		 (char *)&One,sizeof(int));
      if (connect(NetworkStream,SocketAddress,SocketAddressLength) == 0) break;
      close(NetworkStream);
      sleep(1);
    }
  ios_$get_ec(NetworkStream,ios_$get_ec_key,
	      &EventCountPointers[NetworkEventCount],&Status);
  if (Status.all != status_$ok) pfm_$error_trap(Status);
}

hidden procedure ProcessGraphicsInput()
{
  static boolean MetaDepressed = false;
  gpr_$event_t EventType;
  status_$t Status;
  byte EventData[1];
  gpr_$position_t EventPosition;
  short AcquireCount;
  while (true)
    {
      gpr_$cond_event_wait(&EventType,&EventData[0],&EventPosition,&Status);
      if (EventType == gpr_$no_event) break;
      if (Status.all != status_$ok) pfm_$error_trap(Status);
      if (EventType == gpr_$coded_keys)
	{
	  int Keystroke = EventData[0];
	  OutputByte(N_Keystroke);
	  if (MetaDepressed)
	    OutputByte(Meta(Keystroke));
	  else OutputByte(Keystroke);
	}
      else if (EventType == gpr_$function_keys)
	{
	  int Keystroke = EventData[0];
	  if (FixedMappings[Keystroke] > 0)
	    {
	      OutputByte(N_Keystroke);
	      if (MetaDepressed)
		OutputByte(Meta(FixedMappings[Keystroke]));
	      else OutputByte(FixedMappings[Keystroke]);
	    }
	  else
	    {
	      OutputByte(N_FunctionKey);
	      OutputByte(Ctrl('^'));
	      OutputByte((MetaDepressed ? 0x4C : 0x48)
			 | (Keystroke >> 6));
	      OutputByte(0x40 | (Keystroke & 0x3F));
	    }
	}

      else if (EventType == gpr_$physical_keys)
	MetaDepressed = (EventData[0] <KBD3_$KEY_UP);
      else if (EventType == gpr_$locator_update)
	{
	  AcquireDisplay(false);
	  MouseCursorPosition = EventPosition;
	  ReleaseDisplay(ColorDisplay);
	}
      else if (EventType == gpr_$buttons)
	{
	  AcquireDisplay(false);
	  MouseCursorPosition = EventPosition;
	  ReleaseDisplay(ColorDisplay);
	  OutputByte(N_Buttons);
	  OutputByte(Ctrl('^'));
	  OutputByte(EventData[0]+(MetaDepressed ? 16 : 0));
	  OutputByte(8+(MouseCursorPosition.x_coord
			+MouseCursorXOrigin)/FontWidth);
	  OutputByte(8+(MouseCursorPosition.y_coord
			+MouseCursorYOrigin)/FontHeight);
	}
      else if (EventType == gpr_$entered_window)
	{
	  AcquireDisplay(false);
	  MouseCursorInWindow = true;
	  MouseCursorPosition = EventPosition;
	  ReleaseDisplay(ColorDisplay);
	}
      else if (EventType == gpr_$left_window)
	{
	  AcquireDisplay(false);
	  MouseCursorInWindow = false;
	  ReleaseDisplay(ColorDisplay);
	}
    }
  if (OutputPointer > OutputBuffer)
    {
      gpr_$force_release(&AcquireCount,&Status);
      pad_$pop_push_window(ios_$stdout,1,true,&Status);
      while (--AcquireCount >= 0)
	ScreenUnobscured = gpr_$acquire_display(&Status);
    }
  FlushOutputBuffer();
}

hidden procedure InitializeGraphics()
{
  pad_$string_t KeyboardType;
  fontn_$table_tver1 *FontTable;
  pad_$window_desc_t CurrentWindow;
  pad_$position_t IconPosition;
  gpr_$offset_t ScreenBitmapSize, MouseCursorBitmapSize;
  gpr_$rgb_plane_t HiPlane;
  gpr_$position_t WindowOrigin;
  gpr_$attribute_desc_t CursorAttributeBlock;
  gpr_$keyset_t KeySet;
  status_$t Status;
  word LengthMapped;
  short KeyboardLength, WindowCount, FontID, Key;
  short *MouseCursorBitmapPointer, CursorLineWidth, i;
  boolean IconMoved;
  pad_$inq_kbd(ios_$stdin,sizeof(KeyboardType),KeyboardType,
	       &KeyboardLength,&Status);
  Keyboard = KeyboardType[0]-'0';
  AcquireCount = 0;
  while (true)
    {
      pad_$icon_wait(ios_$stdout,1,&IconMoved,&IconPosition,&Status);
      if (!IconMoved) break;
    }
  pad_$inq_font(ios_$stdout,&FontWidth,&FontHeight,
		FontName,sizeof(FontName),&FontNameLength,&Status);
  if (Status.all != status_$ok)
    {
      printf("Pad_$Inq_Font failed, using f7x13.b\n");
      strcpy(FontName,"/sys/dm/fonts/f7x13.b");
      FontNameLength = strlen(FontName);
    }
  pad_$raw(ios_$stdin,&Status);
  FontTable =
    (fontn_$table_tver1 *) ms_$mapl(FontName,FontNameLength,
				    0,sizeof(fontn_$table_tver1),
				    ms_$nr_xor_1w,ms_$r,
				    false,&LengthMapped,&Status);
  FontHeight = FontTable->max_height + FontTable->v_spacing;
  FontWidth = FontTable->space_size + FontTable->inter_space_dflt;
  FontOffset = FontTable->baseline_offset + FontTable->v_spacing/2-1
	       - (FontTable->font_height+1-FontTable->max_height)/2;
  FontVerticalSpacing = FontTable->v_spacing;
  FontHorizontalSpacing = FontTable->inter_space_dflt;
  ms_$unmap(FontTable,LengthMapped,&Status);

  pad_$set_scale(ios_$stdout,1,1,&Status);
  pad_$inq_windows(ios_$stdout,&CurrentWindow,1,&WindowCount,&Status);
  ScreenLines = (CurrentWindow.height-2)/FontHeight;
  ScreenColumns = (CurrentWindow.width-2)/FontWidth;
  ScreenBitmapSize.x_size = gpr_$max_x_size;
  ScreenBitmapSize.y_size = gpr_$max_y_size;
  gpr_$init(gpr_$direct,ios_$stdout,ScreenBitmapSize,
	    gpr_$highest_plane,&ScreenBitmap,&Status);
  gpr_$inq_bitmap_dimensions(ScreenBitmap,&ScreenBitmapSize,&HiPlane,&Status);
  ColorDisplay = (HiPlane > 0);
  Background = (ColorDisplay ? gpr_$inq_background(&Status) : 0);
  Characters = (ColorDisplay ? 0 : 1);
  if (ColorDisplay)
    {
      NormalPlaneMask = (1<<4)-1;
      FullPlaneMask = (1<<(HiPlane+1))-1;
      PlaneMask123 = (1<<1)|(1<<2)|(1<<3);
      gpr_$set_plane_mask_32(NormalPlaneMask,&Status);
    } 
  WindowOrigin.x_coord = (CurrentWindow.width-ScreenColumns*FontWidth)/2;
  WindowOrigin.y_coord = (CurrentWindow.height-ScreenLines*FontHeight)/2;
  gpr_$set_coordinate_origin(WindowOrigin,&Status);
  gpr_$set_obscured_opt(gpr_$input_ok_if_obs,&Status);
  gpr_$set_refresh_entry(RefreshProcedure,(gpr_$rhdm_pr_t)NIL,&Status);
  gpr_$load_font_file(FontName,FontNameLength,&FontID,&Status);
  gpr_$set_text_font(FontID,&Status);
  MouseCursorBitmapSize.x_size = MouseCursorWidth;
  MouseCursorBitmapSize.y_size = MouseCursorHeight;
  gpr_$allocate_attribute_block(&CursorAttributeBlock,&Status);
  gpr_$allocate_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
		       &MouseCursorBitmap,&Status);
  gpr_$acquire_display(&Status);
  gpr_$allocate_hdm_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
			   &MouseCursorTempBitmap,&Status);
  if (Status.all != status_$ok)
    gpr_$allocate_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
			 &MouseCursorTempBitmap,&Status);
  gpr_$release_display(&Status);

  gpr_$inq_bitmap_pointer(MouseCursorBitmap,(char **)&MouseCursorBitmapPointer,
			  &CursorLineWidth,&Status);
  for (i=0; i<MouseCursorHeight; i++)
    {
      *MouseCursorBitmapPointer = MouseCursorPattern[i];
      MouseCursorBitmapPointer += CursorLineWidth;
    }
  bzero(FixedMappings,sizeof(FixedMappings));
  FixedMappings[KBD3_$ESC] = Ctrl('[');
  FixedMappings[KBD3_$TAB] = Ctrl('I');
  FixedMappings[KBD3_$RET] = Ctrl('M');
  FixedMappings[KBD3_$BS] = 0x7F;
  FixedMappings[KBD3_$DEL] = 0x7F;
  for (i=0; i<=9; i++)
    FixedMappings[KBD3_$NP0+i] = '0'+i;
  FixedMappings[KBD3_$NPE] = Ctrl('M');
  FixedMappings[KBD3_$NPF] = '-';
  FixedMappings[KBD3_$NPG] = '+';
  FixedMappings[KBD3_$NPP] = '.';
  lib_$init_set(CodedKeysSet,256);
  lib_$init_set(FunctionKeysSet,256);
  lib_$init_set(PhysicalKeysSet,256);
  lib_$init_set(MouseButtonsSet,256);
  for (Key=0; Key<0x80; Key++)
    lib_$add_to_set(CodedKeysSet,256,Key);
  for (Key=0; Key<256; Key++)
    if (FixedMappings[Key] > 0) lib_$add_to_set(FunctionKeysSet,256,Key);
  SetApolloMetaKey(KBD3_$L9);
  gpr_$enable_input(gpr_$coded_keys,CodedKeysSet,&Status);
  gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
  gpr_$enable_input(gpr_$locator_update,MouseButtonsSet,&Status);
  gpr_$enable_input(gpr_$entered_window,MouseButtonsSet,&Status);
  gpr_$enable_input(gpr_$left_window,MouseButtonsSet,&Status);
  gpr_$set_quit_event(gpr_$keystroke,'\377',&Status);
  gpr_$get_ec(gpr_$input_ec,&EventCountPointers[GraphicsEventCount],&Status);
  EventCountValues[GraphicsEventCount] = 0;
}


hidden procedure SendDisplayConfiguration()
{
  status_$t Status;
  byte Buffer[3];
  Buffer[0] = Keyboard;
  Buffer[1] = ScreenLines;
  Buffer[2] = ScreenColumns;
  ios_$put(NetworkStream,ios_$no_put_get_opts,
	   (char *)Buffer,sizeof(Buffer),&Status);
  if (Status.all != status_$ok)
    FatalError("Client from wrong Host\n","");
}

visible int main(ArgCount,ArgVector,Environment)
	int ArgCount;
	char *ArgVector[], *Environment[];
{
  status_$t Status;
  if (ArgCount != 2)
    FatalError("Usage: ngt <network-display>\n","");
  InitializeNetwork(ArgVector[1]);
  InitializeGraphics();
  SendDisplayConfiguration();
  while (true)
    {
      switch (ec2_$wait(EventCountPointers,EventCountValues,2,&Status)-1)
	{
	  case GraphicsEventCount:
	    EventCountValues[GraphicsEventCount] =
	      1+ec2_$read(*EventCountPointers[GraphicsEventCount]);
	    ProcessGraphicsInput();
	    break;
	  case NetworkEventCount:
	    EventCountValues[NetworkEventCount] =
	      1+ec2_$read(*EventCountPointers[NetworkEventCount]);
	    ProcessNetworkInput();
	    break;
	}
    }
}
