#include "config.h"

/* Only compile for Apollo.  */

#ifdef TM_APOLLO

#include <stdio.h>

#include <apollo/base.h>
#include <apollo/loader.h>

#include "rtl.h"
#include "tree.h"
#include "c-tree.h"
#include "output.h"
#include "regs.h"
#include "flags.h"
#include "obstack.h"



/* These field accessors would normally be in tree.h.  They are
   only needed here.  */

#define IDENTIFIER_SEARCHED_P(NODE)  ((NODE)->identifier.apollo_searched_flag)
#define IDENTIFIER_NEEDS_PIC_P(NODE) ((NODE)->identifier.apollo_needs_pic_flag)
#define IDENTIFIER_A0_RETURN(NODE)   ((NODE)->identifier.apollo_a0_return)
#define IDENTIFIER_PIC_OFFSET(NODE)  ((NODE)->identifier.apollo_pic_offset)

/* This was taken from varasm.c.  */

/* This macro gets just the user-specified name
   out of the string in a SYMBOL_REF.  On most machines,
   we discard the * if any and that's all.  */
#ifndef STRIP_NAME_ENCODING
#define STRIP_NAME_ENCODING(VAR,SYMBOL_NAME) \
  (VAR) = ((SYMBOL_NAME) + ((SYMBOL_NAME)[0] == '*'))
#endif

/* These definitions were copied from tree.c for obstack support.  */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

extern void gcc_obstack_init ();   /* in tree.c */

/* This is the list of symbols needing offset table entries for PIC code.
   It is an array of pointers to the IDENTIFIER_NODEs for each symbol.
   It is built once per source file.  */

static struct obstack apollo_obstack;

/* This is the number of entries in apollo_obstack.  */

static int apollo_table_size = 0;

/* This is TRUE if PIC references are needed for externals found in the
   Apollo Known Global Table (KGT) but not for other symbols.  This is the
   normal mode.  Setting -fpic or -fPIC will make this FALSE in
   apollo_pic_init ().  */

static int apollo_flag_pic = 0;

/* Move this to apollo68.h later */
/* These are used as the prefixes for labels put into functions for use
   in the .unwind section.  The decimal value of unwind_label_counter
   appended to them to make unique names.  */

#define UNWIND_START_PREFIX ".L.unwind_start."
#define UNWIND_END_PREFIX   ".L.unwind_end."

/* This is used to count label pairs inserted into functions for use by
   the .unwind section entries.  */

static int unwind_label_counter = 0;

/* output_unwind_record () writes assembler code for an unwind record to
   the current section.  It may be used in output_function_epilogue to
   generate .unwind section entries or Function Control Blocks in the
   .text section.  The caller must switch sections or do alignments.
   This code covers only one form of unwind record for functions that use
   a frame pointer.  It is intended to do only the bare minimum needed
   by the Apollo traceback (tb) command.  Without .blocks and .lines
   sections, the tb command can not print symbol names, only offsets.  */

static void output_unwind_record (stream, offset, regmask, fpregmask)
     FILE *stream;
     int  offset;    /* from FP to first saved register */
     int  regmask;   /* A7 A6 ... D0 save mask */
     int  fpregmask; /* FP7 ...  FP0 save mask */
{
  /* Output an unwind_$descriptor_t.  The first field is start_pc.  */
  asm_fprintf (stream, "\t.long %s%d\n", UNWIND_START_PREFIX,
	       unwind_label_counter);

  /* length:  24 bit function length.  */
  asm_fprintf (stream, "\t.int24 %s%d-%s%d\n", UNWIND_END_PREFIX,
	       unwind_label_counter, UNWIND_START_PREFIX,
	       unwind_label_counter);

  /* kind:  always 1 for udk$m68k_compressed.  */
  asm_fprintf (stream, "\t.byte 1\n");

  /* Next is a short form unwind_$m68k_desc_t.  saved_a_regs and
     saved_d_regs are combined into one field here.  It is a mask
     in the form A7 A6 ... A0 D7 D6 ... D0.  Normally only explicitly
     saved registers appear in it.  (D2-D7, A2-A5) */
  asm_fprintf (stream, "\t.short %d\n", regmask);

  /* saved_fp_regs.  It is a mask with FP0 being the LSB.  Normally
     only explicitly saved registers appear in it.  (FP2-FP7) */
  asm_fprintf (stream, "\t.short %d\n", fpregmask);

  /* saved_reg_off.  Offset from FP to register save area.  */      
  asm_fprintf (stream, "\t.long %d\n", -offset);
}



/* This initializes the offset table builder.  It should be called at the
   start of each source file, usually from toplev.c.
   Note that flag_pic is always set nonzero afterward.  */

void
apollo_pic_init ()
{
  apollo_table_size = 0;
  gcc_obstack_init (&apollo_obstack);

  /* Reset the compiler PIC flag to the appropriate mode.  */
  if (flag_pic == 0)
    {
      apollo_flag_pic = 1;
      flag_pic = 2;
    }
}



/* This writes the offset table to the assembler output file.  It should
   be called at the end of each source file before the identifer space
   is freed.  */

void
apollo_pic_finish ()
{
  tree*   offset_table = obstack_finish (&apollo_obstack);

  /* Generate assembler code for the offset table.  */
  if (apollo_table_size > 0   ||   apollo_pic_all_p ())
    {
      data_section ();
      ASM_OUTPUT_ALIGN (asm_out_file, 1);
      ASM_OUTPUT_LABEL (asm_out_file, GLOBAL_OFFSET_TABLE_NAME);
      while (apollo_table_size-- > 0)
        {
          asm_fprintf (asm_out_file, "\t.long ");
          assemble_name (asm_out_file,
                         IDENTIFIER_POINTER (*offset_table ++));
          asm_fprintf (asm_out_file, "\n");
        }
    }
  /* end if */

  /* Free the entire list.  */
  obstack_free (&apollo_obstack, 0);
}



/* This returns TRUE if all symbol references require PIC relocation.
   The test of flag_pic is not currently needed, but is here in case
   the flag settings in apollo_pic_init () are changed.  */

int
apollo_pic_all_p ()
{
  return   ! apollo_flag_pic   &&   flag_pic;
}



/* Nonzero if the constant value OP is a legitimate general operand
   when generating PIC code.  It is given that flag_pic is on and 
   that X satisfies CONSTANT_P or is a CONST_DOUBLE.  This function
   has the side effect of adding to the global offset table any
   non-PIC symbols in OP.  */

int
apollo_legitimate_pic_operand_p (op)
     rtx op;
{
  void*                    address;
  int                      legitimate;
  loader_$kg_lookup_opts   lookup_opts = 0;
  char*                    name;
  tree                     id;

  switch (GET_CODE (op))
    {
    case SYMBOL_REF:
      name = XSTR (op, 0);
      STRIP_NAME_ENCODING (name, name);
      id   = get_identifier (name);
      if (id == 0)
	abort ();

      /* Look up the identifier in the KGT if this has not already
	 been done.  Local labels do require a KGT search.  */
      if (IDENTIFIER_SEARCHED_P (id))
	legitimate = ! IDENTIFIER_NEEDS_PIC_P (id);
      else
	{
	  IDENTIFIER_NEEDS_PIC_P (id)
	    = (     apollo_pic_all_p ()
	       ||   (     name[0] != '.'   &&   name[1] != 'L'
		     &&   loader_$kg_lookup (name,   strlen (name),
					     loader_$kg_symbol,
					     lookup_opts,   &address) != 0));
	  legitimate = ! IDENTIFIER_NEEDS_PIC_P (id);

	  if (! legitimate)
	    {
	      /* Add the symbol to the offset table and
		 mark it as referenced.  */
	      IDENTIFIER_PIC_OFFSET (id) = UNITS_PER_WORD * apollo_table_size;
	      obstack_ptr_grow_fast (&apollo_obstack,  id);
	      apollo_table_size ++;
	      TREE_SYMBOL_REFERENCED (id) = 1;
	    }

	  IDENTIFIER_SEARCHED_P (id) = 1;
	}
      break;

    case CONST:
      /* Usually a PLUS expression.  If the operands are not legitimate, look
	 for the special form produced by legitimize_pic_address ().  */
      op = XEXP (op, 0);
      legitimate = (     (     apollo_legitimate_pic_operand_p (XEXP (op, 0))
		          &&   apollo_legitimate_pic_operand_p (XEXP (op, 1)))
		    ||   (     GET_CODE (op) == PLUS
			  &&   XEXP (op, 0) == pic_offset_table_rtx
			  &&   (     GET_CODE (XEXP (op, 1)) == SYMBOL_REF
				||   GET_CODE (XEXP (op, 1)) == LABEL_REF)));
      break;
    default:
      legitimate = 1;
    }
  /* end switch */

  return legitimate;
}



/* This converts ORIG to make it a legitimate PIC address through
   register REG.  Most of this code was taken from config/m68k.c.  */

rtx
legitimize_pic_address (orig, mode, reg)
     rtx orig;
     enum machine_mode mode;
     rtx reg;
{
  rtx   pic_ref = orig;
  char* name;

  /* Handle a SYMBOL_REF by accumulating a "global offset table" of symbols
     and converting the symbol names into numeric offsets from the start
     of this table.  This code places every non-PIC symbol into the table.

     TODO someday:  Speedups might be possible for special cases like text
     section labels or defined data symbols at short offsets from the global
     offset table.  Example:  for nearby text section ".LC" labels, generate
     short PC-relative references directly and keep the label out of the
     global offset table.  Long offsets might need two instructions.
     Since the offsets are not known until assembly time, adding new
     assembler directives similar to JBSR would be the best approach.  */

  if (     GET_CODE (orig) == SYMBOL_REF
      &&   ! apollo_legitimate_pic_operand_p (orig))
    {
      if (reg == 0)
	abort ();

      /* Convert the name into a numeric offset.  */
      name = XSTR (orig, 0);
      STRIP_NAME_ENCODING (name, name);
      pic_ref = plus_constant_for_output
	(pic_offset_table_rtx,
	 IDENTIFIER_PIC_OFFSET (get_identifier (name)));
      pic_ref = gen_rtx (MEM, Pmode, pic_ref);
      current_function_uses_pic_offset_table = 1;
      RTX_UNCHANGING_P (pic_ref) = 1;
      emit_move_insn (reg, pic_ref);
      pic_ref = reg;
    }
  else if (GET_CODE (orig) == LABEL_REF)
    {
      if (reg == 0)
	abort ();

      pic_ref = gen_rtx (MEM, Pmode,
			 gen_rtx (PLUS, Pmode,
				  pic_offset_table_rtx, orig));
      current_function_uses_pic_offset_table = 1;
      RTX_UNCHANGING_P (pic_ref) = 1;
      emit_move_insn (reg, pic_ref);
      pic_ref = reg;
    }
  else if (GET_CODE (orig) == CONST)
    {
      rtx base, offset;

      /* Make sure this is CONST has not already been legitimized */
      if (GET_CODE (XEXP (orig, 0)) == PLUS
	  && XEXP (XEXP (orig, 0), 0) == pic_offset_table_rtx)
	return orig;

      if (reg == 0)
	abort ();

      /* legitimize both operands of the PLUS */
      if (GET_CODE (XEXP (orig, 0)) == PLUS)
	{
	  base = legitimize_pic_address (XEXP (XEXP (orig, 0), 0), Pmode, reg);
	  orig = legitimize_pic_address (XEXP (XEXP (orig, 0), 1), Pmode,
					 base == reg ? 0 : reg);
	}
      else abort ();

      if (GET_CODE (orig) == CONST_INT)
	return plus_constant_for_output (base, INTVAL (orig));
      pic_ref = gen_rtx (PLUS, Pmode, base, orig);
      /* Likewise, should we set special REG_NOTEs here?  */
    }
  return pic_ref;
}



/* This generates the assembly code to declare a function name.  For full
   Apollo PIC, it puts the function symbol in the data area and generates
   the entry sequence.  */

void
apollo_asm_declare_function_name (out_file, fn_name, fn_decl)
     FILE* out_file;
     char* fn_name;
     tree  fn_decl;
{
  if (apollo_pic_all_p ())
    {
      data_section ();
      ASM_OUTPUT_ALIGN (out_file, 1);
      ASM_OUTPUT_LABEL (out_file, fn_name);
      asm_fprintf (out_file, "\tlea %0U%s,%Ra0\n\tjmp %LL.PIC.",
		   GLOBAL_OFFSET_TABLE_NAME);
      assemble_name (out_file, fn_name);
      asm_fprintf (out_file, "\n");
      text_section ();
      ASM_OUTPUT_ALIGN (out_file, 1);
      asm_fprintf (out_file, "%LL.PIC.");
      ASM_OUTPUT_LABEL (out_file, fn_name);
    }
  else
    ASM_OUTPUT_LABEL (asm_out_file, fn_name);
}



/* This routine generates the assembly code for function entry.
   This was taken almost unchanged from m68k.c.  Code was added
   for making labels for the .unwind section entry, and the PIC
   portion was changed to use the Apollo conventions.  */

void
output_function_prologue (stream, size)
     FILE *stream;
     int size;
{
  register int regno;
  register int mask = 0;
  int num_saved_regs = 0;
  extern char call_used_regs[];
  int fsize = (size + 3) & -4;
  

  if (frame_pointer_needed)
    {
      if (fsize == 0 && TARGET_68040)
	{
	/* on the 68040, pea + move is faster than link.w 0 */
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tpea (%s)\n\tmove.l %s,%s\n",
	       reg_names[FRAME_POINTER_REGNUM], reg_names[STACK_POINTER_REGNUM],
	       reg_names[FRAME_POINTER_REGNUM]);
#else
	  asm_fprintf (stream, "\tpea %s@\n\tmovel %s,%s\n",
	       reg_names[FRAME_POINTER_REGNUM], reg_names[STACK_POINTER_REGNUM],
	       reg_names[FRAME_POINTER_REGNUM]);
#endif
	}
      else if (fsize < 0x8000)
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tlink.w %s,%0I%d\n",
		       reg_names[FRAME_POINTER_REGNUM], -fsize);
#else
	  asm_fprintf (stream, "\tlink %s,%0I%d\n",
		       reg_names[FRAME_POINTER_REGNUM], -fsize);
#endif
	}
      else if (TARGET_68020)
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tlink.l %s,%0I%d\n",
		       reg_names[FRAME_POINTER_REGNUM], -fsize);
#else
	  asm_fprintf (stream, "\tlink %s,%0I%d\n",
		       reg_names[FRAME_POINTER_REGNUM], -fsize);
#endif
	}
      else
	{
      /* Adding negative number is faster on the 68040.  */
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tlink.w %s,%0I0\n\tadd.l %0I%d,%Rsp\n",
		       reg_names[FRAME_POINTER_REGNUM], -fsize);
#else
	  asm_fprintf (stream, "\tlink %s,%0I0\n\taddl %0I%d,%Rsp\n",
		       reg_names[FRAME_POINTER_REGNUM], -fsize);
#endif
	}
      /* Make the starting label for use by the .unwind section entry.  */
      asm_fprintf (stream, "%s%d:\n", UNWIND_START_PREFIX,
		   unwind_label_counter);
    }
  else if (fsize)
    {
      /* Adding negative number is faster on the 68040.  */
      if (fsize + 4 < 0x8000)
	{
	/* asm_fprintf() cannot handle %. */
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tadd.w %0I%d,%Rsp\n", - (fsize + 4));
#else
	  asm_fprintf (stream, "\taddw %0I%d,%Rsp\n", - (fsize + 4));
#endif
	}
      else
	{
	/* asm_fprintf() cannot handle %. */
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tadd.l %0I%d,%Rsp\n", - (fsize + 4));
#else
	  asm_fprintf (stream, "\taddl %0I%d,%Rsp\n", - (fsize + 4));
#endif
	}
    }
#ifdef SUPPORT_SUN_FPA
  for (regno = 24; regno < 56; regno++)
    if (regs_ever_live[regno] && ! call_used_regs[regno])
      {
#ifdef MOTOROLA
	asm_fprintf (stream, "\tfpmovd %s,-(%Rsp)\n",
		     reg_names[regno]);
#else
	asm_fprintf (stream, "\tfpmoved %s,%Rsp@-\n",
		     reg_names[regno]);
#endif
      }
#endif
  for (regno = 16; regno < 24; regno++)
    if (regs_ever_live[regno] && ! call_used_regs[regno])
       mask |= 1 << (regno - 16);
  if ((mask & 0xff) != 0)
    {
#ifdef MOTOROLA
      asm_fprintf (stream, "\tfmovm %0I0x%x,-(%Rsp)\n", mask & 0xff);
#else
      asm_fprintf (stream, "\tfmovem %0I0x%x,%Rsp@-\n", mask & 0xff);
#endif
    }
  mask = 0;
  for (regno = 0; regno < 16; regno++)
    if (regs_ever_live[regno] && ! call_used_regs[regno])
      {
        mask |= 1 << (15 - regno);
        num_saved_regs++;
      }
  if (frame_pointer_needed)
    {
      mask &= ~ (1 << (15 - FRAME_POINTER_REGNUM));
      num_saved_regs--;
    }

#if NEED_PROBE
  fprintf (stream, "\ttstl sp@(%d)\n", NEED_PROBE - num_saved_regs * 4);
#endif

  if (num_saved_regs <= 2)
    {
      /* Store each separately in the same order moveml uses.
         Using two movel instructions instead of a single moveml
         is about 15% faster for the 68020 and 68030 at no expense
         in code size */

      int i;

      /* Undo the work from above. */
      for (i = 0; i< 16; i++)
        if (mask & (1 << i))
          asm_fprintf (stream,
#ifdef MOTOROLA
		       "\t%Omove.l %s,-(%Rsp)\n",
#else
		       "\tmovel %s,%Rsp@-\n",
#endif
		       reg_names[15 - i]);
    }
  else if (mask)
    {
#ifdef MOTOROLA
      asm_fprintf (stream, "\tmovm.l %0I0x%x,-(%Rsp)\n", mask);
#else
      asm_fprintf (stream, "\tmoveml %0I0x%x,%Rsp@-\n", mask);
#endif
    }

  /* Apollo will always have flag_pic set.  It normally uses a partial PIC
     model where only shared library objects require a PIC reference.  */

  if (flag_pic && current_function_uses_pic_offset_table)
    {
      if (apollo_pic_all_p ())
        {
	  /* The function entry in the data area loads register A0 with
	     the table pointer.  True PIC code is required.  */

#ifdef MOTOROLA
	  asm_fprintf (stream, "\t%Omove.l %s,%s\n", reg_names[8],
		       reg_names [PIC_OFFSET_TABLE_REGNUM]);
#else
	  asm_fprintf (stream, "\tmovel %s,%s\n", reg_names[8],
		       reg_names [PIC_OFFSET_TABLE_REGNUM]);
#endif
	}
      else
        {
          /* Use a direct (non-PIC) reference to the table, since the
	     Apollo loader does not handle PC-relative references.  */

          asm_fprintf (stream, "\tlea %0U%s,%s\n",
		       GLOBAL_OFFSET_TABLE_NAME,
		       reg_names [PIC_OFFSET_TABLE_REGNUM]);
        }
    }
}



/* This generates code for function exit.  It was taken almost unchanged
   from m68k.c with code added for making the .unwind section record.  */

void
output_function_epilogue (stream, size)
     FILE *stream;
     int size;
{
  register int regno;
  register int mask, fmask, uwfmask;
  register int nregs;
  int offset, foffset, fpoffset, uwoffset;
  extern char call_used_regs[];
  int fsize = (size + 3) & -4;
  int big = 0;
  rtx insn = get_last_insn ();
  
#ifdef FUNCTION_EXTRA_EPILOGUE
  FUNCTION_EXTRA_EPILOGUE (stream, size);
#endif
  nregs = 0;  fmask = 0; fpoffset = 0; uwfmask = 0;
#ifdef SUPPORT_SUN_FPA
  for (regno = 24 ; regno < 56 ; regno++)
    if (regs_ever_live[regno] && ! call_used_regs[regno])
      nregs++;
  fpoffset = nregs * 8;
#endif
  nregs = 0;
  for (regno = 16; regno < 24; regno++)
    if (regs_ever_live[regno] && ! call_used_regs[regno])
      {
        nregs++;
	fmask |= 1 << (23 - regno);
	uwfmask |= 1 << (regno - 16);
      }
  foffset = fpoffset + nregs * 12;
  nregs = 0;  mask = 0;
  if (frame_pointer_needed)
    regs_ever_live[FRAME_POINTER_REGNUM] = 0;
  for (regno = 0; regno < 16; regno++)
    if (regs_ever_live[regno] && ! call_used_regs[regno])
      {
        nregs++;
	mask |= 1 << regno;
      }
  offset = foffset + nregs * 4;
  uwoffset = size + offset;  /* FP to register save area offset
				for .unwind section.  */

  /* If the last insn was a BARRIER, we don't have to write any code.
     The .unwind record is still needed.  This code was moved down from
     before the register mask generation.  */
  if (GET_CODE (insn) == NOTE)
    insn = prev_nonnote_insn (insn);
  if (insn && GET_CODE (insn) == BARRIER)
    {
      /* Make the ending label for use by the .unwind section record.  */
      if (frame_pointer_needed)
	asm_fprintf (stream, "%s%d:\n", UNWIND_END_PREFIX,
		     unwind_label_counter);
      /* Output just a no-op so that debuggers don't get confused
	 about which function the pc is in at this address.  */
      asm_fprintf (stream, "\tnop\n");

      /* Generate the Apollo .unwind section record.  This is described
	 in a comment at the end of this routine.  */
      if (frame_pointer_needed)
	{
	  unwind_section ();
	  output_unwind_record (stream, uwoffset, mask, uwfmask);
	  unwind_label_counter ++;
	}
      return;
    }

  if (offset + fsize >= 0x8000
      && frame_pointer_needed
      && (mask || fmask || fpoffset))
    {
#ifdef MOTOROLA
      asm_fprintf (stream, "\t%Omove.l %0I%d,%Ra0\n", -fsize);
#else
      asm_fprintf (stream, "\tmovel %0I%d,%Ra0\n", -fsize);
#endif
      fsize = 0, big = 1;
    }
  if (nregs <= 2)
    {
      /* Restore each separately in the same order moveml does.
         Using two movel instructions instead of a single moveml
         is about 15% faster for the 68020 and 68030 at no expense
         in code size. */

      int i;

      /* Undo the work from above. */
      for (i = 0; i< 16; i++)
        if (mask & (1 << i))
          {
            if (big)
	      {
#ifdef MOTOROLA
		asm_fprintf (stream, "\t%Omove.l -%d(%s,%Ra0.l),%s\n",
			     offset + fsize,
			     reg_names[FRAME_POINTER_REGNUM],
			     reg_names[i]);
#else
		asm_fprintf (stream, "\tmovel %s@(-%d,%Ra0:l),%s\n",
			     reg_names[FRAME_POINTER_REGNUM],
			     offset + fsize, reg_names[i]);
#endif
	      }
            else if (! frame_pointer_needed)
	      {
#ifdef MOTOROLA
		asm_fprintf (stream, "\t%Omove.l (%Rsp)+,%s\n",
			     reg_names[i]);
#else
		asm_fprintf (stream, "\tmovel %Rsp@+,%s\n",
			     reg_names[i]);
#endif
	      }
            else
	      {
#ifdef MOTOROLA
		asm_fprintf (stream, "\t%Omove.l -%d(%s),%s\n",
			     offset + fsize,
			     reg_names[FRAME_POINTER_REGNUM],
			     reg_names[i]);
#else
		asm_fprintf (stream, "\tmovel %s@(-%d),%s\n",
			     reg_names[FRAME_POINTER_REGNUM],
			     offset + fsize, reg_names[i]);
#endif
	      }
            offset = offset - 4;
          }
    }
  else if (mask)
    {
      if (big)
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tmovm.l -%d(%s,%Ra0.l),%0I0x%x\n",
		       offset + fsize,
		       reg_names[FRAME_POINTER_REGNUM],
		       mask);
#else
	  asm_fprintf (stream, "\tmoveml %s@(-%d,%Ra0:l),%0I0x%x\n",
		       reg_names[FRAME_POINTER_REGNUM],
		       offset + fsize, mask);
#endif
	}
      else if (! frame_pointer_needed)
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tmovm.l (%Rsp)+,%0I0x%x\n", mask);
#else
	  asm_fprintf (stream, "\tmoveml %Rsp@+,%0I0x%x\n", mask);
#endif
	}
      else
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tmovm.l -%d(%s),%0I0x%x\n",
		       offset + fsize,
		       reg_names[FRAME_POINTER_REGNUM],
		       mask);
#else
	  asm_fprintf (stream, "\tmoveml %s@(-%d),%0I0x%x\n",
		       reg_names[FRAME_POINTER_REGNUM],
		       offset + fsize, mask);
#endif
	}
    }
  if (fmask)
    {
      if (big)
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tfmovm -%d(%s,%Ra0.l),%0I0x%x\n",
		       foffset + fsize,
		       reg_names[FRAME_POINTER_REGNUM],
		       fmask);
#else
	  asm_fprintf (stream, "\tfmovem %s@(-%d,%Ra0:l),%0I0x%x\n",
		       reg_names[FRAME_POINTER_REGNUM],
		       foffset + fsize, fmask);
#endif
	}
      else if (! frame_pointer_needed)
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tfmovm (%Rsp)+,%0I0x%x\n", fmask);
#else
	  asm_fprintf (stream, "\tfmovem %Rsp@+,%0I0x%x\n", fmask);
#endif
	}
      else
	{
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tfmovm -%d(%s),%0I0x%x\n",
		       foffset + fsize,
		       reg_names[FRAME_POINTER_REGNUM],
		       fmask);
#else
	  asm_fprintf (stream, "\tfmovem %s@(-%d),%0I0x%x\n",
		       reg_names[FRAME_POINTER_REGNUM],
		       foffset + fsize, fmask);
#endif
	}
    }
  if (fpoffset != 0)
    for (regno = 55; regno >= 24; regno--)
      if (regs_ever_live[regno] && ! call_used_regs[regno])
        {
	  if (big)
	    {
#ifdef MOTOROLA
	      asm_fprintf (stream, "\tfpmovd -%d(%s,%Ra0.l), %s\n",
			   fpoffset + fsize,
			   reg_names[FRAME_POINTER_REGNUM],
			   reg_names[regno]);
#else
	      asm_fprintf (stream, "\tfpmoved %s@(-%d,%Ra0:l), %s\n",
			   reg_names[FRAME_POINTER_REGNUM],
			   fpoffset + fsize, reg_names[regno]);
#endif
	    }
	  else if (! frame_pointer_needed)
	    {
#ifdef MOTOROLA
	      asm_fprintf (stream, "\tfpmovd (%Rsp)+,%s\n",
			   reg_names[regno]);
#else
	      asm_fprintf (stream, "\tfpmoved %Rsp@+, %s\n",
			   reg_names[regno]);
#endif
	    }
	  else
	    {
#ifdef MOTOROLA
	      asm_fprintf (stream, "\tfpmovd -%d(%s), %s\n",
			   fpoffset + fsize,
			   reg_names[FRAME_POINTER_REGNUM],
			   reg_names[regno]);
#else
	      asm_fprintf (stream, "\tfpmoved %s@(-%d), %s\n",
			   reg_names[FRAME_POINTER_REGNUM],
			   fpoffset + fsize, reg_names[regno]);
#endif
	    }
	  fpoffset -= 8;
	}
  if (frame_pointer_needed)
    {
      /* Make the ending label for use by the .unwind section entry,
	 preceeded by unlk and followed by the return instruction.  */
      fprintf (stream, "\tunlk %s\n",
	       reg_names[FRAME_POINTER_REGNUM]);
      asm_fprintf (stream, "%s%d:\n", UNWIND_END_PREFIX, unwind_label_counter);
    }
  else if (fsize)
    {
      if (fsize + 4 < 0x8000)
	{
	/* asm_fprintf() cannot handle %. */
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tadd.w %0I%d,%Rsp\n", fsize + 4);
#else
	  asm_fprintf (stream, "\taddw %0I%d,%Rsp\n", fsize + 4);
#endif
	}
      else
	{
	/* asm_fprintf() cannot handle %. */
#ifdef MOTOROLA
	  asm_fprintf (stream, "\tadd.l %0I%d,%Rsp\n", fsize + 4);
#else
	  asm_fprintf (stream, "\taddl %0I%d,%Rsp\n", fsize + 4);
#endif
	}
    }
  if (current_function_pops_args)
    asm_fprintf (stream, "\trtd %0I%d\n", current_function_pops_args);
  else
    fprintf (stream, "\trts\n");

  /* Generate the Apollo .unwind section record.  Apollo CC duplicates
     this record in the .text section for functions that use floating
     point.  This duplicate is the Frame Control Block.  (FCB) Such functions
     push an FCB address or a NULL pointer before the link instruction.  For
     now, omit the FCB and the address push.  */

  if (frame_pointer_needed)
    {
      unwind_section ();
      output_unwind_record (stream, uwoffset, mask, uwfmask);
      unwind_label_counter ++;
    }
}



/* These routines are used to handle functions that are declared with the
   a0_return attribute.  apollo_set_a0_return (fundecl) marks a function
   as a0_return if it is pointer-valued.  apollo_a0_return_p (fundecl)
   returns true if fundecl is actually a0_return.  The a0_return attribute
   indicates that the function should return its value in register A0
   rather than D0.  It is used by the FUNCTION_VALUE macro.  The Apollo C
   compiler only honors a0_return declarations for pointer-valued
   functions.  This code does the same.  */

void
apollo_set_a0_return (fundecl)
     tree fundecl;
{
  /* Set the flag only for pointer-valued functions.  */
  if (fundecl   &&   TREE_CODE (TREE_TYPE (TREE_TYPE (fundecl))) == POINTER_TYPE)
    IDENTIFIER_A0_RETURN (DECL_NAME (fundecl)) = 1;
  else
    warning_with_decl (fundecl, "`a0_return' attribute ignored");
}



int
apollo_a0_return_p (fundecl)
     tree fundecl;
{
  /* Return true only for functions with the a0_return bit set.  */
  return fundecl   &&   IDENTIFIER_A0_RETURN (DECL_NAME (fundecl));
}



/* Make .mir and .sri sections at the end of a file.  Apollo ar complains
   if .mir is absent.  This .mir record contains only a maker entry that
   points to the first string in the COFF string table, usually
   "gcc2_compiled".  The maker timestamp is the approximate release date of
   the compiler in UNIX time format.  (Count of seconds since January 1, 1970
   GMT) It is not critical.
   The .sri section consists of hardware entries requiring a 68020 CPU and
   68881 FPU, and software entries that state that the normal register saving
   conventions are both followed and assumed by this code.  The systype and
   runtype were purposely omitted, since the loader can set them.  */

void
apollo_asm_file_end (file)
     FILE *file;
{
  /* Make .mir section.  */
  asm_fprintf (file, ".section .mir\n");
  asm_fprintf (file, "\t.long %d\n",  1);          /* # mir records       */
  asm_fprintf (file, "\t.short %d\n", 1);          /* maker_mir_kind      */
  asm_fprintf (file, "\t.short %d\n", 14);         /* record size         */
  asm_fprintf (file, "\t.short %d\n", 0x23C);      /* Version 2.60        */
  asm_fprintf (file, "\t.long %d\n",  0x2E7A9000); /* Maker timestamp     */
  asm_fprintf (file, "\t.long %d\n",  4);          /* String table offset */

  /* Make .sri section.  */
  asm_fprintf (file, ".section .sri\n");
  asm_fprintf (file, "\t.long %d\n",  4);       /* # sri records          */
  asm_fprintf (file, "\t.short %d\n", 1);       /* hardware_sri_kind      */
  asm_fprintf (file, "\t.short %d\n", 4);       /* take_or_rule           */
  asm_fprintf (file, "\t.long %d\n",  4);       /* m020_hardware_flag     */
  asm_fprintf (file, "\t.short %d\n", 1);       /* hardware_sri_kind      */
  asm_fprintf (file, "\t.short %d\n", 4);       /* take_or_rule           */
  asm_fprintf (file, "\t.long %d\n",  0x20000); /* m881_hardware_flag     */
  asm_fprintf (file, "\t.short %d\n", 2);       /* software_sri_kind      */
  asm_fprintf (file, "\t.short %d\n", 4);       /* take_or_rule           */
  asm_fprintf (file, "\t.long %d\n",  0x20000); /* xrs_software_flag      */
  asm_fprintf (file, "\t.short %d\n", 2);       /* software_sri_kind      */
  asm_fprintf (file, "\t.short %d\n", 4);       /* take_or_rule           */
  asm_fprintf (file, "\t.long %d\n",  0x40000); /* reg_save_software_flag */
}

#endif /* TM_APOLLO */
