/* Output dst-format symbol table information from GNU compiler.
   Copyright (C) 1987, 1988, 1992 Free Software Foundation, Inc.

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */



/* Include these first, because they may define MIN and MAX.  */
#include <stdio.h>
#include <errno.h>

#include "config.h"
#include "tree.h"
#include "c-tree.h"
#include "rtl.h"
#include "flags.h"
#include "regs.h"
#include "insn-config.h"
#include "reload.h"
#include "defaults.h"
#include "output.h" /* ASM_OUTPUT_SOURCE_LINE may refer to sdb functions.  */
typedef int boolean;
#include "dst.h"

#ifndef errno
extern int errno;
#endif

/* Nonzero means if the type has methods, only output debugging
   information if methods are actually written to the asm file.  */

static int flag_minimal_debug = 1;

/* Nonzero if we have actually used any of the GDB extensions
   to the debugging format.  The idea is that we use them for the
   first time only if there's a strong reason, but once we have done that,
   we use them whenever convenient.  */

static int have_used_extensions = 0;

char *getpwd ();

/* 1 if PARM is passed to this function in memory.  */

#define PARM_PASSED_IN_MEMORY(PARM) \
 (GET_CODE (DECL_INCOMING_RTL (PARM)) == MEM)

/* A C expression for the integer offset value of an automatic variable
   (N_LSYM) having address X (an RTX).  */
#ifndef DEBUGGER_AUTO_OFFSET
#define DEBUGGER_AUTO_OFFSET(X) \
  (GET_CODE (X) == PLUS ? INTVAL (XEXP (X, 1)) : 0)
#endif

/* A C expression for the integer offset value of an argument (N_PSYM)
   having address X (an RTX).  The nominal offset is OFFSET.  */
#ifndef DEBUGGER_ARG_OFFSET
#define DEBUGGER_ARG_OFFSET(OFFSET, X) (OFFSET)
#endif

/* Stream for writing to assembler file.  */

static FILE *asmfile;

/* Last source file name mentioned in a NOTE insn.  */

static char *lastfile;

/* Current working directory.  */

static char *cwd;

enum typestatus {TYPE_UNSEEN, TYPE_XREF, TYPE_DEFINED};

/* Vector recording the status of describing C data types.
   When we first notice a data type (a tree node),
   we assign it a number using next_type_number.
   That is its index in this vector.
   The vector element says whether we have yet output
   the definition of the type.  TYPE_XREF says we have
   output it as a cross-reference only.  */

enum typestatus *typevec;

/* Number of elements of space allocated in `typevec'.  */

static int typevec_len;

/* In dst output, each type gets a unique number.
   This is the number for the next type output.
   The number, once assigned, is in the TYPE_SYMTAB_ADDRESS field.  */

static int next_type_number = 100;

/* These variables are for dstout_symbol to communicate to
   dstout_finish_symbol.
   current_sym_code is the symbol-type-code, a symbol N_... define in stab.h.
   current_sym_value and current_sym_addr are two ways to address the
   value to store in the symtab entry.
   current_sym_addr if nonzero represents the value as an rtx.
   If that is zero, current_sym_value is used.  This is used
   when the value is an offset (such as for auto variables,
   register variables and parms).  */

static int current_sym_value;
static rtx current_sym_addr;
static char current_sym_letter;

static int next_block_number = 0;
/* Number of chars of symbol-description generated so far for the
   current symbol.  Used by CHARS and CONTIN.  */

static int current_sym_nchars;

/* Report having output N chars of the current symbol-description.  */

#define CHARS(N) (current_sym_nchars += (N))

/* Break the current symbol-description, generating a continuation,
   if it has become long.  */

#ifndef DBX_CONTIN_LENGTH
#define DBX_CONTIN_LENGTH 80
#endif

void dstout_types ();
void dstout_args ();
void dstout_symbol ();
static void dstout_type_name ();
static void dstout_type ();
static void dstout_typedefs ();
static void dst_define_type ();
void dstout_end_scope ();
static void print_int_cst_octal ();
static void print_octal ();
void	dstout_type()
{
}

/* Return the absolutized filename for the given relative
   filename.  Note that if that filename is already absolute, it may
   still be returned in a modified form because this routine also
   eliminates redundant slashes and single dots and eliminates double
   dots to get a shortest possible filename from the given input
   filename.  The absolutization of relative filenames is made by
   assuming that the given filename is to be taken as relative to
   the first argument (cwd) or to the current directory if cwd is
   NULL.  */

#if 0
static char *
abspath (rel_filename)
     char *rel_filename;
{
  /* Setup the current working directory as needed.  */
  char *abs_buffer
    = (char *) alloca (strlen (cwd) + strlen (rel_filename) + 2);
  char *endp = abs_buffer;
  char *outp, *inp;
  char *value;

  /* Copy the filename (possibly preceded by the current working
     directory name) into the absolutization buffer.  */

  {
    char *src_p;

    if (rel_filename[0] != '/')
      {
        src_p = cwd;
        while (*endp++ = *src_p++)
          continue;
        *(endp-1) = '/';        		/* overwrite null */
      }
    src_p = rel_filename;
    while (*endp++ = *src_p++)
      continue;
    if (endp[-1] == '/')
      *endp = '\0';

  /* Now make a copy of abs_buffer into abs_buffer, shortening the
     filename (by taking out slashes and dots) as we go.  */

  outp = inp = abs_buffer;
  *outp++ = *inp++;        	/* copy first slash */
  for (;;)
    {
      if (!inp[0])
        break;
      else if (inp[0] == '/' && outp[-1] == '/')
        {
          inp++;
          continue;
        }
      else if (inp[0] == '.' && outp[-1] == '/')
        {
          if (!inp[1])
                  break;
          else if (inp[1] == '/')
            {
                    inp += 2;
                    continue;
            }
          else if ((inp[1] == '.') && (inp[2] == 0 || inp[2] == '/'))
            {
                    inp += (inp[2] == '/') ? 3 : 2;
                    outp -= 2;
                    while (outp >= abs_buffer && *outp != '/')
              	outp--;
                    if (outp < abs_buffer)
                {
                  /* Catch cases like /.. where we try to backup to a
                     point above the absolute root of the logical file
                     system.  */

              	  fprintf (stderr, "%s: invalid file name: %s\n",
			   pname, rel_filename);
              	  exit (1);
              	}
                    *++outp = '\0';
                    continue;
            }
        }
      *outp++ = *inp++;
    }

  /* On exit, make sure that there is a trailing null, and make sure that
     the last character of the returned string is *not* a slash.  */

  *outp = '\0';
  if (outp[-1] == '/')
    *--outp  = '\0';

  /* Make a copy (in the heap) of the stuff left in the absolutization
     buffer and return a pointer to the copy.  */

  value = (char *) oballoc (strlen (abs_buffer) + 1);
  strcpy (value, abs_buffer);
  return value;
}
#endif

/* At the beginning of compilation, start writing the symbol table.
   Initialize `typevec' and output the standard data types of C.  */

void
dstout_init (asm_file, input_file_name, syms)
     FILE *asm_file;
     char *input_file_name;
     tree syms;
{
  char ltext_label_name[100];

  asmfile = asm_file;

  typevec_len = 100;
  typevec = (enum typestatus *) xmalloc (typevec_len * sizeof typevec[0]);
  bzero (typevec, typevec_len * sizeof typevec[0]);

  next_block_number = 2;
  next_type_number = 100;

  fprintf (asmfile, ".dstblk COMP_UNIT,1.2,\"%s\",\"%s\"\n",
		strcmp("cplusplus", lang_identify())
			? "C"
			: "C++",
		input_file_name);
  fprintf (asmfile, ".dstblk BLOCK,\"%s\",\"module\"\n",
		input_file_name);
  fprintf (asmfile, ".dstsym SCOPE,\"%s\"\n",
		input_file_name);
}
/* Output debugging info to FILE to switch to sourcefile FILENAME.  */

void
dstout_source_file (file, filename)
     FILE *file;
     char *filename;
{
  char ltext_label_name[100];

  if (filename && (lastfile == 0 || strcmp (filename, lastfile)))
    {
      fprintf (file, ".dstfil \"%s\"\n",
	       filename);
      lastfile = filename;
    }
}

/* Output a line number symbol entry into output stream FILE, 
   for source file FILENAME and line number LINENO.  */

void
dstout_source_line (file, filename, lineno)
     FILE *file;
     char *filename;
     int lineno;
{
  static int linnum = 0;

  dstout_source_file (file, filename);

  fprintf(file, ".dstlin %d,.L_LI%d\n.L_LI%d:\n", lineno, linnum, linnum);
  linnum++;
}

/* At the end of compilation, finish writing the symbol table.
   Unless you define DBX_OUTPUT_MAIN_SOURCE_FILE_END, the default is
   to do nothing. */

void
dstout_finish (file, filename)
     FILE *file;
     char *filename;
{
  dstout_end_scope();
  free(typevec);
}


/* Subroutine of `dstout_type'.  Output the type fields of TYPE.
   This must be a separate function because anonymous unions require
   recursive calls.  */

static void
dst_define_type_fields (type)
     tree type;
{
  tree tem;

  for (tem = TYPE_FIELDS (type); tem; tem = TREE_CHAIN (tem))
    dst_define_type ((TREE_CODE (tem) == FIELD_DECL
			&& DECL_BIT_FIELD_TYPE (tem))
		       ? DECL_BIT_FIELD_TYPE (tem)
		       : TREE_TYPE (tem));
}

static void
dstout_type_fields (type)
     tree type;
{
  tree tem;
  /* Output the name, type, position (in bits), size (in bits) of each
     field.  */
  for (tem = TYPE_FIELDS (type); tem; tem = TREE_CHAIN (tem))
    {
      /* For nameless subunions and subrecords, treat their fields as ours.  */
      if (DECL_NAME (tem) == NULL_TREE
	  && (TREE_CODE (TREE_TYPE (tem)) == UNION_TYPE
	      || TREE_CODE (TREE_TYPE (tem)) == RECORD_TYPE))
	dstout_type_fields (TREE_TYPE (tem));
      /* Omit here local type decls until we know how to support them.  */
      else if (TREE_CODE (tem) == TYPE_DECL)
	continue;
      /* Omit here the nameless fields that are used to skip bits.  */
      else if (DECL_NAME (tem) != 0 && TREE_CODE (tem) != CONST_DECL)
	{

	  putc(',', asmfile);

	  fprintf (asmfile, "%s:", IDENTIFIER_POINTER (DECL_NAME (tem)));
	  CHARS (2 + IDENTIFIER_LENGTH (DECL_NAME (tem)));

	  fprintf(asmfile, "%d",
		 TYPE_SYMTAB_ADDRESS( ((TREE_CODE (tem) == FIELD_DECL
			&& DECL_BIT_FIELD_TYPE (tem))
		       ? DECL_BIT_FIELD_TYPE (tem)
		       : TREE_TYPE (tem))));

	  if (TREE_CODE (tem) == VAR_DECL)
	    {
	      /* If TEM is non-static, GDB won't understand it.  */
	      fprintf (asmfile, ":0:0");
	    }
	  else if (TREE_CODE (DECL_FIELD_BITPOS (tem)) == INTEGER_CST)
	    {
	      fprintf (asmfile, ":%d:%d",
		       TREE_INT_CST_LOW (DECL_FIELD_BITPOS (tem)),
		       TREE_INT_CST_LOW (DECL_FIELD_BITPOS (tem))
		        + TREE_INT_CST_LOW (DECL_SIZE (tem)));
	    }
	  else
	    /* This has yet to be implemented.  */
	    abort ();
	  CHARS (23);
	}
    }
}

/* Subroutine of `dstout_type_methods'.  Output debug info about the
   method described DECL.  DEBUG_NAME is an encoding of the method's
   type signature.  ??? We may be able to do without DEBUG_NAME altogether
   now.  */

static void
dstout_type_method_1 (decl, debug_name)
     tree decl;
     char *debug_name;
{
  tree firstarg = TREE_VALUE (TYPE_ARG_TYPES (TREE_TYPE (decl)));
  char c1 = 'A', c2;

  if (TREE_CODE (TREE_TYPE (decl)) == FUNCTION_TYPE)
    c2 = '?';
  else /* it's a METHOD_TYPE.  */
    {
      /* A for normal functions.
	 B for `const' member functions.
	 C for `volatile' member functions.
	 D for `const volatile' member functions.  */
      if (TYPE_READONLY (TREE_TYPE (firstarg)))
	c1 += 1;
      if (TYPE_VOLATILE (TREE_TYPE (firstarg)))
	c1 += 2;

      if (DECL_VINDEX (decl))
	c2 = '*';
      else
	c2 = '.';
    }

  fprintf (asmfile, ":%s;%c%c%c", debug_name,
	   TREE_PRIVATE (decl) ? '0' : TREE_PROTECTED (decl) ? '1' : '2', c1, c2);
  CHARS (IDENTIFIER_LENGTH (DECL_ASSEMBLER_NAME (decl)) + 6
	 - (debug_name - IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl))));
  if (DECL_VINDEX (decl))
    {
      fprintf (asmfile, "%d;",
	       TREE_INT_CST_LOW (DECL_VINDEX (decl)));
      dstout_type (DECL_CONTEXT (decl), 0, 0);
      fprintf (asmfile, ";");
      CHARS (8);
    }
}

/* Subroutine of `dstout_type'.  Output debug info about the methods defined
   in TYPE.  */

static void
dstout_type_methods (type)
     register tree type;
{
#if 0 /* Is this code really necessary? */
  /* C++: put out the method names and their parameter lists */
  tree methods = TYPE_METHODS (type);
  tree type_encoding;
  register tree fndecl;
  register tree last;
  char formatted_type_identifier_length[16];
  register int type_identifier_length;

  if (methods == NULL_TREE)
    return;

  type_encoding = DECL_NAME (TYPE_NAME (type));

  /* C++: Template classes break some assumptions made by this code about
     the class names, constructor names, and encodings for assembler
     label names.  For now, disable output of dst info for them.  */
  {
    char *ptr = IDENTIFIER_POINTER (type_encoding);
    /* This should use index.  (mrs) */
    while (*ptr && *ptr != '<') ptr++;
    if (*ptr != 0)
      {
	return;
      }
  }

  type_identifier_length = IDENTIFIER_LENGTH (type_encoding);

  sprintf(formatted_type_identifier_length, "%d", type_identifier_length);

  if (TREE_CODE (methods) == FUNCTION_DECL)
    fndecl = methods;
  else if (TREE_VEC_ELT (methods, 0) != NULL_TREE)
    fndecl = TREE_VEC_ELT (methods, 0);
  else
    fndecl = TREE_VEC_ELT (methods, 1);

  while (fndecl)
    {
      tree name = DECL_NAME (fndecl);
      int need_prefix = 1;

      /* Group together all the methods for the same operation.
	 These differ in the types of the arguments.  */
      for (last = NULL_TREE;
	   fndecl && (last == NULL_TREE || DECL_NAME (fndecl) == DECL_NAME (last));
	   fndecl = TREE_CHAIN (fndecl))
	/* Output the name of the field (after overloading), as
	   well as the name of the field before overloading, along
	   with its parameter list */
	{
	  /* This is the "mangled" name of the method.
	     It encodes the argument types.  */
	  char *debug_name = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (fndecl));
	  int destructor = 0;


	  last = fndecl;

	  if (DECL_IGNORED_P (fndecl))
	    continue;

	  if (flag_minimal_debug)
	    {
	      /* Detect ordinary methods because their mangled names
		 start with the operation name.  */
	      if (!strncmp (IDENTIFIER_POINTER (name), debug_name,
			    IDENTIFIER_LENGTH (name)))
		{
		  debug_name += IDENTIFIER_LENGTH (name);
		  if (debug_name[0] == '_' && debug_name[1] == '_')
		    {
		      char *method_name = debug_name + 2;
		      char *length_ptr = formatted_type_identifier_length;
		      /* Get past const and volatile qualifiers.  */
		      while (*method_name == 'C' || *method_name == 'V')
			method_name++;
		      /* Skip digits for length of type_encoding. */
		      while (*method_name == *length_ptr && *length_ptr)
			  length_ptr++, method_name++;
		      if (! strncmp (method_name,
				     IDENTIFIER_POINTER (type_encoding),
				     type_identifier_length))
			method_name += type_identifier_length;
		      debug_name = method_name;
		    }
		}
	      /* Detect constructors by their style of name mangling.  */
	      else if (debug_name[0] == '_' && debug_name[1] == '_')
		{
		  char *ctor_name = debug_name + 2;
		  char *length_ptr = formatted_type_identifier_length;
		  while (*ctor_name == 'C' || *ctor_name == 'V')
		    ctor_name++;
		  /* Skip digits for length of type_encoding. */
		  while (*ctor_name == *length_ptr && *length_ptr)
		      length_ptr++, ctor_name++;
		  if (!strncmp (IDENTIFIER_POINTER (type_encoding), ctor_name,
				type_identifier_length))
		    debug_name = ctor_name + type_identifier_length;
		}
	      /* The other alternative is a destructor.  */
	      else
		destructor = 1;

	      /* Output the operation name just once, for the first method
		 that we output.  */
	      if (need_prefix)
		{
		  fprintf (asmfile, "%s\:\:", IDENTIFIER_POINTER (name));
		  CHARS (IDENTIFIER_LENGTH (name) + 2);
		  need_prefix = 0;
		}
	    }

	  dstout_type (TREE_TYPE (fndecl), 0, destructor);

	  dstout_type_method_1 (fndecl, debug_name);
	}
      if (!need_prefix)
	{
          putc (';', asmfile);
	  CHARS (1);
	}
    }
#endif
}

/* Output the named type, and any types it depends on.
 * if the type isn't fully defined yet, we output a forward reference
 * to it.
 */

void
dst_define_type(type)
tree type;
{
  tree tem;

  if (TYPE_SYMTAB_ADDRESS (type) != 0 
      && (TYPE_SYMTAB_ADDRESS (type) < 100
	  || !TYPE_SIZE (type)
	  || typevec[TYPE_SYMTAB_ADDRESS (type)] == TYPE_DEFINED))
    return;

  if (TYPE_MAIN_VARIANT (type) != type)
    {
       /* Variant types don't really need to be defined separately
	* for debugging purposes - output the main variant instead,
	* and set this variant's symtab address to the address of the
	* main variant. Note that we don't bother checking that its
	* been done - it's cheaper in all respects to just do the
	* recursion and assignment.
	*/
      dst_define_type (TYPE_MAIN_VARIANT (type));
      TYPE_SYMTAB_ADDRESS (type) =
	TYPE_SYMTAB_ADDRESS (TYPE_MAIN_VARIANT (type));
      return;
    }
  else if (TYPE_SYMTAB_ADDRESS (type))
    ; /* We're defining a previous forward reference. Don't change its number */
  else if (type == short_integer_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_int16_type;
  else if (type == integer_type_node)
    TYPE_SYMTAB_ADDRESS(type) = dst_int32_type;
  else if (type == long_integer_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_int32_type;
  else if (type == short_unsigned_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_uint16_type;
  else if (type == unsigned_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_uint32_type;
  else if (type == long_unsigned_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_uint32_type;
  else if (type == ptrdiff_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_uint32_type;
  else if (type == unsigned_char_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_uchar_type;
  else if (type == signed_char_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_char_type;
  else if (type == char_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_char_type;
  else if (type == float_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_real32_type;
  else if (type == double_type_node)
    TYPE_SYMTAB_ADDRESS (type) = dst_real64_type;
  else if (type == void_type_node)
    TYPE_SYMTAB_ADDRESS(type) = dst_void_type;
  else
    TYPE_SYMTAB_ADDRESS (type) = next_type_number++;

  if (typevec_len <= TYPE_SYMTAB_ADDRESS (type))
    {
      enum typestatus *new_typevec;

      new_typevec = (enum typestatus *) malloc (sizeof (enum typestatus)
						* (typevec_len + 100));
      memcpy (new_typevec, typevec, sizeof (enum typestatus) * typevec_len);
      memset (new_typevec + typevec_len, 0, sizeof(enum typestatus) * 100);
      free (typevec);
      typevec = new_typevec;
      typevec_len += 100;
    }

  if (TYPE_SIZE (type) == 0
      && (TREE_CODE (type) == RECORD_TYPE
	  || TREE_CODE (type) == UNION_TYPE
	  || TREE_CODE (type) == QUAL_UNION_TYPE
	  || TREE_CODE (type) == ENUMERAL_TYPE))
    {
      typevec[TYPE_SYMTAB_ADDRESS (type)] = TYPE_XREF;
      return;
    }
  typevec[TYPE_SYMTAB_ADDRESS (type)] = TYPE_DEFINED;
  if (TYPE_SYMTAB_ADDRESS (type) < 100)
    return;

  switch (TREE_CODE (type))
    {
    case SET_TYPE:
      /* What should we do with these? Let's just call it an integer */
      TYPE_SYMTAB_ADDRESS (type) = dst_int32_type;
      break;

    case ARRAY_TYPE:
      dst_define_type (TREE_TYPE (type));
      fprintf(asmfile, ".dstsym ARRAY,%d,%d,%d,%d\n",
			TYPE_SYMTAB_ADDRESS (type),
			0,
			TYPE_DOMAIN (type)
			 ? TREE_INT_CST_LOW (TYPE_MAX_VALUE (TYPE_DOMAIN(type)))
			 : -1,
			TYPE_SYMTAB_ADDRESS (TREE_TYPE (type)));
      CHARS (28);
      break;

    case RECORD_TYPE:
    case UNION_TYPE:
      {
	int i, n_baseclasses = 0;

	if (TYPE_BINFO (type) != 0 && TYPE_BINFO_BASETYPES (type) != 0)
	  n_baseclasses = TREE_VEC_LENGTH (TYPE_BINFO_BASETYPES (type));

	tem = size_in_bytes (type);

	for (i = 0; i < n_baseclasses; i++)
	  {
	    tree child = TREE_VEC_ELT (BINFO_BASETYPES (TYPE_BINFO (type)), i);

	    dst_define_type (BINFO_TYPE (child));
	  }
	dst_define_type_fields (type);

	/* Identify record or union, and print its size.  */
	fprintf (asmfile, ".dstsym %s,", (TREE_CODE (type) == RECORD_TYPE)
			  ? "RECORD" : "UNION");
        dstout_type_name (type);
	fprintf(asmfile, ",%d",
		TYPE_SYMTAB_ADDRESS (type));

	for (i = 0; i < n_baseclasses; i++)
	  {
	    tree child = TREE_VEC_ELT (BINFO_BASETYPES (TYPE_BINFO (type)), i);
 	    /* Print out the base class information with fields
		which have the same names at the types they hold.  */
	    putc(',', asmfile);
	    dstout_type_name (BINFO_TYPE (child));
	    fprintf(asmfile, ":%d,%d,%d",
			TYPE_SYMTAB_ADDRESS (BINFO_TYPE (child)),
			 TREE_INT_CST_LOW (BINFO_OFFSET (child)) * BITS_PER_UNIT,
			 TREE_INT_CST_LOW (DECL_SIZE (TYPE_NAME (BINFO_TYPE (child)))) * BITS_PER_UNIT);
	    CHARS (20);
	  }
      }

      CHARS (11);

      /* Write out the field declarations.  */
      dstout_type_fields (type);
      putc ('\n', asmfile);
      break;

    case ENUMERAL_TYPE:
      fprintf(asmfile, ".dstsym ENUM,%s,%d,4",
				TYPE_NAME (type),
				TYPE_SYMTAB_ADDRESS (type));
      CHARS (14);
      for (tem = TYPE_VALUES (type); tem; tem = TREE_CHAIN (tem))
	{
	  fprintf (asmfile, ",%s:%d", IDENTIFIER_POINTER (TREE_PURPOSE (tem)),
		   TREE_INT_CST_LOW (TREE_VALUE (tem)));
	  CHARS (11 + IDENTIFIER_LENGTH (TREE_PURPOSE (tem)));
	}
      putc ('\n', asmfile);
      CHARS (1);
      break;

    case POINTER_TYPE:
      CHARS (1);
      dst_define_type (TREE_TYPE (type));
      fprintf(asmfile, ".dstsym POINTER,%d,%d\n",
		TYPE_SYMTAB_ADDRESS (type),
		TYPE_SYMTAB_ADDRESS (TREE_TYPE (type)));
      break;

    case METHOD_TYPE:
#if 0
      if (use_gnu_debug_info_extensions)
	{
	  have_used_extensions = 1;
	  putc ('#', asmfile);
	  CHARS (1);
	  if (flag_minimal_debug /*&& !show_arg_types*/)
	    {
	      /* Normally, just output the return type.
		 The argument types are encoded in the method name.  */
	      putc ('#', asmfile);
	      dstout_type (TREE_TYPE (type), 0, 0);
	      putc (';', asmfile);
	      CHARS (1);
	    }
	  else
	    {
#if 0
	      /* When outputting destructors, we need to write
		 the argument types out longhand.  */
	      dstout_type (TYPE_METHOD_BASETYPE (type), 0, 0);
	      putc (',', asmfile);
	      CHARS (1);
	      dstout_type (TREE_TYPE (type), 0, 0);
	      dstout_args (TYPE_ARG_TYPES (type));
	      putc (';', asmfile);
	      CHARS (1);
#endif
	    }
	}
      else
	{
	  /* Treat it as a function type.  */
	  dstout_type (TREE_TYPE (type), 0, 0);
	}
      break;

    case OFFSET_TYPE:
      dstout_type (integer_type_node, 0, 0);
      break;
#endif

    case REFERENCE_TYPE:
      dst_define_type (TREE_TYPE (type));
      fprintf(asmfile, ".dstsym REFERENCE,%d,%d\n",
		TYPE_SYMTAB_ADDRESS (TREE_TYPE (type)),
		TYPE_SYMTAB_ADDRESS (type));
      break;

    case FUNCTION_TYPE:
      break;
#if 0
    default:
      abort ();
#endif
    }
}

/* Print the value of integer constant C, in octal,
   handling double precision.  */

static void
print_int_cst_octal (c)
     tree c;
{
  unsigned HOST_WIDE_INT high = TREE_INT_CST_HIGH (c);
  unsigned HOST_WIDE_INT low = TREE_INT_CST_LOW (c);
  int excess = (3 - (HOST_BITS_PER_WIDE_INT % 3));

  fprintf (asmfile, "0");

  if (excess == 3)
    {
      print_octal (high, HOST_BITS_PER_WIDE_INT / 3);
      print_octal (low, HOST_BITS_PER_WIDE_INT / 3);
    }
  else
    {
      unsigned HOST_WIDE_INT beg = high >> excess;
      unsigned HOST_WIDE_INT middle
	= ((high & (((HOST_WIDE_INT) 1 << excess) - 1)) << (3 - excess)
	   | (low >> (HOST_BITS_PER_WIDE_INT / 3 * 3)));
      unsigned HOST_WIDE_INT end
	= low & (((HOST_WIDE_INT) 1 << (HOST_BITS_PER_WIDE_INT / 3 * 3)) - 1);
      fprintf (asmfile, "%o%01o", beg, middle);
      print_octal (end, HOST_BITS_PER_WIDE_INT / 3);
    }
}

static void
print_octal (value, digits)
     unsigned HOST_WIDE_INT value;
     int digits;
{
  int i;

  for (i = digits - 1; i >= 0; i--)
    fprintf (asmfile, "%01o", ((value >> (3 * i)) & 7));
}

/* Output the name of type TYPE, with no punctuation.
   Such names can be set up either by typedef declarations
   or by struct, enum and union tags.  */

static void
dstout_type_name (type)
     register tree type;
{
  tree t;
  if (TYPE_NAME (type) == 0)
    return;
  if (TREE_CODE (TYPE_NAME (type)) == IDENTIFIER_NODE)
    {
      t = TYPE_NAME (type);
    }
  else if (TREE_CODE (TYPE_NAME (type)) == TYPE_DECL)
    {
      t = DECL_NAME (TYPE_NAME (type));
    }
  else
    return;

  fprintf (asmfile, "%s", IDENTIFIER_POINTER (t));
  CHARS (IDENTIFIER_LENGTH (t));
}

void
dstout_symbol (decl, local)
     tree decl;
     int local;
{
  tree type = TREE_TYPE (decl);
  tree context = NULL_TREE;
  int regno = -1;
  char *name;

  /* Ignore nameless syms, but don't ignore type tags.  */

  if (DECL_NAME (decl) == 0)
    return;
  if ((DECL_NAME (decl) == 0 && TREE_CODE (decl) != TYPE_DECL)
      || DECL_IGNORED_P (decl))
    return;

  switch (TREE_CODE (decl))
    {
    case CONST_DECL:
    case FUNCTION_DECL:
    case TYPE_DECL:
      return;
    }

  dst_define_type(type);
  if (TREE_CODE(decl) == TYPE_DECL)
    {
#if 0
      fprintf(asmfile, ".dstsym TYPEDEF,%s,%d\n",
			IDENTIFIER_POINTER (DECL_NAME (decl)),
			TYPE_SYMTAB_ADDRESS (type));
#endif
      return;
    }
      if (DECL_RTL (decl) == 0)
	return;
      /* Don't mention a variable that is external.
	 Let the file that defines it describe it.  */
      if (DECL_EXTERNAL (decl))
	return;

      /* If the variable is really a constant
	 and not written in memory, inform the debugger.  */
      if (TREE_STATIC (decl) && TREE_READONLY (decl)
	  && DECL_INITIAL (decl) != 0
	  && ! TREE_ASM_WRITTEN (decl)
	  && (DECL_FIELD_CONTEXT (decl) == NULL_TREE
	      || TREE_CODE (DECL_FIELD_CONTEXT (decl)) == BLOCK))
	{
	  if (TREE_PUBLIC (decl) == 0)
	    {
	      return;
	    }
	  /* else it is something we handle like a normal variable.  */
	}

      DECL_RTL (decl) = eliminate_regs (DECL_RTL (decl), 0, 0);

      /* Don't mention a variable at all
	 if it was completely optimized into nothingness.

	 If DECL was from an inline function, then it's rtl
	 is not identically the rtl that was used in this
	 particular compilation.  */
      if (GET_CODE (DECL_RTL (decl)) == REG)
	{
	  regno = REGNO (DECL_RTL (decl));
	  if (regno >= FIRST_PSEUDO_REGISTER)
	    return;
	}
      else if (GET_CODE (DECL_RTL (decl)) == SUBREG)
	{
	  rtx value = DECL_RTL (decl);
	  int offset = 0;
	  while (GET_CODE (value) == SUBREG)
	    {
	      offset += SUBREG_WORD (value);
	      value = SUBREG_REG (value);
	    }
	  if (GET_CODE (value) == REG)
	    {
	      regno = REGNO (value);
	      if (regno >= FIRST_PSEUDO_REGISTER)
		return;
	      regno += offset;
	    }
	  alter_subreg (DECL_RTL (decl));
	}


      if (GET_CODE (DECL_RTL (decl)) == MEM
	  && GET_CODE (XEXP (DECL_RTL (decl), 0)) == SYMBOL_REF)
	{
	  current_sym_value = 0;
	  if (TREE_PUBLIC (decl))
	    current_sym_letter = '.';
	  else
	    current_sym_letter = ':';
	  current_sym_addr = XEXP (DECL_RTL (decl), 0);
	}
      else if (regno >= 0)
	{
	  current_sym_letter = '%';
	  current_sym_value = regno;
	}
      else if (GET_CODE (DECL_RTL (decl)) == MEM
	       && (GET_CODE (XEXP (DECL_RTL (decl), 0)) == MEM
		   || (GET_CODE (XEXP (DECL_RTL (decl), 0)) == REG
		       && REGNO (XEXP (DECL_RTL (decl), 0)) != FRAME_POINTER_REGNUM)))
	/* If the value is indirect by memory or by a register
	   that isn't the frame pointer
	   then it means the object is variable-sized and address through
	   that register or stack slot.  DBX has no way to represent this
	   so all we can do is output the variable as a pointer.
	   If it's not a parameter, ignore it.
	   (VAR_DECLs like this can be made by integrate.c.)  */
	{
	  if (GET_CODE (XEXP (DECL_RTL (decl), 0)) == REG)
	    {
	      current_sym_letter = '%';
	      current_sym_value =  (REGNO (XEXP (DECL_RTL (decl), 0)));
	    }
	  else
	    {
	      /* DECL_RTL looks like (MEM (MEM (PLUS (REG...) (CONST_INT...)))).
		 We want the value of that CONST_INT.  */
	      current_sym_letter = '$';
	      current_sym_value
		= DEBUGGER_AUTO_OFFSET (XEXP (XEXP (DECL_RTL (decl), 0), 0));
	    }

	  /* Effectively do build_pointer_type, but don't cache this type,
	     since it might be temporary whereas the type it points to
	     might have been saved for inlining.  */
	  /* Don't use REFERENCE_TYPE because dst can't handle that.  */
	  type = make_node (POINTER_TYPE);
	  TREE_TYPE (type) = TREE_TYPE (decl);
	}
      else if (GET_CODE (DECL_RTL (decl)) == MEM
	       && GET_CODE (XEXP (DECL_RTL (decl), 0)) == REG)
	{
	  current_sym_value = DEBUGGER_AUTO_OFFSET (XEXP (DECL_RTL (decl), 0));
	  current_sym_letter = '%';
	}
      else if (GET_CODE (DECL_RTL (decl)) == MEM
	       && GET_CODE (XEXP (DECL_RTL (decl), 0)) == PLUS
	       && GET_CODE (XEXP (XEXP (DECL_RTL (decl), 0), 1)) == CONST_INT)
	{
	  /* DECL_RTL looks like (MEM (PLUS (REG...) (CONST_INT...)))
	     We want the value of that CONST_INT.  */
	  current_sym_value = DEBUGGER_AUTO_OFFSET (XEXP (DECL_RTL (decl), 0));
	  current_sym_letter = '$';
	}
      else if (GET_CODE (DECL_RTL (decl)) == MEM
	       && GET_CODE (XEXP (DECL_RTL (decl), 0)) == CONST)
	{
	  /* Handle an obscure case which can arise when optimizing and
	     when there are few available registers.  (This is *always*
	     the case for i386/i486 targets).  The DECL_RTL looks like
	     (MEM (CONST ...)) even though this variable is a local `auto'
	     or a local `register' variable.  In effect, what has happened
	     is that the reload pass has seen that all assignments and
	     references for one such a local variable can be replaced by
	     equivalent assignments and references to some static storage
	     variable, thereby avoiding the need for a register.  In such
	     cases we're forced to lie to debuggers and tell them that
	     this variable was itself `static'.  */
	  current_sym_letter = '.';
	  current_sym_addr = XEXP (XEXP (DECL_RTL (decl), 0), 0);
	}
      else
	/* Address might be a MEM, when DECL is a variable-sized object.
	   Or it might be const0_rtx, meaning previous passes
	   want us to ignore this variable.  */
	return;

      /* One slight hitch: if this is a VAR_DECL which is a static
	 class member, we must put out the mangled name instead of the
	 DECL_NAME.  */
  if (DECL_LANG_SPECIFIC (decl))
    name = IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl));
  else
    name = IDENTIFIER_POINTER (DECL_NAME (decl));

  if (current_sym_letter == '$' || current_sym_letter == '%')
    fprintf(asmfile,
	    ".dstsym VAR,%c%d,%s,%d\n",
	    current_sym_letter,
	    current_sym_value,
	    name,
	    TYPE_SYMTAB_ADDRESS (type));
  else
    {
      fprintf(asmfile,
	    ".dstsym VAR,%c",
	    current_sym_letter);
      output_addr_const (asmfile, current_sym_addr);
      fprintf(asmfile, ",%s,%d\n",
	    name,
	    TYPE_SYMTAB_ADDRESS (type));
    }
}

/* Output definitions of all the decls in a chain.  */

void
dstout_syms (syms)
     tree syms;
{
  while (syms)
    {
      dstout_symbol (syms, 1);
      syms = TREE_CHAIN (syms);
    }
}


/* Output everything about a symbol block (a BLOCK node
   that represents a scope level),
   including recursive output of contained blocks.

   BLOCK is the BLOCK node.
   DEPTH is its depth within containing symbol blocks.
   ARGS is usually zero; but for the outermost block of the
   body of a function, it is a chain of PARM_DECLs for the function parameters.
   We output definitions of all the register parms
   as if they were local variables of that block.

   If -g1 was used, we count blocks just the same, but output nothing
   except for the outermost block.

   Actually, BLOCK may be several blocks chained together.
   We handle them all in sequence.  */

static void
dstout_block (block, depth, args)
     register tree block;
     int depth;
     tree args;
{
  int blocknum;

  while (block)
    {
      /* Ignore blocks never expanded or otherwise marked as real.  */
      if (TREE_USED (block))
	{
	  if (depth > 0 && debug_info_level != DINFO_LEVEL_TERSE)
	    {
	      char buf[20];
	      blocknum = next_block_number++;
	      ASM_GENERATE_INTERNAL_LABEL (buf, "LBB", blocknum);

#if 0
	      if (BLOCK_HANDLER_BLOCK (block))
		{
		  /* A catch block.  Must precede N_LBRAC.  */
		  tree decl = BLOCK_VARS (block);
		  while (decl)
		    {
		      fprintf (asmfile, "%s \"%s:C1\",%d,0,0,", ASM_STABS_OP,
			       IDENTIFIER_POINTER (DECL_NAME (decl)), N_CATCH);
		      assemble_name (asmfile, buf);
		      fprintf (asmfile, "\n");
		      decl = TREE_CHAIN (decl);
		    }
		}
#endif

	      fprintf (asmfile, ".dstblk BLOCK,\"");
	      assemble_name (asmfile, buf);
	      fprintf (asmfile, "\",\"scope\"\n");
	      fprintf (asmfile, ".dstsym SCOPE,\"");
	      assemble_name (asmfile, buf);
	      fprintf (asmfile, "\"\n");
	    }
	  else if (depth > 0)
	    /* Count blocks the same way regardless of debug_info_level.  */
	    next_block_number++;

	  if (debug_info_level != DINFO_LEVEL_TERSE || depth == 0)
	    dstout_syms (BLOCK_VARS (block));
#if 0
	  if (args)
	    dstout_reg_parms (args);
#endif

	  /* Output the subblocks.  */
	  dstout_block (BLOCK_SUBBLOCKS (block), depth + 1, NULL_TREE);

	  /* Refer to the marker for the end of the block.  */
	  if (depth > 0 && debug_info_level != DINFO_LEVEL_TERSE)
	    {
	      char buf[20];
	      ASM_GENERATE_INTERNAL_LABEL (buf, "LBE", blocknum);
	      fprintf (asmfile, ".dstsym END_SCOPE,");
	      assemble_name (asmfile, buf);
	      fprintf (asmfile, "\n");
	    }
	}
      block = BLOCK_CHAIN (block);
    }
}

/* The following two functions output definitions of function parameters.
   Each parameter gets a definition locating it in the parameter list.
   Each parameter that is a register variable gets a second definition
   locating it in the register.

   Printing or argument lists in gdb uses the definitions that
   locate in the parameter list.  But reference to the variable in
   expressions uses preferentially the definition as a register.  */

/* Output definitions, referring to storage in the parmlist,
   of all the parms in PARMS, which is a chain of PARM_DECL nodes.  */
void
dstout_parms (parms)
     tree parms;
{
  for (; parms; parms = TREE_CHAIN (parms))
    if (DECL_NAME (parms) && TREE_TYPE (parms) != error_mark_node)
      {
	/* Perform any necessary register eliminations on the parameter's rtl,
	   so that the debugging output will be accurate.  */
	DECL_INCOMING_RTL (parms)
	  = eliminate_regs (DECL_INCOMING_RTL (parms), 0, NULL_RTX);
	DECL_RTL (parms) = eliminate_regs (DECL_RTL (parms), 0, NULL_RTX);

	if (PARM_PASSED_IN_MEMORY (parms))
	  {
	    rtx addr = XEXP (DECL_INCOMING_RTL (parms), 0);

	    /* ??? Here we assume that the parm address is indexed
	       off the frame pointer or arg pointer.
	       If that is not true, we produce meaningless results,
	       but do not crash.  */
	    if (GET_CODE (addr) == PLUS
		&& GET_CODE (XEXP (addr, 1)) == CONST_INT)
	      current_sym_value = INTVAL (XEXP (addr, 1));
	    else
	      current_sym_value = 0;

	    current_sym_addr = 0;

	    if (GET_CODE (DECL_RTL (parms)) == REG
		&& REGNO (DECL_RTL (parms)) >= 0
		&& REGNO (DECL_RTL (parms)) < FIRST_PSEUDO_REGISTER)
	      dstout_type (DECL_ARG_TYPE (parms), 0, 0);
	    else
	      {
		int original_value = current_sym_value;

		/* This is the case where the parm is passed as an int or double
		   and it is converted to a char, short or float and stored back
		   in the parmlist.  In this case, describe the parm
		   with the variable's declared type, and adjust the address
		   if the least significant bytes (which we are using) are not
		   the first ones.  */
#if BYTES_BIG_ENDIAN
		if (TREE_TYPE (parms) != DECL_ARG_TYPE (parms))
		  current_sym_value += (GET_MODE_SIZE (TYPE_MODE (DECL_ARG_TYPE (parms)))
					- GET_MODE_SIZE (GET_MODE (DECL_RTL (parms))));
#endif

		if (GET_CODE (DECL_RTL (parms)) == MEM
		    && GET_CODE (XEXP (DECL_RTL (parms), 0)) == PLUS
		    && GET_CODE (XEXP (XEXP (DECL_RTL (parms), 0), 1)) == CONST_INT
		    && INTVAL (XEXP (XEXP (DECL_RTL (parms), 0), 1)) == current_sym_value)
		  dst_define_type (DECL_ARG_TYPE (parms));
		else
		  {
		    current_sym_value = original_value;
		    dstout_type (DECL_ARG_TYPE (parms), 0, 0);
		  }
	      }
	    current_sym_value = DEBUGGER_ARG_OFFSET (current_sym_value, addr);
	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + IDENTIFIER_LENGTH (DECL_NAME (parms));

		fprintf (asmfile, ".dstsym VAR,$%d,%s,%d\n",
			 current_sym_value,
			 IDENTIFIER_POINTER (DECL_NAME (parms)),
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, ".dstsym VAR,$%d,(anon),%d\n",
			 current_sym_value,
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }
	  }
	else if (GET_CODE (DECL_RTL (parms)) == REG)
	  {
	    rtx best_rtl;
	    char regparm_letter;
	    /* Parm passed in registers and lives in registers or nowhere.  */

	    current_sym_addr = 0;

	    /* If parm lives in a register, use that register;
	       pretend the parm was passed there.  It would be more consistent
	       to describe the register where the parm was passed,
	       but in practice that register usually holds something else.  */
	    if (REGNO (DECL_RTL (parms)) >= 0
		&& REGNO (DECL_RTL (parms)) < FIRST_PSEUDO_REGISTER)
	      best_rtl = DECL_RTL (parms);
	    /* If the parm lives nowhere,
	       use the register where it was passed.  */
	    else
	      best_rtl = DECL_INCOMING_RTL (parms);
	    current_sym_value = REGNO (best_rtl);

	    dst_define_type (DECL_ARG_TYPE (parms));
	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + IDENTIFIER_LENGTH (DECL_NAME (parms));
		fprintf (asmfile, ".dstsym VAR,%%%d,%s,%d\n",
			 current_sym_value,
			 IDENTIFIER_POINTER (DECL_NAME (parms)),
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, ".dstsym VAR,%%%d,(anon),%d\n",
			 current_sym_value,
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }

	  }
	else if (GET_CODE (DECL_RTL (parms)) == MEM
		 && GET_CODE (XEXP (DECL_RTL (parms), 0)) == REG)
/*		 && rtx_equal_p (XEXP (DECL_RTL (parms), 0),
				 DECL_INCOMING_RTL (parms))) */
	  {
	    /* Parm was passed via invisible reference.
	       That is, its address was passed in a register.
	       Output it as if it lived in that register.
	       The debugger will know from the type
	       that it was actually passed by invisible reference.  */

	    /* Parm passed in registers and lives in registers or nowhere.  */

	    /* DECL_RTL looks like (MEM (REG...).  Get the register number.  */
	    current_sym_value = REGNO (XEXP (DECL_RTL (parms), 0));
	    current_sym_addr = 0;

	    dst_define_type (DECL_ARG_TYPE (parms));

	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + strlen (IDENTIFIER_POINTER (DECL_NAME (parms)));

		fprintf (asmfile, ".dstsym VAR,%%%d,%s,%d\n",
			 current_sym_value,
			 IDENTIFIER_POINTER (parms),
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, ".dstsym VAR,%%%d,(anon),%d\n",
			 current_sym_value,
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }
	  }
	else if (GET_CODE (DECL_RTL (parms)) == MEM
		 && XEXP (DECL_RTL (parms), 0) != const0_rtx)
	  {
	    /* Parm was passed in registers but lives on the stack.  */

	    /* DECL_RTL looks like (MEM (PLUS (REG...) (CONST_INT...))),
	       in which case we want the value of that CONST_INT,
	       or (MEM (REG ...)) or (MEM (MEM ...)),
	       in which case we use a value of zero.  */
	    if (GET_CODE (XEXP (DECL_RTL (parms), 0)) == REG
		|| GET_CODE (XEXP (DECL_RTL (parms), 0)) == MEM)
	      current_sym_value = 0;
	    else
	      current_sym_value = INTVAL (XEXP (XEXP (DECL_RTL (parms), 0), 1));
	    current_sym_addr = 0;

	    dst_define_type (DECL_ARG_TYPE (parms));

	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + strlen (IDENTIFIER_POINTER (DECL_NAME (parms)));

		fprintf (asmfile, ".dstsym VAR,%%%d,%s,%d\n",
			 current_sym_value,
			 IDENTIFIER_POINTER (DECL_NAME (parms)),
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, ".dstsym VAR,%%%d,(anon),%d\n",
			 current_sym_value,
			 TYPE_SYMTAB_ADDRESS (DECL_ARG_TYPE (parms)));
	      }

	    current_sym_value
	      = DEBUGGER_ARG_OFFSET (current_sym_value,
				     XEXP (DECL_RTL (parms), 0));
	    dstout_type (TREE_TYPE (parms), 0, 0);
	  }
      }
}

/* Output definitions for the places where parms live during the function,
   when different from where they were passed, when the parms were passed
   in memory.

   It is not useful to do this for parms passed in registers
   that live during the function in different registers, because it is
   impossible to look in the passed register for the passed value,
   so we use the within-the-function register to begin with.

   PARMS is a chain of PARM_DECL nodes.  */

#if 0
void
dstout_reg_parms (parms)
     tree parms;
{
  for (; parms; parms = TREE_CHAIN (parms))
    if (DECL_NAME (parms))
      {

	/* Report parms that live in registers during the function
	   but were passed in memory.  */
	if (GET_CODE (DECL_RTL (parms)) == REG
	    && REGNO (DECL_RTL (parms)) >= 0
	    && REGNO (DECL_RTL (parms)) < FIRST_PSEUDO_REGISTER
	    && PARM_PASSED_IN_MEMORY (parms))
	  {
	    current_sym_code = N_RSYM;
	    current_sym_value = REGNO (DECL_RTL (parms));
	    current_sym_addr = 0;

	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + IDENTIFIER_LENGTH (DECL_NAME (parms));
		fprintf (asmfile, "%s \"%s:r", ASM_STABS_OP,
			 IDENTIFIER_POINTER (DECL_NAME (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, "%s \"(anon):r", ASM_STABS_OP);
	      }
	    dstout_type (TREE_TYPE (parms), 0, 0);
	    dstout_finish_symbol (parms);
	  }
	/* Report parms that live in memory but not where they were passed.  */
	else if (GET_CODE (DECL_RTL (parms)) == MEM
		 && GET_CODE (XEXP (DECL_RTL (parms), 0)) == PLUS
		 && GET_CODE (XEXP (XEXP (DECL_RTL (parms), 0), 1)) == CONST_INT
		 && PARM_PASSED_IN_MEMORY (parms)
		 && ! rtx_equal_p (DECL_RTL (parms), DECL_INCOMING_RTL (parms)))
	  {
#if 0 /* ??? It is not clear yet what should replace this.  */
	    int offset = DECL_OFFSET (parms) / BITS_PER_UNIT;
	    /* A parm declared char is really passed as an int,
	       so it occupies the least significant bytes.
	       On a big-endian machine those are not the low-numbered ones.  */
#if BYTES_BIG_ENDIAN
	    if (offset != -1 && TREE_TYPE (parms) != DECL_ARG_TYPE (parms))
	      offset += (GET_MODE_SIZE (TYPE_MODE (DECL_ARG_TYPE (parms)))
			 - GET_MODE_SIZE (GET_MODE (DECL_RTL (parms))));
#endif
	    if (INTVAL (XEXP (XEXP (DECL_RTL (parms), 0), 1)) != offset) {...}
#endif
	    current_sym_code = N_LSYM;
	    current_sym_value = DEBUGGER_AUTO_OFFSET (XEXP (DECL_RTL (parms), 0));
	    current_sym_addr = 0;
	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + IDENTIFIER_LENGTH (DECL_NAME (parms));
		fprintf (asmfile, "%s \"%s:", ASM_STABS_OP,
			 IDENTIFIER_POINTER (DECL_NAME (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, "%s \"(anon):", ASM_STABS_OP);
	      }
	    dstout_type (TREE_TYPE (parms), 0, 0);
	    dstout_finish_symbol (parms);
	  }
#if 0
	else if (GET_CODE (DECL_RTL (parms)) == MEM
		 && GET_CODE (XEXP (DECL_RTL (parms), 0)) == REG)
	  {
	    /* Parm was passed via invisible reference.
	       That is, its address was passed in a register.
	       Output it as if it lived in that register.
	       The debugger will know from the type
	       that it was actually passed by invisible reference.  */

	    current_sym_code = N_RSYM;

	    /* DECL_RTL looks like (MEM (REG...).  Get the register number.  */
	    current_sym_value = REGNO (XEXP (DECL_RTL (parms), 0));
	    current_sym_addr = 0;

	    if (DECL_NAME (parms))
	      {
		current_sym_nchars = 2 + strlen (IDENTIFIER_POINTER (DECL_NAME (parms)));

		fprintf (asmfile, "%s \"%s:r", ASM_STABS_OP,
			 IDENTIFIER_POINTER (DECL_NAME (parms)));
	      }
	    else
	      {
		current_sym_nchars = 8;
		fprintf (asmfile, "%s \"(anon):r", ASM_STABS_OP);
	      }

	  }
#endif
      }
}
#endif


/* Output dst data for a function definition.
   This includes a definition of the function name itself (a symbol),
   definitions of the parameters (locating them in the parameter list)
   and then output the block that makes up the function's body
   (including all the auto variables of the function).  */

void
dstout_function (decl)
     tree decl;
{
   fprintf (asmfile, ".dstblk BLOCK,\"%s\",\"function\"\n",
	       IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl)));
   fprintf (asmfile, ".dstsym SCOPE,\"%s\"\n",
	       IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (decl)));
   dstout_parms (DECL_ARGUMENTS (decl));
   dstout_block (DECL_INITIAL (decl), 0, DECL_ARGUMENTS (decl));
}

void
dstout_end_scope ()
{
  static esnum = 0;

  fprintf (asmfile, ".dstsym END_SCOPE,.LES%d\n.LES%d:\n",
		esnum, esnum);
  esnum++;
}
