

#include <stdio.h>
#include <ctype.h>
#include "config.h"
#include "as.h"
#include "hash.h"
#include "obstack.h"		/* For "symbols.h" */
#include "struc-symbol.h"
#include "symbols.h"
#include "read.h"
#include "dst.h"
#include "mir.h"
#include "frags.h"
#include "obj-format.h"

#define	MAX_FILES	100

extern	char	*demand_copy_string();

struct	string_entry
{
	char	*string;
	struct	string_entry *next;
};

typedef enum
{
	rt_symbols,
	rt_lines,
	rt_blocks,
	rt_text,
	rt_data,
	rt_name_table,
	rt_label,
	rt_type_desc,
	rt_forward_type,
	rt_symbol_names
} reloc_type_t;

struct	relocation_entry
{
	reloc_type_t reloc_type;
	long	*location;
	long	blk_num;
	struct relocation_entry *next;
};

struct	generic_entry
{
	char	*data;
	int	size;
	struct	generic_entry *next;
	struct	relocation_entry *relocation_list;
};

struct	line_entry
{
	int	filenum;		/* The file number */
	long	linenum;		/* The line number */
	long	pc;			/* The PC value */
	char	*symbol;		/* The symbol name */
	struct	line_entry *next;	/* The next entry */
};

struct	line_list
{
	struct	line_entry *head;
	struct	line_entry *tail;
};

struct	sym_data
{
	long	length;
	long	base;
	char	*data;
	struct	sym_data *next;
	char	*common_var;
	struct	relocation_entry *relocation_list;
};

struct	block_entry
{
	int	num;
	dst_rec_block_t data;
	struct block_entry *next;
	struct block_entry *parent;
	struct block_entry *first_child;
	struct block_entry *next_sibling;
	long	blocks_start;
	long	lines_start;
	long	symbols_start;
	struct relocation_entry *relocation_list;
	struct relocation_entry *sym_relocs;
	struct	line_list lines;
	char	*name;
	char	*es_name;
	struct	sym_data *symdata;
};

struct	internal_type
{
	struct	sym_data *symdata;
	int	type;
	struct	internal_type *next;
};

static	struct	block_entry *blocks = NULL;
static	struct	block_entry *current_block = NULL;
static	struct	internal_type *type_list = NULL;
static	long	blocks_section_start;
static	long	lines_section_start;
static	long	symbols_section_start;

#define	TEXT_SECTION_ENTRY	1
#define	DATA_SECTION_ENTRY	2
#define	BSS_SECTION_ENTRY	3
#define	LINE_TABLE_ENTRY	4
#define	SYMBOL_TABLE_ENTRY	5
#define	BLOCK_TABLE_ENTRY	6
#define	TOTAL_SECTIONS		6

static	char	*section_names[] =
{
	".wtext", ".data", ".bss", ".lines", ".symbols", ".blocks"
};

dst_rec_section_tab_t section_table = { TOTAL_SECTIONS };
long	section_tab_vaddr;

static	struct	sym_data *locate_type(int);

static	void	skip_separator(c)
char	c;
{
	SKIP_WHITESPACE();
	if (*input_line_pointer == c)
		input_line_pointer++;
	SKIP_WHITESPACE();
}

static	void	add_section_data(void *data, long size)
{
	char	*buffer;

	if (!size)
		return;
	buffer = frag_more(size);
	if (!buffer)
		as_fatal("Couldn't write debugging tables");
	memcpy(buffer, data, size);
}

struct relocation_entry *relocation_list = NULL;

static	void	add_relocation_entry(reloc_type_t type,
				long *location,
				long blk_num,
				struct relocation_entry **relocation_list)
{
	struct relocation_entry *entry;

	entry = (struct relocation_entry *)
			malloc(sizeof(struct relocation_entry));
	entry->next = *relocation_list;
	entry->reloc_type = type;
	entry->location = location;
	entry->blk_num = blk_num;
	*relocation_list = entry;
}

static	void	do_offset_relocations(struct relocation_entry *list,
				long location)
{
	long	value;
	symbolS *symbolP;
	struct	sym_data *symdata;
	dst_type_t *type_desc;
	dst_var_loc_short_t *varloc;
	

	for (;list;list = list->next)
	{
		switch(list->reloc_type)
		{
		case rt_symbols:
		case rt_lines:
		case rt_blocks:
		case rt_text:
		case rt_data:
			value = 0;
			break;
		case rt_symbol_names:
			value = 6;
			break;
		case rt_forward_type:
			type_desc = (dst_type_t *) list->location;
			symdata = locate_type(list->blk_num);
			if (symdata)
			{
				type_desc->user_type.user_defined_type = 1;
				type_desc->user_type.doffset = symdata->base -
					location;
			}
			else
			{
				/* If we don't know by this stage, it's null */
				memset(type_desc, 0, sizeof(*type_desc));
			}
			continue;
		case rt_type_desc:
			type_desc = (dst_type_t *) list->location;
			symdata = (struct sym_data *) list->blk_num;
			type_desc->user_type.user_defined_type = 1;
			type_desc->user_type.doffset = symdata->base -
							location;
			continue;
		case rt_label:
			symdata = (struct sym_data *) list->blk_num;
			symbolP = symbol_find((char *) symdata->common_var);
			varloc = (dst_var_loc_short_t *) list->location;
			if (!symbolP)
			{
				varloc->loc_type = dst_var_loc_abs;
				varloc->loc_index = 0;
				varloc->location = 0;
				free(symdata->common_var);
				symdata->common_var = NULL;
			}
			else if (S_GET_SEGMENT (symbolP) == SEG_DATA)
			{
				varloc->loc_index = DATA_SECTION_ENTRY;
				varloc->location = S_GET_VALUE(symbolP)+
					symbolP->sy_frag->fr_address;
				free(symdata->common_var);
				symdata->common_var = NULL;
			}
			else
			{
				varloc->loc_type = dst_var_loc_abs;
				varloc->loc_index = 0;
				varloc->location = 0;
			}
			continue;
		case rt_name_table:
			value = blocks_section_start +
				sizeof(dst_rec_comp_unit_t) + 8;
			break;
		}
		*list->location += value - location;
	}
}

static	char	*lookup_string(struct string_entry *list, int number)
{
	while(number--)
		list = list->next;
	return list->string;
}

static	int	locate_string(struct string_entry **list, char *string, long *position)
{
	struct string_entry	**entry;
	long	i = 0;
	long	size = 0;

	for (entry = list; *entry; entry = &(*entry)->next)
	{
		if (!strcmp((*entry)->string, string))
		{
			if (position)
			{
				*position = size;
			}
			return i;
		}
		size += strlen((*entry)->string) + 1;
		i++;
	}
	*entry = (struct string_entry *) malloc(sizeof(struct string_entry));
	(*entry)->string = (char *) malloc(strlen(string) + 1);
	strcpy((*entry)->string, string);
	(*entry)->next = NULL;
	if (position)
		*position = size;
	return i++;
}

static	long	size_strings(struct string_entry *list, int make_even)
{
	long	size = 0;

	while (list)
	{
		size += strlen(list->string) + 1;
		list = list->next;
	}
	if (make_even && (size & 1))
		size++;
	return size;
}

static	long	count_strings(struct string_entry *list)
{
	long	count;

	for (count = 0; list; list = list->next, count++);
	return count;
}


static	struct	string_entry *file_table = NULL;
static	struct	string_entry *name_table = NULL;
static	struct	string_entry *symbol_strings = NULL;

static	long	current_line;
static	long	current_file;

static	dst_rec_comp_unit_t	comp_unit_info;
static	dst_rec_file_tab_t	file_table_info;

static	void	skip_to_next_field(void)
{
	while (*input_line_pointer != '\n' && *input_line_pointer != ',')
		input_line_pointer++;
	skip_separator(',');
	while (*input_line_pointer == ' ' || *input_line_pointer == '\t')
		input_line_pointer++;
}

static	char	*copy_until(c)
char c;
{
	char	*cp;
	int	len;
	char	*rslt;

	cp = input_line_pointer;
	while (*input_line_pointer != '\n' && *input_line_pointer != c)
		input_line_pointer++;
	len = input_line_pointer - cp;
	rslt = (char *) malloc(len + 1);
	strncpy(rslt, cp, len);
	rslt[len] = '\0';
	return rslt;
}

#define	copy_rest_of_field() copy_until(',');
#define	copy_rest_of_bit() copy_until(':');

static	int	debugging_on = 0;

void	create_target_segments(void)
{
	subseg_new(".mir", 0);
}

static	void	build_comp_unit(void)
{
	int	major, minor;
	char	*language;
	char	*file;
	int	lang_len = 100;
	int	file_len = 512;

	debugging_on = 1;
	/* Create all the sections we need */

	bzero(&comp_unit_info, sizeof(comp_unit_info));

	sscanf(input_line_pointer, "%d.%d", &major, &minor);
	comp_unit_info.version.major_part = major;
	comp_unit_info.version.minor_part = minor;

	skip_to_next_field();
	language = demand_copy_string(&lang_len);
	if (!strcmp(language, "fortran"))
		comp_unit_info.lang_type = dst_lang_ftn;
	else if (!strcmp(language, "C"))
		comp_unit_info.lang_type = dst_lang_c;
	else if (!strcmp(language, "Pascal"))
		comp_unit_info.lang_type = dst_lang_pas;
	else if (!strcmp(language, "Modula-2"))
		comp_unit_info.lang_type = dst_lang_mod2;
	else if (!strcmp(language, "ADA"))
		comp_unit_info.lang_type = dst_lang_ada;
	else if (!strcmp(language, "C++"))
		comp_unit_info.lang_type = dst_lang_cxx;
	else
		comp_unit_info.lang_type = dst_lang_unk;
	skip_to_next_field();
	file = demand_copy_string(&file_len);
	locate_string(&file_table, file, NULL);
	free(language);
	free(file);
}

static	void	build_block(void)
{
	char	*block_name;
	int	block_len = 512;
	char	*type_name;
	int	type_len = 512;
	long	name_location;
	struct	block_entry *block;
	struct	block_entry *back_link;

	block_name = demand_copy_string(&block_len);
	skip_to_next_field();
	type_name = demand_copy_string(&type_len);
	block = (struct block_entry *) malloc(sizeof(struct block_entry));
	bzero((char *) block, sizeof(struct block_entry));
	if (block_name[0] != '.' || block_name[1] != 'L')
		block->name = lookup_string(name_table,
			locate_string(&name_table, block_name, &name_location));
	else
	{
		block->name = block_name;
		name_location = 0;
	}
	block->parent = current_block;
	if (block->parent)
	{
		if (block->parent->first_child)
		{
			for (back_link = block->parent->first_child;
			     back_link->next_sibling;
			     back_link = back_link->next_sibling);
			back_link->next_sibling = block;
		}
		else
			block->parent->first_child = block;
	}
	if (current_block)
		block->num = current_block->num + 1;
	else
		block->num = 0;
	block->next = blocks;
	if (!strcmp(type_name, "module"))
	{
		block->data.block_type = dst_block_module;
	}
	else if (!strcmp(type_name, "function"))
	{
		block->data.block_type = dst_block_function;
		block->data.n_of_code_ranges = 1;
		block->data.flags = 1<<dst_block_executable;
		block->data.code_ranges[0].code_start.sect_index = 
					TEXT_SECTION_ENTRY;
		block->data.code_ranges[0].code_size = 0;
	}
	else if (!strcmp(type_name, "scope"))
	{
		block->data.block_type = dst_block_block_data;
		block->data.n_of_code_ranges = 1;
		block->data.flags = 1>>dst_block_executable;
		block->data.code_ranges[0].code_start.sect_index =
					TEXT_SECTION_ENTRY;
		block->data.code_ranges[0].code_size = 0;
	}

	block->relocation_list = NULL;
	block->data.noffset = name_location;
	if (name_location)
		add_relocation_entry(rt_name_table, &block->data.noffset,
			     name_location, &block->relocation_list);
	block->lines.head = NULL;
	block->lines.tail = NULL;
	block->es_name = NULL;
	block->symdata = NULL;
	block->sym_relocs = NULL;
	blocks = block;
	current_block = block;
	if (name_location)
		free(block_name);
	free(type_name);
}

static	struct sym_data	*add_sym_data(struct block_entry *block, char *data, long size)
{
	struct	sym_data **sym_list;
	struct	sym_data *new_sym;
	int	areasize;

	if (size&1)
		areasize = size+1;
	else
		areasize = size;
	for (sym_list = &block->symdata; *sym_list;
			sym_list = &(*sym_list)->next);
	*sym_list = new_sym = (struct sym_data *)
				malloc(sizeof(struct sym_data));
	new_sym->data = (char *) malloc(areasize);
	new_sym->length = areasize;
	new_sym->base = 0;
	new_sym->next = NULL;
	new_sym->common_var = NULL;
	new_sym->relocation_list = NULL;
	memcpy(new_sym->data, data, size);
	return new_sym;
};

static	void	end_scope(void)
{
	dst_rec_t data;

	if (current_block)
	{
		data.rec_type = dst_typ_end_scope;
		data.rec_flags = 0;
		add_sym_data(current_block, (char *) &data, 2);
		current_block->es_name = copy_rest_of_field();
		current_block = current_block->parent;
	}
}

static	void	begin_scope(void)
{
	dst_rec_t	data;
	char	*name;
	int	string_len = 256;
	long	name_location;
	char	*buffer;
	struct	sym_data *symdata;

	if (current_block)
	{
		name = demand_copy_string(&string_len);
		data.rec_type = dst_typ_scope;
		data.rec_flags = 0;
		DST_scope(&data).start_line.file_index =
		    DST_scope(&data).end_line.file_index = 0;
		if (name[0] != '.' || name[1] != 'L')
			locate_string(&symbol_strings, name, &name_location);
		else
			name_location = 0;
		DST_scope(&data).noffset = name_location;
		symdata = add_sym_data(current_block, (char *) &data,
					2 + sizeof(dst_rec_scope_t));
		if (name_location)
		{
			buffer = symdata->data;
			add_relocation_entry(rt_symbol_names,
				(long *) (buffer + ((char *)
				&DST_scope(&data).noffset -
				  (char *) &data)),
				name_location, &symdata->relocation_list);
			free(name);
		}
	}
}

static	void	fix_type_desc(reloc_list, type_desc, type)
struct relocation_entry **reloc_list;
dst_type_t *type_desc;
int type;
{
	struct internal_type *type_entry;

	if (type >= 100)
	{
		for (type_entry = type_list; type_entry; type_entry =
			type_entry->next)
		{
			if (type_entry->type == type)
				break;
		}
		type_desc->user_type.user_defined_type = 1;
		if (type_entry)
			add_relocation_entry(rt_type_desc,
					(long *) type_desc,
					(long) type_entry->symdata,
					reloc_list);
		else /* It's a forward reference */
			add_relocation_entry(rt_forward_type,
					(long *) type_desc,
					type,
					reloc_list);
	}
	else
	{
		type_desc->std_type.user_defined_type = 0;
		type_desc->std_type.must_be_zero = 0;
		type_desc->std_type.dtc = type;
	}
}

static	struct	sym_data *locate_type(type)
int type;
{
	struct internal_type *type_entry;

	for (type_entry = type_list; type_entry; type_entry =
		type_entry->next)
	{
		if (type_entry->type == type)
			break;
	}
	if (type_entry)
		return type_entry->symdata;
	return NULL;
}

static	void	add_type_desc(symdata, type)
struct sym_data *symdata;
int type;
{
	struct internal_type *new_type;

	new_type = (struct internal_type *)
			malloc(sizeof(struct internal_type));
	new_type->symdata = symdata;
	new_type->type = type;
	new_type->next = type_list;
	type_list = new_type;
}

static	void	pointer_reference_entry(rec_type)
int	rec_type;
{
	dst_rec_t data;
	struct	sym_data *symdata;
	char	*buffer;
	int	type;
	int	new_type;

	new_type = get_absolute_expression();
	skip_separator(',');
	type = get_absolute_expression();
	data.rec_type = rec_type;
	data.rec_flags = 0;
	DST_pointer(&data).noffset = 0;
	DST_pointer(&data).src_loc.file_index = 0;
	DST_pointer(&data).src_loc.line_number = 0;
	symdata = add_sym_data(current_block, (char *) &data,
				2 + sizeof(dst_rec_pointer_t));
	buffer = symdata->data;
	fix_type_desc(&symdata->relocation_list,
		      (dst_type_t *) (buffer +
			((char *) &DST_pointer(&data).type_desc -
			 (char *) &data)),
			type);
	add_type_desc(symdata, new_type);
}

static	void	pointer_entry(void)
{
	pointer_reference_entry(dst_typ_pointer);
}

static	void	reference_entry(void)
{
	pointer_reference_entry(dst_typ_reference);
}

static	void	array_entry(void)
{
	dst_rec_t data;
	struct	sym_data *symdata;
	char	*buffer;
	int	type;
	int	new_type;
	int	low_index;
	int	high_index;

	new_type = get_absolute_expression();
	skip_separator(',');
	low_index = get_absolute_expression();
	skip_separator(',');
	high_index = get_absolute_expression();
	skip_separator(',');
	type = get_absolute_expression();
	data.rec_type = dst_typ_array;
	data.rec_flags = 0;
	DST_array(&data).noffset = 0;
	DST_array(&data).src_loc.file_index = 0;
	DST_array(&data).src_loc.line_number = 0;
	DST_array(&data).indx_type_desc.std_type.user_defined_type = 0;
	DST_array(&data).indx_type_desc.std_type.must_be_zero = 0;
	DST_array(&data).indx_type_desc.std_type.dtc = dst_int32_type;
	DST_array(&data).lo_bound = low_index;
	DST_array(&data).hi_bound = high_index;
	DST_array(&data).span = 1;
	DST_array(&data).size = high_index - low_index + 1;
	DST_array(&data).span_comp = 0;
	DST_array(&data).multi_dim = 0;
	DST_array(&data).is_packed = 0;
	DST_array(&data).column_major = 0;
	DST_array(&data).reserved = 0;
	symdata = add_sym_data(current_block, (char *) &data,
				2 + sizeof(dst_rec_array_t));
	buffer = symdata->data;
	fix_type_desc(&symdata->relocation_list,
		      (dst_type_t *) (buffer +
			((char *) &DST_array(&data).elem_type_desc -
			 (char *) &data)),
			type);
	add_type_desc(symdata, new_type);
}

struct	member_list
{
	char	*name;
	int	type;
	int	startbit;
	int	endbit;
	struct	member_list *next;
};

static	void	record_union_entry(rec_type)
int	rec_type;
{
	dst_rec_t *data;
	struct	sym_data *symdata;
	char	*buffer;
	int	new_type;
	char	*name;
	int	size = 512;
	struct	member_list *list = NULL, *element, *last_element;
	int	members = 0;
	long	name_location;
	int	largest_bit = 0;
	int	i;

	name = copy_rest_of_field();
	skip_separator(',');
	new_type = get_absolute_expression();
	skip_separator(',');
	while (*input_line_pointer != '\n')
	{
		element = (struct member_list *)
				malloc(sizeof(struct member_list));
		size = 512;
		element->name = copy_rest_of_bit();
		skip_separator(':');
		element->type = get_absolute_expression();
		skip_separator(':');
		element->startbit = get_absolute_expression();
		skip_separator(':');
		element->endbit = get_absolute_expression();
		skip_separator(',');
		element->next = NULL;
		if (list)
			last_element->next = element;
		else
			list = element;
		members++;
		if (element->endbit > largest_bit)
			largest_bit = element->endbit;
		last_element = element;
	}
	symdata = add_sym_data(current_block, (char *) &data,
				2 + sizeof(dst_rec_record_t) +
					(members - 100) *
					sizeof(dst_field_t));
	buffer = symdata->data;
	data = (dst_rec_ptr_t) buffer;
	locate_string(&symbol_strings, name, &name_location);
	data->rec_type = rec_type;
	data->rec_flags = 0;
	DST_record(data).noffset = name_location;
	add_relocation_entry(rt_symbol_names,
		(long *) &DST_record(data).noffset,
		name_location, &symdata->relocation_list);
	DST_record(data).src_loc.file_index = 0;
	DST_record(data).src_loc.line_number = 0;
	DST_record(data).size = largest_bit / 8 + (largest_bit%8?1:0);
	DST_record(data).nfields = members;
	for (element = list, i = 0; element; element = element->next, i++)
	{
		locate_string(&symbol_strings, element->name, &name_location);
		DST_record(data).f.fields[i].noffset = name_location;
		add_relocation_entry(rt_symbol_names,
				(long *) &DST_record(data).f.fields[i].noffset,
				name_location, &symdata->relocation_list);
		fix_type_desc(&symdata->relocation_list,
			     (long *) &DST_record(data).f.fields[i].type_desc,
				element->type);
		if ((element->endbit - element->startbit) % 8 ||
		    element->startbit % 8)
		{
			DST_record(data).f.fields[i].f.field_bit.format_tag =
								dst_field_bit;
			DST_record(data).f.fields[i].f.field_bit.nbits =
					element->endbit - element->startbit;
			DST_record(data).f.fields[i].f.field_bit.is_signed = 0;
			DST_record(data).f.fields[i].f.field_bit.bit_offset = 
						element->startbit % 8;
			DST_record(data).f.fields[i].f.field_bit.byte_offset = 
						element->startbit / 8;
			DST_record(data).f.fields[i].f.field_bit.pad = 0;
		}
		else
		{
			DST_record(data).f.fields[i].f.field_byte.format_tag =
								dst_field_byte;
			DST_record(data).f.fields[i].f.field_byte.offset =
						element->startbit / 8;
		}
	}
	add_type_desc(symdata, new_type);
	free(name);
	for (element = list; element; element = list)
	{
		list = element->next;
		free(element->name);
		free(element);
	}
}

static	void	record_entry(void)
{
	record_union_entry(dst_typ_record);
}

static	void	union_entry(void)
{
	record_union_entry(dst_typ_union);
}

struct	value_list
{
	char	*name;
	int	value;
	struct	value_list *next;
};

static	void	enum_entry(void)
{
	dst_rec_t *data;
	struct	sym_data *symdata;
	char	*buffer;
	int	new_type;
	char	*name;
	int	size = 512;
	struct	value_list *list = NULL, *element, *last_element;
	int	members = 0;
	long	name_location;
	int	i;
	int	len;

	name = copy_rest_of_field();
	skip_separator(',');
	new_type = get_absolute_expression();
	skip_separator(',');
	len = get_absolute_expression();
	skip_separator(',');
	while (*input_line_pointer != '\n')
	{
		element = (struct value_list *)
				malloc(sizeof(struct value_list));
		size = 512;
		element->name = copy_rest_of_bit();
		skip_separator(':');
		element->value = get_absolute_expression();
		skip_separator(',');
		element->next = NULL;
		if (list)
			last_element->next = element;
		else
			list = element;
		members++;
		last_element = element;
	}
	symdata = add_sym_data(current_block, (char *) &data,
				2 + sizeof(dst_rec_explicit_enum_t) +
					(members - 100) *
					sizeof(dst_enum_elem_t));
	buffer = symdata->data;
	data = (dst_rec_ptr_t) buffer;
	locate_string(&symbol_strings, name, &name_location);
	data->rec_type = dst_typ_explicit_enum;
	data->rec_flags = 0;
	DST_explicit_enum(data).noffset = name_location;
	add_relocation_entry(rt_symbol_names,
		(long *) &DST_record(data).noffset,
		name_location, &symdata->relocation_list);
	DST_explicit_enum(data).src_loc.file_index = 0;
	DST_explicit_enum(data).src_loc.line_number = 0;
	DST_explicit_enum(data).size = len;
	DST_explicit_enum(data).nelems = members;
	for (element = list, i = 0; element; element = element->next, i++)
	{
		locate_string(&symbol_strings, element->name, &name_location);
		DST_explicit_enum(data).elems[i].noffset = name_location;
		add_relocation_entry(rt_symbol_names,
			(long *) &DST_explicit_enum(data).elems[i].noffset,
				name_location, &symdata->relocation_list);
		DST_explicit_enum(data).elems[i].value = element->value;
	}
	add_type_desc(symdata, new_type);
	free(name);
	for (element = list; element; element = list)
	{
		list = element->next;
		free(element->name);
		free(element);
	}
}

static	void	var_entry(void)
{
	dst_rec_t data;
	int	stack_off = 0;
	int	reg_num;
	int	type_num;
	char	*name;
	char	*asm_name;
	long	name_location;
	struct	sym_data *symdata;
	char	*buffer;
	int	is_stack = 0;
	int	is_absolute = 0;
	int	is_static = 0;
	int	is_reg;

	if (*input_line_pointer == '$')
	{
		input_line_pointer++;
		if (*input_line_pointer == '+')
			input_line_pointer++; /* g_a_e is braindead */
		stack_off = get_absolute_expression();
		is_stack = 1;
	}
	else if (*input_line_pointer == '.')
	{
		is_absolute = 1;
		input_line_pointer++;
		asm_name = copy_rest_of_field();
	}
	else if (*input_line_pointer == ':')
	{
		is_static = is_absolute = 1;
		input_line_pointer++;
		asm_name = copy_rest_of_field();
	}
	else if (*input_line_pointer == '%')
	{
		is_reg = 1;
		input_line_pointer++;
		if (*input_line_pointer == 'd')
		{
			input_line_pointer++;
			reg_num = *input_line_pointer++ - '0';
		}
		else if (*input_line_pointer == 'a')
		{
			input_line_pointer++;
			reg_num = *input_line_pointer++ - '0' + 8;
		}
		else
		{
			reg_num = get_absolute_expression();
		}
	}
	else
		return;

	if (*input_line_pointer == ',')
		input_line_pointer++;
	name = copy_rest_of_field();
	if (*input_line_pointer == ',')
		input_line_pointer++;
	type_num = get_absolute_expression();
	locate_string(&symbol_strings, name, &name_location);
	data.rec_type = dst_typ_var;
	data.rec_flags = 0;
	DST_var(&data).noffset = name_location;
	DST_var(&data).src_loc.file_index = 0;
	DST_var(&data).src_loc.line_number = 0;
	DST_var(&data).attributes = 0;
	DST_var(&data).no_of_locs = 1;
	DST_var(&data).short_locs = 1;
	if (is_stack)
	{
		DST_var(&data).locs.shorts[0].loc_type = dst_var_loc_reg_rel;
		DST_var(&data).locs.shorts[0].loc_index = 14;
		DST_var(&data).locs.shorts[0].location = stack_off;
		DST_var(&data).attributes = 0;
	}
	else if (is_absolute)
	{
		DST_var(&data).locs.shorts[0].loc_type = dst_var_loc_sect_off;
		switch(now_seg)
		{
		case SEG_DATA:
			DST_var(&data).locs.shorts[0].loc_index =
							DATA_SECTION_ENTRY;
			break;
		default:
			DST_var(&data).locs.shorts[0].loc_index =
							BSS_SECTION_ENTRY;
			break;
		}
		if (!is_static)
			DST_var(&data).attributes = (1<<dst_var_attr_global);
		DST_var(&data).attributes |= (1<<dst_var_attr_static);
	}
	else if (is_reg)
	{
		DST_var(&data).locs.shorts[0].loc_type = dst_var_loc_reg;
		DST_var(&data).locs.shorts[0].loc_index = reg_num;
		DST_var(&data).locs.shorts[0].location = 0;
	}

	DST_var(&data).locs.shorts[0].start_line =
		DST_var(&data).locs.shorts[0].end_line = 0;

	symdata = add_sym_data(current_block, (char *) &data,
				2 + sizeof(dst_rec_var_t) -
				dst_dummy_array_size *
				sizeof(dst_var_loc_long_t) +
				sizeof(dst_var_loc_short_t));
	buffer = symdata->data;
	fix_type_desc(&symdata->relocation_list,
		      (dst_type_t *) (buffer +
			((char *) &DST_var(&data).type_desc -
			 (char *) &data)),
			type_num);
	add_relocation_entry(rt_symbol_names,
		(long *) (buffer + ((char *) &DST_var(&data).noffset -
			  (char *) &data)),
		name_location, &symdata->relocation_list);
	if (is_absolute)
	{
		symdata->common_var = asm_name;
		add_relocation_entry(rt_label,
			(long *) (buffer +
			((char *) &DST_var(&data).locs.shorts[0] -
			 (char *) &data)),
		(long) symdata, &symdata->relocation_list);
	}
	free(name);
	while (*input_line_pointer != '\n')
		input_line_pointer++;
}

void	dst_block(void)
{
	char	entry_type[20];
	char	*c;

	SKIP_WHITESPACE();
	c = entry_type;
	while (*input_line_pointer != ',' && *input_line_pointer != '\n')
		*c++ = *input_line_pointer++;
	*c = '\0';
	skip_to_next_field();
	SKIP_WHITESPACE();
	if (!strcmp(entry_type, "COMP_UNIT"))
		build_comp_unit();
	else if (!strcmp(entry_type, "BLOCK"))
		build_block();
	while ( *input_line_pointer != '\n' )
		input_line_pointer++;
}

void	dst_file(void)
{
	char	*filename;
	int	string_length = 512;

	filename = demand_copy_string(&string_length);
	current_file = locate_string(&file_table, filename, NULL);
	free(filename);
}

void	dst_line(void)
{
	struct	line_entry *entry;
	char	*label;

	current_line = get_absolute_expression();
	if (*input_line_pointer == ',')
		input_line_pointer++;
	else
	{
		as_warn("Expected comma after line number");
		while (*input_line_pointer != '\n')
			input_line_pointer++;
		return;
	}
	if (current_block)
	{
		entry = (struct line_entry *) malloc(sizeof(struct line_entry));
		entry->linenum = current_line;
		entry->filenum = current_file;
		label = copy_rest_of_field();
		entry->symbol = label;
		entry->next = NULL;
		if (current_block->lines.tail)
		{
			current_block->lines.tail->next = entry;
			current_block->lines.tail = entry;
		}
		else
		{
			current_block->lines.head =
				current_block->lines.tail = entry;
		}
	}
	
	while ( *input_line_pointer != '\n' )
		input_line_pointer++;
}


struct
{
	char	*keyword;
	void	(*func)();
} dst_symbol_keywords[] =
{
	{	"ARRAY",	array_entry	},
	{	"END_SCOPE",	end_scope	},
	{	"ENUM",		enum_entry	},
	{	"POINTER",	pointer_entry   },
	{	"REFERENCE",	reference_entry },
	{	"SCOPE",	begin_scope	},
	{	"RECORD",	record_entry	},
	{	"UNION",	union_entry	},
	{	"VAR",		var_entry	},
	{	NULL,		NULL		}
};


void	dst_symbol(void)
{
	char	entry_type[20];
	int	i;
	char	*c;

	SKIP_WHITESPACE();
	c = entry_type;
	while (*input_line_pointer != ',' && *input_line_pointer != '\n')
		*c++ = *input_line_pointer++;
	*c = '\0';
	skip_to_next_field();
	SKIP_WHITESPACE();
	for (i = 0; dst_symbol_keywords[i].keyword; i++)
	{
		if (!strcmp(entry_type, dst_symbol_keywords[i].keyword))
		{
			(*dst_symbol_keywords[i].func)();
			return;
		}
	}
	as_warn("Unkown debugging keyword: %s", entry_type);
	demand_empty_rest_of_line();
}

dst_rec_t	dst_record;

static	void write_dst(char *data, int size, char type, char flags)
{
	char	zero = 0;

	add_section_data(&type, 1);
	add_section_data(&flags, 1);
	add_section_data(data, size);
	if (size & 1)
		add_section_data(&zero, 1);
}

static	long	write_string_table(struct string_entry *strings,
			   char type)
{
	char	zero = 0;
	long	size;

	add_section_data(&type, 1);
	add_section_data(&zero, 1);
	size = size_strings(strings, 0);
	add_section_data(&size, 4);
	while (strings)
	{
		add_section_data(strings->string, strlen(strings->string)+1);
		strings = strings->next;
	}
	if (size&1)
	{
		add_section_data(&zero, 1);
		size++;
	}
	return size + 6;
}

static	void	reorder_dst_blocks(void)
{
	long size;
	struct block_entry *block, *next_block, *prev_block;

	next_block = blocks;
	prev_block = block = NULL;

	size = 0;
	while (next_block)
	{
		block = next_block;
		next_block = block->next;
		block->next = prev_block;
		prev_block = block;
		size++;
	}
	blocks = block;
	comp_unit_info.number_of_blocks = size;
}

int	build_dst_lines_section()
{
	struct	block_entry *block;
	struct	line_entry *line;
	symbolS *symbolP, *procP, *endP;
	dst_ln_entry_t entry;
	long	size = 0;
	long	last_pc, last_ln;
	long	pc_delta;
	long	ln_delta;
	dst_src_loc_t source_loc;
	int	last_file;
	unsigned char	dpc1;
	signed char	dln1;
	unsigned short	dpc2;
	short		dln2;
	unsigned long	dpc4;
	unsigned long	dln4;
	char	entry_number = '\0';

	if (!debugging_on)
		return;
	reorder_dst_blocks();

	subseg_new(".lines", 0);
	if (!debugging_on)
		return;
	frag_now->fr_type = rs_machine_dependent;
	symbolP = symbol_new(".L__lines", now_seg, 0, &zero_address_frag);
	symbol_table_insert(symbolP);
	for (block = blocks; block; block = block->next)
	{
		procP = symbol_find(block->name);
		if (block->es_name)
			endP = symbol_find(block->es_name);
		else
			endP = 0;
		if (!procP)
			continue;
		last_pc = S_GET_VALUE(procP) + procP->sy_frag->fr_address;
		if (endP)
			block->data.code_ranges[0].code_size =
				S_GET_VALUE(endP)+ endP->sy_frag->fr_address -
				last_pc;
		block->data.code_ranges[0].lines_start.sect_index =
				LINE_TABLE_ENTRY;
		block->data.code_ranges[0].lines_start.sect_offset =
				size;
		block->data.code_ranges[0].code_start.sect_offset = last_pc;
		block->data.symbols_start.sect_offset = 0;
		block->data.symbols_start.sect_index = 0;
		entry.esc.esc_flag = -8;
		entry.esc.esc_code = dst_ln_file;
		add_section_data(&entry, 1);
		source_loc.reserved = 0;
		if (block->lines.head)
		{
			last_ln = block->lines.head->linenum;
			last_file = block->lines.head->filenum;
			source_loc.file_index = last_file + 1;
			source_loc.line_number = last_ln;
		}
		else
		{
			source_loc.file_index = 0;
			source_loc.line_number = 0;
			last_ln = last_file = 0;
		}
		add_section_data(&source_loc, sizeof(source_loc));
		entry.esc.esc_code = dst_ln_entry;
		add_section_data(&entry, 1);
		add_section_data(&entry_number, 1);
		size += 3 + sizeof(source_loc);
		for (line = block->lines.head; line; line = line->next)
		{
			if (line->filenum != last_file)
			{
				last_file = line->filenum;
				last_ln = line->linenum;
				source_loc.file_index = last_file + 1;
				source_loc.line_number = last_ln;
				add_section_data(&source_loc, sizeof(source_loc));
				size += sizeof(source_loc);
			}
			symbolP = symbol_find(line->symbol);
			if (!symbolP)
				line->pc = 0;
			else
				line->pc = S_GET_VALUE(symbolP) +
					symbolP->sy_frag->fr_address;
			pc_delta = line->pc - last_pc;
			ln_delta = line->linenum - last_ln;
			last_pc = line->pc;
			last_ln = line->linenum;
			if (pc_delta < 16 && ln_delta > -8 && ln_delta < 8)
			{
				entry.delta.ln_delta = ln_delta;
				entry.delta.pc_delta = pc_delta / 2;
				add_section_data(&entry, 1);
				size++;
			}
			else if (pc_delta < 256 && ln_delta > -129 &&
				ln_delta < 128)
			{
				entry.esc.esc_flag = -8;
				entry.esc.esc_code = dst_ln_dln1_dpc1;
				add_section_data(&entry, 1);
				dln1 = ln_delta;
				dpc1 = pc_delta / 2;
				add_section_data(&dln1, 1);
				add_section_data(&dpc1, 1);
				size += 3;
			}
			else if (pc_delta < 65536 && ln_delta > -32769 &&
				ln_delta < 32768)
			{
				entry.esc.esc_flag = -8;
				entry.esc.esc_code = dst_ln_dln2_dpc2;
				add_section_data(&entry, 1);
				dln2 = ln_delta;
				dpc2 = pc_delta / 2;
				add_section_data(&dln2, 2);
				add_section_data(&dpc2, 2);
				size += 5;
			}
			else
			{
				entry.esc.esc_flag = -8;
				entry.esc.esc_code = dst_ln_ln4_pc4;
				add_section_data(&entry, 1);
				dln4 = last_ln;
				dpc4 = last_pc;
				add_section_data(&dln4, 4);
				add_section_data(&dpc4, 4);
				size += 9;
			}
		}
		entry.esc.esc_flag = -8;
		entry.esc.esc_code = dst_ln_exit;
		add_section_data(&entry, 1);
		entry.esc.esc_flag = -8;
		entry.esc.esc_code = dst_ln_end;
		add_section_data(&entry, 1);
		size += 2;
		while  (size&4)
		{
			entry.esc.esc_flag = -8;
			entry.esc.esc_code = dst_ln_pad;
			add_section_data(&entry, 1);
			size++;
		}
	}

      frag_wane (frag_now);
      frag_align (SUB_SEGMENT_ALIGN (now_seg), 0);
      frag_wane (frag_now);
      frag_now->fr_fix = 0;
      know (frag_now->fr_next == NULL);
 
	return size;
}

void	build_dst_symbols_section(void)
{
	struct	block_entry *block;
	long	current_address;
	struct	sym_data *symdata;
	symbolS	*symbolP;

	if (!debugging_on)
		return;
	subseg_new(".symbols", 0);
	frag_now->fr_type = rs_machine_dependent;

	symbolP = symbol_new(".L__symbols", now_seg, 0, &zero_address_frag);
	symbol_table_insert(symbolP);

	write_string_table(symbol_strings, dst_typ_string_tab);
	current_address = size_strings(symbol_strings, 1) + 6;
	for (block = blocks; block; block = block->next)
	{
		for (symdata = block->symdata; symdata; symdata = symdata->next)
		{
			symdata->base = current_address;
			current_address += symdata->length;
		}
		if (block->symdata)
		{
			block->data.symbols_start.sect_index =
					SYMBOL_TABLE_ENTRY;
			block->data.symbols_start.sect_offset =
					block->symdata->base;
		}
	}
	for (block = blocks; block; block = block->next)
	{
		for (symdata = block->symdata; symdata; symdata = symdata->next)
		{
			do_offset_relocations(symdata->relocation_list,
					symdata->base);
			add_section_data(symdata->data, symdata->length);
			if (symdata->common_var)
			{
				/* We have a common symbol relocation.
				 * we know this is a VAR type field.
				 * make the fix known
				 */
				symbolS *symbolP;
				int	size = symdata->length;
				dst_rec_ptr_t rec = (dst_rec_ptr_t)
							symdata->data;

				symbolP = symbol_find(symdata->common_var);
				free(symdata->common_var);
				symdata->common_var = NULL;
				if (!symbolP)
					continue;
				if (size & 1)
					size++;
				size -= (char *) &DST_var(rec).locs.shorts[0].
							location -
					(char *) rec;
				fix_new(frag_now,
					frag_now_fix() - size,
					4, symbolP, 0, 0, 0);
			}
		}
	}

      frag_wane (frag_now);
      frag_align (SUB_SEGMENT_ALIGN (now_seg), 0);
      frag_wane (frag_now);
      frag_now->fr_fix = 0;
      know (frag_now->fr_next == NULL);
}

int	build_dst_block_section(long blocks_base)
{
	long	total_size;
	struct block_entry *block;
	struct block_entry *prev_block;
	struct block_entry *next_block;
	long	size, table_size;
	long	string_table_loc;
	long	current_loc;
	struct string_entry *string;
	symbolS	*symbolP;
	int	i;
	char	labelname[80];

	if (!debugging_on)
		return;
	subseg_new(".blocks", 0);
	frag_now->fr_type = rs_machine_dependent;
	blocks_section_start = blocks_base;

	symbolP = symbol_new(".L__blocks", now_seg, 0, &zero_address_frag);
	symbol_table_insert(symbolP);

	comp_unit_info.root_block_offset =
			sizeof(comp_unit_info) +
			size_strings(name_table, 1) +
			8;

	current_loc = comp_unit_info.root_block_offset + blocks_base;
	for (block = blocks; block; block = block->next)
	{
		block->blocks_start = current_loc;
		current_loc += sizeof(dst_rec_block_t) +
			(block->data.n_of_code_ranges -
			 dst_dummy_array_size) *
			sizeof(dst_code_range_t) + 2;
	}

	comp_unit_info.section_table = current_loc - blocks_base;
	comp_unit_info.file_table = comp_unit_info.section_table +
			size_strings(file_table, 1) + 6 +
			sizeof(section_table) +
			(section_table.number_of_sections -
			 dst_dummy_array_size) * sizeof(long) + 2;
	comp_unit_info.data_size = comp_unit_info.file_table +
			4 + count_strings(file_table) * sizeof(dst_file_desc_t);
	if (comp_unit_info.data_size & 2)
		comp_unit_info.data_size += 2;

	write_dst((char *) &comp_unit_info, sizeof(comp_unit_info),
		dst_typ_comp_unit, 0);
	total_size = sizeof(comp_unit_info) + 2;
	total_size += write_string_table(name_table,
				dst_typ_global_name_tab);


	for (block = blocks; block; block = block->next)
	{
		if (block->first_child)
			block->data.child_block_off =
				block->first_child->blocks_start -
				block->blocks_start;
		if (block->next_sibling)
			block->data.sibling_block_off =
				block->next_sibling->blocks_start -
				block->blocks_start;
		do_offset_relocations(block->relocation_list,
				      block->blocks_start);
		size = sizeof(dst_rec_block_t) +
			(block->data.n_of_code_ranges -
			 dst_dummy_array_size) *
			sizeof(dst_code_range_t);
		write_dst((char *) &block->data, size,
			dst_typ_block, 0);
		total_size += size + 2;
	}

	size = sizeof(section_table) +
		(section_table.number_of_sections -
		 dst_dummy_array_size) * sizeof(long);
	section_tab_vaddr = blocks_base + total_size + 4;
	write_dst((char *) &section_table, size,
				dst_typ_section_tab, 0);
	for (i = 0; i < TOTAL_SECTIONS; i++)
	{
		sprintf(labelname, ".L__%s", section_names[i]+1);
		if (i < 3)
		{
			symbolP = symbol_new(labelname, i+1, 0,
					&zero_address_frag);
			symbol_table_insert(symbolP);
		}
		else
		{
			symbolP = symbol_find(labelname);
		}
		if (!symbolP)
			continue;
		fix_new(frag_now, frag_now_fix() -
			(TOTAL_SECTIONS - i) * 4,
			4, symbolP, 0, 0, 0);
	}
	total_size += size + 2;

	total_size += write_string_table(file_table, dst_typ_string_tab);

	table_size = size_strings(file_table, 1);
	file_table_info.number_of_files = 0;
	size = 0;
	for (string = file_table; string; string = string->next)
	{
		file_table_info.files[file_table_info.number_of_files].dtm = 0;
		file_table_info.files[file_table_info.number_of_files].noffset=
			size - table_size;
		file_table_info.number_of_files++;
		size += strlen(string->string) + 1;
	}
	size = sizeof(short) + sizeof(dst_file_desc_t) *
		file_table_info.number_of_files;
	write_dst((char *) &file_table_info, size, dst_typ_file_tab, 0);
	total_size += size + 2;

	if (total_size & 2)
	{
		write_dst((char *) &file_table_info, 0, dst_typ_pad, 0);
		total_size += 2;
	}


      frag_wane (frag_now);
      frag_align (SUB_SEGMENT_ALIGN (now_seg), 0);
      frag_wane (frag_now);
      frag_now->fr_fix = 0;
      know (frag_now->fr_next == NULL);
	return total_size;
}

void	build_dst_mir_section(void)
{
	extern long string_byte_count;
	mirhdr	mir_hdr;
	mir mir_rec;
	char	version_string[256];

	subseg_new(".mir", 0);

	mir_hdr.nmirs = 1;
	add_section_data(&mir_hdr, sizeof(mir_hdr));
	mir_rec.kind = maker_mir_kind;
	mir_rec.mir_size = MIR_MAKER_SIZE;
	mir_rec.info.mir_maker.version =  0;
	mir_rec.info.mir_maker.creation_time = 0;
	mir_rec.info.mir_maker.maker_name_offset = string_byte_count;
	sprintf(version_string, "GNU assembler version %s (%s)",
				GAS_VERSION, TARGET_ALIAS);
	string_byte_count += strlen(version_string) + 1;
	add_section_data(&mir_rec, MIR_MAKER_SIZE);

      frag_wane (frag_now);
      frag_align (SUB_SEGMENT_ALIGN (now_seg), 0);
      frag_wane (frag_now);
      frag_now->fr_fix = 0;
      know (frag_now->fr_next == NULL);
}
