{[r+,b+,o=80] pasmat options}

MODULE gone_routines;

 {These routines implement the "gone" command.}

    {When the user wants to leave the node unattended, the gone routine borrows
    the display, puts a message on the screen, and waits for the user to type
    a password.

    To make this flexible, the user can specify the password and the message
    font with a configuration file.  This file is in the user's ~user_data
    directory.  The first line is a password, the second is a font pathname.
    If not available, the password defaults to just a <RETURN>, and the
    font pathname defaults to a standard.

    To avoid damaging the screen, it is switched to reverse video and back
    occasionally.  On pre-SR9.5 systems, this has the nice side effect of
    defeating the screen timeout so the message is always visible.  On
    systems after SR9.6, the screen timeout is disabled by the program during
    the time the message is on the screen.

    Written by:
      Dennis Cottel (dennis@nosc.mil)
      Naval Ocean Systems Center, San Diego, CA  }

 {allow DN3000, DN570/DN580 displays, call to gpr_$inq_display_characteristics}
 %var sr9.2;

 {use SR9.6 calls to disable screen timeout}
 %var sr9.6;

 %nolist;
 %include '/sys/ins/base.ins.pas';
 %include '/sys/ins/pad.ins.pas';
 %include '/sys/ins/error.ins.pas';
 %include '/sys/ins/streams.ins.pas';
 %include '/sys/ins/time.ins.pas';
 %include '/sys/ins/ec2.ins.pas';
 %include '/sys/ins/pm.ins.pas';
 %include '/sys/ins/pgm.ins.pas';
 %include '/sys/ins/gpr.ins.pas';
 %include '/sys/ins/kbd.ins.pas';
 %include '/sys/ins/fault.ins.pas';
 %include '/sys/ins/smdu.ins.pas';
 %list;

  DEFINE
    gone;

    %include 'gone.ins.pas';

{*--------------------* 
 | local declarations | 
 *--------------------*}

  CONST
    config_file_name = '/user_data/gone.config'; {relative to home directory}
    config_file_name_len = 22; {chars in configuration file name}

    default_font_path = '/sys/dm/fonts/std';
    default_font_path_len = 17; {chars in default font path}

  VAR
    {force RCS string to be embedded in code}
    ident: string_t :=
'$Header: //pumpkin_patch/local/source/gone/gone_routines.pas,v 1.15 88/01/12 10:44:23 dennis Exp $'
    ;

    debug: boolean := false; {use VAR not CONST to avoid warnings}
    status: status_$t;

    password: string_t; {password read from configuration file}
    password_len: pinteger; {length of password string}

    font_path: string_t; {font path from configuration file}
    font_path_len: pinteger; {length of font path name}

    font_id: integer; {identifying id for the display font}
    vert_spacing: integer; {pixels between lines of the message}
    need_a_space: boolean; {insert space before word if true}

    screen_bitmap: gpr_$bitmap_desc_t; {unique descriptor for screen bitmap}
    max_screen_x, max_screen_y: integer; {actual size of display}
    save_timeout: time_$clock_t; {original screen timeout value}

{*-------------------* 
 | Internal Routines | 
 *-------------------*}


  PROCEDURE check( IN str: string_t);
    INTERNAL;

   {Prints an error message and terminates the program on bad status.}

    BEGIN
      IF status.all <> status_$ok THEN
        BEGIN
        writeln('** Error in Gone ** :', str);
        error_$print(status);
        pgm_$exit;
        END {not ok} ;
    END {check} ;

{*-------------------------* 
 | Initialization routines | 
 *-------------------------*}


  PROCEDURE read_config_file;

    {This reads the user's "gone" configuration file for two parameters.
    The first is a password for resuming the program, and the second
    is a path name for the font to use when displaying the message.

    If for some reason either of the two parameters can't be found,
    then we use the defaults.}

    VAR
      home: string_t; {path to user's home directory}
      home_len: pinteger; {length of home directory path}
      cfile: text;
      open_status: integer32; {returned from OPEN call}
      no_password, no_font: boolean;
      i: pinteger;
      temp: string_t; {needed to copy the string constant}

    BEGIN
      pm_$get_home_txt(max_string, home, home_len);
      IF debug THEN writeln('home directory is "', home: home_len, '"');
      temp := config_file_name; {trick to allow one-at-a-time char copying}
      FOR i := 1 TO config_file_name_len DO {append to home directory}
        home[home_len + i] := temp[i];
      home_len := home_len + config_file_name_len;
      home[home_len + 1] := ' '; {needed for OPEN to work}
      IF debug THEN writeln('config file name is "', home: home_len, '"');

      open(cfile, home, 'OLD', open_status);
      IF open_status <> 0 THEN {no configuration file}
        BEGIN
        IF debug THEN writeln('gone config file not found');
        no_password := true;
        no_font := true;
        END {no config file}
      ELSE {read the config file}
        BEGIN
        reset(cfile);

        IF eof(cfile) THEN no_password := true
        ELSE {read the password}
          BEGIN
          i := 0;
          WHILE NOT eoln(cfile) DO
            BEGIN
            i := i + 1;
            read(cfile, password[i]);
            END {while not eoln} ;
          readln(cfile);
          password_len := i;
          no_password := false;
          END {read the password} ;
        IF debug THEN
          BEGIN
          write('password: "');
          FOR i := 1 TO password_len DO
            write(password[i]);
          writeln('"');
          END {debug} ;

        IF eof(cfile) THEN no_font := true
        ELSE {read the font path}
          BEGIN
          i := 0;
          WHILE NOT eoln(cfile) DO
            BEGIN
            i := i + 1;
            read(cfile, font_path[i]);
            END {while not eoln} ;
          readln(cfile);
          font_path_len := i;
          no_font := false;
          END {read the font path} ;
        IF debug THEN
          BEGIN
          write('fontpath: "');
          FOR i := 1 TO font_path_len DO
            write(font_path[i]);
          writeln('"');
          END {debug} ;
        END {read the config file} ;
      close(cfile);

      IF no_password THEN {use default password}
        BEGIN
        password := '';
        password_len := 0;
        END {use default password} ;
      IF no_font THEN {use default font}
        BEGIN
        font_path := '';
        font_path_len := 0;
        END {use default font} ;
    END {read_config_file} ;


  PROCEDURE set_screen_size;

    {Find out what display type is in use and set the global variables
    describing the maximum screen size.  The values max_screen_x and
    max_screen_y actually represent the largest allowed value, not the
    total size.}

    VAR
      disp: gpr_$disp_char_t;
      disp_len: integer16;

      {pre-sr9.2}
      config: gpr_$display_config_t;
      x, y: integer;

    BEGIN
      %IF sr9.2 %THEN;
      gpr_$inq_disp_characteristics(gpr_$borrow, 1, 56, disp, disp_len, status);
      check('gpr_$inq_disp_characteristics');
      max_screen_x := disp.x_window_size - 1;
      max_screen_y := disp.y_window_size - 1;
      IF debug THEN
        WITH disp DO
          writeln('window_size: ', x_window_size: 1, ', ', y_window_size: 1);

      %ELSE; {pre-sr9.2}
      gpr_$inq_config(config, status);
      check('gpr_$inq_config');
      CASE config OF

      { black and white displays }

        gpr_$bw_800x1024:
          BEGIN
          x := 800;
          y := 1024;
          END;
        gpr_$bw_1024x800:
          BEGIN
          x := 1024;
          y := 800;
          END;

        { color displays }

        gpr_$color_1024x1024x4, gpr_$color_1024x1024x8:
          BEGIN
          x := 1024;
          y := 1024;
          END;
        gpr_$color_1024x800x4, gpr_$color_1024x800x8:
          BEGIN
          x := 1024;
          y := 800;
          END;

        OTHERWISE {assume the worst}
          BEGIN
          x := 800;
          y := 800;
          END;
        END {case} ;
      max_screen_x := x - 1;
      max_screen_y := y - 1;
      %ENDIF; {sr9.2}

    END {set_screen_size} ;


  PROCEDURE set_vert_spacing;
    INTERNAL;

    {Since we don't know what font will be used for the message, we
    go through all the printable characters looking for the biggest.}

    CONST
      min_chr = 32; {space} {first printing character in ASCII}
      max_chr = 126; {~} {last printing character in ASCII}

    VAR
      str: PACKED ARRAY [1..1] OF char;
      size: gpr_$offset_t; {x_size, y_size}
      i: integer;
      max_y_size: integer;

    BEGIN
      max_y_size := 0;
      FOR i := min_chr TO max_chr DO
        BEGIN
        str[1] := chr(i);
        gpr_$inq_text_extent(str, 1, size, status);
        check('from gpr_$inq_text_extent in set_vert_spacing');
        IF size.y_size > max_y_size THEN max_y_size := size.y_size;
        END;
      vert_spacing := round(1.4 * max_y_size);
    END; {set_vert_spacing}


  PROCEDURE pad_init;
    INTERNAL;

    VAR
      mode: gpr_$display_mode_t; {borrow, direct, frame, etc}
      unit: stream_$id_t; {stream id of window for frame mode otherwise 1}
      size: gpr_$offset_t; {size of initial bitmap}
      hi_plane_id: gpr_$plane_t; {highest bit plane}
      text_color: gpr_$pixel_value_t; {drawing color for text}
      timeout: time_$clock_t; {for turning off screen timeout}

    BEGIN
      set_screen_size;
      mode := gpr_$borrow;
      unit := 1; {use 1 for frame mode}
      hi_plane_id := 0;
      size.x_size := max_screen_x + 1;
      size.y_size := max_screen_y + 1;
      gpr_$init(mode, unit, size, hi_plane_id, screen_bitmap, status);
      check('gpr_$init');

      %IF sr9.6 %THEN;
      {disable screen timeout}

      gpr_$inq_blank_timeout(save_timeout, status); {save original value}
      check('gpr_$inq_blank_timeout');
      timeout.high16 := 0;
      timeout.low32 := 0;
      gpr_$set_blank_timeout(timeout, status); {turn timeout off}
      check('gpr_$set_blank_timeout');
      %ENDIF; {sr9.6}

      {set up the fonts}

      gpr_$load_font_file(font_path, font_path_len, font_id, status);
      IF status.all <> status_$ok THEN {user's font not found - use default}
        gpr_$load_font_file(default_font_path, default_font_path_len, font_id,
                            status);
      check('gpr_$load_font_file');
      gpr_$set_text_font(font_id, status);
      check('gpr_$set_text_font');

      {set up color table so both black and white and color use white letters}

      text_color := 1; {always use color table location 1 for text}
      gpr_$set_text_value(text_color, status);
      check('gpr_$set_text_value');
      gpr_$set_color_map(1 {start} , 1 {how many slots} , gpr_$white, status);
      check('gpr_$set_color_map');

      set_vert_spacing;

      need_a_space := false;
      gpr_$move(0, 2 * vert_spacing, status); {leave a blank line at top}
      check('gpr_$move in pad_init');

    END; {pad_init}


  PROCEDURE initialize;
    INTERNAL;

    BEGIN
      read_config_file;
      pad_init;
    END {initialize} ;

{*----------------------* 
 | Termination routines | 
 *----------------------*}


  PROCEDURE pad_fin;
    INTERNAL;

    VAR
      delete_display: boolean; {ignored in borrow mode}

    BEGIN
      %IF sr9.6 %THEN;
      gpr_$set_blank_timeout(save_timeout, status); {restore original value}
      check('gpr_$set_blank_timeout');
      %ENDIF; {sr9.6}
      delete_display := false;
      gpr_$terminate(delete_display, status);
      check('gpr_$terminate');
    END {pad_fin} ;


  PROCEDURE finalize;
    INTERNAL;

    BEGIN
      pad_fin;
    END {finalize} ;

{*---------------------------------------* 
 | Routines to handle text on the screen | 
 *---------------------------------------*}


  PROCEDURE get_word( IN msg: string_t; {source of the message}
                     IN msg_len: pinteger; {total length of the message}
                     IN OUT cursor: pinteger; {current location in msg}
                     OUT word: string_t; {resulting word}
                     OUT word_len: pinteger {chars in resulting word} );

    {Find the next space-separated word in the input message.  Start
    with the current cursor position and skip over leading blanks.}

    BEGIN
      word_len := 0;
      WHILE (msg[cursor] = ' ') AND (cursor <= msg_len) DO {skip spaces}
        cursor := cursor + 1;
      WHILE (msg[cursor] <> ' ') AND (cursor <= msg_len) DO {get chars}
        BEGIN
        word_len := word_len + 1;
        word[word_len] := msg[cursor];
        cursor := cursor + 1;
        END {while - pick up chars in word}
    END {get_word} ;


  PROCEDURE new_line;
    INTERNAL;

   {Start a new line on the display.}

    VAR
      x, y: gpr_$coordinate_t; {current position - 2 byte integer}

    BEGIN
      gpr_$inq_cp(x, y, status); {find the current bitmap position}
      check('gpr_$inq_cp');
      y := y + vert_spacing;
      IF (y + vert_spacing) > max_screen_y THEN y := vert_spacing;
      gpr_$move(0, y, status);
      check('gpr_$move in new_line');
      need_a_space := false;
    END {new_line} ;


  PROCEDURE wrt_word( IN str: string_t;
                     IN len: integer);
    INTERNAL;

    {Write a word of text to the screen.  It is supposed to stay
    together, so if it won't fit, start on a new line.}

    VAR
      spacesize: integer; {width of a space}
      size: gpr_$offset_t; {size of text lump - x_size, y_size}
      x, y: gpr_$coordinate_t; {current position - 2 byte integer}

    BEGIN
      IF debug THEN writeln('wrt_word');
      gpr_$inq_space_size(font_id, spacesize, status); {find the space size}
      check('gpr_$inq_space_size');
      gpr_$inq_text_extent(str, len, size, status); {find the word size}
      check('gpr_$inq_text_extent');
      gpr_$inq_cp(x, y, status); {find the current bitmap position}
      check('gpr_$inq_cp');
      IF NOT need_a_space THEN spacesize := 0;
      IF (x + spacesize + size.x_size) > max_screen_x THEN new_line; {too big
        to fit}
      IF need_a_space THEN {add a space}
        BEGIN
        x := x + spacesize;
        gpr_$move(x, y, status);
        check('gpr_$move for a space');
        END {add a space} ;
      gpr_$text(str, len, status); {write the text to the screen}
      check('wrt_str');
      need_a_space := true; {next time, space first}
    END {wrt_word} ;


  PROCEDURE display_message(msg: string_t;
                            msg_len: pinteger);
    INTERNAL;

   {Display the given message on the screen.}

    VAR
      cursor: pinteger; {current position in string}
      word: string_t;
      word_len: pinteger;

    BEGIN
      IF debug THEN writeln('display_message()');
      cursor := 1;
      WHILE cursor <= msg_len DO {do something with each character}
        BEGIN
        get_word(msg, msg_len, cursor, word, word_len);
        wrt_word(word, word_len);
        END {while < msg_len} ;
    END {display_message} ;

{*--------------------------------* 
 | Routines to get keyboard input | 
 *--------------------------------*}


  PROCEDURE invert_screen;

    {To protect the screen if the message is left for a long time,
    invert all the pixels.  This routine assumes that the background
    color is position zero in the color table, and the text is written
    using location one in the color table.

    In order for this to properly protect the screen, each pixel must be
    completely off half the time, so the color values must be restricted
    to those which completely turn a color pixel on or off, and the
    background and foreground color values must not contain the same
    color pixel.}

    VAR
      color_table: gpr_$color_vector_t;
      temp: gpr_$pixel_value_t;

    BEGIN
      gpr_$inq_color_map(0, 2, color_table, status);
      check('gpr_$inq_color_map');
      temp := color_table[0];
      color_table[0] := color_table[1];
      color_table[1] := temp;
      gpr_$set_color_map(0, 2, color_table, status);
      check('gpr_$set_color_map');
    END {invert_screen} ;


  PROCEDURE get_keyboard(OUT c: char);
    INTERNAL;

    {Wait for an event from either the keyboard or the timer.  If the event
    is a valid keystroke, return the character.  If it is a timer event, then
    reverse the screen video.}

    CONST
      time_ec = 1; {event index for timeout}
      keys_ec = 2; {event index for keyboard (from GMR)}
      wait_time = 30; {seconds desired} {reverse video interval}

    VAR
      ec2_ptr: ARRAY [1..2] OF ec2_$ptr_t; {event count pointers}
      ec2_val: ARRAY [1..2] OF integer32; {event count trigger values}
      which: integer; {which of the two event types occurred}
      got_a_char: boolean; {true when a char came in from keyboard}

      unobscured: boolean;
      position: gpr_$position_t;
      event_type: gpr_$event_t;

    BEGIN
      time_$get_ec(time_$clockh_key, ec2_ptr[time_ec], status);
      check('time_$get_ec');
      gpr_$get_ec(gpr_$input_ec, ec2_ptr[keys_ec], status);
      check('gpr_$get_ec');

      ec2_val[time_ec] := ec2_$read(ec2_ptr[time_ec]^) + wait_time * 4;
      ec2_val[keys_ec] := ec2_$read(ec2_ptr[keys_ec]^);

      gpr_$enable_input(gpr_$keystroke, [chr(0)..chr(16), chr(17),
                        chr(18)..chr(127), kbd_$f1..kbd_$f8, kbd_$bs, kbd_$cr,
                        kbd_$tab], status);
      check('gpr_$enable_input');

      got_a_char := false;
      REPEAT
        which := ec2_$wait(ec2_ptr, ec2_val, 2 {entries} , status);
        check('ec2_$wait');
        CASE which OF
          time_ec: {timed out - reverse the video}
            BEGIN
            ec2_val[time_ec] := ec2_$read(ec2_ptr[time_ec]^) + wait_time * 4;
            invert_screen;
            IF debug THEN writeln('timeout event');
            END {timeout event} ;
          keys_ec: {looks like a keyboard entry}
            BEGIN
            ec2_val[keys_ec] := ec2_$read(ec2_ptr[keys_ec]^) + 1;
            unobscured := gpr_$cond_event_wait(event_type, c, position, status);
            got_a_char := (event_type = gpr_$keystroke);
            IF debug THEN writeln('key event, got_a_char =', got_a_char);
            END {keyboard event} ;
          END {case} ;
      UNTIL got_a_char;
    END; {get_keyboard}


  PROCEDURE wait_for_password;
    INTERNAL;

    {Read characters from the keyboard and test that they match the
    password read from the user's configuration file.  Backspaces
    work to delete the previously entered character, and a <RETURN>
    starts over.}

    CONST
      bs = 8; {ascii back space}
      cr = 13; {ascii carriage return}
      tab = 9; {ascii tab}

    VAR
      c_kbd: char; {character read from keyboard}
      str: string_t; {assemble trial password from user}
      str_l: integer; {length of trial password}
      good_pwd: boolean; {true when trial matches user's password}
      i: integer;

    BEGIN {wait_for_password}
      IF debug THEN writeln('wait_for_password()');
      str_l := 0;
      good_pwd := false;

      {disable quit character altogether for as long as we're in borrow mode}
      smd_$set_quit_char(chr(0), status);
      check('smd_$set_quit_char');

      REPEAT
        get_keyboard(c_kbd);
        IF c_kbd = kbd_$bs THEN c_kbd := chr(bs)
        ELSE IF c_kbd = kbd_$cr THEN c_kbd := chr(cr)
        ELSE IF c_kbd = kbd_$tab THEN c_kbd := chr(tab);
        IF c_kbd = chr(cr) THEN
          BEGIN
          IF str_l = password_len THEN
            BEGIN
            i := 1;
            WHILE (i <= password_len) AND (str[i] = password[i]) DO
              i := i + 1;
            IF (i - 1) = password_len THEN good_pwd := true;
            END;
          str_l := 0;
          END
        ELSE IF c_kbd = chr(bs) THEN
          BEGIN
          str_l := str_l - 1;
          IF str_l < 0 THEN str_l := 0;
          END
        ELSE
          BEGIN
          str_l := str_l + 1;
          IF str_l > max_string THEN str_l := max_string;
          str[str_l] := c_kbd;
          END;
      UNTIL good_pwd;
    END; {wait_for_password}

{*---------------* 
 | entry routine | 
 *---------------*}


  PROCEDURE gone { (msg: string_t; msg_len: pinteger) } ;

    {This is called when the user wants to leave the node unattended
    for a time.}

    BEGIN
      IF debug THEN writeln('gone()');
      initialize;
      display_message(msg, msg_len);
      wait_for_password;
      finalize;
    END {gone} ;

  {END MODULE gone_routines}
