/*   showgif.c - Displays a GIF image on the apollo screen.
 *   version 8d,  23 jun 91 - By Brett W. Davis,  macpd@icaen.uiowa.edu
 *
 * showgif supports all known HP/Apollo displays, even
 * the old four bit displays. showgif can read GIF87a
 * and GIF89a files, however new features added by GIF89a
 * are piped to /dev/null.
 * Bugs: GIF files that contain multiple images are not
 *       treated properly. When I find one of these mythical
 *       multi-GIF files I will fix this.
 *
 * This program is hereby placed in the public domain.
 * There are no restrictions on the use of all or any
 * part of this program. 
 * 
 * Derived from gif2gpr by Ben Samit,  March 1989
 * samit@demon.siemens.com
 * The GIF decode code was lifted right out 
 * of Scott Hemphill's gif to postscript converter.
 * Since Scott Hemphill was kind enough to put his code 
 * in the public domain, I feel obliged to do the same. 
 * My copy of gif2gpr.c included:
 * Save-to-a-bitmap feature added by Roque D. Oliveira,  April 1989.
 * oliveria@caen.engin.umich.edu  
 * 
 * Set_gray_map() and dither patterns from Roger Black, feb 1990.
 * jrblack@umaxc.weeg.uiowa.edu
 * Cleanup handler pascal source from David B. Funk, feb 1990.
 * dbfunk@icaen.uiowa.edu
 * I would appreciate hearing of any suggestions for added features
 * (and bug reports) regarding this program.  Brett Davis.
 */


#include <stdio.h>
#include <string.h>
#include <setjmp.h>
#include <apollo/base.h>
#include <apollo/cal.h>
#include <apollo/gpr.h>
#include <apollo/pad.h>
#include <apollo/error.h>
#include <apollo/pgm.h>
#include <apollo/pfm.h>

#define min(x,y) ((x) < (y) ? (x) : (y))
#define max(x,y) ((x) > (y) ? (x) : (y))
#define FALSE 0
#define TRUE 1

/* Globals used in the setup of each image. */
int firsttimeshown;
int oneimage;
unsigned int aspectratio;
unsigned int backgroundcolor;
int white, black;
int  WIDTH,  HEIGHT;
int eWIDTH, eHEIGHT;
status_$t            status;
gpr_$window_t        display;
gpr_$rgb_plane_t     hi_plane;
gpr_$disp_char_t     display_characteristics;
static short int     disp_len = sizeof(gpr_$disp_char_t);
       short int     disp_len_returned;
gpr_$offset_t        init_size, eSIZE;
gpr_$position_t      eORIGIN;
gpr_$bitmap_desc_t   display_bitmap_desc;
stream_$id_t         graphics_stream , unit;
pad_$window_desc_t   pad_window ;
gpr_$window_t        text_rect;
int apollobits;
unsigned int screenwidth;           /* The dimensions of the screen */
unsigned int screenheight;          /*   (not those of the image)   */
int nameheight;

/* Globals used in file loop. */
gpr_$bitmap_desc_t      curs_pat_desc;
gpr_$position_t         curs_position, curs_origin;
gpr_$raster_op_array_t  curs_raster_op;
boolean                 curs_active;
gpr_$keyset_t    keys;        /* keyset used to advance to next picture in userfeedback mode. */
int endoflist = false;
pfm_$cleanup_rec cl_rec;
status_$t pfm_status;
int quit;
char buf[255];                     /* temp buffer used several places. */
jmp_buf env;                /* longjump var */
int abort;                 /* longjump var */

/* getopt stuff */
extern int optind;     /* index of which argument is next */
extern char *optarg;    /* pointer to argument of this option */ 

/* Font stuff. */
char font_name[] = "charter-bold24";
short int font_name_size = sizeof(font_name)-1;
gpr_$offset_t    textsize;
short int        font_id;

/* Globals relating to files. */
FILE *filelist;
FILE *infile;
char *file_name;
char *filelist_name;
char *bitmap_file_name = NULL;
char *progname;				/* the name of this program */

/* Globals used by LZW decompression. */
typedef int bool;              
typedef struct codestruct    
 {                              
  struct codestruct *prefix;    
  unsigned char first,suffix;   
 } codetype;                    
unsigned char *raster;              /* Decoded image data */
codetype *codetable;                /* LZW compression code data */
int datasize,codesize,codemask;     /* Decoder working variables */
int clear,eoi;                      /* Special code values */
int scanlinebyte;
int scanwidth;

/* Globals relating to colormaps. */
bool global;                        /* Is there a global color map? */
int globalbits;                     /* Number of bits of global colors */
unsigned char globalmap[1024];    /* RGB values for global color map */
bool local;                        /* Is there a local color map? */
int localbits;                     /* Number of bits of local colors */
int imagebits;
unsigned char localmap[1024];     /* RGB values for local color map */
unsigned char mono4map[1024];     /* RGB values for mono and 4bit color map */
int mono4bits;
gpr_$color_vector_t  color_map; 

/* Globals used by draw scanline calls. */
gpr_$pixel_value_t *foo;             /* Scan line buffer */                 
int *ix_inc_table;                   /* Resize x table buffer */
bool interleaved;
int *interleavetable;                /* Scan line interleave table */
gpr_$pixel_value_t gray[256];	     /* Four-bit values for b/w dither, also 4bit color */
gpr_$window_t    destination_window;
int mono4color1;
int currentline;
int gifline;
int *scanlinetable;
int *scanlineused;

/* list of globals set by getopt. */
int monocrome = false;
int four_bit_color = false;      
int full_screen = false;             /* Don't expand to max size by default */
int infinite = false;
int listofgifs = false;
int showname = true;
int save_to_bitmap = false;
int printinfo = false;
int showgif = true;
int sleeptime = 0;
int userfeedback = true;

/* Here is the dithering array.  The four octal numbers on each  
   line represent a four-by-four bitmap that is displayed on the 
   Apollo screen for that color code.  There are, of course, 16  
   color codes from 0 to 15. */                                  
                                                                 
static gpr_$pixel_value_t dither[256] = {                                    
         1,1,1,1, 1,1,1,1, 1,1,1,1, 1,1,1,1,
         1,1,1,1, 1,1,0,1, 1,1,1,1, 1,1,1,1,
         1,1,1,1, 1,1,0,1, 1,1,1,1, 0,1,1,1,
         1,0,1,1, 1,1,0,1, 1,1,1,1, 0,1,0,1,
         1,1,1,1, 1,1,0,1, 1,1,1,1, 0,1,0,1,
         1,0,1,1, 1,1,0,1, 1,0,1,1, 0,1,0,1,
         1,0,1,0, 1,1,0,1, 1,0,1,1, 0,1,0,1,
         1,0,1,0, 1,1,0,1, 1,0,1,0, 0,1,0,1,
         1,0,1,0, 0,1,0,1, 1,0,1,0, 0,1,0,1,
         1,0,1,0, 0,1,0,1, 1,0,1,0, 0,1,0,0,
         1,0,1,0, 0,0,0,1, 1,0,1,0, 0,1,0,0,
         1,0,1,0, 0,0,0,0, 1,0,1,0, 0,1,0,0,
         1,0,1,0, 0,0,0,0, 1,0,1,0, 0,0,0,0,
         1,0,0,0, 0,0,0,0, 1,0,1,0, 0,0,0,0,
         1,0,0,0, 0,0,0,0, 0,0,1,0, 0,0,0,0,
         0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0,0,0,
         };  


void main(argc,argv);
   void get_options(argc,argv);
   void Initialize();
   void save_color_map();
      void check(messagex);
   void process_gif_file();
      void checksignature();
      void read_global_header();
         void read_color_map(colormap,colorbits);
      void readimage();
         void read_local_header();
         void image_setup();
            void set_gray_map();
            void set_four_bit_color_map();
            void set_mono_color_map();
            void set_color_map();
            void find_black_white();
            void fit_image_to_display();
         void window_setup();
            void draw_banner();
         void readextension();
         void readraster(WIDTH,HEIGHT);
            void process(code,fill);
               void outcode(p,fill);
                  void draw_scanline();
                     void draw_gray();
                     void draw_resized_gray();
                     void draw_resized_4bit_color();
                     void draw_color();
                     void draw_resized_color();
   void KbEnable();
   void Close();
   void reset_color_map();

/* void main(argc,argv);  */
/* void Initialize();     */
void usage();
void error(s);
void fatal(s);  
void SaveImage();

gpr_$rwin_pr_t  rwin = process_gif_file;        /*  entry point for refreshing window */
gpr_$rhdm_pr_t  rhdm = NULL;          /*  entry point for refreshing hidden display memory */



/***********************************/
void get_options(argc,argv) 
int argc;
char **argv;
{
int c;

  progname = strrchr(argv[0], '/');
  if (progname) progname++;
  else progname = argv[0];
  
  while ((c=getopt(argc,argv,"14filnopst:u?"))!=EOF) 
    {
     switch (c)
       {                
        case '1':    monocrome = true;   /*** testing mode ***/
                     break;
  
        case '4':    four_bit_color = true;   /*** testing mode ***/
                     break;
  
        case 'f':    full_screen = true;
                     break;
  
        case 'i':    infinite = true;
                     break;
  
        case 'l':    listofgifs = true;
                     break;
  
        case 'n':    showname = false;
                     break;
  
        case 'o':    save_to_bitmap = true;
                     break;      
  
        case 'p':    printinfo = true;
                     break;
  
        case 's':    showgif = false;
                     break;
  
        case 't':    sscanf(optarg, "%d", &sleeptime);
                     fprintf(stdout,"view time is %d seconds\n",sleeptime);
                     break;      
  
        case 'u':    userfeedback = false;
                     break;
  
        case '?':
        default :
                     usage();
       }
    }
  
  if (argc == optind)
    {
     usage();
     exit(-1);
    }
}

/***********************************/
void save_color_map() 
{
  gpr_$acquire_display(&status);
  check("in save_color_map, after first try to gpr$acquire_display"); 
  gpr_$inq_color_map((gpr_$pixel_value_t) 0,(short) (1<<apollobits), color_map, &status);
  check("in save_color_map after calling gpr_$inq_color_map");
  gpr_$release_display(&status);
}

/***********************************/
void check(messagex)
char *messagex;
{
   if (status.all)
   {   error_$print (status);
       printf("Error occurred while %s.\n", messagex);
   }    
}

/***********************************/
void process_gif_file()
{
int ch;

    rewind(infile);
    checksignature();
    read_global_header();
    oneimage = true;

    do 
     {    
       quit = false;
       ch = getc(infile);
       switch (ch)
       {
        case 0:     break; 

        case 44:    readimage();
                    sleep(sleeptime);   
                    break;

        case 59:    quit = true;
                    break;

        case 33:    getc(infile);
/*                    fprintf(stdout,"%s  : This has an extension!.\n",progname); */
                    readextension();
                    break;

        default:    fprintf(stdout,"illegal GIF block type:%i\n", ch);
                    quit = true;
                    break;
       }
     } while (!quit);
}

/***********************************/
void checksignature()
{
  fread(buf,1,6,infile);
  if (strncmp(buf,"GIF",3)) fatal("file is not a GIF file");
  if (strncmp(&buf[3],"87a",3)) 
  {    if (strncmp(&buf[3],"89a",3)) fatal("unknown GIF version number");
  }
}

/***********************************/
/* Get information which is global to all the images stored in the file */
void read_global_header()
{
unsigned char buf[9];

  fread(buf,1,7,infile);
/*  globalwidth = buf[0] + (buf[1] << 8);  */
/*  globalheight = buf[2] + (buf[3] << 8); */
  backgroundcolor = buf[5];
  aspectratio = buf[6];
/* printf("aspect %d, background %d\n", aspectratio, backgroundcolor);  */
  global = buf[4] & 0x80;
  if (global)
  {
    globalbits = (buf[4] & 0x07) + 1;
/*    printf("global bitmap  : %d colors\n",(1<<globalbits));  */
    read_color_map(globalmap, globalbits);
  }
}

/***********************************/
void read_color_map(colormap,colorbits) 
unsigned char *colormap;
int colorbits;
{
int i,j;

  for(i=0;i<(1<<colorbits);i++) 
    {
     *colormap++ = 0;
     for (j=1;j<4;j++) *colormap++ = getc(infile);
    }
}

/***********************************/
void readimage()
{
  read_local_header();
  if (printinfo & firsttimeshown)
    {
     printf("%s  %d x %d, %d colors",file_name,WIDTH,HEIGHT,(1<<imagebits)); 
     if (interleaved) printf(" interleaved");
     printf("\n");
    }

  if (showgif) 
   {
    if (firsttimeshown) image_setup();
    window_setup();
    readraster(WIDTH,HEIGHT);
   }
  else 
   {
    getc(infile); 
    readextension();   /* pipe to /dev/nul */
   }

/*  if (!oneimage) fprintf(stdout,"%s  : Multi Image?\n",progname); */
  firsttimeshown = false;
  oneimage = false;
}

/***********************************/
void read_local_header()
{
  unsigned char buf[9];
  unsigned left,top,width,height;
  register row;
  register i;

  fread(buf,1,9,infile);
/*  left        = buf[0] + (buf[1] << 8);  */
/*  top         = buf[2] + (buf[3] << 8);   */
  WIDTH       = buf[4] + (buf[5] << 8);
  HEIGHT      = buf[6] + (buf[7] << 8);
  scanwidth   = WIDTH -1;
  local       = buf[8] & 0x80;
  interleaved = buf[8] & 0x40;
  imagebits   = globalbits;
  if (local)
   {
    localbits = (buf[8] & 0x7) + 1;
    imagebits = localbits;
    read_color_map(localmap, localbits);
   }
  else if (!global)
        {
         fatal("no colormap present for image");
        } 

  row = 0;
  if (interleaved)
   {
    for (i = 0; i < HEIGHT; i += 8) interleavetable[row++] = i;
    for (i = 4; i < HEIGHT; i += 8) interleavetable[row++] = i;
    for (i = 2; i < HEIGHT; i += 4) interleavetable[row++] = i;
    for (i = 1; i < HEIGHT; i += 2) interleavetable[row++] = i;
   } 
  else
   {
    for (i = 0; i < HEIGHT; i++) interleavetable[i] = row++;
   }
   
  if (imagebits < 5) four_bit_color = false;
  if ((hi_plane == 3) & (imagebits > 4)) four_bit_color = true;
}                       

/***********************************/
void image_setup()
{
   if (monocrome)
      {
       set_gray_map();
       set_mono_color_map();
      }
   else
      {         
       if (four_bit_color) 
          {
           set_gray_map();
           set_four_bit_color_map();
          }
      }  
   set_color_map();
   if (showname) find_black_white();
   fit_image_to_display();
}

/***********************************/
/* This routine turns the 24-bit RGB color map into a 4-bit mono map.  The 
   resulting numbers are fed through the dither table to get the actual bit
   patterns displayed on the screen.  Different results can be obtained by 
   playing around with the algorithm used here. */                         
                                                                           
void set_gray_map()
{                                                          
  int i,red,green,blue, colorbits;
  unsigned char *colormap;
  
  if (local) 
    {
     colormap  = localmap;
     colorbits = localbits;
    }
  else 
    {
     colormap  = globalmap;
     colorbits = globalbits;
    }

  for(i=0;i<(1<<colorbits);i++) 
    {
     *colormap++;                                         
     red   = *colormap++;                                                 
     green = *colormap++;                                               
     blue  = *colormap++;                                                
     gray[i]=((red + green + blue)/3) >> 4;                                
    }                                                                        
}                                                                          

/***********************************/
void set_four_bit_color_map() 
{
  int i, shade, gray; 
  unsigned char *colormap;

  colormap  = mono4map;
  mono4bits = 4;
  shade     = 0;
  gray      = 16;
  for(i = 0 ; i < 16 ; i++) 
    {
     *colormap++ = 0;
     *colormap++ = shade;
     *colormap++ = shade;
     *colormap++ = shade;
     shade += gray;
    }
}

/***********************************/
void set_mono_color_map() 
{
  mono4bits = 1;

  mono4map[0] = 0;
  mono4map[1] = 255;
  mono4map[2] = 255;
  mono4map[3] = 255;
  mono4map[4] = 0;
  mono4map[5] = 0;
  mono4map[6] = 0;
  mono4map[7] = 0;
}

/***********************************/
void set_color_map() 
{
  void *colormap; 
  int colorbits;
  
  if (monocrome | four_bit_color) 
    {
     colormap  = mono4map;
     colorbits = mono4bits;
    }
  else if (local) 
    {
     colormap  = localmap;
     colorbits = localbits;
    }
  else 
    {
     colormap  = globalmap;
     colorbits = globalbits;
    }

  gpr_$acquire_display(&status);
  gpr_$set_color_map((gpr_$pixel_value_t) 0,(short) (1<<colorbits), colormap, &status);
  check("in set_color_map after calling gpr_$set_color_map");
  gpr_$release_display(&status);
}

/***********************************/
void find_black_white()
{
   int brightness, whiteness, blackness, i;
   int nameheight;
   unsigned char *colormap; 
   int colorbits;

   if (monocrome | four_bit_color) 
     {
      colormap  = mono4map;
      colorbits = mono4bits;
     }
   else if (local) 
     {
      colormap  = localmap;
      colorbits = localbits;
     }
   else 
     {
      colormap  = globalmap;
      colorbits = globalbits;
     }

   blackness = 32000;
   whiteness = 0;
   for(i=0;i<(1<<colorbits);i++) 
       {
        *colormap++;
        brightness = *colormap++ + *colormap++ + *colormap++;
        if (brightness < blackness) 
            {
             blackness = brightness;
             black = i;
            }
        if (brightness > whiteness) 
            {
             whiteness = brightness;
             white = i;
            }
       }
/* fprintf(stdout,"white %i, black %i\n", white, black); */
}

/***********************************/
void fit_image_to_display()
{
   float expand;
   float expand_X, expand_Y; 
   float fraction;
   int skip, xskip, yskip;
   int ex, ey, temp;
   int count, found;
   int banner_height;
   int colorbits;

    eWIDTH  = WIDTH;
    eHEIGHT = HEIGHT;
    if (showname) banner_height = nameheight;
    else banner_height = 0;

    if ((full_screen) | (screenwidth < WIDTH) | ((screenheight - banner_height) < HEIGHT))   
      {
       expand_X = screenwidth / (float)WIDTH;
       expand_Y = (screenheight - banner_height) / (float)HEIGHT;
       /* expand gets the smaller value for a even X,Y expansion. */  
       expand = min(expand_X, expand_Y);
       eWIDTH  = WIDTH * expand;
       eHEIGHT = HEIGHT * expand;
      }

     if (local) colorbits = localbits;
     else colorbits = globalbits;
     if (colorbits == 1) monocrome = false;  /* b&w image, don't dither. */

     if (monocrome & (full_screen != true)) {
         if (WIDTH < screenwidth) {              
           xskip = screenwidth / WIDTH;              
         }                                         
         if (HEIGHT < (screenheight - banner_height)) {            
           yskip = (screenheight - banner_height) / HEIGHT;    
         }                                         
         skip = xskip;                             
         if (yskip < xskip) skip = yskip;  
         if (skip > 1) {
           eWIDTH = eWIDTH * skip;
           eHEIGHT = eHEIGHT * skip;
         } 
     }                                             

/*    if ((pad_window.width == eWIDTH) & (pad_window.height == (eHEIGHT + banner_height))) */
/*      {    */
       gpr_$acquire_display(&status);
       gpr_$clear( (gpr_$pixel_value_t) backgroundcolor, &status);
       gpr_$release_display(&status);
/*      }  */
    pad_window.top = 0;
    pad_window.left = (screenwidth - eWIDTH) / 2;
    pad_window.width = eWIDTH;
    pad_window.height = eHEIGHT + banner_height;

    pad_$set_full_window(unit,(short) 1,&pad_window,&status);
}
  
/***********************************/
void window_setup()
{
int ex, ey, temp;
int banner_height;

   if (showname) banner_height = nameheight;
   else banner_height = 0;

   gpr_$acquire_display(&status);
   gpr_$inq_bitmap_position  (display_bitmap_desc, &eORIGIN , &status);
   gpr_$inq_bitmap_dimensions(display_bitmap_desc, &eSIZE   , &hi_plane , &status);
   gpr_$release_display(&status);

   eWIDTH  = eSIZE.x_size ;
   eHEIGHT = eSIZE.y_size - banner_height;;

   if (showname) draw_banner();

   destination_window.window_base.x_coord = 0;
   destination_window.window_size.x_size = eWIDTH;
   destination_window.window_size.y_size = mono4color1;
        
   for (ex=0;  ex<=eWIDTH;  ex += 1) 
        {
         ix_inc_table[ex] = (WIDTH * ex) / eWIDTH; 
        }   
   
   for (ey=0;  ey<2047;  ey+=1) 
        {
         scanlineused[ey] = 0;
         scanlinetable[ey] = 0;
        }

   for (ey=0;  ey<=eHEIGHT;  ey+=mono4color1)
        {    
         temp = (HEIGHT * ey) / eHEIGHT;
         scanlineused[temp] += 1;
         if (scanlineused[temp] == 1) scanlinetable[temp] = ey;
        }         

   currentline = 0;
   gifline = 0;
   scanlinebyte = 0;
}

/***********************************/
void draw_banner() 
{
short int  name_len;

  gpr_$acquire_display(&status);
  text_rect.x_coord = 0;
  text_rect.y_coord = eHEIGHT;
  text_rect.x_size = eWIDTH;
  text_rect.y_size = nameheight;  
  gpr_$set_fill_value((gpr_$pixel_value_t) white, &status);
  gpr_$rectangle(text_rect, &status);
  gpr_$set_text_background_value((gpr_$pixel_value_t) white, &status);
  gpr_$set_text_value((gpr_$pixel_value_t) black, &status);

  name_len = strlen(file_name);
  gpr_$inq_text_extent ( file_name, name_len, &textsize, &status);
  if (eWIDTH > textsize.x_size) 
    {
     gpr_$move( (gpr_$coordinate_t) (eWIDTH-textsize.x_size)/2, (gpr_$coordinate_t) (eHEIGHT + textsize.y_size), &status); 
    }
  else
    {
     gpr_$move( (gpr_$coordinate_t) (eWIDTH-textsize.x_size-10), (gpr_$coordinate_t) (eHEIGHT + textsize.y_size), &status); 
    }
  gpr_$text( file_name, name_len, &status);
  gpr_$release_display(&status);
}

/***********************************/
/* Read a GIF extension block (and do nothing with it). */
void readextension()
 {
  unsigned char code,count;

  while (count = getc(infile)) fread(buf,1,count,infile);
 }

/***********************************/
/* Decode a raster image */
void readraster(width,height)
unsigned width,height;
{
  unsigned char *fill = raster;
  unsigned char buf[255];
  register bits=0;
  register unsigned count,datum=0;
  register unsigned char *ch;
  register int code;

  datasize  = getc(infile);
  clear     = 1 << datasize;
  eoi       = clear+1;
  codesize  = datasize + 1;
  codemask  = (1 << codesize) - 1;
  for (code = 0; code < clear; code++)
   {
    codetable[code].prefix = (codetype*)0;
    codetable[code].first  = code;
    codetable[code].suffix = code;
   }
  for (count = getc(infile); count > 0; count = getc(infile))
   {
    fread(buf,1,count,infile);
    for (ch=buf; count-- > 0; ch++)
     {
      datum += *ch << bits;
      bits  += 8;
      while (bits >= codesize)
       {
        code = datum & codemask;
        datum >>= codesize;
        bits -= codesize;
        if (code == eoi) goto exitloop;  /* This kludge put in
                                            because some GIF files
                                            aren't standard */
        process(code,&fill);
       }
     }
   }
exitloop:
}

/***********************************/
/* Process a compression code.  "clear" resets the code table.  Otherwise
   make a new code table entry, and output the bytes associated with the code. */
void process(code,fill)
register code;
unsigned char **fill;
{
  static avail,oldcode;
  register codetype *p;

  if (code == clear)
  {
    codesize = datasize + 1;
    codemask = (1 << codesize) - 1;
    avail = clear + 2;
    oldcode = -1;
  }
   else if (code < avail)
   {
    outcode(&codetable[code],fill);
    if (oldcode != -1)
     {
      p = &codetable[avail++];
      p->prefix = &codetable[oldcode];
      p->first  = p->prefix->first;
      p->suffix = codetable[code].first;
      if ((avail & codemask) == 0 && avail < 4096)
          {
           codesize++;
           codemask += avail;
          }
     }
    oldcode = code;
   } 
   else if (code == avail && oldcode != -1)
   {
    p = &codetable[avail++];
    p->prefix = &codetable[oldcode];
    p->first  = p->prefix->first;
    p->suffix = p->first;
    outcode(p,fill);
    if ((avail & codemask) == 0 && avail < 4096)
      {
        codesize++;
        codemask += avail;
      }
   oldcode = code;
   } else
      {
         fatal("illegal code in raster data");
      }
}

/***********************************/
/* Output the bytes associated with a code to the raster array */
void outcode(p,fill)
register codetype *p;
register unsigned char **fill;
{
  if (p->prefix) outcode(p->prefix,fill);
  *(*fill)++ = p->suffix;
  
  if (scanlinebyte++ == scanwidth)
     {
      scanlinebyte = 0;
      draw_scanline();
      gifline += 1;              
      *fill = raster;
     }
}

/***********************************/
void draw_scanline() 
{

      if ( (WIDTH != eWIDTH) || (HEIGHT != eHEIGHT) ) 
        {
            if (monocrome) draw_resized_gray();
                else if (four_bit_color) draw_resized_4bit_color();
                     else draw_resized_color();
        }
      else
        {
            if (monocrome) draw_gray();
                else if (four_bit_color) draw_resized_4bit_color();
                     else draw_color();
        }
}

/***********************************/
void draw_gray()
{
    register int g;
    gpr_$pixel_value_t *scan1, *scan2, *scan3, *scan4;
    int x, line, count;
     
    gpr_$acquire_display(&status);
                       
    line = interleavetable[gifline];
    for (count = 0; count < scanlineused[line]; count +=1)
      {
       scan1 = foo;
       scan2 = scan1 + WIDTH;
       scan3 = scan2 + WIDTH;
       scan4 = scan3 + WIDTH;
       for(x=0;x<eWIDTH-3;)
           {                     
            g = ((gray[raster[x++]] + gray[raster[x++]] + gray[raster[x++]] + gray[raster[x++]]) >> 2 )<< 4; 
     
            *scan1++ = dither[g++];
            *scan1++ = dither[g++];
            *scan1++ = dither[g++];
            *scan1++ = dither[g++]; 
     
            *scan2++ = dither[g++];
            *scan2++ = dither[g++];
            *scan2++ = dither[g++];
            *scan2++ = dither[g++];  
     
            *scan3++ = dither[g++];
            *scan3++ = dither[g++];
            *scan3++ = dither[g++];
            *scan3++ = dither[g++];  
     
            *scan4++ = dither[g++];
            *scan4++ = dither[g++];
            *scan4++ = dither[g++];
            *scan4++ = dither[g];
           }

      destination_window.window_base.y_coord = scanlinetable[line] + (count << 2);
      gpr_$write_pixels(foo, destination_window, &status);                
     } 
    gpr_$release_display(&status);
}  

/***********************************/
void draw_resized_gray()
{
    register int g;
    gpr_$pixel_value_t *scan1, *scan2, *scan3, *scan4;
    int x, line, count;
     
    gpr_$acquire_display(&status);

    line = interleavetable[gifline];
    for (count = 0; count < scanlineused[line]; count +=1)
       {    
        scan1 = foo;
        scan2 = scan1 + eWIDTH;
        scan3 = scan2 + eWIDTH;
        scan4 = scan3 + eWIDTH;
        for(x=0;x<eWIDTH-3;)
            {                     
             g = ((gray[raster[ix_inc_table[x++]]] 
                 + gray[raster[ix_inc_table[x++]]] 
                 + gray[raster[ix_inc_table[x++]]]
                 + gray[raster[ix_inc_table[x++]]] ) >> 2 )<< 4;
        
             *scan1++ = dither[g++];
             *scan1++ = dither[g++];
             *scan1++ = dither[g++];
             *scan1++ = dither[g++]; 
        
             *scan2++ = dither[g++];
             *scan2++ = dither[g++];
             *scan2++ = dither[g++];
             *scan2++ = dither[g++];  
        
             *scan3++ = dither[g++];
             *scan3++ = dither[g++];
             *scan3++ = dither[g++];
             *scan3++ = dither[g++];  
        
             *scan4++ = dither[g++];
             *scan4++ = dither[g++];
             *scan4++ = dither[g++];
             *scan4++ = dither[g];
            }

        destination_window.window_base.y_coord = scanlinetable[line] + (count << 2);
        gpr_$write_pixels(foo, destination_window, &status);                
/*
fprintf(stdout,"line %i,  scantable %i, scanlineused %i, gifline %i\n",
line,  scanlinetable[line], scanlineused[line], gifline );
*/
       }

    gpr_$release_display(&status);
}

/***********************************/
void draw_resized_4bit_color()
{
    int  ex, line, count;
    register gpr_$pixel_value_t *foop;
    register int *ix_table;
                        
    ix_table = ix_inc_table;
    foop = foo;
    for (ex=0;  ex<eWIDTH;  ex++)
       {
        *foop++ = gray[raster[*ix_table++]];
       }

    gpr_$acquire_display(&status);
                       
    line = interleavetable[gifline];
    for (count = 0; count < scanlineused[line]; count +=1)
       {
        destination_window.window_base.y_coord = scanlinetable[line] + count;
        gpr_$write_pixels(foo, destination_window, &status);
       } 
    gpr_$release_display(&status);
}

/***********************************/
void draw_color()
{
    register int x;
    int count, line;
    register gpr_$pixel_value_t *foop;
    unsigned char *rast;
                        
    foop = foo;
    rast = raster;
    for(x=0;x<WIDTH; x++)
      {
       *foop++ = (gpr_$pixel_value_t) *rast++;
      }
    
    gpr_$acquire_display(&status);
    destination_window.window_base.y_coord = interleavetable[gifline];
    gpr_$write_pixels(foo, destination_window, &status);                
    gpr_$release_display(&status);
/* fprintf(stdout,"draw_color being used.\n");  */
}  

/***********************************/
void draw_resized_color()
{
    int count, line;
    register gpr_$pixel_value_t *foop;
    register int *ix_table;
    register int ex;
                    
    ix_table = ix_inc_table;
    foop = foo;
    for (ex=0;  ex<eWIDTH;  ex++)
       {
        *foop++ = (gpr_$pixel_value_t) raster[*ix_table++];
       }

    gpr_$acquire_display(&status);
    
    line = interleavetable[gifline];
    for (count = 0; count < scanlineused[line]; count +=1)
       {
        destination_window.window_base.y_coord = scanlinetable[line] + count;
        gpr_$write_pixels(foo, destination_window, &status);
     } 
    gpr_$release_display(&status);
}

/***********************************/
void KbEnable() 
{
  gpr_$event_t     ev_type;
  gpr_$position_t  ev_pos;
  unsigned char    ev_char;

  gpr_$enable_input (gpr_$keystroke, keys, &status);
  check("in KbEnable after calling gpr_$enable_input");
  gpr_$event_wait (&ev_type, &ev_char, &ev_pos, &status);

  if (ev_char == 's') SaveImage();
  if (ev_char == 'S') SaveImage();
  if (ev_char == 'q') exit(0);
  if (ev_char == 'Q') exit(0);

  gpr_$disable_input (gpr_$keystroke, &status);
}

/***********************************/
void Close() 
{
  int i=0;                                            
                                                      
  gpr_$terminate (false, &status);                    
  while(status.fail && i<10)                          
   {                                                  
    i++;                                              
    gpr_$terminate (false, &status);                  
    printf("attempting to terminate i= %d\n",i);      
   }                                                  
  free(codetable);
  free(interleavetable);
  free(ix_inc_table);
  free(foo);
  free(raster);
  free(scanlinetable);
  free(scanlineused);
}

/***********************************/
void reset_color_map()
{
  gpr_$acquire_display(&status);
  gpr_$set_color_map((gpr_$pixel_value_t) 0,(short) (1<<apollobits), color_map, &status);
  check("in reset_color_map after calling gpr_$set_color_map");
  gpr_$release_display(&status);
}


/***********************************/
void main(argc,argv)
int argc;
char **argv;
{
   get_options(argc,argv);
   Initialize();
   save_color_map();
   pfm_status = pfm_$cleanup(&cl_rec);    /*** set up the cleanup handler ***/
   if (pfm_status.all != pfm_$cleanup_set)
     {     /*** we're here because of a fault, do our clean-up work ***/
           /***  restore old color map ***/
      reset_color_map();
      Close();
       
      if (pfm_status.all = status_$ok)  pgm_$exit;
        else pfm_$signal(pfm_status);    /*** quit ***/
     }  

   do
     {
      filelist_name = argv[optind];
      filelist = fopen(argv[optind],"r");
      optind++;
      if (!filelist) fprintf(stdout,"%s  : couldn't open input file %s \n",progname,argv[optind]);
     } while (!filelist & (optind < argc));

/***** start of file loop *****/
   while (filelist && (optind < argc + 1))
     {
      do 
        { 
         if (listofgifs)
           {
            do 
              { 
               endoflist = fscanf(filelist,"%s",file_name) == EOF;
               if( endoflist & infinite )
                 {
                  rewind(filelist);
                  fscanf(filelist,"%s",file_name);
                  endoflist = false;
                 }
               if (!endoflist)
                 {
                  infile = fopen(file_name,"r");
                  if (!infile)
                    {
                     fprintf(stdout,"%s  : couldn't open file :%s:\n",progname,file_name);
                     perror("error text: ");
                    }
                 }
              } while (!infile & !endoflist);
           }
         else
           {
            infile = filelist;
            file_name = filelist_name;
           }
        
         if (infile && !endoflist)
           {
            firsttimeshown = true;

            abort = setjmp(env);
            if (abort == false)
               {
                process_gif_file();
               }

            if (userfeedback & showgif)
              {
               /*   Establish the refresh function. Once established, if the window needs to be                */
               /*   refreshed as the result of a pop or grow, GPR will automatically call process_gif_file. */
               gpr_$set_refresh_entry (rwin,rhdm, &status);                                             
               check("called gpr_$set_refresh_entry");                                                  
     
               gpr_$inq_cursor (&curs_pat_desc, curs_raster_op, &curs_active, &curs_position, &curs_origin,&status);
               if(!curs_active) gpr_$set_cursor_active(true,&status); /* disable the cursor */
        
               KbEnable();                                                                              
               gpr_$set_refresh_entry (rhdm,rhdm, &status);                                             
           
               gpr_$inq_cursor (&curs_pat_desc, curs_raster_op, &curs_active, &curs_position, &curs_origin,&status);
               if(curs_active) gpr_$set_cursor_active(false,&status); /* disable the cursor */
               fclose(infile);  
              }
            if (save_to_bitmap & showgif) SaveImage();
            /* color code used to display b&w gifs, this resets it to monocrome afterwords. */
            if (hi_plane == 0) monocrome = true;
           }

          fclose(infile);
        } while (listofgifs & !endoflist);

      filelist = NULL;
      if (optind < argc)
        {
         do
           {
            filelist_name = argv[optind];
            filelist = fopen(filelist_name,"r");
            optind++;
            if (!filelist) fprintf(stdout,"%s  : couldn't open input file %s \n",progname,filelist_name);
           } while (!filelist & (optind < argc));
        }
     }
/***** file loop end *****/

   reset_color_map();                                                                     
/*   we've done the work, blow off the clean-up handler  */
   pfm_$rls_cleanup(&cl_rec,&status);                                                    
   Close();                         
}

/***********************************/
void Initialize()
{ 
gpr_$display_mode_t  mode = gpr_$direct;
short int KBD_$CR = 0x96;

  lib_$init_set(keys, (short)256);
  lib_$add_to_set(keys, (short)256, ' ');
  lib_$add_to_set(keys, (short)256, 's');
  lib_$add_to_set(keys, (short)256, 'S');
  lib_$add_to_set(keys, (short)256, 'q');
  lib_$add_to_set(keys, (short)256, 'Q');
  lib_$add_to_set(keys, (short)256, KBD_$CR);

   gpr_$inq_cursor (&curs_pat_desc, curs_raster_op, &curs_active, &curs_position, &curs_origin,&status);
   if(curs_active) gpr_$set_cursor_active(false,&status); /* disable the cursor */

   pad_$isa_dm_pad(unit, &status);
   if (status.all) fatal("");

   gpr_$inq_disp_characteristics(mode,unit,disp_len,&display_characteristics,&disp_len_returned,&status);
   check("in Initialize after calling gpr_$inq_display_characteristics");
   
   screenwidth = display_characteristics.x_visible_size;
   screenheight = display_characteristics.y_visible_size;

   apollobits = display_characteristics.n_planes;
   hi_plane  = display_characteristics.n_planes - 1;
   if ( (hi_plane > 0) & !monocrome ) 
     {
      mono4color1 = 1;
     }
   else
     {
      monocrome = true;
      mono4color1 = 4;
     }

    file_name = (char*)malloc(255);
    if (!file_name) fatal("not enough memory for file_name");
    codetable = (codetype*)malloc(4096*sizeof(codetype));
    if (!codetable) fatal("not enough memory for code table");
    foo = (gpr_$pixel_value_t *) malloc(screenwidth * mono4color1 * sizeof(gpr_$pixel_value_t));
    if (!foo) fatal("in Initialize: not enough memory for scan line buffer");
    interleavetable = (int*)malloc(screenheight*sizeof(int));
    if (!interleavetable) fatal("not enough memory for interleave table");
    ix_inc_table = (int *) malloc((screenwidth) * sizeof(int)); 
    if (!ix_inc_table) fatal("in Initialize: not enough memory for scan line table");
    scanlineused = (int*)malloc(2048 * sizeof(int));
    if (!scanlineused) fatal("not enough memory for scanlineused table");
    scanlinetable = (int*)malloc(2048 * sizeof(int));
    if (!scanlinetable) fatal("not enough memory for scanline table");
    raster = (unsigned char*)malloc(screenwidth);
    if (!raster) fatal("not enough memory for image");
                      
    pad_window.top    = 0;
    pad_window.left   = (screenwidth >> 1) - 5;
    pad_window.width  = 20;
    pad_window.height = 20; 

  pad_$create_window((char *)0,(short)0,pad_$transcript,(short)1,pad_window,&graphics_stream,&status);

  unit = graphics_stream ;

  pad_$set_full_window(unit,(short) 1,&pad_window,&status);
  pad_$set_auto_close(unit, (short) 1, true, &status );
  pad_$set_border (unit,(short) 1, false, &status);
  pad_$set_scale (unit,(short) 1,(short) 1, &status);

  init_size.x_size = 2048;
  init_size.y_size = 2048;
  gpr_$init(mode,unit, init_size, hi_plane, &display_bitmap_desc, &status);
  check("in Initialize after calling gpr_$init"); 

  gpr_$load_font_file(font_name,font_name_size, &font_id, &status);
  gpr_$set_text_font(font_id, &status);       
  gpr_$inq_text_extent ( "Hy", 2, &textsize, &status);
  nameheight = textsize.y_size + 10;
}

/***********************************/
void usage()
{
    fprintf(stderr,"usage: %s [-f] [-u] [-t seconds] [-l] [-i] [-n] [-s] [-p] [-o] file_names...\n",progname);
    fprintf(stderr,"\noptions                   description \n\n");
    fprintf(stderr,"  -f              Display image in full screen mode. \n");
    fprintf(stderr,"  -u              Turn off user feedback. \n");
    fprintf(stderr,"  -t seconds      Time delay between images. \n");
    fprintf(stderr,"  -l              file_name contains a list of gif filenames, not a gif file. \n");
    fprintf(stderr,"  -i              infinite loop of file_names. \n");
    fprintf(stderr,"  -n              Turn off banner. \n");
    fprintf(stderr,"  -s              Do not show gif, used with -p. \n");
    fprintf(stderr,"  -p              Print info about gif. \n");
    fprintf(stderr,"  -o              Save image to a GPR bitmap file file_name.gpr. \n");
    exit(0);
}

/***********************************/
void error(s)
char *s;
{
  fprintf(stderr,"%s: %s in file %s\n",progname,s,file_name);
}

/***********************************/
void fatal(s)
char *s;
{
  fprintf(stderr,"%s: %s error in file %s\n",progname,s,file_name);
  longjmp(env, false);
}

/***********************************/
void SaveImage()
/* gpr_$bitmap_desc_t  source_bitmap_desc; */
/* char                *file_name; */
{
    status_$t                       status;
    gpr_$bmf_group_header_array_t   header; 
    gpr_$color_vector_t             color_map;
    gpr_$version_t                  version;
    short int                       groups;
    gpr_$attribute_desc_t           attribs;  
    gpr_$window_t                   source_bitmap;     
    gpr_$bitmap_desc_t              disk_bitmap_desc;
    gpr_$offset_t                   source_bitmap_size;    
    gpr_$rgb_plane_t                hi_plane;
    gpr_$position_t                 disk_bitmap_origin , source_bitmap_origin;
    boolean                         disk_bmf_created;
  static char dotgpr[] = ".gpr";  
  int colorbits;
  int name_len;


  if (monocrome | four_bit_color) colorbits = mono4bits;
  else if (local) colorbits = localbits;
  else colorbits = globalbits;

       gpr_$acquire_display (&status);

       gpr_$inq_cursor (&curs_pat_desc, curs_raster_op, &curs_active, &curs_position, &curs_origin,&status);
       if(curs_active) gpr_$set_cursor_active(false,&status); /* disable the cursor */

/* set the current bitmap the source bitmap */
       gpr_$set_bitmap(display_bitmap_desc,&status);
 check("in SaveImage");

/* get the size and the highest plane id of the source bitmap */
       gpr_$inq_bitmap_dimensions(display_bitmap_desc,&source_bitmap_size,&hi_plane,&status);  
       check("in SaveImage");

/* get the position of the upper left corner of the source bitmap */
/*       gpr_$inq_bitmap_position(display_bitmap_desc,source_bitmap_origin,&status) ;
         check("in SaveImage");
         fprintf(stdout,"source_bitmap_origin.x_coord=%d source_bitmap_origin.y_coord=%d \n",source_bitmap_origin.x_coord,source_bitmap_origin.y_coord);
*/
       gpr_$inq_color_map ((gpr_$pixel_value_t) 0,(short) 256,color_map,&status);   
       check("in SaveImage");

      header[0].n_sects = colorbits;  /* # of sections in a group */
      header[0].pixel_size = 1;         /* # of bits per pixel in each section of a group */
      header[0].allocated_size  = 0;
      header[0].bytes_per_line  = 0;    /* # of bytes in one row of a bitmap. If =0, GPR takes care of it automatically. */
      header[0].bytes_per_sect  = 0;
      header[0].storage_offset  = 0;
      groups = (short) 1;
      version.gpr_$major =(short) 1;
      version.gpr_$minor =(short) 1;
/*
    source_bitmap.window_base.x_coord = source_bitmap_origin.x_coord;
    source_bitmap.window_base.y_coord = source_bitmap_origin.y_coord;
*/
    source_bitmap.window_base.x_coord = 0;
    source_bitmap.window_base.y_coord = 0;
    source_bitmap.window_size.x_size  = source_bitmap_size.x_size;
    source_bitmap.window_size.y_size  = source_bitmap_size.y_size; 
     
    gpr_$allocate_attribute_block(&attribs, &status);  

/* save the whole thing to disk */
    strcpy(buf, file_name);
    name_len = strlen(buf);
    if (!strncmp(buf + name_len -4,".gif",4)) buf[name_len -4] = 0;
    strcat(buf, dotgpr);
    gpr_$open_bitmap_file(gpr_$create, buf, (short)strlen(buf),
       &version, &source_bitmap_size, &groups,header, attribs, &disk_bitmap_desc, &disk_bmf_created, &status);  
    check("in SaveImage");

/* set the current bitmap the disk bitmap */
    gpr_$set_bitmap(disk_bitmap_desc,&status); 
    check("in SaveImage");
  
/* set the color map in the disk bitmap file to the color map the application used. You can change the color map here */
    gpr_$set_bitmap_file_color_map (disk_bitmap_desc,(short) 0,(short) (1<<colorbits),color_map,&status);
    check("in SaveImage");

     disk_bitmap_origin.x_coord = 0;
     disk_bitmap_origin.y_coord = 0;

/* now just move the pixels from the source bitmap to the disk bitmap file */
    gpr_$pixel_blt(display_bitmap_desc,source_bitmap,disk_bitmap_origin,&status);
    check("in SaveImage");

/* set current bitmap back to the source bitmap */
    gpr_$set_bitmap(display_bitmap_desc, &status);    
    check("in SaveImage");

/* the disk bitmap file can be unlocked and available for print */
    gpr_$deallocate_bitmap(disk_bitmap_desc,&status);
    check("in SaveImage");

    gpr_$deallocate_attribute_block(attribs, &status);  
    check("in SaveImage");

  if ( hi_plane > 0 ) 
     {
      fprintf(stdout,"The bitmap file %s has been created. The source bitmap has %d planes. \n",buf,colorbits);
     }
  else
     {
      fprintf(stdout,"The bitmap file %s has been created. The source bitmap has %d plane.  \n",buf,colorbits);
     }

    if(curs_active) gpr_$set_cursor_active(true,&status);

    gpr_$release_display(&status) ;
}
