/*   dsbmf.c  - displays a GPR bitmap file on the apollos
 *
 *                 Version 1.03   04/04/1991
 *
 *  This file contains system-specific graphics routines.
 *  These routines use Apollo Graphics Primitive Resource (GPR) calls.
 *
 *  Authors:   Roque D. Oliveira      oliveria@caen.engin.umich.edu
 * 
 * 
 *  This program was derived from a Fortran program available in the ADUS tape
 * 
 *
 */

/* Changes from Version 1.00
 *
 * - support for frame mode (-f option) added in Version 1.01 , 04/10/1990 
 * - checked for header[0].pixel_size == 1 when reading bitmap file, Version 1.02 , 12/21/1990 
 * - added more entries in print_display_info() 04/04/1991
 *
 */

#include <stdio.h>
#include <string.h>
#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/pad.h>
#include <apollo/time.h>
#include <apollo/tone.h>
#include <apollo/error.h>

char *progname;				/* the name of this program */

status_$t            status;
gpr_$color_vector_t  display_color_map; 
gpr_$rgb_plane_t     hi_plane;
gpr_$disp_char_t     display_characteristics;

static short int     disp_len = sizeof(gpr_$disp_char_t);
       short int     disp_len_returned;

gpr_$offset_t        init_size;
gpr_$bitmap_desc_t   display_bitmap_desc;
gpr_$display_mode_t  mode=gpr_$borrow;
stream_$id_t         unit;

gpr_$attribute_desc_t           attribs;
short int                       groups = (short) 1;
gpr_$version_t                  version;
gpr_$offset_t                   disk_bm_size;
gpr_$bmf_group_header_array_t   header;
gpr_$bitmap_desc_t              disk_bm_desc;
gpr_$color_vector_t             disk_color_map;
boolean                         disk_bm_created;
unsigned int                    lcolormap;
short int                       n_colors;

gpr_$window_t                   source_bitmap;     
gpr_$position_t                 dest_bitmap_origin;
          
unsigned short WIDTH, HEIGHT;   /* image width & height */

#define VERSION    "1"
#define PATCHLEVEL "03"

unsigned int  ldirect   = 0;
unsigned int  lframe    = 0;
unsigned int  lupperleft= 0;
unsigned int  lstat     = 0;
unsigned int  linfo     = 0;
unsigned int  lcursor   = 0;
unsigned int  lbeep     = 0;
unsigned int  lversion  = 0;
unsigned int  lquiet    = 0;
unsigned int  lpause    = 0;
unsigned int  lexec     = 0;
unsigned int  lDebug    = 0;

unsigned int lblackwhite = 0;

void usage()
  {
    (void) fprintf(stderr,"usage: %s [-d | -f] [-s | -q] [-i] [-c] [-b] [-v] [-e prog] [-p secs] file1 [file2 ... filen] \n",progname);
    (void) fprintf(stderr,"\noptions                   description \n\n");
    (void) fprintf(stderr,"  -d          display bitmap file(s) in Direct mode \n");
    (void) fprintf(stderr,"  -f          display bitmap file(s) in Frame mode \n");
    (void) fprintf(stderr,"  -s          only display Statistics about bitmap file(s) \n");
    (void) fprintf(stderr,"  -i          display Information about the display. No bitmap file(s) required. \n");
    (void) fprintf(stderr,"  -c          show Cursor after each bitmap file is displayed \n");
    (void) fprintf(stderr,"  -b          Beep after each bitmap file is displayed \n");
    (void) fprintf(stderr,"  -u          display bitmap file(s) at the Upper-left corner \n");
    (void) fprintf(stderr,"  -v          show Version and patchlevel \n");
    (void) fprintf(stderr,"  -q          Quiet. Do not print informational messages while displaying bitmap file(s) \n");
    (void) fprintf(stderr,"  -e prog     Execute an external program (or system command) after each bitmap file is displayed. Caution with its option. \n");
    (void) fprintf(stderr,"  -p secs     Pause by secs seconds before displaying the next bitmap file \n");
    exit(-1);
  }

void check(messagex)
char *messagex;
  {
     if (status.all)
       { 
        error_$print (status);
        (void) fprintf(stderr,"Error occurred while %s.\n", messagex);
        exit(-1);
       }    
  }

void pause(t)
double t;
{
time_$clock_t  time_to_wait;
    
   time_to_wait.high16 = 0;
   time_to_wait.low32  = 250000 * t;
   time_$wait (time_$relative, time_to_wait, &status);
}
   
void beep(t)
float     t;
{
time_$clock_t  time_to_beep;
    
   time_to_beep.high16 = 0;
   time_to_beep.low32  = 250000 * t;
   tone_$time(time_to_beep);
}

/******************************************************************************/

main(argc, argv)
int argc;
char **argv;
{
 int i , c ;
 double t_secs , atof(); 
 char *bitmap_file_name;
 char *prog_to_exec;
 extern int	 optind;             /* index of which argument is next */
 extern char *optarg;            /* pointer to argument of this option */ 

 progname = strrchr(argv[0], '/');
 if (progname)
    progname++;
 else
    progname = argv[0];
                           
 while ((c=getopt(argc,argv,"dfsicburvqDe:p:")) != EOF) 
 switch (c)
       {
        case 'd':                  /* display bitmaps in direct mode (default is borrow mode) */
                 ldirect = 1;      
                 break;
        case 'f':                  /* display bitmaps in frame mode (default is borrow mode) */
                 lframe = 1;      
                 break;
        case 's':                  /* only display statistics about the bitmap */
                 lstat = 1;        
                 break;
        case 'i':                  /* display information about the display. No bitmap required. */
                 linfo = 1;        
                 break;
        case 'c':                  /* show cursor once bitmap has been displayed */
                 lcursor = 1;      
                 break;
        case 'b':                  /* beep once bitmap has been displayed */
                 lbeep = 1;      
                 break;
        case 'u':                  /* display the bitmap at the upper-left corner (default is centered) */
                 lupperleft = 1;      
                 break;
        case 'v':                  /* show version number and patchlevel of this program */
                 lversion = 1;     
                 break;
        case 'q':                  /* do not write any informational messages to the display */
                 lquiet = 1;       
                 break;
        case 'D':                  /* undocumented -- for debugging */
                 lDebug = 1;
                 break;
        case 'p':                  /* pause between pictures */
                 lpause = 1;
                 t_secs = atof(optarg);
                 break;
        case 'e':                  /* execute an external program (or system command) after bitmap is displayed */
                 lexec = 1;
                 prog_to_exec = optarg;
                 break;
        case '?':
        default :
                 usage();
       }

 if ((argc - optind < 1) && !linfo)
   {
     usage();
   }

 if (lDebug)
   {
    (void) printf("ldirect =%d \n",ldirect );
    (void) printf("lframe =%d \n",lframe );
    (void) printf("lstat =%d \n",lstat );
    (void) printf("linfo =%d \n",linfo );
    (void) printf("lcursor =%d \n", lcursor);
    (void) printf("lbeep =%d \n", lbeep);
    (void) printf("lupperleft =%d \n", lupperleft);
    (void) printf("lversion =%d \n", lversion);
    (void) printf("lquiet =%d \n", lquiet);
    (void) printf("lexec =%d \n", lexec);
    if(lexec) (void) printf("prog_to_exec=%s \n",prog_to_exec);
    (void) printf("lpause =%d \n", lpause);
    if(lpause) (void) printf("t_secs=%f \n",t_secs);
    (void) printf("argc =%d \n",argc );
    (void) printf("optind =%d \n",optind );
   }

 if (lstat && lquiet)
  {
   (void) fprintf(stderr,"-s and -q are mutually exclusive \n");
   exit(-1);
  }

 if (ldirect && lframe)
  {
   (void) fprintf(stderr,"-d and -f are mutually exclusive \n");
   exit(-1);
  }

 if (!lquiet && lversion)
  {
   (void) fprintf(stdout,"%s version %s patchlevel %s by Roque D. Oliveira, oliveria@caen.engin.umich.edu \n",progname,VERSION, PATCHLEVEL);
  }


 if (lstat)           
  {           
   mode = gpr_$direct;  /* should be gpr_$no_display but there is a bug in sr10.1 (not 10.2) */        
  }
 else
  {
   if (ldirect) mode = gpr_$direct; 
   if (lframe)  mode = gpr_$frame; 
  }
   
 gpr_$inq_disp_characteristics(mode,(short) 1,disp_len,&display_characteristics,&disp_len_returned,&status);
 check("in main after calling gpr_$inq_display_characteristics");

 if ( display_characteristics.n_buffers < 0 )
  {
   (void) fprintf(stderr,"This device does not have a display. \n");
    exit(-1);
  } 

 hi_plane  = display_characteristics.n_planes - 1; 
    
 init_size.x_size = display_characteristics.x_visible_size;   /* x dimension of visible screen area in pixels */ 
 init_size.y_size = display_characteristics.y_visible_size;   /* y dimension of visible screen area in pixels */ 
             
 if (lframe)       /* set initial size to that of a large display */
  {
   init_size.x_size = 1280;
   init_size.y_size = 1024;
  }

 if (linfo)
  {
   print_display_info();
   if (argc - optind < 1) exit(0);
  }

 gpr_$init(mode, (short) 1, init_size, hi_plane, &display_bitmap_desc, &status);
 check("in main after calling gpr_$init");
                           


 for (i = optind ; i < argc ; i++)    /* repeat for each input file */
 {                           
   bitmap_file_name = argv[i];

   inq_bitmap_info(bitmap_file_name);

   if (lstat && !lquiet) 
    {
     print_stat(bitmap_file_name);
     continue;
    }

   if (ldirect || lframe) init_pad(mode,WIDTH,HEIGHT); 

   gpr_$allocate_attribute_block(&attribs, &status);  
   gpr_$open_bitmap_file(gpr_$readonly,bitmap_file_name,(short) strlen(bitmap_file_name),&version,
       &disk_bm_size, &groups,header, attribs, &disk_bm_desc, &disk_bm_created, &status);  
   check("in main, after calling gpr_$open_bitmap_file");

   if (!lframe)
    {
     gpr_$wait_frame(&status) ;
     check("in main after calling gpr_$wait_frame"); 
    }

   if (lcolormap || lblackwhite)
    {
     save_color_map(); 
     set_color_map(n_colors);
    }
 
   source_bitmap.window_base.x_coord = 0;
   source_bitmap.window_base.y_coord = 0;
   source_bitmap.window_size.x_size  = disk_bm_size.x_size;
   source_bitmap.window_size.y_size  = disk_bm_size.y_size;
    
   if (lupperleft || ldirect || lframe)
    {
     dest_bitmap_origin.x_coord = 0;
     dest_bitmap_origin.y_coord = 0;
    }
   else
    {
     dest_bitmap_origin.x_coord = (display_characteristics.x_visible_size - WIDTH)/2; 
     dest_bitmap_origin.y_coord = (display_characteristics.y_visible_size - HEIGHT)/2; 
    }
    
/* now just move the pixels from the disk bitmap to the display */
   gpr_$pixel_blt(disk_bm_desc,source_bitmap,dest_bitmap_origin,&status);
   check("in main after calling gpr_$pixel_blt"); 
        
   gpr_$deallocate_bitmap(disk_bm_desc,&status);
   check("in main 2 after calling gpr_$deallocate_bitmap ");
   gpr_$deallocate_attribute_block(attribs, &status);  
   check("in main 2 after calling gpr_$deallocate_attribute_block ");
       
   if (lexec)
    {
     gpr_$release_display(&status);
     exec_external_program(prog_to_exec); 
     gpr_$acquire_display(&status);
    }

   if (lcursor && !lpause) gpr_$set_cursor_active(true,&status);
   if (!lpause && lbeep) beep(0.15);
   if (!lpause) KbEnable();

   if (lpause) pause(t_secs);

   if (lcolormap || lblackwhite)
   {  
     reset_color_map();
   }
             
    if (ldirect || lframe)
     {   
      gpr_$release_display(&status) ;
      gpr_$terminate (true, &status);
      stream_$close (&unit, &status);
      check("in main, after calling stream_$close");
      gpr_$init(mode,(short) 1, init_size, hi_plane, &display_bitmap_desc, &status);
      check("in main, after calling gpr_$init with gpr_$direct or gpr_$frame");
     }
    else
     {
      if (lcursor) gpr_$set_cursor_active(false,&status);
      gpr_$clear ((gpr_$pixel_value_t) -2, &status);
     } 
    
 } /* end for */
 
 gpr_$terminate (false, &status);
 check("in main, after calling gpr_$terminate");

} /* end main */


/******************************************************************************/
inq_bitmap_info(bitmap_file_name)
char  *bitmap_file_name;
{
   gpr_$acquire_display(&status);
   gpr_$allocate_attribute_block(&attribs, &status);  
   gpr_$open_bitmap_file(gpr_$readonly,bitmap_file_name,(short) strlen(bitmap_file_name),&version,
       &disk_bm_size, &groups,header, attribs, &disk_bm_desc, &disk_bm_created, &status);  
   check("in inq_bitmap_info, after calling gpr_$open_bitmap_file");

   WIDTH  = disk_bm_size.x_size;
   HEIGHT = disk_bm_size.y_size;

   if (WIDTH > display_characteristics.x_visible_size)
    {
     WIDTH = display_characteristics.x_visible_size;
     if (!lstat && !lquiet) (void) fprintf(stdout,"image width  too large, using width  = %d pixels\n",WIDTH);
    }
   if (HEIGHT > display_characteristics.y_visible_size)
    {
     HEIGHT = display_characteristics.y_visible_size;
     if (!lstat && !lquiet) (void) fprintf(stdout,"image height too large, using height = %d pixels\n",HEIGHT);
    }

   lblackwhite    =  0;

   if (header[0].n_sects == 1 && header[0].pixel_size == 1) 
    {  
     disk_color_map[0] =  gpr_$black ;
     disk_color_map[1] =  gpr_$white ;
     lcolormap      =  0;
     lblackwhite    =  1;
     n_colors       =  2;
     if(!lstat && !lquiet) (void) fprintf(stdout,"  Internal Colormap : false . Using Black and White. \n");
    }
   else
    {
      n_colors     = 256;
      gpr_$inq_bitmap_file_color_map(disk_bm_desc, (short) 0,(short) 256, disk_color_map, &status);   
      if ((status.all == gpr_$no_color_map_in_file) || (status.all == gpr_$invalid_color_map) )
        {        
         lcolormap    =  0;
         if(!lstat && !lquiet) (void) fprintf(stdout,"  Internal Colormap : false . Using this display's color map. \n");
        }
      else  
        {
         check("in inq_bitmap_info, after calling gpr_$inq_bitmap_file_color_map");
         lcolormap    =  1;
         if(!lstat && !lquiet) (void) fprintf(stdout,"  Internal Colormap : true \n");
        }
    } 

   gpr_$deallocate_bitmap(disk_bm_desc,&status);
   check("in inq_bitmap_info, after calling gpr_$deallocate_bitmap ");
   gpr_$deallocate_attribute_block(attribs, &status);  
   check("in inq_bitmap_info, after calling gpr_$deallocate_attribute_block ");
   gpr_$release_display(&status);
}

/******************************************************************************/

init_pad(mode,w,h)
gpr_$display_mode_t  mode;
unsigned short w,h;
{ 
  pad_$window_desc_t   pad_window ; 

  if (lupperleft)
    {
     pad_window.top    = 0;
     pad_window.left   = 0;
    }
  else
    { 
     pad_window.top    = (display_characteristics.y_visible_size - HEIGHT)/2; 
     pad_window.left   = (display_characteristics.x_visible_size - WIDTH )/2;
    } 
  pad_window.width  = w;
  pad_window.height = h; 

  gpr_$terminate (true, &status);
  check("in init_pad, after calling gpr_$terminate");

  pad_$create_window((char *)0,(short)0,pad_$transcript,(short)1,pad_window,&unit,&status);
  check("in init_pad after calling pad_$create_window");

  pad_$set_full_window(unit,(short) 1,&pad_window,&status);
  pad_$set_auto_close(unit, (short) 1, true, &status );
  pad_$set_border (unit,(short) 1, false, &status);
  pad_$set_scale (unit,(short) 1,(short) 1, &status);

  gpr_$init(mode,unit, init_size, hi_plane, &display_bitmap_desc, &status);
  check("in init_pad, after calling gpr_$init");
  gpr_$set_auto_refresh(true, &status);
  gpr_$acquire_display(&status);

}

/******************************************************************************/

set_color_map(n_colors) 
short int     n_colors;
{
  if (lDebug) (void) fprintf(stdout,"in set_color_map, n_colors=%d \n",n_colors);
  gpr_$acquire_display(&status);
  gpr_$set_color_map((gpr_$pixel_value_t) 0,n_colors, disk_color_map, &status);
  check("in set_color_map after calling gpr_$set_color_map");
  gpr_$release_display(&status);
}

/******************************************************************************/

save_color_map() 
{
  if (lDebug) (void) fprintf(stdout,"in save_color_map \n");
  gpr_$acquire_display(&status);
  gpr_$inq_color_map((gpr_$pixel_value_t) 0,(short) 256, display_color_map, &status);
  check("in save_color_map after calling gpr_$inq_color_map");
  gpr_$release_display(&status);
}

/******************************************************************************/

reset_color_map()
{
  if (lDebug) (void) fprintf(stdout,"in reset_color_map \n");
  gpr_$acquire_display(&status);
  gpr_$set_color_map((gpr_$pixel_value_t) 0,(short) 256, display_color_map, &status);
  check("in reset_color_map after calling gpr_$set_color_map");
  gpr_$release_display(&status);
}

/******************************************************************************/
     
KbEnable() 
{
  gpr_$keyset_t    keys;
  gpr_$event_t     ev_type;
  gpr_$position_t  ev_pos;
  unsigned char    ev_char;
  short int        KBD_$CR=0x96;

  if (lDebug) (void) fprintf(stdout,"in KbEnable \n");
  lib_$init_set(keys, (short)256);
  lib_$add_to_set(keys, (short)256, ' ');
  lib_$add_to_set(keys, (short)256, 'q');
  lib_$add_to_set(keys, (short)256, 'Q');
  lib_$add_to_set(keys, (short)256, KBD_$CR);
  gpr_$enable_input (gpr_$keystroke, keys, &status);
  check("in KbEnable after calling gpr_$enable_input");
  gpr_$event_wait (&ev_type, &ev_char, &ev_pos, &status);
}

/******************************************************************************/

print_stat(bitmap_file_name)
char   *bitmap_file_name;
{
 (void) fprintf(stdout,"\nBitmap file name = %s \n",bitmap_file_name);
 (void) fprintf(stdout,"Image Characteristics    \n");
 (void) fprintf(stdout,"  Image Size        : %d  %d \n",disk_bm_size.x_size,disk_bm_size.y_size);
 (void) fprintf(stdout,"  Number of groups  : %d \n",groups);
 (void) fprintf(stdout,"  Number of planes  : %d \n",header[0].n_sects);
 (void) fprintf(stdout,"  Pixel size        : %d \n",header[0].pixel_size);
 (void) fprintf(stdout,"  Allocated Size    : %d \n",header[0].allocated_size);
 (void) fprintf(stdout,"  Bytes per line    : %d \n",header[0].bytes_per_line);
 (void) fprintf(stdout,"  Bytes per section : %d \n",header[0].bytes_per_sect);

   if (lcolormap)
   {
    (void) fprintf(stdout,"  Internal Colormap : true \n");
   }
   else
   {
    (void) fprintf(stdout,"  Internal Colormap : false \n");
   }
}

/******************************************************************************/

print_display_info()
{
(void) fprintf(stdout,"\nDisplay Characteristics \n");
(void) fprintf(stdout,"  controller_type         =%d  /* type of graphics controller */ \n",display_characteristics.controller_type  );
(void) fprintf(stdout,"  accelerator_type        =%d  /* type of graphics accelerator */  \n",display_characteristics.accelerator_type  );
(void) fprintf(stdout,"  x_window_origin         =%d  /* x origin of window screen area in pixels */  \n",display_characteristics.x_window_origin );
(void) fprintf(stdout,"  y_window_origin         =%d  /* y origin of window screen area in pixels */  \n",display_characteristics.y_window_origin );
(void) fprintf(stdout,"  x_window_size           =%d  /* x dimension of window screen area in pixels */  \n",display_characteristics.x_window_size  );
(void) fprintf(stdout,"  y_window_size           =%d  /* y dimension of window screen area in pixels */  \n",display_characteristics.y_window_size  );
(void) fprintf(stdout,"  x_visible_size          =%d  /* x dimension of visible screen area in pixels */  \n",display_characteristics.x_visible_size );
(void) fprintf(stdout,"  y_visible_size          =%d  /* y dimension of visible screen area in pixels */  \n",display_characteristics.y_visible_size );
(void) fprintf(stdout,"  x_extension_size        =%d  /* x dimension of maximum extended bitmap size in pixels */  \n",display_characteristics.x_extension_size  );
(void) fprintf(stdout,"  y_extension_size        =%d  /* y dimension of maximum extended bitmap size in pixels */  \n",display_characteristics.y_extension_size  );
(void) fprintf(stdout,"  x_total_size            =%d  /* x dimension of total buffer area in pixels */  \n",display_characteristics.x_total_size );
(void) fprintf(stdout,"  y_total_size            =%d  /* y dimension of total buffer area in pixels */  \n",display_characteristics.y_total_size );
(void) fprintf(stdout,"  x_pixels_per_cm         =%d  /* number of pixels in x dimension per centimeter */  \n",display_characteristics.x_pixels_per_cm );
(void) fprintf(stdout,"  y_pixels_per_cm         =%d  /* number of pixels in y dimension per centimeter */  \n",display_characteristics.y_pixels_per_cm );
(void) fprintf(stdout,"  n_planes                =%d  /* number of planes available */  \n",display_characteristics.n_planes  );
(void) fprintf(stdout,"  n_buffers               =%d  /* number of display buffers available */  \n",display_characteristics.n_buffers );
(void) fprintf(stdout,"  delta_x_per_buffer      =%d  /* relative displacement of buffers in x */  \n",display_characteristics.delta_x_per_buffer  );
(void) fprintf(stdout,"  delta_y_per_buffer      =%d  /* relative displacement of buffers in y */  \n",display_characteristics.delta_y_per_buffer  );
(void) fprintf(stdout,"  delta_planes_per_buffer =%d  /* relative displacement of buffers in depth */  \n",display_characteristics.delta_planes_per_buffer );
(void) fprintf(stdout,"  mem_overlaps            =%d  /* set of overlaps among classes of buffer memory */  \n",display_characteristics.mem_overlaps );
(void) fprintf(stdout,"  x_zoom_max              =%d  /* maximum pixel-replication zoom factor for x */  \n",display_characteristics.x_zoom_max  );
(void) fprintf(stdout,"  y_zoom_max              =%d  /* maximum pixel-replication zoom factor for y */  \n",display_characteristics.y_zoom_max  );
(void) fprintf(stdout,"  video_refresh_rate      =%d  /* refresh rate in hz */  \n",display_characteristics.video_refresh_rate  );
(void) fprintf(stdout,"  n_primaries             =%d  /* number of primary colors (1 -> monochrome; 3 -> color */  \n",display_characteristics.n_primaries );
(void) fprintf(stdout,"  lut_width_per_primary   =%d  /* number of bits in possible shortensity values per primary */  \n",display_characteristics.lut_width_per_primary  );
(void) fprintf(stdout,"  avail_formats           =%d  /* set of available shorteractive/imaging formats */  \n",display_characteristics.avail_formats  );
(void) fprintf(stdout,"  avail_access            =%d  /* set of available pixel sizes for direct access */  \n",display_characteristics.avail_access  );
(void) fprintf(stdout,"  access_address_space    =%d  /* number of 1kb pages of address space available for direct access */  \n",display_characteristics.access_address_space  );
(void) fprintf(stdout,"  invert                  =%d  /* INVert implemention */ \n",display_characteristics.invert );
(void) fprintf(stdout,"  num_lookup_tables       =%d  /* Number of color lookup tables */ \n",display_characteristics.num_lookup_tables  );
(void) fprintf(stdout,"  rgb_color               =%d  /* Modes for separate values for RGB */ \n",display_characteristics.rgb_color );
(void) fprintf(stdout,"  default_cursor_mode     =%d  /* type of cursor - software/hardware  */ \n",display_characteristics.default_cursor_mode );
(void) fprintf(stdout,"  avail_cursor_modes      =%d  /* cursors supported by device  */ \n",display_characteristics.avail_cursor_modes );
(void) fprintf(stdout,"  n_mult_clips            =%d  /* number of clip rects supported  */ \n\n",display_characteristics.n_mult_clips );
}

/******************************************************************************/

exec_external_program(program)    /* execute an external program (or system command) */
char                 *program;
{                                         
 system(program);
}

/******************************************************************************/

