/*********************************************
 *      GIF to Sun Rasterfile Converter      *
 *                                           *
 *      May 16, 1988  by Scott Hemphill      *
 *                                           *
 * I wrote this program, and hereby place it *
 * in the public domain, i.e. there are no   *
 * copying restrictions of any kind.         *
 *					     *
 * Converted from giftops into giftoras by:  *
 * Mic Kaczmarczik 6/01/88   		     *
 *********************************************/

#include <sys/types.h>
#include <netinet/in.h>
#include <stdio.h>
#include <rasterfile.h>
char *malloc();
int strncmp();

#define min(x,y) ((x) < (y) ? (x) : (y))
#define FALSE 0
#define TRUE 1

typedef int bool;
typedef struct codestruct {
            struct codestruct *prefix;
            unsigned char first,suffix;
        } codetype;

FILE *infile;
unsigned int screenwidth;           /* The dimensions of the screen */
unsigned int screenheight;          /*   (not those of the image)   */
bool global;                        /* Is there a global color map? */
int globalbits;                     /* Number of bits of global colors */
unsigned char globalmap[256][3];    /* RGB values for global color map */
unsigned char colortable[256 * 3];   /* RGB values for image */
unsigned char *raster;              /* Decoded image data */
codetype *codetable;                /* LZW compression code data */
int datasize,codesize,codemask;     /* Decoder working variables */
int clear,eoi;                      /* Special code values */

int encode = FALSE;		    /* Write out encoded Sun rasterfile? */
FILE *outfile;			    /* Our output file			 */
#define	ENCODER  "/usr/lib/rasfilters/convert.2"

void usage()
{
        fprintf(stderr,"usage: giftoras [-c] [GIF-file]\n");
        exit(-1);
}

void fatal(s)
char *s;
{
        fprintf(stderr,"giftoras: %s\n",s);
        exit(-1);
}

void checksignature()
{
        char buf[6];

        fread(buf,1,6,infile);
        if (strncmp(buf,"GIF",3)) fatal("file is not a GIF file");
        if (strncmp(&buf[3],"87a",3)) fatal("unknown GIF version number");
}

/* Get information which is global to all the images stored in the file */

void readscreen()
{
        unsigned char buf[7];

        fread(buf,1,7,infile);
        screenwidth = buf[0] + (buf[1] << 8);
        screenheight = buf[2] + (buf[3] << 8);
        global = buf[4] & 0x80;
        if (global) {
            globalbits = (buf[4] & 0x07) + 1;
            fread(globalmap,3,1<<globalbits,infile);
        }
}

/*
 * Initialize current image's color map by copying the triplets into
 * three arrays of red, green, and blue values.
 */

void initcolors(colortable,colormap,ncolors)
unsigned char colortable[256 * 3];
unsigned char colormap[256][3];
int ncolors;
{
	register int entry, rgb;
	register unsigned char *cp;

	cp = colortable;
	for (rgb = 0; rgb < 3 ; rgb++)
		for (entry = 0; entry < ncolors ; entry++)
			*(cp)++ = colormap[entry][rgb];
}

void writeheader(width,height,ncolors)
unsigned width,height,ncolors;
{
	struct rasterfile ras;
	int linewidth;

	linewidth = (width & 1) ? (width + 1) : width;

	ras.ras_magic	= htonl(RAS_MAGIC);
	ras.ras_width	= htonl(width);
	ras.ras_height	= htonl(height);
	ras.ras_depth	= htonl(8);
	ras.ras_length	= htonl(linewidth * height);	/* 1 byte per pixel */
        ras.ras_type	= htonl(RT_STANDARD);
	ras.ras_maptype	= htonl(RMT_EQUAL_RGB);
	ras.ras_maplength = htonl(ncolors * 3);

	fwrite(&ras, sizeof(ras), 1, outfile);
}

/* Output the bytes associated with a code to the raster array */

void outcode(p,fill)
register codetype *p;
register unsigned char **fill;
{
        if (p->prefix) outcode(p->prefix,fill);
        *(*fill)++ = p->suffix;
}

/* Process a compression code.  "clear" resets the code table.  Otherwise
   make a new code table entry, and output the bytes associated with the
   code. */

void process(code,fill)
register code;
unsigned char **fill;
{
        static avail,oldcode;
        register codetype *p;

        if (code == clear) {
            codesize = datasize + 1;
            codemask = (1 << codesize) - 1;
            avail = clear + 2;
            oldcode = -1;
        } else if (code < avail) {
            outcode(&codetable[code],fill);
            if (oldcode != -1) {
                p = &codetable[avail++];
                p->prefix = &codetable[oldcode];
                p->first = p->prefix->first;
                p->suffix = codetable[code].first;
                if ((avail & codemask) == 0 && avail < 4096) {
                    codesize++;
                    codemask += avail;
                }
            }
            oldcode = code;
        } else if (code == avail && oldcode != -1) {
            p = &codetable[avail++];
            p->prefix = &codetable[oldcode];
            p->first = p->prefix->first;
            p->suffix = p->first;
            outcode(p,fill);
            if ((avail & codemask) == 0 && avail < 4096) {
                codesize++;
                codemask += avail;
            }
            oldcode = code;
        } else {
            fatal("illegal code in raster data");
        }
}

/* Decode a raster image */

void readraster(width,height)
unsigned width,height;
{
        unsigned char *fill = raster;
        unsigned char buf[255];
        register bits=0;
        register unsigned count,datum=0;
        register unsigned char *ch;
        register int code;

        datasize = getc(infile);
        clear = 1 << datasize;
        eoi = clear+1;
        codesize = datasize + 1;
        codemask = (1 << codesize) - 1;
        codetable = (codetype*)malloc(4096*sizeof(codetype));
        if (!codetable) fatal("not enough memory for code table");
        for (code = 0; code < clear; code++) {
            codetable[code].prefix = (codetype*)0;
            codetable[code].first = code;
            codetable[code].suffix = code;
        }
        for (count = getc(infile); count > 0; count = getc(infile)) {
            fread(buf,1,count,infile);
            for (ch=buf; count-- > 0; ch++) {
                datum += *ch << bits;
                bits += 8;
                while (bits >= codesize) {
                    code = datum & codemask;
                    datum >>= codesize;
                    bits -= codesize;
                    if (code == eoi) goto exitloop;  /* This kludge put in
                                                        because some GIF files
                                                        aren't standard */
                    process(code,&fill);
                }
            }
        }
exitloop:
        if (fill != raster + width*height) fatal("raster has the wrong size");
        free(codetable);
}

/* Read a row out of the raster image and write it to the output file */
/* Pad the line to an even width if necessary, to match rasterfile format */

void rasterize(row,width)
int row,width;
{
        register unsigned char *scanline;
        register i;

        scanline = raster + row*width;
	fwrite(scanline, 1, width, outfile);
	if (width & 1)
		fputc(0, outfile);	/* pad to 16 bit boundaries */
}

/* write image trailer to standard output */

void writetrailer()
{
}

/* Read image information (position, size, local color map, etc.) and convert
   to a rasterfile. */

void readimage()
{
        unsigned char buf[9];
        unsigned left,top,width,height;
        bool local,interleaved;
        char localmap[256][3];
        int localbits;
	int ncolors;
        int *interleavetable;
        register row;
        register i;

        fread(buf,1,9,infile);
        left = buf[0] + (buf[1] << 8);
        top = buf[2] + (buf[3] << 8);
        width = buf[4] + (buf[5] << 8);
        height = buf[6] + (buf[7] << 8);
        local = buf[8] & 0x80;
        interleaved = buf[8] & 0x40;

	/*
	 * set up the color map for this image
	 */
        if (local) {
            localbits = (buf[8] & 0x7) + 1;
	    ncolors = 1 << localbits;
            fread(localmap,3,ncolors,infile);
            initcolors(colortable,localmap,ncolors);
        } else if (global) {
            initcolors(colortable,globalmap,ncolors = (1 << globalbits));
        } else {
            fatal("no colormap present for image");
        }

	/*
	 * read in the raster
	 */
        raster = (unsigned char*)malloc(width*height);
        if (!raster) fatal("not enough memory for image");
        readraster(width,height);

	/*
	 * write out the header and color table
	 */
        writeheader(width,height,ncolors);
	fwrite(colortable, ncolors, 3, outfile);

	/*
	 * write out the raster lines, a line at a time.
	 */

        if (interleaved) {
            interleavetable = (int*)malloc(height*sizeof(int));
            if (!interleavetable)
		    fatal("not enough memory for interleave table");
            row = 0;
            for (i = top; i < top+height; i += 8) interleavetable[i] = row++;
            for (i = top+4; i < top+height; i += 8) interleavetable[i] = row++;
            for (i = top+2; i < top+height; i += 4) interleavetable[i] = row++;
            for (i = top+1; i < top+height; i += 2) interleavetable[i] = row++;
            for (row = top; row < top+height; row++)
		    rasterize(interleavetable[row],width);
            free(interleavetable);
        } else {
            for (row = top; row < top+height; row++)
		    rasterize(row,width);
        }
        free(raster);
}

/* Read a GIF extension block (and do nothing with it). */

void readextension()
{
        unsigned char code,count;
        char buf[255];

        code = getc(infile);
        while (count = getc(infile)) fread(buf,1,count,infile);
}

extern char    *optarg;
extern int      optind;

main(argc,argv)
int argc;
char *argv[];
{
        int quit = FALSE, stat;
	int opt;
        char ch;
	FILE *encoder;

	while ((opt = getopt(argc, argv, "c")) != EOF) {
		switch (opt) {
		case 'c':
			encode = TRUE;
			break;
		default:
			usage();
			break;
		}
	}

	outfile = stdout;
	if (argv[optind] == NULL)
		infile = stdin;
	else if (argv[optind + 1] != NULL)
		usage();
	else {
		infile = fopen(argv[optind],"r");
		if (infile == NULL) {
			perror("giftoras");
			exit(-1);
		}
	}

	/*
	 * If we're to encode the output, use popen() to invoke the
	 * standard conversion program and insert it into the food chain.
	 */
	if (encode) {
		if ((encoder = popen(ENCODER, "w")) == NULL)
			fatal("can't popen() rasterfile encoder");
		outfile = encoder;
	}

        checksignature();
        readscreen();
        do {
            ch = getc(infile);
            switch (ch) {
                case '\0':  break;  /* this kludge for non-standard files */
                case ',':   readimage();
                            break;
                case ';':   quit = TRUE;
                            break;
                case '!':   readextension();
                            break;
                default:    fatal("illegal GIF block type");
                            break;
            }
        } while (!quit);

	if (encode && (stat = pclose(encoder)) != 0) {
		fprintf(stderr,"pclose: encoder returned status %d\n", stat);
		fatal("error in run-byte encoding process");
	}
}

