/*

$Log:	kbs_patch.c,v $
 * Revision 1.4  01/01/05  17:37:27  paul
 * Expand on note about running patched kernel (It is OK long term).
 * Fix minor spelling errors.
 * 
 * Revision 1.3  99/07/23  19:50:24  paul
 * Correct error message for not finding pattern.  The word 'find' was missing.
 * 
 * Revision 1.2  99/01/23  19:57:29  paul
 * Correct Version macro.
 * Minor error message tweak.
 * 
 * Revision 1.1  99/01/23  18:31:56  paul
 * Initial revision
 * 

    This program corrects a limitation in the Domain/OS kernel that
prevents SCSI drives with a capacity greater than 4GB (4*1024**3) from
being INVOL'd.  When attempting to INVOL such a drive, the reported
capacity is reported as 4GB less than the true capacity.  For example, a
4,569,599,488 byte drive was reported as having a capacity of
approximately 268MB.
    The problem is that the kernel code that determines the capacity of
a physical volume is not written for capacities that exceed 2^32 bytes.
The OS determines how many blocks (1K or 4K) the drive can support by
issuing the ReadCapacity SCSI command.  This command returns the number
of blocks on the drive, and the size of a block.  Then the system
multiplies these numbers together, which yields the capacity of the
drive, in bytes.  For the last step, the system divides the byte
capacity by the system page size (1K or 4K).  However, a byte capacity
that is greater than 2^32-1 will have overflowed, and the result will be
incorrect.  One example if a drive with a reported capacity of 8,924,999
blocks, at 512 bytes per block, which comes to 4,569,599,488 total
bytes (0x1105e8e00).  This number exceeds 32 bits, and becomes
0x105e8e00, which divides by the page size of the node to become either
0x105e8 or 0x417a3 blocks, depending on the page size.
    Here is a code fragment for the capacity calculation:
        MOVE.l      (a2),d0            * d0 = numblocks
        MULS.l      ($4,a2),d0         * d0 *= blocksize
        MOVE.l      d0,d1              * d1=d0
        LSR.l       #$8,d1             * d1 /= 4096 (page size)
        LSR.l       #$4,d1             *
        MOVE.l      d1,($FFFFFFE0,a6)  * local_b_per_pvol = d1
On a machine with a 1K pagesize, the divisor is 1K, and the second shift
instruction has a count of 2.
    The simplest fix for the problem is to reorder the code that does
the multiply/divide.  Note that the division is done by two adjacent LSR
instructions.  If one of these is moved to operate on the blocksize,
then the multiply won't overflow.  Here is an example:
        MOVE.l      ($4,a2),d0         * d0 = blocksize
        LSR.l       #$8,d0             * d0 /= 256
        MULS.l      (a2),d0            * d0 *= numblocks
        MOVE.l      d0,d1              * d1=d0
        LSR.l       #$4,d1             * d1 /= 16
        MOVE.l      d1,($FFFFFFE0,a6)  * local_b_per_pvol = d1
This solution worked in the case of the 4.5G drive mentioned earlier.

Notes:
    This solution depends on the drive's block size being a multiple of
256.  If the block size does not meet this requirement, then the results
will be incorrect.
    The patched kernel is only necessary when the drive is being
INVOL'd.  Once invol has completed, the drive may be mounted on a normal
system.  On the other hand, it does not hurt to continue to run the
patched kernel.  I have run a patched kernel on a 5500 for approximately
two years, without any problems.

*/

#systype "sys5.3"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <apollo/base.h>
#include <apollo/error.h>
#include <apollo/ms.h>

/* prototype for undocumented function */
unsigned long getpagesize(void);

/* version */
#define Version (strtok("$Revision: 1.4 $"+10, " "))

/* search/replace information for machines with a 1k page size */
unsigned short search_pattern1k[] = {0x2012, 0x4C2A, 0x0000, 0x0004, 0x2200, 0xE089, 0xE489, 0x2D41};
#define search_pattern_length1k (sizeof(search_pattern1k)/sizeof(search_pattern1k[0]))
unsigned short replace_pattern1k[] = {0x202a, 0x0004, 0xe088, 0x4c12, 0x0000, 0x2200};
#define replace_pattern_length1k (sizeof(replace_pattern1k)/sizeof(replace_pattern1k[0]))

/* search/replace information for machines with a 4K page size */
unsigned short search_pattern4k[] = {0x2012, 0x4C2A, 0x0000, 0x0004, 0x2200, 0xE089, 0xE889, 0x2D41};
#define search_pattern_length4k (sizeof(search_pattern4k)/sizeof(search_pattern4k[0]))
unsigned short replace_pattern4k[] = {0x202a, 0x0004, 0xe088, 0x4c12, 0x0000, 0x2200};
#define replace_pattern_length4k (sizeof(replace_pattern4k)/sizeof(replace_pattern4k[0]))


/* Search for the pattern, return VA of the pattern, or NULL if not found. */
unsigned short *search(unsigned short *base, unsigned long Words, unsigned short *search_pattern, unsigned long search_length)
   {
   int count;
   unsigned short *retval;

   /* Adjust Words for pattern length */
   Words -= (search_length-1);
   retval = NULL;
   while (Words--)
      {
      /* See if we match at the current location */
      for (count = 0; count < search_length; count++)
         if (base[count] != search_pattern[count])
            break;
      if (count == search_length)
         {
         if (retval)
            {
            printf("Error - pattern found more than once.\n");
            return NULL;
            }
         retval = base;
         }
      /* Increment base pointer */
      base++;
      }
   return retval;
   }


void do_help(void)
   {
   puts("\
kbs_patch - Patch kernel to support big SCSI drives\n\
\n\
Usage:\n\
    kbs_patch [1024 | 4096] <kernel_path>\n\
\n\
Examples:\n\
    kbs_patch /sau14/domain_os\n\
    kbs_patch 1024 /sau7/domain_os\n\
\n\
Description:\n\
    The Domain/OS kernel selected by <kernel_path> will be patched to support\n\
big SCSI drives.  This changes the code that calculates the number of blocks\n\
that the drive contains.  The pagesize argument is optional, and will default to\n\
the pagesize of the machine running the command.\n\
    The <kernel_path> argument must be the last one on the command line.  Only\n\
one file may be patched at a time.\n\
    Note that the patch that this program uses will not work on devices\n\
that have a block size that is not a multiple of 256 bytes.\n\
    See source code for technical details.\n\
\n\
Disclaimer:\n\
    No warranty for this program or its result are expressed or implied.  The\n\
author assumes no liability for data loss or any other damage caused by the use\n\
of this program.\n\
");
   }


int main(int argc, char *argv[])
   {
   int count;
   unsigned short *search_pattern, *replace_pattern;
   unsigned long search_length, replace_length;
   unsigned short *base, *patch_point;
   unsigned long pagesize;
   unsigned long length;
   struct stat stat_buf;
   status_$t status;

   /* say hello */
   printf("kbs_patch, version %s, built " __DATE__ " at " __TIME__ ".\n", Version);
   if (argc == 2)
      {
      pagesize = getpagesize();
      printf("Pagesize of machine is %d bytes.\n", pagesize);
      }
   else if (argc == 3)
      {
      pagesize = atol(argv[1]);
      printf("Selected page size is %d bytes.\n", pagesize);
      argv[1] = argv[2];
      }
   else
      {
      do_help();
      exit(EXIT_FAILURE);
      }
   /* Select the appropriate search/replace patterns, based on pagesize */
   switch (pagesize)
      {
      case 1024:
         search_pattern = search_pattern1k;
         search_length = search_pattern_length1k;
         replace_pattern = replace_pattern1k;
         replace_length = replace_pattern_length1k;
         break;
      case 4096:
         search_pattern = search_pattern4k;
         search_length = search_pattern_length4k;
         replace_pattern = replace_pattern4k;
         replace_length = replace_pattern_length4k;
         break;
      default:
         printf("Unrecognized page size of %d bytes, can't patch file.\n", pagesize);
         exit(EXIT_FAILURE);
      }
   printf(" Search pattern is");
   for (count = 0; count < search_length; count++)
      printf(" 0x%04x", search_pattern[count]);
   printf("\n Replace pattern is");
   for (count = 0; count < replace_length; count++)
      printf(" 0x%04x", replace_pattern[count]);
   puts("");

   /* Get the file length */
   if (stat(argv[1], &stat_buf))
      {
      perror("Can't get file size");
      exit(EXIT_FAILURE);
      }
   /* Map the file */
   printf("Mapping file '%s' for 0x%x bytes.\n", argv[1], stat_buf.st_size);
   base = ms_$mapl(argv[1], strlen(argv[1]), 0, (unsigned long)stat_buf.st_size, ms_$nr_xor_1w, ms_$wr, false, &length, &status);
   if (status.all != status_$ok)
      {
      printf("Error - Can't map file '%s'.\n", argv[1]);
      error_$print(status);
      exit(EXIT_FAILURE);
      }
   if (length == stat_buf.st_size)
      {
      /* Search for the patch point */
      patch_point = search(base, length/2, search_pattern, search_length);
      if (patch_point != NULL)
         {
         /* Update the file */
         for (count = 0; count < replace_length;count++)
            patch_point[count] = replace_pattern[count];
         printf("Pattern found, 0x%x Words changed at byte offset 0x%x.\n", replace_length, ((char *)patch_point)-((char *)base));
         }
      else
         {
         printf("Error - couldn't find exactly one occurrence of search pattern.\n");
         }
      }
   else
      {
      printf("Error - Wanted to map 0x%x bytes, only mapped 0x%x bytes.\n", stat_buf.st_size, length);
      exit(EXIT_FAILURE);
      }
   /* Unmap the file */
   ms_$unmap(base, length, &status);
   if (status.all != status_$ok)
      {
      printf("Error - Can't unmap file '%s'.\n", argv[1]);
      error_$print(status);
      exit(EXIT_FAILURE);
      }
   }
