/*
 * Do full cylinder buffered writes to slow devices.  Uses a simple
 * buffered read/delayed write algorithm
 */

#include <stdio.h>
#include "msdos.h"
#ifdef apollo
#include "flopdrv.h"
#endif

extern int fd, disk_size, disk_dirty, dir_start, dir_len;
extern int clus_size;
extern long disk_offset, disk_current;
extern unsigned char *disk_buf;
static int read_cyl();

#ifdef apollo
extern unsigned int flop_sectors_per_track;

void
disk_write(
   long start,
   unsigned char *buf,
   int len
)
{
   unsigned int cyl, head, sect;
   unsigned int track_size;
   unsigned int i, sector, num_sectors;
   unsigned char *buf_ptr, *disk_ptr;
   char *memcpy();
   void exit(), disk_flush();

   num_sectors = len / MSECTOR_SIZE;
   track_size = flop_sectors_per_track;

   if (disk_size == 1) {
      /* no cylinder caching, write sector by sector */
      for (i = 0; i < num_sectors; i++) {
         sector = start+i;
         cyl = sector / (2*track_size);
         head = (sector / track_size) & 0x0001;
         sect = sector + 1 - head*track_size - 2*cyl*track_size;
         if (!flopdrv_write_sector(cyl, head, sect, &buf[i*MSECTOR_SIZE])) {
            fprintf(stderr, "disk_write: flopdrv_write_sector failed\n");
            exit(1);
         }
      }
   }
   else {
      /* cylinder caching */
      for (i = 0; i < num_sectors; i++) {
         sector = start+i;
         if (sector < disk_current || sector >= disk_current + disk_size) {
            /* a "cache" miss */
            if (disk_dirty) disk_flush();
            disk_current = (sector / disk_size) * disk_size;
            /* If there is something on the new cylinder that you want to keep,
               you'll have to read it first before writing. */
            if (read_cyl(disk_current)) {
               cyl = disk_current / (2*track_size);
               if (!flopdrv_read_cylinder(cyl, disk_buf)) {
                  fprintf(stderr, "disk_read: flopdrv_read_cylinder failed\n");
                  exit(1);
               }
            }
         }
         /* a cache hit... */
         buf_ptr = buf + i*MSECTOR_SIZE;
         disk_ptr = disk_buf + ((sector - disk_current) * MSECTOR_SIZE);
         memcpy((char *) disk_ptr, (char *) buf_ptr, MSECTOR_SIZE);
         disk_dirty = 1;
      }
   }
}

/*
 * Flush a dirty buffer to disk.  Resets disk_dirty to zero.
 * All errors are fatal.
 */

void
disk_flush()
{
   unsigned int cyl;
   void exit();

   if (fd < 0 || disk_current < 0L || !disk_dirty) return;

   cyl = disk_current / (2*flop_sectors_per_track);
   if (!flopdrv_write_cylinder(cyl, disk_buf)) {
      fprintf(stderr, "disk_flush: write\n");
      exit(1);
   }
   disk_dirty = 0;
}

#else

void
disk_write(start, buf, len)
long start;
unsigned char *buf;
int len;
{
	register long i;
	int length;
	unsigned char *buf_ptr, *disk_ptr;
	char *memcpy();
	void perror(), exit(), disk_flush();
	long where, tail, lseek();

					/* don't use cache? */
	if (disk_size == 1) {
		where = (start * MSECTOR_SIZE) + disk_offset;
		if (lseek(fd, where, 0) < 0) {
			perror("disk_write: lseek");
			exit(1);
		}
					/* write it! */
		if (write(fd, (char *) buf, (unsigned int) len) != len) {
			perror("disk_write: write");
			exit(1);
		}
		return;
	}

	tail = start + (len / MSECTOR_SIZE) - 1;
	for (i = start; i <= tail; i++) {
					/* a cache miss... */
		if (i < disk_current || i >= disk_current + disk_size) {

			if (disk_dirty)
				disk_flush();

			disk_current = (i / disk_size) * disk_size;
			/*
			 * If there is something on the new cylinder that
			 * you want to keep, you'll have to read it first
			 * before writing.
			 */
			if (read_cyl(disk_current)) {
				where = (disk_current * MSECTOR_SIZE) + disk_offset;
				length = disk_size * MSECTOR_SIZE;

					/* move to next location */
				if (lseek(fd, where, 0) < 0) {
					perror("disk_write: lseek");
					exit(1);
				}
					/* read it! */
				if (read(fd, (char *) disk_buf, (unsigned int) length) != length) {
					perror("disk_write: read");
					exit(1);
				}
			}
		}
					/* a cache hit ... */
		buf_ptr = buf + ((i - start) * MSECTOR_SIZE);
		disk_ptr = disk_buf + ((i - disk_current) * MSECTOR_SIZE);
		memcpy((char *) disk_ptr, (char *) buf_ptr, MSECTOR_SIZE);
		disk_dirty = 1;
	}
	return;
}

/*
 * Flush a dirty buffer to disk.  Resets disk_dirty to zero.
 * All errors are fatal.
 */

void
disk_flush()
{
	int len;
	long where, lseek();
	void perror(), exit();

	if (fd < 0 || disk_current < 0L || !disk_dirty)
		return;

	where = (disk_current * MSECTOR_SIZE) + disk_offset;
	if (lseek(fd, where, 0) < 0) {
		perror("disk_flush: lseek");
		exit(1);
	}
					/* write it! */
	len = disk_size * MSECTOR_SIZE;
	if (write(fd, (char *) disk_buf, (unsigned int) len) != len) {
		perror("disk_flush: write");
		exit(1);
	}
	disk_dirty = 0;
	return;
}
#endif

/*
 * Determine if the cylinder has some useful information on it.  Returns a 1
 * if it needs to be read.
 */

static int
read_cyl(sector)
long sector;
{
	register unsigned int i;
	int start_clus, end_clus;
	unsigned int start, end, fat_decode();

	if (!sector)
		return(1);

/* APOLLO: First file on blank 5.25 floppy will be corrupt without the following */
        if ((sector - dir_start - dir_len) < 0) sector = dir_start + dir_len;

	start = ((sector - dir_start - dir_len) / clus_size) + 2;
	end = ((sector + disk_size - dir_start - dir_len) / clus_size) + 2;

	for (i = start; i < end; i++) {
					/* if fat_decode returns non-zero */
		if (fat_decode(i))
			return(1);
	}

	return(0);
}

