/*
 * flopdrv_call.c
 *   GPIO call-side routine(s) for the MSDOS floppy driver.
 *
 *   Version 1.1 - July 22, 1993
 *   Jef Rijmenants
 */

#include <stdio.h>
#include <apollo/base.h>
#include <apollo/error.h>
#include "pbu.h"
#include "flopdrv_priv.h"
#include "flopdrv.h"


/*
 * Standard AT port addresses for floppy controller:
 * 
 *                         | Primary | Secondary | Access
 *                         | Contrlr | Contrlr   |
 * ------------------------| --------|-----------|-------------
 * Base Address            | 3F0     | 370       | -
 * Digital Output Register | 3F2     | 372       | Write
 * Status Register         | 3F4     | 374       | Read
 * Data Register           | 3F5     | 375       | Read/Write
 * Digital Input Register  | 3F7     | 377       | Read
 * +Configuration Register | 3F7     | 377       | Write
 * 
 * Interrupt Request Line: 6 (IRQ6)
 * DMA channel: 2 (DRQ2)
 */

/*
 * DOS standard formats:
 * 
 * Floppy size         | 5.25" | 5.25" | 3.5"  | 3.5"  | 3.5"
 * Floppy type         | DSDD  | DSHD  | DSDD  | DSHD  | DSED
 * Capacity            | 360k  | 1.2M  | 720k  | 1.44M | 2.88M
 * Number of tracks    | 40    | 80    | 80    | 80    | 80
 * Sectors per track   | 9     | 15    | 9     | 18    | 36
 * Data transfer speed | 250   | 500   | 250   | 500   | 1000
 * 
 * Capacity is in bytes (formatted).
 * Data transfer speed is in kbits/sec.
 * Floppy type codes:
 *   DS = Double Sided
 *   DD = Double Density
 *   HD = High Density
 *   ED = Extra High Density
 */

/* drive/disk configuration */
unsigned int flopdrv_drive_type;  /* indicates drive type (3.5" or 5.25"), is extracted from DDF user_info */
unsigned int flopdrv_disk_type;   /* indicates disk type (DSHD or DSDD) */

/* drive characteristics
 * first byte:   STT (STep Time) 4 bits, HUT (Head Up Time) 4 bits
 * second byte:  HDT (Head Down Time) 7 bits, NDMA (No DMA Mode) 1 bit
 *
 *             Data Transfer Speed
 *          1M     500k   300k    250k  (in k bits per second)
 * -----------------------------------------
 *   STT    STep Time (in ms)
 * -----------------------------------------
 *    0h    8.0    16     26.7    32
 *    1h    7.5    15     25      30
 *    2h    7.0    14     23.3    28
 *    ..    ...    ...    ...     ...
 *    Eh    1.0    2      3.3     4
 *    Fh    0.5    1      1.7     2
 * -----------------------------------------
 *   HUT    Head Up Time (in ms)
 * -----------------------------------------
 *    0h    128    256    426     512
 *    1h    8      16     26.7    32
 *    2h    16     32     53.5    64
 *    ..    ...    ...    ...     ...
 *    Eh    112    224    373     448
 *    Fh    120    240    400     480
 * -----------------------------------------
 *   HDT    Head Down Time (in ms)
 * -----------------------------------------
 *    0h    128    256    426     512
 *    1h    1      2      3.3     4
 *    2h    2      4      6.7     8
 *    ..    ...    ...    ...     ...
 *   7Eh    126    252    420     504
 *   7Fh    127    254    423     508
 */

/*                                          3H   3D      5H   5D */
unsigned int flopdrv_num_tracks[2][2] = {{  80,  80}, {  80,  40}};  /* number of tracks */
unsigned int flopdrv_track_size[2][2] = {{  18,   9}, {  15,   9}};  /* sectors per track */
unsigned int flopdrv_data_speed[2][2] = {{   0,   2}, {   0,   2}};  /* data transfer speed (0=500k, 1=300k, 2=250k)*/
unsigned int flopdrv_stt_hut[2][2]    = {{0x88,0x88}, {0x88,0x88}};  /* step time + head up time */
unsigned int flopdrv_hdt_ndma[2][2]   = {{0x80,0x80}, {0x80,0x80}};  /* head down time + no-dma mode*/

#define FLOPDRV_NUM_TRACKS  flopdrv_num_tracks[flopdrv_drive_type][flopdrv_disk_type]
#define FLOPDRV_TRACK_SIZE  flopdrv_track_size[flopdrv_drive_type][flopdrv_disk_type]
#define FLOPDRV_DATA_SPEED  flopdrv_data_speed[flopdrv_drive_type][flopdrv_disk_type]
#define FLOPDRV_STT_HUT     flopdrv_stt_hut[flopdrv_drive_type][flopdrv_disk_type]
#define FLOPDRV_HDT_NDMA    flopdrv_hdt_ndma[flopdrv_drive_type][flopdrv_disk_type]

/* interrupt routine stuff */
long flopdrv_device_timeout;              /* timeout (in msec) for hardware interrupt */
boolean flopdrv_quit_enable;              /* enables CTRL_C during floppy IO */

/* DMA stuff */
#define FLOPDRV_CYL_SIZE  2*18            /* max number of sectors per cylinder, for DMA buffer */
char *flopdrv_dma_buffer;                 /* must be page-aligned (=on a 1024 byte boundary) */
linteger flopdrv_dma_buffer_size;         /* the DN3000 does not support more than one page (1024 bytes) */
char *flopdrv_buffer;                     /* used for page-alignment of the dma_buffer, see driver init routine */
linteger flopdrv_buffer_size;
boolean flopdrv_dma_buffer_wired = false; /* indicates that the DMA buffer is made non-swapable */
pbu_$dma_channel_t flopdrv_dma_channel;   /* is fixed to channel 2, the AT standard */
linteger flopdrv_iomap_iova;
linteger flopdrv_buffer_iova;

/* control */
unsigned char flopdrv_command[16];        /* global buffer for command sequences */
unsigned char flopdrv_status[16];         /* global buffer for return status info */
int flopdrv_retry_ok;                     /* a fatal error can disable retries (eg: a write-protected floppy) */

int flopdrv_debug = 0;


/*
 * user callable routines
 */

/*
 * flopdrv_read_sector
 *   Reads the specified sector. If the operation fails it will recalibrate
 *   the drive and reposition the head a number of times before quiting.
 */
int flopdrv_read_sector(
   unsigned int cyl_num,
   unsigned int head_num,
   unsigned int sect_num,
   unsigned char *sector_buffer
)
{
   int try;

   flopdrv_retry_ok = FLOPDRV_TRUE;
   for (try = 0; (try < FLOPDRV_MAX_RETRIES) && flopdrv_retry_ok; try++) {
      if (flopdrv_position_head_cmd(cyl_num))
         if (flopdrv_read_sector_cmd(cyl_num, head_num, sect_num, sector_buffer))
            return FLOPDRV_OK;
      flopdrv_calibrate_drive_cmd(); /* go to track 0, and try again */
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_write_sector
 *   Writes the specified sector. If the operation fails it will recalibrate
 *   the drive and reposition the head a number of times before quiting.
 */
int flopdrv_write_sector(
   unsigned int cyl_num,
   unsigned int head_num,
   unsigned int sect_num,
   unsigned char *sector_buffer
)
{
   int try;

   flopdrv_retry_ok = FLOPDRV_TRUE;
   for (try = 0; (try < FLOPDRV_MAX_RETRIES) && flopdrv_retry_ok; try++) {
      if (flopdrv_position_head_cmd(cyl_num))
         if (flopdrv_write_sector_cmd(cyl_num, head_num, sect_num, sector_buffer))
            return FLOPDRV_OK;
      flopdrv_calibrate_drive_cmd(); /* go to track 0, and try again */
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_read_cylinder
 *   Reads the specified cylinder. If the operation fails it will recalibrate
 *   the drive and reposition the head a number of times before quiting.
 */
int flopdrv_read_cylinder(
   unsigned int cyl_num,
   unsigned char *cylinder_buffer
)
{
   int try;

   if (!flopdrv_iomap_present) return FLOPDRV_FALSE;

   flopdrv_retry_ok = FLOPDRV_TRUE;
   for (try = 0; (try < FLOPDRV_MAX_RETRIES) && flopdrv_retry_ok; try++) {
      if (flopdrv_position_head_cmd(cyl_num))
         if (flopdrv_read_cylinder_cmd(cyl_num, cylinder_buffer))
            return FLOPDRV_OK;
      flopdrv_calibrate_drive_cmd(); /* go to track 0, and try again */
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_write_cylinder
 *   Writes the specified cylinder. If the operation fails it will recalibrate
 *   the drive and reposition the head a number of times before quiting.
 */
int flopdrv_write_cylinder(
   unsigned int cyl_num,
   unsigned char *cylinder_buffer
)
{
   int try;

   if (!flopdrv_iomap_present) return FLOPDRV_FALSE;

   flopdrv_retry_ok = FLOPDRV_TRUE;
   for (try = 0; (try < FLOPDRV_MAX_RETRIES) && flopdrv_retry_ok; try++) {
      if (flopdrv_position_head_cmd(cyl_num))
         if (flopdrv_write_cylinder_cmd(cyl_num, cylinder_buffer))
            return FLOPDRV_OK;
      flopdrv_calibrate_drive_cmd(); /* go to track 0, and try again */
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_format_track
 *   Formats the specified sector. If the operation fails it will recalibrate
 *   the drive and reposition the head a number of times before quiting.
 */
int flopdrv_format_track(
   unsigned int cyl_num,
   unsigned int head_num
)
{
   int try;

   flopdrv_retry_ok = FLOPDRV_TRUE;
   for (try = 0; (try < FLOPDRV_MAX_RETRIES) && flopdrv_retry_ok; try++) {
      if (flopdrv_position_head_cmd(cyl_num))
         if (flopdrv_format_track_cmd(cyl_num, head_num)) return FLOPDRV_OK;
      flopdrv_calibrate_drive_cmd(); /* go to track 0, and try again */
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_format_disk
 *   Formats the entire disk.
 */
int flopdrv_format_disk()
{
   unsigned int cyl;
   for (cyl = 0; cyl < FLOPDRV_NUM_TRACKS; cyl++) {
      if (!flopdrv_format_track(cyl, 0)) return FLOPDRV_FAIL;
      if (!flopdrv_format_track(cyl, 1)) return FLOPDRV_FAIL;
/* FIXME: correct this for DD disks! */
      if (cyl == 16) fprintf(stderr, "Formatting 20%% complete...\n");
      if (cyl == 32) fprintf(stderr, "Formatting 40%% complete...\n");
      if (cyl == 48) fprintf(stderr, "Formatting 60%% complete...\n");
      if (cyl == 64) fprintf(stderr, "Formatting 80%% complete...\n");
   }
   fprintf(stderr, "Formatting done.\n");
   return FLOPDRV_OK;
}


/*
 * flopdrv_get_drive_type
 *   Returns the type of drive connected to the controller 
 *   (= as specified in the DDF file).
 */
int flopdrv_get_drive_type()
{
   return flopdrv_drive_type;
}


/*
 * flopdrv_cylinder_io_possible
 *   Indicates if machine can do full cylinder IO or not (limited DMA size on DN30000)
 */
int flopdrv_cylinder_io_possible()
{
   return flopdrv_iomap_present;
}


/*
 * flopdrv_start_drive
 *   Initializes the controller, activates and recalibrates the drive.
 *   If this fails, it will retry some recalibrations before quiting.
 */
int flopdrv_start_drive(
   unsigned int disk_type
)
{
   int try;

   if ( (disk_type != FLOPDRV_DSHD_FLOPPY) && (disk_type != FLOPDRV_DSDD_FLOPPY)) {
      fprintf(stderr, "ERROR: Invalid floppy disk type.\n");
      return FLOPDRV_FAIL;
   }
   flopdrv_disk_type = disk_type;
   if (!flopdrv_start_drive_cmd()) return FLOPDRV_FAIL;
   for (try = 0; try < FLOPDRV_MAX_RETRIES; try++)
      if (flopdrv_calibrate_drive_cmd()) return FLOPDRV_OK;
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_stop_drive
 *   Stops and unselects the drive, and deactivates the FDC.
 */
int flopdrv_stop_drive()
{
   return flopdrv_stop_drive_cmd();
}


/*
 * GPIO interface routines (init and cleanup routines)
 */

/*
 * flopdrv_init(
 *   The GPIO driver init routine. This routine gets called when the
 *   device is acquired (with pbu_$acquire).
 */
void flopdrv_init(
   pbu_$unit_t  &pbu_unit,
   pbu_$ddf_ptr_t  &ddf_ptr,
   pbu_$csr_page_ptr_t  &csr_ptr,
   status_$t  *status
)
{
   unsigned char reg;
   int controller_present;
   short pbu_info[3];
   status_$t sts;

   flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   flopdrv_cb.ddf_p = ddf_ptr;
   flopdrv_cb.csr_p = (flopdrv_csr_page_p)(csr_ptr);
   flopdrv_cb.pbu_unit = pbu_unit;

   /* check controller presence by activating/deactivating the FDC
      and checking the MRQ bit in the status register (maybe too simplistic
      but my FDC has no feet) */
   controller_present = FLOPDRV_TRUE;
   flopdrv_cb.csr_p->do_reg = 0x00; /* deactive controller */
   wait(1);
   if ((flopdrv_cb.csr_p->stat_reg & 0x80) != 0x00) controller_present = FLOPDRV_FALSE;
   flopdrv_cb.csr_p->do_reg = 0x0C; /* activate controller */
   wait(1);
   if ((flopdrv_cb.csr_p->stat_reg & 0x80) != 0x80) controller_present = FLOPDRV_FALSE;
   flopdrv_cb.csr_p->do_reg = 0x00; /* deactive controller */
   wait(1);
   if ((flopdrv_cb.csr_p->stat_reg & 0x80) != 0x00) controller_present = FLOPDRV_FALSE;
   if (!controller_present) {
      fprintf(stderr, "ERROR: Floppy disk controller not present on this node.\n");
      status->s.fail = 1;
      return;
   }

   /* determine drive type from the first chars of the user-info specification in the DDF */
   if (strncmp(flopdrv_cb.ddf_p->user_info, "3_5", 3) == 0) 
      flopdrv_drive_type = FLOPDRV_3_5_DRIVE;
   else if (strncmp(flopdrv_cb.ddf_p->user_info, "5_25", 4) == 0) 
      flopdrv_drive_type = FLOPDRV_5_25_DRIVE;
   else {
      flopdrv_drive_type = FLOPDRV_3_5_DRIVE;
      fprintf(stderr, "WARNING: Invalid drive_type specification in ddf user_info, defaulted to 3.5\" drive\n");
   }
   flopdrv_disk_type = FLOPDRV_DSHD_FLOPPY;
   flopdrv_device_timeout = 30 * 1000; /* milliseconds */
   flopdrv_quit_enable = true; /* enables CTRL_C interrupts during IO (may corrupt disks) */
   flopdrv_dma_channel = 2;  /* the AT-standard */

   pbu_$get_info(6, pbu_info, &sts);
   flopdrv_iomap_present = (pbu_info[2] == 2); /* node has IO map for AT bus */

   /* prepare a page-aligned buffer, use a part of flopdrv_buffer */
   if (flopdrv_iomap_present)
      flopdrv_dma_buffer_size = FLOPDRV_CYL_SIZE * FLOPDRV_BYTES_PER_SECTOR; /* full cylinder DMA */
   else 
      flopdrv_dma_buffer_size = FLOPDRV_BYTES_PER_SECTOR;  /* single sector DMA */
   flopdrv_buffer_size = flopdrv_dma_buffer_size + bytes_per_page;
   flopdrv_buffer = (char *) malloc(flopdrv_buffer_size);
   flopdrv_dma_buffer = flopdrv_buffer;
   flopdrv_dma_buffer = (char *) (((unsigned long)flopdrv_dma_buffer + bytes_per_page-1)
      & ~(bytes_per_page-1)); /* round up to page boundary (think in bits to see how this works) */
   flopdrv_dma_buffer_wired = false;


   /* "wires" the DMA-buffer, ie. makes it un-swappable so that the DMA
       hardware can do its job correctly. */
   if (flopdrv_iomap_present) {
/*HELP: I am not sure the following is correct ! */
      flopdrv_iomap_iova = pbu2_$allocate_map(flopdrv_cb.pbu_unit, flopdrv_dma_buffer_size,
         false, (linteger)0, &sts);
      if (sts.all != status_$ok) error_$print(sts);
      pbu2_$wire(flopdrv_cb.pbu_unit, flopdrv_dma_buffer, flopdrv_dma_buffer_size, &sts);
      if (sts.all != status_$ok) error_$print(sts);
      flopdrv_buffer_iova = pbu2_$map(flopdrv_cb.pbu_unit, flopdrv_dma_buffer,
         flopdrv_dma_buffer_size, flopdrv_iomap_iova, &sts);
      if (sts.all != status_$ok) error_$print(sts);
   }
   else {
      pbu2_$wire(flopdrv_cb.pbu_unit, flopdrv_dma_buffer, flopdrv_dma_buffer_size, &sts);
      if (sts.all != status_$ok) error_$print(sts);
   }
   flopdrv_dma_buffer_wired = true;
}


/*
 * flopdrv_cleanup
 *   GPIO driver cleanup routine. This routine gets called when the device
 *   is released (using pbu_$release).
 */
void flopdrv_cleanup(
   pbu_$unit_t  *pbu_unit,
   boolean   *force_flag,
   status_$t  *status
)
{
   boolean touch;
   status_$t sts;

   touch = false;

   /* unwire the buffer */
   if (flopdrv_dma_buffer_wired) {
      if (flopdrv_iomap_present) {
         pbu2_$unmap(flopdrv_cb.pbu_unit, flopdrv_dma_buffer, flopdrv_dma_buffer_size,
            flopdrv_buffer_iova, &sts);
         if (sts.all != status_$ok) error_$print(sts);
         pbu2_$unwire(flopdrv_cb.pbu_unit, flopdrv_dma_buffer, flopdrv_dma_buffer_size, touch, &sts);
         if (sts.all != status_$ok) error_$print(sts);
         pbu2_$free_map(flopdrv_cb.pbu_unit, &sts);
         if (sts.all != status_$ok) error_$print(sts);
      }
      else {
         pbu2_$unwire(flopdrv_cb.pbu_unit, flopdrv_dma_buffer, flopdrv_dma_buffer_size, touch, &sts);
         if (sts.all != status_$ok) error_$print(sts);
      }
      flopdrv_dma_buffer_wired = false;
      free(flopdrv_buffer);
   }

   flopdrv_cb.csr_p->do_reg = 0x00; /* deactivate controller */
}


/*
 * basic floppy commands
 */

/*
 * flopdrv_start_drive
 *   Activates the FDC, enables FDC DMA and interrupt requests. Selects
 *   the drive and starts the drive motor (the drive is always drive 0 or
 *   drive A).
 */
int flopdrv_start_drive_cmd()
{
   status_$t sts;
   int ret_sts;

   /* activate the controller, enable DMA en IRQ, start motor A */
   flopdrv_cb.csr_p->do_reg = 0x1C;

   /* select data tranfer rate (500k-bits/s for 1.44Mb and 1.2M) */
   flopdrv_cb.csr_p->di_reg = FLOPDRV_DATA_SPEED;

   /* send "setting drive characteristics" command */
   flopdrv_command[0] = 0x03; /* set drive characteristics */
   flopdrv_command[1] = FLOPDRV_STT_HUT;  /* step-time / head-up time */
   flopdrv_command[2] = FLOPDRV_HDT_NDMA; /* head-down time / no-DMA */
   if (!flopdrv_put_command(flopdrv_command, 3)) return FLOPDRV_FAIL;
   flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;

   /* enable interrupts (=interrupt handling by GPIO) */
   pbu_$enable_device(flopdrv_cb.pbu_unit, &sts);
   if (sts.all != status_$ok) {
      error_$print(sts);
      return FLOPDRV_FAIL;
   }
   return FLOPDRV_OK;
}


/*
 * flopdrv_stop_drive
 *   Stops the drive, disables interrupts and deactivates the controller.
 */
int flopdrv_stop_drive_cmd()
{
   status_$t sts;

   /* disable interrupts */
   pbu_$disable_device(flopdrv_cb.pbu_unit, &sts);
   /* if (sts.all != status_$ok) error_$print(sts); */

   /* disable DRQ and IRQ, stop drive motor, disable controller */
   flopdrv_cb.csr_p->do_reg = 0x00; 

   return FLOPDRV_OK;
}


/*
 * flopdrv_check_interrupt_status_cmd
 *   Checks interrupt status for some commands that generate interrupts
 *   on termination but no status report.
 */
int flopdrv_check_interrupt_status_cmd(
   unsigned char *st0,
   unsigned char *cyl
)
{
   /* send "check interrupt_status" command */
   flopdrv_command[0] = 0x08;
   if (!flopdrv_put_command(flopdrv_command, 1)) return FLOPDRV_FAIL;

   if (!flopdrv_get_status(flopdrv_status, 2)) return FLOPDRV_FAIL;
   *st0 = flopdrv_status[0];   /* st0 status */
   *cyl = flopdrv_status[1];   /* current cylinder */
   if ((*st0 & 0xC0) == 0x00) return FLOPDRV_OK;
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_calibrate_drive_cmd
 *   Calibrates the drive, ie. steps down the head to track 0. This should
 *   be done if head positioning errors occur (floppy drives use open-loop
 *   head positioning).
 */
int flopdrv_calibrate_drive_cmd()
{
   pbu_$wait_index_t wait_index;
   status_$t sts;
   unsigned char st0, cyl;

   /* send "calibrate drive" command */
   flopdrv_command[0] = 0x07;
   flopdrv_command[1] = 0x00;
   if (!flopdrv_put_command(flopdrv_command, 2)) return FLOPDRV_FAIL;

   /* wait for "command finished" interrupt */
   wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
   if (sts.all != status_$ok) error_$print(sts);
   flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;

   /* check interrupt status */
   if (wait_index == WAIT_EC_ADVANCE) {
      if (!flopdrv_check_interrupt_status_cmd(&st0, &cyl)) return FLOPDRV_FAIL;
      if ((st0 & 0xC0) == 0x00) return FLOPDRV_OK;
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_position_head_cmd
 *   Positions the head to the specified track or cylinder.
 */
int flopdrv_position_head_cmd(
   unsigned int cylinder
)
{
   pbu_$wait_index_t wait_index;
   status_$t sts;
   unsigned char st0, cyl;

   if ((cylinder < 0) || (cylinder > (FLOPDRV_NUM_TRACKS-1))) return FLOPDRV_FAIL;

   /* send "position head" command */
   flopdrv_command[0] = 0x0F;
   flopdrv_command[1] = 0x00;
   flopdrv_command[2] = cylinder;
   if (!flopdrv_put_command(flopdrv_command, 3)) return FLOPDRV_FAIL;

   /* wait for "command finished" interrupt */
   wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
   if (sts.all != status_$ok) error_$print(sts);
   flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;

   /* check interrupt status */
   if (wait_index == WAIT_EC_ADVANCE) {
      if(!flopdrv_check_interrupt_status_cmd(&st0, &cyl)) return FLOPDRV_FAIL;
      if (((st0 & 0xC0) == 0x00) && (cyl == cylinder)) return FLOPDRV_OK;
   }
   return FLOPDRV_FAIL;
}


/*
 * flopdrv_read_id_cmd
 *   Reads the sector ID of the first sector the controller can lock-on.
 *   This was used for debugging and experiments, it is not used by the 
 *   flop program.
 */
int flopdrv_read_id_cmd()
{
   pbu_$wait_index_t wait_index;
   status_$t sts;
   unsigned char st0, st1, st2, cyl, head, sec_num, sec_size;
   int ret_sts;

   ret_sts = FLOPDRV_FAIL;

   /* send "read ID" command */
   flopdrv_command[0] = 0x4A;
   flopdrv_command[1] = 0x00;
   if (flopdrv_put_command(flopdrv_command, 2)) {
   
      /* wait for "command finished" interrupt */
      wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
      if (sts.all != status_$ok) error_$print(sts);
      flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   
      /* check interrupt status */
      if (wait_index == WAIT_EC_ADVANCE) {
         if (flopdrv_get_status(flopdrv_status, 7)) {
            st0 = flopdrv_status[0];
            st1 = flopdrv_status[1];
            st2 = flopdrv_status[2];
            cyl = flopdrv_status[3]; /* cylinder */
            head = flopdrv_status[4]; /* head */
            sec_num = flopdrv_status[5]; /* sector number */
            sec_size = flopdrv_status[6]; /* sector size */
            if (flopdrv_debug) fprintf(stderr,
               "ST0=%X, ST1=%X, ST2=%X, cyl=%d, head=%d, sec=%d, siz=%d\n",
               (int)st0, (int)st1, (int)st2, (int)cyl, (int)head, (int)sec_num, (int)sec_size);
            if ((st0 & 0xC0) == 0x00) ret_sts = FLOPDRV_OK; 
         }
      }
   }
   return ret_sts;
}


/*
 * flopdrv_read_sector_cmd
 *   Reads the specified sector.
 */
int flopdrv_read_sector_cmd(
   unsigned int cyl_num,
   unsigned int head_num,
   unsigned int sect_num,
   unsigned char *sector_buffer
)
{
   unsigned char st0, st1, st2, cyl, head, sec_num, sec_size;
   pbu_$wait_index_t wait_index;
   status_$t sts;
   pinteger dma_len;
   pbu_$dma_opts_t dma_options;
   long dma_rest;
   int i, ret_sts;

   ret_sts = FLOPDRV_FAIL;

   /* start dma */
   dma_len = FLOPDRV_BYTES_PER_SECTOR;
   dma_options = 0;
   pbu_$dma_start(flopdrv_cb.pbu_unit, flopdrv_dma_channel, pbu_$dma_read,
      flopdrv_dma_buffer, dma_len, dma_options, &sts);
   if (sts.all != status_$ok) error_$print(sts);

   /* send "read sector" command */
   flopdrv_command[0] = 0x46;     /* single track, MFM */
   if (head_num == 0)
       flopdrv_command[1] = 0x00;
   else
       flopdrv_command[1] = 0x04; /* head, drive A */
   flopdrv_command[2] = cyl_num;  /* cylinder */
   flopdrv_command[3] = head_num; /* head */
   flopdrv_command[4] = sect_num; /* start sector number */
   flopdrv_command[5] = 0x02;     /* sector size = 512b */
   flopdrv_command[6] = sect_num; /* last sector number */
   if (flopdrv_drive_type == FLOPDRV_5_25_DRIVE) 
      flopdrv_command[7] = 42;    /* GAP 3 size = 42 */
   else 
      flopdrv_command[7] = 27;    /* GAP 3 size = 27 */
   flopdrv_command[8] = 0xff;     /* data length */
   if (flopdrv_put_command(flopdrv_command, 9)) {

      /* wait for DMA to complete */
      wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
      if (sts.all != status_$ok)  error_$print(sts);
      flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   
      if (wait_index == WAIT_EC_ADVANCE) {
         if (flopdrv_get_status(flopdrv_status, 7)) {
            st0 = flopdrv_status[0];
            st1 = flopdrv_status[1];
            st2 = flopdrv_status[2];
            cyl = flopdrv_status[3]; /* cylinder */
            head = flopdrv_status[4]; /* head */
            sec_num = flopdrv_status[5]; /* sector number */
            sec_size = flopdrv_status[6]; /* sector size */
      
            if ((st0 & 0xC0) == 0x00) {
               for (i = 0; i < FLOPDRV_BYTES_PER_SECTOR; i++) sector_buffer[i] = flopdrv_dma_buffer[i];
               ret_sts = FLOPDRV_OK;
            }
            else if (flopdrv_debug) {
               fprintf(stderr, "ST0=%X, ST1=%X, ST2=%X, cyl=%d, head=%d, sec=%d, siz=%d\n",
                  (int)st0, (int)st1, (int)st2, (int)cyl, (int)head, (int)sec_num, (int)sec_size);
               flopdrv_print_status_info(0, st0);
               flopdrv_print_status_info(1, st1);
               flopdrv_print_status_info(2, st2);
            }
         }
      }

    }
   /* stop DMA operation, may not run to completion if command fails */
   dma_rest = pbu_$dma_stop(flopdrv_cb.pbu_unit, flopdrv_dma_channel, &sts);
   /* if (sts.all != status_$ok) error_$print(sts); */

   return ret_sts;
}


/*
 * flopdrv_write_sector_cmd
 *   Writes the specified sector.
 */
int flopdrv_write_sector_cmd(
   unsigned int cyl_num,
   unsigned int head_num,
   unsigned int sect_num,
   unsigned char *sector_buffer
)
{
   unsigned char st0, st1, st2, cyl, head, sec_num, sec_size;
   pbu_$wait_index_t wait_index;
   status_$t sts;
   pinteger dma_len;
   pbu_$dma_opts_t dma_options;
   long dma_rest;
   int i, ret_sts;

   ret_sts = FLOPDRV_FAIL;

   /* copy sector buffer into dma buffer */
   for (i = 0; i < FLOPDRV_BYTES_PER_SECTOR; i++) flopdrv_dma_buffer[i] = sector_buffer[i];

   /* start dma */
   dma_len = FLOPDRV_BYTES_PER_SECTOR;
   dma_options = 0;
   pbu_$dma_start(flopdrv_cb.pbu_unit, flopdrv_dma_channel, pbu_$dma_write,
      flopdrv_dma_buffer, dma_len, dma_options, &sts);
   if (sts.all != status_$ok) error_$print(sts);

   /* send "write sector" command */
   flopdrv_command[0] = 0x45;     /* single track, MFM */
   if (head_num == 0) flopdrv_command[1] = 0x00; else flopdrv_command[1] = 0x04; /* head, drive A */
   flopdrv_command[2] = cyl_num;  /* cylinder */
   flopdrv_command[3] = head_num; /* head */
   flopdrv_command[4] = sect_num; /* start sector number */
   flopdrv_command[5] = 0x02;     /* sector size = 512b */
   flopdrv_command[6] = sect_num; /* last sector number */
   if (flopdrv_drive_type == FLOPDRV_5_25_DRIVE) 
      flopdrv_command[7] = 42;    /* GAP 3 size = 42 */
   else
      flopdrv_command[7] = 27;    /* GAP 3 size = 27 */
   flopdrv_command[8] = 0xff;     /* data length */
   if (flopdrv_put_command(flopdrv_command, 9)) {
      /* wait for DMA to complete */
      wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
      if (sts.all != status_$ok)  error_$print(sts);
      flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   
      if (wait_index == WAIT_EC_ADVANCE) {
         if (flopdrv_get_status(flopdrv_status, 7)) {
            st0 = flopdrv_status[0];
            st1 = flopdrv_status[1];
            st2 = flopdrv_status[2];
            cyl = flopdrv_status[3]; /* cylinder */
            head = flopdrv_status[4]; /* head */
            sec_num = flopdrv_status[5]; /* sector number */
            sec_size = flopdrv_status[6]; /* sector size */
            if ((st0 & 0xC0) == 0x00) {
               ret_sts = FLOPDRV_OK;
            }
            else if ((st1 & 0x02) == 0x02) {
               flopdrv_retry_ok = FLOPDRV_FALSE;
               fprintf(stderr, "ERROR: Floppy is write protected.\n");
            }
            else if (flopdrv_debug) {
               fprintf(stderr, "ST0=%X, ST1=%X, ST2=%X, cyl=%d, head=%d, sec=%d, siz=%d\n",
                  (int)st0, (int)st1, (int)st2, (int)cyl, (int)head, (int)sec_num, (int)sec_size);
               flopdrv_print_status_info(0, st0);
               flopdrv_print_status_info(1, st1);
               flopdrv_print_status_info(2, st2);
            }
         }
      }
   }

   /* stop DMA operation, may not run to completion if command fails */
   dma_rest = pbu_$dma_stop(flopdrv_cb.pbu_unit, flopdrv_dma_channel, &sts);
   /* if (sts.all != status_$ok) error_$print(sts); */

   return ret_sts;
}


/*
 * flopdrv_read_cylinder_cmd
 *   Reads the specified cylinder.
 */
int flopdrv_read_cylinder_cmd(
   unsigned int cyl_num,
   unsigned char *cylinder_buffer
)
{
   unsigned char st0, st1, st2, cyl, head, sec_num, sec_size;
   pbu_$wait_index_t wait_index;
   status_$t sts;
   linteger dma_len;
   pbu_$dma_opts_t dma_options;
   long dma_rest;
   int i, ret_sts;

   ret_sts = FLOPDRV_FAIL;

   /* start dma */
   dma_len = 2 * FLOPDRV_TRACK_SIZE * FLOPDRV_BYTES_PER_SECTOR;
   dma_options = 0;

/*HELP: It does not work without the following pbu2_$map call, but I do not know why! */
   flopdrv_buffer_iova = pbu2_$map(flopdrv_cb.pbu_unit, flopdrv_dma_buffer,
      flopdrv_dma_buffer_size, flopdrv_iomap_iova, &sts);
   if (sts.all != status_$ok) error_$print(sts);
   pbu2_$dma_start(flopdrv_cb.pbu_unit, flopdrv_dma_channel, pbu_$dma_read,
      flopdrv_dma_buffer, flopdrv_iomap_iova, dma_len, dma_options, &sts);
   if (sts.all != status_$ok) error_$print(sts);

   /* send "read sector" command */
   flopdrv_command[0] = 0xC6;     /* multiple track, MFM */
   flopdrv_command[1] = 0x00;     /* head, drive A */
   flopdrv_command[2] = cyl_num;  /* cylinder */
   flopdrv_command[3] = 0;        /* head */
   flopdrv_command[4] = 1;        /* start sector number */
   flopdrv_command[5] = 0x02;     /* sector size = 512b */
   flopdrv_command[6] = FLOPDRV_TRACK_SIZE; /* last sector number */
   if (flopdrv_drive_type == FLOPDRV_5_25_DRIVE) 
      flopdrv_command[7] = 42;    /* GAP 3 size = 42 */
   else 
      flopdrv_command[7] = 27;    /* GAP 3 size = 27 */
   flopdrv_command[8] = 0xff;     /* data length */
   if (flopdrv_put_command(flopdrv_command, 9)) {

      /* wait for DMA to complete */
      wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
      if (sts.all != status_$ok)  error_$print(sts);
      flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   
      if (wait_index == WAIT_EC_ADVANCE) {
         if (flopdrv_get_status(flopdrv_status, 7)) {
            st0 = flopdrv_status[0];
            st1 = flopdrv_status[1];
            st2 = flopdrv_status[2];
            cyl = flopdrv_status[3]; /* cylinder */
            head = flopdrv_status[4]; /* head */
            sec_num = flopdrv_status[5]; /* sector number */
            sec_size = flopdrv_status[6]; /* sector size */

            if ((st0 & 0xC0) == 0x00) {
               for (i = 0; i < dma_len; i++) cylinder_buffer[i] = flopdrv_dma_buffer[i];
               ret_sts = FLOPDRV_OK;
            }
            else if (flopdrv_debug) {
               fprintf(stderr, "ST0=%X, ST1=%X, ST2=%X, cyl=%d, head=%d, sec=%d, siz=%d\n",
                  (int)st0, (int)st1, (int)st2, (int)cyl, (int)head, (int)sec_num, (int)sec_size);
               flopdrv_print_status_info(0, st0);
               flopdrv_print_status_info(1, st1);
               flopdrv_print_status_info(2, st2);
            }
         }
      }

    }
   /* stop DMA operation, may not run to completion if command fails */
   dma_rest = pbu2_$dma_stop(flopdrv_cb.pbu_unit, flopdrv_dma_channel, &sts);
   /* if (sts.all != status_$ok) error_$print(sts); */

   return ret_sts;
}


/*
 * flopdrv_write_cylinder_cmd
 *   Writes the specified cylinder.
 */
int flopdrv_write_cylinder_cmd(
   unsigned int cyl_num,
   unsigned char *cylinder_buffer
)
{
   unsigned char st0, st1, st2, cyl, head, sec_num, sec_size;
   pbu_$wait_index_t wait_index;
   status_$t sts;
   linteger dma_len;
   pbu_$dma_opts_t dma_options;
   long dma_rest;
   int i, ret_sts;

   ret_sts = FLOPDRV_FAIL;

   dma_len = 2 * FLOPDRV_TRACK_SIZE * FLOPDRV_BYTES_PER_SECTOR;

   /* copy sector buffer into dma buffer */
   for (i = 0; i < dma_len; i++) flopdrv_dma_buffer[i] = cylinder_buffer[i];

   /* start dma */
   dma_options = 0;
   pbu2_$dma_start(flopdrv_cb.pbu_unit, flopdrv_dma_channel, pbu_$dma_write,
      flopdrv_dma_buffer, flopdrv_iomap_iova, dma_len, dma_options, &sts);
   if (sts.all != status_$ok) error_$print(sts);

   /* send "write sector" command */
   flopdrv_command[0] = 0xC5;     /* multiple track, MFM */
   flopdrv_command[1] = 0x00;     /* head, drive A */
   flopdrv_command[2] = cyl_num;  /* cylinder */
   flopdrv_command[3] = 0;        /* head */
   flopdrv_command[4] = 1;        /* start sector number */
   flopdrv_command[5] = 0x02;     /* sector size = 512b */
   flopdrv_command[6] = FLOPDRV_TRACK_SIZE; /* last sector number */
   if (flopdrv_drive_type == FLOPDRV_5_25_DRIVE) 
      flopdrv_command[7] = 42;    /* GAP 3 size = 42 */
   else
      flopdrv_command[7] = 27;    /* GAP 3 size = 27 */
   flopdrv_command[8] = 0xff;     /* data length */
   if (flopdrv_put_command(flopdrv_command, 9)) {
      /* wait for DMA to complete */
      wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
      if (sts.all != status_$ok)  error_$print(sts);
      flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   
      if (wait_index == WAIT_EC_ADVANCE) {
         if (flopdrv_get_status(flopdrv_status, 7)) {
            st0 = flopdrv_status[0];
            st1 = flopdrv_status[1];
            st2 = flopdrv_status[2];
            cyl = flopdrv_status[3]; /* cylinder */
            head = flopdrv_status[4]; /* head */
            sec_num = flopdrv_status[5]; /* sector number */
            sec_size = flopdrv_status[6]; /* sector size */
            if ((st0 & 0xC0) == 0x00) {
               ret_sts = FLOPDRV_OK;
            }
            else if ((st1 & 0x02) == 0x02) {
               flopdrv_retry_ok = FLOPDRV_FALSE;
               fprintf(stderr, "ERROR: Floppy is write protected.\n");
            }
            else if (flopdrv_debug) {
               fprintf(stderr, "ST0=%X, ST1=%X, ST2=%X, cyl=%d, head=%d, sec=%d, siz=%d\n",
                  (int)st0, (int)st1, (int)st2, (int)cyl, (int)head, (int)sec_num, (int)sec_size);
               flopdrv_print_status_info(0, st0);
               flopdrv_print_status_info(1, st1);
               flopdrv_print_status_info(2, st2);
            }
         }
      }
   }

   /* stop DMA operation, may not run to completion if command fails */
   dma_rest = pbu2_$dma_stop(flopdrv_cb.pbu_unit, flopdrv_dma_channel, &sts);
   /* if (sts.all != status_$ok) error_$print(sts); */

   return ret_sts;
}


/*
 * flopdrv_format_track_cmd
 *   Formats a single track according to drive/disk combination. 
 */
int flopdrv_format_track_cmd(
   unsigned int cyl_num,
   unsigned int head_num
)
{
   unsigned char st0, st1, st2, cyl, head, sec_num, sec_size;
   pbu_$wait_index_t wait_index;
   status_$t sts;
   pinteger dma_len;
   pbu_$dma_opts_t dma_options;
   long dma_rest;
   int i, j, ret_sts;

   ret_sts = 0;
      
   /* prepare format buffer (contains for each sector: cyl, head, sec_num, size) */
   for (i = 1; i <= FLOPDRV_TRACK_SIZE; i++) {
      j = (i-1)*4;
      flopdrv_dma_buffer[j] = cyl_num;     /* cylinder num for this sector */
      flopdrv_dma_buffer[j+1] = head_num;  /* head num for this sector */
      flopdrv_dma_buffer[j+2] = i;         /* its own sector number */
      flopdrv_dma_buffer[j+3] = 2;         /* sector size is 512 bytes */
   }

   /* start dma */
   dma_len = FLOPDRV_TRACK_SIZE*4;
   dma_options = 0;
   pbu_$dma_start(flopdrv_cb.pbu_unit, flopdrv_dma_channel, pbu_$dma_write,
      flopdrv_dma_buffer, dma_len, dma_options, &sts);
   if (sts.all != status_$ok) error_$print(sts);

   /* send "format track" command */
   flopdrv_command[0] =  0x4D;     /* MFM */
   if (head_num == 0)
      flopdrv_command[1] = 0x00;
   else
      flopdrv_command[1] = 0x04;   /* head, drive A */
   flopdrv_command[2] = 0x02;      /* sector size = 512b */
   flopdrv_command[3] = FLOPDRV_TRACK_SIZE;  /* track size, num sectors per track */
   if (flopdrv_drive_type == FLOPDRV_5_25_DRIVE) 
      flopdrv_command[4] = 80;     /* GAP 3 size for 5.25" */
   else
      flopdrv_command[4] = 84;     /* GAP 3 size for 3.5" */
   flopdrv_command[5] = 0xf6;      /* format filler byte, standard value */
   if (flopdrv_put_command(flopdrv_command, 6)) {
   
      /* wait for DMA to complete */
      wait_index = pbu_$wait(flopdrv_cb.pbu_unit, flopdrv_device_timeout, flopdrv_quit_enable, &sts);
      if (sts.all != status_$ok)  error_$print(sts);
      flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   
      if (wait_index == WAIT_EC_ADVANCE) {
         if (flopdrv_get_status(flopdrv_status, 7)) {
            st0 = flopdrv_status[0];
            st1 = flopdrv_status[1];
            st2 = flopdrv_status[2];
            cyl = flopdrv_status[3]; /* cylinder */
            head = flopdrv_status[4]; /* head */
            sec_num = flopdrv_status[5]; /* sector number */
            sec_size = flopdrv_status[6]; /* sector size */
            if ((st0 & 0xC0) == 0x00) {
               ret_sts = FLOPDRV_OK;
            }
            else if ((st1 & 0x02) == 0x02) {
               fprintf(stderr, "ERROR: Floppy is write protected.\n");
               flopdrv_retry_ok = FLOPDRV_FALSE;
            }
            else if (flopdrv_debug) {
               fprintf(stderr, "ST0=%X, ST1=%X, ST2=%X, cyl=%d, head=%d, sec=%d, siz=%d\n",
                  (int)st0, (int)st1, (int)st2, (int)cyl, (int)head, (int)sec_num, (int)sec_size);
               flopdrv_print_status_info(0, st0);
               flopdrv_print_status_info(1, st1);
               flopdrv_print_status_info(2, st2);
            }
         }
      }
   }

   /* stop DMA operation, DMA may not run to completion if command fails */
   dma_rest = pbu_$dma_stop(flopdrv_cb.pbu_unit, flopdrv_dma_channel, &sts);
   /* if (sts.all != status_$ok) error_$print(sts); */

   return ret_sts;
}


/*
 * basic control functions
 */

/*
 * flopdrv_put_command
 *   sends commands to the floppy controller. A command is a byte sequence
 *   containing an opcode with modifiers and parameters. Commands are fed to
 *   the data register. The MRQ (Main Request ?) bit in the status register
 *   indicates that the data register is ready to accept a command byte.
 */
int flopdrv_put_command(
   unsigned char command[],
   unsigned int num_bytes
)
{
   int mrq, i, j, timed_out;

   flopdrv_cb.command_sts = FLOPDRV_CMD_BUSY;
   for (i = 0; i < num_bytes; i++) {
      timed_out = FLOPDRV_TRUE;
      for (j = 0; j < FLOPDRV_TIMEOUT_COUNT; j++) {
         mrq = flopdrv_cb.csr_p->stat_reg & 0x80;
         if (mrq == 0x80) {
            flopdrv_cb.csr_p->data_reg = command[i];
            timed_out = FLOPDRV_FALSE;
            break;
         }
      }
   }
   if (timed_out) flopdrv_cb.command_sts = FLOPDRV_CMD_NONE;
   return !timed_out;
}


/*
 * flopdrv_get_status
 *   most controller commands generate an interrupt when completed and
 *   return a status report to indicate success or failure of the command.
 *   Status reports are byte sequences (variable in length) that must be
 *   read from the data register otherwise the controller will not accept
 *   other commands. The MRQ (Main Request ?) bit in the status register
 *   indicates that the data register contains a valid status byte.
 */
int flopdrv_get_status(
   unsigned char status[],
   unsigned int num_bytes
)
{
   int mrq, i, j, timed_out;

   for (i = 0; i < num_bytes; i++) {
      timed_out = FLOPDRV_TRUE;
      for (j = 0; j < FLOPDRV_TIMEOUT_COUNT; j++) {
         mrq = flopdrv_cb.csr_p->stat_reg & 0x80;
         if (mrq == 0x80) {
            status[i] = flopdrv_cb.csr_p->data_reg;
            timed_out = FLOPDRV_FALSE;
            break;
         }
      }
   }
   return !timed_out;
}


/*
 * debug functions
 */

/*
 * flopdrv_print_status_reg
 *   Prints out the bits in the status register in a human-readable format.
 *   The status register layout is as follows:
 *
 *        bit7                               bit0
 *        MRQ  DIO  NDMA BUSY ACTD ACTC ACTB ACTA 
 *
 *   Where:
 *        MRQ  : Main request (1 = data reg ready, 0 = data reg not-ready)
 *        DIO  : Data IO (1 = FDC -> CPU, 0 = CPU -> FDC)
 *        NDMA : No-DMA mode (0 = FDC in DMA mode (normal))
 *        BUSY : FDC busy flag (1 = FDC busy, 0 = FDC not executing a command)
 *        ACTD-ACTA : Drive D-A seeking (1 = seeking, 0 = not-seeking)
 */
void flopdrv_print_status_reg()
{
   unsigned char reg;
   reg = flopdrv_cb.csr_p->stat_reg;
   fprintf(stderr, "Status Reg: ");
   if (reg & 0x80) fprintf(stderr,"MRQ "); else fprintf(stderr, "noMRQ ");
   if (reg & 0x40) fprintf(stderr,"DOUT "); else fprintf(stderr, "DIN ");
   if (reg & 0x20) fprintf(stderr,"noDMA "); else fprintf(stderr, "DMA ");
   if (reg & 0x10) fprintf(stderr,"BUSY "); else fprintf(stderr, "READY ");
   if (reg & 0x08) fprintf(stderr,"ACTD "); else fprintf(stderr, "noACTD ");
   if (reg & 0x04) fprintf(stderr,"ACTC "); else fprintf(stderr, "noACTC ");
   if (reg & 0x02) fprintf(stderr,"ACTB "); else fprintf(stderr, "noACTB ");
   if (reg & 0x01) fprintf(stderr,"ACTA "); else fprintf(stderr, "noACTA ");
   fprintf(stderr, "\n");
}


/*
 * flopdrv_print_status_info
 */
void flopdrv_print_status_info(
   unsigned int stat_type,
   unsigned char stat
)
{
   if (stat_type == 0) {
      fprintf(stderr, "ST0= ");
      if ((stat & 0xC0) == 0x00) fprintf(stderr, "NORMAL_TERM ");
      if ((stat & 0xC0) == 0x40) fprintf(stderr, "ABNORMAL_TERM ");
      if ((stat & 0xC0) == 0x80) fprintf(stderr, "INVALID_CMD ");
      if ((stat & 0xC0) == 0xC0) fprintf(stderr, "ABNORMAL_POLL ");
      if ((stat & 0x20) == 0x20) fprintf(stderr, "SEARCH_END ");   /* seek end */
      if ((stat & 0x10) == 0x10) fprintf(stderr, "CALIB_ERR ");    /* equipment check error */
      if ((stat & 0x08) == 0x08) fprintf(stderr, "NOT_READY ");    /* not ready */
      if ((stat & 0x04) == 0x04) fprintf(stderr, "HEAD_1 ");       /* head */
      if ((stat & 0x04) == 0x00) fprintf(stderr, "HEAD_0 ");
      if ((stat & 0x03) == 0x00) fprintf(stderr, "DRIVE_O ");      /* drive select mask */
      if ((stat & 0x03) == 0x01) fprintf(stderr, "DRIVE_1 ");
      if ((stat & 0x03) == 0x02) fprintf(stderr, "DRIVE_2 ");
      if ((stat & 0x03) == 0x03) fprintf(stderr, "DRIVE_3 ");
      fprintf(stderr, "\n");
   }
   else if (stat_type == 1) {
      fprintf(stderr, "ST1= ");
      if ((stat & 0x80) == 0x80) fprintf(stderr, "CYLINDER_END ");  /* end of cylinder */
      if ((stat & 0x20) == 0x20) fprintf(stderr, "CRC_ERR ");       /* CRC error in data or addr */
      if ((stat & 0x10) == 0x10) fprintf(stderr, "TIMEOUT ");       /* CPU signal timeout */
      if ((stat & 0x04) == 0x04) fprintf(stderr, "NO_DATA ");       /* no data, unreadable */
      if ((stat & 0x02) == 0x02) fprintf(stderr, "WRITE_PROTECT "); /* write protected */
      if ((stat & 0x01) == 0x01) fprintf(stderr, "NO_ID ");         /* missing address mark */
      fprintf(stderr, "\n");
   }
   else if (stat_type == 2) {
      fprintf(stderr, "ST2= ");
      if ((stat & 0x40) == 0x40) fprintf(stderr, "ZERO_ADDR ");     /* control mark = deleted */
      if ((stat & 0x20) == 0x20) fprintf(stderr, "CRC_DATA ");      /* CRC error in data field */
      if ((stat & 0x10) == 0x10) fprintf(stderr, "WRONG_CYL ");     /* wrong cylinder */
      if ((stat & 0x04) == 0x04) fprintf(stderr, "SEARCH_ERR ");    /* scan not satified */
      if ((stat & 0x02) == 0x02) fprintf(stderr, "BAD_CYL ");       /* bad cylinder */
      if ((stat & 0x01) == 0x01) fprintf(stderr, "NO_ADDR ");       /* missing address mark */
      fprintf(stderr, "\n");
   }
   else if (stat_type == 3) {
      fprintf(stderr, "ST3= ");
      if ((stat & 0x40) == 0x40) fprintf(stderr, "WRITE_PROTECT "); /* write protect */
      if ((stat & 0x10) == 0x10) fprintf(stderr, "TRACK_O ");       /* track zero signal (1=track 0) */
      if ((stat & 0x08) == 0x08) fprintf(stderr, "DOUBLE_SIDED ");
      if ((stat & 0x04) == 0x04) fprintf(stderr, "HDSEL ");
      if ((stat & 0x02) == 0x02) fprintf(stderr, "DS1 ");
      if ((stat & 0x01) == 0x01) fprintf(stderr, "DS0 ");
      fprintf(stderr, "\n");
   }
   else fprintf(stderr, "STATUS: Bummer!!!\n");
}


/*
 * flopdrv_dump_sector
 *   Dumps a sector in decimal, hex and ascii to stdout.
 */
void flopdrv_dump_sector(
   unsigned int cyl_num,
   unsigned int head_num,
   unsigned int sect_num
)
{
   int i, j;
   unsigned char c, sector_buffer[FLOPDRV_BYTES_PER_SECTOR];

   if (!flopdrv_read_sector(cyl_num, head_num, sect_num, sector_buffer)) return;

   for (j=0; j<16; j++) {
      /* dump decimal */
      for (i=0; i<32; i++) printf("%d ", sector_buffer[j*32+i]);
      printf("\n");
   
      /* dump hex */
      for (i=0; i<32; i++) {
         c = sector_buffer[j*32+i];
         printf("%x ", c);
      }
      printf("\n");
   
      /* dump ascii */
      for (i=0; i<32; i++) {
         c = sector_buffer[j*32+i];
         if (c > 32) printf("%c", c); else printf(".");
      }
      printf("\n");
   }
}


#ifdef NEVER
/*
 * flopdrv_test
 *   A sequence of basic commands, used during development of this driver.
 */
int flopdrv_test()
{
   unsigned char sector_buffer[FLOPDRV_BYTES_PER_SECTOR];
   int i;

   flopdrv_start_drive_cmd();
   flopdrv_calibrate_drive_cmd();

   flopdrv_position_head_cmd(40);
   flopdrv_read_id_cmd();
   flopdrv_read_sector(1, 1, 5, sector_buffer);
   flopdrv_dump_sector(0, 0, 1);

/*
   flopdrv_position_head_cmd(40);
   flopdrv_read_id_cmd();
*/

   flopdrv_format_disk();
   flopdrv_dump_sector(0, 0, 1);

   for (i = 0; i < FLOPDRV_BYTES_PER_SECTOR; i++) sector_buffer[i] = 0xA4;
   flopdrv_write_sector(1, 1, 5, sector_buffer);
   flopdrv_dump_sector(1, 1, 5);

   flopdrv_stop_drive();
}
#endif

