/*
 * Add an MSDOS filesystem to a low level formatted diskette.
 *
 * Emmet P. Gray			US Army, HQ III Corps & Fort Hood
 * ...!uunet!uiucuxc!fthood!egray	Attn: AFZF-DE-ENV
 * fthood!egray@uxc.cso.uiuc.edu	Directorate of Engineering & Housing
 * 					Environmental Management Office
 * 					Fort Hood, TX 76544-5057
 */

#ifdef apollo

/*
 *   low-level formatting for Mtools on Apollo
 *   Jef Rijmenants - 19 july 1993
 */

#include <stdio.h>
#include <ctype.h>
#include "msdos.h"
#include "patchlevel.h"
#include "flopdrv.h"

#define SECTOR_SIZE 512  
#define DIR_ENTRY_SIZE 32
#define FAT_BYTE_SIZE ((0xfff * 3) / 2)
#define RESERVED_SECTORS 1
#define FAT_COPIES 2
#define NUM_SIDES 2
#define HIDDEN_SECTORS 0

/* calculates logical sector (0-n) from track, side and sector */
#define SECTOR(track, side, sector) ((sector-1) + \
  (side+1)*(TRACK_SIZE*track))

/* floppy drive type */
#define DRIVE_3_5   0
#define DRIVE_5_25  1

/* floppy disk type */
#define DSHD_FLOPPY  0
#define DSDD_FLOPPY  1

/* configuration */
unsigned int drive_type;  /* indicates drive type (3.5" or 5.25") */
unsigned int disk_type;   /* indicates disk type (DSHD or DSDD) */

/*
    Drive type          | 5.25" | 5.25" | 3.5"  | 3.5"  | 3.5"
    Floppy type         | DSDD  | DSHD  | DSDD  | DSHD  | DSED
    Capacity            | 360k  | 1.2M  | 720k  | 1.44M | 2.88M
    Number of tracks    | 40    | 80    | 80    | 80    | 80
    Sectors per track   | 9     | 15    | 9     | 18    | 36
    Data transfer speed | 250   | 500   | 250   | 500   | 1000
*/

/*                                   3HD   3DD     5HD   5DD  */
unsigned int num_tracks[2][2]   = {{  80,   80}, {  80,   40}};  /* num tracks */
unsigned int track_size[2][2]   = {{  18,    9}, {  15,    9}};  /* sectors per track */
unsigned int num_sectors[2][2]  = {{2880, 1440}, {2400,  720}};  /* number of sectors */
unsigned int dir_entries[2][2]  = {{ 224,  112}, { 224,  112}};  /* number of root dir entries */
unsigned int FAT_size[2][2]     = {{   9,    5}, {   9,    5}};  /* size of FAT */
unsigned int cluster_size[2][2] = {{   1,    2}, {   1,    2}};  /* cluster size in sectors */
unsigned int format_code[2][2]  = {{0xF0, 0xF9}, {0xF9, 0xFD}};  /* MSDOS format code */

#define NUM_TRACKS    num_tracks[drive_type][disk_type]
#define TRACK_SIZE    track_size[drive_type][disk_type]
#define NUM_SECTORS   num_sectors[drive_type][disk_type]
#define DIR_ENTRIES   dir_entries[drive_type][disk_type]
#define FAT_SIZE      FAT_size[drive_type][disk_type]
#define CLUSTER_SIZE  cluster_size[drive_type][disk_type]
#define FORMAT_CODE   format_code[drive_type][disk_type]

typedef unsigned char  t_byte;


/* shut up linker */
int fd, dir_dirty;
long dir_chain[MAX_DIR_SECS];
unsigned char *dir_buf;
void disk_write() {}

unsigned char label[12];

/*
 * write_sector
 *   Writes one logical sector.
 */
int write_sector(
   int  sector_id,   /* logical sector number 0-n */
   t_byte sector_buf[],
   int  sector_size
)
{
   unsigned int cyl, head, sector;

   cyl = sector_id / (2*TRACK_SIZE);
   head = (sector_id / TRACK_SIZE) & 0x0001;
   sector = sector_id + 1 - head*TRACK_SIZE - 2*cyl*TRACK_SIZE;
   return flopdrv_write_sector(cyl, head, sector, sector_buf);
}


/*
 * flop_format
 */
int flop_format()
{
   int i, j, first_sector, last_sector, copy;
   t_byte sector_buf[SECTOR_SIZE];
   t_byte FAT[FAT_BYTE_SIZE];
   long time(), now;
   char *memcpy();
   struct directory *dir, *mk_entry();

   /* low-level formatting */
   if (!flopdrv_format_disk()) return 0;

   /* initialize and write boot_sector */
   for (i = 0; i < SECTOR_SIZE; i++) sector_buf[i] = 0x00;
   sector_buf[0] = 0xeb;
   sector_buf[1] = 0x34;
   sector_buf[2] = 0x90;
   sector_buf[3] = 'M';
   sector_buf[4] = 't';
   sector_buf[5] = 'o';
   sector_buf[6] = 'o';
   sector_buf[7] = 'l';
   sector_buf[8] = 's';
   sector_buf[9] = ' ';
   sector_buf[10] = ' ';
   sector_buf[11] = SECTOR_SIZE & 0xff;
   sector_buf[12] = SECTOR_SIZE / 256;
   sector_buf[13] = CLUSTER_SIZE & 0xff;
   sector_buf[14] = RESERVED_SECTORS & 0xff;
   sector_buf[15] = RESERVED_SECTORS / 256;
   sector_buf[16] = FAT_COPIES & 0xff;
   sector_buf[17] = DIR_ENTRIES & 0xff;
   sector_buf[18] = DIR_ENTRIES / 256;
   sector_buf[19] = NUM_SECTORS & 0xff;
   sector_buf[20] = NUM_SECTORS / 256;
   sector_buf[21] = FORMAT_CODE;
   sector_buf[22] = FAT_SIZE & 0xff;
   sector_buf[23] = FAT_SIZE / 256;
   sector_buf[24] = TRACK_SIZE & 0xff;
   sector_buf[25] = TRACK_SIZE / 256;
   sector_buf[26] = NUM_SIDES & 0xff;
   sector_buf[27] = NUM_SIDES / 256;
   sector_buf[28] = HIDDEN_SECTORS & 0xff;
   sector_buf[29] = HIDDEN_SECTORS / 256;
   sector_buf[510] = 0x55;
   sector_buf[511] = 0xaa;
   if (!write_sector(SECTOR(0,0,1), sector_buf, SECTOR_SIZE)) return 0;
   
   /* initialize and write FAT */
   for (i=0; i < FAT_BYTE_SIZE; i++) FAT[i] = 0x00;
   FAT[0] = FORMAT_CODE;
   FAT[1] = 0xff;
   FAT[2] = 0xff;
   for (copy=0; copy < FAT_COPIES; copy++) {
      first_sector = RESERVED_SECTORS + FAT_SIZE*copy;
      last_sector = first_sector + FAT_SIZE;
      for (i = first_sector, j = 0; i < last_sector; i++, j = j+SECTOR_SIZE)
         if (!write_sector(i, &FAT[j], SECTOR_SIZE)) return 0;
      }
   
   /* write directory sectors */
   for (i = 0; i < SECTOR_SIZE; i++) sector_buf[i] = 0x00;
   first_sector = RESERVED_SECTORS + FAT_SIZE * FAT_COPIES;
   last_sector = first_sector + (DIR_ENTRIES * DIR_ENTRY_SIZE) / SECTOR_SIZE;

   if (label[0] != '\0') {
      time(&now);
      dir = mk_entry(label, 0x08, 0, 0L, now);
      memcpy((char *) sector_buf, (char *) dir, MDIR_SIZE);
      if (!write_sector(first_sector, sector_buf, SECTOR_SIZE)) return 0;
      for (i = 0; i < SECTOR_SIZE; i++) sector_buf[i] = 0x00;
      first_sector++;
   }
   for (i = first_sector; i < last_sector; i++) {
      if (!write_sector(i, sector_buf, SECTOR_SIZE)) return 0;
   }
   return 1;
}


main(argc, argv)
int argc;
char *argv[];
{
	extern int optind;
	extern char *optarg;
	extern struct device devices[];
	int i, c, oops, tracks, heads, sectors, fat_len, dir_len, clus_size;
	char drive, *name, *expand();
	char *strncpy(), *memset(), *memcpy();
	struct device *dev;
	void exit(), perror();

	oops = 0;
	tracks = 0;
	heads = 0;
	sectors = 0;
	label[0] = '\0';
					/* get command line options */
	while ((c = getopt(argc, argv, "t:h:s:l:")) != EOF) {
		switch (c) {
			case 't':
				tracks = atoi(optarg);
				break;
			case 'h':
				heads = atoi(optarg);
				break;
			case 's':
				sectors = atoi(optarg);
				break;
			case 'l':
				sprintf((char *) label, "%-11.11s", optarg);
				break;
			default:
				oops = 1;
				break;
		}
	}

	if (oops || (argc - optind) != 1) {
		fprintf(stderr, "Mtools version %s, dated %s\n", VERSION, DATE);
		fprintf(stderr, "Usage: %s [-t tracks] [-h heads] [-s sectors] [-l label] device\n", argv[0]);
		exit(1);
	}

	drive = argv[argc -1][0];
	if (islower(drive))
		drive = toupper(drive);

					/* check out the drive letter */
	dev = devices;
	while (dev->drive) {
		if (dev->drive == drive)
			break;
		dev++;
	}
	if (!dev->drive) {
		fprintf(stderr, "Drive '%c:' not supported\n", drive);
		exit(1);
	}
	if (dev->tracks == 0) {
		fprintf(stderr, "Non-removable media is not supported\n");
		exit(1);
	}
					/* find the right one */
	if (!dev->gioctl) {
		while (dev->drive == drive) {
			if ((!tracks || dev->tracks == tracks) && (!heads || dev->heads == heads) && (!sectors || dev->sectors == sectors))
				break;
			dev++;
		}
	}
	if (dev->drive != drive) {
		fprintf(stderr, "%s: Paramaters not supported\n", argv[0]);
		exit(1);
	}

					/* fill in the blanks */
	if (!tracks)
		tracks = dev->tracks;
	if (!heads)
		heads = dev->heads;
	if (!sectors)
		sectors = dev->sectors;

					/* open the device */
	name = expand(dev->name);
   if (!flopdrv_acquire(name)) exit(1);

   drive_type = flopdrv_get_drive_type();
   disk_type = DSHD_FLOPPY;
   if (sectors == 9) disk_type = DSDD_FLOPPY;
   if (!flopdrv_start_drive(disk_type)) exit(1);
   
   if (!flop_format())
      fprintf(stderr, "ERROR: Failed to format floppy!\n");

   flopdrv_stop_drive();
   flopdrv_release();

}

#else

#include <stdio.h>
#include <ctype.h>
#include "msdos.h"
#include "patchlevel.h"

int fd, dir_dirty, dir_entries;
long dir_chain[MAX_DIR_SECS];
unsigned char *dir_buf;

main(argc, argv)
int argc;
char *argv[];
{
	extern int optind;
	extern char *optarg;
	extern struct device devices[];
	struct bootsector boot;
	int i, c, oops, tracks, heads, sectors, fat_len, dir_len, clus_size;
	int tot_sectors, num_clus, fat_guess;
	long time(), now, lseek();
	char drive, *name, *expand();
	char *strncpy(), *memset(), *memcpy();
	unsigned char media, label[12], buf[MSECTOR_SIZE];
	struct device *dev;
	struct directory *dir, *mk_entry();
	void exit(), perror();

	oops = 0;
	tracks = 0;
	heads = 0;
	sectors = 0;
	label[0] = '\0';
					/* get command line options */
	while ((c = getopt(argc, argv, "t:h:s:l:")) != EOF) {
		switch (c) {
			case 't':
				tracks = atoi(optarg);
				break;
			case 'h':
				heads = atoi(optarg);
				break;
			case 's':
				sectors = atoi(optarg);
				break;
			case 'l':
				sprintf((char *) label, "%-11.11s", optarg);
				break;
			default:
				oops = 1;
				break;
		}
	}

	if (oops || (argc - optind) != 1) {
		fprintf(stderr, "Mtools version %s, dated %s\n", VERSION, DATE);
		fprintf(stderr, "Usage: %s [-t tracks] [-h heads] [-s sectors] [-l label] device\n", argv[0]);
		exit(1);
	}

	drive = argv[argc -1][0];
	if (islower(drive))
		drive = toupper(drive);

					/* check out the drive letter */
	dev = devices;
	while (dev->drive) {
		if (dev->drive == drive)
			break;
		dev++;
	}
	if (!dev->drive) {
		fprintf(stderr, "Drive '%c:' not supported\n", drive);
		exit(1);
	}
	if (dev->tracks == 0) {
		fprintf(stderr, "Non-removable media is not supported\n");
		exit(1);
	}
					/* find the right one */
	if (!dev->gioctl) {
		while (dev->drive == drive) {
			if ((!tracks || dev->tracks == tracks) && (!heads || dev->heads == heads) && (!sectors || dev->sectors == sectors))
				break;
			dev++;
		}
	}
	if (dev->drive != drive) {
		fprintf(stderr, "%s: Paramaters not supported\n", argv[0]);
		exit(1);
	}
					/* open the device */
	name = expand(dev->name);
	if ((fd = open(name, 2 | dev->mode)) < 0) {
		perror("init: open");
		exit(1);
	}
					/* fill in the blanks */
	if (!tracks)
		tracks = dev->tracks;
	if (!heads)
		heads = dev->heads;
	if (!sectors)
		sectors = dev->sectors;

					/* set parameters, if needed */
	if (dev->gioctl) {
		if ((*(dev->gioctl)) (fd, tracks, heads, sectors))
			exit(1);
	}
					/* do a "test" read */
	if (read(fd, (char *) buf, MSECTOR_SIZE) != MSECTOR_SIZE) {
		fprintf(stderr, "%s: Error reading from '%s', wrong parameters?\n", argv[0], name);
		exit(1);
	}
					/* get the parameters */
	tot_sectors = tracks * heads * sectors;
	switch (tot_sectors) {
		case 320:		/* 40t * 1h * 8s = 160k */
			media = 0xfe;
			clus_size = 1;
			dir_len = 4;
			fat_len = 1;
			break;
		case 360:		/* 40t * 1h * 9s = 180k */
			media = 0xfc;
			clus_size = 1;
			dir_len = 4;
			fat_len = 2;
			break;
		case 640:		/* 40t * 2h * 8s = 320k */
			media = 0xff;
			clus_size = 2;
			dir_len = 7;
			fat_len = 1;
			break;
		case 720:		/* 40t * 2h * 9s = 360k */
			media = 0xfd;
			clus_size = 2;
			dir_len = 7;
			fat_len = 2;
			break;
		case 1440:		/* 80t * 2h * 9s = 720k */
			media = 0xf9;
			clus_size = 2;
			dir_len = 7;
			fat_len = 3;
			break;
		case 2400:		/* 80t * 2h * 15s = 1.2m */
			media = 0xf9;
			clus_size = 1;
			dir_len = 14;
			fat_len = 7;
			break;
		case 2880:		/* 80t * 2h * 18s = 1.44m */
			media = 0xf0;
			clus_size = 1;
			dir_len = 14;
			fat_len = 9;
			break;
		default:		/* a non-standard format */
			media = 0xf0;
			if (heads == 1)
				clus_size = 1;
			else
				clus_size = (tot_sectors > 2000) ? 1 : 2;
			if (heads == 1)
				dir_len = 4;
			else
				dir_len = (tot_sectors > 2000) ? 14 : 7;
			/*
			 * Estimate the fat length, then figure it out.  The
			 * 341 is the number of 12 bit fat entries in a sector.
			 */
			fat_guess = ((tot_sectors / clus_size) / 341.0) + 0.95;
			num_clus = (tot_sectors -dir_len - (2 * fat_guess) -1) / clus_size;
			fat_len = (num_clus / 341.0) + 1;
			break;
	}
					/* the boot sector */
	memset((char *) &boot, '\0', MSECTOR_SIZE);
	boot.jump[0] = 0xeb;
	boot.jump[1] = 0x44;
	boot.jump[2] = 0x90;
	strncpy((char *) boot.banner, "Mtools  ", 8);
	boot.secsiz[0] = 512 % 0x100;
	boot.secsiz[1] = 512 / 0x100;
	boot.clsiz = (unsigned char) clus_size;
	boot.nrsvsect[0] = 1;
	boot.nrsvsect[1] = 0;
	boot.nfat = 2;
	boot.dirents[0] = (dir_len * 16) % 0x100;
	boot.dirents[1] = (dir_len * 16) / 0x100;
	boot.psect[0] = tot_sectors % 0x100;
	boot.psect[1] = tot_sectors / 0x100;
	boot.descr = media;
	boot.fatlen[0] = fat_len % 0x100;
	boot.fatlen[1] = fat_len / 0x100;
	boot.nsect[0] = sectors % 0x100;
	boot.nsect[1] = sectors / 0x100;
	boot.nheads[0] = heads % 0x100;
	boot.nheads[1] = heads / 0x100;

					/* write the boot */
	lseek(fd, 0L, 0);
	write(fd, (char *) &boot, MSECTOR_SIZE);
					/* first fat */
	memset((char *) buf, '\0', MSECTOR_SIZE);
	buf[0] = media;
	buf[1] = 0xff;
	buf[2] = 0xff;
	write(fd, (char *) buf, MSECTOR_SIZE);
	memset((char *) buf, '\0', MSECTOR_SIZE);
	for (i = 1; i < fat_len; i++)
		write(fd, (char *) buf, MSECTOR_SIZE);
					/* second fat */
	buf[0] = media;
	buf[1] = 0xff;
	buf[2] = 0xff;
	write(fd, (char *) buf, MSECTOR_SIZE);
	memset((char *) buf, '\0', MSECTOR_SIZE);
	for (i = 1; i < fat_len; i++)
		write(fd, (char *) buf, MSECTOR_SIZE);
					/* the root directory */
	if (label[0] != '\0') {
		time(&now);
		dir = mk_entry(label, 0x08, 0, 0L, now);
		memcpy((char *) buf, (char *) dir, MDIR_SIZE);
	}
	write(fd, (char *) buf, MSECTOR_SIZE);
	memset((char *) buf, '\0', MSECTOR_SIZE);
	for (i = 1; i < dir_len; i++)
		write(fd, (char *) buf, MSECTOR_SIZE);
	close(fd);
	exit(0);
}

/* hooks for the missing parts */
void disk_write() {}

#endif

