/*
 * Do full cylinder buffered reads from slow devices.  Uses a simple
 * buffered read/delayed write algorithm.
 */

#include <stdio.h>
#include "msdos.h"
#ifdef apollo
#include "flopdrv.h"
#endif

unsigned char *disk_buf;		/* disk read/write buffer */
int disk_size;				/* size of read/write buffer */
long disk_current;			/* first sector in buffer */
int disk_dirty;				/* is the buffer dirty? */

extern int fd;
extern long disk_offset;


#ifdef apollo
extern unsigned int flop_sectors_per_track;

void
disk_read(
   long start,
   unsigned char *buf,
   int len
)
{
   unsigned int cyl, head, sect;
   unsigned int track_size;
   unsigned int i, sector, num_sectors;
   unsigned char *buf_ptr, *disk_ptr;
   char *memcpy();
   void exit(), disk_flush();

   num_sectors = len / MSECTOR_SIZE;
   track_size = flop_sectors_per_track;

   if (disk_size == 1) {
      /* no cylinder caching, read sector by sector */
      for (i = 0; i < num_sectors; i++) {
         sector = start+i;
         cyl = sector / (2*track_size);
         head = (sector / track_size) & 0x0001;
         sect = sector + 1 - head*track_size - 2*cyl*track_size;
         if (!flopdrv_read_sector(cyl, head, sect, &buf[i*MSECTOR_SIZE])) {
            fprintf(stderr, "disk_read: flopdrv_read_sector failed\n");
            exit(1);
         }
      }
   }
   else {
      /* cylinder caching */
      for (i = 0; i < num_sectors; i++) {
         sector = start+i;
         if (sector < disk_current || sector >= disk_current + disk_size) {
            /* a "cache" miss */
            if (disk_dirty) disk_flush();
            disk_current = (sector / disk_size) * disk_size;
            cyl = disk_current / (2*track_size);
            if (!flopdrv_read_cylinder(cyl, disk_buf)) {
               fprintf(stderr, "disk_read: flopdrv_read_cylinder failed\n");
               exit(1);
            }
         }
         /* a cache hit... */
         buf_ptr = buf + i*MSECTOR_SIZE;
         disk_ptr = disk_buf + ((sector - disk_current) * MSECTOR_SIZE);
         memcpy((char *) buf_ptr, (char *) disk_ptr, MSECTOR_SIZE);
      }
   }
}

#else

void
disk_read(start, buf, len)
long start;
unsigned char *buf;
int len;
{
	register long i;
	int length;
	unsigned char *buf_ptr, *disk_ptr;
	char *memcpy();
	long where, tail, lseek();
	void perror(), exit(), disk_flush();

					/* don't use cache? */
	if (disk_size == 1) {
		where = (start * MSECTOR_SIZE) + disk_offset;
		if (lseek(fd, where, 0) < 0) {
			perror("disk_read: lseek");
			exit(1);
		}
					/* read it! */
		if (read(fd, (char *) buf, (unsigned int) len) != len) {
			perror("disk_read: read");
			exit(1);
		}
		return;
	}

	tail = start + (len / MSECTOR_SIZE) - 1;
	for (i = start; i <= tail; i++) {
					/* a "cache" miss */
		if (i < disk_current || i >= disk_current + disk_size) {

			if (disk_dirty)
				disk_flush();

			disk_current = (i / disk_size) * disk_size;
			where = (disk_current * MSECTOR_SIZE) + disk_offset;
			length = disk_size * MSECTOR_SIZE;

					/* move to next location */
			if (lseek(fd, where, 0) < 0) {
				perror("disk_read: lseek");
				exit(1);
			}
					/* read it! */
			if (read(fd, (char *) disk_buf, (unsigned int) length) != length) {
				perror("disk_read: read");
				exit(1);
			}
		}
					/* a cache hit... */
		buf_ptr = buf + ((i - start) * MSECTOR_SIZE);
		disk_ptr = disk_buf + ((i - disk_current) * MSECTOR_SIZE);
		memcpy((char *) buf_ptr, (char *) disk_ptr, MSECTOR_SIZE);
	}
	return;
}
#endif
