/*****************************************************************************
    Exabyte Driver for Boeing FCTR

    11/21/92    mdl     written
    12/10/92    mdl     made more robust
    01/15/93    mdl     fixed several minor bugs, changed call format to K&R
                        per Tom Crewson (Boeing) instructions
    02/01/93    mdl     changed timeout to vary between 90 seconds & 140 minutes
    02/02/93    mdl     reduced data structures to range [0-6] from [0-7]
    02/08/93    mdl     added ex_prewired and ex_preunwired calls
    09/24/93    mdl     added "short erase" to erase call. OK, so it's not so clean....
****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include </usr/apollo/include/base.h>
#include </usr/apollo/include/name.h>
#include </usr/apollo/include/scsi.h>
#include </usr/apollo/include/pfm.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include </mts/driver.h>

/*-----------------------------------*/
#define cmd_test_unit_ready		0x00
#define cmd_rewind				0x01
#define cmd_request_status		0x03
#define cmd_read_block_limits	0x05
#define cmd_read				0x08
#define cmd_write				0x0a
#define cmd_write_filemarks		0x10
#define cmd_space				0x11
#define cmd_inquiry				0x12
#define cmd_verify              0x13
#define cmd_mode_select			0x15
#define cmd_erase				0x19
#define cmd_mode_sense			0x1a
#define cmd_unload				0x1b
#define cmd_locate              0x2b
#define cmd_read_position       0x34
/*-----------------------------------*/

typedef union {
      scsi_$ext_sense_data_t sense;
      scsi_$buffer_t         buffer;
} scsi_data_buffer_t;

static scsi_$handle_t  scsi_handle[7];          /* SCSI manager handle for each SCSI ID */
static scsi_$cdb_t     scsi_cdb[7];             /* SCSI CDB for each SCSI ID */
static pfm_$cleanup_rec scsi_cleanup[7];        /* cleanup record for each SCSI ID */
static boolean         scsi_id_initialized[7] = {false,false,false,false,false,false,false};
                                                /* set to true if manager handle for this ID is valid */
static char            *wired_buffer_ptr[7];    /* addresses of wired area(s) */
static linteger        wired_buffer_length[7];  /* length of wired buffer area(s) */
static linteger        timeout[7];              /* number of seconds to wait before timeout */
static boolean         buffer_wired[7] = {false,false,false,false,false,false,false};
                                                /* true if buffer(s) wired */
static boolean         driver_has_sense[7] = {false,false,false,false,false,false,false};
                                                /* true if we've stored sense data for this ID */
boolean                ex_prewired[7];          /* true if buffer(s) prewired */
linteger               scsi_$max_xfer_length;   /* This is the most the system can transfer */

scsi_data_buffer_t     request_sense_data;
scsi_$ext_sense_data_t mode_sense_data;
int                    scsi_dev, space_count;
pinteger               scsi_device_namelen, arglen, infolen, return_count;
scsi_$info_t           info;
status_$t              domain_status;
scsi_$operation_id_t   scsi_op;
scsi_$wait_index_t     wait_index;
scsi_$op_status_t      scsi_status_list;
driver_$status_t       scsi_stat;
/* --------------------------------------------------------------------- */
union {
  struct {                                     /* extended sense (byte:bit position) */  
       unsigned short     valid:1;             /* (0:7) validity of information bytes */
       unsigned short     pad2:7;              /* (0:6-0) err class & code are fixed */
       unsigned short     segment:8;           /* (1) segment number */
       unsigned short     filemark:1;          /* (2:7) at a filemark */
       unsigned short     eom:1;               /* (2:6) End Of Medium */
       unsigned short     ili:1;               /* (2:5) Incorrect Length Indicator */
       unsigned short     pad3:1;              /* (2:4) reserved for future use */
       unsigned short     sense_key:4;         /* (2:3-0) */
       unsigned short     info_length1:8;      /* (3-6) Information bytes */
       unsigned short     info_length2:8;      /* (3-6) Information bytes */
       unsigned short     info_length3:8;      /* (3-6) Information bytes */
       unsigned short     info_length4:8;      /* (3-6) Information bytes */
       unsigned short     add_len:8;           /* (7)   length of additional sense data */
       unsigned short     rsrvd_8_9:16;        /* (8-9) reserved */
       unsigned short     rsrvd_10:8;          /* (10)  reserved */
       unsigned short     underover:8;         /* (11)  underrun/overrun counter */
       unsigned short     add_sense:8;         /* (12)  additional sense code */
       unsigned short     add_qual:8;          /* (13)  additional sense code qualifier */
       unsigned short     rsrvd_14_15:16;      /* (14-15) reserved */
       unsigned short     rd_wr_cnt:24;        /* (16-18) read/write error counter */
       unsigned short     pf:1;                /* (19:7)  power-fail/reset */
       unsigned short     bpe:1;               /* (19:6)  SCSI bus parity error */
       unsigned short     fpe:1;               /* (19:5)  data buffer parity error */
       unsigned short     me:1;                /* (19:4)  media error */
       unsigned short     eco:1;               /* (19:3)  error counter overflow */
       unsigned short     tme:1;               /* (19:2)  tape motion error */
       unsigned short     tnp:1;               /* (19:1)  tape not present */
       unsigned short     lbot:1;              /* (19:0)  tape at LBOT */
       unsigned short     rsvd_10_7:1;         /* (20:7)  reserved */
       unsigned short     tmd:1;               /* (20:6)  tape mark detect error */
       unsigned short     wp:1;                /* (20:5)  write protect */
       unsigned short     fmke:1;              /* (20:4)  filemark write error */
       unsigned short     ure:1;               /* (20:3)  data underrun error */
       unsigned short     wei:1;               /* (20:2)  max rewrites exceeded */
       unsigned short     sse:1;               /* (20:1)  servo system error */
       unsigned short     fe:1;                /* (20:0)  data formatter error */
       unsigned short     rsvd_21_7:2;         /* (21:7-6)  reserved */
       unsigned short     rrr:1;               /* (21:5)  reverse retries required */
       unsigned short     clnd:1;              /* (21:4)  drive has just been cleaned */
       unsigned short     cln:1;               /* (21:3)  time to clean drive */
       unsigned short     peot:1;              /* (21:2)  tape at PEOT */
       unsigned short     wseb:1;              /* (21:1)  write splice error (blank tape) */
       unsigned short     wseo:1;              /* (21:0)  write splice error (Overshoot) */
       unsigned short     rsrvd_22:8;          /* (22)  reserved */
       unsigned short     tape_left:24;        /* (23-25) tape remaining */
       unsigned short     track_retries:8;     /* (26)  tracking retry counter */
       unsigned short     rw_retries:8;        /* (27)  read/write retry counter */
       unsigned short     fault_code:8;        /* (28)  fault symptom code */
  } sd;
  unsigned char all[29];
} ex_sense_data;
unsigned char   stored_sense[7][29];            /* where we store sense data for each SCSI ID */
/* ---------------------------------------------------------------------
    Initialize the device library
   --------------------------------------------------------------------- */
void ex_initialize(scsi_id, d_status)
unsigned char scsi_id;
status_$t *d_status;
{
  status_$t temp_status;
  pinteger namelen;
  static char device_name[]="/dev/scsi0"; /* name of scsi ddf for exabyte */
  int i;

  if (scsi_id_initialized[scsi_id])
  {
    (*d_status).all = scsi_$already_acquired;
    return;
  }
  if (scsi_id > 7)
  {
    (*d_status).all = scsi_$bad_parm;
    return;
  }

  namelen = sizeof (device_name) - 1;
  device_name[namelen-1] = device_name[namelen-1] + scsi_id;
  scsi_$acquire( device_name, namelen, &scsi_handle[scsi_id], d_status);
  if ((*d_status).all != status_$ok)
  {
    /*
    printf("Unable to open %s\n", device_name);
    pfm_$error_trap((*d_status));
    */
    return;
  }

  infolen = 8; /* length of the info block */
  scsi_$get_info( scsi_handle[scsi_id], infolen, &info, d_status);
  if ((*d_status).all != status_$ok)
  {
    /*
    printf("Unable to obtain system scsi information - %h\n",(*d_status).all);
    */
    ex_terminate( scsi_id, &temp_status);      /* release the device */
    return;
  }

  scsi_$max_xfer_length = info.max_xfer;
  scsi_id_initialized[scsi_id] = true;
  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  ex_prewired[scsi_id] = false;                /* nothing wired (yet) */
  timeout[scsi_id] = 90*1000;                  /* set timeout to 90 seconds */
}
/* ---------------------------------------------------------------------
   Close & release scsi device
   ---------------------------------------------------------------------
*/
void ex_terminate( scsi_id, d_status)
status_$t *d_status;
unsigned char scsi_id;
{
  if (scsi_id_initialized[scsi_id])
  {
    scsi_id_initialized[scsi_id] = false;
    scsi_$release(scsi_handle[scsi_id], d_status);
  }
}
/* --------------------------------------------------------------------- */
void ex_prewire( scsi_id, buffer_ptr, length, status)
unsigned char scsi_id;
linteger length;
scsi_$buffer_t buffer_ptr;
status_$t *status;
{

/* I/O buffers must be in physical memory and stay there during I/O.
   The wire & unwire calls fix the buffers in memory and release them
   from memory, respectively */

  (*status).all = status_$ok;
  if (length == 0) return;
  if (ex_prewired[scsi_id])
  {
    if (buffer_wired[scsi_id])                          /* search for and unwire all wired buffers */
    {
      scsi_$unwire( scsi_handle[scsi_id], wired_buffer_ptr[scsi_id], wired_buffer_length[scsi_id], false, status);
      buffer_wired[scsi_id] = false;
    }
  }
  scsi_$wire( scsi_handle[scsi_id], buffer_ptr, length, status);
  if ((*status).all != status_$ok) return;
  ex_prewired[scsi_id] = true;
  buffer_wired[scsi_id] = true;                   /* remember we're wired */
  wired_buffer_ptr[scsi_id] = buffer_ptr;
  wired_buffer_length[scsi_id] = length;
  return;
}
/* --------------------------------------------------------------------- */
void ex_preunwire( scsi_id, status )
unsigned char scsi_id;
status_$t *status;
{

  if (ex_prewired[scsi_id])
  {
    if (buffer_wired[scsi_id])                          /* search for and unwire all wired buffers */
    {
      scsi_$unwire( scsi_handle[scsi_id], wired_buffer_ptr[scsi_id], wired_buffer_length[scsi_id], false, status);
      buffer_wired[scsi_id] = false;
    }
    else (*status).all = status_$ok;
  }
  else (*status).all = status_$ok;
  ex_prewired[scsi_id] = false;
  return;
}
/* --------------------------------------------------------------------- */
execute_command( scsi_id, buffer_ptr, length, scsi_dir)
unsigned char scsi_id;
linteger length;
scsi_$buffer_t buffer_ptr;
scsi_$direction_t scsi_dir;
{
status_$t wait_status;
int i;

/* I/O buffers must be in physical memory and stay there during I/O.
   The wire & unwire calls fix the buffers in memory and release them
   from memory, respectively */

  if ((length != 0) && (ex_prewired[scsi_id] == 0))
  {

  /* Play it safe and establish a cleanup handler in case the program dies during
     I/O. Don't want to leave bits & pieces of memory wired if we can help it. */

    domain_status = pfm_$cleanup( &scsi_cleanup[scsi_id] );
    if (domain_status.all != pfm_$cleanup_set)        /* if !=, then we're in a fault situation */
    {
      for (i=0;i<8;i++)
      {
        if (buffer_wired[i])                          /* search for and unwire all wired buffers */
        {
          scsi_$unwire( scsi_handle[i], wired_buffer_ptr[i], wired_buffer_length[i], false, &wait_status);
          buffer_wired[i] = false;
        }
      }
      pfm_$signal( domain_status );                   /* go to the next cleanup level */
    }
    scsi_$wire( scsi_handle[scsi_id], buffer_ptr, length, &domain_status);
    if (domain_status.all != status_$ok)
    {
      /*
      printf("(execute_command) Unable to wire data buffer\n");
      */
      pfm_$error_trap(domain_status);
      return(10);
    }
    buffer_wired[scsi_id] = true;                   /* remember we're wired */
    wired_buffer_ptr[scsi_id] = buffer_ptr;
    wired_buffer_length[scsi_id] = length;
  }

  scsi_$do_command( scsi_handle[scsi_id], scsi_cdb[scsi_id], buffer_ptr, length, scsi_dir, &scsi_op, &domain_status);
  if (domain_status.all != status_$ok)
  {
    /* printf("(execute_command) Unable to execute scsi command\n"); */
    if ((length != 0) && (ex_prewired[scsi_id] == 0))
    {
      scsi_$unwire( scsi_handle[scsi_id], buffer_ptr, length, false, &domain_status);
      buffer_wired[scsi_id] = false;                                /* forget we're wired */
      pfm_$rls_cleanup( &scsi_cleanup[scsi_id], &domain_status );   /* get rid of cleanup handler */
    }
    return(11);
  }

  /* The wait interval is in milliseconds, so let's give it a long (maximum) wait period */
  wait_index = scsi_$wait( scsi_handle[scsi_id], 60*60*1000, true, scsi_op, 1, &scsi_status_list, &return_count, &wait_status);

  /* enable paging of buffer area */
  if ((length != 0) && (ex_prewired[scsi_id] == 0))
  {
    scsi_$unwire( scsi_handle[scsi_id], buffer_ptr, length, false, &domain_status);
    buffer_wired[scsi_id] = false;                                  /* forget we're wired */
    pfm_$rls_cleanup( &scsi_cleanup[scsi_id], &domain_status );     /* get rid of cleanup handler */
    if (domain_status.all != status_$ok)
    {
      /*
      printf("(execute_command) Unable to unwire scsi data buffer\n");
      */
      pfm_$error_trap(domain_status);
      return(13);
    }
  }

  /* now see if wait was OK */
  domain_status = wait_status;
  if (domain_status.all != status_$ok)
  {
    /*
    printf("(execute_command) SCSI wait unsuccessful\n");
    pfm_$error_trap(domain_status);
    */
    return(13);
  }

  scsi_stat = scsi_status_list.op_status;

  /* check status of scsi command execution */
  switch (wait_index)
  {
	case scsi_device_advance:		break;
	case scsi_timeout:       		domain_status.all = scsi_$operation_timeout;
    default:                        /* printf("(execute_command) wait index not zero: %d\n",wait_index); */
    return(wait_index);
  }	/* end wait_for_done */

return(0);
}
/* ---------------------------------------------------------------------
   Erase the scsi device media
   ---------------------------------------------------------------------
*/

void ex_erase( scsi_id, cdb_byte_01, status )/* erase command */
unsigned char     scsi_id;                  /* scsi id of target device */
unsigned char     cdb_byte_01;              /* 1st byte of CDB */
driver_$status_t  *status;                   /* scsi status */
{
  int i;

  timeout[scsi_id] = 2*70*60*1000;             /* change timeout to 140 minutes */
  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_erase;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;     /* let caller set lun & other bits */
  scsi_cdb[scsi_id].all[02] = 0;               /* reserved & vendor unique bits must be zero */
  scsi_cdb[scsi_id].all[03] = 0;               /* reserved & vendor unique bits must be zero */
  if (cdb_byte_01 == 0)                        /* is this a "short" erase? */
    scsi_cdb[scsi_id].all[04] = 1;             /* yes - assume this is not an Exabyte, and set length to 1 */
  else
    scsi_cdb[scsi_id].all[04] = 0;             /* reserved & vendor unique bits must be zero */
  scsi_cdb[scsi_id].all[05] = 0;               /* reserved & vendor unique bits must be zero */

  i = execute_command( scsi_id, 0, 0, scsi_write);
  timeout[scsi_id] = 90*1000;                  /* change timeout back to 90 seconds */
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Execute a generic SCSI command
   ---------------------------------------------------------------------
*/
void ex_generic( scsi_id, cdb, cdb_length, buffer_ptr, length, write, time, status )
unsigned char scsi_id, cdb_length;
unsigned char *cdb, *buffer_ptr;
linteger length, time;
driver_$direction_t write;
driver_$status_t  *status;
{
  int i;
  linteger old_time;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  old_time = timeout[scsi_id];                 /* remember old timeout value */
  timeout[scsi_id] = time;
  for (i=0; i < cdb_length; i++,cdb++)
     scsi_cdb[scsi_id].all[i] = *cdb;  /* user-defined CDB */
  i = execute_command( scsi_id, buffer_ptr, length, write);
  timeout[scsi_id] = old_time;                 /* restore original timeout value */
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                            /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Inquiry of the scsi device
   ---------------------------------------------------------------------
*/
void ex_inquiry( scsi_id, cdb_byte_01, buffer_ptr, length, status )
unsigned char scsi_id, cdb_byte_01;
driver_$status_t  *status;
unsigned char *buffer_ptr;
linteger length;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_inquiry;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;	/* let caller set lun & other bits */
  if (length > 255)
    scsi_cdb[scsi_id].len = 255;			/* try for at most 255 bytes of data */
  else
    scsi_cdb[scsi_id].len = length;
  scsi_cdb[scsi_id].all[02] = 0;			/* reserved & vendor unique bits must be zero */
  scsi_cdb[scsi_id].all[03] = 0;			/* reserved & vendor unique bits must be zero */
  scsi_cdb[scsi_id].all[05] = 0;			/* reserved & vendor unique bits must be zero */

  i = execute_command( scsi_id, buffer_ptr, length, scsi_read);
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   (Un)load the scsi device
   ---------------------------------------------------------------------
*/
void ex_load( scsi_id, cdb_byte_04, status)
driver_$status_t  *status;
unsigned char scsi_id, cdb_byte_04;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_unload;
  scsi_cdb[scsi_id].all[01] = 0;	/* reserved bits must be zero */
  scsi_cdb[scsi_id].all[02] = 0;	/* reserved bits must be zero */
  scsi_cdb[scsi_id].all[03] = 0;	/* reserved bits must be zero */
  scsi_cdb[scsi_id].all[04] = cdb_byte_04;  /* user-defined value for this byte */
  scsi_cdb[scsi_id].all[05] = 0;	/* reserved bits must be zero */

  i = execute_command( scsi_id, 0, 0, scsi_read);
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Locate forward or backward
   ---------------------------------------------------------------------
*/
void ex_locate( scsi_id, cdb_byte_01, number, status)
unsigned char scsi_id, cdb_byte_01;
linteger number;
driver_$status_t  *status;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_locate;
  scsi_cdb[scsi_id].all[1] = cdb_byte_01;/* user-defined */
  scsi_cdb[scsi_id].all[2] = 0;           /* reserved */
  scsi_cdb[scsi_id].all[3] = ((number & 0xff000000) >> 24);  /* MSB block address */
  scsi_cdb[scsi_id].all[4] = ((number & 0x00ff0000) >> 16);
  scsi_cdb[scsi_id].all[5] = ((number & 0x0000ff00) >>  8);
  scsi_cdb[scsi_id].all[6] =  (number & 0x000000ff);         /* LSB block address */
  scsi_cdb[scsi_id].all[7] = 0;           /* reserved */
  scsi_cdb[scsi_id].all[8] = 0;           /* partitions not defined on Exabytes */
  scsi_cdb[scsi_id].all[9] = 0;           /* directories not supported, nor any vendor unique bits defined */

  i = execute_command( scsi_id, 0, 0, scsi_read);
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Set mode (mode select) the scsi device
   ---------------------------------------------------------------------
*/
void ex_mode_select( scsi_id, cdb_byte_01, buffer_ptr, length, status )
unsigned char scsi_id, cdb_byte_01;
driver_$status_t  *status;
unsigned char *buffer_ptr;
linteger length;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_mode_select;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;     /* user-defined bits */
  if (length > 255)
    scsi_cdb[scsi_id].len = 255;               /* try for at most 255 bytes of select data */
  else
    scsi_cdb[scsi_id].len = length;            /* try for '*length' bytes of select data */
  scsi_cdb[scsi_id].vend_unique = 0;           /* nothing unique needed here */
  scsi_cdb[scsi_id].rsv0_1 = 0;                /* reserved bits must be zero */
  scsi_cdb[scsi_id].flag = 0;
  scsi_cdb[scsi_id].link = 0;                  /* not linked */

  i = execute_command( scsi_id, buffer_ptr, length, scsi_write);
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Sense the current mode (mode sense) of the scsi device
   ---------------------------------------------------------------------
*/
void ex_mode_sense( scsi_id, cdb_byte_01, cdb_byte_02, buffer_ptr, length, status )
unsigned char scsi_id, cdb_byte_01, cdb_byte_02;
driver_$status_t  *status;
unsigned char *buffer_ptr;
linteger length;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_mode_sense;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;     /* user-defined bits */
  scsi_cdb[scsi_id].all[02] = cdb_byte_02;     /* user-defined bits */
  scsi_cdb[scsi_id].all[03] = 0;               /* reserved bits */
  if (length > 255)
    scsi_cdb[scsi_id].all[04] = 255;           /* try for at most 255 bytes of sense data */
  else
    scsi_cdb[scsi_id].all[04] = length;        /* try for '*length' bytes of sense data */
  scsi_cdb[scsi_id].all[05] = 0;               /* no vendor unique bits, just reserved bits */

  i = execute_command( scsi_id, buffer_ptr, length, scsi_read);
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Read a buffer from the scsi device
   ---------------------------------------------------------------------
*/
void ex_read( scsi_id, cdb_byte_01, buffer_ptr, length, length_read, status )
unsigned char scsi_id, cdb_byte_01;
driver_$status_t  *status;
unsigned char *buffer_ptr;
linteger length, *length_read;
{
  union {
         linteger lint;                     /* standard integer form of length */
         struct {
                 unsigned char msb,         /* most significant byte */
                               msm,         /* most significant middle byte */
                               lsm,         /* least significant middle byte */
                               lsb;         /* least significant byte */
                } byte;
        } info_length;
  int i;
  driver_$status_t  scsi_status;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_read;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;	/* let caller set lun & SILI & Fixed bits */
  scsi_cdb[scsi_id].len = length;           /* try for '*length' bytes of data */
  scsi_cdb[scsi_id].all[05] = 0;            /* reserved & vendor unique bits must be zero */

  i = execute_command( scsi_id, buffer_ptr, length, scsi_read);
  *status = scsi_stat;                      /* return scsi operation status */
  *length_read = length;                    /* assume we got what caller asked for */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    *length_read = 0;                       /* say we read nothing */
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
    {
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
      return;
    }
  }
  if (!scsi_stat) return;                   /* if status is good, return */

  /* bad read status of some sort -- ask the drive what went wrong             */
  /* also, save the request sense results in case the caller asks for it later */

  ex_request_sense( scsi_id, 0, ex_sense_data.all, sizeof(ex_sense_data), &scsi_status );
  if (scsi_status != 0) return;
  driver_has_sense[scsi_id] = true;        /* remember that we have the sense data already */
  if (ex_sense_data.sd.valid)
  {
    info_length.byte.msb = ex_sense_data.sd.info_length1;  /* compiler likes to align 4-byte quantities */
    info_length.byte.msm = ex_sense_data.sd.info_length2;  /* --fewer instructions than move/shift, etc.-- */
    info_length.byte.lsm = ex_sense_data.sd.info_length3;
    info_length.byte.lsb = ex_sense_data.sd.info_length4;
    if (!(info_length.lint & 0x80000000)) *length_read = length - info_length.lint;    /* tape block shorter than requested? */
    /* else info_length.lint is the # of bytes not read from this block */
  }
  for (i=0;i<29;i++) stored_sense[scsi_id][i] = ex_sense_data.all[i];  /* store sense data */
}
/* ---------------------------------------------------------------------
   Read block limits of the scsi device
   ---------------------------------------------------------------------
*/
void ex_read_block_limits( scsi_id, buffer_ptr, status )
unsigned char scsi_id;
driver_$status_t  *status;
unsigned char *buffer_ptr;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_read_block_limits;
  scsi_cdb[scsi_id].all[01] = 0;           /* 0 for lun */
  scsi_cdb[scsi_id].all[02] = 0;			/* reserved bits must be zero */
  scsi_cdb[scsi_id].all[03] = 0;			/* reserved bits must be zero */
  scsi_cdb[scsi_id].all[04] = 0;			/* reserved bits must be zero */
  scsi_cdb[scsi_id].all[05] = 0;			/* reserved & vendor unique bits must be zero */

  i = execute_command( scsi_id, buffer_ptr, 6, scsi_read); /* requires user buffer of 6 bytes */
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Read position of the media in the scsi device
   ---------------------------------------------------------------------
*/
void ex_read_position( scsi_id, cdb_byte_01, buffer_ptr, status )
unsigned char scsi_id, cdb_byte_01;
driver_$status_t  *status;
unsigned char *buffer_ptr;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_read_position;
  scsi_cdb[scsi_id].all[1] = cdb_byte_01;    /* user defined */
  scsi_cdb[scsi_id].all[2] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[3] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[4] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[5] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[6] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[7] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[8] = 0;               /* reserved */
  scsi_cdb[scsi_id].all[9] = 0;               /* reserved */

  i = execute_command( scsi_id, buffer_ptr, 20, scsi_read); /* requires user buffer of 20 bytes, minimum */
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Read status (request sense) from the scsi device
   ---------------------------------------------------------------------
*/
void ex_request_sense( scsi_id, cdb_byte_05, buffer_ptr, length, status )
unsigned char scsi_id, cdb_byte_05;
driver_$status_t  *status;
unsigned char *buffer_ptr;
linteger length;
{
  int i, j;
  driver_$status_t  scsi_status;

  if (cdb_byte_05 & 0x80)                /* if we don't want to clear error counters, then.... */
  {
    if (driver_has_sense[scsi_id])       /* do we recycle old sense data? */
    {
      j = 29;
      if (length < j) j = length;
      for (i=0;i<j;i++) *buffer_ptr++ = stored_sense[scsi_id][i];  /* recall stored sense data */
      driver_has_sense[scsi_id] = false; /* recycle sense data only one time */
      *status = driver_good_status;
      return;
    }
  }
  scsi_cdb[scsi_id].cmd = cmd_request_status;
  scsi_cdb[scsi_id].lun = 0;			  /* always lun 0 */
  scsi_cdb[scsi_id].rsv0_0 = 0;	          /* reserved bits must be zero */
  if (length > 255)
    scsi_cdb[scsi_id].len = 255;          /* too much space requested */
  else
    scsi_cdb[scsi_id].len = length;       /* try for length bytes of sense data */
  scsi_cdb[scsi_id].all[05] = cdb_byte_05;	/* nothing unique needed here */

  i = execute_command( scsi_id, buffer_ptr, length, scsi_read);
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Rewind the scsi device
   ---------------------------------------------------------------------
*/
void ex_rewind( scsi_id, status)
driver_$status_t  *status;
unsigned char scsi_id;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_rewind;
  scsi_cdb[scsi_id].lun = 0;            /* always lun 0 */
  scsi_cdb[scsi_id].rsv0_0 = 0;         /* reserved bits must be 0 */
  scsi_cdb[scsi_id].len = 0;
  scsi_cdb[scsi_id].vend_unique = 0;	/* nothing unique needed here */
  scsi_cdb[scsi_id].rsv0_1 = 0;	        /* reserved bits must be zero */
  scsi_cdb[scsi_id].flag = 0;
  scsi_cdb[scsi_id].link = 0;           /* not linked */

  i = execute_command( scsi_id, 0, 0, scsi_read);
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Space forward or backward
   ---------------------------------------------------------------------
*/
void ex_space( scsi_id, cdb_byte_01, cdb_byte_05, number, status)
unsigned char scsi_id, cdb_byte_01, cdb_byte_05;
linteger number;
driver_$status_t  *status;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_space;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;  /* user-defined */
                                            /* 0 - logical blocks
                                               1 - filemarks
                                               2 - *not supported*
                                               3 - end record */
  scsi_cdb[scsi_id].len = number;           /* space 'number' blocks, etc. */
  scsi_cdb[scsi_id].all[05] = cdb_byte_05;  /* user-defined */

  i = execute_command( scsi_id, 0, 0, scsi_read);
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Test Unit Ready
   ---------------------------------------------------------------------
*/
void ex_test_unit_ready( scsi_id, status )
unsigned char scsi_id;
driver_$status_t  *status;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_test_unit_ready;
  scsi_cdb[scsi_id].all[01] = 0;               /* reserved bits */
  scsi_cdb[scsi_id].all[02] = 0;               /* reserved bits */
  scsi_cdb[scsi_id].all[03] = 0;               /* reserved bits */
  scsi_cdb[scsi_id].all[04] = 0;               /* reserved bits */
  scsi_cdb[scsi_id].all[05] = 0;               /* reserved bits */

  i = execute_command( scsi_id, 0, 0, scsi_read);
  *status = scsi_stat;                          /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok))  /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Verify a tape in the scsi device
   ---------------------------------------------------------------------
*/
void ex_verify( scsi_id, cdb_byte_01, number, status )
unsigned char scsi_id, cdb_byte_01;
driver_$status_t  *status;
linteger number;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_verify;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;     /* user-defined bits */
  scsi_cdb[scsi_id].len = number;              /* try for '*number' bytes/blocks to be verified */
  scsi_cdb[scsi_id].all[05] = 0;               /* more reserved bits */

  i = execute_command( scsi_id, 0, 0, scsi_read);
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
/* ---------------------------------------------------------------------
   Write a buffer to the scsi device
   ---------------------------------------------------------------------
*/
void ex_write( scsi_id, cdb_byte_01, buffer_ptr, length, status )
unsigned char scsi_id, cdb_byte_01;
driver_$status_t  *status;
unsigned char *buffer_ptr;
linteger length;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_write;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;	   /* user defined */
  scsi_cdb[scsi_id].len = length;              /* try to write length bytes of data */
  scsi_cdb[scsi_id].all[05] = 0;               /* nothing unique here */

  i = execute_command( scsi_id, buffer_ptr, length, scsi_write);
  *status = scsi_stat;                         /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
  return;
}
/* ---------------------------------------------------------------------
   Write a filemark to the scsi device
   ---------------------------------------------------------------------
*/
void ex_write_filemarks( scsi_id, cdb_byte_01, cdb_byte_05, number, status )
unsigned char scsi_id, cdb_byte_01, cdb_byte_05;
driver_$status_t  *status;
linteger number;
{
  int i;

  driver_has_sense[scsi_id] = false;           /* new operation - forget old sense data */
  scsi_cdb[scsi_id].cmd = cmd_write_filemarks;
  scsi_cdb[scsi_id].all[01] = cdb_byte_01;    /* user-defined bits */
  scsi_cdb[scsi_id].len = number;             /* try for '*number' filemarks to be written */
  scsi_cdb[scsi_id].all[05] = cdb_byte_05;    /* more user-defined bits */

  i = execute_command( scsi_id, 0, 0, scsi_write);
  *status = scsi_stat;                      /* return scsi operation status */
  if ((i) | (domain_status.all != status_$ok)) /* if we have a bad, but non-SCSI status, then... */
  {
    if (!scsi_stat)                         /* if there is no SCSI status besides, then...*/
      *status = driver_undefined_status;    /* can't translate why, so say it's undefined */
  }
}
