/* ---------------------------------------------------------------------------------------------
   MTS - mt utility for SCSI devices under Domain OS
         Written by Michael Lampi, MDL Corporation, 15301 NE 90th Street, Redmond, WA 98052
                    lampi@mdlcorp.com           (800) 800-3766           fax (206) 861-6767

    Copyright (c) 1993 by MDL Corporation

    01/20/93    mdl     written
    02/01/93    mdl     added floppy & change definition stuff
    02/02/93    mdl     added cartridge retension support
    02/08/93    mdl     added some disk read/write support
    02/09/93    mdl     added some disk drive exercising
    02/22/93    mdl     expanded device density display & added  C1716/C mode select stuff
    05/17/93    mdl     cleaned up request sense info report
    05/27/93    mdl     put in a request sense to clean up any initial error status (unit attn, etc.),
                        add tape capacity remaining report, add retension
    06/09/93    mdl     added support for scsi driver library options
    06/10/93    mdl     fixed string match bug
    06/11/93    mdl     added reinit, type82 and typex options
    06/15/93    mdl     added "ce_mode" to enable hex dumps of returned status info
    06/22/93    mdl     made "8200 response mode set", et al to ce_mode for status
    07/06/93    mdl     fleshed out mode sense and added user mode select stuff
    09/01/93    mdl     added HP 35480A DAT compression/decompression & changeable mode select display
    09/08/93    mdl     added compression info to status message (IWorks version 1.10)
   --------------------------------------------------------------------------------------------- */
#include <stdio.h>
#include <sys/types.h>
#include </usr/apollo/include/base.h>
#include </usr/apollo/include/name.h>
#include </usr/apollo/include/pfm.h>
#include </usr/apollo/include/scsi.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include </mts/driver.h>
#include </mts/sdriver.h>

#define scsi_$device_in_use              0x00380003 /* requested device in use */
#define maxbufsize 240000

#define disk_type 1
#define tape_type 2
#define optical_type 3

unsigned char mode_buffer[1024];
union {
  unsigned char bigbuffer[maxbufsize];
  linteger      biglbuffer[maxbufsize/4];
  } bf;

unsigned char scsi_id;
union {
  struct {                                     /* extended sense (byte:bit position) */  
       unsigned short     valid:1;             /* (0:7) validity of information bytes */
       unsigned short     pad2:7;              /* (0:6-0) err class & code are fixed */
       unsigned short     segment:8;           /* (1) segment number */
       unsigned short     filemark:1;          /* (2:7) at a filemark */
       unsigned short     eom:1;               /* (2:6) End Of Medium */
       unsigned short     ili:1;               /* (2:5) Incorrect Length Indicator */
       unsigned short     pad3:1;              /* (2:4) reserved for future use */
       unsigned short     sense_key:4;         /* (2:3-0) */
       unsigned short     info_length1:8;      /* (3-6) Information bytes */
       unsigned short     info_length2:8;      /* (3-6) Information bytes */
       unsigned short     info_length3:8;      /* (3-6) Information bytes */
       unsigned short     info_length4:8;      /* (3-6) Information bytes */
       unsigned short     add_len:8;           /* (7)   length of additional sense data */
       unsigned short     rsrvd_8_9:16;        /* (8-9) reserved */
       unsigned short     rsrvd_10:8;          /* (10)  reserved */
       unsigned short     underover:8;         /* (11)  underrun/overrun counter */
       unsigned short     add_sense:8;         /* (12)  additional sense code */
       unsigned short     add_qual:8;          /* (13)  additional sense code qualifier */
       unsigned short     rsrvd_14_15:16;      /* (14-15) reserved */
       unsigned short     rd_wr_cnt:24;        /* (16-18) read/write error counter */
       unsigned short     pf:1;                /* (19:7)  power-fail/reset */
       unsigned short     bpe:1;               /* (19:6)  SCSI bus parity error */
       unsigned short     fpe:1;               /* (19:5)  data buffer parity error */
       unsigned short     me:1;                /* (19:4)  media error */
       unsigned short     eco:1;               /* (19:3)  error counter overflow */
       unsigned short     tme:1;               /* (19:2)  tape motion error */
       unsigned short     tnp:1;               /* (19:1)  tape not present */
       unsigned short     lbot:1;              /* (19:0)  tape at LBOT */
       unsigned short     rsvd_10_7:1;         /* (20:7)  reserved */
       unsigned short     tmd:1;               /* (20:6)  tape mark detect error */
       unsigned short     wp:1;                /* (20:5)  write protect */
       unsigned short     fmke:1;              /* (20:4)  filemark write error */
       unsigned short     ure:1;               /* (20:3)  data underrun error */
       unsigned short     wei:1;               /* (20:2)  max rewrites exceeded */
       unsigned short     sse:1;               /* (20:1)  servo system error */
       unsigned short     fe:1;                /* (20:0)  data formatter error */
       unsigned short     rsvd_21_7:2;         /* (21:7-6)  reserved */
       unsigned short     rrr:1;               /* (21:5)  reverse retries required */
       unsigned short     clnd:1;              /* (21:4)  drive has just been cleaned */
       unsigned short     cln:1;               /* (21:3)  time to clean drive */
       unsigned short     peot:1;              /* (21:2)  tape at PEOT */
       unsigned short     wseb:1;              /* (21:1)  write splice error (blank tape) */
       unsigned short     wseo:1;              /* (21:0)  write splice error (Overshoot) */
       unsigned short     rsrvd_22:8;          /* (22)  reserved */
       unsigned short     tape_left:24;        /* (23-25) tape remaining */
       unsigned short     track_retries:8;     /* (26)  tracking retry counter */
       unsigned short     rw_retries:8;        /* (27)  read/write retry counter */
       unsigned short     fault_code:8;        /* (28)  fault symptom code */
  } sd;
  unsigned char all[80];
} sense_data;

#define maxcmd  32
static char command[maxcmd][8]={"eof", "weof",    /* 0, 1 */
                              "fsf",            /* 2 */
                              "fsr",            /* 3 */
                              "bsf",            /* 4 */
                              "bsr",            /* 5 */
                              "erase",          /* 6 */
                              "reten",          /* 7 */
                              "rewind", "rew",  /* 8, 9 */
                              "offline", "rewoffl", /* 10, 11 */
                              "status",         /* 12 */
                              "high",           /* 13 */
                              "low",            /* 14 */
                              "where",          /* 15 */
                              "moveto",         /* 16 */
                              "fse",            /* 17 - forward space to end of data */
                              "wseof",          /* 18 - write short filemark */
                              "tur",            /* 19 - test unit ready */
                              "version",        /* 20 - report version number */
                              "debug",          /* 21 - debug/interactive mode */
                              "finit",          /* 22 - initialize floppy mode */
                              "reten",          /* 23 - retension cartridge tape */
                              "qic11",          /* 24 - QIC-11  density */
                              "qic24",          /* 25 - QIC-24  density */
                              "qic120",         /* 26 - QIC-120 density */
                              "qic150",         /* 27 - QIC-150 density */
                              "qic525",         /* 28 - QIC-525 density */
                              "8200",           /* 29 - 8200 tape mode */
                              "8500",           /* 30 - 8500 tape mode */
                              "serase"};        /* 31 - perform a short erase on the tape drive */

int ce_mode = 0;    /* set to 1 if we are in "ce mode" */
driver_$status_t scsi_status = driver_good_status;
#define ce_mode_value 1234

/* ---------------------------------------------------------------------
    Formatted output routines
   --------------------------------------------------------------------- */
dump_data( buffer, size )
unsigned char *buffer;
int size;
{
  int i;

  if (ce_mode == 0) return;

  i=0;
  printf("%4x  ",i);
  for (i=0;i<size;i++)
  {
    printf("%2x ",buffer[i]);
    if (((i+1)&7) == 0)
    {
      if (i<size-1)
        printf("\n%4x  ",i+1);
      else
        printf("\n");
    }
  }
  printf("\n");
}
dump_data_2( buffer1, buffer2, size )
unsigned char *buffer1, *buffer2;
int size;
{
  int i;

  if (ce_mode == 0) return;

  for (i=0;i<size;i++)
  {
    if ((i & 7) == 0)
      printf("%4x  ",i);
    printf("%2x ",buffer1[i]);
    if ((i&7) == 7)
    {
      printf("\n      %2x %2x %2x %2x %2x %2x %2x %2x - changeable bits\n",
             buffer2[i-7], buffer2[i-6], buffer2[i-5], buffer2[i-4],
             buffer2[i-3], buffer2[i-2], buffer2[i-1], buffer2[i]);
    }
  }
  if ((size & 7) > 0)
  {
    printf("\n      ");
    for (i=(size & 0xfffff8);i<size;i++)
      printf("%2x ", buffer2[i]);
    printf(" - changeable bits\n");
  }
}
/* ---------------------------------------------------------------------
   Display request sense data from the scsi device
   ---------------------------------------------------------------------
*/
display_sense_data( scsi_id )
unsigned char scsi_id;
{
  linteger i, j, number;
  int k;
  float x, y, z;
  union {
         linteger lint;                     /* standard integer form of length */
         struct {
                 unsigned char msb,         /* most significant byte */
                               msm,         /* most significant middle byte */
                               lsm,         /* least significant middle byte */
                               lsb;         /* least significant byte */
                } byte;
        } info_length;
  unsigned char request_sense_data[20];

  if (scsi_status == driver_undefined_status)
  {
    printf("DomainOS-related error occurred\n");
    return;
  }
  ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status );
  if (scsi_status != 0)
  {
    printf("Unable to perform request_sense command: %x\n",scsi_status);
    return;
  }
  printf("Request Sense Fields:\n");

  printf("  Sense Key: %2x - ",sense_data.sd.sense_key);
  if (sense_data.sd.sense_key != 0) {
    switch (sense_data.sd.sense_key)  {
      case 1:  printf("recovered error\n"); break;
      case 2:  printf("not ready\n"); break;
      case 3:  printf("medium error\n"); break;
      case 4:  printf("hardware error\n"); break;
      case 5:  printf("illegal request\n");
               if (sense_data.sd.add_len >= 11) {   /* we can tell user which bit(s) are bad */
                 if (sense_data.all[15] & 0x80) {   /* sense key specific valid? */
                   if (sense_data.all[15] & 0x40)   /* if in the command block.... */
                     printf("  command descriptor block error");
                   else
                     printf("  data parameter error");
                   if (sense_data.all[15] & 0x10)   /* bit pointer field valid? */
                     printf(" at bit %d", sense_data.all[15] & 0x0f);
                   k = ( ((int) sense_data.all[16] ) << 8) + (int) sense_data.all[17];
                   printf(" at byte %d (decimal) %x (hex)\n", k, k);
                 }
               }
               break;
      case 6:  printf("unit attention\n"); break;
      case 7:  printf("write protected\n"); break;
      case 8:  printf("blank check\n"); break;
      case 9:  printf("EXABYTE-special positioning error\n"); break;
      case 10: printf("copy aborted\n"); break;
      case 11: printf("aborted command\n"); break;
      case 12: printf("equal\n"); break;
      case 13: printf("volume overflow\n"); break;
      case 14: printf("miscompare\n"); break;
      default: printf("unsupported error code\n");
    }
    if (sense_data.sd.add_len > 5)
      printf("  Additional sense code: %2x, qualifier: %2x\n",sense_data.sd.add_sense,sense_data.sd.add_qual);
  }
  else printf("no sense\n");
             
  if (sense_data.sd.filemark) printf("  filemark detected\n");
  if (sense_data.sd.eom)      printf("  at EOM\n");
  if (sense_data.sd.ili)      printf("  illegal length\n");
  if (sense_data.sd.valid)
  {
/*
    info_length = (((((sense_data.sd.info_length1  << 8) | sense_data.sd.info_length2) << 8) |
                       sense_data.sd.info_length3) << 8) | sense_data.sd.info_length4;
*/
    info_length.byte.msb = sense_data.sd.info_length1;  /* compiler likes to align 4-byte quantities */
    info_length.byte.msm = sense_data.sd.info_length2;  /* --fewer instructions than move/shift, etc.-- */
    info_length.byte.lsm = sense_data.sd.info_length3;
    info_length.byte.lsb = sense_data.sd.info_length4;
    printf("  %d bytes/blocks unprocessed\n",info_length.lint);
  }

  if (sense_data.sd.add_len > 3)
    if (sense_data.sd.underover != 0) printf("  overrun/underrun count: %d\n", sense_data.sd.underover);
  if (sense_data.sd.add_len > 12) {
    if (sense_data.sd.pf)    printf("  power-on/reset\n");
    if (sense_data.sd.bpe)   printf("  SCSI bus parity error\n");
    if (sense_data.sd.fpe)   printf("  data buffer parity error\n");
    if (sense_data.sd.me)    printf("  media error\n");
    if (sense_data.sd.eco)   printf("  error counter overflow\n");
    if (sense_data.sd.tme)   printf("  tape motion error\n");
    if (sense_data.sd.tnp)   printf("  tape not present\n");
    if (sense_data.sd.lbot)  printf("  tape at LBOT\n");
  }
  if (sense_data.sd.add_len > 13) {
    if (sense_data.sd.tmd)   printf("  tape mark detect error\n");
    if (sense_data.sd.wp)    printf("  write protect\n");
    if (sense_data.sd.fmke)  printf("  filemark write error\n");
    if (sense_data.sd.ure)   printf("  data underrun error\n");
    if (sense_data.sd.wei)   printf("  max rewrites exceeded\n");
    if (sense_data.sd.sse)   printf("  servo system error\n");
    if (sense_data.sd.fe)    printf("  data formatter error\n");
  }
  if (sense_data.sd.add_len > 14) {
    if (sense_data.sd.rrr)   printf("  reverse retries required\n");
    if (sense_data.sd.clnd)  printf("  drive has just been cleaned\n");
    if (sense_data.sd.cln)   printf("  time to clean drive\n");
    if (sense_data.sd.peot)  printf("  tape at PEOT\n");
    if (sense_data.sd.wseb)  printf("  write splice error (blank tape)\n");
    if (sense_data.sd.wseo)  printf("  write splice error (Overshoot)\n");
  }

  if (sense_data.sd.add_len > 19) {
    i = sense_data.sd.rw_retries;
    if (i != 0) printf("  read/write retry count: %d\n", i);
  }

  if (sense_data.sd.add_len > 10) {
    i = sense_data.sd.rd_wr_cnt;
    if (i != 0) printf("  read/write error count: %d\n",i);
  }

  if (sense_data.sd.add_len > 18) {
    i = sense_data.sd.track_retries;
    if (i != 0) printf("  tracking retry count:   %d\n", i);
  }

  if (sense_data.sd.add_len > 17) {
    if (!sense_data.sd.tnp) {
      /* ask for 17 bytes of non-page mode sense data to be returned */
      ex_mode_sense( scsi_id, 0, 0, request_sense_data, 17, &scsi_status);
      switch( request_sense_data[1] ) {
         case 0x81: i =  588; break;   /* P6-15 */
         case 0x82: i = 1175; break;   /* P6-30 */
         case 0x83: i = 2348; break;   /* P6-60 */
         case 0x84: i = 3522; break;   /* P6-90 */
         case 0x85: i = 4695; break;   /* P6-120 */
         case 0xc1: i =  600; break;   /* P5-15 (rough guesstimate) */
         case 0xc2: i = 1200; break;   /* P5-30 (rough guesstimate) */
         case 0xc3: i = 3600; break;   /* P5-60 (rough guesstimate) */
         case 0xc4: i = 4944; break;   /* P5-90 */
         default:   i = 0;
      }
      number = request_sense_data[5];
      number = (number << 8) + request_sense_data[6];
      number = (number << 8) + request_sense_data[7];
      printf("  blocks on tape:   %d\n",number);

      j = sense_data.sd.tape_left;
      printf("  blocks remaining: %d\n", j);

      x = j;y = i;z = number; if (z < 1.0) z = 1.0;
      x = (x * y) / z;          /*  j = (j * i) / number; */
      switch (request_sense_data[4]) {
         case 0x13: printf("DAT format\n"); break;
         case 0x14: printf("  megabytes remaining: %4.0f\n", x/2.0); break; /* Exabyte 8200 2.3GB */
         case 0x15: printf("  megabytes remaining: %4.0f\n", x); break;   /* Exabyte 8500 5GB */
         default:   printf("  megabytes remaining: %4.0f\n", x); /* default (8500) density */
      }
    }
  }

  dump_data( sense_data.all, (7 + sense_data.sd.add_len) );
}

/* ---------------------------------------------------------------------------
   Set the mode select buffer to known values
   ---------------------------------------------------------------------------
*/
set_mode()
{
  mode_buffer[0] = 0;     /* the first two bytes are reserved */
  mode_buffer[1] = 0;
  mode_buffer[2] = 0x10;  /* 0x00 - unbuffered mode, 0x10 - buffered mode */
  mode_buffer[3] = 8;     /* block descriptor length */

  mode_buffer[4] = 0;     /* set tape drive to 0x14 - 8200 mode */
  mode_buffer[5] = 0;     /*                   0x15 - 8500 mode */
  mode_buffer[6] = 0;
  mode_buffer[7] = 0;
  mode_buffer[8] = 0;     /* reserved */
  mode_buffer[9] = 0;     /* block length MSB - 0 = variable */
  mode_buffer[10] = 0;
  mode_buffer[11] = 0;    /* block length LSB and the end of the block descriptor */

  mode_buffer[12] = 0x10; /* beginning of device configuration page */
  mode_buffer[13] = 0x0d; /* length of this page */
  mode_buffer[14] = 0;	  /* don't change active format */
  mode_buffer[15] = 0;	  /* don't support active partitions */
  mode_buffer[16] = 0x80; /* write buffer full ratio - 512KB */
  mode_buffer[17] = 0x80; /* read buffer full ratio  - 512KB */
  mode_buffer[18] = 0;	  /* write delay MSB */
  mode_buffer[19] = 0x80; /*             LSB - 0x80 = 12.8 sec */
  mode_buffer[20] = 1;	  /* early warning of EOT on reads */
  mode_buffer[21] = 0;	  /* gap size */
  mode_buffer[22] = 0x18; /* EEG & SEW set to 1 */
  mode_buffer[23] = 0;	  /* Early warning buffer size MSB */
  mode_buffer[24] = 0;	  /*  */
  mode_buffer[25] = 0;	  /* LSB */
  mode_buffer[26] = 0;	  /* select data compression algorithm */
}
/* --------------------------------------------------------------------------- */
set_mode_c1716(mode_buffer)
unsigned char mode_buffer[30];
{
  mode_buffer[0] = 0;     /* the first two bytes are reserved */
  mode_buffer[1] = 0x03;  /* optical erasable */
  mode_buffer[2] = 0;
  mode_buffer[3] = 8;     /* block descriptor length */

  mode_buffer[4] = 0x03;  /* set density to optical erasable */
  mode_buffer[5] = 0;
  mode_buffer[6] = 0;
  mode_buffer[7] = 0;
  mode_buffer[8] = 0;     /* reserved */
  mode_buffer[9] = 0;     /* block length MSB */
  mode_buffer[10] = 0;
  mode_buffer[11] = 0;    /* block length LSB and the end of the block descriptor */

  mode_buffer[12] = 0x21; /* beginning of vendor unique device configuration page */
  mode_buffer[13] = 0x10; /* length of this page */
  mode_buffer[14] = 0x03; /* DTIS | DAIR */
  mode_buffer[15] = 0;
  mode_buffer[16] = (1000 & 0xff0000) >> 16; /* Maximum buffer latency msb */
  mode_buffer[17] = (1000 & 0x00ff00) >>  8;
  mode_buffer[18] = (1000 & 0x0000ff);       /* lsb */
  mode_buffer[19] = 2;    /* drive retry count */
  mode_buffer[20] = 150;  /* autochanger eject distance */
  mode_buffer[21] = 5;    /* Phase retry count */
  mode_buffer[22] = 0x18; /* EEG & SEW set to 1 */
  mode_buffer[23] = 0;	  /* reserved */
}
/* --------------------------------------------------------------------------- */
direct_write( scsi_id, number, addr, limit, buflen )
  /* number is the total number of blocks to write
     addr   is the address on the disk at which to start writing
     limit  is the maximum number of blocks to write per scsi command
     buflen is the size of the I/O data buffer in bytes
  */
  unsigned char scsi_id;
  linteger limit, addr, number, buflen;

{
  int i, j;
  linteger iteraddr;
  status_$t d_status;
  unsigned char cdb[12];
  driver_$status_t scsi_status;
  pfm_$cleanup_rec scsi_cleanup;    /* cleanup record */

  /* Play it safe and establish a cleanup handler in case the program dies during
     I/O. Don't want to leave bits & pieces of memory wired if we can help it. */

  d_status = pfm_$cleanup( &scsi_cleanup );
  if (d_status.all != pfm_$cleanup_set)        /* if !=, then we're in a fault situation */
  {
    ex_preunwire(scsi_id, &d_status);              /* release the buffer */
    pfm_$signal( d_status );                       /* go to the next cleanup level */
  }

  ex_prewire( scsi_id, bf.bigbuffer, buflen, &d_status);
  if (d_status.all != status_$ok)
  {
     pfm_$rls_cleanup( &scsi_cleanup, &d_status );   /* get rid of cleanup handler */
     printf("Unable to prewire I/O data buffer: %x\n", d_status.all);
     return;
  }
  iteraddr = 0;
  while (number > limit)
  {
    cdb[0] = 0x2a;        /* use 10 byte CDB */
    cdb[1] = 0x00;
    cdb[2] = ((iteraddr + addr) >> 24) & 0xFF;
    cdb[3] = ((iteraddr + addr) >> 16) & 0xFF;
    cdb[4] = ((iteraddr + addr) >>  8) & 0xFF;
    cdb[5] = ((iteraddr + addr)      ) & 0xFF;
    cdb[6] = 0x00;
    cdb[7] = (limit >> 8) & 0xFF;     
    cdb[8] = limit & 0xFF;
    cdb[9] = 0x00;
    ex_generic( scsi_id, cdb, 10, bf.bigbuffer, limit*512, driver_write, 1000, &scsi_status );
    if (scsi_status)
    {
      ex_preunwire(scsi_id, &d_status);              /* release the buffer */
      pfm_$rls_cleanup( &scsi_cleanup, &d_status );   /* get rid of cleanup handler */
      printf("write non-zero return - %d - for block address %d\n",scsi_status,iteraddr);
      display_sense_data(scsi_id);
      return;
    }
    iteraddr = iteraddr + limit;
    number = number - limit;
  }
  cdb[0] = 0x2a;        /* use 10 byte CDB */
  cdb[1] = 0x00;
  cdb[2] = ((iteraddr + addr) >> 24) & 0xFF;
  cdb[3] = ((iteraddr + addr) >> 16) & 0xFF;
  cdb[4] = ((iteraddr + addr) >>  8) & 0xFF;
  cdb[5] = ((iteraddr + addr)      ) & 0xFF;
  cdb[6] = 0x00;
  cdb[7] = (number >> 8) & 0xFF;     
  cdb[8] = number & 0xFF;
  cdb[9] = 0x00;
  ex_generic( scsi_id, cdb, 10, bf.bigbuffer, number*512, driver_write, 1000, &scsi_status );
  ex_preunwire(scsi_id, &d_status);              /* release the buffer */
  pfm_$rls_cleanup( &scsi_cleanup, &d_status );   /* get rid of cleanup handler */
  if (scsi_status)
  {
    printf("write non-zero return - %d - for block address %d\n",scsi_status,iteraddr);
    display_sense_data(scsi_id);
    return;
  }
}
/* --------------------------------------------------------------------------- */
direct_read( scsi_id, number, addr, limit, buflen, input )
  /* number is the total number of blocks to write
     addr   is the address on the disk at which to start writing
     limit  is the maximum number of blocks to write per scsi command
     buflen is the size of the I/O data buffer in bytes
     input  is 'y' if the data is to be displayed
  */
  unsigned char scsi_id;
  linteger number, addr, limit, buflen;
  char input;

{
  int i, j;
  linteger iteraddr;
  status_$t d_status;
  unsigned char cdb[12];
  driver_$status_t scsi_status;
  pfm_$cleanup_rec scsi_cleanup;    /* cleanup record */

  /* Play it safe and establish a cleanup handler in case the program dies during
     I/O. Don't want to leave bits & pieces of memory wired if we can help it. */

  d_status = pfm_$cleanup( &scsi_cleanup );
  if (d_status.all != pfm_$cleanup_set)        /* if !=, then we're in a fault situation */
  {
    ex_preunwire(scsi_id, &d_status);          /* release the buffer */
    pfm_$signal( d_status );                   /* go to the next cleanup level */
  }

  ex_prewire( scsi_id, bf.bigbuffer, buflen, &d_status);
  if (d_status.all != status_$ok)
  {
     printf("Unable to prewire I/O data buffer: %x\n",d_status.all);
     pfm_$rls_cleanup( &scsi_cleanup, &d_status );   /* get rid of cleanup handler */
     return;
  }
  iteraddr = 0;
  while (number > limit)
  {
    cdb[0] = 0x28;        /* use 10 byte CDB */
    cdb[1] = 0x00;
    cdb[2] = ((iteraddr + addr) >> 24) & 0xFF;
    cdb[3] = ((iteraddr + addr) >> 16) & 0xFF;
    cdb[4] = ((iteraddr + addr) >>  8) & 0xFF;
    cdb[5] = ((iteraddr + addr)      ) & 0xFF;
    cdb[6] = 0x00;
    cdb[7] = (limit >> 8) & 0xFF;     
    cdb[8] = limit & 0xFF;
    cdb[9] = 0x00;
    ex_generic( scsi_id, cdb, 10, bf.bigbuffer, limit*512, driver_read, 1000, &scsi_status );
    if (scsi_status)
    {
      ex_preunwire(scsi_id, &d_status);               /* release the buffer */
      pfm_$rls_cleanup( &scsi_cleanup, &d_status );   /* get rid of cleanup handler */
      printf("read non-zero return - %d - for block address %d\n",scsi_status,iteraddr);
      display_sense_data(scsi_id);
      return;
    }
    if ((input == 'y') | (input == 'Y'))
    {
      printf("\nRead block %d data buffer contains:\n",iteraddr);
      dump_data( bf.bigbuffer, 40); /* show what we have read */
    }
    iteraddr = iteraddr + limit;
    number = number - limit;
  }
  cdb[0] = 0x28;        /* use 10 byte CDB */
  cdb[1] = 0x00;
  cdb[2] = ((iteraddr + addr) >> 24) & 0xFF;
  cdb[3] = ((iteraddr + addr) >> 16) & 0xFF;
  cdb[4] = ((iteraddr + addr) >>  8) & 0xFF;
  cdb[5] = ((iteraddr + addr)      ) & 0xFF;
  cdb[6] = 0x00;
  cdb[7] = (number >> 8) & 0xFF;     
  cdb[8] = number & 0xFF;
  cdb[9] = 0x00;
  ex_generic( scsi_id, cdb, 10, bf.bigbuffer, number*512, driver_read, 1000, &scsi_status );
  ex_preunwire(scsi_id, &d_status);               /* release the buffer */
  pfm_$rls_cleanup( &scsi_cleanup, &d_status );   /* get rid of cleanup handler */
  if (scsi_status)
  {
    printf("read non-zero return - %d - for block address %d\n",scsi_status,iteraddr);
    display_sense_data(scsi_id);
    return;
  }
  if ((input == 'y') | (input == 'Y'))
  {
    printf("\nRead block %d data buffer contains:\n",iteraddr);
    dump_data( bf.bigbuffer, 40);   /* show what we have read */
  }
}
/* --------------------------------------------------------------------------- */
read_capacity( scsi_id, nblocks, blocksize )
  unsigned char scsi_id;
  linteger *nblocks, *blocksize;

{
  status_$t d_status;
  unsigned char cdb[12];
  driver_$status_t scsi_status;
  cdb[0] = 0x25;        /* use 10 byte CDB */
  cdb[1] = 0x00;
  cdb[2] = 0x00;
  cdb[3] = 0x00;
  cdb[4] = 0x00;
  cdb[5] = 0x00;
  cdb[6] = 0x00;
  cdb[7] = 0x00;
  cdb[8] = 0x00;
  cdb[9] = 0x00;
  ex_generic( scsi_id, cdb, 10, bf.bigbuffer, 8, driver_read, 100, &scsi_status );
  if (scsi_status)
  {
    printf("read capacity non-zero return - %d - for block address 0\n",scsi_status);
    display_sense_data(scsi_id);
    return(1);
  }
  *nblocks = (bf.bigbuffer[0] << 24) | (bf.bigbuffer[1] << 16) | (bf.bigbuffer[2] << 8) | (bf.bigbuffer[3]);
  *blocksize = (bf.bigbuffer[4] << 24) | (bf.bigbuffer[5] << 16) | (bf.bigbuffer[6] << 8) | (bf.bigbuffer[7]);
  return(0);
}
/* --------------------------------------------------------------------------- */
debug_stuff()
{
  int i, j, k, pattern, cmd, input_status, dev_type, bits;
  status_$t d_status;
  unsigned char cdb_byte_01, cdb_byte_05, cdb[12], cdb_length, mode_buffer[256];
  driver_$status_t scsi_status;
  char input, vinput;
  linteger length_read, number, addr, uid, iteraddr, limit, buflen, sector, qsector;
  unsigned char request_sense_data[1024];
  driver_$rpd_t rpd_buffer;
  driver_$direction_t dir;
  pfm_$cleanup_rec scsi_cleanup;    /* cleanup record */

  input_status = EOF + 1;
  cmd = 999;                        /* fake a command to print command list */

  ce_mode = 1;                      /* enable hex dumps of returned data */
  dev_type = -1;                    /* device type is set to be bogus */

  while (input_status != EOF)
  {
    switch (cmd)
    {
      case 0: {     /* exit */
        return;
      }
      case 1: {     /* load */
        ex_load( scsi_id, 0x01, &scsi_status);
        if (scsi_status)
        {
          printf("load non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 2: {     /* unload */
        ex_load( scsi_id, 0x00, &scsi_status);
        if (scsi_status)
        {
          printf("unload non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 3: {     /* rewind */
        ex_rewind( scsi_id, &scsi_status);
        if (scsi_status)
        {
          printf("rewind non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 4: {     /* erase */
        printf("Erase takes up to 2 hours on an Exabyte 8500. Continue? [y/n] ");
        if (scanf("%s",bf.bigbuffer)==EOF)  break;
        if ((bf.bigbuffer[0] != 'y') && (bf.bigbuffer[0] != 'Y')) break;
        ex_erase( scsi_id, 1, &scsi_status);    /* if 1 is used, then a LONG erase (to PEOT) occurs */
        if (scsi_status)                        /* this takes about 2 hours and will probably time out */
        {
          printf("erase non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 5: {     /* inquiry */
        ex_inquiry( scsi_id, 0, mode_buffer, 255, &scsi_status);
        if (scsi_status)
        {
          printf("inquiry non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        else
        {
          printf("Inquiry data: \n");
          printf("  Device:  ");
          switch(mode_buffer[0])
          {
            case 0: printf("direct access\n"); dev_type = disk_type; break;
            case 1: printf("sequential access\n"); dev_type = tape_type; break;
            case 7: printf("optical\n"); dev_type = optical_type; break;
          }
          printf("  Vendor:  ");
          for (i=8;i<16;i++) putchar(mode_buffer[i]);
          printf("\n  Product: ");
          for (i=16;i<32;i++) putchar(mode_buffer[i]);
          printf("\n  Revision: ");
          for (i=32;i<36;i++) putchar(mode_buffer[i]);
          printf("\n  Serial #: ");
          for (i=96;i<105;i++) putchar(mode_buffer[i]);
          printf("\n");
          if (mode_buffer[0] == 0x01) printf("  tape device\n");
          else if (mode_buffer[0] == 0x02) printf("  disk device (02)\n");
          else if (mode_buffer[0] == 0x7f) printf("  disk device (7f)\n");
          else if (mode_buffer[0] == 0x00) printf("  disk device (00)\n");
          if ((mode_buffer[3] & 0x0f) == 0) printf("  drive SCSI level is SCSI-1\n");
          else if ((mode_buffer[3] & 0x0f) == 0x01) printf("  drive SCSI level is SCSI-1/CCS\n");
          else if ((mode_buffer[3] & 0x0f) == 0x02) printf("  drive SCSI level is SCSI-2\n");
          else printf("  drive SCSI level is unknown: %d\n",mode_buffer[3] & 0x0f);
          printf("  ANSI version %d\n",(mode_buffer[2] & 0x07));
          if (mode_buffer[1] & 0x80) printf("  removable media\n");
          dump_data( mode_buffer, 106 );
        }
        break;
      }
      case 6: {     /* read block limits */
        ex_read_block_limits( scsi_id, mode_buffer, &scsi_status);
        if (scsi_status)
        {
          printf("read_block_limits non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        else
        {
          printf("Read Block Limits response: \n");
          number = (mode_buffer[1] << 16) | (mode_buffer[2] << 8) | mode_buffer[3];
          printf("  Maximum block size is %d\n",number);
          number = (mode_buffer[4] << 8) | mode_buffer[5];
          printf("  Minimum block size is %d\n",number);
        }
        break;
      }
      case 7: {     /* space */
        printf("Enter 0 to space over blocks\n      1 to space over filemarks\n      3 to space to EOD\n: ");
        if ((scanf("%1d",&cmd)==EOF) | ((cmd<0) | (cmd>3) | (cmd==2))) break;
        if (cmd != 3)
        {
          printf("Enter number to space: ");
          if (scanf("%d",&number)==EOF)  break;
        }
        cdb_byte_01 = cmd;
        ex_space( scsi_id, cdb_byte_01, 0x80, number, &scsi_status);
        if (scsi_status)
        {
          printf("space non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 8: {     /* verify */
        printf("Enter number of bytes to verify: ");
        if (scanf("%d",&number)==EOF)  break;
        ex_verify( scsi_id, 0, number, &scsi_status);
        if (scsi_status)
        {
          printf("verify non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 9: {     /* read position */
        ex_read_position( scsi_id, 0, rpd_buffer.all, &scsi_status);
        if (scsi_status)
        {
          printf("read_position non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
          break;
        }
        if (rpd_buffer.rpd.bop) printf("The tape is positioned at BOT\n");
        else if (rpd_buffer.rpd.eop) printf("The tape is positioned between LEOT and PEOT ");
        else if (rpd_buffer.rpd.bpu) printf("The tape positioned is unknown ");
        printf("The current tape block address is %d\n",rpd_buffer.rpd.first_block);
        break;
      }
      case 10: {     /* write_filemarks */
        printf("Enter 0 to write short filemarks or 1 to write long filemarks: ");
        if ((scanf("%1d",&cmd)==EOF) | ((cmd!=0) & (cmd!=1))) break;
        printf("Enter number of filemarks to write: ");
        if (scanf("%d",&number)==EOF)  break;
        if (cmd==1)
          cdb_byte_05 = 0;          /* long */
        else
          cdb_byte_05 = 0x80;       /* short */
        ex_write_filemarks( scsi_id, 0, cdb_byte_05, number, &scsi_status);
        if (scsi_status)
        {
          printf("write_filemarks non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 11: {    /* write */
        printf("Enter 0 for variable-length mode writes, 1 for fixed length writes: ");
        if ((scanf("%d",&j)==EOF) | (j < 0) | (j > 1)) break;
        cdb_byte_01 = j;
        printf("Enter number of bytes per block to write [1-%d]: ",scsi_$max_xfer_length);
        if ((scanf("%d",&number)==EOF) | (number < 0) | (number >= maxbufsize)) break;
        printf("Enter data pattern to write (in hex): ");
        if (scanf("%x",&pattern)==EOF) break;
        j = number;
        if (j > sizeof(bf.bigbuffer)) j = sizeof(bf.bigbuffer);
        for (i=0;i<j;i++) bf.bigbuffer[i] = pattern;           /* load data pattern into buffer */
        printf("Enter number of blocks to write: ");
        if (scanf("%d",&j)==EOF) break;
        for (i=0;i<j;i++)
        {
          ex_write( scsi_id, cdb_byte_01, bf.bigbuffer, number, &scsi_status );
          if (scsi_status)
          {
            printf("write non-zero return - %d - for block %d\n",scsi_status,i);
            display_sense_data(scsi_id);
            break;
          }
        }
        break;
      }
      case 12: {     /* locate */
        /* move to an absolute scsi block address on this tape */
        printf("Enter block address to move to: ");
        if (scanf("%d",&number)==EOF)  break;
        ex_locate( scsi_id, 0, number, &scsi_status);
        if (scsi_status)
        {
          printf("locate non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 13: {    /* read */
        printf("Enter 0 for variable-length mode reads, 1 for fixed length reads: ");
        if ((scanf("%d",&j)==EOF) | (j < 0) | (j > 2)) break;
        cdb_byte_01 = j;
        /* read as big an amount of data as we possibly can on this machine */
        printf("Enter number of bytes to read: ");
        if ((scanf("%d",&number)==EOF) | (number < 0)) break;
        if (number > sizeof(bf.bigbuffer)) number = sizeof(bf.bigbuffer);     /* don't overflow buffer storage! */
        printf("Enter number of blocks to read: ");
        if (scanf("%d",&j)==EOF) break;
        printf("Display data read? ");
        getchar();          /* eliminate '\n' */
        input=getchar();    /* get response */
        if (input==EOF) break;
        for (i=0;i<j;i++)
        {
          /*  - suppress illegal length indicator (SILI) bit set--
          ex_read( scsi_id, 0x02, bf.bigbuffer, number, &length_read, &scsi_status );
          */
          ex_read( scsi_id, cdb_byte_01, bf.bigbuffer, number, &length_read, &scsi_status );
          printf("read block %d - %d bytes:\n",i,length_read);
          if (scsi_status)
          {
            printf("read block %d - non-zero return - %d\n",i,scsi_status);
            display_sense_data(scsi_id);
          }
          if ((length_read > 0) & ((input == 'y') | (input == 'Y')))
          {
            printf("\nRead block %d data buffer contains:\n",i);
            dump_data( bf.bigbuffer, 40); /* show what we have read */
          }
        }
        break;
      }
      case 14: {    /* request sense */
        display_sense_data(scsi_id);
        break;
      }
      case 15: {    /* mode_select */
        printf("Enter 1 for default mode select,\n");
        printf("      2 for C1716/C set inquiry response\n");
        printf("      3 for 8200 mode\n");
        printf("      4 for 8500 mode\n");
        printf("      5 for user-defined mode\n");
        printf(": ");
        getchar();          /* eliminate '\n' */
        input=getchar();    /* get response */
        if (input==EOF) break;
        if (input == '1')
        {
          set_mode();
          ex_mode_select( scsi_id, 0x10, mode_buffer, 27, &scsi_status);
        } else if (input == '2')
        {
          ex_mode_sense( scsi_id, 0, 0xe1, request_sense_data, 24, &scsi_status);
          if (scsi_status)
          {
            printf("mode_sense non-zero break - %d\n",scsi_status);
            display_sense_data(scsi_id);
            break;
          }
          printf("Device currently in");
          if ((request_sense_data[14] & 0x01) == 1) printf(" Direct Access Device inquiry mode\n");
          else                                      printf(" Optical Device inquiry mode\n");
          printf("Enter 1 to change to Direct Access Device, 2 to change to Optical [1 or 2]: ");
          getchar();          /* eliminate '\n' */
          input=getchar();    /* get response */
          if (input==EOF) break;
          if (input == '1')     /* direct access device mode */
          {
            request_sense_data[0]  = 0;
            request_sense_data[2]  = 0;
            request_sense_data[8]  = 0;
            request_sense_data[12] = 0x21; /* beginning of vendor unique device configuration page */
            request_sense_data[13] = 0x0a; /* length of this page which is NOT 0x10 !!*/
            request_sense_data[14] = 0x03; /* DTIS | DAIR */
          }
          else if (input == '2')   /* optical device mode */
          {
            request_sense_data[0]  = 0;
            request_sense_data[2]  = 0;
            request_sense_data[8]  = 0;
            request_sense_data[12] = 0x21; /* beginning of vendor unique device configuration page */
            request_sense_data[13] = 0x0a; /* length of this page which is NOT 0x10 !!*/
            request_sense_data[14] = 0x02; /* DTIS only */
          }
          else break;
          /* if (i==0) set_mode_c1716(request_sense_data); */
          ex_mode_select( scsi_id, 0x10, request_sense_data, 24, &scsi_status);    /* set 'current' mode */
          if (scsi_status)
          {
            printf("mode_select (current) non-zero break - %d\n",scsi_status);
            display_sense_data(scsi_id);
            break;
          }
          ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status ); /* get the 'unit attention' that happens here */
          if (scsi_status != 0)
          {
            printf("Unable to perform request_sense command: %x\n",scsi_status);
            break;
          }
          ex_mode_select( scsi_id, 0x11, request_sense_data, 24, &scsi_status);    /* and save these bits */
          if (scsi_status)
          {
            printf("mode_select (saved) non-zero break - %d\n",scsi_status);
            display_sense_data(scsi_id);
            break;
          }
          ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status ); /* get the 'unit attention' that happens here */
          if (scsi_status != 0)
          {
            printf("Unable to perform request_sense command: %x\n",scsi_status);
            break;
          }
          ex_mode_sense( scsi_id, 0, 0xe1, request_sense_data, 24, &scsi_status);
          if (scsi_status)
          {
            printf("mode_sense non-zero break - %d\n",scsi_status);
            display_sense_data(scsi_id);
            break;
          }
          printf("Device now in");
          if ((request_sense_data[14] & 0x01) == 1) printf(" Direct Access Device inquiry mode\n");
          else                                      printf(" Optical Device inquiry mode\n");
        }
        else if (input == '3') {    /* 8200 density - mode_select */
          mode_buffer[0] = 0;
          mode_buffer[1] = 0;
          mode_buffer[2] = 0x10;    /* buffer mode & tape speed */
          mode_buffer[3] = 8;
          mode_buffer[4] = 0x14;    /* set density to 8200 mode */
          mode_buffer[5] = 0;
          mode_buffer[6] = 0;
          mode_buffer[7] = 0;
          mode_buffer[8] = 0;
          mode_buffer[9] = 0;       /* variable block length (0 bytes) */
          mode_buffer[10] = 0;
          mode_buffer[11] = 0;
        
          ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
          if (scsi_status)
          {
            printf("mode_select non-zero return - %d\n",scsi_status);
            display_sense_data(scsi_id);
          }
          break;
        }
        else if (input == '4') {    /* 8500 density - mode_select */
          mode_buffer[0] = 0;
          mode_buffer[1] = 0;
          mode_buffer[2] = 0x10;    /* buffer mode & tape speed */
          mode_buffer[3] = 8;
          mode_buffer[4] = 0;       /* set density to 8500 mode */
          mode_buffer[5] = 0;
          mode_buffer[6] = 0;
          mode_buffer[7] = 0;
          mode_buffer[8] = 0;
          mode_buffer[9] = 0;       /* variable block length (0 bytes) */
          mode_buffer[10] = 0;
          mode_buffer[11] = 0;
          ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
          if (scsi_status)
          {
            printf("mode_select non-zero return - %d\n",scsi_status);
            display_sense_data(scsi_id);
          }
          break;
        }
        else if (input == '5') {    /* user defined mode select */
          printf("Enter (hex) mode page to modify: ");
          getchar();          /* eliminate '\n' */
          if (scanf("%x",&i)==EOF)  break;
          ex_mode_sense( scsi_id, 0, (char) i, request_sense_data, 255, &scsi_status);
          if (scsi_status)
          {
            printf("mode_sense (page %x) non-zero return - %d\n", i, scsi_status);
            display_sense_data(scsi_id);
            break;
          }

          k = ((int) request_sense_data[0]) + 1;      /* number of bytes in the mode select buffer */
          request_sense_data[0] = 0x00;
          bits = 1;

          ex_mode_sense( scsi_id, 0, (char) i+0x40, mode_buffer, 255, &scsi_status);
          if (scsi_status)
          {
            printf("mode_sense (page %x changeable values) non-zero return - %d\n", i, scsi_status);
            display_sense_data(scsi_id);
            ex_mode_sense( scsi_id, 0x10, (char) i+0x40, mode_buffer, 255, &scsi_status);
            if (scsi_status)
            {
              printf("SCSI-2 mode_sense (page %x changeable values) non-zero return - %d\n", i, scsi_status);
              display_sense_data(scsi_id);
              printf("Page %d currently contains:\n", i);
              dump_data( request_sense_data, k );
              bits--;      /* no "changeable bits" display possible?!? */
            }
            else
            {
              printf("Page %x currently contains:\n", i);
              dump_data_2( request_sense_data, mode_buffer, k );
            }
          }
          else
          {
            printf("Page %x currently contains:\n", i);
            dump_data_2( request_sense_data, mode_buffer, k );
          }
          scsi_status = 0;
          printf("Enter ff for the offset to exit,\n      fe to display current select settings,\n      fd to quit,\n      fc to exit and NOT set save parameters bit\n");
          for (i=0;(i!=0xff) && (i!=0xfd) && (i!=0xfc);) {
            printf("Enter byte offset (hex): ");
            if (scanf("%x",&i)==EOF)  break;
            if (i < 0xf0) {
              printf("Contains 0x%x - Enter new value (hex): ", (int) request_sense_data[i]);
              if (scanf("%x",&j)==EOF)  break;
              request_sense_data[i] = (char) j;
            }
            else if (i == 0xfe) {
              printf("Current select settings are: \n");
              if (bits)
                dump_data_2( request_sense_data, mode_buffer, k );
              else
                dump_data( request_sense_data, k );
            }
            else if (i == 0xfd) break;
          }
          if (i != 0xfd) {
            printf("Changing to: \n");
            dump_data( request_sense_data, k );
            if (i == 0xfc)      /* don't set save page bit */
              ex_mode_select( scsi_id, 0x10, request_sense_data, k, &scsi_status);
            else
              ex_mode_select( scsi_id, 0x11, request_sense_data, k, &scsi_status);
            if (scsi_status)
            {
              printf("mode_select non-zero return - %d\n",scsi_status);
              display_sense_data(scsi_id);
            }
            break;
          }
        }
        else break;
        if (scsi_status)
        {
          printf("mode_select non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        break;
      }
      case 16: {     /* mode sense */
        if (dev_type == -1) {
          ex_inquiry( scsi_id, 0, mode_buffer, 255, &scsi_status);
          if (scsi_status)
          {
            printf("(mode sense) inquiry non-zero return - %d\n",scsi_status);
            display_sense_data(scsi_id);
            break;
          }
          else
          {
            switch(mode_buffer[0])
            {
              case 0: dev_type = disk_type; break;
              case 1: dev_type = tape_type; break;
              case 7: dev_type = optical_type; break;
            }
          }
          if (dev_type == -1) {
            if (mode_buffer[0] == 0x01) dev_type = tape_type;
            else if (mode_buffer[0] == 0x02) dev_type = disk_type;
            else if (mode_buffer[0] == 0x7f) dev_type = disk_type;
            else if (mode_buffer[0] == 0x00) dev_type = disk_type;
            }
          }

        if ((dev_type == tape_type) || (dev_type == -1)) {
          /* ask for 17 bytes of non-page mode sense data to be returned */
          ex_mode_sense( scsi_id, 0, 0, request_sense_data, 17, &scsi_status);
          if (scsi_status)
          {
            printf("mode_sense non-zero return - %d\n",scsi_status);
            display_sense_data(scsi_id);
            break;
          }

          printf("Non-page format mode sense returned %d bytes\n", (int) request_sense_data[0]);
          printf("    data cartridge type loaded is %x - ", (int) request_sense_data[1]);
          switch( request_sense_data[1] ) {
             case 0:    printf("(none)\n"); break;    /* Exabyte tape types */
             case 0x81: printf("P6-15\n"); break;
             case 0x82: printf("P6-30\n"); break;
             case 0x83: printf("P6-60\n"); break;
             case 0x84: printf("P6-90\n"); break;
             case 0x85: printf("P6-120\n"); break;
             case 0xc1: printf("P5-15\n"); break;
             case 0xc2: printf("P5-30\n"); break;
             case 0xc3: printf("P5-60\n"); break;
             case 0xc4: printf("P5-90\n"); break;
             default: printf("\n");
          }
          printf("    cartridge is ");
          if ((request_sense_data[2] & 0x80) == 0) printf("not ");
          printf("write protected\n");
          if (request_sense_data[3] == 0)
             break;                                   /* no block descriptor?! */
          printf("    unit is in ");
          switch (request_sense_data[4])
          {
             case 0x00: printf("device default format\n"); break;
             case 0x01: printf("9 track 800 bpi format\n"); break;
             case 0x02: printf("9 track 1600 bpi format\n"); break;
             case 0x03: printf("9 track 6250 bpi format\n"); break;
             case 0x04: printf("QIC-11  4 or 9 track  8000 bpi format\n"); break;
             case 0x05: printf("QIC-24  9 track  10000 ftpi format\n"); break;
             case 0x06: printf("9 track 3200 bpi format\n"); break;
             case 0x07: printf("QIC  4 track 6400 bpi format\n"); break;
             case 0x08: printf("cassette 4 track 8000 bpi format\n"); break;
             case 0x09: printf("HIC 18 track (DEC-TK50) format\n"); break;
             case 0x0a: printf("HIC 22 track 6667 bpi format\n"); break;
             case 0x0b: printf("QIC  4 track 1600 bpi format\n"); break;
             case 0x0c: printf("HIC 24 track 12690 bpi format\n"); break;
             case 0x0d: printf("HIC 24 track 25380 bpi format\n"); break;
             case 0x0f: printf("QIC-120 format\n"); break;
             case 0x10: printf("QIC-150 format\n"); break;
             case 0x11: printf("QIC-320 format\n"); break;
             case 0x12: printf("QIC-1350 format\n"); break;
             case 0x13: printf("DAT format\n"); break;
             case 0x14: printf("8mm (Exabyte 8200 2.3GB) ANSI format\n"); break;
             case 0x15: printf("8mm (Exabyte 8500 5GB) international format\n"); break;
             case 0x16: printf("HIC 48 track 10000 bpi format\n"); break;
             case 0x17: printf("HIC 48 track 42500 bpi (DEC TK-857) format\n"); break;
             default: printf("an unknown format - %x\n", (int) request_sense_data[4]);
          }
          number = request_sense_data[5];
          number = (number << 8) + request_sense_data[6];
          number = (number << 8) + request_sense_data[7];
          printf("    tape capacity is %d 1K blocks\n",number);

          /* now get the compression page, if it exists on this device */

          ex_mode_sense( scsi_id, 0, 0x0f, mode_buffer, 255, &scsi_status);
          if (scsi_status)  /* oops, no compression page support */
          {
            ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status ); /* get the 'illegal request' that may happen here */
          }
          else
          {
            if ((mode_buffer[0x02] & 0x10) == 0x10)
              printf("    device buffering enabled\n");
            if ((mode_buffer[0x0e] & 0x80) == 0x80)
              printf("    compression enabled\n");
            else
            {           /* compression either disabled or not supported */
              if ((mode_buffer[0x0e] & 0x40) == 0x40)
                printf("    compression disabled\n");
            }
            if ((mode_buffer[0x0f] & 0x80) == 0x80)
              printf("    decompression enabled\n");
            else
            {           /* compression either disabled or not supported */
              if ((mode_buffer[0x0e] & 0x40) == 0x40)
                printf("    decompression disabled\n");
            }
          }
          break;
        }
      else if (dev_type == disk_type) {

        /* ask for page 8 (cache) mode sense data to be returned */
        ex_mode_sense( scsi_id, 0, 0x08, request_sense_data, 255, &scsi_status);
        if (scsi_status)
        {
          printf("mode_sense (page 8) non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        else
        {
          printf("page 8 (cache) returned %d bytes\n", (int) request_sense_data[0]);
          printf("    drive is ");
          if ((request_sense_data[2] & 0x80) == 0) printf("not ");
          printf("write protected\n");
          if (request_sense_data[3] == 0)
             break;                                   /* no block descriptor?! */
          i = (((int) request_sense_data[9]) << 16) + (((int) request_sense_data[10]) << 8) + ((int) request_sense_data[11]);
          printf("    sector size is %d bytes\n", i);
          number = request_sense_data[5];
          number = (number << 8) + request_sense_data[6];
          number = (number << 8) + request_sense_data[7];
          printf("    drive capacity is %d blocks\n",number);

          printf("    cache page byte 2 is %x\n", (int) request_sense_data[14]);
          printf("      IC:   %d\n", (request_sense_data[14] & 0x80) >> 7);
          printf("      ABPF: %d\n", (request_sense_data[14] & 0x40) >> 6);
          printf("      CAP : %d\n", (request_sense_data[14] & 0x20) >> 5);
          printf("      DISC: %d\n", (request_sense_data[14] & 0x10) >> 4);
          printf("      SIZE: %d\n", (request_sense_data[14] & 0x08) >> 3);
          printf("      WCE : %d\n", (request_sense_data[14] & 0x04) >> 2);
          printf("      MF  : %d\n", (request_sense_data[14] & 0x02) >> 1);
          printf("      RCD : %d\n",  request_sense_data[14] & 0x01);
          printf("    retention byte is    %x\n", (int) request_sense_data[15]);
          printf("    disable prefetch len %d\n", ((int) request_sense_data[16] * 256) + (int) request_sense_data[17] );
          printf("    minimum prefetch len %d\n", ((int) request_sense_data[18] * 256) + (int) request_sense_data[19] );
          printf("    maximum prefetch len %d\n", ((int) request_sense_data[20] * 256) + (int) request_sense_data[21] );
          printf("    ceiling prefetch len %d\n", ((int) request_sense_data[22] * 256) + (int) request_sense_data[23] );
          if (request_sense_data[13] >= 12) {
            printf("    cache page byte  12: %x\n", (int) request_sense_data[24]);
            printf("      FSW:  %d\n", (request_sense_data[24] & 0x80) >> 7);
            printf("      DRA:  %d\n", (request_sense_data[24] & 0x20) >> 5);
            if (request_sense_data[13] >= 13) {
              printf("    # of cache segments: %d\n", (int) request_sense_data[25]);
              if (request_sense_data[13] >= 15) {
                printf("    cache segment size:  %d\n", ((int) request_sense_data[26] * 256) + (int) request_sense_data[27] );
                if (request_sense_data[13] >= 19) {
                  i = (((int) request_sense_data[29]) << 16) + (((int) request_sense_data[30]) << 8) + ((int) request_sense_data[31]);
                  printf("    non-cache seg size:  %d\n", i);
                }
              }
            }
          }
          dump_data( request_sense_data, (int) request_sense_data[0] + 1);
        }

        /* ask for page 4 (geometry) mode sense data to be returned */
        ex_mode_sense( scsi_id, 0, 0x04, request_sense_data, 255, &scsi_status);
        if (scsi_status)
        {
          printf("mode_sense (page 4) non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        else
        {
          printf("page 4 (geometry) returned %d bytes\n", (int) request_sense_data[0]);

          i = (((int) request_sense_data[14]) << 16) + (((int) request_sense_data[15]) << 8) + ((int) request_sense_data[16]);
          printf("    number of cylinders: %d\n", i);
          printf("    number of heads:     %d\n", (int) request_sense_data[17]);
          i = (((int) request_sense_data[18]) << 16) + (((int) request_sense_data[19]) << 8) + ((int) request_sense_data[20]);
          printf("    write precomp start: %d\n", i);
          i = (((int) request_sense_data[21]) << 16) + (((int) request_sense_data[22]) << 8) + ((int) request_sense_data[23]);
          printf("    reduced wrt current: %d\n", i);
          i = (((int) request_sense_data[24]) << 8) + ((int) request_sense_data[25]);
          printf("    drive step rate:     %d\n", i);
          i = (((int) request_sense_data[26]) << 16) + (((int) request_sense_data[27]) << 8) + ((int) request_sense_data[28]);
          printf("    landing zone cyl:    %d\n", i);
          i = (((int) request_sense_data[29]) << 8) + ((int) request_sense_data[30]);
          printf("    rotational rate:     %d\n", i);
          dump_data( request_sense_data, (int) request_sense_data[0] + 1);
        }

        /* ask for page 2 (disconnect/reconnect) mode sense data to be returned */
        ex_mode_sense( scsi_id, 0, 0x02, request_sense_data, 255, &scsi_status);
        if (scsi_status)
        {
          printf("mode_sense (page 2) non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        else
        {
          printf("page 2 (disconnect/reconnect) returned %d bytes\n", (int) request_sense_data[0]);
          printf("    buffer full ratio:   %d\n", (int) request_sense_data[14]);
          printf("    buffer empty ratio:  %d\n", (int) request_sense_data[15]);
          i = (((int) request_sense_data[16]) << 8) + ((int) request_sense_data[17]);
          printf("    bus inactivity limit %d\n", i);
          i = (((int) request_sense_data[18]) << 8) + ((int) request_sense_data[19]);
          printf("    disconnect timelimit %d\n", i);
          i = (((int) request_sense_data[20]) << 8) + ((int) request_sense_data[21]);
          printf("    connect time limit:  %d\n", i);
          i = (((int) request_sense_data[22]) << 8) + ((int) request_sense_data[23]);
          printf("    maximum burst size:  %d\n", i);
          dump_data( request_sense_data, (int) request_sense_data[0] + 1);
        }

        /* ask for page 1 (error recovery) mode sense data to be returned */
        ex_mode_sense( scsi_id, 0, 0x01, request_sense_data, 255, &scsi_status);
        if (scsi_status)
        {
          printf("mode_sense (page 1) non-zero return - %d\n",scsi_status);
          display_sense_data(scsi_id);
        }
        else
        {
          printf("page 1 (error recovery) returned %d bytes\n", (int) request_sense_data[0]);
          printf("    error recovery page byte 2 is %x\n", (int) request_sense_data[14]);
          printf("      AWRE: %d\n", (request_sense_data[14] & 0x80) >> 7);
          printf("      ARRE: %d\n", (request_sense_data[14] & 0x40) >> 6);
          printf("      TB  : %d\n", (request_sense_data[14] & 0x20) >> 5);
          printf("      RC  : %d\n", (request_sense_data[14] & 0x10) >> 4);
          printf("      EER : %d\n", (request_sense_data[14] & 0x08) >> 3);
          printf("      PER : %d\n", (request_sense_data[14] & 0x04) >> 2);
          printf("      DTE : %d\n", (request_sense_data[14] & 0x02) >> 1);
          printf("      DCR : %d\n",  request_sense_data[14] & 0x01);
          printf("    read retry count:    %d\n", (int) request_sense_data[15]);
          printf("    write retry count:   %d\n", (int) request_sense_data[20]);
          dump_data( request_sense_data, (int) request_sense_data[0] + 1);
        }

        break;
        }
      }
      case 17: {    /* test unit ready */
        ex_test_unit_ready( scsi_id, &scsi_status );
        if (scsi_status) printf("unit is not ready\n",scsi_status);
        else printf("unit is ready\n");
        break;
      }
      case 18: {    /* generic SCSI command */
        printf("Generic SCSI command\nEnter values in hex for a 10-byte (max) CDB:\n");
        for (cdb_length=0;cdb_length<10;cdb_length++)
        {
          printf("CDB(%2x) = ",cdb_length);
          if ((scanf("%2x",&cmd)==EOF) ) break;
          cdb[cdb_length] = cmd;
        }
        cdb_length--;
        printf("Enter number of bytes in buffer to initialize: ");
        if (scanf("%d",&number)==EOF)  break;
        for (i=0;i<number;i++)
        {
          printf("buffer(%2x) = ",i);
          if ((scanf("%2x",&cmd)==EOF) ) break;
          bf.bigbuffer[i] = cmd;
        }
        printf("Enter size of buffer to be used (if different from initialized buffer): ");
        if (scanf("%d",&i)==EOF)  break;
        if (i > number) number = i;
        printf("Enter 0 if reading from device, 1 if writing to device: ");
        if (scanf("%d",&dir)==EOF)  break;
        if (dir = 0) dir = driver_read;
        else if (dir = 1) dir = driver_write;
        else break;
        printf("Enter number of milliseconds to wait before command times out: ");
        if (scanf("%d",&length_read)==EOF)  break;
        ex_generic( scsi_id, cdb, cdb_length, bf.bigbuffer, number, dir, length_read, &scsi_status );
        if (scsi_status) printf("generic command execution had non-zero return - %d\n",scsi_status);
        else printf("successful execution\n");
        break;
      }
      case 19: {    /* format unit command */
        cdb[0] = 0x04;
        cdb[1] = 0x00;      /* don't bother with defect lists, etc. */
        cdb[2] = 0x00;
        cdb[3] = 0x00;      /* default interleave */
        cdb[4] = 0x00;
        cdb[5] = 0x00;
        ex_generic( scsi_id, cdb, 6, bf.bigbuffer, 0, driver_write, 60*60*1000, &scsi_status );
        if (scsi_status) printf("format command execution had non-zero return - %d\n",scsi_status);
        else printf("format complete\n");
        break;
      }
      case 20: {    /* change definition command */
        printf("Enter 1 for SCSI-1, 2 for SCSI-1/CCS or 3 for SCSI-2: ");
        if (scanf("%d",&number)==EOF)  break;
        if ((number < 0) || (number > 3)) break;
        cdb[0] = 0x40;
        cdb[1] = 0x00;
        cdb[2] = 0x01;      /* save this */
        cdb[3] = number;
        cdb[4] = 0x00;
        cdb[5] = 0x00;
        cdb[6] = 0x00;
        cdb[7] = 0x00;
        cdb[8] = 0x00;
        cdb[9] = 0x00;
        ex_generic( scsi_id, cdb, 10, bf.bigbuffer, 0, driver_write, 5, &scsi_status );
        if (scsi_status) printf("change definition command execution had non-zero return - %d\n",scsi_status);
        else printf("definition changed\n");
        break;
      }
      case 21:                /* initialize floppy to 1.2MB format */
      {
        printf("Enter 0 for 1.2MB, 1 for 360KB drive density: ");
        if (scanf("%d",&number)==EOF)  break;
        if (number == 0)        /* 1.2MB format */
        {
          bf.bigbuffer[0] = 0x00;  /* set up header */
          bf.bigbuffer[1] = 0x1b;
          bf.bigbuffer[2] = 0x00;
          bf.bigbuffer[3] = 0x00;
          ex_mode_select( scsi_id, 0x10, bf.bigbuffer, 4, &scsi_status );
        }
        else if (number == 1)   /* 360KB format */
        {
          bf.bigbuffer[0] = 0x00;  /* set up header */
          bf.bigbuffer[1] = 0x27;
          bf.bigbuffer[2] = 0x00;
          bf.bigbuffer[3] = 0x08;

          bf.bigbuffer[4] = 0x00;    /* set up block descriptor */
          bf.bigbuffer[5] = 0x00;
          bf.bigbuffer[6] = 0x00;
          bf.bigbuffer[7] = 0x00;
          bf.bigbuffer[8] = 0x00;
          bf.bigbuffer[9] = 0x00;
          bf.bigbuffer[10] = 0x02;
          bf.bigbuffer[11] = 0x00;

          bf.bigbuffer[12] = 0x05;   /* page 5 descriptor */
          bf.bigbuffer[13] = 0x1e;
          bf.bigbuffer[14] = 0x00;
          bf.bigbuffer[15] = 0xfa;
          bf.bigbuffer[16] = 0x02;
          bf.bigbuffer[17] = 0x09;
          bf.bigbuffer[18] = 0x02;
          bf.bigbuffer[19] = 0x00;

          bf.bigbuffer[20] = 0x00;
          bf.bigbuffer[21] = 0x28;
          bf.bigbuffer[22] = 0x00;
          bf.bigbuffer[23] = 0x00;
          bf.bigbuffer[24] = 0x00;
          bf.bigbuffer[25] = 0x00;
          bf.bigbuffer[26] = 0x00;
          bf.bigbuffer[27] = 0x3c;

          bf.bigbuffer[28] = 0x00;
          bf.bigbuffer[29] = 0x00;
          bf.bigbuffer[30] = 0x96;
          bf.bigbuffer[31] = 0x05;
          bf.bigbuffer[32] = 0x46;
          bf.bigbuffer[33] = 0x60;
          bf.bigbuffer[34] = 0x02;
          bf.bigbuffer[35] = 0x00;

          bf.bigbuffer[36] = 0x00;
          bf.bigbuffer[37] = 0x00;
          bf.bigbuffer[38] = 0x25;
          bf.bigbuffer[39] = 0x00;
          bf.bigbuffer[40] = 0x00;
          bf.bigbuffer[41] = 0x00;
          bf.bigbuffer[42] = 0x00;
          bf.bigbuffer[43] = 0x00;
          ex_mode_select( scsi_id, 0x10, bf.bigbuffer, 44, &scsi_status );
        }
        else break;
        if (scsi_status) printf("initialize floppy command execution had non-zero return - %d\n",scsi_status);
        else printf("initialize complete\n");
        break;
      }
      case 22:                /* retension cartridge tape drive */
      {
        ex_load( scsi_id, 0x03, &scsi_status);
        if (scsi_status) printf("retension command execution had non-zero return - %d\n",scsi_status);
        else printf("retension complete\n");
        break;
      }
      case 23: {    /* direct access write */
        printf("Enter disk address at which to start writing: ");
        if ((scanf("%d",&addr)==EOF)) break;
        printf("Enter number of blocks per write to write [1-%d]: ",scsi_$max_xfer_length/512);
        if ((scanf("%d",&number)==EOF) || (number < 1) || (number > (scsi_$max_xfer_length/512))) break;
        buflen = number * 512 / 4;
        if (buflen > (sizeof(bf.biglbuffer) / 4)) buflen = sizeof(bf.biglbuffer) / 4;
        for (uid=0;uid<buflen;uid++) bf.biglbuffer[uid] = uid+addr;     /* load unique data pattern into buffer */
        printf("Enter number of blocks to write: ");
        if (scanf("%d",&limit)==EOF) break;
        direct_write( scsi_id, limit, addr, number, buflen * 4 );
        break;
      }
      case 24: {     /* seek */
        /* move to an absolute scsi block address on this disk */
        printf("Enter block address to seek to: ");
        if (scanf("%d",&number)==EOF)  break;
        cdb[0] = 0x28;        /* use 10 byte CDB */
        cdb[1] = 0x00;
        cdb[2] = (number >> 24) & 0xFF;
        cdb[3] = (number >> 16) & 0xFF;
        cdb[4] = (number >>  8) & 0xFF;
        cdb[5] =  number        & 0xFF;
        cdb[6] = 0x00;
        cdb[7] = 0x00;
        cdb[8] = 0x00;          /* zero length reads perform seeks */
        cdb[9] = 0x00;
        ex_generic( scsi_id, cdb, 10, bf.bigbuffer, 0, driver_read, 100, &scsi_status );
        if (scsi_status)
        {
          printf("seek non-zero return - %d - for block address %d\n",scsi_status,number);
          display_sense_data(scsi_id);
          break;
        }
        break;
      }
      case 25: {    /* direct access read */
        printf("Enter disk address at which to start reading: ");
        if ((scanf("%d",&addr)==EOF)) break;
        printf("Enter number of blocks per read to read [1-%d]: ",scsi_$max_xfer_length/512);
        if ((scanf("%d",&number)==EOF) || (number < 1) || (number > (scsi_$max_xfer_length/512))) break;
        j = number * 512;
        if (j > sizeof(bf.bigbuffer)) j = sizeof(bf.bigbuffer);
        printf("Enter number of blocks to read: ");
        if (scanf("%d",&limit)==EOF) break;
        printf("Display data read? ");
        getchar();          /* eliminate '\n' */
        input=getchar();    /* get response */
        if (input==EOF) break;
        direct_read( scsi_id, limit, addr, number, buflen, input );
        break;
      }
      case 26: {     /* read capacity */
        if (read_capacity( scsi_id, &limit, &number ) == 0)
          printf("Drive capacity is %d (0x%x) blocks. Block size is %d bytes\n",limit,limit,number);
        break;
      }
      case 27: {     /* exercise drive */
        if (read_capacity( scsi_id, &limit, &number) != 0)  /* get sector size & disk size */
        {
          printf("Unable to compute size of drive -- aborting\n");
          break;
        }
        sector = number;                                    /* sector size in bytes */
        qsector = sector / 4;                               /* sector size in lintegers */
        buflen = scsi_$max_xfer_length / number;            /* divide buffer into sector-sized chunks */
        if (scsi_$max_xfer_length > sizeof(bf.biglbuffer)) buflen = sizeof(bf.biglbuffer) / number;
        printf("Initialize disk? ");
        getchar();                                          /* eliminate '\n' */
        input=getchar();                                    /* get response */
        if (input==EOF) break;
        printf("Verify during scans? ");
        getchar();                                          /* eliminate '\n' */
        vinput=getchar();                                   /* get response */
        if (vinput==EOF) break;
        if ((vinput=='y') || (vinput=='Y')) vinput = 'y';
        if ((input=='y') || (input=='Y'))
        {
          printf("Verify initialization? ");
          getchar();                                        /* eliminate '\n' */
          input=getchar();                                  /* get response */
          if (input==EOF) break;
          else if (input == 'Y') input = 'y';
          iteraddr = (limit - buflen) / 10;                 /* compute increment for progress report */
          number = iteraddr;
          i = 10;
          printf("initializing drive...\n");
          for (addr = 0; addr < (limit-buflen); addr = addr+buflen)
          {
            for (uid = 0; uid < (buflen*qsector); uid++)
              bf.biglbuffer[uid] = uid + (addr * qsector);              /* load unique data pattern into buffer */
            direct_write( scsi_id, buflen, addr, buflen, buflen * sector );
            if (input=='y')
            {
              direct_read( scsi_id, buflen, addr, buflen, buflen * sector );
              for (uid=0;uid<buflen*qsector;uid++)
              {
                if (bf.biglbuffer[uid] != uid + (addr * qsector))         /* see if things are as they should be */
                  printf("data comparison error at block %7d byte %2x - %x should be %x\n",addr+uid/sector,uid&0xff,bf.biglbuffer[uid],uid+(addr*qsector));
              }
            }
            if (addr > number)
            {
              printf("   %d\%\n",i);
              i = i + 10;
              number = number + iteraddr;
            }
          }
        }
        printf("long scanning drive...\n");
        i = 1000;
        k = 1000;
        addr = 100;
        buflen = 3;         /* just do 3 blocks */
        direct_read( scsi_id, buflen, addr, buflen, buflen * sector ); /* load data pattern */
        for (j=0;j<10000;j++)
        {
          direct_write( scsi_id, buflen, addr+buflen, buflen, buflen * sector );
          direct_read( scsi_id, buflen, addr, buflen, buflen * sector );
          if (vinput == 'y')
          {
            for (uid=0;uid<(buflen*qsector);uid++)
            {
              if (bf.biglbuffer[uid] != uid + (addr * qsector))   /* see if things are as they should be */
                printf("data error at block %d byte %2x - %x should be %x\n",addr+uid/sector,uid&0xff,bf.biglbuffer[uid],uid+(addr*qsector));
            }
          }
          if (j > i)
          {
            printf("   %d\%\n",i*10/k);
            i = i + k;
          }
        }
        if (vinput != 'y')
        {
          for (uid=0;uid<(buflen*qsector);uid++)
          {
            if (bf.biglbuffer[uid] != uid + (addr * qsector))   /* see if things are as they should be */
              printf("data error at block %d byte %2x - %x should be %x\n",addr+uid/sector,uid&0xff,bf.biglbuffer[uid],uid+(addr*qsector));
          }
        }
        printf("short scanning drive...\n");
        i = 1000;
        k = 1000;
        addr = 2000;
        buflen = 1;         /* just do 1 block */
        direct_read( scsi_id, buflen, addr, buflen, buflen * sector );
        for (j=0;j<10000;j++)
        {
          direct_write( scsi_id, buflen, addr, buflen, buflen * sector );
          direct_read( scsi_id, buflen, addr, buflen, buflen * sector );
          if (vinput == 'y')
          {
            for (uid=0;uid<(buflen*qsector);uid++)
            {
              if (bf.biglbuffer[uid] != uid + (addr * qsector))   /* see if things are as they should be */
                printf("data error at block %d byte %2x - %x should be %x\n",addr+uid/sector,uid&0xff,bf.biglbuffer[uid],uid+(addr*qsector));
            }
          }
          if (j > i)
          {
            printf("   %d\%\n",i*10/k);
            i = i + k;
          }
        }
        if (vinput != 'y')
        {
          for (uid=0;uid<(buflen*qsector);uid++)
          {
            if (bf.biglbuffer[uid] != uid + (addr * qsector))   /* see if things are as they should be */
              printf("data error at block %d byte %2x - %x should be %x\n",addr+uid/sector,uid&0xff,bf.biglbuffer[uid],uid+(addr*qsector));
          }
        }
        printf("exercise complete");
        break;
      }
      default: {
        printf(" 0 - Exit                 14 - Request Sense\n");
        printf(" 1 - Load                 15 - Mode Select\n");
        printf(" 2 - Unload               16 - Mode Sense\n");
        printf(" 3 - Rewind               17 - Test Unit Ready\n");
        printf(" 4 - Erase                18 - Generic SCSI Command\n");
        printf(" 5 - Inquiry              19 - Format Unit\n");
        printf(" 6 - Read Block Limits    20 - Change Definition\n");
        printf(" 7 - Space                21 - Initialize Floppy\n");
        printf(" 8 - Verify               22 - Retension Cartridge\n");
        printf(" 9 - Read Position        23 - Direct Access Write\n");
        printf("10 - Write Filemarks      24 - Seek\n");
        printf("11 - Write                25 - Direct Access Read\n");
        printf("12 - Locate               26 - Read Capacity\n");
        printf("13 - Read                 27 - Exercise Disk\n");
      }
    }
    printf("\nEnter desired SCSI command: ");
    input_status = scanf("%d",&cmd);
    if (input_status==0) return(0);
  }
}
/* --------------------------------------------------------------------------- */
main(argc,argv)
int argc;
char *argv[];

{
  int i, j, pattern, cmd, exitstat, dev_type;
  char *ptr1, *ptr2, *ptrx;
  status_$t d_status;
  unsigned char cdb_byte_01, cdb_byte_05, cdb[12], cdb_length;
  char input;
  linteger length_read, number;
  unsigned char request_sense_data[1024];
  driver_$rpd_t rpd_buffer;
  driver_$direction_t dir;

  if ((argc < 3) | (argc > 4))
  {
    printf("Usage: %s scsi_id command [count]\n",argv[0]);
    printf("called with %s ",argv[0]);
    for (j=1;j<argc;j++)
    {
      printf("%s ",argv[j]);
    }
    printf("\n");
    exit(1);
  }
  sscanf(argv[1],"%d",&j);
  if ((j<0) | (j>6))
  {
    printf("SCSI ID must be in range [0-6]: %d\n",j);
    exit(1);
  }
  scsi_id = j;

  for (cmd=0;cmd<maxcmd;cmd++)       /* decode the command */
  {
    ptr1 = argv[2];                 /* point to the command specifier argument */
    ptr2 = &command[cmd][0];        /* point to the next command entry */
    for (i=0; ((*ptr1!=NULL) && (*ptr2!=NULL)) && (*ptr1==*ptr2); i++, ptr1++, ptr2++); /* compare to this entry */
    if ((*ptr2 == NULL) && (*ptr1 == NULL)) break;      /* if we are NULL, then we found a match, so break */
  }
  if (*ptr1 != *ptr2)
  {
    printf("Unknown command %s\n",argv[2]);
    exit(1);                        /* if no match, then complain & quit */
  }

  j = 1;                            /* assume at least 1 */
  if (argc > 2)
    sscanf(argv[3],"%d",&j);
  number = j;

  ex_initialize( scsi_id, &d_status );
  if (d_status.all != status_$ok)
  {
    if ((d_status.all & 0xffffff) != scsi_$device_in_use)
      printf("Unable to open /dev/scsi%1d\n - run make_devices\n",j);
    else
      printf("Unable to open /dev/scsi%1d\n - device is in use\n",j);
    exit(2);                /* command failed */
  }

/* Do the command */
  exitstat = 0;             /* assume successful exit */

  ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status ); /* get the 'unit attention' that may happen here */

  switch (cmd)
  {
    case 0:
    case 1:                 /* write (long) filemarks */
      {
      cdb_byte_05 = 0;      /* long filemark */
      ex_write_filemarks( scsi_id, 0, cdb_byte_05, number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 2:                 /* space forward filemarks */
    {
      ex_space( scsi_id, 1, 0x80, number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 3:                 /* space forward blocks */
    {
      ex_space( scsi_id, 0, 0x80, number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 4:                 /* space backward filemarks */
    {
      ex_space( scsi_id, 1, 0x80, -number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 5:                 /* space backward blocks */
    {
      ex_space( scsi_id, 0, 0x80, -number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 6:                 /* erase */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_erase( scsi_id, 1, &scsi_status);    /* if 1 is used, then a LONG erase (to PEOT) occurs */
      if (scsi_status) exitstat = 2;          /* this takes about 2 hours and will probably time out */
      break;
    }
    case 7:                 /* retension - retension & load, actually */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_load( scsi_id, 0x03, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 8:                 /* rewind & rew */
    case 9:
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_rewind( scsi_id, &scsi_status);
      if (scsi_status)
      {                                             /* clear the error */
        ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status );
        if (scsi_status != 0)
        {
          printf("Unable to perform request_sense command: %x\n",scsi_status);
          exitstat = 2;
          break;
        }
        ex_rewind( scsi_id, &scsi_status);          /* try again */
        if (scsi_status != 0)
        {
          display_sense_data(scsi_id);
          exitstat = 2;
        }
      }
      break;
    }
    case 10:                /* unload = offline & rewoffl */
    case 11:
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_load( scsi_id, 0x00, &scsi_status);
      if (scsi_status)
      {                                             /* clear the error */
        ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status );
        if (scsi_status != 0)
        {
          printf("Unable to perform request_sense command: %x\n",scsi_status);
          exitstat = 2;
          break;
        }
        ex_load( scsi_id, 0x00, &scsi_status);      /* try again */
        if (scsi_status != 0)
        {
          display_sense_data(scsi_id);
          exitstat = 2;
        }
      }
      break;
    }
    case 12:                /* status - request sense */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      display_sense_data(scsi_id);

      ex_inquiry( scsi_id, 0, mode_buffer, 255, &scsi_status);  /* see which kind of drive this is */
      if (scsi_status)
      {
        printf("(status) inquiry non-zero return - %d\n",scsi_status);
        display_sense_data(scsi_id);
        exitstat = 2;
        break;
      }
      if (mode_buffer[0] != 1) break;   /* don't do anything else if this is not a tape device */

      ex_mode_sense( scsi_id, 0, 0x0f, mode_buffer, 255, &scsi_status);
      if (scsi_status)
      {
        ex_request_sense( scsi_id, 0, sense_data.all, sizeof(sense_data), &scsi_status ); /* get the 'unit attention' that may happen here */
      }
      else
      {
        if ((mode_buffer[0x02] & 0x10) == 0x10)
          printf("device buffering enabled\n");
        if ((mode_buffer[0x0e] & 0x80) == 0x80)
          printf("compression enabled\n");
        else
        {           /* compression either disabled or not supported */
          if ((mode_buffer[0x0e] & 0x40) == 0x40)
            printf("compression disabled\n");
        }
        if ((mode_buffer[0x0f] & 0x80) == 0x80)
          printf("decompression enabled\n");
        else
        {           /* compression either disabled or not supported */
          if ((mode_buffer[0x0e] & 0x40) == 0x40)
            printf("decompression disabled\n");
        }
      }

      break;
    }
    case 13:               /* high density - mode_select - 8500 mode */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */

      ex_inquiry( scsi_id, 0, mode_buffer, 255, &scsi_status);  /* see which kind of drive this is */
      if (scsi_status)
      {
        printf("(mode sense) inquiry non-zero return - %d\n",scsi_status);
        display_sense_data(scsi_id);
        exitstat = 2;
        break;
      }
      switch(mode_buffer[0])
      {
        case 0: dev_type = disk_type; break;
        case 1: dev_type = tape_type; break;
        case 2: dev_type = disk_type; break;
        case 7: dev_type = optical_type; break;
        case 0x7f: dev_type = disk_type; break;
        default: dev_type = -1;
      }

      if (dev_type != tape_type) break;     /* don't do anything if this is not a tape device */

      if ((mode_buffer[16] == 'H') && (mode_buffer[17] == 'P') &&
          (mode_buffer[18] == '3') && (mode_buffer[19] == '5') &&
          (mode_buffer[20] == '4') && (mode_buffer[21] == '8'))     /* if this is an HP 35480, then */
      {
        ex_mode_sense( scsi_id, 0, 0x0f, mode_buffer, 255, &scsi_status);
        if (scsi_status)
        {
          printf("mode_sense (page %x) non-zero return - %d\n", i, scsi_status);
          display_sense_data(scsi_id);
          exitstat = 2;
          break;
        }
        i = ((int) mode_buffer[0]) + 1; /* number of bytes returned by mode sense */
        mode_buffer[0x00] = 0;        /* zap length */
        mode_buffer[0x02] = 0x10;     /* enable buffering */
        mode_buffer[0x0e] = 0xc0;     /* enable compression */
        mode_buffer[0x0f] = 0x80;     /* enable decompression, and no s/w decompression */
        mode_buffer[0x13] = 0x20;     /* compression algorithm 1, er, 20 (HP manual lies!) */
        ex_mode_select( scsi_id, 0x10, mode_buffer, i, &scsi_status);
      }
      else
      {
        set_mode();
        ex_mode_select( scsi_id, 0x10, mode_buffer, 27, &scsi_status);
      }

      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 14:               /* low density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_inquiry( scsi_id, 0, mode_buffer, 255, &scsi_status);  /* see which kind of drive this is */
      if (scsi_status)
      {
        printf("(mode sense) inquiry non-zero return - %d\n",scsi_status);
        display_sense_data(scsi_id);
        exitstat = 2;
        break;
      }
      switch(mode_buffer[0])
      {
        case 0: dev_type = disk_type; break;
        case 1: dev_type = tape_type; break;
        case 2: dev_type = disk_type; break;
        case 7: dev_type = optical_type; break;
        case 0x7f: dev_type = disk_type; break;
        default: dev_type = -1;
      }
      if (dev_type != tape_type) break;     /* don't do anything if this is not a tape device */

      if ((mode_buffer[16] == 'H') && (mode_buffer[17] == 'P') &&
          (mode_buffer[18] == '3') && (mode_buffer[19] == '5') &&
          (mode_buffer[20] == '4') && (mode_buffer[21] == '8'))     /* if this is an HP 35480, then */
      {
        ex_mode_sense( scsi_id, 0, 0x0f, mode_buffer, 255, &scsi_status);
        if (scsi_status)
        {
          printf("mode_sense (page %x) non-zero return - %d\n", i, scsi_status);
          display_sense_data(scsi_id);
          exitstat = 2;
          break;
        }
        i = ((int) mode_buffer[0]) + 1; /* number of bytes returned by mode sense */
        mode_buffer[0x00] = 0;        /* zap length */
        mode_buffer[0x02] = 0x10;     /* enable buffering */
        mode_buffer[0x0e] = 0x40;     /* disable compression */
        mode_buffer[0x0f] = 0x80;     /* enable decompression, and no s/w decompression */
        mode_buffer[0x13] = 0;        /* no compression algorithm */
        ex_mode_select( scsi_id, 0x10, mode_buffer, i, &scsi_status);
      }
      else                                                           /* else use default method */
      {
        set_mode();
        mode_buffer[4] = 0x14;  /* set tape drive to 0x14 - 8200 mode */
        ex_mode_select( scsi_id, 0x10, mode_buffer, /*27*/ 12, &scsi_status);
      }
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 15:                /* read position */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_read_position( scsi_id, 0, rpd_buffer.all, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
        break;
      }
      if (rpd_buffer.rpd.bop) printf("BOT ");
      else if (rpd_buffer.rpd.eop) printf("LEOT-PEOT ");
      else if (rpd_buffer.rpd.bpu) printf("unknown ");
      printf("%d\n",rpd_buffer.rpd.first_block);
      break;
    }
    case 16:
    {                       /* moveto - locate */
      /* move to an absolute scsi block address on this tape */
      ex_locate( scsi_id, 0, number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 17:                /* space to eod */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_space( scsi_id, 3, 0x80, number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 18:                /* write short filemarks */
      {
      cdb_byte_05 = 0x80;   /* short filemark */
      ex_write_filemarks( scsi_id, 0, cdb_byte_05, number, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 19:                /* test unit ready */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_test_unit_ready( scsi_id, &scsi_status );
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 20:                /* report version number of this program */
    {
      printf("version 1.10\n");
      break;
    }
    case 21:                /* interactive mode */
    {
      printf("MTS (debug) v1.10\n\nUsing SCSI device %1d\n",scsi_id);
      printf("Maximum SCSI transfer length on this system is %d bytes\n",scsi_$max_xfer_length);

      debug_stuff();
      exitstat = 0;         /* say everything is hunky-dory */
      break;
    }
    case 22:                /* initialize floppy to 1.2MB format */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      bf.bigbuffer[0] = 0x00;  /* set up block descriptor header */
      bf.bigbuffer[1] = 0x1b;
      bf.bigbuffer[2] = 0x00;
      bf.bigbuffer[3] = 0x00;
      ex_mode_select( scsi_id, 0x10, bf.bigbuffer, 4, &scsi_status );
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
    }
    case 23:                /* retension cartridge tape drive */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_load( scsi_id, 0x03, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 24:               /* QIC-11 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0;       /* buffer mode & tape speed */
      mode_buffer[3] = 8;

      mode_buffer[4] = 0x04;    /* set density to 60MB mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* fixed block length of 512 bytes */
      mode_buffer[10] = 0x02;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 25:               /* QIC-24 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0;       /* buffer mode & tape speed */
      mode_buffer[3] = 8;

      mode_buffer[4] = 0x05;    /* set density to other 60MB mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* fixed block length of 512 bytes */
      mode_buffer[10] = 0x02;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 26:               /* QIC-120 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0;       /* buffer mode & tape speed */
      mode_buffer[3] = 8;

      mode_buffer[4] = 0x0f;    /* set density to 120MB mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* fixed block length of 512 bytes */
      mode_buffer[10] = 0x02;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 27:               /* QIC-150 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0;       /* buffer mode & tape speed */
      mode_buffer[3] = 8;

      mode_buffer[4] = 0x10;    /* set density to 150MB mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* fixed block length of 512 bytes */
      mode_buffer[10] = 0x02;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 28:               /* QIC-525 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0;       /* buffer mode & tape speed */
      mode_buffer[3] = 8;

      mode_buffer[4] = 0x11;    /* set density to 525MB mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* fixed block length of 512 bytes */
      mode_buffer[10] = 0x02;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 29:               /* 8200 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0x10;    /* buffer mode & tape speed */
      mode_buffer[3] = 8;
      mode_buffer[4] = 0x14;    /* set density to 8200 mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* variable block length (0 bytes) */
      mode_buffer[10] = 0;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 30:               /* 8500 density - mode_select */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      mode_buffer[0] = 0;
      mode_buffer[1] = 0;
      mode_buffer[2] = 0x10;    /* buffer mode & tape speed */
      mode_buffer[3] = 8;
      mode_buffer[4] = 0;       /* set density to 8500 mode */
      mode_buffer[5] = 0;
      mode_buffer[6] = 0;
      mode_buffer[7] = 0;
      mode_buffer[8] = 0;
      mode_buffer[9] = 0;       /* variable block length (0 bytes) */
      mode_buffer[10] = 0;
      mode_buffer[11] = 0;

      ex_mode_select( scsi_id, 0x00, mode_buffer, 12, &scsi_status);
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    case 31:                 /* short erase */
    {
      if (number == ce_mode_value) ce_mode = 1;     /* enable hex dumps of the cdb */
      ex_erase( scsi_id, 0, &scsi_status);    /* if 1 is used, then a LONG erase (to PEOT) occurs */
      if (scsi_status) {
        display_sense_data(scsi_id);
        exitstat = 2;
      }
      break;
    }
    default: exitstat = 2;
  }
  ex_terminate( scsi_id, &d_status );
  if (d_status.all != status_$ok)
  {
    printf("Unable to release /dev/scsi1\n");
    pfm_$error_trap(d_status);
    exit(2);          /* command failed */
  }
  exit(exitstat);
}
