/*
Copyright (c) 1993 Regents of the University of Michigan.
All rights reserved.

Redistribution and use in source and binary forms are permitted provided
that this notice is preserved and that due credit is given to the University
of Michigan at Ann Arbor.  The name of the University may not be used to
endorse or promote products derived from this software without specific
prior written permission.  This software is provided ``as is'' without
express or implied warranty.

Written by Jim Rees.
*/

#include <sys/param.h>
#include <sys/reboot.h>
#include <sys/disklabel.h>

#include "stand.h"
#include "prom.h"

unsigned int bootdev;

static struct {
	char *name;
	ctype_t ctype;
} ctypes[] = {
	"wd",	ctype_winchester,
	"fd",	ctype_floppy,
	"dr",	ctype_ring_xmit,
	"sd",	ctype_smd,
	"ct",	ctype_cartridge,
	"en",	ctype_ether,
	NULL,	(ctype_t) 0
};

#define PAGESIZE 1024	/* How much you get from the prom */
#define DBPERPAGE (PAGESIZE / DEV_BSIZE)

static struct {
	char buf[2 * PAGESIZE], *page;
	short ppn;
} di;

struct pvl {
	u_short version;
	char apolloname[6];	/* "APOLLO" */
	char name[32];
	long id[2];		/* aegis: volume uid */
	u_short pad0;
	u_short dtype;
	long xbperpv;		/* blocks per physical volume (minus diag, bad) */
	u_short spertr;		/* sectors per track */
	u_short trpercyl;	/* tracks per cylinder */
	daddr_t lvoff[10];
	daddr_t altlv[10];
	daddr_t bad;
	daddr_t diag;
	u_short sector_start;
	u_short sector_size;
	u_short pre_comp;
	char pad1[6];
	long bperpv;		/* blocks per physical volume */
	daddr_t boot;
};

promopen(f, adapt, ctlr, part)
struct open_file *f;
int adapt, ctlr, part;
{
	u_int dev;
	static u_int lastdev;
	int i;
	char blkhdr[32];
	struct pvl *pvlp;
	long st;

	if (!lastdev)
		lastdev = bootdev;

	/* If device has changed since last time, must call init_disk */

	dev = (u_int) f->f_devdata;
	if (B_TYPE(dev) != B_TYPE(lastdev) || B_UNIT(dev) != B_UNIT(lastdev)) {
		for (i = 0; ctypes[i].name; i++)
			if (!strcmp(ctypes[i].name, f->f_dev->dv_name))
				break;
		if (!ctypes[i].name) {
			printf("no dev %s\n", f->f_dev->dv_name);
			return ENODEV;
		}
		init_disk(ctypes[i].ctype, B_UNIT(dev), &st);
		if (st != 0) {
			prstatus("prom init", st);
			return EIO;
		}
		lastdev = dev;
	}

	if (di.page == NULL) {
		/* The prom disk driver insists on reading into aligned pages.
		   This gives us the address of a page somewhere inside of diskbuf. */
		di.page = (char *) ((int) di.buf + 1023 & ~0x3ff);
		di.ppn = (int) di.page >> 10;
	}
#ifdef AEGIS_COMPAT
	if (part > 0 && part < 10) {
		/* Read pv label */
		read_disk(0, di.ppn, blkhdr, &st);
		pvlp = (struct pvl *) di.page;
		partoff = pvlp->lvoff[part] * DBPERPAGE;
	}
#endif
	return 0;
}

promstrategy(ss, func, dblk, size, buf, rsize)
struct sd_softc *ss;
int func;
daddr_t dblk;
u_int size;
char *buf;
u_int *rsize;
{
	daddr_t bn;
	int cc = size;
	char *ma = buf;
	int count;
	char blkhdr[32], *bhp;
	long st;

	if (size == 0)
		return(0);

	if (func == F_WRITE)
		bzero(blkhdr, sizeof blkhdr);

	bn = dblk / DBPERPAGE;
	bhp = blkhdr;

	while (cc > 0) {
		count = (cc < PAGESIZE) ? cc : PAGESIZE;
		if (func == F_READ) {
			read_disk(bn, di.ppn, bhp, &st);
			bcopy(di.page, ma, count);
		} else if (func == F_WRITE) {
			bcopy(ma, di.page, count);
			write_disk(bn, di.ppn, bhp, &st);
		}
		if (st != 0) {
			prstatus("prom read", st);
			return EIO;
		}
		bn++;
		cc -= count;
		ma += count;
	}
	*rsize = size;
	return 0;
}

char *
ctype_to_name(ctype)
ctype_t ctype;
{
	int i;

	for (i = 0; ctypes[i].name; i++)
		if (ctypes[i].ctype == ctype)
			break;
	return (ctypes[i].name);
}

/*
 * Print a character on console.
 */
void
putchar(c)
int c;
{
	if (c == '\0')
		return;
	if (c == '\n')
		putchar('\r');
	put_char(c);
}

getchar()
{
	int c;

	c = get_char();
	if (c == 0x1b)
		_rtt();
	if (c == '\r')
		c = '\n';
	return c;
}

static struct {
	int status;
	char *msg;
} promstatus[] = {
	0x12, "controller timeout",
	0x13, "illegal controller command",
	0x15, "dma memory parity",
	0x16, "dma overrun",
	0x17, "dma not at end of range",
	0x1d, "tape busy",
	0x1e, "disk busy",
	0x1f, "controller busy",
	0x30, "illegal drive command",
	0x31, "file mark",
	0x32, "bom",
	0x33, "end of tape on read",
	0x34, "no data",
	0x35, "filler",
	0x36, "bad block",
	0x37, "read abort",
	0x38, "end of tape",
	0x39, "no drive",
	0x3a, "no tape in drive",
	-1, NULL
};

prstatus(s, st)
char *s;
long st;
{
	int i;
	char *msg = NULL;

	st >>= 24;
	for (i = 0; promstatus[i].status != -1; i++)
		if (promstatus[i].status == st)
			msg = promstatus[i].msg;
	if (msg != NULL)
		printf("%s: %s\n", s, msg);
	else
		printf("%s: %x\n", s, st);
}
