/*
    This library requires that the main program be linked with a 
    large STACK size - otherwise it will die                     */
    
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>

#include "ParseLib.h"
 
#ifdef _DOS
   #define FILETEXT    ".\\_$/"       /* added / for testing */
   #ifndef MALLOC_DEF_FLAG
      #define MALLOC_DEF_FLAG
      #define MALLOC(X)   _fmalloc(X)
      #define FREE(X)     _ffree(X)
   #endif
 
#else
   #define FILETEXT    "._-+~/$"
   #ifndef MALLOC_DEF_FLAG
      #define MALLOC_DEF_FLAG
      #define MALLOC(X)   malloc(X)
      #define FREE(X)     free(X)
   #endif
#endif

Parse_input_ptr_t        Parse_input_list               ;
Parse_input_ptr_t        Parse_input_pntr               ;
Parse_input_ptr_t        Parse_input_buff               ;
Parse_input_ptr_t        Parse_input_last               ;
Parse_input_ptr_t        Parse_input_walk               ;

short                    Parse_lineno                   ;
short                    Parse_posit                    ;
short                    Parse_blanks                   ;
short                    Parse_trails                   ;
Boolean_t                Parse_Error                    ;
Boolean_t                Parse_int_flag                 ;
Boolean_t                Parse_text_flag                ;
Boolean_t                Parse_blank_flag               ;
Boolean_t                Parse_string_flag              ;
Boolean_t                Parse_dquote_flag              ;
Boolean_t                Parse_squote_flag              ;
Boolean_t                Parse_comment_flag             ;
Boolean_t                Parse_special_flag             ;
Boolean_t                Parse_endline_flag             ;
Boolean_t                Parse_endfile_flag             ;
Boolean_t                Parse_begfile_flag             ;

char                     Parse_message[512]             ;

int                    (*Parse_error_routine)(Boolean_t, char *) = NULL  ;

Parse_simple_ptr_t       Parse_simple_point = NULL ;
short                    Parse_simple_count = 0    ;


short                 Parse_index(char *string, char search)
{
   short          i     ;
   char           char1 ;
 
   for( i=0 ; (char1 = string[i]) ; i++ )
   {
      if( char1 == search )
         return i + 1 ;
   }
   return 0 ;
}
 

Parse_argument_ptr_t  Parse_argument(Parse_input_ptr_t *pointer)
{
   /* the purpose of this routine is to return an ARGUMENT where
      ARGUMENT is defined as

      1) a QUOTED string
      2) anything else up to a standard delimited:
         spaces, end of line, end of file. eg. 125/10 would
         be returned as one argument.
   */

   char             *string = NULL  ;
   int               strlen = 0     ;

   Boolean_t         first  = TRUE  ;
 
   static Parse_argument_t  argument ;

   if( *pointer == NULL )                                                     /* if points to nowhere               */
     return NULL ;                                                            /* return nothing                     */

   Parse_input_pntr = NULL     ;                                              /* set last position to NULL          */
   Parse_input_walk = *pointer ;                                              /* set where I am walking too         */

   while( Parse_input_walk != NULL )                                          /* as long as we have somewhere to go */
   {
      if( Parse_input_walk->endtype == PARSE_END_ENDFILE )                    /* if we got an endfile               */
         return NULL ;                                                        /* return off to nowhere              */

      if( Parse_input_walk->blanks != 0 )                                     /* we found something with blanks     */
      {
         if( !first )                                                         /* if this is not the first time      */
            break ;                                                           /* break out of loop                  */
      }
 
      if( first )                                                             /* first time through?                */
      {
         argument.lineno = Parse_input_walk->lineno ;                         /* set my starting lineno             */
         argument.posit  = Parse_input_walk->posit  ;                         /* set my starting character posit    */
      }

      if     ( Parse_input_walk->toktype == PARSE_TOK_STRING )                /* we have something in quotes?       */
      {
         if( first )                                                          /* if this is the first time          */
         {
            strlen           = strlen + Parse_input_walk->strlen ;            /* its okay, add string to length     */
            Parse_input_pntr = Parse_input_walk                  ;            /* set my end pointer to here         */
         }

         break ;                                                              /* break out of loop in any case      */
      }

      else if( Parse_input_walk->toktype == PARSE_TOK_TEXT    )               /* we have something in quotes?       */
      {
         strlen           = strlen + Parse_input_walk->strlen ;               /* its okay, add string to length     */
         Parse_input_pntr = Parse_input_walk                  ;               /* set my end pointer to here         */
      }

      else if( Parse_input_walk->toktype == PARSE_TOK_INTEGER )               /* we have an integer?                */
      {
         strlen           = strlen + Parse_input_walk->strlen ;               /* its okay, add string to length     */
         Parse_input_pntr = Parse_input_walk                  ;               /* set my end pointer to here         */
      }

      else if( Parse_input_walk->toktype == PARSE_TOK_SPECIAL )               /* we have something special          */
      {
         if( strpbrk(Parse_input_walk->strptr, "=,") != 0 )                   /* if we find these delimiters        */
         {
            if( !first )                                                      /* if not the first character         */
            {
               Parse_input_pntr = Parse_input_walk ;                          /* set my pointer here and break      */
               break ;                                                        /* break out of loop                  */
            }
 
            else                                                              /* not the first character            */
            {
               strlen++                            ;
               Parse_input_pntr = Parse_input_walk ;
               break                               ;
            }
         }
 
         else
         {
            strlen++                            ;                             /* just one character                 */
            Parse_input_pntr = Parse_input_walk ;                             /* set my end pointer to here         */
         }
      }

      else                                                                    /* a special, but not a good one      */
         break ;                                                              /* break                              */

      if( Parse_input_walk->endtype == PARSE_END_ENDLINE && strlen != 0)      /* is this the end of the line?       */
         break ;                                                              /* break out of loop in any case      */

      if( Parse_input_walk->endtype == PARSE_END_ENDFILE               )      /* is this the end of the line?       */
         break ;                                                              /* break out of loop in any case      */

      Parse_input_walk = Parse_input_walk->next ;                             /* set my pointer to next             */

      first            = FALSE                  ;                             /* not the first time through         */
   }

   if( strlen != 0 )                                                          /* we didnt get a string?             */
   {
       string = (char *) MALLOC( strlen + 2 ) ;                               /* allocate space                     */
       if( string == NULL )                                                   /* got a problem!                     */
          return( (Parse_argument_ptr_t) -1 ) ;                               /* return error                       */

       string[0] = '\0' ;                                                     /* clear up length                    */
   }


   /* lets walk through again, this time building string */
   /* -------------------------------------------------- */
 
   argument.strlen = strlen ;                                                 /* set the return length              */
   argument.strptr = string ;                                                 /* set the string location            */

   if( Parse_input_pntr != NULL)
   {
      Parse_input_walk = *pointer ;                                           /* set where I am walking too         */

      while( Parse_input_walk != NULL )                                       /* as long as we have somewhere to go */
      {
         if( Parse_input_walk->endtype == PARSE_END_ENDFILE )
            break ;

         else
         {
            strcat(string, Parse_input_walk->strptr) ;                        /* concat string                      */

            if( Parse_input_walk == Parse_input_pntr )                        /* did we hit the last one?           */
            {
               *pointer = Parse_input_walk->next ;                            /* set next position                  */
               return (&argument)                ;                            /* return the string position         */
            }

            else                                                              /* otherwise                          */
               Parse_input_walk = Parse_input_walk->next ;                    /* point to next                      */
         }
      }                                                                       /* keep looping                       */

      sprintf(Parse_message, "ERROR: Loop fall through in Parse_argument.") ; /* inform user of problem             */
      Parse_error(TRUE, Parse_message)                                      ; /* output to wherever                 */
      exit(1)                                                               ; /* stop run                           */
   }

   else
      return (&argument) ;
}


Parse_input_ptr_t  Parse_check(Parse_input_ptr_t *pointer)
{
   /* the purpose of this routine is to return the address of the current
      ITEM and bump the pointer to the next one                           */

   Parse_input_ptr_t     address = *pointer ;

   if     ( address == NULL )                                                   /* if points to nowhere               */
     return NULL ;                                                              /* return nothing                     */
 
   else if( address->endtype == PARSE_END_ENDFILE )                             /* if we got an endfile               */
      return NULL ;                                                             /* return off to nowhere              */
 
   else
   {
      *pointer = address->next  ;                                               /* set pointer to next                */
      return (address)          ;                                               /* return the address                 */
   }
}


Parse_argument_ptr_t  Parse_token(Parse_input_ptr_t *pointer)
{
   /* the purpose of this routine is to return the next Token */

   static Parse_argument_t     address ;

   if     ( *pointer == NULL )                                                  /* if points to nowhere               */
     return NULL ;                                                              /* return nothing                     */
 
   Parse_input_walk = *pointer ;                                                /* set the walk pointer               */
 
   if( Parse_input_walk->endtype == PARSE_END_ENDFILE )                         /* if we got an endfile               */
      return NULL ;                                                             /* return off to nowhere              */
 
   else
   {
      address.lineno = Parse_input_walk->lineno ;                               /* save lineno of this item           */
      address.posit  = Parse_input_walk->posit  ;                               /* save posit  of this item           */
      address.strlen = Parse_input_walk->strlen ;                               /* save strlen of this item           */
      address.strptr = Parse_input_walk->strptr ;                               /* save strptr of this item           */
 
      *pointer       = Parse_input_walk->next   ;                               /* set my pointer to the next         */
 
      return (&address)                         ;                               /* return the item pointer            */
   }
}


Parse_argument_ptr_t  Parse_filename(Parse_input_ptr_t *pointer)
{
   char             *string = NULL  ;
   int               strlen = 0     ;

   char              char1          ;
   Boolean_t         first  = TRUE  ;
 
   static Parse_argument_t  argument ;

   if( *pointer == NULL )                                                       /* if points to nowhere               */
     return NULL ;                                                              /* return nothing                     */

   Parse_input_pntr = NULL     ;                                                /* set last position to NULL          */
   Parse_input_walk = *pointer ;                                                /* set where I am walking too         */

   while( Parse_input_walk != NULL )                                            /* as long as we have somewhere to go */
   {
      if( Parse_input_walk->blanks  )                                           /* we found something with blanks     */
      {
         if( !first )                                                           /* if this is the first time          */
            break ;                                                             /* break out of loop                  */
      }
 
      if( first )                                                               /* first time through?                */
      {
         argument.lineno = Parse_input_walk->lineno ;                           /* set my starting lineno             */
         argument.posit  = Parse_input_walk->posit  ;                           /* set my starting character posit    */
      }

      if     ( Parse_input_walk->toktype == PARSE_TOK_STRING     )              /* we have something in quotes?       */
      {
         if( first )                                                            /* if this is the first time          */
         {
            strlen           = strlen + Parse_input_walk->strlen ;              /* its okay, add string to length     */
            Parse_input_pntr = Parse_input_walk                  ;              /* set my end pointer to here         */
         }

         break ;                                                                /* break out of loop in any case      */
      }

      else if( Parse_input_walk->toktype == PARSE_TOK_TEXT    ||                /* we have some text?                 */
               Parse_input_walk->toktype == PARSE_TOK_INTEGER    )              /* or an integer, both okay in context*/
      {
         strlen           = strlen + Parse_input_walk->strlen ;                 /* its okay, add string to length     */
         Parse_input_pntr = Parse_input_walk                  ;                 /* set my end pointer to here         */
      }

      else if( Parse_input_walk->toktype == PARSE_TOK_SPECIAL    )              /* we have something in quotes?       */
      {
         char1 = Parse_input_walk->strptr[0] ;                                  /* get this char                      */
 
         if( Parse_index(FILETEXT, char1) )
         {
            strlen++                            ;                               /* just one character                 */
            Parse_input_pntr = Parse_input_walk ;                               /* set my end pointer to here         */
         }

         else                                                                   /* a special, but not a good one      */
            break ;                                                             /* break                              */
      }

      if( Parse_input_walk->endtype == PARSE_END_ENDLINE && strlen != 0 )       /* is this the end of the line?       */
         break ;                                                                /* break out of loop in any case      */

      if( Parse_input_walk->endtype == PARSE_END_ENDFILE                )       /* is this the end of the line?       */
         break ;                                                                /* break out of loop in any case      */

      Parse_input_walk = Parse_input_walk->next ;                               /* set my pointer to next             */

      first            = FALSE                  ;                               /* not the first time through         */
   }

   if( strlen != 0 )                                                            /* we didnt get a string?             */
   {
       string = (char *) MALLOC( strlen + 2 ) ;                                 /* allocate space                     */
       if( string == NULL )                                                     /* got a problem!                     */
          return( (Parse_argument_ptr_t) -1 ) ;                                 /* return error                       */

       string[0] = '\0' ;                                                       /* clear up length                    */
   }


   /* lets walk through again, this time building string */
   /* -------------------------------------------------- */
 
   argument.strlen = strlen ;                                                   /* set the return length              */
   argument.strptr = string ;                                                   /* set the string location            */

   if( Parse_input_pntr != NULL)
   {
      Parse_input_walk = *pointer ;                                             /* set where I am walking too         */

      while( Parse_input_walk != NULL )                                         /* as long as we have somewhere to go */
      {
         if( Parse_input_walk->endtype == PARSE_END_ENDFILE )
            break ;

         else
         {
            strcat(string, Parse_input_walk->strptr) ;                          /* concat string                      */

            if( Parse_input_walk == Parse_input_pntr )                          /* did we hit the last one?           */
            {
               *pointer = Parse_input_walk->next ;                              /* set next position                  */
               return (&argument)                ;                              /* return the string position         */
            }

            else                                                                /* otherwise                          */
               Parse_input_walk = Parse_input_walk->next ;                      /* point to next                      */
         }
      }                                                                         /* keep looping                       */

      sprintf(Parse_message, "ERROR: Loop fall through in Parse_Filename.") ;   /* inform user of problem             */
      Parse_error(TRUE, Parse_message)                                      ;   /* output to wherever                 */
      exit(1)                                                               ;   /* stop run                           */
   }
}


void   Parse_release(Parse_input_ptr_t  pointer)
{
   Parse_input_pntr = pointer ;                                              /* set to first position               */

   while( Parse_input_pntr )                                                 /* while we have a pointer             */
   {
      Parse_input_last = Parse_input_pntr->next ;                            /* save the next location              */

      FREE(Parse_input_pntr->strptr) ;                                       /* clear this location->strptr         */
      FREE(Parse_input_pntr        ) ;                                       /* clear this location                 */

      Parse_input_pntr = Parse_input_last ;                                  /* reset the location                  */
   }
}


short  Parse_next(Parse_input_ptr_t *pointer)
{
   Parse_input_walk = *pointer ;

   if     ( Parse_input_walk          == NULL              )
      return 0 ;

   else if( Parse_input_walk->endtype == PARSE_END_ENDFILE )
      return 0 ;

   else if( Parse_input_walk->next    == NULL              )
      return 0 ;

   else
   {
      Parse_input_walk = Parse_input_walk->next ;
      if( Parse_input_walk->endtype == PARSE_END_ENDFILE )
         return 0 ;

      else
      {
         *pointer = Parse_input_walk ;
         return 1                    ;
      }
   }
}


int    Parse_save(char *string, short *strlen)
{

static unsigned long allocated = 0 ;
static          long count     = 0 ;

   if( *strlen == 0 )
      return(0) ;
 
   Parse_input_buff = (Parse_input_ptr_t) MALLOC( sizeof(Parse_input_t) ) ;
   if( Parse_input_buff == 0 )
      return(-1) ;
 
   Parse_input_buff->strptr = (char *) MALLOC( *strlen + 2 ) ;
   if( Parse_input_buff->strptr == 0 )
      return(-1) ;
 
   strcpy(Parse_input_buff->strptr, string) ;
 
   Parse_input_buff->strlen  = *strlen            ;
   Parse_input_buff->lineno  = Parse_lineno       ;
   Parse_input_buff->posit   = Parse_posit        ;
   Parse_input_buff->blanks  = Parse_blanks       ;
   Parse_input_buff->trails  = 0                  ;
   Parse_input_buff->strtype = PARSE_STR_NQUOTE   ;
 
   if     ( Parse_text_flag    )
      Parse_input_buff->toktype  = PARSE_TOK_TEXT    ;

   else if( Parse_string_flag  )
   {
      Parse_input_buff->toktype  = PARSE_TOK_STRING  ;
      if( Parse_dquote_flag )
         Parse_input_buff->strtype = PARSE_STR_DQUOTE ;
 
      else
         Parse_input_buff->strtype = PARSE_STR_SQUOTE ;
   }

   else if( Parse_int_flag     )
      Parse_input_buff->toktype  = PARSE_TOK_INTEGER ;

   else
      Parse_input_buff->toktype  = PARSE_TOK_SPECIAL ;
 
   if     ( Parse_endline_flag )
   {
      Parse_input_buff->endtype  = PARSE_END_ENDLINE ;
      Parse_input_buff->trails   = Parse_trails      ;
   }

   else if( Parse_endfile_flag )
      Parse_input_buff->endtype  = PARSE_END_ENDFILE ;

   else
      Parse_input_buff->endtype  = PARSE_END_NOTEND  ;
 
   Parse_input_buff->begflag = Parse_begfile_flag ;

   Parse_input_buff->next    = NULL               ;
 
   if( Parse_input_list == NULL )
      Parse_input_list = Parse_input_buff ;

   else
      Parse_input_last->next = Parse_input_buff ;
 
   Parse_input_last   = Parse_input_buff ;
 
   *strlen            = 0     ;
   strcpy(string, "")         ;
 
   Parse_posit        = -1    ;
   Parse_blanks       = 0     ;
   Parse_int_flag     = FALSE ;
   Parse_text_flag    = FALSE ;
   Parse_blank_flag   = FALSE ;
   Parse_string_flag  = FALSE ;
   Parse_special_flag = FALSE ;
   Parse_endline_flag = FALSE ;
   Parse_endfile_flag = FALSE ;
   Parse_begfile_flag = FALSE ;

   return(1) ;
}


void   Parse_lower(char *string)
{
   while( *string )                                                       /* spin through string        */
   {
      if( *string >= 'A' && *string <= 'Z' )                              /* test case of character     */
         *string += ' ';                                                  /* change case if necessary   */

      string++ ;                                                          /* bump string pointer        */
   }
}


void   Parse_upper(char *string)
{
   while( *string )                                                       /* spin through string        */
   {
      if( *string >= 'a' && *string <= 'z' )                              /* test case of character     */
         *string -= ' ';                                                  /* change case if necessary   */

      string++ ;                                                          /* bump string pointer        */
   }
}


void   Parse_debug(Parse_input_ptr_t  pointer)
{
   Parse_input_ptr_t    address = pointer ;                                /* set to first position               */

   while( address )                                                        /* while we have a pointer             */
   {
      printf("%5d %3d %3d %d %2d %d %d %3d  %s\n",
             address->lineno  , address->posit   ,
             address->blanks  , address->toktype ,
             address->strtype ,
             address->begflag , address->endtype ,
             address->strlen  , address->strptr   ) ;

      address = address->next ;                                            /* reset to next location              */
   }
}


void   Parse_linklist_debug(Parse_string_list_ptr_t  linklist)
{
   while( linklist )
   {
      printf("Debug: %s", linklist->strptr) ;
 
      linklist = linklist->next ;
   }
}


void      Parse_error_install( int (*error_routine)(Boolean_t, char *) )
{
   Parse_error_routine = error_routine ;
}


void      Parse_error(Boolean_t fatal, char *string)
{
   if( Parse_error_routine )
      Parse_error_routine(fatal, string) ;
 
   else
      fprintf(stderr, "ParseLib: %s\n", string) ;
 
   if( fatal )
      exit(1) ;
}


Parse_input_ptr_t   Parse_file(FILE *finp, Parse_comment_t comments)
{
   char                    *strptr               ;
   char                     string1[2048]        ;
 
   Parse_string_list_ptr_t  linklist_strt = NULL ;
   Parse_string_list_ptr_t  linklist_last        ;
   Parse_string_list_ptr_t  linklist_work        ;
 
   Parse_input_ptr_t        return_value         ;
                           
   /* read the input file, saving each line */
   /* ------------------------------------- */

   while( TRUE )                                                       /* start spining outer loop      */
   {
      strptr = fgets(string1, sizeof(string1), finp) ;                 /* get the string                */
      if( strptr )                                                     /* if the string exists ...      */
      {
         linklist_work = MALLOC( sizeof(Parse_simple_t) )      ;
         if( linklist_work == NULL )
         {
            sprintf(Parse_message, "ERROR: Unable to allocate required storage.") ;
            Parse_error(FALSE, Parse_message)                                     ;
            Parse_release(Parse_input_list)                                       ;
            return(NULL)                                                          ;
         }
 
         linklist_work->strptr = MALLOC( strlen(strptr) + 2 )                     ;
         if( linklist_work->strptr == NULL )
         {
            sprintf(Parse_message, "ERROR: Unable to allocate required storage.") ;
            Parse_error(FALSE, Parse_message)                                     ;
            Parse_release(Parse_input_list)                                       ;
            return(NULL)                                                          ;
         }
 
         if( linklist_strt )
            linklist_last->next = linklist_work ;
 
         else
            linklist_strt       = linklist_work ;
 
         linklist_work->next    = NULL          ;
         linklist_last          = linklist_work ;
 
         strcpy(linklist_work->strptr, strptr)  ;
      }
 
      else
         break ;
   }
 
   return_value = Parse_linklist(linklist_strt, comments) ;
 
   Parse_linklist_release(linklist_strt) ;
 
   return( return_value ) ;
}
 
 
void                Parse_linklist_release(Parse_string_list_ptr_t linklist)
{
   Parse_string_list_ptr_t  linklist_next ;
   Parse_string_list_ptr_t  linklist_work ;
 
   linklist_work = linklist ;
   while( linklist_work )
   {
      if( linklist_work->strptr )
         FREE(linklist_work->strptr) ;
 
      linklist_next = linklist_work->next ;
 
      FREE(linklist_work)                 ;
 
      linklist_work = linklist_next       ;
   }
}
 

Parse_input_ptr_t   Parse_linklist(Parse_string_list_ptr_t linklist,
                                   Parse_comment_t         comments )
{
   /* note: this routine expects that the last input line looks like:

      text ....   <nl><eof>

      if this is not the case, not a <nl> before the <eof> changes
      WILL be necessary (see Parse_Next)
   */
 
   Parse_string_list_ptr_t  linklist_work              ;

   char                     string1 [1024]             ;
   short                    strlen1                    ;
   short                    strtmp1                    ;
   char                     string2 [1024]             ;
   short                    strlen2                    ;

   short                    comment_line               ;
   char                     comment_pasc               ;

   char                     char1                      ;

   short                    indx                       ;


   /* read the input file, parsing each line */
   /* -------------------------------------- */

   Parse_Error        = FALSE    ;                                     /* clear ERROR flag              */
   Parse_dquote_flag  = FALSE    ;                                     /* clear DOUBLE QUOTE flag       */
   Parse_squote_flag  = FALSE    ;                                     /* clear SINGLE QUOTE flag       */
   Parse_comment_flag = FALSE    ;                                     /* clear COMMENT flag            */
   Parse_endfile_flag = FALSE    ;                                     /* clear ENDFILE flag            */
   Parse_lineno       = 0        ;                                     /* clear line number             */
 
   Parse_input_last   = NULL     ;                                     /* need to reset for multi calls */
   Parse_input_list   = NULL     ;                                     /* need to reset for multi calls */
 
   linklist_work      = linklist ;                                     /* set the initial link list     */
   while( linklist_work )                                              /* start spining outer loop      */
   {
      strlen1            = 0     ;                                     /* clear length                  */
      strtmp1            = 0     ;                                     /* clear length                  */
      strlen2            = 0     ;                                     /* clear length                  */
      Parse_blanks       = 0     ;                                     /* clear length                  */
      Parse_trails       = 0     ;                                     /* clear length                  */

      Parse_int_flag     = FALSE ;                                     /* clear INT flag                */
      Parse_text_flag    = FALSE ;                                     /* clear TEXT flag               */
      Parse_blank_flag   = FALSE ;                                     /* clear BLANK flag              */
      Parse_string_flag  = FALSE ;                                     /* clear STRING flag             */
      Parse_special_flag = FALSE ;                                     /* clear SPECIAL flag            */
      Parse_endline_flag = FALSE ;                                     /* clear ENDLINE flag            */
      Parse_begfile_flag = TRUE  ;

      Parse_lineno++             ;                                     /* bump line count               */
 
      strcpy(string2, linklist_work->strptr) ;                         /* copy the string to work area  */
 
      linklist_work = linklist_work->next    ;                         /* set the work to the next      */
 
      strlen2       = strlen(string2) + 1    ;
 
      for( indx=0 ; indx < strlen2 ; indx++ )                          /* spin through loop             */
      {
         char1 = string2[indx] ;                                       /* get the char                  */
         if     ( char1 == '\n' )                                      /* if got a return               */
            break ;                                                    /* break out                     */
 
         else if( char1 == '\t' )                                      /* if it is a tab                */
            char1 = ' ' ;                                              /* convert it to a blank         */
 
         else if( char1 == '\0' )                                      /* if it is a ZERO               */
            break ;                                                    /* treat it like a CR            */
 
         string1[indx] = char1 ;                                       /* save the character            */
 
         if( char1 != ' ' )                                            /* if not a space                */
         {
            strtmp1      = indx + 1 ;                                  /* save the last non blank       */
            Parse_trails = 0        ;                                  /* clear trail count             */
         }
 
         else
            Parse_trails += 1       ;                                  /* bump trail counter            */
      }
 
      string1[strtmp1] = '\0'    ;                                     /* set the terminating blank     */
      strlen1          = strtmp1 ;                                     /* set the length                */
 
      strlen2          = 0       ;                                     /* clear the length of the string*/
      strcpy(string2, "")        ;                                     /* string2 is used internally    */
 
      Parse_posit      = -1      ;                                     /* unknown position              */

      /* break the line(s) apart */
      /* ----------------------- */
 
      indx = 0 ;                                                       /* point to beginning of line    */
      while( (char1 = string1[indx++]) )                               /* get each character            */
      {
         if( string1[indx] == '\0' )
            Parse_endline_flag = TRUE ;
 
         /* are we in a comment */
         /* ------------------- */

         if     ( Parse_comment_flag )                                 /* are we in a comment ?         */
         {
            if( comment_pasc )                                         /* is this a PASCAL type         */
            { 
               if( char1 == '}' )                                      /* close of comment              */
               {
                  Parse_comment_flag = FALSE ;                         /* clear comment flag            */
               }            
            }
            
            else                                                       /* otherwise, C type             */
            {
               if( char1 == '*' && string1[indx] == '/' )              /* close of the comment          */
               {
                  indx++                     ;                         /* bump past /                   */
                  Parse_comment_flag = FALSE ;                         /* clear comment flag            */
               }
            }
            continue ;                                                 /* keep going                    */
         }
 

         /* are we in a double quote string? */
         /* -------------------------------- */

         else if( Parse_dquote_flag )                                  /* if we are in a double quote   */
         {
            if     ( char1 == '"' )                                    /* did we found closing dquote ? */
            {
               if( Parse_save(string2, &strlen2) == -1 )               /* try to save the string        */
                  return ( (Parse_input_ptr_t) -1 ) ;

               Parse_dquote_flag = FALSE     ;                         /* clear the dquote flag         */
               Parse_string_flag = FALSE     ;                         /* clear the string flag         */
            }
 
            else if( char1 == '\\' )                                   /* if break character encountered*/
            {
               /* we have a 'break' character, is this a PASCAL type */
               
               if( comments.pastyp )                                   /* if this a pascal type run     */
                  string2[strlen2++] = char1           ;
                  
               else
               {                  
                  string2[strlen2++] = char1           ;               /* save the / character          */
                  string2[strlen2++] = string1[indx++] ;               /* and save the next character   */
               }
               
               string2[strlen2  ] = '\0' ;                             /* terminate the string          */
            }
 
            else                                                       /* found a standard character    */
            {
               string2[strlen2++] = char1 ;                            /* save the character            */
               string2[strlen2  ] = '\0'  ;                            /* place the closing 0           */
            }
            continue ;                                                 /* keep spinning                 */
         }


         /* are we in a single quote string? */
         /* -------------------------------- */

         else if( Parse_squote_flag )                                  /* if we are in a single quote   */
         {
            if     ( char1 == '\'' )                                   /* did we found closing dquote ? */
            {             
               if( Parse_save(string2, &strlen2) == -1 )               /* try to save the string        */
                  return ( (Parse_input_ptr_t) -1 ) ;

               Parse_squote_flag = FALSE     ;                         /* clear the dquote flag         */
               Parse_string_flag = FALSE     ;                         /* clear the string flag         */
            }
 
            else if( char1 == '\\' )                                   /* if break character encountered*/
                        { 
               /* we have a 'break' character, is this a PASCAL type */
               
               if( comments.pastyp )                                   /* if this a pascal type run     */
                  string2[strlen2++] = char1           ;
                  
               else
               {                  
                  string2[strlen2++] = char1           ;               /* save the / character          */
                  string2[strlen2++] = string1[indx++] ;               /* and save the next character   */
               }
               
               string2[strlen2  ] = '\0' ;                             /* terminate the string          */
            }
 
            else                                                       /* found a standard character    */
            {
               string2[strlen2++] = char1 ;                            /* save the character            */
               string2[strlen2  ] = '\0'  ;                            /* place the closing 0           */
            }
            continue ;                                                 /* keep spinning                 */
         }


         /* are we entering a single quote */
         /* ------------------------------ */

         else if( char1 == '\'' )                                      /* found a single quote          */
         {
            Parse_posit       = indx ;                                 /* position where quote starts   */
            Parse_squote_flag = TRUE ;                                 /* set the flag                  */
            Parse_string_flag = TRUE ;                                 /* set string flag               */
            continue                 ;                                 /* continue                      */
         }


         /* are we entering a double quote */
         /* ------------------------------ */

         else if ( char1 == '"' )                                      /* found a double quote          */
         {
            Parse_posit       = indx ;                                 /* position where quote starts   */
            Parse_dquote_flag = TRUE ;                                 /* set the flag                  */
            Parse_string_flag = TRUE ;                                 /* set string flag               */
            continue                 ;                                 /* continue                      */
         }
 
 
         /* are we going into a # comment */
 
         else if( comments.pound && char1 == '#' )                     /* are we starting a # comment   */
            break ;
 
 
         /* are we going into a // comment */
 
         else if( comments.slashd && char1 == '/' && string1[indx] == '/' ) /* are we starting a comment*/ 
            break ;


         /* are we going into a / * comment */
         /* ------------------------------- */

         else if( comments.slashs && char1 == '/' && string1[indx] == '*' ) /* are we starting a comment*/
         {
            indx++                            ;                        /* bump past *                   */
            Parse_comment_flag = TRUE         ;                        /* set the flag                  */
            comment_line       = Parse_lineno ;                        /* save the line number          */
            comment_pasc       = FALSE        ;                        /* NOT a pascal type             */
            continue                          ;                        /* continue in the loop          */
         } 
         
         
         /* are we going into a pascal { comment */
         /* ------------------------------------ */
         
         else if( comments.pastyp && char1 == '{' )                       
         {
            Parse_comment_flag = TRUE         ;                        /* set the flag                  */
            comment_line       = Parse_lineno ;                        /* save the line number          */
            comment_pasc       = TRUE         ;                        /* a pascal type                 */
            continue                          ;                        /* continue in the loop          */
         } 

         /* are we going into a blank */
         /* ------------------------- */

         else if( char1 == ' ' )                                       /* is this a blank string?       */
         {
            Parse_blanks      = 0     ;                                /* clear blank count             */
            indx--                    ;                                /* backup one char               */

            while( (char1 = string1[indx++]) )                         /* get each character            */
            {
               if( char1 == ' ' )                                      /* if this char is a blank       */
                  Parse_blanks++ ;                                     /* bump the blank count          */

               else                                                    /* otherwise,                    */
                  break ;                                              /* break out of loop             */
            }

            if( char1 == '\0' )                                        /* if we hit end of line,        */
               break ;                                                 /* break out of loop             */
          
            else                                                       /* otherwise                     */
            {
               indx--   ;                                              /* backup one char               */
               continue ;                                              /* keep going                    */
            }
         }


         /* are we going into a string */
         /* -------------------------  */

         else if( (char1 >= 'a' && char1 <= 'z') ||
                  (char1 >= 'A' && char1 <= 'Z')   )
         {
            strlen2         = 0    ;                                   /* clear strlen2                 */
            Parse_posit     = indx ;                                   /* position where string starts  */
            Parse_text_flag = TRUE ;                                   /* set TEXT flag                 */
            indx--                 ;                                   /* backup one char               */

            while( (char1 = string1[indx++]) )                         /* get each character            */
            {
               if( (char1 >= 'a' && char1 <= 'z') ||                   /* how does the rest of          */
                   (char1 >= 'A' && char1 <= 'Z') ||                   /* the input                     */
                   (char1 >= '0' && char1 <= '9') ||                   /* look?                         */
                    char1 == '_'                    )                  /*                               */
               {
                  string2[strlen2++] = char1 ;
                  string2[strlen2  ] = '\0'  ;
               }

               else
                  break ;
            }

            if( char1 == '\0' )                                        /* if we hit end of line,        */
            {
               Parse_endline_flag = TRUE     ;                         /* flag as end of line           */
               if( Parse_save(string2, &strlen2) == -1 )
                  return ( (Parse_input_ptr_t) -1 ) ;

               break                         ;                         /* break out of loop             */
            }

            else                                                       /* otherwise                     */
            {
               if( Parse_save(string2, &strlen2) == -1 )
                  return ( (Parse_input_ptr_t) -1 ) ;

               indx--                        ;                         /* backup one char               */
               continue                      ;                         /* keep going                    */
            }
         }


         /* are we going into a integer */
         /* --------------------------  */

         else if( char1 >= '0' && char1 <= '9' )
         {
            strlen2         = 0    ;                                   /* clear strlen2                 */
            Parse_posit     = indx ;                                   /* position where string starts  */
            Parse_int_flag  = TRUE ;                                   /* set INT flag                  */
            indx--                 ;                                   /* backup one char               */

            while( (char1 = string1[indx++]) )                         /* get each character            */
            {
               if( (char1 >= '0' && char1 <= '9') )                    /* how does the rest look        */
               {
                  string2[strlen2++] = char1 ;
                  string2[strlen2  ] = '\0'  ;
               }

               else
                  break ;
            }

            if( char1 == '\0' )                                        /* if we hit end of line,        */
            {
               Parse_endline_flag = TRUE     ;                         /* flag as end of line           */
               if( Parse_save(string2, &strlen2) == -1 )
                  return ( (Parse_input_ptr_t) -1 ) ;

               break                         ;                         /* break out of loop             */
            }

            else                                                       /* otherwise                     */
            {
               if( Parse_save(string2, &strlen2) == -1 )
                  return ( (Parse_input_ptr_t) -1 ) ;

               indx--                        ;                         /* backup one char               */
               continue                      ;                         /* keep going                    */
            }
         }


         /* how about a signed integer? */
         /* --------------------------- */

         else if( char1 == '-' || char1 == '+'  )
         {
            Parse_posit       = indx   ;                               /* position where string starts  */
            strlen2            = 0     ;                               /* clear strlen2                 */
            string2[strlen2++] = char1 ;                               /* save this char                */
            string2[strlen2  ] = '\0'  ;                               /* set next to zero              */

            while( (char1 = string1[indx++]) )                         /* get each character            */
            {
               if( (char1 >= '0' && char1 <= '9') )                    /* how does the rest look        */
               {
                  string2[strlen2++] = char1 ;
                  string2[strlen2  ] = '\0'  ;
               }

               else
                  break ;
            }

            if( strlen2 == 1)                                          /* if just one char              */
               Parse_special_flag = TRUE ;                             /* it is a special character     */

            else                                                       /* otherwise                     */
               Parse_int_flag     = TRUE ;                             /* it is an integer              */

            if( char1 == '\0' )                                        /* if we hit end of line,        */
            {
               Parse_endline_flag = TRUE     ;                         /* flag as end of line           */
 
               if( Parse_save(string2, &strlen2) == -1 )
                  return ( (Parse_input_ptr_t) -1 ) ;

               break                         ;                         /* break out of loop             */
            }

            else                                                       /* otherwise                     */
            {
               if( Parse_save(string2, &strlen2) == -1 )
                  return ( (Parse_input_ptr_t) -1 ) ;

               indx--                        ;                         /* backup one char               */
               continue                      ;                         /* keep going                    */
            }
         }


         /* everyone else is a special character */
         /* ------------------------------------ */

         else
         {
            Parse_posit        = indx     ;                            /* position where string starts  */
            strlen2            = 0        ;                            /* clear strlen2                 */
            string2[strlen2++] = char1    ;                            /* save this char                */
            string2[strlen2  ] = '\0'     ;                            /* set next to zero              */
            Parse_special_flag = TRUE     ;                            /* set special flag              */
            if( Parse_save(string2, &strlen2) == -1 )
               return ( (Parse_input_ptr_t) -1 ) ;
         }
      }
      
      if( Parse_input_last )                                           /* if last exists                */
         Parse_input_last->endtype = PARSE_END_ENDLINE ;               /* set prev field as end of line */                                                                /* we are finished file          */

      if( Parse_dquote_flag || Parse_squote_flag )
      {
         sprintf(Parse_message, "ERROR: Unterminated quoted text started on line %d.",
                 Parse_lineno                                                         ) ;
         Parse_error(FALSE, Parse_message)                                              ;
         Parse_release(Parse_input_list)                                                ;
         return(NULL)                                                                   ;
      }
   }                                                                   /* we are finished file          */

   if     ( Parse_comment_flag )                                       /* are we still in a comment     */
   {
      sprintf(Parse_message, "ERROR: Unterminated comment line started on line %d.", comment_line) ;
      Parse_error(FALSE, Parse_message)                                                            ;
      Parse_release(Parse_input_list)                                                              ;
      return(NULL)                                                                                 ;
   }

   Parse_endfile_flag = TRUE     ;
   string2[0]         = ' '      ;
   string2[1]         = '\0'     ;
   strlen2            =  1       ;
 
   if( Parse_save(string2, &strlen2) == -1 )
      return ( (Parse_input_ptr_t) -1 ) ;

   return(Parse_input_list) ;
}


void                 Parse_simple_release()
{
   short i ;
 
   if( Parse_simple_point )
   {
      for( i=0 ; i < Parse_simple_count + 1 ; i++ )
      {
         if( Parse_simple_point[i].string )
            FREE( Parse_simple_point[i].string ) ;
      }
 
      FREE(Parse_simple_point)  ;

      Parse_simple_point = NULL ;
      Parse_simple_count = 0    ;
   }
}


Parse_simple_ptr_t   Parse_simple_alloc(short  count)
{
   Parse_simple_ptr_t     point ;
   short                  i     ;

   if( Parse_simple_point )
      Parse_simple_release() ;
 
   point = (Parse_simple_ptr_t) MALLOC( (count + 1) * sizeof(Parse_simple_t)) ;

   if( point == NULL )
      return (Parse_simple_ptr_t) (-1) ;

   else
   {
      for( i=0 ; i < count + 1 ; i++ )
      {
         point[i].search  = NULL           ;
         point[i].string  = NULL           ;
         point[i].convert = PARSE_CVT_NONE ;
      }

      Parse_simple_point = point ;
      Parse_simple_count = count ;

      return point ;
   }
}


Boolean_t        Parse_strcmp(char *string1, char *string2, Boolean_t insens)
{
   char              char1     ;
   char              char2     ;
   short             i     = 0 ;

   while( TRUE )
   {
      char1 = string1[i  ] ;
      char2 = string2[i++] ;
 
      if     ( char1 == '\0' && char1 == '\0' )
         return TRUE ;
 
      else if( char1 == '\0' || char1 == '\0' )
         return FALSE ;
 
      if( insens )
      {
         if( char1 >= 'a' && char1 <= 'z' ) char1 -= ' ' ;
         if( char2 >= 'a' && char2 <= 'z' ) char2 -= ' ' ;
      }
 
      if( char1 != char2 )
         return FALSE ;
   }
}


Boolean_t        Parse_simple_parse(Parse_simple_ptr_t  point,
                                    Parse_input_ptr_t   input )
{
   Parse_simple_ptr_t    sim_walk = point ;
   Parse_input_ptr_t     inp_walk = input ;
 
   Parse_argument_ptr_t  address          ;
 
   Boolean_t             error    = FALSE ;
 
   Boolean_t             found    = FALSE ;

   char                 *string           ;
   short                 strlen           ;
   short                 lineno           ;
   short                 posit            ;
 
   short                 i                ;

   /* clear the found flags for all the inputs */

   while( sim_walk->search )
   {
      sim_walk->found = FALSE ;
      sim_walk++              ;
   }

   /* check to verify that we point to input */

   if( inp_walk == NULL )
      return FALSE ;

   /* clear all the USED flags within the input record */

   while( inp_walk )
   {
      inp_walk->used = FALSE          ;
      inp_walk       = inp_walk->next ;
   }


   /* reset the flags that we changed */

   inp_walk = input ;

   while( TRUE )
   {
      if( (address = Parse_token(&inp_walk)) == NULL )
         break ;

      else
      {
         sim_walk = point           ;                           /* set the walk point           */
         string   = address->strptr ;                           /* copy the string pointer      */
         lineno   = address->lineno ;
         posit    = address->posit  ;
         found    = FALSE           ;                           /* clear found flag             */
 
         while( sim_walk->search )                              /* while we are still pointing  */
         {
            if( Parse_strcmp(string, sim_walk->search, sim_walk->insens) )
            {
               found = TRUE ;                                   /* flag as found                */
 
               if( sim_walk->found )                            /* have we already found it?    */
               {
                  sprintf(Parse_message,
                          "ERROR: Duplicate Setting found for '%s' at line %d, position %d.",
                          string, lineno, posit                                              ) ;
                  Parse_error(FALSE, Parse_message)                                            ;
                  error = TRUE                                                                 ;
               }
 
               /* check if the argument is followed by an = */
 
               if( (address = Parse_argument(&inp_walk)) == NULL )
               {
                  sprintf(Parse_message, "ERROR: Unexpected <END_OF_FILE> encountered.") ;
                  Parse_error(FALSE, Parse_message)                                      ;
                  error = TRUE                                                           ;
                  break                                                                  ;
               }
 
 
               /* I know know why this is but if I used address->strptr where I
                  use string, I get unpredictable results ???                   */
 
               string = address->strptr ;
               strlen = address->strlen ;
               lineno = address->lineno ;
               posit  = address->posit  ;
 
               if( strcmp(string, "=") != 0 )                   /* should be an equal           */
               {
                  sprintf(Parse_message,
                          "ERROR: Expected a '=', but found '%s' at line %d, position %d.",
                          string, lineno, posit                                            ) ;
                  Parse_error(FALSE, Parse_message)                                          ;
                  error = TRUE                                                               ;
                  break                                                                      ;
               }
 
               if( (address = Parse_argument(&inp_walk)) == NULL )
               {
                  sprintf(Parse_message, "ERROR: Unexpected <END_OF_FILE> encountered.") ;
                  Parse_error(FALSE, Parse_message)                                      ;
                  error = TRUE                                                           ;
                  break                                                                  ;
               }
 
               /* verify that the field exists */
 
               string = address->strptr ;
               strlen = address->strlen ;
               lineno = address->lineno ;
               posit  = address->posit  ;
 
               switch( sim_walk->inptype )
               {
                  case  PARSE_INP_TEXT     : if( *string == '\0' )
                                             {
                                                sprintf(Parse_message, "%s '%s' at line %d, position %d.",
                                                        "ERROR: Missing Text found for field",
                                                        sim_walk->search, lineno, posit                   ) ;
                                                Parse_error(FALSE, Parse_message)                           ;
                                                error = TRUE                                                ;
                                             }
                                             break ;
 
 
                  case  PARSE_INP_STRING   : if( *string == '\0' )
                                             {
                                                sprintf(Parse_message, "%s '%s' at line %d, position %d.",
                                                        "ERROR: Missing String found for field",
                                                        sim_walk->search, lineno, posit                   ) ;
                                                Parse_error(FALSE, Parse_message)                           ;
                                                error = TRUE                                                ;
                                             }
                                             break ;
 
 
                  case  PARSE_INP_INTEGER  : for( i=0, found=FALSE ; string[i] ; i++ )
                                             {
                                                if( string[i] >= '0' && string[i] <= '9' )
                                                   found = TRUE ;
 
                                                else
                                                {
                                                   sprintf(Parse_message, "%s '%s' at line %d, position %d.",
                                                           "ERROR: Illegal Integer found for field",
                                                           sim_walk->search, lineno, posit                   ) ;
                                                   Parse_error(FALSE, Parse_message)                           ;
                                                   error = TRUE                                                ;
                                                   break                                                       ;
                                                }
                                             }
 
                                             if( !found )
                                             {
                                                sprintf(Parse_message, "%s '%s' at line %d, position %d.",
                                                        "ERROR: Missing Integer found for field",
                                                        sim_walk->search, lineno, posit                   ) ;
                                                Parse_error(FALSE, Parse_message)                           ;
                                                error = TRUE                                                ;
                                             }
                                             break ;
 
 
                  case  PARSE_INP_FILENAME : for( i=0, found=FALSE ; string[i] ; i++ )
                                             {
                                                if( ( string[i] >= '0' && string[i] <= '9' ) ||
                                                    ( string[i] >= 'a' && string[i] <= 'z' ) ||
                                                    ( string[i] >= 'A' && string[i] <= 'Z' ) ||
                                                    Parse_index(FILETEXT, string[i])        )
                                                   found = TRUE ;
 
                                                else
                                                {
                                                   sprintf(Parse_message, "%s '%s' at line %d, position %d.",
                                                           "ERROR: Illegal Filename found for field",
                                                           sim_walk->search, lineno, posit                   ) ;
                                                   Parse_error(FALSE, Parse_message)                           ;
                                                   error = TRUE                                                ;
                                                   break                                                       ;
                                                }
                                             }
 
                                             if( !found )
                                             {
                                                sprintf(Parse_message, "%s '%s' at line %d, position %d.",
                                                        "ERROR: Missing Filename found for field",
                                                        sim_walk->search, lineno, posit                   ) ;
                                                Parse_error(FALSE, Parse_message)                           ;
                                                error = TRUE                                                ;
                                             }
                                             break ;
 
 
                  default                  : sprintf(Parse_message, "%s '%s'.",
                                                     "ERROR: Illegal Field Type for field",
                                                     sim_walk->search                       ) ;
                                             Parse_error(FALSE, Parse_message)                ;
                                             error = TRUE                                     ;
               }
 
               sim_walk->string  = string ;
               sim_walk->lineno  = lineno ;
               sim_walk->posit   = posit  ;
               sim_walk->found   = TRUE   ;
 
               if( sim_walk->convert && sim_walk->string )
               {
                  if     ( sim_walk->convert == PARSE_CVT_LOWER )
                     Parse_lower(sim_walk->string) ;
 
                  else if( sim_walk->convert == PARSE_CVT_UPPER )
                     Parse_upper(sim_walk->string) ;
               }
            }
            sim_walk = sim_walk + 1 ;                           /* walk to next in list         */
         }                                                      /* end of SIM_WALK loop         */
 
         if( !found )                                           /* if this key was not found    */
         {
            sprintf(Parse_message,
                    "ERROR: Unknown Keyword found '%s' at line %d, position %d.",
                    string, lineno, posit                                        ) ;
            Parse_error(FALSE, Parse_message)                                      ;
            error = TRUE                                                           ;
         }
      }
   }
 
 
   /* now to walk through the requests and see what was required but
      wasnt resolved                                                 */

   sim_walk = point ;                                           /* set the walk point           */
 
   while( sim_walk->search )                                    /* while we are still pointing  */
   {
      if( ( !sim_walk->found ) && sim_walk->required )
      {
         sprintf(Parse_message, "ERROR: Required Field '%s' was not found.",
                 sim_walk->search                                           ) ;
         Parse_error(FALSE, Parse_message)                                    ;
         error = TRUE                                                         ;
      }
 
      sim_walk = sim_walk + 1 ;
   }
 
   return error ;
}


short              Parse_reserved_word(char *string)
{
   /* This routine verifies that field names are not reserved words for C */
 
   static char  *rwords_c[ ]   = { "auto"    , "break"   , "case"   , "char"  , "const"     , "continue",
                                   "default" , "do"      , "double" , "else"  , "enum"      , "extern"  ,
                                   "float"   , "for"     , "goto"   , "if"    , "int"       , "long"    ,
                                   "register", "return"  , "short"  , "signed", "sizeof"    , "static"  ,
                                   "struct"  , "switch"  , "typedef", "union" , "unsigned"  , "void"    ,
                                   "volatile", "while"   , NULL                                           } ;
 
   static char  *rwords_cpp[ ] = { "asm"     , "auto"    ,"break"   , "case"     ,"catch"   , "char"    ,
                                   "class"   , "const"   ,"continue", "default"  ,"delete"  , "do"      ,
                                   "double"  , "else"    ,"enum"    , "extern"   ,"float"   , "for"     ,
                                   "friend"  , "goto"    ,"if"      , "inline"   ,"int"     , "long"    ,
                                   "new"     , "operator","private" , "protected","public"  , "register",
                                   "return"  , "short"   ,"signed"  , "sizeof"   ,"static"  , "stuct"   ,
                                   "switch"  , "template","this"    , "throw"    ,"try"     , "typedef" ,
                                   "union"   , "unsigned","virtual" , "void"     ,"volatile", "while"   ,
                                   NULL                                                                   } ;
 
   short        i                                                                                           ;
 
   for( i=0 ; *rwords_c[i] ; i++ )
   {
      if( strcmp(string, rwords_c[i]) == 0 )
         return 1 ;
   }
 
   for( i=0 ; *rwords_cpp[i] ; i++ )
   {
      if( strcmp(string, rwords_cpp[i]) == 0 )
         return 2 ;
   }
 
   return 0 ;
}
