/*
 * piodrv_call.c
 *   GPIO call-side routine(s) for the parallel port driver.
 *   Jef Rijmenants, 19-may-1993.
 */

#include <stdio.h>
#include <apollo/base.h>
#include <pbu.h>
#include "piodrv.h"

/*
 * The parallel port has three registers:
 *   1. the data register    (offset 00h, read-write)
 *   2. the status register  (offset 01h, read-only)
 *   3. the control register (offset 02h, read-write)
 * The base IO address is: 378h, 278h, 3bch or 2bch.
 */

typedef struct {
   unsigned char data_reg;
   unsigned char stat_reg;
   unsigned char ctrl_reg;
   unsigned char pad[bytes_per_page-3];
} device_t __attribute((__device));

static device_t *piodrv_csr_p;
pbu_$unit_t piodrv_unit ;  /* pbu unit number */


/* The status register bits are as follows:
 *
 *   BUSY_L ACK_L OUTPAP ONLINE ERR_L X X X
 *
 *   BUSY_L : (0=printer busy, 1=not busy)
 *   ACK_L  : (1=transfer busy, 0=transfer ended)
 *   OUTPAP : (1=out of paper)
 *   ONLINE : (1=printer on-line)
 *   ERR_L  : printer error (1=ok, 0=error)
 *   X      : unused (normally 1)
 */
/* status register bit masks */
#define PIODRV_BUSY_L 0x80
#define PIODRV_ACK_L  0x40
#define PIODRV_OUTPAP 0x20
#define PIODRV_ONLINE 0x10
#define PIODRV_ERR_L  0x08

/* The control register bits are as follows:
 *
 *   X X X INTRUP SELECT INIT_L AUTOLF STROBE
 *
 *   INTRUP : hardware interrupt when data transfer done (0=no interrupt)
 *   SELECT : device select (1=select, 0=unselect)
 *   INIT_L : device init (1=normal, 0=init)
 *   AUTOLF : auto linefeed (1=by printer, 0= by host)
 *   STROBE : strobe data transfer (1=transfer, 0=no transfer)
 *   X      : unused (normally 1)
 */
/* control register bit masks */
#define PIODRV_INTRUP 0x10
#define PIODRV_SELECT 0x08
#define PIODRV_INIT_L 0x04
#define PIODRV_AUTOLF 0x02
#define PIODRV_STROBE 0x01

#define PIODRV_STROBE_WAIT 2


/*
 * piodrv_init
 *   The GPIO driver init routine. This routines gets called when the
 *   device is acquired (with pbu_$acquire).
 */
void piodrv_init(
   pbu_$unit_t          &unit,
   pbu_$ddf_ptr_t       &ddf_ptr,
   pbu_$csr_page_ptr_t  &csr_ptr,
   status_$t            *sts
)
{
   int i;
   piodrv_csr_p = (device_t*)csr_ptr;

   /* check if parallel io port is present and initialized */
   if (piodrv_csr_p->ctrl_reg != 0xEC) {
      piodrv_reset();
      /* wait while printer resets */
      sleep(30); /* an HP550C takes about 25 seconds */
   }
   /* check status register */
   if (piodrv_csr_p->stat_reg != 0xDF) {
      fprintf(stderr, "ERROR: Device not present or inactive.\n");
      sts->s.fail = 1;
      return;
   }

   piodrv_csr_p->ctrl_reg = PIODRV_SELECT | PIODRV_INIT_L;
   piodrv_unit = unit ;
   sts->all = status_$ok ;
}


/*
 * piodrv_cleanup
 *   GPIO driver cleanup routine. This routine gets called when the device
 *   is released (using pbu_$release).
 */
void piodrv_cleanup(
   pbu_$unit_t &pbu_unit,
   boolean     &force_flag,
   status_$t   *sts
)
{
   piodrv_csr_p->ctrl_reg = PIODRV_SELECT | PIODRV_INIT_L;
}


/*
 * piodrv_reset
 *   reset printer
 */
int piodrv_reset()
{
   int wait;
   piodrv_csr_p->ctrl_reg = 0x00;
   for (wait = 0; wait < 1000; wait++);
   piodrv_csr_p->ctrl_reg = PIODRV_SELECT | PIODRV_INIT_L;
}


/*
 * int piodrv_put_char
 *   write one char to the parallel port (low-level routine)
 */
int piodrv_put_char(
   char c
)
{
   int wait;

   /* wait for not-busy */
/*
   for (wait = 0; wait < PIODRV_TIMEOUT; wait++) {
      if (piodrv_csr_p->stat_reg & PIODRV_BUSY_L) break;
   }
   if (wait == PIODRV_TIMEOUT) return 0;
*/
   while (!(piodrv_csr_p->stat_reg & PIODRV_BUSY_L));

   /* load data register */
   piodrv_csr_p->data_reg = c ;

   /* activate strobe */
   for (wait = 0; wait < PIODRV_STROBE_WAIT; wait++);
   piodrv_csr_p->ctrl_reg = PIODRV_SELECT | PIODRV_INIT_L | PIODRV_STROBE;

   /* deactivate strobe to transfer data */
   for (wait = 0; wait < PIODRV_STROBE_WAIT; wait++);
   piodrv_csr_p->ctrl_reg = PIODRV_SELECT | PIODRV_INIT_L;

   /* wait for ack */
/*
   for (wait = 0; wait < PIODRV_ACK_TIMEOUT; wait++) {
      if (!(piodrv_csr_p->stat_reg & PIODRV_ACK_L)) break;
   }
   if (wait == PIODRV_ACK_TIMEOUT) return 0;
*/

   return 1;
}


/*
 * piodrv_paper_empty
 */                                         
int piodrv_paper_empty()
{
   return piodrv_csr_p->stat_reg & PIODRV_OUTPAP;
}


