/*
 * pop.c
 *
 * This program simply connects to a POP3 server and sucks over
 * any new mail.
 */

#include <apollo/base.h>
#include <apollo/cal.h>
#include <apollo/error.h>
#include <apollo/ios.h>
#include <apollo/pgm.h>
#include <apollo/pm.h>
#include <apollo/time.h>
#include <apollo/type_uids.h>
#include <apollo/vfmt.h>

#include "cl.h"
#include "help.h"
#include "lib.h"
#include "pm.h"
#include "socket.h"
#include "strl.h"

#define string(x) (x), (sizeof(x)-1)
#define program string("pop")
#define version string("1.0")

boolean monitor, verbose, anything;
char hostname[ 64 ], username[ 64 ], password[ 64 ];
short hostname_len, username_len, password_len;
char prefix[ name_$long_pnamlen_max ];
short prefix_len;

typedef enum
{
    pop_$null,
    pop_$user,
    pop_$pass,
    pop_$quit,
    pop_$stat,
    pop_$list,
    pop_$listx,
    pop_$retr,
    pop_$dele,
    pop_$noop,
    pop_$last,
    pop_$rset,
    pop_$top,
    pop_$rpop
}
pop_$command_t;

char *pop_$command_text[] =
{
    "",                   /*    pop_$null,  */
    "USER %A%$",          /*    pop_$user,  */
    "PASS %A%$",          /*    pop_$pass,  */
    "QUIT%$",             /*    pop_$quit,  */
    "STAT%$",             /*    pop_$stat,  */
    "LIST %ULD%$",        /*    pop_$list,  */
    "LIST%$",             /*    pop_$listx, */
    "RETR %ULD%$",        /*    pop_$retr,  */
    "DELE %ULD%$",        /*    pop_$dele,  */
    "NOOP%$",             /*    pop_$noop,  */
    "LAST%$",             /*    pop_$last,  */
    "RSET%$",             /*    pop_$rset,  */
    "TOP %ULD %ULD%$",    /*    pop_$top,   */
    "RPOP %A%$"           /*    pop_$rpop   */
};

boolean pop_$command_multi[] =
{
    false,                /*    pop_$null,  */
    false,                /*    pop_$user,  */
    false,                /*    pop_$pass,  */
    false,                /*    pop_$quit,  */
    false,                /*    pop_$stat,  */
    false,                /*    pop_$list,  */
    true,                 /*    pop_$listx, */
    true,                 /*    pop_$retr,  */
    false,                /*    pop_$dele,  */
    false,                /*    pop_$noop,  */
    false,                /*    pop_$last,  */
    false,                /*    pop_$rset,  */
    true,                 /*    pop_$top,   */
    false                 /*    pop_$rpop   */
};

short pop_$resp_n[] =
{
    0,                    /*    pop_$null,  */
    0,                    /*    pop_$user,  */
    0,                    /*    pop_$pass,  */
    0,                    /*    pop_$quit,  */
    1,                    /*    pop_$stat,  */
    1,                    /*    pop_$list,  */
    2,                    /*    pop_$listx, */
    1,                    /*    pop_$retr,  */
    0,                    /*    pop_$dele,  */
    0,                    /*    pop_$noop,  */
    1,                    /*    pop_$last,  */
    0,                    /*    pop_$rset,  */
    0,                    /*    pop_$top,   */
    0                     /*    pop_$rpop   */
};


void
pop_$init
(
    void
);

ios_$id_t
pop_$connect
(
    void
);

void
pop_$do_args
(
    void
);

void
pop_$get_line
(
    ios_$id_t  s,
    char      *line,
    long      *line_len,
    long       line_max
);

void
pop_$put_line
(
    ios_$id_t  s,
    char      *line,
    long       line_len
);

boolean
pop_$get_response
(
    ios_$id_t  s,
    boolean    multiline,
    char      *data,
    long      *data_len,
    long       data_max
);

boolean
pop_$get_message
(
    ios_$id_t s,    /* socket */
    ios_$id_t f     /* file */
);

int
pop_$do_command
(
    ios_$id_t       s,
    pop_$command_t  command,
    void           *arg1,
    void           *arg2
);

void
pop_$process
(
    ios_$id_t s
);


void
pop_$init
(
    void
)
{
    status_$t status;
    ios_$id_t socket;

    help_$args(program, version);
    error_$init_std_format(ios_$errout, '?', program);
    pm_$set_my_name(program, &status);
    pop_$do_args();

    socket = pop_$connect();
    pop_$process(socket);

    pgm_$exit();
}


ios_$id_t
pop_$connect
(
    void
)
{
    status_$t status;
    socket_$addr_family_t family;
    socket_$addr_t address;
    unsigned long addr_len;
    ios_$id_t socket;
    socket_$address_t sa;
    short sa_len;

    /*
     * We default to internet address, but we'll take anything.
     * We tell the difference by looking for the colon.
     */

    if( strl_$index(hostname, hostname_len, string(":")) > 0 )
    {
        family = socket_$unspec;
    }
    else
    {
        family = socket_$internet;
    }

    socket_$from_name(family, hostname, hostname_len, 110, &address, &addr_len, &status);

    if( status.all != status_$ok )
    {
        error_$std_format(status, "Cannot find host \"%A\"%$", hostname, &hostname_len);
        pgm_$set_severity(pgm_$error);
        pgm_$exit();
    }

    socket_$create((socket_$domain_t)family, socket_$stream_type, socket_$unspec_protocol, &socket, &status);

    if( status.all != status_$ok )
    {
        error_$std_format(status, "Cannot create socket%$");
        pgm_$set_severity(pgm_$internal_fatal);
        pgm_$exit();
    }

    if( password_len == 0 )
    {
        /*
         * The user wants us to "rpop" in without a password.
         * This means we'll have to get a "priviliged" port.
         */

        socket_$address_t me;
        unsigned len, port;
        short slen;

        for( port = 511; port > 0; port-- )
        {
            len = sizeof me;
            me.family = (socket_$domain_t)family;
            lib_$data_zero(&me.data, socket_$sizeof_data);
            socket_$set_port((socket_$addr_t *)&me, &len, port, &status);

            if( status.all != status_$ok )
            {
                error_$std_format(status, "Cannot set port address %ULD%$", &port);
                pgm_$set_severity(pgm_$internal_fatal);
                pgm_$exit();
            }

            slen = len;
            socket_$bind(socket, me, slen, &status);

            if( status.all != status_$ok )
            {
                if( status.all == 0x090C0030 )
                {
                    continue;
                }

                error_$std_format(status, "Cannot bind local socket address%$");
                pgm_$set_severity(pgm_$internal_fatal);
                pgm_$exit();
            }
        }

        if( port == 0 )
        {
            status.all = 0x090C000B;
            error_$std_format(status, "No available priviliged ports%$");
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }
    }

    sa = *(socket_$address_t *)&address;
    sa_len = sizeof sa;

    socket_$connect(socket, sa, sa_len, &status);

    if( status.all != status_$ok )
    {
        error_$std_format(status, "Cannot connect to %A%$", hostname, &hostname_len);
        pgm_$set_severity(pgm_$error);
        pgm_$exit();
    }

    return socket;
}

void
pop_$do_args
(
    void
)
{
    status_$t status;
    short i;

    cl_$init(cl_$std_options, program);

    if( cl_$check_flag("-host[id]", 1) )
    {
        cl_$get_arg(cl_$next, hostname, &hostname_len, sizeof hostname);
    }
    else
    {
        strl_$copy(string("pop"), hostname, &hostname_len, sizeof hostname);
    }

    if( cl_$check_flag("-user[name]", 1) )
    {
        cl_$get_arg(cl_$next, username, &username_len, sizeof username);
    }
    else
    {
        short index;

        pm_$get_sid_txt(sizeof username, username, &username_len);
        index = strl_$index(username, username_len, string("."));

        if( index > 1 )
        {
            username_len = index - 1;
        }
    }

    if( cl_$check_flag("-pass[word]", 1) )
    {
        cl_$get_arg(cl_$next, password, &password_len, sizeof password);
    }
    else
    {
        password_len = 0;
    }

    if( cl_$check_flag("-pre[fix]", 1) )
    {
        name_$long_pname_t p;
        short p_len;

        cl_$get_arg(cl_$next, p, &p_len, sizeof p);
        strl_$concat(p, p_len, string(".%ULD%$"), prefix, &prefix_len, sizeof prefix);
    }
    else
    {
        time_$clock_t clock_value;
        cal_$timedate_rec_t date_time;
        name_$long_pname_t p;
        short p_len;

        cal_$get_local_time(&clock_value);
        cal_$decode_time(clock_value, &date_time);

        vfmt_$encode("./newmail.%4ZWD.%5(%2ZWD.%)%$", p, sizeof p, &p_len, 
            &date_time.year, &date_time.month, &date_time.day,
            &date_time.hour, &date_time.minute, &date_time.second);

        strl_$concat(p, p_len, string("%ULD%$"), prefix, &prefix_len, sizeof prefix);
    }

    monitor = cl_$get_flag("-mon[itor]", &i);
    verbose = cl_$get_flag("-ver[bose]", &i);

    cl_$check_unclaimed();

    return;
}

void
pop_$get_line
(
    ios_$id_t  s,
    char      *line,
    long      *line_len,
    long       line_max
)
{
    status_$t status;
    char *p;
    short plen;

    p = line;
    plen = 0;

    while( 1 )
    {
        long new;
        short end;

        new = ios_$get(s, ios_$preview_opt, p, line_max - plen, &status);

        if( status.all != status_$ok )
        {
            error_$std_format(status, "Cannot preview data from the server%$");
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }

        end = strl_$index(line, plen + new, string("\r\n"));

        if( end > 0 && end < (plen + new) )
        {
            /* we got one! */
            ios_$get(s, ios_$no_put_get_opts, p, end - plen + 1, &status);

            if( status.all != status_$ok )
            {
                error_$std_format(status, "Cannot read a line from the server%$");
                pgm_$set_severity(pgm_$error);
                pgm_$exit();
            }

            *line_len = end - 1;
            return;
        }

        new = ios_$get(s, ios_$no_put_get_opts, p, new, &status);

        if( status.all != status_$ok )
        {
            error_$std_format(status, "Cannot get more data from the server%$");
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }

        plen += new;
        p += new;

        if( plen == line_max )
        {
            *line_len = line_max;
            return;
        }
    }
}

void
pop_$put_line
(
    ios_$id_t  s,
    char      *line,
    long       line_len
)
{
    status_$t status;

    ios_$put(s, ios_$no_put_get_opts, line, line_len, &status);

    if( status.all != status_$ok )
    {
        error_$std_format(status, "Cannot say \"%A\"%$", line, &line_len);
        pgm_$set_severity(pgm_$error);
        pgm_$exit();
    }
}

boolean
pop_$get_response
(
    ios_$id_t  s,
    boolean    multiline,
    char      *data,
    long      *data_len,
    long       data_max
)
{
    boolean response;

    pop_$get_line(s, data, data_len, data_max);

    if( monitor )
    {
        vfmt_$ws(ios_$errout, "S: %A%.", data, data_len);
    }

    if( strl_$eq(data, 3, "+OK", 3) == true )
    {
        response = true;
    }
    else
    {
        response = false;
    }

    if( response == true && multiline == true )
    {
        short dl;

        do
        {
            pop_$get_line(s, data, data_len, data_max);
        
            if( monitor )
            {
                vfmt_$ws(ios_$errout, "S: %A%.", data, data_len);        
            }

            dl = *data_len;
        }
        while( strl_$eq(data, dl, string(".")) == false );
    }

    return response;
}

boolean
pop_$get_message
(
    ios_$id_t s,    /* socket */
    ios_$id_t f     /* file */
)
{
    boolean response;
    char data[ 1024 ];
    long data_len;
    time_$clock_t start_time, end_time;
    int bytes;

    pop_$get_line(s, data, &data_len, sizeof data);

    if( monitor )
    {
        vfmt_$ws(ios_$errout, "S: %A%.", data, &data_len);
    }

    if( strl_$eq(data, 3, "+OK", 3) == true )
    {
        response = true;
    }
    else
    {
        response = false;
    }

    if( response == true )
    {
        if( verbose )
        {
            short dl = data_len, ip = 5;
            bytes = strl_$s2dec(data, dl, &ip);
            time_$clock(&start_time);
        }

        while( 1 )
        {
            short dl;

            pop_$get_line(s, data, &data_len, sizeof data);

            dl = data_len;

            if( strl_$eq(data, dl, string(".")) == true )
            {
                break;
            }

            vfmt_$ws(f, "%A%.", data, &data_len);
        }

        if( verbose )
        {
            double time, rate;
            time_$clock(&end_time);
            cal_$sub_clock(&end_time, start_time);

            if( end_time.c2.high16 > 0 )    /* a few hours.. */
            {
                vfmt_$ws(ios_$errout, "Message received, finally.%.");
            }

            time = (double)end_time.c2.low32 * 0.000004;
            rate = (double)bytes / time;

            vfmt_$ws(ios_$errout, "%ULD bytes received in %LF seconds, for %LF b/s.%.", &bytes, &time, &rate);
        }
    }

    return response;
}

int
pop_$do_command
(
    ios_$id_t       s,
    pop_$command_t  command,
    void           *arg1,
    void           *arg2
)
{
    char line[ 1024 ];
    long line_len;
    short llen;
    int response = -1;

    if( command != pop_$null )
    {
        vfmt_$encode(pop_$command_text[ command ], line, sizeof line, &llen, arg1, arg2);
        
        if( monitor )
        {
            vfmt_$ws(ios_$errout, "C: %A%.", line, &llen);
        }
        
        strl_$append(string("\r\n"), line, &llen, sizeof line);
        line_len = llen;
        
        pop_$put_line(s, line, line_len);
    }

    if( command == pop_$retr )
    {
        /*
         * In this case *arg2 is the stream ID to which we write
         * the data.
         */

        ios_$id_t f = *(ios_$id_t *)arg2;

        if( pop_$get_message(s, f) == false )
        {
            vfmt_$ws(ios_$errout, "RETR bombed%.");
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }
    }
    else
    {
        short i, sxp = 0, llen;

        if( pop_$get_response(s, pop_$command_multi[ command ], line, &line_len, sizeof line) == false )
        {
            vfmt_$ws(ios_$errout, "Command bombed%.");
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }

        llen = line_len;

        for( i = 0; i <= pop_$resp_n[ command ]; i++ )
        {
            char token[ 16 ];
            short token_len, num_offset = 1;

            strl_$gettok(line, llen, &sxp, token, &token_len, sizeof token);

            if( token_len > 0 )
            {
                response = strl_$s2dec(token, token_len, &num_offset);
            }
            else
            {
                vfmt_$ws(ios_$errout, "Didn't get the %WD response parameters I expected%.", &pop_$resp_n[ command ]);
                pgm_$set_severity(pgm_$error);
                pgm_$exit();
            }
        }
    }

    return response;
}

void
pop_$process
(
    ios_$id_t s
)
{
    int quantity, i;

    pop_$do_command(s, pop_$null, 0, 0);

    pop_$do_command(s, pop_$user, username, &username_len);

    if( password_len == 0 )
    {
        pop_$do_command(s, pop_$rpop, username, &username_len);
    }
    else
    {
        pop_$do_command(s, pop_$pass, password, &password_len);
    }

    if( verbose )
    {
        vfmt_$write("Logged in as %A%.", username, &username_len);
    }

    quantity = pop_$do_command(s, pop_$stat, 0, 0);

    if( verbose )
    {
        vfmt_$write("User %A has %LD new messages.%.", username, &username_len, &quantity);
    }

    for( i = 1; i <= quantity; i++ )
    {
        status_$t status;
        name_$long_pname_t name;
        short name_len;
        ios_$id_t file;

        vfmt_$encode(prefix, name, sizeof name, &name_len, &i);
        ios_$create(name, name_len, unstruct_$uid, ios_$no_pre_exist_mode, ios_$write_opt, &file, &status);

        if( status.all != status_$ok )
        {
            error_$std_format(status, "Cannot create object \"%A\"%$", name, &name_len);
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }

        pop_$do_command(s, pop_$retr, &i, &file);

        ios_$close(file, &status);

        if( status.all != status_$ok )
        {
            error_$std_format(status, "Cannot close object \"%A\"%$", name, &name_len);
            pgm_$set_severity(pgm_$error);
            pgm_$exit();
        }

        if( verbose )
        {
            vfmt_$write("Retrieved message %LD to file %A .%.", &i, name, &name_len);
        }

        pop_$do_command(s, pop_$dele, &i, 0);
    }

    pop_$do_command(s, pop_$quit, 0, 0);

    if( verbose )
    {
        vfmt_$write("Retrieved %LD message(s).%.", &quantity);
    }

    if( quantity > 0 )
    {
        pgm_$set_severity(pgm_$true);
    }
    else
    {
        pgm_$set_severity(pgm_$false);
    }

    return;
}

main(){ pop_$init(); }
