{*****************************************************************************
 *****                                                                   *****
 *****                           NETWORK.PAS                             *****
 *****                                                                   *****
 *****      Library of routines to get info about the nodes              *****
 *****      and users on the Apollo DOMAIN network.                      *****
 *****                            Version 6                              *****
 *****                  David M. Krowitz March 17, 1987.                 *****
 *****                                                                   *****
 *****      Copyright (c) 1987                                           *****
 *****      David M. Krowitz                                             *****
 *****      Massachusetts Institute of Technology                        *****
 *****      Department of Earth, Atmospheric, and Planetary Sciences     *****
 *****************************************************************************
}
MODULE NETWORK;

%NOLIST;
%INCLUDE '/sys/ins/base.ins.pas';
%INCLUDE '/sys/ins/name.ins.pas';
%INCLUDE '/sys/ins/pm.ins.pas';
%INCLUDE '/sys/ins/pgm.ins.pas';
%INCLUDE '/sys/ins/streams.ins.pas';
%LIST;


CONST

    {Definitions of some standard ascii control characters}

    etx = chr(3);           {etx (control-C) character}
    lf  = chr(10);          {line-feed character}
    ff  = chr(12);          {form-feed character}
    cr  = chr(13);          {carriage-return character}
    sub = chr(26);          {sub (control-Z) character}
    esc = chr(27);          {escape character}
    rs  = chr(30);          {rs character}


TYPE

    network_$node_id_t      =   array[1..5] of char;
    network_$entry_dir_t    =   array[1..32] of char;
    network_$user_name_t    =   array[1..32] of char;
    network_$user_sid_t     =   array[1..140] of char;
    network_$pathname_t     =   array[1..256] of char;

    network_$node_data_t    =   RECORD
                        node_id:        network_$node_id_t;     {5 digit hex. node ID}
                        entry_dir:      network_$entry_dir_t;   {Entry directory of node}
                        entry_len:      integer16;              {Length of entry dir}
                        catalog_dir:    network_$entry_dir_t;   {Name node cataloged by (same as entry dir for nodes with disks)}
                        catalog_len:    integer16;              {Length of catalog name (0 if diskless node not catalogged)}
                        diskless:       boolean;                {TRUE if node is diskless}
                        partner_id:     network_$node_id_t;     {Node ID of partner for diskless nodes}
                        END;

    network_$node_array_t   =   array[1..1024] of network_$node_data_t;

    network_$user_t         =   RECORD
                        user_sid:       network_$user_sid_t;    {SID of user logged into nodes' DM}
                        user_len:       integer16;              {Length of user SID}
                        node_id:        network_$node_id_t;     {5 digit hex. node ID}
                        entry_dir:      network_$entry_dir_t;   {Entry directory of node}
                        entry_len:      integer16;              {Length of entry dir}
                        catalog_dir:    network_$entry_dir_t;   {Name node cataloged by (same as entry dir for nodes with disks)}
                        catalog_len:    integer16;              {Length of catalog name (0 if diskless node not catalogged)}
                        diskless:       boolean;                {TRUE if node is diskless}
                        END;

    network_$user_array_t   =   array[1..1024] of network_$user_t;

    network_$index_list_t   =   array[1..1024] of pinteger;






    {List all nodes which are currently responding on the Apollo
     DOMAIN network. Use the LCNODE command to get a list of all
     of the currently catalogged nodes.}

    PROCEDURE network_$list_nodes (
                                OUT node_array:     network_$node_array_t;
                                OUT num_nodes:      pinteger
                                );


    TYPE
        input_line_t    =   array[1..256] of char;

    VAR
        i,j,k:          INTEGER16;                      {Counters}
        args:           array[1..2] of ^PGM_$ARG;       {Arguments to invoked program}
        conn_vec:       array[0..1] of STREAM_$ID_T;    {Stream connection vector}
        mode:           PGM_$MODE;                      {Program mode}
        reserved:       array[1..8] of char;            {Reserved for future use}
        buffptr:        ^input_line_t;                  {Pointer to buffer for reading LCNODE output}
        lineptr:        ^input_line_t;                  {Pointer to data returned by GET_REC}
        linelen:        INTEGER32;                      {Number of bytes of data returned}
        seek_key:       STREAM_$SK_T;                   {Stream seek-key returned by GET_REC}
        status:         STATUS_$T;                      {Status returned by PGM calls}


    BEGIN

        {Invoke CTNODE -UPDATE to make sure we have the nodes all
         catalogged correctly before doing the LCNODE command.}

        conn_vec[0] := STREAM_$STDIN;
        NEW (args[1]);
        NEW (args[2]);
        args[1]^.len := 6;                  {Length of argument}
        args[1]^.chars := 'ctnode';         {1st arg. is the program name}
        args[2]^.len := 7;                  {Length of 2nd argument}
        args[2]^.chars := '-update';        {2nd arg. is the arg to the program}
        mode := [PGM_$WAIT];
        PGM_$INVOKE ('/com/ctnode',11,2,args,1,conn_vec,mode,reserved,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - CTNODE command failed ****');
            PGM_$EXIT;
        END;
         
        {Create a temporary file to hold the output of the LCNODE program.
         Set up the connection-vector so that this stream will become
         LCNODE's standard output stream.}

        STREAM_$CREATE ('',0,STREAM_$OVERWRITE,STREAM_$NO_CONC_WRITE,conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - can not create stream ****');
            PGM_$EXIT;
        END;


        {Invoke LCNODE to get the info we need, and then close the
         stream so we can reopen it for reading.}

        args[1]^.len := 6;                  {Length of argument}
        args[1]^.chars := 'lcnode';         {1st arg. is the program name}
        mode := [PGM_$WAIT];

        PGM_$INVOKE ('/com/lcnode',11,1,addr(args),2,conn_vec,
                     mode,reserved,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - LCNODE command failed ****');
            PGM_$EXIT;
        END;


        {Reset the file and read the LCNODE data.
         The first line is blank, the 2nd line contains the node-ID
         of this node, the 3rd line contains the number of nodes on
         the network - 1, the 4th line is blank, the 5th line is
         the column headings of the node listing, the 6th line is
         blank, and the node listings start on the 7th line. The
         first node listed is the node making this call.}

        STREAM_$SEEK (conn_vec[1],STREAM_$CHR,STREAM_$ABSOLUTE,1,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - can not reset stream ****');
            PGM_$EXIT;
        END;
                                 

        {Get the number of other nodes on the network. It starts at
         the 2nd character of the 3rd line of the listing. The actual
        number of nodes is one more than the number given in the listing.}

        NEW (buffptr);
        STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
        STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
        STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - can not read number of nodes ****');
            PGM_$EXIT;
        END;
        num_nodes := 0;
        i := 2;
        WHILE (lineptr^[i] <> ' ') DO BEGIN
            num_nodes := num_nodes*10+ORD(lineptr^[i])-ORD('0');
            i := i+1;
        END;
        num_nodes := num_nodes+1;

        {Read the node listings into the return array. The 2nd
         through 6th characters of each line are the node-ID.
         The entry directory starts with the 52nd character for nodes
         with disks. For diskless nodes, the name the node is catalogged
         by starts with the 52nd character. If the node is diskless and
         not catalogged with any name, the 5nd character is a '*' (start
         of the string '*** DISKLESS *** partner node:').}

        STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
        STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
        STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);

        FOR i := 1 TO num_nodes DO BEGIN
            STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
            IF (status.all <> 0) THEN BEGIN
                writeln ('**** NETWORK_$LIST_NODES: Error - can not read node info ****');
                PGM_$EXIT;
            END;


            {Extract the node ID from the line. The node ID is left justified
             and padded with blanks in the LCNODE output, so we can read it
             directly into the node ID array.}

            WITH node_array[i] DO BEGIN

                FOR j := 1 TO 5 DO BEGIN
                    node_id[j] := lineptr^[j+1];
                END;


                {Extract the directory name by which the node is catalogged
                 from the line. If the node is not diskless, then this is
                 also the entry directory. If the node is diskless and not
                 catalogged with any name, then return a length of 0 for
                 the node's catalog name. If node is diskless, extract the
                 5 character node ID of the partner node. The node ID will
                 be left justified and padded with blanks if necessary.}

                diskless := FALSE;
                FOR j := 1 TO linelen DO BEGIN
                    IF (lineptr^[j] = '*') THEN diskless := TRUE;
                END;
                IF (diskless = FALSE) THEN BEGIN
                    j := 52;
                    k := 1;
                    WHILE (lineptr^[j] <> lf) DO BEGIN
                        entry_dir[k] := lineptr^[j];
                        j := j+1;
                        k := k+1;
                    END;
                    entry_len := j-52;
                    FOR j := 1 TO entry_len DO BEGIN
                        catalog_dir[j] := entry_dir[j];
                    END;
                    catalog_len := entry_len;
                END
                ELSE BEGIN
                    IF (lineptr^[52] = '/') THEN BEGIN
                        j := 52;
                        k := 1;
                        WHILE (lineptr^[j] <> ' ') DO BEGIN
                            catalog_dir[k] := lineptr^[j];
                            j := j+1;
                            k := k+1;
                        END;
                        catalog_len := k-1;
                    END
                    ELSE BEGIN
                        catalog_len := 0;
                    END;
                    FOR k := 1 TO 5 DO partner_id[k] := ' ';
                    j := linelen;
                    WHILE (lineptr^[j-1] <> ' ') DO BEGIN
                        j := j-1;
                    END;
                    k := 1;
                    WHILE (lineptr^[j] <> lf) DO BEGIN
                        partner_id[k] := lineptr^[j];
                        j := j+1;
                        k := k+1;
                    END;
                END;
            END; {End of WITH node_array[i] DO ...}    
        END;


        {Find the entry directories for any diskless nodes.}

        FOR i := 1 TO num_nodes DO WITH node_array[i] DO BEGIN
            IF (node_array[i].diskless = TRUE) THEN BEGIN
                FOR j := 1 TO num_nodes DO BEGIN
                    IF (i <> j) AND THEN
                    (partner_id[1] = node_array[j].node_id[1]) AND THEN
                    (partner_id[2] = node_array[j].node_id[2]) AND THEN
                    (partner_id[3] = node_array[j].node_id[3]) AND THEN
                    (partner_id[4] = node_array[j].node_id[4]) AND THEN
                    (partner_id[5] = node_array[j].node_id[5]) THEN BEGIN
                        FOR k := 1 TO node_array[j].entry_len DO BEGIN
                            entry_dir[k] := node_array[j].entry_dir[k];
                        END;
                        entry_len := node_array[j].entry_len;
                        EXIT;
                    END;
                END;
            END;
        END;

        {All done. Close and delete the file.}

        STREAM_$CLOSE (conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - can not close stream ****');
            PGM_$EXIT;
        END;

    END;            {End of procedure NETWORK_$LIST_NODES.}





    {Check that a given node is on the DOMAIN network and find
     the index to the node's entry in the list of nodes supplied
     by NETWORK_$LIST_NODES so that we can look up its node ID.}

    PROCEDURE network_$find_node (
                                IN  catalog_dir:    network_$entry_dir_t;
                                IN  catalog_len:    pinteger;
                                IN  node_list:      network_$node_array_t;
                                IN  num_nodes:      pinteger;
                                OUT index:          pinteger
                                );

    VAR
        i,j,k:          INTEGER16;                      {Counters}


    {Some useful internal functions for doing string manipulations}

    FUNCTION lowercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['A'..'Z']) THEN BEGIN
                lowercase := CHR(ORD(character)-ORD('A')+ORD('a'))
            END
            ELSE BEGIN
                lowercase := character;
            END;
        END;        {End of function LOWERCASE.}



    FUNCTION uppercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['a'..'z']) THEN BEGIN
                uppercase := CHR(ORD(character)-ORD('a')+ORD('A'))
            END
            ELSE BEGIN
                uppercase := character;
            END;
        END;        {End of function UPPERCASE.}



    FUNCTION compare_entries (
                            IN  string1:    network_$entry_dir_t;
                            IN  len1:       pinteger;
                            IN  string2:    network_$entry_dir_t;
                            IN  len2:       pinteger
                            ):BOOLEAN;
    
        VAR
            i:  pinteger;

        BEGIN
            IF (len1 <> len2) THEN BEGIN
                compare_entries := FALSE;
                RETURN;
            END;
            compare_entries := TRUE;
            FOR i := 1 TO len1 DO BEGIN
                IF (uppercase(string1[i]) <> uppercase(string2[i])) THEN BEGIN
                    compare_entries := FALSE;
                    RETURN;
                END;
            END;
        END;        {End of function COMPARE_ENTRIES.}


    {Beginning of main body of NETWORK_$FIND_NODE}

    BEGIN
        index := 0;
        FOR i := 1 TO num_nodes DO BEGIN
            IF (compare_entries(catalog_dir,catalog_len,node_list[i].catalog_dir,node_list[i].catalog_len) = TRUE) THEN BEGIN
                index := i;
                EXIT;
            END;
        END;
    END;            {End of procedure NETWORK_$FIND_NODE.}





    {Check that a given node is on the DOMAIN network and find
     the index to the node's entry in the list of nodes supplied
     by NETWORK_$LIST_NODES so that we can look up its entry
     directory and see whether or not it is a diskless node.}

    PROCEDURE network_$find_node_id (
                                IN  node_id:        network_$node_id_t;
                                IN  node_list:      network_$node_array_t;
                                IN  num_nodes:      pinteger;
                                OUT index:          pinteger
                                );

    VAR
        i,j,k:          INTEGER16;                      {Counters}


    {Some useful internal functions for doing string manipulations}

    FUNCTION lowercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['A'..'Z']) THEN BEGIN
                lowercase := CHR(ORD(character)-ORD('A')+ORD('a'))
            END
            ELSE BEGIN
                lowercase := character;
            END;
        END;        {End of function LOWERCASE.}



    FUNCTION uppercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['a'..'z']) THEN BEGIN
                uppercase := CHR(ORD(character)-ORD('a')+ORD('A'))
            END
            ELSE BEGIN
                uppercase := character;
            END;
        END;        {End of function UPPERCASE.}



    FUNCTION compare_node_ids (
                            IN  string1:    network_$node_id_t;
                            IN  string2:    network_$node_id_t
                            ):BOOLEAN;
    
        VAR
            i:  pinteger;

        BEGIN
            compare_node_ids := TRUE;
            FOR i := 1 TO 5 DO BEGIN
                IF (uppercase(string1[i]) <> uppercase(string2[i])) THEN BEGIN
                    compare_node_ids := FALSE;
                    RETURN;
                END;
            END;
        END;        {End of function COMPARE_NODE_IDS.}


    {Beginning of main body of NETWORK_$FIND_NODE_ID}

    BEGIN
        index := 0;
        FOR i := 1 TO num_nodes DO BEGIN
            IF (compare_node_ids(node_id,node_list[i].node_id)) THEN BEGIN
                index := i;
                EXIT;
            END;
        END;
    END;            {End of procedure NETWORK_$FIND_NODE_ID.}





    {List all users logged into the Apollo DOMAIN network.
     Use the LUSR command to get a list of all of the people who
     are logged in to the display manager of each node on the 
     network.}

    PROCEDURE network_$list_users (
                                OUT user_array:     network_$user_array_t;
                                OUT num_users:      pinteger
                                );


    TYPE
        input_line_t    =   array[1..256] of char;

    VAR
        i,j,k:          INTEGER16;                      {Counters}
        args:           array[1..2] of ^PGM_$ARG;       {Arguments to invoked program}
        conn_vec:       array[0..1] of STREAM_$ID_T;    {Stream connection vector}
        mode:           PGM_$MODE;                      {Program mode}
        reserved:       array[1..8] of char;            {Reserved for future use}
        buffptr:        ^input_line_t;                  {Pointer to buffer for reading LUSR output}
        lineptr:        ^input_line_t;                  {Pointer to data returned by GET_REC}
        linelen:        INTEGER32;                      {Number of bytes of data returned}
        seek_key:       STREAM_$SK_T;                   {Stream seek-key returned by GET_REC}
        status:         STATUS_$T;                      {Status returned by PGM calls}


    BEGIN

        {Invoke CTNODE -UPDATE to make sure we have the nodes all
         catalogged correctly before doing the LUSR command.}

        conn_vec[0] := STREAM_$STDIN;
        NEW (args[1]);
        NEW (args[2]);
        args[1]^.len := 6;                  {Length of argument}
        args[1]^.chars := 'ctnode';         {1st arg. is the program name}
        args[2]^.len := 7;                  {Length of 2nd argument}
        args[2]^.chars := '-update';        {2nd arg. is the arg to the program}
        mode := [PGM_$WAIT];
        PGM_$INVOKE ('/com/ctnode',11,2,args,1,conn_vec,mode,reserved,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_NODES: Error - CTNODE command failed ****');
            PGM_$EXIT;
        END;


        {Create a temporary file to hold the output of the LUSR program.
         Set up the connection-vector so that this stream will become
         LUSR's standard output stream. Then invoke LUSR -FULL to get
         the desired info.}

        STREAM_$CREATE ('',0,STREAM_$OVERWRITE,STREAM_$NO_CONC_WRITE,
                        conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_USERS: Error - can not create stream ****');
            PGM_$EXIT;
        END;

        args[1]^.len := 4;              {Length of 1st argument}
        args[1]^.chars := 'lusr';       {1st arg. is the program name}
        args[2]^.len := 5;              {Length of 2nd argument}
        args[2]^.chars := '-full';      {2nd arg. is the arg to the program}
        mode := [PGM_$WAIT];

        PGM_$INVOKE ('/com/lusr',9,2,args,2,conn_vec,mode,reserved,status);    
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_USERS: Error - LUSR command failed ****');
            PGM_$EXIT;
        END;


        {Reset the file and read the LUSR data.
         Each line contains a full user name (SID) of the user who
         is logged into the DM on that node. Nodes which don't have
         anyone logged into the DM aren't listed. The last line of
         the file tells the number of nodes listed out of the total
         number of nodes on the network which responded to the 
         LUSR program. Each user name is followed by the entry
         directory of the node or by the entry directory of the
         node's partner if it is a diskless node.}

        STREAM_$SEEK (conn_vec[1],STREAM_$CHR,STREAM_$ABSOLUTE,1,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_USERS: Error - can not reset stream ****');
            PGM_$EXIT;
        END;
        num_users := 0;


        {Read a line, if it's not blank, then the line contains a user
         name and an entry directory. Otherwise, the line is the second
         to last line in the listing. The user name starts with the 5th
         character of the line and ends with the first space following
         that character. The node-id is the last 1 to 5 characters of the
         user name. The entry directory starts with the 37th character
         unless the node is diskless. If the node is diskless and it is
         catalogged with some name in the // directory, then its catalog
         name starts in column 50 (with '//'). If the node does not have
         a name that it is catalogged by then return 0 for the length of
         the catalog name.}

        i := 0;
        NEW (buffptr);
        REPEAT
            STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,
                             linelen,seek_key,status);
            IF (status.all <> 0) THEN BEGIN
                writeln ('**** NETWORK_$LIST_USERS: Error - can not read user info ****');
                PGM_$EXIT;
            END;
            IF (lineptr^[1] <> lf) AND (linelen <> 1) THEN BEGIN
                i := i+1;


                {Extract the user's SID from the line.}

                j := 5;
                k := 1;
                WHILE (lineptr^[j] <> ' ') DO BEGIN
                    user_array[i].user_sid[k] := lineptr^[j];
                    j := j+1;
                    k := k+1;
                END;
                user_array[i].user_len := k-1;


                {Extract the node's ID from the user's SID.}

                FOR j := 1 TO 5 DO BEGIN
                    user_array[i].node_id[j] := ' ';
                END;
                j := 1;
                FOR k := 1 TO 3 DO BEGIN
                    WHILE (user_array[i].user_sid[j] <> '.') DO j := j+1;
                    j := j+1;
                END;
                k := 1;
                WHILE (j <= user_array[i].user_len) DO BEGIN
                    IF (user_array[i].user_sid[j] <> '.') THEN BEGIN
                        user_array[i].node_id[k] := user_array[i].user_sid[j];
                        k := k+1;
                        j := j+1;
                    END
                    ELSE EXIT;
                END;


                {Extract the node's entry directory from the line. If
                 the node is not diskless then this is the same as the
                 node's catalogged name.}

                WITH user_array[i] DO BEGIN
                    diskless := FALSE;
                    FOR j := 1 TO linelen DO BEGIN
                        IF (lineptr^[j] = '*') THEN diskless := TRUE;
                    END;
                    j := linelen;
                    WHILE (lineptr^[j-1] <> ' ') DO BEGIN
                        j := j-1;
                    END;
                    FOR k := j to linelen DO BEGIN
                        entry_dir[k-j+1] := lineptr^[k];
                    END;
                    entry_len := linelen-j;
                    IF (diskless = FALSE) THEN BEGIN
                        FOR j := 1 TO entry_len DO BEGIN
                            catalog_dir[j] := entry_dir[j];
                        END;
                        catalog_len := entry_len;
                    END
                    ELSE IF (lineptr^[50] = '/') THEN BEGIN
                        j := 50;
                        k := 1;
                        WHILE (lineptr^[j] <> ' ') DO BEGIN
                            catalog_dir[k] := lineptr^[j];
                            j := j+1;
                            k := k+1;
                        END;
                        catalog_len := j-50;
                    END
                    ELSE BEGIN
                        catalog_len := 0;
                    END;
                END;
            END;
        UNTIL (lineptr^[1] = lf) AND (linelen = 1);


        {All done. Return total number of users list and close the
         file. Since it's temporary it will be deleted upon closing.}

        num_users := i;
        STREAM_$CLOSE (conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$LIST_USERS: Error - can not close stream ****');
            PGM_$EXIT;
        END;

    END;            {End of procedure NETWORK_$LIST_USERS.}

                                  



    {Find all the nodes that a given user is logged into on the DOMAIN network.
     Use the list of users logged into each node's display manager which
     is returned by NETWORK_$LIST_USERS to find which nodes a given user
     is logged into. Return a list of indices into the user-list for the
     nodes which they are logged into. Return an index_count of 0 if they
     are not logged in anywhere.}

    PROCEDURE network_$find_user (
                                IN  user_name:      network_$user_name_t;
                                IN  user_name_len:  pinteger;
                                IN  user_list:      network_$user_array_t;
                                IN  num_users:      pinteger;
                                OUT index_list:     network_$index_list_t;
                                OUT index_count:    pinteger
                                );

    VAR
        i,j,k:          INTEGER16;                      {Counters}
        name:           network_$user_name_t;           {User name extracted from user SID}
        name_len:       pinteger;                       {Length of user name}


    {Some useful internal functions for doing string manipulations}

    FUNCTION lowercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['A'..'Z']) THEN BEGIN
                lowercase := CHR(ORD(character)-ORD('A')+ORD('a'))
            END
            ELSE BEGIN
                lowercase := character;
            END;
        END;        {End of function LOWERCASE.}



    FUNCTION uppercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['a'..'z']) THEN BEGIN
                uppercase := CHR(ORD(character)-ORD('a')+ORD('A'))
            END
            ELSE BEGIN
                uppercase := character;
            END;
        END;        {End of function UPPERCASE.}



    FUNCTION compare_names (
                            IN  string1:    network_$user_name_t;
                            IN  len1:       pinteger;
                            IN  string2:    network_$user_name_t;
                            IN  len2:       pinteger
                            ):BOOLEAN;
    
        VAR
            i:  pinteger;

        BEGIN
            IF (len1 <> len2) THEN BEGIN
                compare_names := FALSE;
                RETURN;
            END;
            compare_names := TRUE;
            FOR i := 1 TO len1 DO BEGIN
                IF (uppercase(string1[i]) <> uppercase(string2[i])) THEN BEGIN
                    compare_names := FALSE;
                    RETURN;
                END;
            END;
        END;        {End of function COMPARE_NAMES.}



    PROCEDURE extract_name_from_sid (
                                IN  sid:        network_$user_sid_t;
                                IN  sid_len:    pinteger;
                                OUT name:       network_$user_name_t;
                                OUT name_len:   pinteger
                                );
    
        VAR
            i:  pinteger;

        BEGIN
        i := 1;
        WHILE (sid[i] <> '.') DO BEGIN
            name[i] := sid[i];
            i := i+1;
        END;
        name_len := i-1;
        END;        {End of function EXTRACT_NAME_FROM_SID.}




    {Beginning of main body of NETWORK_$FIND_USER}

    BEGIN
        index_count := 0;
        FOR i := 1 TO num_users DO WITH user_list[i] DO BEGIN
            extract_name_from_sid (user_sid,user_len,name,name_len);
            IF (compare_names(user_name,user_name_len,name,name_len)) THEN BEGIN
                index_count := index_count+1;
                index_list[index_count] := i;
            END;
        END;
    END;            {End of procedure NETWORK_$FIND_USER.}





    {Check that a user is logged into a particular node on the DOMAIN network.
     Use the list of users logged into each node's display manager which
     is returned by NETWORK_$LIST_USERS and the list of nodes on the
     network returned by NETWORK_$LIST_NODES to check if a user is
     logged into a particular node. Return both an index into the user-list
     and an index into the node-list. Return index counts of 0 if the
     user is not logged into the particular node.}

    PROCEDURE network_$find_user_at_node_id (
                                IN  node:           network_$node_id_t;
                                IN  user_name:      network_$user_name_t;
                                IN  user_name_len:  pinteger;
                                IN  user_list:      network_$user_array_t;
                                IN  num_users:      pinteger;
                                OUT user_index:     pinteger
                                );

    VAR
        i,j,k:          INTEGER16;                      {Counters}
        name:           network_$user_name_t;           {User name extracted from user SID}
        name_len:       pinteger;                       {Length of user name}


    {Some useful internal functions for doing string manipulations}

    FUNCTION lowercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['A'..'Z']) THEN BEGIN
                lowercase := CHR(ORD(character)-ORD('A')+ORD('a'))
            END
            ELSE BEGIN
                lowercase := character;
            END;
        END;        {End of function LOWERCASE.}



    FUNCTION uppercase (IN  character:  CHAR):CHAR;
        BEGIN
            IF (character IN ['a'..'z']) THEN BEGIN
                uppercase := CHR(ORD(character)-ORD('a')+ORD('A'))
            END
            ELSE BEGIN
                uppercase := character;
            END;
        END;        {End of function UPPERCASE.}



    FUNCTION compare_names (
                            IN  string1:    network_$user_name_t;
                            IN  len1:       pinteger;
                            IN  string2:    network_$user_name_t;
                            IN  len2:       pinteger
                            ):BOOLEAN;
    
        VAR
            i:  pinteger;

        BEGIN
            IF (len1 <> len2) THEN BEGIN
                compare_names := FALSE;
                RETURN;
            END;
            compare_names := TRUE;
            FOR i := 1 TO len1 DO BEGIN
                IF (uppercase(string1[i]) <> uppercase(string2[i])) THEN BEGIN
                    compare_names := FALSE;
                    RETURN;
                END;
            END;
        END;        {End of function COMPARE_NAMES.}



    FUNCTION compare_nodes (
                            IN  string1:    network_$node_id_t;
                            IN  string2:    network_$node_id_t
                            ):BOOLEAN;
    
        VAR
            i:  pinteger;

        BEGIN
            compare_nodes := TRUE;
            FOR i := 1 TO 5 DO BEGIN
                IF (uppercase(string1[i]) <> uppercase(string2[i])) THEN BEGIN
                    compare_nodes := FALSE;
                    RETURN;
                END;
            END;
        END;        {End of function COMPARE_NODES.}



    PROCEDURE extract_name_from_sid (
                                IN  sid:        network_$user_sid_t;
                                IN  sid_len:    pinteger;
                                OUT name:       network_$user_name_t;
                                OUT name_len:   pinteger
                                );
    
        VAR
            i:  pinteger;

        BEGIN
        i := 1;
        WHILE (sid[i] <> '.') DO BEGIN
            name[i] := sid[i];
            i := i+1;
        END;
        name_len := i-1;
        END;        {End of function EXTRACT_NAME_FROM_SID.}




    {Beginning of main body of NETWORK_$FIND_USER_AT_NODE}

    BEGIN
        user_index := 0;
        FOR i := 1 TO num_users DO WITH user_list[i] DO BEGIN
            extract_name_from_sid (user_sid,user_len,name,name_len);
            IF (compare_names(user_name,user_name_len,name,name_len)) AND
            (compare_nodes(node,node_id))THEN BEGIN
                user_index := i;
                EXIT;
            END;
        END;
    END;            {End of procedure NETWORK_$FIND_USER_AT_NODE.}

                                  



    {Return the name of the user who owns this process.
     Extract the user name from the full user SID of this
     process (ie. throw away the project, organization, and
     node ID numbers).}

    PROCEDURE network_$get_proc_user_name (
                                OUT user_name:      network_$user_name_t;
                                OUT name_length:    pinteger
                                );
    VAR

        sid:            array[1..140] of char;      {The complete SID returned for this process}
        sid_len:        pinteger;                   {The length of the SID returned}
        i:              pinteger;                   {Counter}

    BEGIN
        PM_$GET_SID_TXT (140,sid,sid_len);
        i := 1;
        WHILE (sid[i] <> '.') DO BEGIN
            user_name[i] := sid[i];
            i := i+1;
        END;
        name_length := i-1;
    END;            {End of procedure NETWORK_$GET_PROC_USER_NAME.}





    {Return the full SID of the owner of this process.
     Included with the NETWORK calls for completeness of
     the library even though it duplicates a PM call.}

    PROCEDURE network_$get_proc_user_sid (
                                OUT user_sid:       network_$user_sid_t;
                                OUT sid_length:     pinteger
                                );
    BEGIN
        PM_$GET_SID_TXT (140,user_sid,sid_length);
    END;            {End of procedure NETWORK_$GET_PROC_USER_SID.}





    {Return the node ID number of the node running this process.
     Extract the node ID from the full user SID of this process
     (ie. throw away the user name and the project and
     organization numbers). Note that the node ID is the 4th
     field of up to 5 fields in the SID (ie. LUSR -ALLP will
     show some processes as: user.server.none.252C.LOGIN).}

    PROCEDURE network_$get_proc_node_id (
                                OUT node_id:        network_$node_id_t
                                );
    VAR

        sid:            array[1..140] of char;      {The complete SID returned for this process}
        sid_len:        pinteger;                   {The length of the SID returned}
        i,j:            pinteger;                   {Counters}

    BEGIN
        PM_$GET_SID_TXT (140,sid,sid_len);
        FOR i := 1 TO 5 DO node_id[i] := ' ';
        j := 1;
        FOR i := 1 TO 3 DO BEGIN
            WHILE (sid[j] <> '.') DO j := j+1;
            j := j+1;
        END;
        i := 1;
        WHILE (j <= sid_len) DO BEGIN
            IF (sid[j] <> '.') THEN BEGIN
                node_id[i] := sid[j];
                i := i+1;
                j := j+1;
            END
            ELSE EXIT;
        END;
    END;            {End of procedure NETWORK_$GET_PROC_NODE_ID.}





    {Return the entry directory of the node running this process.
     Extract this info from the full pathname of the `NODE_DATA
     directory of the node.}

    PROCEDURE network_$get_proc_entry_dir (
                                OUT entry_dir:      network_$entry_dir_t;
                                OUT entry_len:      integer16;
                                OUT diskless_flag:  boolean
                                );
    VAR
        i,j,k:          INTEGER16;                      {Counters}
        full_path:      array[1..256] of char;          {Full pathname of `NODE_DATA directory}
        full_len:       INTEGER16;                      {Length of full pathname}
        status:         STATUS_$T;                      {Status returned by NAME calls}


    BEGIN

        {Get the full pathname of the `NODE_DATA directory of the node
         running this process. Then extract the entry directory from
         the full pathname.}

        NAME_$GET_PATH ('`node_data',10,full_path,full_len,status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$GET_PROC_ENTRY_DIR: Error - can not get full pathname ****');
            PGM_$EXIT;
        END;

        IF (full_len <= 2) THEN BEGIN
            writeln ('**** NETWORK_$GET_PROC_ENTRY_DIR: Error - full pathname  too short ****');
            PGM_$EXIT;
        END;

        entry_dir[1] := '/';
        entry_dir[2] := '/';
        i := 3;
        WHILE (full_path[i] <> '/') DO BEGIN
            entry_dir[i] := full_path[i];
            i := i+1;
        END;
        entry_len := i-1;

        diskless_flag := FALSE;
        j := i+1;
        WHILE (j <= full_len) DO BEGIN
            IF (full_path[j] = '.') THEN BEGIN
                diskless_flag := TRUE;
                EXIT;
            END
            ELSE BEGIN
                j := j+1;
            END;
        END;

    END;            {End of procedure NETWORK_$GET_PROC_ENTRY_DIR.}






    {Return the home directory of the specified user.
     The user name can be short (eg. "krowitz") or long
     with project and organization fields, but not a node ID
     field (eg. "krowitz.none.jordan"). The name may contain
     wildcards (eg. "krowitz.%.jordan"), in which case the
     first matching entry from the registry is returned.
     Note that the user name is in a variable of the type
     NETWORK_$USER_SID_T not NETWORK_$USER_NAME_Tin order
     to provide enough characters for long names and wild cards.}

    PROCEDURE network_$get_user_home_dir (
                                IN  user_name:      network_$user_sid_t;
                                IN  name_len:       pinteger;
                                OUT home_dir:       network_$pathname_t;
                                OUT dir_len:        pinteger
                                );

    TYPE
        input_line_t    =   array[1..256] of char;

    VAR
        i,j,k:          INTEGER16;                      {Counters}
        args:           array[1..3] of ^PGM_$ARG;       {Arguments to invoked program}
        conn_vec:       array[0..1] of STREAM_$ID_T;    {Stream connection vector}
        mode:           PGM_$MODE;                      {Program mode}
        reserved:       array[1..8] of char;            {Reserved for future use}
        buffptr:        ^input_line_t;                  {Pointer to buffer for reading LCNODE output}
        lineptr:        ^input_line_t;                  {Pointer to data returned by GET_REC}
        linelen:        INTEGER32;                      {Number of bytes of data returned}
        seek_key:       STREAM_$SK_T;                   {Stream seek-key returned by GET_REC}
        status:         STATUS_$T;                      {Status returned by PGM calls}


    BEGIN

        {Create a temporary file to hold the output of the EDACCT program.
         Set up the connection-vector so that this stream will become
         EDACCT's standard output stream.}

        conn_vec[0] := STREAM_$STDIN;
        STREAM_$CREATE ('',0,STREAM_$OVERWRITE,STREAM_$NO_CONC_WRITE,conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$GET_USER_HOME_DIR: Error - can not create stream ****');
            PGM_$EXIT;
        END;


        {Invoke EDACCT -L to get the info we need, and then close the
         stream so we can reopen it for reading.}

        NEW (args[1]);
        NEW (args[2]);
        NEW (args[3]);
        args[1]^.len := 6;                  {Length of argument}
        args[1]^.chars := 'edacct';         {1st arg. is the program name}
        args[2]^.len := 2;                  {Length of 2nd argument}
        args[2]^.chars := '-l';             {2nd arg. is the arg to the program}
        i := 1;
        WHILE (i <= name_len) DO BEGIN
            IF (user_name[i] <> ' ') THEN BEGIN
                args[3]^.chars[i] := user_name[i];
                i := i+1;
            END
            ELSE EXIT;
        END;
        args[3]^.len := i-1;

        mode := [PGM_$WAIT];
        PGM_$INVOKE ('/com/edacct',11,3,args,2,conn_vec,
                     mode,reserved,status);


        {If status was non-zero, then EDACCT couldn't find the user name
         in the registry. Just return a null home directory.}

        IF (status.all <>  0) THEN BEGIN
            dir_len := 0;
            FOR i := 1 TO 256 DO home_dir[i] := ' ';
        END
        ELSE BEGIN


            {Reset the file and read the EDACCT data.
             The first line contains the username, project, and organization
             seperated by spaces followed by the home directory which begins
             in column 35.}

            STREAM_$SEEK (conn_vec[1],STREAM_$CHR,STREAM_$ABSOLUTE,1,status);
            IF (status.all <> 0) THEN BEGIN
                writeln ('**** NETWORK_$GET_USER_HOME_DIR: Error - can not reset stream ****');
                PGM_$EXIT;
            END;
                                 

            NEW (buffptr);
            STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
            IF (status.all <> 0) THEN BEGIN
                writeln ('**** NETWORK_$GET_USER_HOME_DIR: Error - can not read user info ****');
                PGM_$EXIT;
            END;

                i := 35;
                j := 1;
                WHILE (lineptr^[i] <> ' ') AND (lineptr^[i] <> lf) DO BEGIN
                    home_dir[j] := lineptr^[i];
                    j := j+1;
                    i := i+1;
                END;
                dir_len := j-1;
        END;
            

        {All done. Close and delete the file.}

        STREAM_$CLOSE (conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$GET_USER_HOME_DIR: Error - can not close stream ****');
            PGM_$EXIT;
        END;

    END;            {End of procedure NETWORK_$GET_USER_HOME_DIR.}

                                  



    {Return the full name of the specified user.
     The user name is in the short form without wildcards,
     project, or organization (eg. "krowitz" but not
     "krowitz.sys_admin" or "krowitz.%.%"). Note that the
     full name is returned in a variable of the type
     NETWORK_$USER_SID_T in order to have enough space for
     the user's full name.}

    PROCEDURE network_$get_user_full_name (
                                IN  user_name:      network_$user_name_t;
                                IN  name_len:       pinteger;
                                OUT full_name:      network_$user_sid_t;
                                OUT full_len:       pinteger
                                );

    TYPE
        input_line_t    =   array[1..256] of char;

    VAR
        i,j,k:          INTEGER16;                      {Counters}
        args:           array[1..3] of ^PGM_$ARG;       {Arguments to invoked program}
        conn_vec:       array[0..1] of STREAM_$ID_T;    {Stream connection vector}
        mode:           PGM_$MODE;                      {Program mode}
        reserved:       array[1..8] of char;            {Reserved for future use}
        buffptr:        ^input_line_t;                  {Pointer to buffer for reading LCNODE output}
        lineptr:        ^input_line_t;                  {Pointer to data returned by GET_REC}
        linelen:        INTEGER32;                      {Number of bytes of data returned}
        seek_key:       STREAM_$SK_T;                   {Stream seek-key returned by GET_REC}
        status:         STATUS_$T;                      {Status returned by PGM calls}


    BEGIN

        {Create a temporary file to hold the output of the EDPPO program.
         Set up the connection-vector so that this stream will become
         EDPPO's standard output stream.}

        conn_vec[0] := STREAM_$STDIN;
        STREAM_$CREATE ('',0,STREAM_$OVERWRITE,STREAM_$NO_CONC_WRITE,conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$GET_USER_FULL_NAME: Error - can not create stream ****');
            PGM_$EXIT;
        END;


        {Invoke EDPPO -L to get the info we need, and then close the
         stream so we can reopen it for reading.}

        NEW (args[1]);
        NEW (args[2]);
        NEW (args[3]);
        args[1]^.len := 5;                  {Length of argument}
        args[1]^.chars := 'edppo';          {1st arg. is the program name}
        args[2]^.len := 3;                  {Length of 2nd argument}
        args[2]^.chars := '-lf';            {2nd arg. is the arg to the program}
        i := 1;
        WHILE (i <= name_len) DO BEGIN
            IF (user_name[i] <> ' ') THEN BEGIN
                args[3]^.chars[i] := user_name[i];
                i := i+1;
            END
            ELSE EXIT;
        END;
        args[3]^.len := i-1;

        mode := [PGM_$WAIT];
        PGM_$INVOKE ('/com/edppo',10,3,args,2,conn_vec,
                     mode,reserved,status);

        {If status was non-zero, then EDACCT couldn't find the user name
         in the registry. Just return a null full name.}

        IF (status.all <>  0) THEN BEGIN
            full_len := 0;
            FOR i := 1 TO 140 DO full_name[i] := ' ';
        END
        ELSE BEGIN


            {Reset the file and read the EDPPO data.
             The first line contains the username followed by the full user
             name which begins in column 20.}

            STREAM_$SEEK (conn_vec[1],STREAM_$CHR,STREAM_$ABSOLUTE,1,status);
            IF (status.all <> 0) THEN BEGIN
                writeln ('**** NETWORK_$GET_USER_FULL_NAME: Error - can not reset stream ****');
                PGM_$EXIT;
            END;
                                 

            NEW (buffptr);
            STREAM_$GET_REC (conn_vec[1],buffptr,256,lineptr,linelen,seek_key,status);
            IF (status.all <> 0) THEN BEGIN
                writeln ('**** NETWORK_$GET_USER_FULL_NAME: Error - can not read user info ****');
                PGM_$EXIT;
            END;

            i := 20;
            j := 1;
            WHILE (lineptr^[i] <> lf) DO BEGIN
                full_name[j] := lineptr^[i];
                j := j+1;
                i := i+1;
            END;
            full_len := j-1;
        END;
            

        {All done. Close and delete the file.}

        STREAM_$CLOSE (conn_vec[1],status);
        IF (status.all <> 0) THEN BEGIN
            writeln ('**** NETWORK_$GET_USER_FULL_NAME: Error - can not close stream ****');
            PGM_$EXIT;
        END;

    END;            {End of procedure NETWORK_$GET_USER_FULL_NAME.}




{***** End of module NETWORK *****}
