/*
 *	ansitape.c - ANSI tape headers
 */

#include "lang.h"
#include "std.h"
#include "sio.h"
#include "char.h"
#include "str.h"
#include "order.h"
#include "lim.h"
#include "timestamp.h"
#include "tape.h"
#include "format.h"
#include "ansitape.h"
#include "remote.h"
#include "error.h"

int	brec = 0;		/* current record no. within tape/disk block buffer */

char	devunix[DEV_UNIX_LEN];

char	*devname;
int	devtype = DEV_NONE;

int	dblocksz = MT_BLKSIZE;	/* wbak data block size */
int	bufsz = 80;		/* ANSI header buffer = record size */
int	bf = 1;			/* records per tape/disk block */

FILE	*rfp;

devnametab	devnametable[] = {
#if	defined(DEV_UNIX_M)  && defined(DEV_OFFSET_M)
	{ DEV_APOLLO_M, 	DEV_UNIX_M,	DEV_OFFSET_M,	DEV_TAPE },
#endif	/* defined(DEV_UNIX_M)  && defined(DEV_OFFSET_M)  */

#if	defined(DEV_UNIX_MT) && defined(DEV_OFFSET_MT)
	{ DEV_APOLLO_MT,	DEV_UNIX_MT,	DEV_OFFSET_MT,	DEV_TAPE },
#endif	/* defined(DEV_UNIX_MT) && defined(DEV_OFFSET_MT) */

#if	defined(DEV_UNIX_CT) && defined(DEV_OFFSET_CT)
	{ DEV_APOLLO_C,		DEV_UNIX_CT,	DEV_OFFSET_CT,	DEV_TAPE },
#endif	/* defined(DEV_UNIX_C) && defined(DEV_OFFSET_CT) */

#if	defined(DEV_UNIX_CT) && defined(DEV_OFFSET_CT)
	{ DEV_APOLLO_CT,	DEV_UNIX_CT,	DEV_OFFSET_CT,	DEV_TAPE },
#endif	/* defined(DEV_UNIX_CT) && defined(DEV_OFFSET_CT) */

#if	defined(DEV_UNIX_F)  && defined(DEV_OFFSET_F)
	{ DEV_APOLLO_F, 	DEV_UNIX_F,	DEV_OFFSET_F,	DEV_FLOPPY },
#endif	/* defined(DEV_UNIX_F)  && defined(DEV_OFFSET_F)  */

	{ (char *) 0,   	(char *) 0,	0,      	0 }
};

char	fu_dtype[] = "invalid device type";
char	fu_dunit[] = "invalid device unit";

extern char	mode_read[];

extern int	verbose;

#ifdef	__STDC__
void	open_tape(char *path, int type)
#else
void	open_tape(path, type)
char	*path;
int	type;
#endif	/* __STDC__ */
{
	devname = path;
	devtype = type;

	switch (devtype) {
		case DEV_STDIN:
			reset_id();

			rfp = stdin;
			break;
		case DEV_TAPE:
		case DEV_FLOPPY:
			if (*devname != SLASH) {
				devnametab	*dnp;

				if (*devname == 'c') {
					dblocksz = CT_BLKSIZE;
					bufsz = CT_BLKSIZE;
					bf = 16;
				}

				for (dnp = devnametable;
					dnp->dn_apollo != (char *) 0;
						dnp++) {
					int	unit;

					unit = match_tape(devname,
					                  dnp->dn_apollo);
#ifdef sgi
					if (*devname == 'c' && unit == 0) {
						devname=devunix;
						(void) sprintf(devname, DEV_CT_DEF);
					} else if (*devname == 'm' && unit == 0) {
						devname=devunix;
						(void) sprintf(devname, DEV_MT_DEF);
					} else
#endif

					if (unit >= 0) {
						devname = devunix;
						devtype = dnp->dn_type;

						unit += dnp->dn_offset;
						(void) sprintf(devname,
						               dnp->dn_unix,
						               unit);

						break;
					}
				}

				if (devname != devunix)
					erroru(fu_dtype);
			}
			/* fall through to next case */
		case DEV_FILE:
			reset_id();

			if ((rfp = fopen(devname, mode_read)) == (FILE *) NULL)
				errorp(devname);
			break;
		case DEV_REMCT:
			dblocksz = CT_BLKSIZE;
			bufsz = CT_BLKSIZE;
			devtype = DEV_REMOTE;
			/* fall through to next case */
		case DEV_REMOTE:
			rmt_open();

			reset_id();
			break;
		default:
			errorm(fu_dtype);
			break;
	}
	if (verbose) printf("Device: %s\n",devname);
}

#ifdef	__STDC__
int	match_tape(char *s, char *d)
#else
int	match_tape(s, d)
char	*s;
char	*d;
#endif	/* __STDC__ */
{
	while (*s == *d) {
		if (*s == CNULL)
			return(0);

		s++;
		d++;
	}

	if (*d == CNULL) {
		int	c = DEV_MAXDIGITS;
		int	m = 0;
		int	n = 0;

		while (*s != CNULL) {
			if (!isdigit(*s))
				if (m > 0)
					erroru(fu_dunit);
				else
					return(-1);

			m++;

			if ((n > 0) || !iszero(*s)) {
				if (c-- <= 0)
					erroru(fu_dunit);

				n *= 10;
				n += todigit(*s);
			}

			s++;
		}

		return(n);
	}

	return(-1);
}

#define	mt_rew()	cmd_tape(MTREW, 1)
#define	mt_fsf(count)	cmd_tape(MTFSF, count)
#define	mt_off()	cmd_tape(MTOFFL, 1)
#ifdef	MTRETEN
#define	mt_ret()	cmd_tape(MTRETEN, 1)
char	rt_mesg[] = "\nRetensioning cartridge tape... Please wait.\n";
#endif	/* MTRETEN */

ansitab	ansitable[] = {
	/*  ident	  idswab	  action	   vol req	count */
	{ VOL1_IDENT,	VOL1_IDSWAB,	ansi_VOL1,	    1,  1,	  0 },
	{ VOL2_IDENT,	VOL2_IDSWAB,	ansi_VOL2,	    1,  0,	  0 },
	{ UVL1_IDENT,	UVL1_IDSWAB,	ansi_UVL1,	    1,  0,	  0 },
	{ HDR1_IDENT,	HDR1_IDSWAB,	ansi_HDR1,	    0,  1,	  0 },
	{ HDR2_IDENT,	HDR2_IDSWAB,	ansi_HDR2,	    0,  1,	  0 },
	{ HDR3_IDENT,	HDR3_IDSWAB,	ansi_HDR3,	    0,  0,	  0 },
	{ HDR4_IDENT,	HDR4_IDSWAB,	ansi_HDR4,	    0,  0,	  0 },
	{ UHL1_IDENT,	UHL1_IDSWAB,	ansi_UHL1,	    0,  1,	  0 },
	{ (char *) 0,	(char *) 0,	(void (*)()) 0,	    0,  0,	  0 }
};

#define elemsize(s,e)	sizeof(((s *) 0)->e)

char	*ansi_ident;

char	ansi_vid[elemsize(VOL1,v1_label)+1];
char	ansi_own[elemsize(VOL1,v1_owner)+1];
char	ansi_fid[elemsize(HDR1,h1_name)+1];

int	ansi_fileno;
int	ansi_filesec = 1;

int	ansi_year;
int	ansi_yday;

time_unix	ansi_time;

extern int	req_fileno;
extern char	*req_fid;

extern int	opt_reo;
extern int	opt_anys;

extern int	opt_sla;

extern int	opt_reten;
extern int	opt_rewind;
extern int	opt_unload;

#ifdef	__STDC__
void	find_tape(void)
#else
void	find_tape()
#endif	/* __STDC__ */
{
	int	tape_fileno;

	if ((devtype != DEV_TAPE) && (devtype != DEV_REMOTE))
		return;

	if (req_fileno >= 0) {
		mt_rew();

#ifdef	MTRETEN
		if (opt_reten && dblocksz==CT_BLKSIZE) {
			(void) printf(rt_mesg);
			mt_ret();
			mt_rew();
		}
#endif	/* MTRETEN */

		tape_fileno =  1;
	} else
		tape_fileno = -1;

	while (1) {
		ansitab	*ap;
		ansibuf	buf;

		int	count = 0;

		for (ap = ansitable; ap->at_ident != (char *) 0; ap++)
			ap->at_count = 0;

		while (read_tape((char *) &buf, bufsz)) {
			if (verbose>1) shobuf((char *)&buf,80);

#define SBMATCH(x) (strncmp((x), buf.ab_SYSB.sb_ident, strlen(x)) == 0)

			if (SBMATCH(SYSB_IDENT) || SBMATCH(SYSB_IDSWAB)) {
				skip_sysboot();
				continue;
			}

			ansi_ident = buf.ab_ident;

			for (ap = ansitable; ap->at_ident != (char *) 0; ap++) {
				int	found = 0;

#define	IDMATCH(id) \
	(strncmp((id), buf.ab_ident, (str_size_t) sizeof(buf.ab_ident)) == 0)

				if (IDMATCH(ap->at_ident)) {
					++found;
				} else if (IDMATCH(ap->at_idswab)) {
#ifdef	DEBUGSWAP
#define	DEBUGSFMT	"%s: DEBUG: swap header: %s -> %s\n"

					(void) fprintf(stderr,
					               DEBUGSFMT,
					               prog,
					               ap->at_idswab,
					               ap->at_ident);
#endif	/* DEBUGSWAP */
					swap_tape((char *) &buf, sizeof(buf));
					++found;
				}

				if (found) {
					if (ap->at_count++ > 0)
						errora("duplicate ANSI header");

					break;
				}
			}

			/*
			  On SGI, using DAT, intermittently get a read of garbage
			  after the valid headers, which generates an error here.
			  But the tape is actually good, so best to just continue.
			*/
			if (ap->at_count == 0) break;
				/*errora("unrecognized ANSI header");*/

			if (ap->at_vol && (tape_fileno > 1))
				errora("unexpected ANSI volume header");

			(*ap->at_action)(&buf);

			count++;
		}

		if (count == 0) {
			(void) fprintf(stderr, "%s: file ", prog);

			if (req_fileno > 0)
				(void) fprintf(stderr, "#%d, ", req_fileno);

			if (req_fid != (char *) 0)
				(void) fprintf(stderr, "\"%s\", ", req_fid);

			(void) fprintf(stderr, "section #%d ", ansi_filesec);
			(void) fprintf(stderr, "not found\n");

			exit(E_USAGE);
			/*NOTREACHED*/
		}

		if (devtype != DEV_REMOTE)
			clearerr(rfp);	/* clear EOF for subsequent reads */

		for (ap = ansitable; ap->at_ident != (char *) 0; ap++)
			if (ap->at_req && (ap->at_count == 0) &&
			    (!ap->at_vol || (tape_fileno == 1))) {
				ansi_ident = ap->at_ident;
				errora("missing ANSI header");
			}

		if (req_fileno < 0) {
			req_fileno = ansi_fileno;
			tape_fileno = ansi_fileno;
		}

		if (ansi_fileno != tape_fileno)
			errora("file number mismatch");

		if ((req_fileno != 0) &&
		    (req_fileno == ansi_fileno))
			break;

		if ((req_fid != (char *) 0) &&
		    (strcmp(req_fid, ansi_fid) == 0))
			break;

		mt_fsf(2);

		tape_fileno++;
	}

	{
		struct tm	*tmp = localtime(&ansi_time);

		ansi_ident = HDR1_IDENT;


		if (opt_sla) {
#ifdef	TZ_APOLLO
			cal_$timezone_rec_t	tz;

			cal_$get_info(&tz);
#endif	/* TZ_APOLLO */
			if (tmp->tm_year != ansi_year)
				errora("file creation year mismatch");

			/* Sometimes differ by 1, sometimes 0 */
			if ( (tmp->tm_yday != (ansi_yday-1)) &&
			     (tmp->tm_yday !=  ansi_yday) )
				errora("file creation date mismatch");

			(void) putchar(NL);
			(void) printf("Label:\n");
			(void) printf("   Volume ID:     %s\n", ansi_vid);
			(void) printf("   Owner ID:      %s\n",
			                (*ansi_own ? ansi_own :
                                                   "(no owner specified)"));
			(void) printf("   File number:   %d\n", ansi_fileno);
			(void) printf("   File section:  %d\n", ansi_filesec);
			(void) printf("   File ID:       %s\n",
			                (*ansi_fid ? ansi_fid :
                                                   "(no id specified)"));
			(void) printf("   File written:  ");
			(void) printf("%4d/%02d/%02d %02d:%02d:%02d %s\n",
			                  tmp->tm_year+1900,
			                  tmp->tm_mon+1,
			                  tmp->tm_mday,
			                  tmp->tm_hour,
			                  tmp->tm_min,
			                  tmp->tm_sec,
#ifdef	TZ_APOLLO
			                  tz.tz_name
#endif	/* TZ_APOLLO */
#ifdef	TZ_SYSV
			                  tzname[tmp->tm_isdst]
#endif	/* TZ_SYSV */
#ifdef	TZ_BSD
			                  tmp->tm_zone
#endif	/* TZ_BSD */
			             );
			(void) putchar(NL);
		}
	}
}

char	tbuf[BLOCK_SIZE];	/* tape (or disk) block buffer */
char	*tp;			/* pointer to current record in tape block */
int	crec;			/* no. records read from tape */

#ifdef	__STDC__
int	read_tape(char *buf, int size)
#else
int	read_tape(buf, size)
char	*buf;
int	size;
#endif	/* __STDC__ */
{

	/*
	   On SGI, read after EOF sometimes returns crec>0,
	   so need to compute when to return(0)
	*/
	if (brec>0 && brec==crec && crec<bf) { brec = 0; crec = 0; return(0); }
	if (brec >= crec) brec = 0;
	if (brec == 0) {
		if (devtype == DEV_REMOTE)
			crec = rmt_read(tbuf, size);
		else {
			crec = fread((fread_buf_t)  tbuf,
				     (fread_size_t) size,
				     (fread_size_t) bf,
						    rfp);
			if (ferror(rfp))
				errorm("tape read error");
		}
		tp = tbuf;
	}
	brec++;
	if (crec) { for(; size; size--) *buf++ = *tp++; }

	return(crec);
}

#ifndef	SWAB_OVERLAP
#ifdef	__STDC__
void	swap_tape(char *buf, int size)
#else
void	swap_tape(buf, size)
char	*buf;
int	size;
#endif	/* __STDC__ */
{
	char	*sp = buf;
	char	*dp = buf;

	size /= 2;

	while (size-- > 0) {
		char	ctemp;

		ctemp = *sp++;
		*dp++ = *sp++;
		*dp++ = ctemp;
	}
}
#endif	/* SWAB_OVERLAP */

#ifdef	__STDC__
void	close_tape(void)
#else
void	close_tape()
#endif	/* __STDC__ */
{
	if (opt_rewind)
		mt_rew();

	if (opt_unload)
		mt_off();

	if (devtype == DEV_REMOTE)
		rmt_close();
	else {
		if (fclose(rfp) == EOF)
			errorp(devname);
	}
}

#ifdef	__STDC__
void	cmd_tape(int op, int count)
#else
void	cmd_tape(op, count)
int	op;
int	count;
#endif	/* __STDC__ */
{
	if (devtype == DEV_REMOTE)
		rmt_ioctl(op, count);
	else {
		struct mtop	mtop_cmd;

		mtop_cmd.mt_op = op;
		mtop_cmd.mt_count = count;

		if (ioctl(fileno(rfp), MTIOCTOP, (char *) &mtop_cmd) < 0)
			errorp(devname);
	}
}

#ifdef	__STDC__
void	ansi_VOL1(ansibuf *bp)
#else
void	ansi_VOL1(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
	if (!opt_sla) return;

	(void) strtrim(ansi_vid, bp->ab_VOL1.v1_label, elemsize(VOL1,v1_label));

	if (bp->ab_VOL1.v1_access != VOL1_ACCESS)
		errora("invalid volume access code");

	(void) strtrim(ansi_own, bp->ab_VOL1.v1_owner, elemsize(VOL1,v1_owner));

	if (bp->ab_VOL1.v1_ansi_level != VOL1_ANSI_LEVEL)
		errora("invalid ANSI standard level");
}

/*ARGSUSED*/
#ifdef	__STDC__
void	ansi_VOL2(ansibuf *bp)
#else
void	ansi_VOL2(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
}

char	ft_apollo[] = FORMAT_TIME_APOLLO;

#ifdef	__STDC__
void	ansi_UVL1(ansibuf *bp)
#else
void	ansi_UVL1(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
	time_apollo	ta;

	if (!opt_sla) return;

	if (sscanf(bp->ab_UVL1.uv_time_apollo, ft_apollo,
	           &ta.ta_time, &ta.ta_extra) != 2)
		errora("invalid volume time");
}

char	ft_fileno[] = FORMAT_FILENO;
char	ft_date[] = FORMAT_DATE_ANSI;

#ifdef	__STDC__
void	ansi_HDR1(ansibuf *bp)
#else
void	ansi_HDR1(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
	(void) strtrim(ansi_fid, bp->ab_HDR1.h1_name, elemsize(HDR1,h1_name));

	if (strncmp(HDR1_FILESET,
	            bp->ab_HDR1.h1_fileset,
	            (str_size_t) elemsize(HDR1,h1_fileset)) != 0)
		errora("invalid file set");

/*
	if (!opt_anys && strncmp(HDR1_VOLUME_NUM,
	            bp->ab_HDR1.h1_volume_num,
	            (str_size_t) elemsize(HDR1,h1_volume_num)) != 0)
		errora("invalid volume number");
*/
	sscanf(bp->ab_HDR1.h1_volume_num,"%4d",&ansi_filesec);

	if ((sscanf(bp->ab_HDR1.h1_file_num, ft_fileno, &ansi_fileno) != 1) ||
	    (ansi_fileno <= 0))
		errora("invalid file number");

	if (!opt_sla) return;

	if (strncmp(HDR1_GEN,
	            bp->ab_HDR1.h1_gen,
	            (str_size_t) elemsize(HDR1,h1_gen)) != 0)
		errora("invalid generation number");

	if (strncmp(HDR1_GENVER,
	            bp->ab_HDR1.h1_genver,
	            (str_size_t) elemsize(HDR1,h1_genver)) != 0)
		errora("invalid generation version number");

	if (sscanf(bp->ab_HDR1.h1_created, ft_date,
	           &ansi_year, &ansi_yday) != 2)
		errora("invalid creation date");

	if (strncmp(bp->ab_HDR1.h1_created,
	            bp->ab_HDR1.h1_expires,
	            (str_size_t) elemsize(HDR1,h1_expires)) != 0)
		errora("invalid expiration date");

	if (bp->ab_HDR1.h1_access != HDR1_ACCESS)
		errora("invalid file access code");

	if (strncmp(HDR1_BLOCKCOUNT,
	            bp->ab_HDR1.h1_blockcount,
	            (str_size_t) elemsize(HDR1,h1_blockcount)) != 0)
		errora("invalid block count");

	if (strncmp(HDR1_TAPESYS,
	            bp->ab_HDR1.h1_tapesys,
	            (str_size_t) elemsize(HDR1,h1_tapesys)) != 0)
		errora("invalid tape system");
}

#ifdef	__STDC__
void	ansi_HDR2(ansibuf *bp)
#else
void	ansi_HDR2(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{

	int blocklen,reclen;

	if (!opt_sla) return;

	if (bp->ab_HDR2.h2_recfmt != HDR2_RECFMT)
		errora("invalid record format");

	sscanf(bp->ab_HDR2.h2_blocklen,"%5d",&blocklen);
	if (dblocksz != blocklen)
		errora("invalid block length");

	sscanf(bp->ab_HDR2.h2_reclen,"%5d",&reclen);
	if (dblocksz != reclen)
		errora("invalid record length");

	if (bp->ab_HDR2.h2_density != HDR2_DENSITY)
		errora("invalid tape density");

	if (bp->ab_HDR2.h2_vol_switch != HDR2_VOL_SWITCH)
		errora("invalid volume switch");

	if (strncmp(HDR2_JOB,
	            bp->ab_HDR2.h2_job,
	            (str_size_t) elemsize(HDR2,h2_job)) != 0)
		errora("invalid job name");

	if (strncmp(HDR2_RECORDING,
	            bp->ab_HDR2.h2_recording,
	            (str_size_t) elemsize(HDR2,h2_recording)) != 0)
		errora("invalid recording mode");

	if (bp->ab_HDR2.h2_carriage_control != HDR2_CARRIAGE_CONTROL)
		errora("invalid carriage control");

	if (bp->ab_HDR2.h2_blocked_records != HDR2_BLOCKED_RECORDS)
		errora("invalid blocked records indicator");

	if (strncmp(HDR2_BLOCK_OFFSET,
	            bp->ab_HDR2.h2_block_offset,
	            (str_size_t) elemsize(HDR2,h2_block_offset)) != 0)
		errora("invalid block offset");
}

/*ARGSUSED*/
#ifdef	__STDC__
void	ansi_HDR3(ansibuf *bp)
#else
void	ansi_HDR3(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
}

/*ARGSUSED*/
#ifdef	__STDC__
void	ansi_HDR4(ansibuf *bp)
#else
void	ansi_HDR4(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
}

char	ft_gmdate[] = FORMAT_DATE_GMT;
char	ft_gmtime[] = FORMAT_TIME_GMT;

#ifdef	__STDC__
void	ansi_UHL1(ansibuf *bp)
#else
void	ansi_UHL1(bp)
ansibuf	*bp;
#endif	/* __STDC__ */
{
	if (!opt_sla) return;

	{
		time_apollo	ta;

		if (sscanf(bp->ab_UHL1.uh_time_apollo, ft_apollo,
		           &ta.ta_time, &ta.ta_extra) != 2)
			errora("invalid file time");

		ansi_time = time_atou(ta.ta_time);
	}

	{
		struct tm	tg;

		if (sscanf(bp->ab_UHL1.uh_date_gmt, ft_gmdate,
		           &tg.tm_year, &tg.tm_mon, &tg.tm_mday) != 3)
			errora("invalid file GMT date");

		if (sscanf(bp->ab_UHL1.uh_time_gmt, ft_gmtime,
		           &tg.tm_hour, &tg.tm_min, &tg.tm_sec)  != 3)
			errora("invalid file GMT time");
	}
}

#ifdef	__STDC__
void	reset_id(void)
#else
void	reset_id()
#endif	/* __STDC__ */
{
	if (setuid(getuid()) < 0)
		errorm("setuid botch");

	if (setgid(getgid()) < 0)
		errorm("setgid botch");
}

#ifdef	__STDC__
char	*strtrim(char *s1, char *s2, int n)
#else
char	*strtrim(s1, s2, n)
char	*s1;
char	*s2;
int	n;
#endif	/* __STDC__ */
{
	char	*s1min = s1;
	char	*s2max = s2 + n;

	while ((s2 <  s2max) && (*s2 != CNULL))
		*s1++ = *s2++;

	*s1-- = CNULL;

	while ((s1 >= s1min) && (*s1 == SPACE))
		*s1-- = CNULL;

	return(s1min);
}
#ifdef	__STDC__
void	skip_sysboot(void)
#else
void	skip_sysboot()
#endif	/* __STDC__ */
{
	static char	sb_mesg[] = "\nSkipping over SYSBOOT found at beginning of volume.\n";

	(void) printf(sb_mesg);
	brec = 0;			/* clear buffered records */ 
	mt_rew();
	mt_fsf(1);
}

#ifdef	__STDC__
int	trailers(void)
#else
int	trailers()
#endif	/* __STDC__ */
{
	extern 	FILE *rfp;
	ansibuf	buf;
	int	eovt = 0, eoft = 0;
	
#define IDEMATCH(x) (strncmp((x), buf.ab_EOFV.ef_ident, strlen(x)) == 0)

	if ((devtype != DEV_TAPE) && (devtype != DEV_REMOTE))
		return(0);
	brec = 0;			/* clear buffered records */ 
	if (devtype != DEV_REMOTE) clearerr(rfp);

	while (read_tape((char *) &buf, bufsz)) {
		if (verbose>1) shobuf((char *)&buf,80);
		if (IDEMATCH(EOF1_IDENT) || IDEMATCH(EOF1_IDSWAB) ) eoft++;
		if (IDEMATCH(EOF2_IDENT) || IDEMATCH(EOF2_IDSWAB) ) eoft++;
		if (IDEMATCH(EOV1_IDENT) || IDEMATCH(EOV1_IDSWAB) ) eovt++;
		if (IDEMATCH(EOV2_IDENT) || IDEMATCH(EOV2_IDSWAB) ) eovt++;
	}
	brec = 0;			/* clear buffered records */ 
	if (devtype != DEV_REMOTE) clearerr(rfp);
	return(eovt);
}

#ifdef	__STDC__
void	tape_action_only(void)
#else
void	tape_action_only()
#endif	/* __STDC__ */
{
	if ((devtype != DEV_TAPE) && (devtype != DEV_REMOTE))
		return;
#ifdef	MTRETEN
	if (opt_reten>0 && dblocksz==CT_BLKSIZE) {
		(void) printf(rt_mesg);
		mt_ret();
	}
#endif	/* MTRETEN */
	close_tape();
	exit(0);
}

/* hack: char buffer displayer - for debug */
#ifdef	__STDC__
void	shobuf(char *ip, int sz)
#else
void	shobuf(ip, sz)
char	*ip;
int	sz;
#endif	/* __STDC__ */
{
	int	ii;
	printf("[");
	for (ii=0; ii<sz; ii++,ip++){
		if (*ip>31 && *ip<127) printf("%c",*ip);/* ASCII printables */
		else if (*ip==0) printf("~");		/* null */
		else printf(".");			/* non-printables */
	}
	printf("]\n"); (void)fflush(NULL);
}
