/*
 *	rbak.c - command line parsing and file operations
 */

#include "lang.h"
#include "std.h"
#include "sio.h"
#include "char.h"
#include "str.h"
#include "order.h"
#include "lim.h"
#include "finfo.h"
#include "pathnames.h"
#include "timestamp.h"
#include "format.h"
#include "type_uids.h"
#include "rbak.h"
#include "tape.h"
#include "ansitape.h"
#include "remove.h"
#include "error.h"

/*ARGSUSED*/
#ifdef	__STDC__
int	main(int argc, char **argv)
#else
int	main(argc, argv)
int	argc;
char	**argv;
#endif	/* __STDC__ */
{
	set_opts(argv);
	run_rbak();

	exit(E_NONE);
	/*NOTREACHED*/
}

char	cur_fileno[] = "cur";

int	req_fileno;
char	*req_fid;

int	opt_reo;
int	opt_anys = 0;

int	mode_index = -1;
int	mode_exist = -1;

int	list_dir;
int	list_file;
int	list_link;

int	verbose = 0;
int	downcase = 1;

int	opt_recconv = 1;
int	opt_types = 1;

int	opt_sla = 1;
int	opt_sacl;
int	opt_pdt;

int	opt_reten = -1;
int	opt_rewind;
int	opt_unload;

int	all_path;
pathtab	*pathtabroot = (pathtab *) 0;

extern	int	devtype;

extern	int	dblocksz;
extern	int 	bf;

char	fu_rmult[] = "more than one restore mode was specified";
char	fu_emult[] = "more than one preexisting mode was specified";
char	fu_imult[] = "only one backup media specification is allowed";
char	fu_dnone[] = "device type is missing";
char	fu_rnone[] = "remote host name and device path are missing";
char	fu_aspre[] = "invalid -as option: no pathname precedes it";
char	fu_anone[] = "invalid -as option: pathname required";
char	fu_knone[] = "invalid -pr option: pathname(s) required";
char	fu_aprms[] = "-pr is allowed only when used with -ms";
char	fu_fnone[] = "-from requires a pathname as its argument";
char	fu_allpn[] = "-all is not allowed when pathnames are specified";
char	fu_allre[] = "redundant -all specification";
char	fu_nfinv[] = "invalid backup file number";
char	fu_nzero[] = "backup file number must be greater than zero";
char	fu_nnone[] = "-f[ile] option not followed by backup file number";
char	fu_inone[] = "file id does not follow -f[ile]id option";
char	fu_consn[] = "conflicting arguments: -stdin and -f[ile]";
char	fu_consi[] = "conflicting arguments: -stdin and -f[ile]id";
char	fu_confn[] = "conflicting arguments: -from and -f[ile]";
char	fu_confi[] = "conflicting arguments: -from and -f[ile]id";
char	fu_redni[] = "redundant backup file number or id specification";
char	fu_inret[] = "invalid options: -reten and -nreten are incompatible";
char	fu_fcret[] = "invalid options: -f[ile] cur and -reten are incompatible";
char	fu_reret[] = "invalid options: -reo and -reten are incompatible";
char	fu_valun[] = "option -unload is valid only for tapes";
char	fu_valrw[] = "option -rewind is valid only for tapes";
char	fu_valrt[] = "option -reten is valid only for tapes";
char	fu_valnr[] = "option -nreten is valid only for tapes";

/*
Just because a type is defined here doesn't mean it's been properly
tested.  Types that aren't defined show up as "unknown", but will 
most likely restore OK unless there is a header.
*/
uid_tab uid_table[] = {
	/* Most common first as this is sequentially searched */
	/*  uid value	    description     header size */
	{ {UID_UNSTRUCT } , DESC_UNSTRUCT   ,  0 },
	{ {UID_UASC     } , DESC_UASC       , 32 },
	{ {UID_REC      } , DESC_REC        , 32 },
	{ {UID_HDRU     } , DESC_HDRU       , 32 },
	{ {UID_COFF     } , DESC_COFF       ,  0 },
	{ {UID_OBJ      } , DESC_OBJ        ,  0 },
	{ {UID_NIL      } , DESC_NIL        ,  0 },
	{ {UID_COMPRESS } , DESC_COMPRESS   ,  0 },
	{ {UID_BITMAP   } , DESC_BITMAP     ,  0 },
	{ {UID_CASE_HM  } , DESC_CASE_HM    ,  0 },
	{ {UID_CMPEXE   } , DESC_CMPEXE     ,  0 },
	{ {UID_D3M_AREA } , DESC_D3M_AREA   ,  0 },
	{ {UID_D3M_SCH  } , DESC_D3M_SCH    ,  0 },
	{ {UID_DIR      } , DESC_DIR        ,  0 },
	{ {UID_DM_EDIT  } , DESC_DM_EDIT    ,  0 },
	{ {UID_IPAD     } , DESC_IPAD       ,  0 },
	{ {UID_MBX      } , DESC_MBX        ,  0 },
	{ {UID_MT       } , DESC_MT         ,  0 },
	{ {UID_NULL     } , DESC_NULL       ,  0 },
	{ {UID_PAD      } , DESC_PAD        ,  0 },
	{ {UID_PIPE     } , DESC_PIPE       ,  0 },
	{ {UID_PTY_SLAVE} , DESC_PTY_SLAVE  ,  0 },
	{ {UID_PTY      } , DESC_PTY        ,  0 },
	{ {UID_SIO      } , DESC_SIO        ,  0 },
	{ {UID_TCP      } , DESC_TCP        ,  0 },
	{ {UID_OSIO     } , DESC_OSIO       ,  0 },
	{ {UID_DEV_TTY  } , DESC_DEV_TTY    ,  0 },
	{ {-1, -1       } , ""              , -1 }
};


#define	match_opts(x)	(strcmp(*opts, x) == 0)
#define	Case_opts(x)	if match_opts(x)
#define	case_opts(x)	else Case_opts(x)
#define	Case_alts(x, y)	if (match_opts(x) || match_opts(y))
#define	case_alts(x, y)	else Case_alts(x, y)
#define	default_opts	else if (**opts == DASH)
#define	default_path	else

#ifdef	__STDC__
void	set_opts(char **opts)
#else
void	set_opts(opts)
char	**opts;
#endif	/* __STDC__ */
{
	if ((prog = rindex(*opts, SLASH)) == (char *) 0)
		prog = *opts;
	else
		++prog;

	while (*++opts != (char *) 0)
		Case_opts("-dev")
			{
				if (devtype != DEV_NONE)
					erroru(fu_imult);

				if (*++opts == (char *) 0)
					erroru(fu_dnone);

				open_tape(*opts, DEV_TAPE);
			}
		case_alts("-rem", "-remote")
			{
				if (devtype != DEV_NONE)
					erroru(fu_imult);

				if (*++opts == (char *) 0)
					erroru(fu_rnone);

				open_tape(*opts, DEV_REMOTE);
			}
		case_alts("-remc", "-remct")
			{
				if (devtype != DEV_NONE)
					erroru(fu_imult);

				if (*++opts == (char *) 0)
					erroru(fu_rnone);

				open_tape(*opts, DEV_REMCT);
			}
		case_alts("-f", "-file")
			{
				if ((req_fileno != 0) ||
				    (req_fid != (char *) 0))
					erroru(fu_redni);

				if (devtype == DEV_STDIN)
					erroru(fu_consn);

				if (devtype == DEV_FILE)
					erroru(fu_confn);

				if (*++opts == (char *) 0)
					erroru(fu_nnone);

				if (strcmp(*opts, cur_fileno) == 0) {
					if (opt_reten > 0)
						erroru(fu_fcret);

					req_fileno = -1;
				} else {
					if (sscanf(*opts, "%d",
					           &req_fileno) != 1)
						erroru(fu_nfinv);

					if (req_fileno == 0)
						erroru(fu_nzero);

					if (req_fileno < 0)
						erroru(fu_nfinv);
				}
			}
		case_alts("-fid", "-fileid")
			{
				if ((req_fileno != 0) ||
				    (req_fid != (char *) 0))
					erroru(fu_redni);

				if (devtype == DEV_STDIN)
					erroru(fu_consi);

				if (devtype == DEV_FILE)
					erroru(fu_confi);

				if (*++opts == (char *) 0)
					erroru(fu_inone);

				req_fid = *opts;
			}
		case_opts("-anys")
			{
				opt_anys++;
			}
		case_opts("-reo")
			{
				if (opt_reten > 0)
					erroru(fu_reret);

				opt_reo++;
			}
		case_opts("-int")
			{
				if (mode_index >= 0)
					erroru(fu_rmult);

				mode_index = 0;
			}
		case_opts("-index")
			{
				if (mode_index >= 0)
					erroru(fu_rmult);

				mode_index = 1;
			}
		case_opts("-cr")
			{
				if (mode_exist >= 0)
					erroru(fu_emult);

				mode_exist = EX_CREATE;
			}
		case_opts("-r")
			{
				if (mode_exist >= 0)
					erroru(fu_emult);

				mode_exist = EX_REPLACE;
			}
		case_opts("-ms")
			{
				if (mode_exist >= 0)
					erroru(fu_emult);

				mode_exist = EX_MERGE_SOURCE;
			}
		case_opts("-md")
			{
				if (mode_exist >= 0)
					erroru(fu_emult);

				mode_exist = EX_MERGE_DEST;
			}
		case_opts("-pr")
			{
				int	kept = 0;

				if (mode_exist != EX_MERGE_SOURCE)
					erroru(fu_aprms);

				while ((*++opts != (char *) 0) &&
				              (**opts != DASH)) {
					add_keep(*opts);
					++kept;
				}

				--opts;

				if (!kept)
					erroru(fu_knone);
			}
		case_opts("-force")
			{
				/* ignore */
			}
		case_opts("-du")
			{
				/* ignore */
			}
		case_opts("-l")
			{
				list_dir++;
				list_file++;
				list_link++;
			}
		case_opts("-ld")
			{
				list_dir++;
			}
		case_opts("-lf")
			{
				list_file++;
			}
		case_opts("-ll")
			{
				list_link++;
			}
		case_opts("-v")
			{
				verbose++;
				list_file++;
			}
		case_alts("-ndc", "-nodowncase")
			{
				downcase = 0;
			}
		case_opts("-rc")
			{
				opt_recconv = 1;
			}
		case_opts("-nrc")
			{
				opt_recconv = 0;
			}
		case_opts("-nty")
			{
				opt_types = 0;
				opt_recconv = 0;
			}
		case_opts("-512")
			{
				dblocksz = 512;
			}
		case_opts("-sla")
			{
				opt_sla = 1;
			}
		case_opts("-nsla")
			{
				opt_sla = 0;
			}
		case_opts("-dacl")
			{
				opt_sacl = 0;
			}
		case_opts("-sacl")
			{
				opt_sacl = 1;
			}
		case_opts("-pdt")
			{
				opt_pdt++;
			}
		case_opts("-reten")
			{
				if (opt_reten == 0)
					erroru(fu_inret);

				if (opt_reo)
					erroru(fu_reret);

				if (req_fileno < 0)
					erroru(fu_fcret);

				if ((devtype == DEV_STDIN) ||
				    (devtype == DEV_FILE))
					erroru(fu_valrt);

				opt_reten = 1;
			}
		case_opts("-nreten")
			{
				if (opt_reten > 0)
					erroru(fu_inret);

				if ((devtype == DEV_STDIN) ||
				    (devtype == DEV_FILE))
					erroru(fu_valnr);

				opt_reten = 0;
			}
		case_opts("-rewind")
			{
				if ((devtype == DEV_STDIN) ||
				    (devtype == DEV_FILE))
					erroru(fu_valrw);

				opt_rewind++;
			}
		case_opts("-unload")
			{
				if ((devtype == DEV_STDIN) ||
				    (devtype == DEV_FILE))
					erroru(fu_valun);

				opt_unload++;
			}
		case_opts("-from")
			{
				if (devtype != DEV_NONE)
					erroru(fu_imult);

				if (req_fileno != 0)
					erroru(fu_confn);

				if (req_fid != (char *) 0)
					erroru(fu_confi);

				if (opt_reten > 0)
					erroru(fu_valrt);

				if (opt_reten == 0)
					erroru(fu_valnr);

				if (opt_rewind)
					erroru(fu_valrw);

				if (opt_unload)
					erroru(fu_valun);

				if (*++opts == (char *) 0)
					erroru(fu_fnone);

				open_tape(*opts, DEV_FILE);
			}
		case_opts("-stdin")
			{
				if (devtype != DEV_NONE)
					erroru(fu_imult);

				if (req_fileno != 0)
					erroru(fu_consn);

				if (req_fid != (char *) 0)
					erroru(fu_consi);

				if (opt_reten > 0)
					erroru(fu_valrt);

				if (opt_reten == 0)
					erroru(fu_valnr);

				if (opt_rewind)
					erroru(fu_valrw);

				if (opt_unload)
					erroru(fu_valun);

				open_tape("(stdin)", DEV_STDIN);
			}
		case_opts("-all")
			{
				if (pathtabroot != (pathtab *) 0)
					erroru(fu_allpn);

				if (all_path)
					erroru(fu_allre);

				all_path++;
			}
		case_opts("-as")
			{
				if (*++opts == (char *) 0)
					erroru(fu_anone);

				rename_path(*opts);
			}
		default_opts
			{
				usage();
			}
		default_path
			{
				if (all_path)
					erroru(fu_allpn);

				add_path(*opts);
			}

	if (devtype == DEV_NONE)
		open_tape(DEV_APOLLO_DEF, DEV_TAPE);

	if ( (!all_path && (pathtabroot == (pathtab *) 0)) &&
             (req_fileno == 0) && (req_fid == (char *) 0) &&
	     (mode_index < 0) && (opt_reten>0 || opt_rewind || opt_unload))
		tape_action_only(); /* no return */

	if (mode_index < 0)
		mode_index = 0;

	if (mode_exist < 0)
		mode_exist = EX_CREATE;

	if (opt_reten < 0)
		opt_reten = 0;

	if (!all_path && (pathtabroot == (pathtab *) 0))
		ask_path();

	if ((req_fileno == 0) && (req_fid == (char *) 0) &&
	    ((devtype == DEV_TAPE) || (devtype == DEV_REMOTE)))
		ask_fileno();
}

char	block[BLOCK_SIZE];
long	seq = 0L;

int	pcount;
int	plevel;

int	alevel;
int	blevel;

int	dlevel;
char	dname[MAXPATHLEN];
attrib	dattrib[MAXDLEVEL];

char	dot[] = ".";

char	fe_toomany[] = "too many names were found in the backup file";
char	fe_notfound[] =
"The following names were requested but were not found in the backup file";

#ifdef	__STDC__
void	run_rbak(void)
#else
void	run_rbak()
#endif	/* __STDC__ */
{
	int tape_file_open = 1;

	find_tape();
	if (verbose>1) printf("Data block size = %d \n\n",dblocksz);

	plevel = 0;

	alevel = 0;
	blevel = 0;

	dlevel = 0;
	(void) strcpy(dname, dot);

	if (mode_index)
		(void) printf("Starting index:\n\n");
	else {
		(void) printf("Starting restore:\n");

		if (list_dir || list_file || list_link)
			(void) putchar(NL);

		(void) dir_attrib(dname, &dattrib[dlevel]);
	}

	while (tape_file_open) {
		while (read_tape(block, dblocksz)) {
			++seq;

			if (read_block() < 0) {
				tape_file_open = 0;
				break;
			}
		}
		if ( (tape_file_open) && 
		     (tape_file_open = trailers()) && 
		     (tape_file_open = ask_vol_swap())) 
			find_tape();
	}

	if (mode_index)
		(void) printf("\nIndex complete.\n");
	else {
		if (list_dir || list_file || list_link)
			(void) putchar(NL);

		(void) printf("Restore complete.\n");
	}

	if (pcount < 0)
		errorm(fe_toomany);

	if (pcount > 0) {
		pathtab	*pp;

		(void) fprintf(stderr, "\n%s:\n", fe_notfound);

		for (pp = pathtabroot; pp != (pathtab *) 0; pp = pp->pt_next)
			if (!pp->pt_found)
				(void) fprintf(stderr, "   %s\n", pp->pt_name);

		(void) putc(NL, stderr);

		exit(E_ANSI);
		/*NOTREACHED*/
	}

	close_tape();
}

FILE	*fp;
char	fname[MAXPATHLEN];
attrib	fattrib;
long	fsize;
int	fcount;
	char	*type_name;
	long	Fsize;

char	fi_dir[]    = "(dir)  %s\n";
char	fi_file[]   = "(file) %s\n";
char	fi_file_v[] = "(file) %s  (%s %d)\n";
char	fi_link[]   = "(link) %s\n";
char	fi_link_v[]   = "(link) %s  -> %s\n";

char	fl_dir[]    = "(dir)  \"%s\" restored.\n";
char	fl_file[]   = "(file) \"%s\" restored.\n";
char	fl_file_v[] = "(file) \"%s\" (%s %d) restored.\n";
char	fl_link[]   = "(link) \"%s\" restored.\n";

int	swapping;
long 	hdr_size = 0;

int	rc_ptr = 0;
int	rc_count = 0;

#define	SEQFIRST	0x00000001L
#define	SEQFSWAP	0x00000100L

#ifdef	__STDC__
int	read_block(void)
#else
int	read_block()
#endif	/* __STDC__ */
{
	char	*bp = block;
	char	*bplast;

	uid__t	fuid;
	static	int do_rcv = 0;

	if ((seq == SEQFIRST) && (get_long(bhp(bp)->bh_seq) == SEQFSWAP)) {
#ifdef	DEBUGSWAP
#define	DEBUGSFMT	"%s: DEBUG: swapping data\n"

		(void) fprintf(stderr,
		               DEBUGSFMT,
		               prog);
#endif	/* DEBUGSWAP */

		++swapping;
	}

	if (swapping)
		swap_tape(block, dblocksz);

	if (get_long(bhp(bp)->bh_seq) != seq) {
		/*fprintf(stderr,"expected %d, found %d\n",seq,get_long(bhp(bp)->bh_seq));*/
		if (opt_anys) seq = get_long(bhp(bp)->bh_seq);
		else errors("invalid sequence number");
	}

	{
		short	bsize = get_short(bhp(bp)->bh_size);

		if ((bsize > dblocksz) || (bsize <= 0))
			errors("invalid block size");

		bplast = bp + bsize;
	}

	bp += sizeof(A_block_header);

	while (bp < bplast) {
		long	mtype = MAGICTYPE(get_short(mhp(bp)->mh_type1),
			                  get_short(mhp(bp)->mh_type2));
		short	msize = get_short(mhp(bp)->mh_size);

#define	namelen(n)	(msize - ((n) - bp))
#define	copy_name(d, s)	copy_namelen(d, s, namelen(s))

		if ((msize > ((bplast-bp)-sizeof(A_magic_header))) ||
		    (msize <= 0))
			errors("invalid data size");

		bp += sizeof(A_magic_header);

#ifdef	DEBUGDUMP
#define	DEBUGDHDR	"%s: DEBUG: type = (%d,%d), size = %d\n"
#define	DEBUGDHEX	"%s: DEBUG: addr:\
  0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f   0123456789abcdef"
#define	DEBUGDOFF	"%s: DEBUG: %04x:"
#define	DEBUGDDAT	" %02x"

		(void) fprintf(stderr,
		               DEBUGDHDR,
		               prog,
		               MAJORTYPE(mtype),
		               MINORTYPE(mtype),
		               msize);

		(void) fprintf(stderr,
		               DEBUGDHEX,
		               prog);

		{
			unsigned short	off = 0;
			unsigned char	*dp = bp;

			while (off < ((unsigned) msize)) {
				if ((off & 0xf) == 0x0) {
					if (off > 0) {
						unsigned char	*cp = dp - 0x10;

						(void) fputs(" | ", stderr);

						while (cp < dp) {
							(void) putc(dchar(*cp),
							            stderr);
							++cp;
						}
					}

					(void) putc(NL, stderr);

					(void) fprintf(stderr,
					               DEBUGDOFF,
					               prog,
					               off);
				}

				(void) fprintf(stderr,
				               DEBUGDDAT,
				               *dp);

				++off;
				++dp;
			}

			{
				unsigned char	*cp = dp - (--off & 0xf) - 1;

				while (++off & 0xf)
					(void) fputs("   ", stderr);

				(void) fputs(" | ", stderr);

				while (cp < dp) {
					(void) putc(dchar(*cp),
					            stderr);
					++cp;
				}
			}

			(void) putc(NL, stderr);
		}
#endif	/* DEBUGDUMP */

		switch (mtype) {
			case   SUB_MAGIC:	case   SUB_OLD_MAGIC:
			case EMPTY_MAGIC:
			case   OPT_MAGIC:	case   OPT_OLD_MAGIC:
			case   ACL_MAGIC:	case   ACL_OLD_MAGIC:
				/* just ignore */
				break;

			case  NAME_MAGIC:	case  NAME_OLD_MAGIC:
				copy_name(fname, nhp(bp)->nh_name);
				if (mtype==NAME_OLD_MAGIC && downcase) downcase_name(fname);
				break;

			case  FILE_MAGIC:	case  FILE_OLD_MAGIC:
				if (!sel_path(fname))
					break;

				fuid.high = get_long(fhp(bp)->fh_common.ch_type_high);
				fuid.low  = get_long(fhp(bp)->fh_common.ch_type_low);
				type_handler(&fuid,&hdr_size,&type_name);
				if (!opt_types) hdr_size = 0;

				if (do_rcv = (strcmp(type_name,DESC_REC)==0 && opt_recconv)) {
					rc_ptr = 0;
					rc_count = 0;
				}
				
				/*Fsize = get_long(fhp(bp)->fh_common.ch_size);*/ /* wrong for pre-SR10 */
				Fsize = get_size(bp, mtype) - hdr_size;
				/*printf("type=%4.4x %4.4x ",fuid.high,fuid.low);*/

				if (mode_index) {
					if (verbose) printf(fi_file_v, fname, type_name, Fsize);
					else (void) printf(fi_file, fname);
					break;
				}

				if (check_exist(fname, F_TYPE) < 0)
					break;

				if (!make_link(fname, bp, mtype)) {
					get_attrib(bp, mtype, &fattrib);

					fsize = get_size(bp, mtype);

					if (fsize < 0)
						errors("invalid file size");

					fcount = 0;

					make_dir(fname, F_TYPE);
					open_file(fname);

					if (fsize == 0)
						close_file(fname, &fattrib);
				} else {
					if (list_file) {
						if (verbose) (void) printf(fl_file_v, fname, type_name, Fsize);
						else (void) printf(fl_file, fname);
					} 
				}

				break;

			case  DATA_MAGIC:
				if (!mode_index && (fp != (FILE *) NULL)) {
					long	lsize = min(fsize, msize);
					long	wsize = lsize;
					char	*wp = bp;
					int	wstat;

					wsize -= min(lsize,hdr_size);
					wp += min(lsize,hdr_size);
					hdr_size -= min(lsize,hdr_size);

					if (wsize > 0L) {
					    if (do_rcv)
						wstat = convert_rec(wp, wsize);
					    else
					    	wstat = fwrite((fwrite_buf_t)  wp,
					    		(fwrite_size_t) wsize,
					    		(fwrite_size_t) 1, fp);
					    if (wstat == 0) errorp(fname);
					}

					fsize -= lsize;

					if (fsize == 0L)
						close_file(fname, &fattrib);
				}

				break;

			case   DIR_OLD_MAGIC:
				copy_name(dname, dohp(bp)->doh_name);
				if (downcase) downcase_name(dname);
				goto dcase;

			case   DIR_MAGIC:
				copy_name(dname, dhp(bp)->dh_name);
			dcase:
				if (++dlevel >= MAXDLEVEL)
					errors("too many directory levels");

				if (!mode_index)
					get_attrib(bp, mtype, &dattrib[dlevel]);

				if (!sel_path(dname))
					break;

				if (!all_path && (plevel <= 0))
					plevel = dlevel;

				if (mode_index) {
					(void) printf(fi_dir, dname);
					break;
				}

				if (check_exist(dname, D_TYPE) < 0) {
					if (plevel == dlevel)
						plevel = 0;

					break;
				}

				if (!make_link(dname, bp, mtype))
					make_dir(dname, D_TYPE);
				else {
					if (list_dir)
						(void) printf(fl_dir, dname);
				}

				break;

			case  POPD_MAGIC:	case  POPD_OLD_MAGIC:
				if (dlevel <= 0) {
					if (opt_anys) dlevel++;	/* UGH! */
					else errors("directory level botch");
				}

				if (!mode_index) {
					if ((all_path || (plevel > 0)) ||
					    ((alevel <= dlevel) &&
					     (dlevel <= blevel)))
						if (set_attrib(dname,
						               &dattrib[dlevel],
						               D_TYPE) < 0)
							errorp(dname);

					if (list_dir &&
					    (all_path || (plevel > 0)))
						(void) printf(fl_dir, dname);
				}

				if (--dlevel > 0) {
					copy_name(dname, php(bp)->ph_name);
					if (mtype==POPD_OLD_MAGIC && downcase) downcase_name(dname);
				}
				else
					(void) strcpy(dname, dot);

				if (dlevel < plevel)
					plevel = 0;

				if (dlevel < blevel)
					blevel = dlevel;

				if (blevel < alevel)
					alevel = blevel = 0;

				if (plevel > 0)
					adjust_path(dname);

				break;

			case  LINK_MAGIC:	case  LINK_OLD_MAGIC:
				{
					char	lname[MAXPATHLEN];
					char	cname[MAXPATHLEN];

					long	llength;
					char	*cp;

					llength = get_long(lhp(bp)->lh_length);

					copy_namelen(lname, lhp(bp)->lh_name,
					              (int) llength);

					if (mtype==LINK_OLD_MAGIC && downcase) downcase_name(lname);

					if (!sel_path(lname))
						break;

					cp = lhp(bp)->lh_name + llength;

					copy_name(cname, cp);
					if (mtype==LINK_OLD_MAGIC && downcase) downcase_name(cname);

					if (mode_index) {
						if (verbose) printf(fi_link_v, lname, cname);
						else (void) printf(fi_link, lname);
						break;
					}

					if (check_exist(lname, L_TYPE) < 0)
						break;

					make_dir(lname, L_TYPE);


					if (symlink(cname, lname) < 0)
						errorp(lname);

					if (set_attrib(lname,
					               &dattrib[dlevel],
					               L_TYPE) < 0)
						/* just ignore - target might not exist yet */
						/*errorp(lname)*/;

					if (list_link)
						(void) printf(fl_link, lname);
				}

				break;

			default:
#ifdef	DEBUGTYPE
#define	DEBUGTFMT	"%s: DEBUG: invalid type = (%d,%d), size = %d\n"

				(void) fprintf(stderr,
				               DEBUGTFMT,
				               prog,
				               MAJORTYPE(mtype),
				               MINORTYPE(mtype),
				               msize);
#else
				errors("invalid type");
#endif	/* DEBUGTYPE */
				break;
		}

		if (!all_path && (pcount <= 0) && (fp == (FILE *) NULL) &&
		    (plevel <= 0) && (alevel <= 0) && (blevel <= 0))
			return(-1);

		bp += msize;
		if (isodd(bplast-bp))
			bp++;
	}

	return(0);
}

char	fe_toolong[] = "file name too long";
char	fe_trunc[] = "file name truncated";

#ifdef	__STDC__
void	copy_namelen(char *d, char *s, int slen)
#else
void	copy_namelen(d, s, slen)
char	*d;
char	*s;
int	slen;
#endif	/* __STDC__ */
{
	if (slen >= MAXPATHLEN)
		errors(fe_toolong);

	if (slen < 0)
		errors(fe_trunc);

	(void) strncpy(d, s, (str_size_t) slen);
	d[slen] = CNULL;
}

char	fw_exist[] = "%s: unable to restore \"%s\" (name already exists)\n";

#ifdef	__STDC__
int	check_exist(char *path, int type)
#else
int	check_exist(path, type)
char	*path;
int	type;
#endif	/* __STDC__ */
{
	struct stat	statbuf;

	if (lstat(path, &statbuf) == 0) {
		if ((mode_exist & EX_MERGE) &&
		    (type == D_TYPE) &&
		    (isdir(statbuf.st_mode) ||
		     (islink(statbuf.st_mode) &&
		      (stat(path, &statbuf) == 0) &&
		      isdir(statbuf.st_mode))))
			return(0);

		if (mode_exist & EX_REPLACE) {
			if ((mode_exist & EX_MERGE) && sel_keep(path))
				return(-1);

			rremove(path, 0);
			return(0);
		} else {
			if (!(mode_exist & EX_MERGE))
				(void) fprintf(stderr, fw_exist, prog, path);

			return(-1);
		}
	}

	if (errno != ENOENT)
		errorp(path);

	return(0);
}

char	kname[MAXPATHLEN];
int	knamelen;
int	koffset;

#ifdef	__STDC__
void	make_dir(char *path, int type)
#else
void	make_dir(path, type)
char	*path;
int	type;
#endif	/* __STDC__ */
{
	char	*sp[MAXDLEVEL];
	int	slevel;

	if (!*path) return;

	slevel = 0;

	while (((sp[slevel] = rindex(path, SLASH)) != (char *) 0) &&
	       (sp[slevel] > path)) {
		struct stat	statbuf;

		*sp[slevel] = CNULL;

		if (lstat(path, &statbuf) == 0)
			break;

		if (errno != ENOENT)
			errorp(path);

		if (++slevel >= MAXDLEVEL)
			errors("too many components in pathname");
	}

	if (knamelen <= 0) {
		int	tlevel;

		tlevel = dlevel;
		if (type == D_TYPE)
			--tlevel;

		if ((tlevel > 0) && (slevel > 0)) {
			if (alevel <= 0) {
				alevel = tlevel - slevel + 1;
				if (alevel <= 0)
					alevel = 1;
			}

			if (alevel > 0)
				blevel = tlevel;
		}
	}

	if (sp[slevel] != (char *) 0)
		*sp[slevel] = SLASH;

	while (slevel-- > 0) {
		if (mkdir(path, DIR_MODE) < 0)
			errorp(path);

		*sp[slevel] = SLASH;
	}

	if ((type == D_TYPE) &&
	    (mkdir(path, DIR_MODE) < 0) &&
	    !((mode_exist & EX_MERGE) && (errno == EEXIST)))
		errorp(path);
}

#ifdef	__STDC__
void	open_file(char *path)
#else
void	open_file(path)
char	*path;
#endif	/* __STDC__ */
{
	if ((fp = fopen(path, "w")) == (FILE *) NULL)
		errorp(path);
}

#ifdef	__STDC__
void	close_file(char *path, attrib *ap)
#else
void	close_file(path, ap)
char	*path;
attrib	*ap;
#endif	/* __STDC__ */
{
	if (fclose(fp) == EOF)
		errorp(path);

	fp = (FILE *) NULL;

	if (set_attrib(path, ap, F_TYPE) < 0)
		errorp(path);

	if (list_file)
		if (verbose) (void) printf(fl_file_v, fname, type_name, Fsize);
		else (void) printf(fl_file, path);
}

char unk_type[] = "unknown";

#ifdef	__STDC__
void	type_handler(uid__t *tuid, long *offset, char **tdesc)
#else
void	type_handler(tuid, offset, tdesc)
uid__t	*tuid;
long	*offset;
char	**tdesc;
#endif	/* __STDC__ */
{
	int i;
	for (i=0; *uid_table[i].desc; i++) {
		if (uid_table[i].uid.high == tuid->high && 
		    uid_table[i].uid.low  == tuid->low ) {
			*offset = uid_table[i].hdr_sz;
			*tdesc  = uid_table[i].desc;
			return;
		}
	}
	*tdesc = unk_type;
	*offset=0;
}

char	fe_magic[] = "invalid magic number";

#ifdef	__STDC__
long	get_size(char *bp, long mtype)
#else
long	get_size(bp, mtype)
char	*bp;
long	mtype;
#endif	/* __STDC__ */
{
	switch (mtype) {
		case  FILE_MAGIC:
		case   DIR_MAGIC:
			return(get_long(chp(bp)->ch_size));

		case  FILE_OLD_MAGIC:
		case   DIR_OLD_MAGIC:
			return(get_long(cohp(bp)->coh_size));

		default:
			errors(fe_magic);
			/*NOTREACHED*/
	}
}

#ifdef	__STDC__
void	dir_attrib(char *path, attrib *ap)
#else
void	dir_attrib(path, ap)
char	*path;
attrib	*ap;
#endif	/* __STDC__ */
{
	struct stat	statbuf;

	if (stat(path, &statbuf) < 0)
		errorp(path);

	if (!isdir(statbuf.st_mode)) {
		errno = ENOTDIR;
		errorp(path);
	}

	ap->at_mtime = statbuf.st_mtime;
	ap->at_mode = statbuf.st_mode;
	ap->at_uid = statbuf.st_uid;
	ap->at_gid = statbuf.st_gid;
}

#ifdef	__STDC__
void	get_attrib(char *bp, long mtype, attrib *ap)
#else
void	get_attrib(bp, mtype, ap)
char	*bp;
long	mtype;
attrib	*ap;
#endif	/* __STDC__ */
{
	switch (mtype) {
		case  FILE_MAGIC:
		case   DIR_MAGIC:
			ap->at_mtime =
				time_atou(get_long(chp(bp)->ch_mtime.at_time));

			ap->at_mode = acl_to_mode(chp(bp)->ch_uacl,
			                          chp(bp)->ch_gacl,
			                          chp(bp)->ch_zacl,
			                          chp(bp)->ch_oacl);

			ap->at_uid = get_long(chp(bp)->ch_uid);
			ap->at_gid = get_long(chp(bp)->ch_gid);

			break;

		case  FILE_OLD_MAGIC:
		case   DIR_OLD_MAGIC:
			ap->at_mtime = time_atou(get_long(cohp(bp)->coh_mtime));

			ap->at_mode = -1;

			ap->at_uid = -1;
			ap->at_gid = -1;

			break;

		default:
			errors(fe_magic);
			break;
	}
}

#ifdef	__STDC__
int	set_attrib(char *path, attrib *ap, int type)
#else
int	set_attrib(path, ap, type)
char	*path;
attrib	*ap;
int	type;
#endif	/* __STDC__ */
{
	switch (type) {
		int	status;

		case F_TYPE:
		case D_TYPE:
			if (opt_pdt) {
#ifdef	UTIMBUF
				struct utimbuf	stimes;
				struct utimbuf	*times = &stimes;

				times->actime = times->modtime = ap->at_mtime;
#else
				time_t	times[2];

				times[0] = times[1] = ap->at_mtime;
#endif	/* UTIMBUF */

				if ((status = utime(path, times)) != 0)
					return(status);
			}

			if (opt_sacl &&
			    (ap->at_mode != -1) &&
			    ((status = chmod(path, ap->at_mode)) != 0))
				return(status);

			/* fall through to next case */
		case L_TYPE:
			if (opt_sacl &&
			    ((ap->at_uid != -1) || (ap->at_gid != -1)) &&
			    ((status = chown(path,
			                       (chown_uid_t) ap->at_uid,
			                       (chown_gid_t) ap->at_gid)) != 0))
				if (errno != EPERM)
					return(status);

			break;
		default:
			errorm("invalid attribute type");
			break;
	}

	return(0);
}

#ifdef	__STDC__
void	get_inode(inode *ip, char *bp, long mtype)
#else
void	get_inode(ip, bp, mtype)
inode	*ip;
char	*bp;
long	mtype;
#endif	/* __STDC__ */
{
	A_inode	*ap;

	switch (mtype) {
		case  FILE_MAGIC:
		case   DIR_MAGIC:
			ap = &chp(bp)->ch_inode;
			break;

		case  FILE_OLD_MAGIC:
		case   DIR_OLD_MAGIC:
			ap = &cohp(bp)->coh_inode;
			break;

		default:
			errors(fe_magic);
			break;
	}

	ip->in_time = get_long(ap->ai_time);
	ip->in_node = get_long(ap->ai_node);
}

#ifdef	__STDC__
void	copy_inode(inode *ip, inode *jp)
#else
void	copy_inode(ip, jp)
inode	*ip;
inode	*jp;
#endif	/* __STDC__ */
{
	ip->in_time = jp->in_time;
	ip->in_node = jp->in_node;
}

#ifdef	__STDC__
int	cmp_inode(inode *ip, inode *jp)
#else
int	cmp_inode(ip, jp)
inode	*ip;
inode	*jp;
#endif	/* __STDC__ */
{
	int	icmp;

	if ((icmp = (ip->in_time - jp->in_time)) == 0)
	       icmp = ip->in_node - jp->in_node;

	return(icmp);
}

#ifdef	__STDC__
int	check_link(char *bp, long mtype)
#else
int	check_link(bp, mtype)
char	*bp;
long	mtype;
#endif	/* __STDC__ */
{
	switch (mtype) {
		case  FILE_MAGIC:
			return(get_short(chp(bp)->ch_nlink) > 1);

		case   DIR_MAGIC:
			return(get_short(chp(bp)->ch_nlink) > 2);

		case  FILE_OLD_MAGIC:
		case   DIR_OLD_MAGIC:
			return(1);	/* worst case */

		default:
			errors(fe_magic);
			/*NOTREACHED*/
	}
}

#ifdef	__STDC__
int	make_link(char *path, char *bp, long mtype)
#else
int	make_link(path, bp, mtype)
char	*path;
char	*bp;
long	mtype;
#endif	/* __STDC__ */
{
	linktab	*lp;

	if (check_link(bp, mtype)) {
		inode	linode;

		get_inode(&linode, bp, mtype);

		if ((lp = find_link(path, &linode)) != (linktab *) 0) {
			make_dir(path, L_TYPE);

			if  (link(lp->lt_name, path) < 0)
				errorp(path);

			return(1);
		}
	}

	return(0);
}

linktab	*linktabroot = (linktab *) 0;

#ifdef	__STDC__
linktab	*find_link(char *path, inode *ip)
#else
linktab	*find_link(path, ip)
char	*path;
inode	*ip;
#endif	/* __STDC__ */
{
	linktab	*ep;
	linktab	*lp;

	for (ep = (linktab *) 0, lp = linktabroot;
		lp != (linktab *) 0;
			ep = lp, lp = lp->lt_next)
				if (cmp_inode(&lp->lt_inode, ip) == 0)
					return(lp);

	lp = (linktab *) get_memory(sizeof(linktab));

	if (ep != (linktab *) 0)
		ep->lt_next = lp;
	else
		linktabroot = lp;

	lp->lt_name = (char *) get_memory((unsigned) (strlen(path)+1));
	(void) strcpy(lp->lt_name, path);
	(void) copy_inode(&lp->lt_inode, ip);
	lp->lt_next = (linktab *) 0;

	return(lp->lt_next);
}

char	path_tty[] = PATH_TTY;
char	mode_read[] = "r";

#ifdef	__STDC__
void	ask_fileno(void)
#else
void	ask_fileno()
#endif	/* __STDC__ */
{
	FILE	*tfp;
	char	buf[MAXLINE];

	if ((tfp = fopen(path_tty, mode_read)) == (FILE *) NULL) 
		erroru("no backup file number or id specified");

	(void) fprintf(stderr, "Backup file number: ");

	while (fgets(buf, MAXLINE, tfp) != (char *) 0) {
		char	*cp = buf;

		while (iswhite(*cp))
			cp++;

		if (strncmp(cp,
		            cur_fileno,
		            (str_size_t) (sizeof(cur_fileno)-1)) == 0) {
			char	*ep = cp + (sizeof(cur_fileno)-1);

			if (iswhite(*ep) || isnl(*ep) || isnull(*ep)) {
				req_fileno = -1;
				break;
			}
		}

		if ((sscanf(cp, "%d", &req_fileno) == 1) && (req_fileno > 0))
			break;

		(void) fprintf(stderr, "Invalid file number, reenter: ");
	}

	(void) fclose(tfp);

	if (req_fileno == 0) {
		exit(E_USAGE);
		/*NOTREACHED*/
	}
}

#ifdef	__STDC__
void	ask_path(void)
#else
void	ask_path()
#endif	/* __STDC__ */
{
	FILE	*tfp;
	int	c;

	(void) fprintf(stderr, "%s: no pathnames or -all specified\n", prog);

	if ((tfp = fopen(path_tty, mode_read)) == (FILE *) NULL) {
		exit(E_USAGE);
		/*NOTREACHED*/
	}

	(void) fprintf(stderr, "%s everything (yes/no)? ",
	                   (mode_index ? "Index" : "Restore"));

	while ((c = getc(tfp)) != EOF) {
		do {
			if (!iswhite(c))
				break;
		} while ((c = getc(tfp)) != EOF);

		if (!isnl(c)) {
			int	n;

			while ((n = getc(tfp)) != EOF)
				if (isnl(n))
					break;
		}

		if (isyes(c)) {
			all_path++;
			break;
		}

		if (isno(c))
			break;

		(void) fprintf(stderr, "Please answer \"yes\" or \"no\": ");
	}

	(void) fclose(tfp);

	if (!all_path) {
		exit(E_USAGE);
		/*NOTREACHED*/
	}
}

pathtab	*pathtablast = (pathtab *) 0;

#ifdef	__STDC__
void	add_path(char *path)
#else
void	add_path(path)
char	*path;
#endif	/* __STDC__ */
{
	pathtab	*pp;

	pp = (pathtab *) get_memory(sizeof(pathtab));

	if (pathtablast != (pathtab *) 0)
		pathtablast->pt_next = pp;
	else
		pathtabroot = pp;

	pp->pt_name = path;
	pp->pt_rename = (char *) 0;
	pp->pt_found = 0;
	pp->pt_next = (pathtab *) 0;

	pathtablast = pp;

	pcount++;
}

#ifdef	__STDC__
void	rename_path(char *path)
#else
void	rename_path(path)
char	*path;
#endif	/* __STDC__ */
{
	if ((pathtablast == (pathtab *) 0) ||
	    (pathtablast->pt_rename != (char *) 0))
		erroru(fu_aspre);

	pathtablast->pt_rename = path;
}

char	fe_dupname[] = "duplicate name found in backup file";

#ifdef	__STDC__
int	sel_path(char *path)
#else
int	sel_path(path)
char	*path;
#endif	/* __STDC__ */
{
	if (all_path)
		return(1);
	else if (plevel > 0) {
		adjust_path(path);

		return(1);
	} else {
		pathtab	*pp;

		for (pp = pathtabroot; pp != (pathtab *) 0; pp = pp->pt_next)
			if (strcmp(pp->pt_name, path) == 0) {
				pcount--;

				if (pp->pt_found++)
					errors(fe_dupname);

				if (pp->pt_rename != (char *) 0) {
					(void) strcpy(kname, pp->pt_rename);
					knamelen = strlen(kname);
					koffset  = strlen(path);
				} else {
					kname[0] = CNULL;
					knamelen = 0;
					koffset  = 0;
				}

				adjust_path(path);

				return(1);
			}

		return(0);
	}
}

char	fe_adjtooshort[] = "adjusted file name too short";
char	fe_adjtoolong[]  = "adjusted file name too long";
char	fe_adjgarbled[]  = "adjusted file name garbled";

#ifdef	__STDC__
void	adjust_path(char *path)
#else
void	adjust_path(path)
char	*path;
#endif	/* __STDC__ */
{
	if (knamelen > 0) {
		char	tname[MAXPATHLEN];
		int	tnamelen = strlen(path) - koffset;

		if (tnamelen < 0)
			errors(fe_adjtooshort);

		if ((knamelen+tnamelen) >= MAXPATHLEN)
			errors(fe_adjtoolong);

		(void) strcpy(tname, path+koffset);

		if ((*tname != SLASH) && (*tname != CNULL))
			errors(fe_adjgarbled);

		(void) strcpy(path, kname);
		(void) strcat(path, tname);
	}
}

keeptab	*keeptabroot = (keeptab *) 0;
keeptab	*keeptablast = (keeptab *) 0;

#ifdef	__STDC__
void	add_keep(char *path)
#else
void	add_keep(path)
char	*path;
#endif	/* __STDC__ */
{
	keeptab	*kp;

	kp = (keeptab *) get_memory(sizeof(keeptab));

	if (keeptablast != (keeptab *) 0)
		keeptablast->kt_next = kp;
	else
		keeptabroot = kp;

	kp->kt_name = path;
	kp->kt_next = (keeptab *) 0;

	keeptablast = kp;
}

#ifdef	__STDC__
int	sel_keep(char *path)
#else
int	sel_keep(path)
char	*path;
#endif	/* __STDC__ */
{
	keeptab	*kp;

	for (kp = keeptabroot; kp != (keeptab *) 0; kp = kp->kt_next)
		if (strcmp(kp->kt_name, path) == 0)
			return(1);

	return(0);
}

#ifdef	__STDC__
vmem_t	get_memory(unsigned int size)
#else
vmem_t	get_memory(size)
unsigned int	size;
#endif	/* __STDC__ */
{
	vmem_t	mp;

	if ((mp = (vmem_t) malloc(size)) == (vmem_t) 0)
		errorm("out of memory");

	return(mp);
}

#ifdef	__STDC__
short	get_short(A_short A_val)
#else
short	get_short(A_val)
A_short	A_val;
#endif	/* __STDC__ */
{
	union {
		short	us_val;
		char	us_buf[SIZEOF_SHORT];
	} u;

#ifdef	WIDE_SHORT
	u.us_val = 0;
#endif	/* WIDE_SHORT */

#define	SHORT_XFER(x)	(u.us_buf[SHORT_INDEX(x)] = A_val[(x)])

	SHORT_XFER(0);
	SHORT_XFER(1);

	return(u.us_val);
}

#ifdef	__STDC__
long	get_long(A_long A_val)
#else
long	get_long(A_val)
A_long	A_val;
#endif	/* __STDC__ */
{
	union {
		long	ul_val;
		char	ul_buf[SIZEOF_LONG];
	} u;

#ifdef	WIDE_LONG
	u.ul_val = 0L;
#endif	/* WIDE_LONG */

#define	LONG_XFER(x)	(u.ul_buf[LONG_INDEX(x)] = A_val[(x)])

	LONG_XFER(0);
	LONG_XFER(1);
	LONG_XFER(2);
	LONG_XFER(3);

	return(u.ul_val);
}

/*
  Convert name to lower case.  This should only be used for "old"-type names
  (ie from SR9 or older systems which were case-insensitive for filesystem
  objects) which were stored in upper-case but were normally displayed in
  lower-case.  Note that the Unix add-on, Domain/IX, required a fudge, since
  it needed case-sensitive names - so ":" was used as an escape to indicate
  the following character was a "real" uppercase.
*/
#ifdef	__STDC__
void 	downcase_name(char *name)
#else
void 	downcase_name(name)
char	*name;
#endif	/* __STDC__ */
{
	char *s = name, *d = name;

	while (*s) {
		if (*s==':') { s++; *d++ = *s++; }
		else { *d++ = tolower(*s++); }
	}
	*d = '\0';
}

/*
  convert_rec converts Apollo rec-type files with leading 4-byte
  count fields to sequential data records.  Variable or constant
  record length.  This will normally fix Fortran direct-access files.
*/
#ifdef	__STDC__
int	convert_rec(unsigned char *b, int n)
#else
int	convert_rec(b, n)
unsigned char	*b;
int		n;
#endif	/* __STDC__ */
{
	char	bf[BLOCK_SIZE];
        char	*bx = bf;

	for (; n; n--,b++) {
		if (rc_ptr++ < 4) {
			if (rc_ptr > 0) rc_count = (rc_count<<8) + (int)*b;
			if (rc_ptr == 4) {
				/*fprintf(stderr,"rc_ptr=%d count=%d\n",rc_ptr,rc_count);*/
				if (rc_count < 4) {	/* switch to straight copy */
					*(int*)bx = rc_count;
					bx += 4;
					rc_count = INT_MAX;
				}
			}
		} else {
			*bx++ = *b;
		        if (rc_ptr == rc_count) { 
				if (isodd(rc_count)) rc_ptr = -1; else rc_ptr = 0;
				rc_count = 0;
			}
		}
	}
	return( fwrite((fwrite_buf_t)  bf, (fwrite_size_t) (bx-bf),
		       (fwrite_size_t) 1, fp) );
}

#ifdef	__STDC__
int	ask_vol_swap(void)
#else
int	ask_vol_swap()
#endif	/* __STDC__ */
{
	FILE	*tfp;
	char	ANS[80];
	int	rc = 1;


	if ((tfp = fopen(path_tty, mode_read)) == (FILE *) NULL) 
		erroru("tape swap - manual intervention required");

	(void) fprintf(stderr, "** Tape swap: Please mount the next volume for reading.\n");
	(void) fprintf(stderr, "   Options:\n");
	(void) fprintf(stderr, "     g[o]    - continue using defaults.\n");
	(void) fprintf(stderr, "     a[bort] - abort search.\n");
	(void) fprintf(stderr, "     Option: ");


	while (rc = (fgets(ANS, sizeof(ANS), stdin)!=0)) {
		(void)downcase_name(ANS);
		if (strlen(ANS)) {
			if (strncmp("abort",ANS,strlen(ANS)) == 0) {
				rc = 0;
				break;
			} else if (strncmp("go",ANS,strlen(ANS)) == 0) {
				rc = 1;
				break;
			}
		}
		(void) fprintf(stderr, "Please answer \"g[o]\" or \"a[bort]\": ");
	}

	(void) fclose(tfp);
	return (rc);
}

#ifdef	notused
#ifdef	__STDC__
char 	*ctime_apollo(time_apollo *tap)
#else
char 	*ctime_apollo(tap)
time_apollo	*tap;
#endif	/* __STDC__ */
{
	char	*cp;
	long	t;

	t = time_atou(tap->ta_time);

	cp = ctime(&t);
	cp[24] = CNULL;

	return(cp);
}
#endif	/* notused */
