/*
 *	ansitape.h - ANSI tape headers
 *
 *	This is based on the ansitape program, by David S. Hayes
 *	     (available from comp.sources.unix, volume 8).
 */

#ifndef	ANSITAPE_H
#define	ANSITAPE_H

typedef	struct VOL1 {
	char	v1_ident[4];		/* VOL1 */
	char	v1_label[6];		/* volume name */
	char	v1_access;		/* access code */
	char	v1_ignore1[20];
	char	v1_ignore2[6];
	char	v1_owner[14];		/* user name */
	char	v1_ignore3[28];
	char	v1_ansi_level;		/* ANSI standard level */
} VOL1;

#define	VOL1_IDENT	"VOL1"
#define	VOL1_IDSWAB	"OV1L"
#define	VOL1_ACCESS	' '
#define	VOL1_ANSI_LEVEL	'3'

typedef	struct VOL2 {
	char	v2_ident[4];		/* VOL2 */
	char	v2_ignore[76];
} VOL2;

#define	VOL2_IDENT	"VOL2"
#define	VOL2_IDSWAB	"OV2L"

typedef	struct UVL1 {
	char	uv_ident[4];		/* UVL1 */
	char	uv_time_apollo[17];	/* volume time (Apollo) */
	char	uv_ignore[59];
} UVL1;

#define	UVL1_IDENT	"UVL1"
#define	UVL1_IDSWAB	"VU1L"

typedef	struct HDR1 {
	char	h1_ident[4];		/* HDR1 */
	char	h1_name[17];		/* filename */
	char	h1_fileset[6];		/* file volume set */
	char	h1_volume_num[4];	/* number of volume in set */
	char	h1_file_num[4];		/* number of file on tape */
	char	h1_gen[4];		/* generation number */
	char	h1_genver[2];		/* generation version number */
	char	h1_created[6];		/* creation date */
	char	h1_expires[6];		/* expiration date */
	char	h1_access;		/* access code */
	char	h1_blockcount[6];	/* number of blocks (zero in header) */
	char	h1_tapesys[13];		/* name of software creating tape */
	char	h1_ignore[7];
} HDR1;

#define	HDR1_IDENT	"HDR1"
#define	HDR1_IDSWAB	"DH1R"
#define	HDR1_FILESET	"BACKUP"
#define	HDR1_VOLUME_NUM	"0001"
#define	HDR1_GEN	"0001"
#define	HDR1_GENVER	"00"
#define	HDR1_ACCESS	' '
#define	HDR1_BLOCKCOUNT	"000000"
#define	HDR1_TAPESYS	"             "

typedef	struct HDR2 {
	char	h2_ident[4];		/* HDR2 */
	char	h2_recfmt;		/* record format */
	char	h2_blocklen[5];		/* file tape block length */
	char	h2_reclen[5];		/* file logical record length */
	char	h2_density;		/* density/recording mode */
	char	h2_vol_switch;		/* file continuation indicator */
	char	h2_job[17];		/* job name/step */
	char	h2_recording[2];	/* parity/conversion, space fill */
	char	h2_carriage_control;	/* carriage control */
	char	h2_ignore1;
	char	h2_blocked_records;	/* blocked records indictor */
	char	h2_ignore2[11];
	char	h2_block_offset[2];	/* bytes to skip for each block */
	char	h2_ignore3[28];
} HDR2;

#define	HDR2_IDENT	"HDR2"
#define	HDR2_IDSWAB	"DH2R"
#define	HDR2_RECFMT	'F'
#define	HDR2_BLOCKLEN	"08192"		/* 00512 for QIC */
#define	HDR2_RECLEN	"08192"		/* 00512 for QIC */
#define	HDR2_DENSITY	' '
#define	HDR2_VOL_SWITCH	' '
#define	HDR2_JOB	"                 "
#define	HDR2_RECORDING	"  "
#define	HDR2_CARRIAGE_CONTROL	' '
#define	HDR2_BLOCKED_RECORDS	' '
#define	HDR2_BLOCK_OFFSET	"00"

typedef	struct HDR3 {
	char	h3_ident[4];		/* HDR3 */
	char	h3_os_reserved[76];	/* reserved for OS */
} HDR3;

#define	HDR3_IDENT	"HDR3"
#define	HDR3_IDSWAB	"DH3R"

typedef	struct HDR4 {
	char	h4_ident[4];		/* HDR4 */
	char	h4_name[63];		/* filename (continued) */
	char	h4_ignore1[2];
	char	h4_ignore2[11];
} HDR4;

#define	HDR4_IDENT	"HDR4"
#define	HDR4_IDSWAB	"DH4R"

typedef	struct UHL1 {
	char	uh_ident[4];		/* UHL1 */
	char	uh_time_apollo[17];	/* file time (Apollo) */
	char	uh_ignore1;
	char	uh_date_gmt[10];	/* file date (GMT) */
	char	uh_ignore2;
	char	uh_time_gmt[8]; 	/* file time (GMT) */
	char	uh_ignore3[39];
} UHL1;

#define	UHL1_IDENT	"UHL1"
#define	UHL1_IDSWAB	"HU1L"

typedef	struct SYSB {
	char	sb_ignore1[16];		/* Might be some more specific id stuff in here */
	char	sb_ident[8];		/* SYSBOOT  */
	char	sb_ignore2[56];		/* actually much longer */
} SYSB;

#define	SYSB_IDENT	"SYSBOOT "
#define	SYSB_IDSWAB	"YSBSOO T"

typedef	struct EOFV {
	char	ef_ident[4];
	char	ef_ignore[76];
} EOFV;

#define	EOF1_IDENT	"EOF1"
#define	EOF1_IDSWAB	"OE1F"
#define	EOF2_IDENT	"EOF2"
#define	EOF2_IDSWAB	"OE2F"

#define	EOV1_IDENT	"EOV1"
#define	EOV1_IDSWAB	"OE1V"
#define	EOV2_IDENT	"EOV2"
#define	EOV2_IDSWAB	"OE2V"

#define	FORMAT_FILENO     	"%4d"

#define	FORMAT_DATE_ANSI	" %2d%3d"

#define	FORMAT_TIME_APOLLO	"%8x.%8x"

#define	FORMAT_DATE_GMT   	"%4d/%2d/%2d"
#define	FORMAT_TIME_GMT   	"%2d:%2d:%2d"

typedef	struct ansitab {
	char	*at_ident;		/* identifier */
	char	*at_idswab;		/* ... ditto, byte-swapped */
	void	(*at_action)();		/* function to call */
	short	at_vol;			/* true if volume header */
	short	at_req;			/* true if required */
	int	at_count;		/* number found */
} ansitab;

typedef	union ansibuf {
	char	ab_ident[4];
	VOL1	ab_VOL1;
	VOL2	ab_VOL2;
	UVL1	ab_UVL1;
	HDR1	ab_HDR1;
	HDR2	ab_HDR2;
	HDR3	ab_HDR3;
	HDR4	ab_HDR4;
	UHL1	ab_UHL1;
	SYSB	ab_SYSB;
	EOFV	ab_EOFV;
	char	FILLER[CT_BLKSIZE];
} ansibuf;

typedef	struct devnametab {
	char	*dn_apollo;		/* Apollo device name */
	char	*dn_unix;		/* UNIX device name */
	int	dn_offset;		/* unit number offset */
	int	dn_type;		/* device type */
} devnametab;

#define	DEV_NONE	(-1)

#define	DEV_STDIN	0
#define	DEV_FILE	1
#define	DEV_TAPE	2
#define	DEV_FLOPPY	3
#define	DEV_REMOTE	4
#define	DEV_REMCT	5

#define	DEV_APOLLO_M	"m"
#define	DEV_APOLLO_MT	"mt"
#define	DEV_APOLLO_C	"c"
#define	DEV_APOLLO_CT	"ct"
#define	DEV_APOLLO_F	"f"

#define	DEV_APOLLO_DEF	"m0"

#define	DEV_MAXDIGITS	MAXDIGITS

#define	TAPE_HIDENSITY	8
#define	TAPE_NOREWIND	4

#define	TAPE_OPTS_BSD	(TAPE_HIDENSITY|TAPE_NOREWIND)

#define	DEV_OFFSET_NONE	0

#ifdef	apollo
#define	DEV_UNIX_M	"/dev/rmt%d"
#define	DEV_UNIX_MT	"/dev/rmts%d"
#define	DEV_UNIX_CT	"/dev/rct%d"
#define	DEV_UNIX_F	"/dev/rfl%da"

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_MT)	/* longest */

#define	DEV_OFFSET_M	TAPE_OPTS_BSD
#define	DEV_OFFSET_MT	TAPE_OPTS_BSD
#define	DEV_OFFSET_CT	TAPE_OPTS_BSD
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* apollo */

#ifdef	vax
#define	DEV_UNIX_M	"/dev/rmt%d"
#define	DEV_UNIX_MT	"/dev/rmt%d"
#define	DEV_UNIX_CT	"/dev/rmt%d"	/* TK-50 */
#define	DEV_UNIX_F	"/dev/rra%dc"	/* RX-50, or "/dev/rrx%dc" for RX-02 */

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_F)	/* longest */

#define	DEV_OFFSET_M	TAPE_OPTS_BSD
#define	DEV_OFFSET_MT	TAPE_OPTS_BSD
#define	DEV_OFFSET_CT	TAPE_OPTS_BSD
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* vax */

#ifdef	sun
#define	DEV_UNIX_M	"/dev/nrmt%d"
#define	DEV_UNIX_MT	"/dev/nrst%d"
#define	DEV_UNIX_CT	"/dev/nrst%d"
#define	DEV_UNIX_F	"/dev/rfd%dc"

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_M)	/* longest */

#define	DEV_OFFSET_M	TAPE_HIDENSITY
#define	DEV_OFFSET_MT	DEV_OFFSET_NONE
#define	DEV_OFFSET_CT	DEV_OFFSET_NONE
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* sun */

#ifdef	__hpux
#define	DEV_UNIX_M	"/dev/rmt/%dmn"
#define	DEV_UNIX_MT	"/dev/rmt/%dmn"
#define	DEV_UNIX_CT	"/dev/rmt/%dmn"		/* or "/dev/rct/c0d%d" */
#if	(_POSIX_VERSION > 198808L) && (_POSIX2_VERSION > 198909L)
#ifdef	__hp9000s700
#define	DEV_UNIX_F	"/dev/rfloppy/c201d%ds0"
#endif	/* __hp9000s700 */
#ifdef	__hp9000s300
#define	DEV_UNIX_F	"/dev/rfloppy/cEd%ds0"
#endif	/* __hp9000s300 */
#else
#define	DEV_UNIX_F	"/dev/rdsk/%ds0"
#endif	/* (_POSIX_VERSION > 198808L) && (_POSIX2_VERSION > 198909L) */

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_F)	/* longest */

#define	DEV_OFFSET_M	DEV_OFFSET_NONE
#define	DEV_OFFSET_MT	DEV_OFFSET_NONE
#define	DEV_OFFSET_CT	DEV_OFFSET_NONE
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* __hpux */

#ifdef	hp300
#define	DEV_UNIX_M	"/dev/rmt%d"
#define	DEV_UNIX_MT	"/dev/rmt%d"
#define	DEV_UNIX_CT	"/dev/rmt%d"
#define	DEV_UNIX_F	"/dev/rrd%dc"	/* HP9122 */

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_F)	/* longest */

#define	DEV_OFFSET_M	TAPE_OPTS_BSD
#define	DEV_OFFSET_MT	TAPE_OPTS_BSD
#define	DEV_OFFSET_CT	TAPE_OPTS_BSD
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* hp300 */

#ifdef	ibm032
#define	DEV_UNIX_M	"/dev/nrst%d"
#define	DEV_UNIX_MT	"/dev/nrst%d"
#define	DEV_UNIX_CT	"/dev/nrst%d"
#define	DEV_UNIX_F	"/dev/rfd%d"

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_M)	/* longest */

#define	DEV_OFFSET_M	DEV_OFFSET_NONE
#define	DEV_OFFSET_MT	DEV_OFFSET_NONE
#define	DEV_OFFSET_CT	DEV_OFFSET_NONE
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* ibm032 */

#ifdef	__386BSD__
#define	DEV_UNIX_M	"/dev/nrst%d"
#define	DEV_UNIX_MT	"/dev/nrst%d"
#define	DEV_UNIX_CT	"/dev/nrst%d"
#define	DEV_UNIX_F	"/dev/rfd%dc"

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_M)	/* longest */

#define	DEV_OFFSET_M	DEV_OFFSET_NONE
#define	DEV_OFFSET_MT	DEV_OFFSET_NONE
#define	DEV_OFFSET_CT	DEV_OFFSET_NONE
#define	DEV_OFFSET_F	DEV_OFFSET_NONE
#endif	/* __386BSD__ */

#define	DEV_UNIX_LEN	(DEV_UNIX_DLEN+(DEV_MAXDIGITS-2)+1)

#ifdef	sgi
#define	DEV_UNIX_M	"/dev/rmt/tps0d%dnrv"	  /* SCSI tape, no rewind, variable block */ 
#define	DEV_UNIX_MT	"/dev/rmt/tps0d%dnrv"	  /* SCSI tape, no rewind, variable block */
#define	DEV_UNIX_CT	"/dev/rmt/tps0d%dnrns"	  /* SCSI tape, no rewind, no swap (QIC) */
#define	DEV_UNIX_F	"/dev/rdsk/fds0d%d.3.5hi" /* 3.5", 1.44MB floppy */

#define	DEV_UNIX_DLEN	sizeof(DEV_UNIX_F)	/* longest */

#define	DEV_OFFSET_M	DEV_OFFSET_NONE
#define	DEV_OFFSET_MT	DEV_OFFSET_NONE
#define	DEV_OFFSET_CT	DEV_OFFSET_NONE
#define	DEV_OFFSET_F	DEV_OFFSET_NONE

#define DEV_CT_DEF 	"/dev/nrtapens"
#define DEV_MT_DEF 	"/dev/nrtape"

#endif	/* sgi */

#ifdef linux
#define	 DEV_UNIX_M	 "XXX"	 /* Where xxx is your no-rewind variable block tape device */
#define	 DEV_UNIX_MT	 DEV_UNIX_M
#define	 DEV_UNIX_CT	 "YYY"	 /* No rewind, no byte swap */
#define	 DEV_UNIX_F	 "/dev/fd%d"
#define	 DEV_UNIX_DLEN	 sizeof(XXX or YYY)  /* Whichever of XXX or YYY is longer */
#define	 DEV_OFFSET_M	 DEV_OFFSET_NONE
#define	 DEV_OFFSET_MT	 DEV_OFFSET_NONE
#define	 DEV_OFFSET_CT	 DEV_OFFSET_NONE
#define	 DEV_OFFSET_F	 DEV_OFFSET_NONE
#endif    /* linux */

void	open_tape PROTO((char *, int));
int	match_tape PROTO((char *, char *));
void	find_tape PROTO((void));
int	read_tape PROTO((char *, int));
#ifdef	SWAB_OVERLAP
#define	swap_tape(buf, size) \
	swab((swab_buf_t) (buf), (swab_buf_t) (buf), (swab_size_t) (size))
#else
void	swap_tape PROTO((char *, int));
#endif	/* SWAB_OVERLAP */
void	close_tape PROTO((void));
void	cmd_tape PROTO((int, int));

void	ansi_VOL1 PROTO((ansibuf *));
void	ansi_VOL2 PROTO((ansibuf *));
void	ansi_UVL1 PROTO((ansibuf *));
void	ansi_HDR1 PROTO((ansibuf *));
void	ansi_HDR2 PROTO((ansibuf *));
void	ansi_HDR3 PROTO((ansibuf *));
void	ansi_HDR4 PROTO((ansibuf *));
void	ansi_UHL1 PROTO((ansibuf *));

void	reset_id PROTO((void));

char	*strtrim PROTO((char *, char *, int));

void	skip_sysboot PROTO((void));

int	trailers PROTO((void));

void	tape_action_only PROTO((void));

void	shobuf PROTO((char *, int));

#endif	/* ANSITAPE_H */
