/*
 *	format.h - rbak format
 *
 *	The "A_" prefix used below means (possibly) unaligned,
 *	and Apollo (big-endian) byte order.
 */

#ifndef	FORMAT_H
#define	FORMAT_H

#ifdef	apollo
#define	A_SIZEOF_CHAR	sizeof(char)
#define	A_SIZEOF_SHORT	sizeof(short)
#define	A_SIZEOF_LONG	sizeof(long)
#else
#define	A_SIZEOF_CHAR	1
#define	A_SIZEOF_SHORT	2
#define	A_SIZEOF_LONG	4
#endif	/* apollo */

#define	  SIZEOF_SHORT	sizeof(short)
#define	  SIZEOF_LONG	sizeof(long)

#define	A_SHORT_REV(x)	((A_SIZEOF_SHORT-1)-(x))
#define	  SHORT_REV(x)	  ((SIZEOF_SHORT-1)-(x))

#define	A_LONG_REV(x)	((A_SIZEOF_LONG-1)-(x))
#define	  LONG_REV(x)	  ((SIZEOF_LONG-1)-(x))

#ifdef	ENDIAN_REVERSE
#define	SHORT_INDEX(x)	A_SHORT_REV(x)
#define	LONG_INDEX(x)	A_LONG_REV(x)
#endif	/* ENDIAN_REVERSE */

#ifdef	ENDIAN_APOLLO
#define	SHORT_INDEX(x)	SHORT_REV(A_SHORT_REV(x))
#define	LONG_INDEX(x)	LONG_REV(A_LONG_REV(x))
#endif	/* ENDIAN_APOLLO */

typedef	char	A_char[A_SIZEOF_CHAR];
typedef	char	A_short[A_SIZEOF_SHORT];
typedef	char	A_long[A_SIZEOF_LONG];

typedef	struct A_time_apollo {
	A_long	at_time;
	A_long	at_extra;
} A_time_apollo;

typedef	struct time_apollo {
	long	ta_time;	/* Apollo time */
	long	ta_extra;	/* extra precision, plus sometimes a node ID */
} time_apollo;

typedef	time_t	time_unix;	/* UNIX time */

#define	TC1	((1e6/4)/0x10000)
#define	TC2	((10*365+2)*24*60*60)

#define	time_atou(x)	((time_t) (((((double) ((unsigned)x))/TC1)+TC2)+0.5))
#define	time_utoa(x)	( (long)  (((((double) (x))-TC2)*TC1)+0.5))

#define	BLOCK_SIZE	0x2000		/* physical block size */
#define	MT_BLKSIZE	8192		/* DAT block size */
#define	CT_BLKSIZE	512		/* QIC block size */

typedef	struct A_block_header {
	A_long		bh_seq;		/* sequence number, starts at 1 */
	A_time_apollo	bh_time;	/* timestamp */
	A_short		bh_size;	/* size of data within block */
} A_block_header;

#define	bhp(bp) 	( (A_block_header *) (bp))

typedef	struct A_magic_header {		/* aligned to even file offset */
	A_short		mh_type1;	/* data type (see below) */
	A_short		mh_size;	/* size of data in next section */
	A_short		mh_type2;	/* data type (see below) */
} A_magic_header;

#define	mhp(bp) 	( (A_magic_header *) (bp))

#define	MAGICSHIFT	16		/* bits per short on Apollo */
#define	MAGICMASK	0xffff		/* mask for short on Apollo */

#define	MAGICTYPE(type1, type2)	(((type1) << MAGICSHIFT) | (type2))

#define	MAJORTYPE(type) 	(((type)  >> MAGICSHIFT) & MAGICMASK)
#define	MINORTYPE(type) 	( (type)                 & MAGICMASK)

#define	iseven(x)	(((x) & 0x1) == 0)
#define	isodd(x)	(((x) & 0x1) != 0)

typedef	struct A_sub_header {
	A_short		sh_ignore1;
	A_time_apollo	sh_time;	/* another timestamp */
	A_short		sh_ignore2;
	/* possibly longer */
} A_sub_header;

#define	shp(bp) 	(   (A_sub_header *) (bp))

#define	  SUB_MAGIC	MAGICTYPE(9,2)
#define	  SUB_OLD_MAGIC	MAGICTYPE(9,1)

typedef	struct A_empty_header {
	A_long		eh_empty;
} A_empty_header;

#define	ehp(bp) 	( (A_empty_header *) (bp))

#define	EMPTY_MAGIC	MAGICTYPE(8,1)

typedef	struct A_opt_header {
	A_char		oh_ignore;
	/* actually longer */
} A_opt_header;

#define	ohp(bp) 	(   (A_opt_header *) (bp))

#define	  OPT_MAGIC	MAGICTYPE(6,2)
#define	  OPT_OLD_MAGIC	MAGICTYPE(6,1)

typedef	struct A_acl_header {
	A_char		ah_ignore;
	/* actually longer */
} A_acl_header;

#define	ahp(bp) 	(   (A_acl_header *) (bp))

#define	  ACL_MAGIC	MAGICTYPE(10,2)
#define	  ACL_OLD_MAGIC	MAGICTYPE(10,1)

typedef	A_time_apollo	A_inode;

#define	ai_time	at_time
#define	ai_node	at_extra

typedef	time_apollo	inode;		/* timestamp used as inode */

#define	in_time	ta_time
#define	in_node	ta_extra

typedef	unsigned char	A_acl_apollo;

typedef	A_acl_apollo	acl_apollo;

typedef	struct A_common_header {	/* used for files/directories */
	A_long		ch_ignore1;
	A_inode		ch_inode;	/* inode of this file/directory */ /* probably actually Apollo UID of the object */
	A_long		ch_type_high;	/* type_uid.high */
	A_long		ch_type_low;	/* type_uid.low */
	A_long		ch_size;	/* actual size */
	A_long		ch_ignore4;
	A_time_apollo	ch_mtime;	/* modification time */
	A_time_apollo	ch_itime1;
	A_time_apollo	ch_itime2;
	A_time_apollo	ch_itime3;
	A_inode		ch_dinode;	/* inode of parent directory */
	A_long		ch_ignore5;
	A_time_apollo	ch_itime4;
	A_time_apollo	ch_itime5;
	A_long		ch_ignore6;
	A_long		ch_ignore7;
	A_acl_apollo	ch_uacl;	/* acl for owner */
	A_acl_apollo	ch_gacl;	/* acl for group */
	A_acl_apollo	ch_zacl;	/* acl for organization */
	A_acl_apollo	ch_oacl;	/* acl for world */
	A_long		ch_ignore8;
	A_long		ch_uid;		/* user id */
	A_long		ch_gid;		/* group id */
	A_long		ch_oid;		/* organization id */
	A_short		ch_nlink;	/* number of hard links */
	A_short		ch_pad;
} A_common_header;

#define	chp(bp) 	((A_common_header *) (bp))

#define	ACL_MASK_UID	0x40
#define	ACL_MASK_GID	0x40
#define	ACL_MASK_VTX	0x40
#define	ACL_MASK_IGN	0x10
#define	ACL_MASK_USR	0x07
#define	ACL_MASK_GRP	0x07
#define	ACL_MASK_OTH	0x07

#define	ACL_SHIFT_UID	5
#define	ACL_SHIFT_GID	4
#define	ACL_SHIFT_VTX	3
#define	ACL_SHIFT_USR	6
#define	ACL_SHIFT_GRP	3
#define	ACL_SHIFT_OTH	0

#define	acl_to_mode(u, g, z, o)	\
	( \
		(((u) & ACL_MASK_UID) << ACL_SHIFT_UID) | \
		(((g) & ACL_MASK_GID) << ACL_SHIFT_GID) | \
		(((o) & ACL_MASK_VTX) << ACL_SHIFT_VTX) | \
		(((u) & ACL_MASK_USR) << ACL_SHIFT_USR) | \
		(((g) & ACL_MASK_GRP) << ACL_SHIFT_GRP) | \
		(((o) & ACL_MASK_OTH) << ACL_SHIFT_OTH) \
	)

#define	mode_to_uacl(m)	\
	( \
		(mode_uid(m) >> ACL_SHIFT_UID) | \
		(mode_usr(m) >> ACL_SHIFT_USR) \
	)
#define	mode_to_gacl(m)	\
	( \
		(mode_gid(m) >> ACL_SHIFT_GID) | \
		(mode_grp(m) >> ACL_SHIFT_GRP) \
	)
#define	mode_to_zacl(m)	ACL_MASK_IGN
#define	mode_to_oacl(m)	\
	( \
		(mode_vtx(m) >> ACL_SHIFT_VTX) | \
		(mode_oth(m) >> ACL_SHIFT_OTH) \
	)

typedef	struct A_common_old_header {	/* used for (old) files/directories */
	A_long		coh_ignore1;
	A_inode		coh_inode;	/* inode of this file/directory */
	A_long		coh_ignore2;
	A_long		coh_ignore3;
	A_inode		coh_iinode;
	A_long		coh_size;	/* actual size */
	A_long		coh_ignore4;
	A_long		coh_atime;	/*    access    time, no extra prec */
	A_long		coh_mtime;	/* modification time, no extra prec */
	A_inode		coh_dinode;	/* inode of parent directory */
	A_short		coh_ignore5;
	A_short		coh_pad;
} A_common_old_header;

#define	cohp(bp) 	((A_common_old_header *) (bp))

typedef	struct A_name_header {
	A_inode		nh_inode;	/* inode of file */
	A_long		nh_ignore;
	A_char		nh_name;	/* file name */
	/* actually longer */
} A_name_header;

#define	nhp(bp) 	(  (A_name_header *) (bp))

#define	 NAME_MAGIC	MAGICTYPE(2,2)
#define	 NAME_OLD_MAGIC	MAGICTYPE(2,1)

typedef	struct A_file_header {
	A_common_header	fh_common;
	A_long		fh_ignore[6];
} A_file_header;

#define	fhp(bp) 	(  (A_file_header *) (bp))

#define	fh_inode	fh_common.ch_inode
#define	fh_size		fh_common.ch_size
#define	fh_mtime	fh_common.ch_mtime
#define	fh_dinode	fh_common.ch_dinode
#define	fh_uacl		fh_common.ch_uacl
#define	fh_gacl		fh_common.ch_gacl
#define	fh_zacl		fh_common.ch_zacl
#define	fh_oacl		fh_common.ch_oacl
#define	fh_uid		fh_common.ch_uid
#define	fh_gid		fh_common.ch_gid
#define	fh_oid		fh_common.ch_oid
#define	fh_nlink	fh_common.ch_nlink

#define	 FILE_MAGIC	MAGICTYPE(0,2)

typedef	struct A_file_old_header {
	A_common_old_header	foh_common;
	A_long			foh_ignore[2];
} A_file_old_header;

#define	fohp(bp) 	(  (A_file_old_header *) (bp))

#define	foh_inode	foh_common.coh_inode
#define	foh_size	foh_common.coh_size
#define	foh_atime	foh_common.coh_atime
#define	foh_mtime	foh_common.coh_mtime
#define	foh_dinode	foh_common.coh_dinode

#define	 FILE_OLD_MAGIC	MAGICTYPE(0,1)

#define	 DATA_MAGIC	MAGICTYPE(1,1)	/* file contents */

typedef	struct A_data_type_header {
	A_long		dth_magic0;
	A_long		dth_ignore1;
	A_long		dth_ignore2;
	A_long		dth_size;	/* file size, including header */
	A_long		dth_magic4;
	A_long		dth_magic5;
	A_long		dth_magic6;
	A_long		dth_magic7;
} A_data_type_header;

#undef	min
#define	min(x, y)	(((x) < (y)) ? (x) : (y))
#undef	max
#define	max(x, y)	(((x) > (y)) ? (x) : (y))

typedef	struct A_dir_header {
	A_common_header	dh_common;
	A_long		dh_ignore[32];
	A_char		dh_name;	/* directory name */
	/* actually longer */
} A_dir_header;

#define	dhp(bp) 	(   (A_dir_header *) (bp))

#define	dh_inode	dh_common.ch_inode
#define	dh_size		dh_common.ch_size
#define	dh_mtime	dh_common.ch_mtime
#define	dh_dinode	dh_common.ch_dinode
#define	dh_uacl		dh_common.ch_uacl
#define	dh_gacl		dh_common.ch_gacl
#define	dh_zacl		dh_common.ch_zacl
#define	dh_oacl		dh_common.ch_oacl
#define	dh_uid		dh_common.ch_uid
#define	dh_gid		dh_common.ch_gid
#define	dh_oid		dh_common.ch_oid
#define	dh_nlink	dh_common.ch_nlink

#define	  DIR_MAGIC	MAGICTYPE(3,3)

typedef	struct A_dir_old_header {
	A_common_old_header	doh_common;
	A_long			doh_ignore[2];
	A_inode			doh_iinode1;
	A_inode			doh_iinode2;
	A_char			doh_name;	/* directory name */
	/* actually longer */
} A_dir_old_header;

#define	dohp(bp) 	(   (A_dir_old_header *) (bp))

#define	doh_inode	doh_common.coh_inode
#define	doh_size	doh_common.coh_size
#define	doh_atime	doh_common.coh_atime
#define	doh_mtime	doh_common.coh_mtime
#define	doh_dinode	doh_common.coh_dinode

#define	  DIR_OLD_MAGIC	MAGICTYPE(3,2)

typedef	struct A_popd_header {
	A_long		ph_ignore;
	A_char		ph_name;	/* parent directory name */
	/* actually longer */
} A_popd_header;

#define	php(bp) 	(  (A_popd_header *) (bp))

#define	 POPD_MAGIC	MAGICTYPE(4,2)
#define	 POPD_OLD_MAGIC	MAGICTYPE(4,1)

typedef	struct A_link_header {
	A_short		lh_ignore;
	A_long		lh_length;	/* length of link name */
	A_char		lh_name;	/* concatenated link name/contents */
	/* actually longer */
	/* link contents immediately follow name */
} A_link_header;

#define	lhp(bp) 	(  (A_link_header *) (bp))

#define	 LINK_MAGIC	MAGICTYPE(5,2)
#define	 LINK_OLD_MAGIC	MAGICTYPE(5,1)

#endif	/* FORMAT_H */
