/*
 * sntp client.  See rfc1361.
 * -p remote port
 * -f set clock even if server is not synced
 * -n don't set clock
 * -q be quiet
 * -v be verbose
 *
 * Jim Rees University of Michigan Jan 1998
 */

#ifdef apollo
#undef __STDC__
#endif

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef __STDC__
#include <unistd.h>
#include <stdlib.h>
#endif
#include <stdio.h>
#include <string.h>

#define PORT 123
#define UNIXOFFSET -2085978496

struct {
    unsigned char livnmode, stratum, poll, prec;
    unsigned int delay, disp, refid;
    unsigned int reftime[2], origtime[2], rcvtime[2], txtime[2];
} pkt;

char *li_s[] = {
    "no warning",
    "last minute has 61 seconds",
    "last minute has 59 seconds",
    "clock not synchronized"
};

char *md_s[] = {
    "reserved",
    "symmetric active",
    "symmetric passive",
    "client",
    "server",
    "broadcast",
    "reserved for NTP control message",
    "reserved for private use"
};

int fflag, nflag, qflag, vflag;

#ifndef __STDC__
extern char *optarg;
extern void *malloc(int);
#endif

int
main(ac, av)
int ac;
char *av[];
{
    int i, fd, len, n, port = PORT, li, vn, md;
    char *hostname, *s;
    struct sockaddr_in saddr;
    struct hostent *hp;
    fd_set fdset;
    struct timeval tv, ntv;
    long offset;

    while ((i = getopt(ac, av, "p:fnqv")) != -1) {
	switch (i) {
	case 'p':
	    port = atoi(optarg);
	    break;
	case 'f':
	    fflag = 1;
	    break;
	case 'n':
	    nflag = 1;
	    break;
	case 'q':
	    qflag = 1;
	    break;
	case 'v':
	    vflag = 1;
	    break;
	}
    }

    if (ac < 2) {
	fprintf(stderr, "usage: sntp [-p port] [-fnq] servername\n");
	exit(1);
    }
    hostname = av[ac-1];

    /* Get a socket */

    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	exit(1);
    }

    /* Fill in the request packet */

    bzero(&pkt, sizeof pkt);
    pkt.livnmode = 0x0b;	/* version 1, mode 3 (client) */

    /* Send packet to server */

    bzero(&saddr, sizeof saddr);
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(port);
    if ((saddr.sin_addr.s_addr = inet_addr(hostname)) == -1) {
	if ((hp = gethostbyname(hostname)) == NULL) {
	    fprintf(stderr, "can't find address for host %s\n", hostname);
	    exit(1);
	}
	bcopy(hp->h_addr, &saddr.sin_addr, sizeof saddr.sin_addr);
	s = malloc(strlen(hostname) + 24);
	sprintf(s, "%s (%s)", hostname, inet_ntoa(saddr.sin_addr));
	hostname = s;
    }

    if (sendto(fd, &pkt, sizeof pkt, 0, (struct sockaddr *) &saddr, sizeof saddr) < 0) {
	perror("sendto");
	exit(1);
    }

    /* Read reply packet from server */

    FD_ZERO(&fdset);
    FD_SET(fd, &fdset);
    tv.tv_sec = 2;
    tv.tv_usec = 0;
    if (select(fd + 1, &fdset, NULL, NULL, &tv) != 1) {
	fprintf(stderr, "no reply from server %s\n", hostname);
	exit(2);
    }

    len = sizeof pkt;
    if ((n = recvfrom(fd, &pkt, sizeof pkt, 0, (struct sockaddr *) &saddr, &len)) < 0) {
	perror("recvfrom");
	exit(1);
    }

    li = (pkt.livnmode & 0xc0) >> 6;
    vn = (pkt.livnmode & 0x38) >> 3;
    md = pkt.livnmode & 0x3;

    if (vflag)
	printf("%s vn=%d mode=%d(%s) stratum=%d txtime=%u:%u\n",
	       li_s[li], vn, md, md_s[md], pkt.stratum,
	       ntohl(pkt.txtime[0]), ntohl(pkt.txtime[1]));

    /* Sanity checks */

    if ((pkt.livnmode & 0xc0) == 0xc0 || !pkt.stratum || !pkt.txtime[0]) {
	fprintf(stderr, "server not synchronized\n");
	if (!fflag) {
	    close(fd);
	    exit(1);
	}
    }

    /*
     * NTP timestamps are represented as a 64-bit unsigned fixed-
     * point number, in seconds relative to 0h on 1 January 1900. The integer
     * part is in the first 32 bits and the fraction part in the last 32 bits.
     */

    ntv.tv_sec = ntohl(pkt.txtime[0]) - UNIXOFFSET;
    ntv.tv_usec = ntohl(pkt.txtime[1]) / 4295;
    gettimeofday(&tv, NULL);
    offset = (ntv.tv_sec - tv.tv_sec) * 1000 + (ntv.tv_usec - tv.tv_usec) / 1000;
    if (vflag)
	printf("remote %ld.%06ld local %ld.%06ld offset %ld msec\n",
	       ntv.tv_sec, ntv.tv_usec, tv.tv_sec, tv.tv_usec, offset);

    if (!nflag && (fflag || abs(offset) > 500)) {
	if (settimeofday(&ntv, NULL) < 0)
	    perror("settimeofday");
    }
    if (!qflag)
	printf("server %s stratum %d offset %s%d.%03d\n",
	       hostname, pkt.stratum, (offset < 0) ? "-" : "",
	       abs(offset) / 1000, abs(offset) % 1000);

    close(fd);
    exit(0);
}

/*
copyright 1997
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
