#include <apollo/base.h>
#include <apollo/cal.h>
#include <apollo/error.h>
#include <apollo/loader.h>
#include <apollo/ms.h>
#include <apollo/pgm.h>
#include <apollo/strl.h>
#include <apollo/time.h>
#include <apollo/vfmt.h>

#include <apollo/a.out.h>
/*
 * <apollo/a.out.h> in turn includes:
 *       <apollo/filehdr.h>
 *       <apollo/aouthdr.h> --> <nlist.h>
 *       <apollo/scnhdr.h>
 *       <apollo/reloc.h>
 *       <apollo/linenum.h>
 *       <apollo/syms.h>
 */
#include <apollo/dst.h>
#include <apollo/mir.h>
#include <apollo/scndata.h>
#include <apollo/sri.h>
#include <apollo/storclass.h>
#include <apollo/symbol.h>
#include <apollo/unwind.h>

#include <ptrace.h>     /* until ptrace_$call, etc. are worked out */
#define _INCLUDE_BSD_SOURCE
#include <sys/types.h>
#include <sys/wait.h>

#define string(x) (x),(sizeof(x)-1)

/*
 * This is arbitrary, but I checked against the system libraries
 */

#define KG_SYMBOL_LENGTH_MAX 64

typedef struct
{
    unsigned short jsr;
    void **trap_addr;
    unsigned short ret;

    char name[ KG_SYMBOL_LENGTH_MAX ];
    unsigned short name_len;
    void *real_addr;

    /* later we put in info regarding number+format of args, etc. */
}
    trap_t __attribute((aligned(2)));

void
name_$resolve
(
    name_$long_pname_t   xpname,
    short               &xpnamelen,
    uid_$t              *rtn_uid,
    status_$t           *status
);

void
file_$get_attributes
(
    uid_$t          &uid,
    short           &opts,
    short           &vtoce_hdr_size,
    unsigned long   *obj_info,
    unsigned long   *vtoce_hdr,
    status_$t       *status
);

void
file_$attributes
(
    uid_$t          &uid,
    unsigned long   *o_vtoce_hdr,
    status_$t       *status
);

int fork(void);

typedef short enum
{
    cal_$relative,
    cal_$absolute
}
    cal_$rel_abs_t;

boolean
cal_$timeval_to_clock
(
    cal_$rel_abs_t  &rel_abs,
    time_$timeval_t &timeval,
    time_$clock_t   *clock
);

void
pgm_$exec_uid
(
    uid_$t          &uid,
    short           &arg_count,
    pgm_$arg_ptr    *arg_vector,
    char            *env_vector[],
    status_$t       *status
);

#define ms_$map_from_top     ((void *) 0xffffffff)
typedef unsigned int ms_$map_flags_set_t;

#define ms_$keep_on_exec      ((ms_$map_flags_set_t) 1<<0)  /* don't unmap on exec */
#define ms_$ok_to_extend      ((ms_$map_flags_set_t) 1<<2)  /* ok to extend length */
#define ms_$no_pager          ((ms_$map_flags_set_t) 1<<3)  /* don't use extensible pager */
#define ms_$anon              ((ms_$map_flags_set_t) 1<<6)  /* use anonymous VM */
#define ms_$share             ((ms_$map_flags_set_t) 1<<7)  /* if ms_$anon: share mapping with forked children */

/* ------------------------------------------------------------------- */
/* M A P L X -- map and lock an object given its name */
extern void *ms_$maplx
    ( char                name[]
    , short int           & len
    , unsigned long int   & start
    , unsigned long int   & length
    , ms_$conc_mode_t     & conc
    , ms_$acc_mode_t      & access
    , void                *&va
    , ms_$map_flags_set_t &flags
    , unsigned long int   * len_mapped
    , status_$t           * status
    );

void main(void)
{
    status_$t status;
    short argument_count;
    pgm_$argv_ptr arg_vector_ptr;
    uid_$t uid;
    unsigned long int coff_file_length, coff_file_mapped;
    void *coff_file;

    FILHDR *fh;
    struct syment *symbol_table;
    long number_of_symbols;
    unsigned long *string_table;
    unsigned short ius;

    SCNHDR *sh = 0;
    RELOC *reloc;
    long nreloc, ils;

    unsigned long trap_vector_size, trap_vector_mapped;
    trap_t *trap_vector;

    loader_$handle_t lib_image;
    void *lib_start;

    int pid;
    struct ptrace_set ptrace_set;
    union wait waitu;

    error_$init_std_format(ios_$errout, '?', string("trap_global"));
    pgm_$get_args(&argument_count, &arg_vector_ptr);

    if( argument_count < 1 )
    {
        pgm_$exit();
    }

    {
        unsigned long o_vtoce_hdr[16];
        name_$resolve(arg_vector_ptr[1]->chars, arg_vector_ptr[1]->len, &uid, &status);
        if( status.all != status_$ok )
        {
            error_$std_format(status, "Can't resolve \"%a\"%$", arg_vector_ptr[1]->chars, &arg_vector_ptr[1]->len);
            pgm_$exit();
        }

        file_$attributes(uid, o_vtoce_hdr, &status);
        if( status.all != status_$ok )
        {
            error_$std_format(status, "Can't get attributes for \"%a\"%$", arg_vector_ptr[1]->chars, &arg_vector_ptr[1]->len);
            pgm_$exit();
        }

        coff_file_length = o_vtoce_hdr[7];
    }
                    
    coff_file = ms_$mapl(arg_vector_ptr[1]->chars, arg_vector_ptr[1]->len, 0, coff_file_length, ms_$nr_xor_1w, ms_$r, false, &coff_file_mapped, &status);
    if( status.all != status_$ok )
    {
        error_$std_format(status, "Can't map \"%a\"%$", arg_vector_ptr[0]->chars, &arg_vector_ptr[0]->len);
        pgm_$exit();
    }

    fh = (struct filehdr *)coff_file;

    if( !ISCOFF(fh->f_magic) )
    {
        status.all = loader_$not_a_program;
        error_$std_format(status, "Cannot dump \"%a\"%$", arg_vector_ptr[1]->chars, &arg_vector_ptr[1]->len);
        pgm_$exit();
    }

    symbol_table = (struct syment *)& ((char *)coff_file)[ fh->f_symptr ];
    number_of_symbols = fh->f_nsyms;
    string_table = (unsigned long *)&symbol_table[ number_of_symbols ];

    for( ius = 1; ius <= fh->f_nscns; ius++ )
    {
        sh = &(((SCNHDR *)&(((char *)coff_file)[ FILHSZ + fh->f_opthdr ])) [ ius-1 ]);

        if( strl_$eq(sh->s_name, 5, ".aptv", 5) == true )
        {
            break;
        }

    }

    if( ius > fh->f_nscns )
    {
        status.all = loader_$invalid_section_number; /* close as I can */
        error_$std_format(status, "There is no .aptv section; punting%$");
        pgm_$exit();
    }

    reloc = (RELOC *)&(((char *)coff_file)[ sh->s_relptr ]);
    if( sh->s_nreloc == S_NRELOC_ESC )
    {
        if( reloc->r_type != R_NRELOC )
        {
            status.all = loader_$invalid_image_handle;
            error_$std_format(status, "Invalid .aptv relocation table%$");
            pgm_$exit();
        }
        else
        {
            nreloc = reloc->r_vaddr;
            reloc++;
        }
    }
    else
    {
        nreloc = sh->s_nreloc;
    }

    trap_vector_size = sizeof(trap_t) * nreloc;
    trap_vector = (trap_t *)ms_$maplx("", 0, 0, trap_vector_size, ms_$cowriters, ms_$wrx, ms_$map_from_top, ms_$keep_on_exec | ms_$anon | ms_$share, &trap_vector_mapped, &status);
    if( status.all != status_$ok )
    {
        error_$std_format(status, "Cannot map temporary file%$");
        pgm_$exit();
    }

    for( ils = 0; ils < nreloc; ils++ )
    {
        RELOC *r = &reloc[ ils ];
        trap_t *t = &trap_vector[ ils ];
        SYMENT *s = &symbol_table[ r->r_symndx ];
        char *name;
        unsigned short name_lim, name_len;

        t->jsr = 0x4EB9;
        t->trap_addr = (void **)r->r_vaddr;
        t->ret = 0x4E75;

/*{ 
  unsigned short *pjsr = &t->jsr, *pret = &t->ret;
  vfmt_$write("0x%8zluh JSR 0x%8zluh%/0x%8zluh RET%.", &pjsr, &t->trap_addr, &pret);
}*/

        if( s->n_zeroes == 0 )
        {
            if( s->n_offset < *string_table )
            {
                long max = *string_table - s->x_offset;
                name = &((char *)string_table)[ s->n_offset ];
                name_lim = max > 65535 ? 65535 : max;
            }
            else
            {
                name = "<name not defined>";
                name_lim = 20;
            }
        }
        else
        {
            name = s->n_name;
            name_lim = 8;
        }
        
        for( name_len = 0; name_len < name_lim; name_len++ )
        {
            if( name[ name_len ] == '\0' )
            {
                break;
            }
        }

        if( strl_$eq(name, 5, "pfm_$", 5) == true ||
            strl_$eq(name, 5, "fpp_$", 5) == true ||
            strl_$eq(name, 2, "m$", 2) == true ||
            strl_$index(name, name_len, "$", 1) == 0 )
        {
            vfmt_$ws(ios_$errout, "Warning: NOT trapping \"%a\"!%.", name, &name_len);
            t->trap_addr = 0;
            continue;
        }

        strl_$copy(name, name_len, t->name, &t->name_len, KG_SYMBOL_LENGTH_MAX);
        (void)loader_$kg_lookup(name, (short)name_len, loader_$kg_symbol, loader_$kg_function_only, &t->real_addr);

/*vfmt_$write("Preparing to trap \"%a\"%.", name, &name_len);*/
    }

    ms_$unmap(coff_file, coff_file_mapped, &status);
    if( status.all != status_$ok )
    {
        error_$std_format(status, "Cannup unmap executable%$");
        pgm_$exit();
    }

    loader_$load(string("trap_global_calls_lib"), loader_$keep_on_exec, &lib_image, &status);
    if( status.all != status_$ok )
    {
        error_$std_format(status, "Can't load \"trap_global_calls_lib\"%$");
        pgm_$exit();
    }

    lib_start = loader_$lookup_start_addr(lib_image, &status);
    if( status.all != status_$ok )
    {
        error_$std_format(status, "Can't find entry point in \"trap_global_calls_lib\"%$");
        pgm_$exit();
    }

    switch( (pid = fork()) )
    {
        extern int errno;
        case -1:
            status.all = 0x090c0000 + errno;
            error_$std_format(status, "Can't fork%$");
            pgm_$exit();
        case 0: /* child */
/*vfmt_$write("Child going%.");*/
            if( ptrace(0, 0, 0, 0, 0, 0) == -1 )
            {
                status.all = 0x090c0000 + errno;
                error_$std_format(status, "Can't ptrace(TRACEME, ...)%$");
                pgm_$exit();
            }
sleep(5);
            pgm_$exec_uid(uid, argument_count-1, &arg_vector_ptr[1], 0, &status);
            error_$std_format(status, "Can't exec%$");
            pgm_$exit();
        default: /* parent */
/*vfmt_$write("Parent going%.");*/
            break;  /* as the others exit, just step out of the switch */
    }

    if( ptrace(19, pid, 0, 0, &ptrace_set, 0) == -1 )
    {
        extern int errno;
        status.all = 0x090c0000 + errno;
        error_$std_format(status, "Can't ptrace(INQUIRE_OPTIONS, ...)%$");
        pgm_$exit();
    }

    ptrace_set.events |= (1<<PTRACE_EXEC);

    if( ptrace(18, pid, 0, 0, &ptrace_set, 0) == -1 )
    {
        extern int errno;
        status.all = 0x090c0000 + errno;
        error_$std_format(status, "Can't ptrace(SET_OPTIONS, ...)%$");
        pgm_$exit();
    }

    if( wait(/*&waitu*/0) == -1 )
    {
        extern int errno;
        status.all = 0x090c0000 + errno;
        error_$std_format(status, "Can't wait%$");
        pgm_$exit();
    }

#if 0
    if( waitu.w_Stopval != 0177 || waitu.w_Stopevent != PTRACE_EXEC )
    {
        /* status.all == .. what? */
        error_$std_format(status, "Unexpected wait response%$");
        pgm_$exit();
    }
#endif

    for( ils = 0; ils < nreloc; ils++ )
    {
        trap_t *t = &trap_vector[ ils ];
/*        *(t->trap_addr) = (void *)&t; */
        if( t->trap_addr == 0 ) continue;

        if( ptrace(17, pid, t->trap_addr, sizeof(void *), &t, 0) == -1 )
        {
            extern int errno;
            status.all = 0x090c0000 + errno;
            error_$std_format(status, "Can't ptrace(WRITEDATA, ...)%$");
            pgm_$exit();
        }

/*vfmt_$write("Child 0x%8zluh now points to 0x%8zluh%.", &t->trap_addr, &t);*/

        t->trap_addr = lib_start;
    }

    if( ptrace(11, pid, 1, 0, 0, 0) == -1 )
    {
        extern int errno;
        status.all = 0x090c0000 + errno;
        error_$std_format(status, "Can't ptrace(DETACH, ...)%$");
        pgm_$exit();
    }

    if( wait(/*&waitu*/0) == -1 )
    {
        extern int errno;
        status.all = 0x090c0000 + errno;
        error_$std_format(status, "Can't wait%$");
        pgm_$exit();
    }

    pgm_$exit();
}
