/*
 * trint - Interpret troff output.  Normally gives x,y position of each
 * character in 1/6 of a point with origin at upper left, along with
 * font and size changes.  -f flag gives only font and size changes,
 * no characters.  -b flag gives cattable output approximating the
 * appearance of the troff output.  -p flag attempts to draw the stuff
 * on a CRT screen, like tc(1).  -x and -y set screen sizes.
 * -g flag for gigi.
 *
 * Jim Rees University of Washington Oct 81
 * Troff code interpreter is lifted from vroff.
 */

#include <stdio.h>
#include <sgtty.h>
#include "chartab.h"

/*#define PFLG	/* Only if you have the Berkeley termcap library */

#define UPI	432
#define UPP	(UPI*11)
#define SPFONT  03
#define RAILMAG 03
#define MCASE   0100
#define PSIZE   017

#define BXSIZ	256
#define BYSIZ	256

struct {
	int scode;
	int rcode;
} ptsizes[] = {
		010, 6,
		0,   7,
		01,  8,
		07,  9,
		02,  10,
		03,  11,
		04,  12,
		05,  14,
		011, 16,
		06,  18,
		012, 20,
		013, 22,
		014, 24,
		015, 28,
		016, 36,
		0,   0
};

FILE *inbuf, *ttyin;

int esc, lead, back, verd;
int oback, olead;
int col, row;
int psize, mcase, railmag;
int doublesize;

int pflg,bflg,fflg,gflg;
static char map[BYSIZ][BXSIZ];
int xsize,ysize;
char *ftnames[] = {
	"Roman",
	"Italic",
	"Bold",
	"Special",
};

#ifdef PFLG
char tcap[1024],tcb[20],*tcbp;
char *cm, *cl;
short ospeed;
int tputc();
#endif

main(ac, av) 
int ac;
char *av[];
{
	register i;

	while (ac > 1 && av[1][0] == '-') {
		switch (av[1][1]) {
		case 'f':
			fflg = 1;
			break;
		case 'p':
			pflg = 1;
			break;
		case 'b':
			bflg = 1;
			break;
		case 'g':
			gflg = 1;
			break;
		case 'x':
			xsize = atoi(av[2]);
			ac--;
			av++;
			break;
		case 'y':
			ysize = atoi(av[2]);
			ac--;
			av++;
			break;
		}
		ac--;
		av++;
	}

	ttyin = fopen("/dev/tty", "r");

	asctab[18] = '-';    /* 3/4 em dash */
	spectab[32] = '_';   /* underrule */
	spectab[102] = '|';     /* box rule */

#ifdef PFLG
	tgetent(tcap, getenv("TERM"));
	if (xsize <= 0)
		xsize = tgetnum("co");
	if (pflg) {
		struct sgttyb sb;

		ioctl(2, TIOCGETP, &sb);
		ospeed = sb.sg_ospeed;
		if (ysize <= 0 && (ysize = tgetnum("li")) <=0)
			ysize = 24;
		tcbp = tcb;
		cl = tcbp;
		tgetstr("cl", &tcbp);
		cm = tcbp;
		tgetstr("cm", &tcbp);
		tputs(cl, 1, tputc);
	} else
#endif
	if (bflg) {
		if (xsize <= 0)
			xsize = 80;
		if (ysize <= 0)
			ysize = 66;
	} else if (gflg) {
		xsize = 768;
		ysize = 480;
		printf("\33PpS(EN0T(S1))W(A0S0N0M1)");
	}

	if(ac > 1)
		{
		while(--ac)
			{
			av++;
			if((inbuf=fopen(av[0], "r")) == NULL) {
				perror(av[0]);
				continue;
			}
			ofile();
			}
		}
	   else
		{
		inbuf = stdin;
		ofile();
		}
	if (gflg)
		printf("\33[24;1H");
#ifdef PFLG
	else if (pflg)
		tputs(tgoto(cm, 0, ysize-1), 1, tputc);
#endif
	exit(0);
}

/* ofile() -- Reads troff codes from FILE *inbuf until EOF.  Calls on
 *      force(c) to display each character after determining row and col.
 */

ofile()
{
	register int c;
	static int initialized;

	while((c = getc(inbuf)) != EOF) {

		if(!c)
			continue;
		if(c & 0200)
			{
			esc += (~c) & 0177;
			continue;
			}
		if(esc)
			{
			if(back)
				esc = -esc;
			col += esc;
			esc = 0;
			}
		if((c & 0377) < 0100)	/*  Purely for efficiency  */
			goto normal_char;
		switch(c)
			{
			case 0100:
				if(!initialized++) {
					row = col = lead = esc = lead = 0;
					verd = back = mcase = railmag = 0;
				}
				else
					goto out;
				break;
			case 0101:	/* lower rail */
				railmag &= ~01;
				break;
			case 0102:	/* upper rail */
				railmag |= 01;
				break;
			case 0103:	/* upper mag */
				railmag |= 02;
				break;
			case 0104:	/* lower mag */
				railmag &= ~02;
				break;
			case 0105:	/* lower case */
				mcase = 0;
				break;
			case 0106:	/* upper case */
				mcase = MCASE;
				break;
			case 0107:	/* escape forward */
				back = 0;
				break;
			case 0110:	/* escape backwards */
				back = 1;
				break;
			case 0111:	/* stop */
				force(c);
				break;
			case 0112:	/* lead forward */
				verd = 0;
				break;
			case 0113:	/* undefined */
				break;
			case 0114:	/* lead backward */
				verd = 1;
				break;
			case 0115:	/* undefined */
			case 0116:
			case 0117:
				break;
			default:
				if((c & 0340) == 0140)	/* leading */
					{
					lead = (~c) & 037;
					if(verd)
						lead = -lead;
					row += lead * 3;
					continue;
					}
				if((c & 0360) == 0120)	/* size change */
					{
					psize = c & PSIZE;
					/* apologies, but this is a patch
					 to simulate real typesetter hardware*/
					if (psize <= 010)
						doublesize = 0;
					else
						doublesize = 55;
					continue;
					}
				if(c & 0300)
					continue;
			normal_char:
				force(c);
			}
		}
	out:
		force(0);
}

/* force(c) -- Displays troff code "c" at "row" and "col" in size "psize"
 *      and font "railmag."
 *      Substitutes from table subs for ligatures.  Waits at the end of
 *      a page for keyboard or joystick input.  Translates from troff
 *      character codes to Megatek character codes with table  asctab or
 *      spectab.  Translates point sizes to Megatek sizes.  Calls on putm()
 *      to display the character.
 */

force(c)
char c;
{
	int i,j,rsize,x,y;
	static int osize,ormag;
	char dl;

	if (row >= UPP || c==0) {       /* End of page */
		eopage();
		if (c==0)
			return;
	}
	c |= mcase;
	if (railmag == SPFONT)
		c = spectab[c];
	else
		c = asctab[c];
	for (i=0; ptsizes[i].rcode != 0; i++)
		if (ptsizes[i].scode == psize)
			break;
	rsize = ptsizes[i].rcode;
	x = (col - doublesize - UPI) / (float) UPI / 6.5 * xsize;
	y = (row - UPI/2.) / (float) UPI / 10. * ysize;

	if (x >= 0 && x < xsize && y >= 0 && y < ysize
	    && c >= ' ' && c <= '~') {
		if (bflg)
			map[y][x] = c;
		else if (gflg) {
			dl = (c == '"') ? '\'' : '"';
			printf("P[%d,%d]T%c%c%c", x, y, dl, c, dl);
		}
#ifdef PFLG
		else if (pflg) {
			tputs(tgoto(cm, x, y), 1, tputc);
			putchar(c);
		}
#endif
	}
	if (!bflg && !pflg && !gflg) {
		if (railmag != ormag || rsize != osize) {
			printf("%s %d point\n", ftnames[railmag], rsize);
			ormag = railmag;
			osize = rsize;
		}
		if (!fflg) {
			if (c < ' ' || c > '~')
				printf("%d,%d %x\n", col, row, c);
			else
				printf("%d,%d %c\n", col, row, c);
		}
	}
}

#ifdef PFLG
#endif

eopage()
{
	int x,y;
	char c;

	if (pflg || gflg) {
		if (gflg)
			printf("\33[24;1HHit <return> for more");
#ifdef PFLG
		else if (pflg) {
			tputs(tgoto(cm, 0, ysize-1), 1, tputc);
			printf("Hit <return> for more");
		}
#endif
		fflush(stdout);
		while (getc(ttyin) != '\n')
			;
		if (gflg)
			printf("\33PpS(E)");
#ifdef PFLG
		else if (pflg)
			tputs(cl, 1, tputc);
#endif
	}
	else if (bflg) {
		for (y=0; y<ysize; y++) {
			for (x=0; x<xsize; x++) {
				c = map[y][x];
				if (c == '\0')
					c = ' ';
				putchar(c);
				map[y][x] = ' ';
			}
			putchar('\n');
		}
	} else
		printf("EOP\n");
	row -= UPP;
}

#ifdef PFLG
tputc(c)
char c;
{
	putchar(c);
}
#endif
