/*
 * uuar
 *
 * uuencode/uudecode function with tar command-line syntax
 * Tries to be robust about ignoring garbage on extract, so you should
 * be able to cat together split files with junk at the front and run
 * them through.
 *
 * Written by Jim Rees, University of Michigan, Sept. 1992
 * Copyright 1996 Regents of The University of Michigan - All Rights Reserved
 * See legal hoo-ha at the end of this file.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

int vflag;

main(ac, av)
int ac;
char *av[];
{
    char *cp, cmd = '\0', *arfilename = NULL, *openmode;
    FILE *arfile;

    if (ac < 2)
	usage();

    for (cp = av[1]; *cp != '\0'; cp++)
	switch (*cp) {
	case 'a':
	case 'c':
	case 't':
	case 'x':
	    if (cmd != '\0')
		usage();
	    cmd = *cp;
	    break;
	case 'v':
	    vflag++;
	    break;
	case 'f':
	    arfilename = av[2];
	    av++;
	    ac--;
	    break;
	default:
	    usage();
	}

    if (cmd == '\0')
	usage();
    if (arfilename == NULL)
	arfilename = "-";
    if (!strcmp(arfilename, "-"))
	arfile = ((cmd == 'a' || cmd == 'c') ? stdout : stdin);
    else {
	if (cmd == 'a')
	    openmode = "a";
	else if (cmd == 'c')
	    openmode = "w";
	else
	    openmode = "r";
	arfile = fopen(arfilename, openmode);
	if (arfile == NULL) {
	    fprintf(stderr, "can't open %s\n", arfilename);
	    exit(1);
	}
    }

    if (cmd == 'a' || cmd == 'c')
	writear(arfile, arfilename, ac - 2, &av[2]);
    else
	readar(arfile, cmd, ac - 2, &av[2]);

    exit(0);
}

usage()
{
    fprintf(stderr, "usage: uuar [cxt](v)(f) (arfile) files ...\n");
    exit(1);
}

writear(arfile, arname, ac, flist)
FILE *arfile;
char *arname;
int ac;
char *flist[];
{
    int i;

    for (i = 0; i < ac; i++) {
	if (i > 0)
	    fprintf(arfile, "\n");
	write1(arfile, flist[i]);
    }
    myfclose(arfile, arname, 0);
}

write1(arfile, name)
FILE *arfile;
char *name;
{
    FILE *infile, *lfile;
    struct stat stb;
    int i, c, word, count;
    char buf[64], *cp;

    infile = fopen(name, "r");
    if (infile == NULL) {
	fprintf(stderr, "can't open %s\n", name);
	return;
    }
    fstat(fileno(infile), &stb);
    if (vflag) {
	lfile = (arfile == stdout) ? stderr : stdout;
	if (vflag > 1)
	    fprintf(lfile, "%03o ", stb.st_mode & 0777);
	fprintf(lfile, "%s\n", name);
    }
    fprintf(arfile, "begin %03o %s\n", stb.st_mode & 0777, name);

    count = 0;
    while (!feof(infile)) {
	if (count == 0)
	    cp = &buf[1];
	word = 0;
	for (i = 0; i < 3; i++) {
	    c = getc(infile);
	    if (c == EOF) {
		if (i == 0)
		    break;
		c = '\0';
	    } else
		count++;
	    word <<= 8;
	    word |= c;
	}
	if (i > 0)
	    for (i = 0; i < 4; i++) {
		c = ((word >> ((3 - i) * 6)) & 0x3f) + ' ';
		if (c == ' ')
		    c += 64;
		*cp++ = c;
	    }
	if (count >= 45 || feof(infile)) {
	    buf[0] = count + ' ';
	    *cp++ = '\n';
	    *cp++ = '\0';
	    fputs(buf, arfile);
	    count = 0;
	}
    }

    fprintf(arfile, "%c\nend\n", ' ' + 64);
    fclose(infile);
}

readar(arfile, cmd, ac, flist)
FILE *arfile;
char cmd;
int ac;
char *flist[];
{
    while (read1(arfile, cmd, ac, flist) >= 0)
	;
}

read1(arfile, cmd, ac, flist)
FILE *arfile;
char cmd;
int ac;
char *flist[];
{
    char *name, buf[120], obuf[90], *cp, *ocp;
    int mode, i, n, word, count, buflen;
    FILE *outfile;

    if (skip(arfile, &name, &mode) < 0)
	return -1;

    /* If flist was supplied, and name not on list, return */
    if (ac > 0) {
	for (i = 0; i < ac; i++)
	    if (!strcmp(flist[i], name))
		break;
	if (i >= ac)
	    return 0;
    }

    /* List */
    if (cmd == 't' || (cmd == 'x' && vflag)) {
	if (cmd == 't' && vflag || (cmd == 'x' && vflag > 1))
	    printf("%03o ", mode);
	printf("%s\n", name);
    }

    /* Extract */
    if (cmd != 'x')
	return 0;
    outfile = fopen(name, "w");
    if (outfile == NULL) {
	fprintf(stderr, "can't create %s\n", name);
	return 0;
    }

    while (fgets(buf, sizeof buf, arfile) != NULL) {
	if (!strcmp("end\n", buf))
	    break;
	cp = buf;
	ocp = obuf;
	n = nextn(&cp);

	/* check line length; some uuencodes append one junk byte */
	buflen = strlen(buf);
	i = ((n + 2) / 3) * 4 + 2;
	if (n <= 0 || n > 45 || (i != buflen && i != buflen - 1))
	    /* garbage */
	    continue;

	for (count = 0; count < n; ) {
	    word = nextword(&cp);
	    if (word < 0)
		break;
	    for (i = 0; i < 3; i++)
		if (count < n) {
		    *ocp++ = (word >> (8 * (2 - i))) & 0xff;
		    count++;
		}
	}
	if (count == n)
	    fwrite(obuf, 1, n, outfile);
    }
    fchmod(fileno(outfile), mode);
    myfclose(outfile, name, 0);
    return 0;
}

/* Return next number, mod-64 + ' ' */

nextn(cpp)
unsigned char **cpp;
{
    while (**cpp != '\0' && (**cpp < ' ' || **cpp >= ' ' + 65))
	(*cpp)++;
    return (**cpp != '\0' ? (*(*cpp)++ - ' ') & 0x3f : -1);
}

nextword(cpp)
unsigned char **cpp;
{
    int i, word;

    word = 0;
    for (i = 0; i < 4; i++) {
	if (**cpp < ' ' || **cpp >= ' ' + 65)
	    return -1;
	word <<= 6;
	word |= (*(*cpp)++ - ' ') & 0x3f;
    }
    return word;
}

/* Skip to beginning of next file in archive */

skip(arfile, namep, modep)
FILE *arfile;
char **namep;
int *modep;
{
    char buf[100];
    static char namebuf[80];

    do {
	if (fgets(buf, sizeof buf, arfile) == NULL)
	    return -1;
    } while (!isprefix("begin", buf));
    if (sscanf(buf, "%*s %o %s", modep, namebuf) != 2) {
	fprintf(stderr, "bad header: %s", buf);
	return -1;
    }
    *namep = namebuf;
    return 0;
}

/* Skip current file in archive */

skipover(arfile)
FILE *arfile;
{
    char buf[100];

    do {
	if (fgets(buf, sizeof buf, arfile) == NULL)
	    return -1;
    } while (!isprefix("end", buf));
    return 0;
}

isprefix(p, s)
char *p, *s;
{
    return (!strncmp(p, s, strlen(p)));
}

myfclose(f, name, dosync)
FILE *f;
char *name;
int dosync;
{
    if (ferror(f) || fflush(f) == EOF ||
	(dosync && fsync(fileno(f)) < 0) || fclose(f) == EOF) {
	fprintf(stderr, "trouble closing %s\n", name);
	return -1;
    }
    return 0;
}

/*
 ******************************************************************************
 * Copyright 1996 Regents of The University of Michigan - All Rights Reserved *
 *                                                                            *
 * Permission to use, copy, modify, and distribute this software and its      *
 * documentation for any purpose and without fee is hereby granted,           *
 * provided that the above copyright notice appears in all copies and that    *
 * both that copyright notice and this permission notice appear in            *
 * supporting documentation, and that the name of The University of Michigan  *
 * not be used in advertising or publicity pertaining to distribution of      *
 * the software without specific, written prior permission. This software     *
 * is supplied as is without expressed or implied warranties of any kind.     *
 *                                                                            *
 *            Center for Information Technology Integration                   *
 *                  Information Technology Division                           *
 *                     The University of Michigan                             *
 *                       535 W. William Street                                *
 *                        Ann Arbor, Michigan                                 *
 *                            313-763-4403                                    *
 *                        info@citi.umich.edu                                 *
 *                                                                            *
 ******************************************************************************
 */
