/*

Simple minded remote file system interconnect.
This is the server.  It runs on any bsd4.2 or 4.3 system.

-d Print debug info.
-s Run as a server, rather than from inetd.
-f Fork and run in the background.
-p xx Use port xx instead of looking for 'v2d' in /etc/services.
-C <dir> cd to <dir> on startup.

Written by Jim Rees, Apollo Computer, 1985.

*/

#include <stdio.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#include "v2d.h"
#include "attr.h"

#define V2PORT	1210

/** End of tunables **/

#define OSTSZ	28

extern char *malloc();
extern struct hostent *gethostbyname(), *gethostbyaddr();
extern int errno;

char *v2_getstr(), *v2_getbuf();
int catch_chld();

char *cmd_names[] = {
	"open",
	"close",
	"read",
	"write",
	"ostat",
	"ofstat",
	"seek",
	"chn",
	"unlink",
	"rdln",
	"mkdir",
	"trunc",
	"chmod",
	"chown",
	"link",
	"slink",
	"readlink",
	"stat",
	"fstat",
	"utime",
	"readdir",
};

struct connection {
	int sfd;
	FILE *sf;
	int fd;
	DIR *dirp;
	char name[256];
};

int debug, fflag, sflag;
int accept_port;
int uid_local, uid_remote;

main (ac, av)
int ac;
char *av[];
{
	char *rootdir = NULL;
	struct connection conn;

	uid_local = getuid();
	if (uid_local == 0)
		rootdir = "/";

	while (ac > 1 && av[1][0] == '-') {
		switch (av[1][1]) {
		case 'd':
			debug = 1;
			break;
		case 'f':
			fflag = 1;
			break;
		case 's':
			sflag = 1;
			break;
		case 'p':
			accept_port = htons(atoi(av[2]));
			ac--;
			av++;
			break;
		case 'C':
			rootdir = av[2];
			ac--;
			av++;
			break;
		}
		ac--;
		av++;
	}

	if (rootdir)
		chdir(rootdir);

	if (sflag)
		do_accept();

	conn.sfd = 0;
	serve(&conn);
}

/* Do all the stuff that inetd normally does for us.
   This routine does not return.  */

do_accept()
{
	int s, slen;
	struct servent *sp;
	struct sockaddr_in saddr;
	struct connection conn;
	int keep_alive = 1;

	if (accept_port == 0) {
		if ((sp = getservbyname("v2d", 0)) != NULL)
			accept_port = sp->s_port;
		else {
			fprintf(stderr, "can't find port for v2d in /etc/services; using %d\n", V2PORT);
			accept_port = htons(V2PORT);
		}
	}

	saddr.sin_family = AF_INET;
	saddr.sin_port = accept_port;
#ifdef BSD4_2
	saddr.sin_addr.S_un.S_addr = INADDR_ANY;
#else
	saddr.sin_addr.s_addr = INADDR_ANY;
#endif

	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket");
		exit(1);
	}

#ifdef BSD4_2
	if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, 0, 0) < 0) {
#else
	if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, &keep_alive, sizeof keep_alive) < 0) {
#endif
		perror("setsockopt");
		exit(1);
	}

	if (bind(s, &saddr, sizeof saddr) < 0) {
		perror("bind");
		exit(1);
	}

	if (listen(s, 4) < 0) {
		perror("listen");
		exit(1);
	}

	if (!debug) {
		if (fork())
			exit(0);
		signal(SIGHUP, SIG_IGN);
		close(0);
		close(1);
		close(2);
	}
	signal(SIGCHLD, catch_chld);

	slen = sizeof saddr;

	for (;;) {
		if ((conn.sfd = accept(s, &saddr, &slen)) < 0) {
			if (debug)
				perror("accept");
			continue;
		}
		if (!fflag) {
			if (fork() == 0) {
				close(s);
				serve(&conn);
				exit(0);
			}
		} else
			serve(&conn);
		close(conn.sfd);
	}
}

catch_chld()
{
	while (wait3(NULL, WNOHANG, NULL) > 0)
		;
}

serve(cp)
struct connection *cp;
{
	struct sockaddr_in saddr;
	int saddrlen = sizeof (struct sockaddr_in);
	int inaddr;
	struct hostent *peernamep;
	int cmd;

	if (getpeername(cp->sfd, &saddr, &saddrlen) < 0)
		return;
	inaddr = ntohl(saddr.sin_addr.s_addr);
	if (debug) {
		if ((peernamep = gethostbyaddr(&saddr.sin_addr.s_addr, sizeof (saddr.sin_addr.s_addr), AF_INET)) != NULL)
			printf("Connection accepted from %s\n", peernamep->h_name);
		else
			printf("Connection accepted from %x\n", inaddr);
	}

	cp->sf = fdopen(cp->sfd, "w");
	cp->fd = -1;
	cp->dirp = NULL;
	strcpy(cp->name, "?");

	uid_remote = v2_getlong(cp->sfd);
	if (uid_local == 0)
		setuid(uid_remote);

	for (;;) {
		errno = 0;
		cmd = v2_getlong(cp->sfd);
		if (cmd < 0) {
			if (errno && debug)
				perror("serve");
			c_close(cp);
			break;
		}
		if (debug)
			printf("%s %s\n", cmd_names[cmd - 1], cp->name);

		switch (cmd) {
		case C_OPEN:
			c_open(cp);
			break;
		case C_CLOSE:
			c_close(cp);
			break;
		case C_READ:
		case C_RDLN:
			c_read(cp, cmd);
			break;
		case C_WRITE:
			c_write(cp);
			break;
		case C_STAT:
		case C_OSTAT:
			c_stat(cp, cmd);
			break;
		case C_FSTAT:
		case C_OFSTAT:
			c_fstat(cp, cmd);
			break;
		case C_SEEK:
			c_seek(cp);
			break;
		case C_CHN:
			c_chn(cp);
			break;
		case C_UNLINK:
			c_unlink(cp);
			break;
		case C_MKDIR:
			c_mkdir(cp);
			break;
		case C_TRUNC:
			c_trunc(cp);
			break;
		case C_CHMOD:
			c_chmod(cp);
			break;
		case C_CHOWN:
			c_chown(cp);
			break;
		case C_LINK:
			c_link(cp);
			break;
		case C_SLINK:
			c_slink(cp);
			break;
		case C_READLINK:
			c_readlink(cp);
			break;
		case C_UTIME:
			c_utime(cp);
			break;
		case C_READDIR:
			c_readdir(cp);
			break;
		default:
			fprintf(stderr, "bad cmd %d\n", cmd);
			c_close(cp);
			break;
		}
	}
	fclose(cp->sf);
}

c_open(cp)
struct connection *cp;
{
	int iflag, flag, type, needwrite;
	char *name;
	struct stat stb;
	static DIR *mydirp;

	if ((name = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	iflag = v2_getlong(cp->sfd);
	if (debug)
		printf("open %s %o\n", name, iflag);

	needwrite = iflag & 3;
	flag = O_RDWR;
	if (iflag & XO_CREAT)
		flag |= O_CREAT;
	if (iflag & XO_TRUNC)
		flag |= O_TRUNC;
	if (iflag & XO_APPEND)
		flag |= O_APPEND;
	if (iflag & XO_EXCL)
		flag |= O_EXCL;

	type = T_RW;

	for (;;) {
		if ((cp->fd = open(name, flag, 0666)) >= 0)
			break;
		if ((flag & O_RDWR) && !needwrite) {
			/* Try again, but for read only */
			flag &= ~3;
			type = T_RO;
		} else {
			v2_putlong(cp->sf, -1 - errno);
			fflush(cp->sf);
			return;
		}
	}

	fstat(cp->fd, &stb);
	strcpy(cp->name, name);
	if (stb.st_mode & S_IFDIR) {
		if (mydirp == NULL) {
			mydirp = opendir(".");
			close(mydirp->dd_fd);
		}
		mydirp->dd_fd = cp->fd;
		rewinddir(mydirp);
		cp->dirp = mydirp;
		type |= T_DIR;
	} else
		type |= T_FILE;
	v2_putlong(cp->sf, type);
	fflush(cp->sf);
}

c_close(cp)
struct connection *cp;
{
	if (cp->fd >= 0)
		close(cp->fd);
	cp->fd = -1;
	cp->dirp = NULL;
}

c_read(cp, cmd)
struct connection *cp;
int cmd;
{
	char *bp;
	int i, n, key, asked;

	asked = v2_getlong(cp->sfd);
	if (asked < 0)
		return;
	bp = v2_getbuf(asked);
	if (cmd == C_RDLN)
		key = lseek(cp->fd, 0L, 1);
	n = read(cp->fd, bp, asked);
	if (cmd == C_RDLN) {
		for (i = 0; i < n; )
			if (bp[i++] == '\n')
				break;
		n = i;
		lseek(cp->fd, key + i, 0);
	}
	if (n < 0)
		n = -1 - errno;
	if (debug)
		printf("read %d got %d\n", asked, n);
	v2_putlong(cp->sf, n);
	while (n > 0) {
		i = fwrite(bp, 1, n, cp->sf);
		n -= i;
		bp += i;
	}
	fflush(cp->sf);
}

#define CDIRECTSIZE (sizeof (struct cdirect) - 2)

c_readdir(cp, cmd)
struct connection *cp;
int cmd;
{
	int maxcnt, bufsize;
	int n, nsent = 0, bytessent = 0;
	long key, ateof = 0;
	struct direct *dp;
	struct cdirect *cdp;
	static char zeros[4];

	maxcnt = v2_getlong(cp->sfd);
	bufsize = v2_getlong(cp->sfd);
	if (debug)
		printf("maxcnt %d bufsize %d\n", maxcnt, bufsize);
	cdp = (struct cdirect *) v2_getbuf(CDIRECTSIZE);

	while (nsent < maxcnt && bytessent < bufsize) {
		key = telldir(cp->dirp);
		cdp->cd_handle = htonl(key);
		if ((dp = readdir(cp->dirp)) == NULL) {
			ateof = 1;
			break;
		}
		n = (CDIRECTSIZE + dp->d_namlen + 4) & ~3;
		if (bytessent + n > bufsize) {
			seekdir(cp->dirp, key);
			break;
		}
		cdp->cd_next = htons(n);
		cdp->cd_type = htons(1);
		cdp->cd_rfu1 = cdp->cd_rfu2 = htonl(dp->d_ino);
		cdp->cd_rfu3 = htonl(0);
		cdp->cd_namelen = htons(dp->d_namlen);
		v2_putlong(cp->sf, n);
		fwrite(cdp, CDIRECTSIZE, 1, cp->sf);
		fwrite(dp->d_name, 1, dp->d_namlen, cp->sf);
		fwrite(zeros, 1, n - (CDIRECTSIZE + dp->d_namlen), cp->sf);
		nsent++;
		bytessent += n;
		if (debug)
			printf("%d %.*s\n", n, dp->d_namlen, dp->d_name);
	}
	v2_putlong(cp->sf, ateof);
	if (debug)
		printf("%d entries in %d bytes, ateof %d\n", nsent, bytessent, ateof);
	fflush(cp->sf);
}

c_write(cp)
struct connection *cp;
{
	int n;
	char *bp;

	bp = v2_getstr(cp->sfd, &n);
	if (bp == NULL)
		return;
	n = write(cp->fd, bp, n);
	if (n < 0)
		n = -1 - errno;
	v2_putlong(cp->sf, n);
	fflush(cp->sf);
}

c_stat(cp, cmd)
struct connection *cp;
int cmd;
{
	char *name;
	struct stat stb;
	struct attr atb;
	int e;

	if ((name = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	if (debug)
		printf("stat %s\n", name);
	if ((e = stat(name, &stb)) < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	if (e >= 0) {
		fixstat(&stb, &atb);
		fwrite(&atb, (cmd == C_STAT ? sizeof atb : OSTSZ), 1, cp->sf);
	}
	fflush(cp->sf);

	/* An inq_only open will do a stat first thing.  Save the name for
	   later operations.  Stupid, disgusting hack.  Should implement
	   fchown/fchmod vs. chown/chmod instead.
	   Given this hack, should fix cp->name in c_chn too.  */

	if (cp->fd < 0)
		strcpy(cp->name, name);
}

c_fstat(cp, cmd)
struct connection *cp;
int cmd;
{
	struct stat stb;
	struct attr atb;

	fstat(cp->fd, &stb);
	fixstat(&stb, &atb);
	fwrite(&atb, (cmd == C_FSTAT ? sizeof atb : OSTSZ), 1, cp->sf);
	fflush(cp->sf);
}

fixstat(s, a)
struct stat *s;
struct attr *a;
{
	a->dev =	htonl(s->st_dev);
	a->ino =	htonl(s->st_ino);
	a->mode =	htonl(s->st_mode);
	a->nlink =	htonl(s->st_nlink);
	a->uid =	htonl((s->st_uid == uid_local && uid_local != 0) ? uid_remote : s->st_uid);
	a->gid =	htonl(s->st_gid);
/*	a->rdev =	htonl(s->st_rdev); */
	a->size =	htonl(s->st_size);
	a->atime =	htonl(s->st_atime);
/*	a->spare1 =	htonl(s->st_spare1); */
	a->mtime =	htonl(s->st_mtime);
/*	a->spare2 =	htonl(s->st_spare2); */
	a->ctime =	htonl(s->st_ctime);
/*	a->spare3 =	htonl(s->st_spare3); */
	a->blksize =	htonl(s->st_blksize);
	a->blocks =	htonl((s->st_blocks + (1024 / DEV_BSIZE) - 1) * DEV_BSIZE / 1024);
}

c_seek(cp)
struct connection *cp;
{
	long key, rkey;
	int how;

	key = v2_getlong(cp->sfd);
	how = v2_getlong(cp->sfd);
	if (cp->dirp == NULL)
		/* ordinary file */
		rkey = lseek(cp->fd, key, how);
	else {
		/* directory */
		if (how == 0) {
			if (key == 0)
				rewinddir(cp->dirp);
			else
				seekdir(cp->dirp, key);
		}
		rkey = telldir(cp->dirp);
	}
	if (debug)
		printf("seek %d %d -> %d\n", how, key, rkey);
	v2_putlong(cp->sf, rkey);
	fflush(cp->sf);
}

c_chn(cp)
struct connection *cp;
{
	char *name, *oldname, *newname;
	int n;

	if ((name = v2_getstr(cp->sfd, &n)) == NULL)
		return;
	oldname = malloc(n + 1);
	strcpy(oldname, name);
	if ((newname = v2_getstr(cp->sfd, &n)) == NULL)
		return;
	if (debug)
		printf("chn %s %s\n", oldname, newname);
	if (cp->dirp != NULL) {
		if (rename(oldname, newname) < 0) {
			v2_putlong(cp->sf, -1 - errno);
			fflush(cp->sf);
			free(oldname);
			return;
		}
	} else {
		if (link(oldname, newname) < 0) {
			v2_putlong(cp->sf, -1 - errno);
			fflush(cp->sf);
			free(oldname);
			return;
		}
		if (unlink(oldname) < 0) {
			v2_putlong(cp->sf, -1 - errno);
			fflush(cp->sf);
			free(oldname);
			unlink(newname);
			return;
		}
	}
	free(oldname);
	v2_putlong(cp->sf, 0);
	fflush(cp->sf);
}

c_unlink(cp)
struct connection *cp;
{
	char *name;
	int e;

	if ((name = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	if (debug)
		printf("unlink %s\n", name);
	if ((e = rmdir(name)) < 0 && errno == ENOTDIR)
		e = unlink(name);
	if (e < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

c_mkdir(cp)
struct connection *cp;
{
	char *name;
	int e;

	if ((name = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	if (debug)
		printf("mkdir %s\n", name);
	if ((e = mkdir(name, 0777)) < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	fflush(cp->sf);

	cp->dirp = opendir(name);
	cp->fd = cp->dirp->dd_fd;
}

c_trunc(cp)
struct connection *cp;
{
	int e;

	if ((e = ftruncate(cp->fd, lseek(cp->fd, 0L, 1))) < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

c_chmod(cp)
struct connection *cp;
{
	int e, mode;

	mode = v2_getlong(cp->sfd);
	if (cp->fd >= 0) {
		if (fchmod(cp->fd, mode) < 0)
			e = -1 - errno;
	} else {
		if (chmod(cp->name, mode) < 0)
			e = -1 - errno;
	}
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

c_chown(cp)
struct connection *cp;
{
	int uid, gid, e;

	uid = v2_getlong(cp->sfd);
	gid = v2_getlong(cp->sfd);
	if (cp->fd >= 0) {
		if ((e = fchown(cp->fd, uid, gid)) < 0)
			e = -1 - errno;
	} else {
		if ((e = chown(cp->name, uid, gid)) < 0)
			e = -1 - errno;
	}
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

c_link(cp)
struct connection *cp;
{
	char buf[100], *name1, *name2;
	int e;

	if ((name1 = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	strcpy(buf, name1);
	name1 = buf;
	if ((name2 = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	if (debug)
		printf("link %s %s\n", name1, name2);
	if ((e = link(name1, name2)) < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

c_slink(cp)
struct connection *cp;
{
	char buf[256], *name1, *name2;
	int e;

	if ((name1 = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	strcpy(buf, name1);
	name1 = buf;
	if ((name2 = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	if (debug)
		printf("symlink %s %s\n", name1, name2);
	if ((e = symlink(name1, name2)) < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

c_readlink(cp)
struct connection *cp;
{
	char *name, buf[256];
	int n;

	if ((name = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	if (debug)
		printf("readlink %s\n", name);
	if ((n = readlink(name, buf, sizeof buf)) < 0)
		n = -1 - errno;
	v2_putlong(cp->sf, n);
	if (n >= 0)
		fwrite(buf, 1, n, cp->sf);
	fflush(cp->sf);
}

c_utime(cp)
struct connection *cp;
{
	char *name;
	time_t timep[2];
	int e;

	if ((name = v2_getstr(cp->sfd, NULL)) == NULL)
		return;
	timep[0] = v2_getlong(cp->sfd);
	timep[1] = v2_getlong(cp->sfd);
	if ((e = utime(name, timep)) < 0)
		e = -1 - errno;
	v2_putlong(cp->sf, e);
	fflush(cp->sf);
}

char *
v2_getstr(fd, lenp)
int fd, *lenp;
{
	int i, n;
	char *bp, *ret;

	if ((n = v2_getlong(fd)) < 0)
		return NULL;
	ret = bp = v2_getbuf(n + 1);
	if (lenp != NULL)
		*lenp = n;
	while (n > 0) {
		if ((i = read(fd, bp, n)) <= 0)
			return NULL;
		n -= i;
		bp += i;
	}
	*bp = '\0';
	return ret;
}

char *
v2_getbuf(n)
int n;
{
	static char *bufp;
	static int size;

	if (n > size) {
		if (bufp != NULL)
			free(bufp);
		size = n;
		bufp = malloc(n);
	}
	return bufp;
}

v2_getlong(fd)
int fd;
{
	long n;

	if (read(fd, &n, sizeof n) != sizeof n)
		return -1;
	n = ntohl(n);
	return n;
}

v2_putlong(f, n)
FILE *f;
long n;
{
	n = htonl(n);
	fwrite(&n, sizeof n, 1, f);
}

v2_putshort(f, n)
FILE *f;
short n;
{
	n = htons(n);
	fwrite(&n, sizeof n, 1, f);
}
